/**
 * OPEN-XCHANGE legal information
 *
 * All intellectual property rights in the Software are protected by
 * international copyright laws.
 *
 *
 * In some countries OX, OX Open-Xchange, open xchange and OXtender
 * as well as the corresponding Logos OX Open-Xchange and OX are registered
 * trademarks of the Open-Xchange, Inc. group of companies.
 * The use of the Logos is not covered by the GNU General Public License.
 * Instead, you are allowed to use these Logos according to the terms and
 * conditions of the Creative Commons License, Version 2.5, Attribution,
 * Non-commercial, ShareAlike, and the interpretation of the term
 * Non-commercial applicable to the aforementioned license is published
 * on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 * Please make sure that third-party modules and libraries are used
 * according to their respective licenses.
 *
 * Any modifications to this package must retain all copyright notices
 * of the original copyright holder(s) for the original code used.
 *
 * After any such modifications, the original and derivative code shall remain
 * under the copyright of the copyright holder(s) and/or original author(s)per
 * the Attribution and Assignment Agreement that can be located at
 * http://www.open-xchange.com/EN/developer/. The contributing author shall be
 * given Attribution for the derivative code and a license granting use.
 *
 *  Copyright (C) 2004-2014 Open-Xchange, Inc.
 *  Mail: info@open-xchange.com
 *
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License, Version 2 as published
 *  by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc., 59
 *  Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package com.openxchange.office_communication.jms.core.camel;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.apache.camel.CamelContext;
import org.apache.camel.StartupListener;
import org.apache.commons.lang3.StringUtils;

//=============================================================================
public class CamelUtils
{
	//-------------------------------------------------------------------------
	private CamelUtils ()
	{}

	//-------------------------------------------------------------------------
	/** wait till the given camel-context is started.
	 * 
	 *  Throws an exception if start failed or timeout was reached.
	 *  Return without an error otherwise.
	 * 
	 *  @param	aContext [IN]
	 *  		the context we want to know if its started.
	 *  
	 *  @param	nTimeout [IN]
	 *  		timeout for wait.
	 *  
	 *  @throws an exception if start failed or timeout occur.
	 */
	public static void waitForContextStart (final CamelContext aContext,
										    final int          nTimeout)
	    throws Exception
	{
		final CountDownLatch aStartSync = new CountDownLatch (1);
		
		aContext.addStartupListener(new StartupListener ()
		{
			@Override
			public void onCamelContextStarted(final CamelContext aContext       ,
										      final boolean      bAlreadyStarted)
				throws Exception
			{
				aStartSync.countDown();
			}
		});

		final boolean bStarted = aStartSync.await(nTimeout, TimeUnit.MILLISECONDS);
		if ( ! bStarted)
			throw new TimeoutException ("Camel context not started in time ["+nTimeout+"].");
	}

	//-------------------------------------------------------------------------
	/** create an absolute path to a class path resource owned by an implementation class.
	 *  
	 *  Hard coding such paths is not a good idea. One package refactor in between and the
	 *  string is out dated and broken. Unfortunately those broken references will be detected
	 *  at runtime ...
	 *  
	 *  This method bind class path resources to it's 'owner implementation' by convention.
	 *  The package will be used as path-root ... and the also given relative path is add.
	 *  
	 *  @param	aOwner [IN]
	 *  		the class which wants to load such resources.
	 *  
	 *	@param	sRelFile [IN]
	 *			the relative path and file name for the requested resource.
	 *
	 * 	@return the absolute resource path and file name.
	 */
	public static String getAbsoluteResFileName (final Class< ? > aOwner  ,
												 final String     sRelFile)
		throws Exception
	{
		String sResPath = aOwner.getName ();
			   sResPath = StringUtils.substringBeforeLast(sResPath, ".");
			   sResPath = StringUtils.replace(sResPath, ".", "/");
			   sResPath = "/" + sResPath;

	    String sResFile = sResPath;
	    if ( ! StringUtils.startsWith(sRelFile, "/"))
	    	sResFile += "/";
	    sResFile += sRelFile;
	    
	    return sResFile;
	}
}
