package com.openexchange.office.tools;

import org.apache.commons.logging.Log;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.exception.OXException;
import com.openexchange.jslob.DefaultJSlob;
import com.openexchange.jslob.JSlobService;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;

/**
 * A helper class to read/write user configuration entries in the
 * context of a specific user.
 *
 * @author Carsten Driesner
 */
public class UserConfigurationHelper {

    /**
     * Initializes a new {@link UserConfigurationHelper}
     *
     * @param services
     *  The service lookup instance to be used by the new
     *  UserConfigurationHelper instance.
     *
     * @param session
     *  The session of the client in that context the user configuration data
     *  should be retrieved/set.
     *
     * @param id
     *  The root path for the configuration bundle, e.g. (io.ox/office). Must
     *  not be null.
     */
	public UserConfigurationHelper(ServiceLookup services, Session session, String id) {
        m_services = services;
        m_session = session;
        m_id = id;
    }

	/**
	 * Retrieves a JSONObject from the user configuration referenced by the
	 * provided relative path.
	 *
	 * @param path
	 *  The relative path to he user configuration entry represented as a
	 *  JSONObject.
	 *
	 * @return
	 *  The value as a JSONObject or null if the entry does not exists, or
	 *  the type is not a JSONObject.
	 */
	public JSONObject getJSONObject(String path) {
        Object value = getGenericValue(path);
        return (value instanceof JSONObject) ? (JSONObject)value : null;
	}

	/**
	 * Retrieves a JSONArray from the user configuration referenced by the
	 * provided relative path.
	 *
	 * @param path
	 *  The relative path to he user configuration entry represented as a
	 *  JSONArray.
	 *
	 * @return
	 *  The value as a JSONArray or null if the entry does not exists, or
	 *  the type is not a JSONArray.
	 */
	public JSONArray getJSONArray(String path) {
		Object value = getGenericValue(path);
		return (value instanceof JSONArray) ? (JSONArray)value : null;
	}

	/**
	 * Retrieves a String from the user configuration referenced by the
	 * provided relative path.
	 *
	 * @param path
	 *  The relative path to he user configuration entry represented as a
	 *  String.
	 *
	 * @return
	 *  The value as a String or null if the entry does not exists, or
	 *  the type is not a String.
	 */
	public String getString(String path) {
		Object value = getGenericValue(path);
		return (value instanceof String) ? (String)value : null;
	}

	/**
	 * Retrieves a Double from the user configuration referenced by the
	 * provided relative path.
	 *
	 * @param path
	 *  The relative path to he user configuration entry represented as a
	 *  Double.
	 *
	 * @return
	 *  The value as a Double or null if the entry does not exists, or
	 *  the type is not a Double.
	 */
	public Double getDouble(String path) {
		Object value = getGenericValue(path);
		return (value instanceof Double) ? (Double)value : null;
	}

	/**
	 * Retrieves a Long from the user configuration referenced by the
	 * provided relative path.
	 *
	 * @param path
	 *  The relative path to he user configuration entry represented as a
	 *  Long.
	 *
	 * @return
	 *  The value as a Long or null if the entry does not exists, or
	 *  the type is not a Long.
	 */
	public long getLong(String path) {
		Object value = getGenericValue(path);
		return ((value instanceof Long) || (value instanceof Integer)) ? (Long)value : null;
	}

	/**
	 * Retrieves a Integer from the user configuration referenced by the
	 * provided relative path.
	 *
	 * @param path
	 *  The relative path to he user configuration entry represented as a
	 *  Integer.
	 *
	 * @return
	 *  The value as a Integer or null if the entry does not exists, or
	 *  the type is not a Integer.
	 */
	public int getInteger(String path) {
		Object value = getGenericValue(path);
		return (value instanceof Integer) ? (Integer)value : null;
	}

	/**
	 * Retrieves a Boolean from the user configuration referenced by the
	 * provided relative path.
	 *
	 * @param path
	 *  The relative path to he user configuration entry represented as a
	 *  Boolean.
	 *
	 * @return
	 *  The value as a Boolean or null if the entry does not exists, or
	 *  the type is not a Boolean.
	 */
	public boolean getBoolean(String path) {
		Object value = getGenericValue(path);
		return (value instanceof Boolean) ? (Boolean)value : null;
	}

    /**
     * Sets the value of the user configuration entry.
     *
     * @param path
	 *  The relative path to the user configuration entry which should get
	 *  the new provided boolean value.
	 *
     * @param value
     *  The new boolean value.
     *
     * @return
     *  TRUE if the value was set successfully, otherwise FALSE.
     */
	public boolean setValue(String path, Boolean value) {
        return setGenericValue(path, value);
    }

    /**
     * Sets the value of the user configuration entry.
     *
     * @param path
	 *  The relative path to the user configuration entry which should get
	 *  the new provided string value.
	 *
     * @param value
     *  The new string value.
     *
     * @return
     *  TRUE if the value was set successfully, otherwise FALSE.
     */
    public boolean setValue(String path, String value) {
        return setGenericValue(path, value);
    }

    /**
     * Sets the value of the user configuration entry.
     *
     * @param path
	 *  The relative path to the user configuration entry which should get
	 *  the new provided integer value.
	 *
     * @param value
     *  The new integer value.
     *
     * @return
     *  TRUE if the value was set successfully, otherwise FALSE.
     */
    public boolean setValue(String path, Integer value) {
    	return setGenericValue(path, value);
    }

    /**
     * Sets the value of the user configuration entry.
     *
     * @param path
	 *  The relative path to the user configuration entry which should get
	 *  the new provided long value.
	 *
     * @param value
     *  The new long value.
     *
     * @return
     *  TRUE if the value was set successfully, otherwise FALSE.
     */
    public boolean setValue(String path, Long value) {
    	return setGenericValue(path, value);
    }

    /**
     * Sets the value of the user configuration entry.
     *
     * @param path
	 *  The relative path to the user configuration entry which should get
	 *  the new provided double value.
	 *
     * @param value
     *  The new double value.
     *
     * @return
     *  TRUE if the value was set successfully, otherwise FALSE.
     */
    public boolean setValue(String path, Double value) {
    	return setGenericValue(path, value);
    }

    /**
     * Sets the value of the user configuration entry.
     *
     * @param path
	 *  The relative path to the user configuration entry which should get
	 *  the new provided JSONObject value.
	 *
     * @param value
     *  The new JSONObject value.
     *
     * @return
     *  TRUE if the value was set successfully, otherwise FALSE.
     */
    public boolean setValue(String path, JSONObject value) {
    	return setGenericValue(path, value);
    }

    /**
     * Sets the value of the user configuration entry.
     *
     * @param path
	 *  The relative path to the user configuration entry which should get
	 *  the new provided JSONArray value.
	 *
     * @param value
     *  The new JSONArray value.
     *
     * @return
     *  TRUE if the value was set successfully, otherwise FALSE.
     */
    public boolean setValue(String path, JSONArray value) {
    	return setGenericValue(path, value);
    }

    /**
     * Retrieves a value from the user configuration.
     *
     * @param path
     *  The relative path to the user configuration entry, e.g.
     *  "portal/recents" or "spreadsheet/maxCells".
     *
     * @return
     *  The value or null if the entry doesn't exists. The type of the value
     *  could be a simple type (Boolean, String, Double, Long, Integer), or a
     *  JSON type if the entry has a complex structure (JSONObject, JSONArray).
     */
	private Object getGenericValue(String path) {
        Object value = null;

        initOnDemand();
        if ((null != m_jslobService) && (null != path)) {
            try {
	            JSONObject configTree = m_jslobService.get(m_id, m_session).getJsonObject();
                JSONObject jsonObject = getParentObject(path, configTree);
                String attrName = getAttributeNameFromPath(path);

                if ((null != attrName) && (null != jsonObject)) {
                	// retrieve value from the parent object
            	    value = jsonObject.opt(attrName);
                }
            } catch (OXException e) {
                LOG.error("RT connection: JSONException catched while accessing JSlobService", e);
            }
        }

        return value;
    }

	/**
	 * Sets a value to the configuration referenced by the provided path.
	 *
	 * @param path
	 *  The relative path to the configuration entry to be written into the
	 *  user configuration.
	 *
	 * @param value
	 *  The value that should be written to the configuration entry. It must
	 *  be a Boolean/Integer/String for simple types or a JSONObject/JSONArray
	 *  for complex types.
	 *
	 * @return
	 */
    private boolean setGenericValue(String path, Object value) {
        initOnDemand();

        if (null != m_jslobService) {
        	try {
                JSONObject configTree = m_jslobService.get(m_id, m_session).getJsonObject();
                JSONObject parent = getParentObject(path, configTree);
                String attrName = getAttributeNameFromPath(path);

                if ((null != attrName) && (null != parent)) {
    			    parent.put(attrName, value);
    			    m_jslobService.set(m_id, new DefaultJSlob(configTree), m_session);
    			    return true;
                }
            } catch (JSONException e) {
                LOG.error("RT connection: JSONException catched while accessing JSlobService", e);
        	} catch (OXException e) {
                LOG.error("RT connection: OXException catched while accessing JSlobService", e);
        	}
        }

        return false;
    }

	/**
	 * Retrieves the parent object containing the referenced entry defined
	 * by the provided path.
	 *
	 * @param path
	 *  A relative path to the user configuration entry.
	 *
	 * @return
	 *  The JSONObject which contains the configuration entry or null if the
	 *  parent object does not exists.
	 */
	private JSONObject getParentObject(String path, JSONObject configTree) {
		JSONObject value = null;

		if (null != configTree) {

	        // cascade through the hierarchical structure to find the correct setting
            int i = 0;
            String[] entries = path.split("/");
            Object parent = null;

            while (i < (entries.length - 1)) {
                String entry = entries[i++];
                 // ignore empty strings
                if (entry.length() > 0) {
                    Object obj = configTree.opt(entry);
                    if (obj instanceof JSONObject) {
                        // step down on JSONObject
                    	configTree = (JSONObject)obj;
                        parent = configTree;
                    } else {
                    	parent = obj;
                    }
                }
            }

            if ((null != parent) && (parent instanceof JSONObject)) {
            	value = (JSONObject)parent;
            }
		}

		return value;
	}

	/**
	 *  Provides the attribute name of a relative configuration entry path.
	 *
	 * @param path
	 *  A relative configuration entry path.
	 *
	 * @return
	 *  The name of the attribute referenced by the provided path or null
	 *  if the attribute name cannot be determined.
	 */
	private String getAttributeNameFromPath(String path) {
		String attributeName = null;

		if (null != path) {
			int lastIndex = path.lastIndexOf('/');
			if ((lastIndex >= 0) && ((lastIndex + 1) < path.length())) {
				attributeName = path.substring(lastIndex+1);
			}
		}

		return attributeName;
	}

    /**
     * Initializes the instance on demand. Reading the configuration values
     * from the user configuration service.
     *
     * @return
     *  TRUE if the initialization was successful otherwise FALSE.
     */
    private boolean initOnDemand() {
        boolean result = false;

        if (!m_bInit && (null != m_services) && (null != m_session) && (null != m_id)) {
            m_jslobService = m_services.getService(JSlobService.class);
            if (null != m_jslobService) {
                try {
                	// just to be sure that id exists and system works correctly
                    m_jslobService.get(m_id, m_session);
                    result = true;
                } catch (OXException e) {
                    LOG.error("RT connection: Exception catched while accesing JSlobService", e);
                }
            }

            m_bInit = true;
        }

        return result;
    }

    private ServiceLookup m_services = null;
    private Session m_session = null;
    private String m_id = null;
    private boolean m_bInit = false;
    private JSlobService m_jslobService = null;

    private static final Log LOG = com.openexchange.log.Log.loggerFor(UserConfigurationHelper.class);
}
