/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.tools;

import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import org.apache.commons.logging.Log;


public class LocalFileMappingManager {
    private static final Log LOG = com.openexchange.log.Log.loggerFor(LocalFileMappingManager.class);

    private final HashMap<String, String> m_localFileMapper = new HashMap<String, String>();
    private final HashMap<String, String> m_pathMapper = new HashMap<String, String>();

    public static final String prefixID = "template:";

    /**
     *
     * Initializes a new {@link LocalFileMappingManager}.
     */
    public LocalFileMappingManager() {
    }

    /**
     * Determines if the local file mapper has a association to the provided
     * path.
     *
     * @param path
     *  The local file path that should be checked.
     *
     * @return
     *  The associated ID of the path or null if no ID is stored for the path.
     */
    public synchronized String getIDFromPath(String path) {
        return m_pathMapper.get(path);
    }

    /**
     * Provides a local file path which is associated with the provided id.
     *
     * @param id
     *  A id which describes a certain local file path (system-dependent).
     *
     * @return
     *  The local file path or null if there is no association is stored for
     *  the provided id.
     */
    public synchronized String getPathFromId(String id) {
        return m_localFileMapper.get(id);
    }

    /**
     * Set a new association between and id and a local file path.
     *
     * @param id
     *  A id which describes a certain local file path.
     *
     * @param path
     *  A local file system path (system-dependent).
     *
     * @return
     *  TRUE if the id was not part of the current map, otherwise FALSE.
     */
    public synchronized boolean associateIdAndPath(String id, String path) {
        String result = m_localFileMapper.put(id, path);
        m_pathMapper.put(path, id);
        return (null == result);
    }

    /**
     * Provides a local file path which is associated with the provided id.
     *
     *
     * @param id
     *  A id which describes a certain local file path.
     *
     * @param path
     *  A local file system path (system-dependent).
     *
     * @return
     *  TRUE if the id and path have been stored, otherwise FALSE.
     */
    public synchronized boolean associateIdAndPath(String id, String path, boolean noOverwrite) {
        String result = m_localFileMapper.get(id);
        if (!(noOverwrite && (null != result))) {
            m_localFileMapper.put(id, path);
            m_pathMapper.put(path, id);
        }
        return (null == result);
    }

    /**
     * Generates a ID from a provided path.
     *
     * @param path
     *  A local file system path;
     *
     * @return
     *  A ID generated from the path;
     */
    public static String generateIDFromPath(String path) {
        String result = null;

        try {
            byte[] message = path.getBytes("UTF-8");
            MessageDigest md = MessageDigest.getInstance("MD5");
            byte[] theDigest = md.digest(message);
            StringBuilder sb = new StringBuilder();
            sb.append(prefixID);
            for (byte b : theDigest) {
                sb.append(String.format("%02X", b));
            }
            return sb.toString();
        } catch (UnsupportedEncodingException e) {
            LOG.error("Exception while creating a id from a local file path", e);
        } catch (NoSuchAlgorithmException e) {
            LOG.error("Exception while creating a id from a local file path", e);
        }
        return result;
    }

    /**
     * Provides an input stream from the file associated with the provided
     * id.
     *
     * @param id
     *  A id which describes a certain local file path.
     *
     * @return
     *  An input stream to access the file data or null if no association
     *  is known or the file is not accessible.
     */
    public InputStream getInputStreamFromID(String id) {
        InputStream stream = null;
        String path = getPathFromId(id);

        if ((null != path) && (path.length() > 0)) {
            stream = FileHelper.getDocumentStreamFromLocalFile(path);
        }
        return stream;
    }
}
