/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import javax.xml.bind.JAXBException;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.docx4j.dml.Theme;
import org.docx4j.docProps.core.CoreProperties;
import org.docx4j.docProps.core.dc.elements.SimpleLiteral;
import org.docx4j.openpackaging.contenttype.ContentTypes;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.io3.Save;
import org.docx4j.openpackaging.packages.OpcPackage;
import org.docx4j.openpackaging.parts.PartName;
import org.docx4j.openpackaging.parts.ThemePart;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.log.LogFactory;
import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.FilterException.ErrorCode;
import com.openexchange.office.tools.ConfigurationHelper;
import com.openexchange.office.tools.ResourceManager;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;

abstract public class OperationDocument {

    private final static Log LOG = LogFactory.getLog(OperationDocument.class);

    private final ResourceManager       resourceManager;
    private final DocumentProperties    documentProperties;
    protected 	  InputStream           inputDocumentStream;
    private final ServiceLookup         services;
    private final Session				session;
    protected final boolean             saveDebugOperations;
    private boolean                     createFinalDocument = false;

    static final int DEFAULT_BUFFER_SIZE = 8192;

    protected OperationDocument(Session session, ServiceLookup services, InputStream inputDocumentStream, ResourceManager resourceManager, DocumentProperties documentProperties) {
        this.services = services;
        this.session = session;
        this.inputDocumentStream = inputDocumentStream;
        this.resourceManager = resourceManager;
        this.documentProperties = documentProperties;
        saveDebugOperations = Boolean.parseBoolean(ConfigurationHelper.getOfficeConfigurationValue(services, session, "debugoperations", "false"));
    }

    public InputStream createDocument(String applyOperations)
        throws FilterException {

        try {
            // applying some document properties...
            CoreProperties coreProperties = getPackage().getDocPropsCorePart(true).getJaxbElement();
            org.docx4j.docProps.core.dc.elements.ObjectFactory of = new org.docx4j.docProps.core.dc.elements.ObjectFactory();
            String userName = documentProperties.optString(DocumentProperties.PROP_LAST_MODIFIED_BY, "");
            coreProperties.setLastModifiedBy(userName);
            coreProperties.setRevision(documentProperties.optString(DocumentProperties.PROP_REVISION, null));
            coreProperties.setVersion(documentProperties.optString(DocumentProperties.PROP_VERSION, null));

            if(userName.length()>0) {
                SimpleLiteral creatorLiteral = coreProperties.getCreator();
                if(creatorLiteral==null) {
                    creatorLiteral = of.createSimpleLiteral();
                    creatorLiteral.getContent().add(userName);
                    coreProperties.setCreator(creatorLiteral);
                }
            }

            applyOperations(applyOperations);
            InputStream debugDocument = null;

            String uniqueDocumentIdentifier = documentProperties.optString(DocumentProperties.PROP_UNIQUE_DOCUMENT_IDENTIFIER, null);
            if (null != uniqueDocumentIdentifier) {
                putUniqueDocumentIdentifier(uniqueDocumentIdentifier);
            }

            if(saveDebugOperations&&inputDocumentStream instanceof ByteArrayInputStream) {
                try {
                    inputDocumentStream.reset();
                    debugDocument = inputDocumentStream;
                }
                catch(IOException e) {
                    //
                }
            }
            return debugSave(applyOperations, debugDocument, resourceManager);
        }
        catch(Throwable e) {
            LOG.error("OOXML Filter could not save the given document:", e);
            throw (e instanceof FilterException) ? (FilterException)e : new FilterException(e, ErrorCode.CRITICAL_ERROR);
        }
    }

    abstract public void applyOperations(String applyOperations)
        throws FilterException;

    abstract protected JSONObject getOperations()
        throws Exception;

    public InputStream save() throws FilterException {
        return debugSave(null, null, null);
    }

    public void setCreateFinalDocument(boolean createFinalDocument) {
        this.createFinalDocument = createFinalDocument;
    }

    public ServiceLookup getServiceLookup() {
    	return this.services;
	}

    public Session getSession() {
    	return this.session;
 	}
    
    /**
     * Add the unique document identifier to the document.
     *
     * @param uniqueIdentifier
     *  The unique document identifier to be saved into the document stream.
     */
    private boolean putUniqueDocumentIdentifier(String uniqueIdentifier) {
        try {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("uniqueDocumentIdentifier", uniqueIdentifier);

            getPackage().getContentTypeManager().addNonPersistContentFile("oxoffice/extendedData.json",
                jsonObject.toString().getBytes("UTF-8"),"json", "application/octet-stream");
        } catch (UnsupportedEncodingException e) {
            LOG.error("OOXML Filter could not save the unique document indentifier:", e);
            return false;
        } catch (JSONException e) {
            LOG.error("OOXML Filter could not create JSON object to store the unique document indentifier:", e);
            return false;
        }

        return true;
    }

    /* For debug purposes we create a folder (debug) within the destination document
     * that stores each information necessary to repeat the last save operation that was done.
     * (original document, the operations just applied and the content of the resource manager. */

    private InputStream debugSave(String debugOperations, java.io.InputStream debugDocument, ResourceManager debugResourceManager)
        throws FilterException {

        try {
        	
        	if (documentProperties.optBoolean(DocumentProperties.PROP_SAVE_TEMPLATE_DOCUMENT, false)) {
        		if(getPackage().getContentTypeManager().getPartNameOverridenByContentType(ContentTypes.WORDPROCESSINGML_DOCUMENT)!=null) {
        			getPackage().getContentTypeManager().addOverrideContentType(new PartName("/word/document.xml").getURI(), ContentTypes.WORDPROCESSINGML_TEMPLATE);
        		}
        		else if(getPackage().getContentTypeManager().getPartNameOverridenByContentType(ContentTypes.SPREADSHEETML_WORKBOOK)!=null) {
        			getPackage().getContentTypeManager().addOverrideContentType(new PartName("/xl/workbook.xml").getURI(), ContentTypes.SPREADSHEETML_TEMPLATE);
        		}
    			else if(getPackage().getContentTypeManager().getPartNameOverridenByContentType(ContentTypes.PRESENTATION)!=null) {
			    	getPackage().getContentTypeManager().addOverrideContentType(new PartName("/ppt/presentation.xml").getURI(), ContentTypes.PRESENTATIONML_TEMPLATE);
        		}
        	}
        	
            final Save saver = new Save(getPackage());
            if(saveDebugOperations) {
                if(debugOperations!=null) {
                    getPackage().getContentTypeManager().addNonPersistContentFile("debug/operationUpdates.dbg",
                        debugOperations.getBytes("UTF-8"),"dbg", "application/octet-stream");
                }
                if(debugDocument!=null) {
                    ZipInputStream zipInputStream = new ZipInputStream(debugDocument);
                    ByteArrayOutputStream outStream = new ByteArrayOutputStream();
                    ZipOutputStream zipOutputStream = new ZipOutputStream(outStream);
                    ZipEntry zipEntry = null;
                    byte[] buffer = new byte[0x10000];  // 64kb buffer
                    while((zipEntry = zipInputStream.getNextEntry())!=null) {

                        // if the source document is containing a debug folder, we will
                        // skip the old debug information and not include it within our copy
                        if(!zipEntry.getName().startsWith("debug/")) {
                            ZipEntry newEntry = new ZipEntry(zipEntry.getName());
                            zipOutputStream.putNextEntry(newEntry);
                            int read;
                            while ((read = zipInputStream.read(buffer))>=0){
                                zipOutputStream.write(buffer,0,read);
                            }
                            zipOutputStream.closeEntry();
                        }
                        zipInputStream.closeEntry();
                    }
                    zipInputStream.close();
                    zipOutputStream.close();

                    getPackage().getContentTypeManager().addNonPersistContentFile("debug/documentStream.dbg",
                        outStream.toByteArray(), "dbg", "application/octet-stream");
                }
                if(debugResourceManager!=null) {

// TODO:  the resourceManager needs to be able to serialize the resources, so we can store these data also within or debug folder
//
//                  getPackage().getContentTypeManager().addNonPersistContentFile("debug/debugResources.dbg",
//                      debugResourceManager., "application/octet-stream");
                }
            }
            saver.setCheckUnusedRelations(createFinalDocument);
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            saver.save(out);
            return new ByteArrayInputStream(out.toByteArray());


        } catch (Throwable e) {
            LOG.error("OOXML Filter could not save the given document:", e);
            throw new FilterException(e, ErrorCode.CRITICAL_ERROR);
        }
    }

    public void writeFile(String path) throws FilterException {
        InputStream  inputStream = null;
        OutputStream outputStream = null;

        try {
            inputStream = save();

            if (null != inputStream) {
                IOUtils.copy(inputStream, outputStream = new FileOutputStream(new File(path)));
                outputStream.flush();
            }
        } catch (Throwable e) {
            LOG.error("OOXML Filter could not save the given document:", e);
            throw new FilterException(e, ErrorCode.CRITICAL_ERROR);
        } finally {
            IOUtils.closeQuietly(inputStream);
            IOUtils.closeQuietly(outputStream);
        }
    }

    abstract public OpcPackage getPackage();

    abstract public Theme getTheme();

    public ResourceManager getResourceManager() {
        return resourceManager;
    }

    public DocumentProperties getDocumentProperties() {
        return documentProperties;
    }

    public String getUserLanguage() {
        return documentProperties.optString(DocumentProperties.PROP_USER_LANGUAGE, "");
    }
    
    public static FilterException rethrowException(Throwable e) {
    	if(e instanceof FilterException) {
    		final FilterException filterException = (FilterException)e;
    		if(filterException.getErrorcode()==ErrorCode.COMPLEXITY_TOO_HIGH) {
    			LOG.warn("could not load/save document, the document complexity is too high");    			
    		}
    		else if(filterException.getErrorcode()==ErrorCode.CRITICAL_ERROR) {
    			LOG.error(e.getMessage(), e);
    		}
    		return (FilterException)e;
    	}
    	else {
    		LOG.error(e.getMessage(), e);
    		return new FilterException(e, ErrorCode.CRITICAL_ERROR);
    	}
    }

    protected ThemePart getDefaultThemePart(PartName partName) {

        final String themeStr =

            "<a:theme xmlns:a=\"http://schemas.openxmlformats.org/drawingml/2006/main\"" +
            "    name=\"${themeName}\">" +
            "    <a:themeElements>" +
            "        <a:clrScheme name=\"${themeName}\">" +
            "            <a:dk1>" + "${dk1}" + " </a:dk1>" +
            "            <a:lt1>" + "${lt1}" + " </a:lt1>" +
            "            <a:dk2>" + "${dk2}" + " </a:dk2>" +
            "            <a:lt2>" + "${lt2}" + " </a:lt2>" +
            "            <a:accent1>" + "${accent1}" + " </a:accent1>" +
            "            <a:accent2>" + "${accent2}" + " </a:accent2>" +
            "            <a:accent3>" + "${accent3}" + " </a:accent3>" +
            "            <a:accent4>" + "${accent4}" + " </a:accent4>" +
            "            <a:accent5>" + "${accent5}" + " </a:accent5>" +
            "            <a:accent6>" + "${accent6}" + " </a:accent6>" +
            "            <a:hlink>"   + "${hlink}"   + " </a:hlink>"   +
            "            <a:folHlink>"+ "${folHlink}"+ " </a:folHlink>"+
            "        </a:clrScheme>" +
            "        <a:fontScheme name=\"${themeName}\">" +
            "            <a:majorFont>" +
            "                <a:latin typeface=\"${majorFont}\" />" +
            "                <a:ea typeface=\"\" />" +
            "                <a:cs typeface=\"\" />" +
            "            </a:majorFont>" +
            "            <a:minorFont>" +
            "                <a:latin typeface=\"${minorFont}\" />" +
            "                <a:ea typeface=\"\" />" +
            "                <a:cs typeface=\"\" />" +
            "            </a:minorFont>" +
            "        </a:fontScheme>" +
            "        <a:fmtScheme name=\"${themeName}\">" +
            "            <a:fillStyleLst>" +
            "                <a:solidFill>" +
            "                    <a:schemeClr val=\"phClr\" />" +
            "                </a:solidFill>" +
            "                <a:gradFill rotWithShape=\"1\">" +
            "                    <a:gsLst>" +
            "                        <a:gs pos=\"0\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:tint val=\"50000\" />" +
            "                                <a:satMod val=\"300000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"35000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:tint val=\"37000\" />" +
            "                                <a:satMod val=\"300000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"100000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:tint val=\"15000\" />" +
            "                                <a:satMod val=\"350000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                    </a:gsLst>" +
            "                    <a:lin ang=\"16200000\" scaled=\"1\" />" +
            "                </a:gradFill>" +
            "                <a:gradFill rotWithShape=\"1\">" +
            "                    <a:gsLst>" +
            "                        <a:gs pos=\"0\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:shade val=\"51000\" />" +
            "                                <a:satMod val=\"130000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"80000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:shade val=\"93000\" />" +
            "                                <a:satMod val=\"130000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"100000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:shade val=\"94000\" />" +
            "                                <a:satMod val=\"135000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                    </a:gsLst>" +
            "                    <a:lin ang=\"16200000\" scaled=\"0\" />" +
            "                </a:gradFill>" +
            "            </a:fillStyleLst>" +
            "            <a:lnStyleLst>" +
            "                <a:ln w=\"9525\" cap=\"flat\" cmpd=\"sng\" algn=\"ctr\">" +
            "                    <a:solidFill>" +
            "                        <a:schemeClr val=\"phClr\">" +
            "                            <a:shade val=\"95000\" />" +
            "                            <a:satMod val=\"105000\" />" +
            "                        </a:schemeClr>" +
            "                    </a:solidFill>" +
            "                    <a:prstDash val=\"solid\" />" +
            "                </a:ln>" +
            "                <a:ln w=\"25400\" cap=\"flat\" cmpd=\"sng\" algn=\"ctr\">" +
            "                    <a:solidFill>" +
            "                        <a:schemeClr val=\"phClr\" />" +
            "                    </a:solidFill>" +
            "                    <a:prstDash val=\"solid\" />" +
            "                </a:ln>" +
            "                <a:ln w=\"38100\" cap=\"flat\" cmpd=\"sng\" algn=\"ctr\">" +
            "                    <a:solidFill>" +
            "                        <a:schemeClr val=\"phClr\" />" +
            "                    </a:solidFill>" +
            "                    <a:prstDash val=\"solid\" />" +
            "                </a:ln>" +
            "            </a:lnStyleLst>" +
            "            <a:effectStyleLst>" +
            "                <a:effectStyle>" +
            "                    <a:effectLst>" +
            "                        <a:outerShdw blurRad=\"40000\" dist=\"20000\" dir=\"5400000\"" +
            "                            rotWithShape=\"0\">" +
            "                            <a:srgbClr val=\"000000\">" +
            "                                <a:alpha val=\"38000\" />" +
            "                            </a:srgbClr>" +
            "                        </a:outerShdw>" +
            "                    </a:effectLst>" +
            "                </a:effectStyle>" +
            "                <a:effectStyle>" +
            "                    <a:effectLst>" +
            "                        <a:outerShdw blurRad=\"40000\" dist=\"23000\" dir=\"5400000\"" +
            "                            rotWithShape=\"0\">" +
            "                            <a:srgbClr val=\"000000\">" +
            "                                <a:alpha val=\"35000\" />" +
            "                            </a:srgbClr>" +
            "                        </a:outerShdw>" +
            "                    </a:effectLst>" +
            "                </a:effectStyle>" +
            "                <a:effectStyle>" +
            "                    <a:effectLst>" +
            "                        <a:outerShdw blurRad=\"40000\" dist=\"23000\" dir=\"5400000\"" +
            "                            rotWithShape=\"0\">" +
            "                            <a:srgbClr val=\"000000\">" +
            "                                <a:alpha val=\"35000\" />" +
            "                            </a:srgbClr>" +
            "                        </a:outerShdw>" +
            "                    </a:effectLst>" +
            "                    <a:scene3d>" +
            "                        <a:camera prst=\"orthographicFront\">" +
            "                            <a:rot lat=\"0\" lon=\"0\" rev=\"0\" />" +
            "                        </a:camera>" +
            "                        <a:lightRig rig=\"threePt\" dir=\"t\">" +
            "                            <a:rot lat=\"0\" lon=\"0\" rev=\"1200000\" />" +
            "                        </a:lightRig>" +
            "                    </a:scene3d>" +
            "                    <a:sp3d>" +
            "                        <a:bevelT w=\"63500\" h=\"25400\" />" +
            "                    </a:sp3d>" +
            "                </a:effectStyle>" +
            "            </a:effectStyleLst>" +
            "            <a:bgFillStyleLst>" +
            "                <a:solidFill>" +
            "                    <a:schemeClr val=\"phClr\" />" +
            "                </a:solidFill>" +
            "                <a:gradFill rotWithShape=\"1\">" +
            "                    <a:gsLst>" +
            "                        <a:gs pos=\"0\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:tint val=\"40000\" />" +
            "                                <a:satMod val=\"350000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"40000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:tint val=\"45000\" />" +
            "                                <a:shade val=\"99000\" />" +
            "                                <a:satMod val=\"350000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"100000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:shade val=\"20000\" />" +
            "                                <a:satMod val=\"255000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                    </a:gsLst>" +
            "                    <a:path path=\"circle\">" +
            "                        <a:fillToRect l=\"50000\" t=\"-80000\" r=\"50000\" b=\"180000\" />" +
            "                    </a:path>" +
            "                </a:gradFill>" +
            "                <a:gradFill rotWithShape=\"1\">" +
            "                    <a:gsLst>" +
            "                        <a:gs pos=\"0\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:tint val=\"80000\" />" +
            "                                <a:satMod val=\"300000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"100000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:shade val=\"30000\" />" +
            "                                <a:satMod val=\"200000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                    </a:gsLst>" +
            "                    <a:path path=\"circle\">" +
            "                        <a:fillToRect l=\"50000\" t=\"50000\" r=\"50000\" b=\"50000\" />" +
            "                    </a:path>" +
            "                </a:gradFill>" +
            "            </a:bgFillStyleLst>" +
            "        </a:fmtScheme>" +
            "    </a:themeElements>" +
            "    <a:objectDefaults />" +
            "    <a:extraClrSchemeLst />" +
            "</a:theme>";

        final String defaultSchemeName = "Office";

        ThemePart themePart = null;
        try {
            themePart = new ThemePart(partName);
            java.util.HashMap<String, String> mappings = new java.util.HashMap<String, String>();
            mappings.put("themeName", defaultSchemeName);
            mappings.put("dk1", "<a:sysClr val=\"windowText\" lastClr=\"000000\" />");
            mappings.put("lt1", "<a:sysClr val=\"window\" lastClr=\"FFFFFF\" />");
            mappings.put("dk2", "<a:srgbClr val=\"44546A\" />");
            mappings.put("lt2", "<a:srgbClr val=\"E7E6E6\" />");
            mappings.put("accent1", "<a:srgbClr val=\"5B9BD5\" />");
            mappings.put("accent2", "<a:srgbClr val=\"ED7D31\" />");
            mappings.put("accent3", "<a:srgbClr val=\"A5A5A5\" />");
            mappings.put("accent4", "<a:srgbClr val=\"FFC000\" />");
            mappings.put("accent5", "<a:srgbClr val=\"4472C4\" />");
            mappings.put("accent6", "<a:srgbClr val=\"70AD47\" />");
            mappings.put("hlink",   "<a:srgbClr val=\"0563C1\" />");
            mappings.put("folHlink","<a:srgbClr val=\"954F72\" />");
            mappings.put("majorFont", "Times New Roman");
            mappings.put("minorFont", "Arial");
            Theme theme = (Theme)org.docx4j.XmlUtils.unmarshallFromTemplate(themeStr, mappings);
            themePart.setJaxbElement(theme);
        }
        catch (JAXBException e) {
            e.printStackTrace();
        }
        catch (InvalidFormatException e) {
            e.printStackTrace();
        }
        return themePart;
    }
}
