/*
 * Copyright 2012 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.odftoolkit.odfdom.component;

import java.util.AbstractList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.json.JSONArray;
import org.json.JSONObject;
import static org.odftoolkit.odfdom.component.JsonOperationConsumer.insertRows;
import org.odftoolkit.odfdom.doc.table.OdfTable;
import org.odftoolkit.odfdom.doc.table.OdfTableRow;
import org.odftoolkit.odfdom.dom.element.OdfStylableElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableCellElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnGroupElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnsElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableHeaderColumnsElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableHeaderRowsElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableRowElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableRowGroupElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableRowsElement;
import org.odftoolkit.odfdom.pkg.OdfElement;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

/**
 * A MultiCoomponent uses a single XML element to represent multiple components.
 * This container can be used for spreadsheet row and cell components using
 * repeated elements via an attribute.
 *
 * @author svante.schubertATgmail.com
 */
public class Table<T> extends Component {

	public Table(OdfElement componentElement, Component parent) {
		super(componentElement, parent);
	}
	private static final String INSERT_BEFORE = "before";
	private static final String INSERT_AFTER = "after";
	private static final String FORMULA_PREFIX = "of:";

	/**
	 * A multiple components can be represented by a single XML element
	 *
	 * @return the number of components the elements represents
	 */
	@Override
	public int repetition() {
		return mRootElement.getRepetition();
	}

	@Override
	// TABLE ONLY -- Svante REMOVE THIS LATER AS THIS IT IS ONLY USED BY TABLES
	public List getChildren() {
		return list(this);
	}

	// Svante ToDo: After all the refactoring this looks like something to change after the release as well.
	private List<Component> list(final Table tableComponent) {
		return new AbstractList<Component>() {
			public int size() {
				return tableComponent.size();
			}

			public Component get(int index) {
				Component c = null;
				TableTableElement tableRoot = (TableTableElement) tableComponent.getRootElement();
				OdfTable table = OdfTable.getInstance(tableRoot);
				OdfTableRow row = table.getRowByIndex(index);
				if (row != null) {
					c = row.getOdfElement().getComponent();
				}
				return c;
			}
		};
	}

	/**
	 * Adds the given component to the root element
	 */
	@Override
	public void addChild(int index, Component c) {
		mRootElement.insert(c.getRootElement(), index);
// 2DO: Svante: ARE THE ABOVE AND THE BELOW EQUIVALENT?
//		OdfElement rootElement = c.getRootElement();
//		if (index >= 0) {
//			mRootElement.insertBefore(rootElement, ((OdfElement) mRootElement).receiveNode(index));
//		} else {
//			mRootElement.appendChild(rootElement);
//		}
	}

	/**
	 * @return either a text node of size 1 or an element being the root element
	 * of a component
	 */
	@Override
	public Node getChildNode(int index) {
		return ((OdfElement) mRootElement).receiveNode(index);

	}

	/**
	 * Removes a component from the text element container. Removes either an
	 * element representing a component or text node of size 1
	 */
	@Override
	public Node remove(int index) {
		Node node = (Node) this.getChildNode(index);
		return mRootElement.removeChild(node);
	}

	/**
	 * All children of the root element will be traversed. If it is a text node
	 * the size is added, if it is an element and a component a size of one is
	 * added, if it is a marker, for known text marker elements (text:span,
	 * text:bookmark) the children are recursive checked
	 *
	 * @return the number of child components
	 */
	@Override
	public int size() {
		OdfTable table = OdfTable.getInstance((TableTableElement) mRootElement);
		return table.getRowCount();
	}

	public void addColumns(int firstColumn, int lastColumn) {
	}

	/**
	 * Formats the complete column of a table starting from columnStartNo to
	 * columnEndNo. Counting starts with 0.
	 */
	public void formatColumn(int columnStartNo, int columnEndNo, JSONObject attrs) {
		changeRange(columnStartNo, columnEndNo, 0, -1, TableRangeChange.newFormatation(attrs));
	}

	/**
	 * Formats the complete row of a table starting from rowStartNo to rowEndNo.
	 * Counting starts with 0.
	 */
	public void formatRow(int rowStartNo, int rowEndNo, JSONObject attrs) {
		changeRange(0, -1, rowStartNo, rowEndNo, TableRangeChange.newFormatation(attrs));
	}
	
	/**
	 * Selects a cell range of the given table and applies the given changes.
	 *
	 * @param tableElement the table the cell range will be selected from
	 * @param rowStart the upper delimiter of the cell range. All positions
	 * start counting with zero.
	 * @param rowEnd the lower delimiter of the cell range. Any negative value
	 * represents eternity, ie. to the end of table.
	 * @param columnStartNo the left delimiter of the cell range.
	 * @param rowEnd the right delimiter of the cell range. Any negative value
	 * represents eternity, ie. to the end of table.
	 * @param changes a list of changes to be applied to each row and cells
	 * during traversing the tree. TableChange may define a subset of selection
	 * and the modification (e.g. insert, format, etc.)
	 */
	// ToDo: Currently only a single Change is supported!!
	public static void modifyRange(TableTableElement tableElement, int rowStartNo, int rowEndNo, int columnStartNo, int columnEndNo, List<TableChange> changes) {
		if (tableElement != null) {
			((Table) tableElement.getComponent()).changeRange(columnStartNo, columnEndNo, rowStartNo, rowEndNo, changes.get(0));
		}
	}

	/**
	 * Selects a cell range of the given table and applies the given changes.
	 *
	 * @param tableElement the table the cell range will be selected from
	 * @param rowStart the upper delimiter of the cell range. All positions
	 * start counting with zero.
	 * @param rowEnd the lower delimiter of the cell range. Any negative value
	 * represents eternity, ie. to the end of table.
	 * @param columnStartNo the left delimiter of the cell range.
	 * @param rowEnd the right delimiter of the cell range. Any negative value
	 * represents eternity, ie. to the end of table.
	 * @param changes a list of changes to be applied to each row and cells
	 * during traversing the tree. TableChange may define a subset of selection
	 * and the modification (e.g. insert, format, etc.)
	 */
	public static void formatRange(TableTableElement tableElement, int rowStartNo, int rowEndNo, int columnStartNo, int columnEndNo, JSONObject formatProperties) {
		if (tableElement != null) {
			Table table = (Table) tableElement.getComponent();
			if (table != null) {
				table.changeRange(columnStartNo, columnEndNo, rowStartNo, rowEndNo, TableRangeChange.newFormatation(formatProperties));
			}
		}
	}

	public static void addRow(TableTableElement tableElement, int sheetNo, int rowStart, int rowEnd, int referenceRow) {
		Table tableComponent = (Table) tableElement.getComponent();

		List children = tableComponent.getChildren();
		try {
			int count = 1;
			int existingRows = children.size();
			if (existingRows < rowStart + 1) {
				count += rowStart - existingRows;
			}
			if (rowEnd != -1) {
				count += rowEnd - rowStart;
			}
// ToDo Performance: Add the parent component if you have it!
// ONE Method with rootComponent redircting to the one with Table as parent
			insertRows(tableComponent.getRootComponent(), new JSONArray().put(sheetNo).put(rowStart), null, count, true, referenceRow, false);
		} catch (Exception ex) {
			Logger.getLogger(JsonOperationConsumer.class.getName()).log(Level.SEVERE, null, ex);
		}
	}
	
	public void changeRange(int columnStartNo, int columnEndNo, int rowStartNo, int rowEndNo, TableChange change) {
		// starting with -1 position for row & column as the 0 would already be the first position found
		traverseTable((TableTableElement) this.getRootElement(), this.getRootElement().getFirstChild(), columnStartNo, columnEndNo, -1, rowStartNo, rowEndNo, -1, change);
	}

	private static TableTableColumnElement newColumnElement(OdfFileDom xmlDoc, int repeated) {
		TableTableColumnElement newColumnElement = xmlDoc.newOdfElement(TableTableColumnElement.class);
		if (repeated != 1) {
			newColumnElement.setTableNumberColumnsRepeatedAttribute(repeated);
		}
		return newColumnElement;
	}

	private static TableTableRowElement newRowElement(OdfFileDom xmlDoc, int repeated) {
		TableTableRowElement newRowElement = xmlDoc.newOdfElement(TableTableRowElement.class);
		if (repeated != 1) {
			newRowElement.setTableNumberRowsRepeatedAttribute(repeated);
		}
		return newRowElement;
	}

	private static TableTableCellElement newCellElement(OdfFileDom xmlDoc, int repeated) {
		TableTableCellElement newCellElement = xmlDoc.newOdfElement(TableTableCellElement.class);
		if (repeated != 1) {
			newCellElement.setTableNumberColumnsRepeatedAttribute(repeated);
		}
		return newCellElement;
	}

	/**
	 * Traverse a table/spreadsheet to dispatch each element of the demanded
	 * range to the change function (if necessary expand the table/spreadsheet)
	 */
	private static void traverseTable(TableTableElement tableElement, Node child, int columnStartNo, int columnEndNo, int currentColumnNo, int rowStartNo, int rowEndNo, int currentRowNo, TableChange change) {
		OdfFileDom xmlDoc = null;
		int maxColumnCount = 1;
		while (child != null) {
			Node nextElement = child.getNextSibling();
			// ensure that nextElement is an element
			while (nextElement != null && !(nextElement instanceof OdfElement)) {
				nextElement = nextElement.getNextSibling();
			}
			if (child instanceof Element) {
				boolean isColumnChangeFinished = false;
				int columnsToExpand = 0;
				// applies the action on the column (split if required)
				if (!isColumnChangeFinished) {

					if (child instanceof TableTableColumnElement) {
						if (change.effectsColumns()) {
							currentColumnNo++;
							currentColumnNo = changeWithinRange((TableTableColumnElement) child, (OdfElement) nextElement, columnStartNo, columnEndNo, currentColumnNo, columnStartNo, columnEndNo, change);
							// traverses all column container
						} else {
							currentColumnNo += ((TableTableColumnElement) child).getRepetition();
						}
					} else if (child instanceof TableTableColumnGroupElement
							|| child instanceof TableTableHeaderColumnsElement
							|| child instanceof TableTableColumnsElement) {
						// search for nested column
						traverseTable(tableElement, child.getFirstChild(), columnStartNo, columnEndNo, currentColumnNo, rowStartNo, rowEndNo, 0, change);
					} else if (child instanceof TableTableRowElement // the last column existing column was reached
							|| child instanceof TableTableRowGroupElement
							|| child instanceof TableTableHeaderRowsElement
							|| child instanceof TableTableRowsElement) {
						// if the rows are reached, but not the columnStart, the gap has to be filled
						if (currentColumnNo >= 0 && currentColumnNo < columnEndNo) {
							// EXPAND TABLE COLUMNS
							if (xmlDoc == null) {
								xmlDoc = (OdfFileDom) child.getOwnerDocument();
							}
							// if not even the start position was reached
							if (currentColumnNo + 1 < columnStartNo) {
								TableTableColumnElement columnGap = newColumnElement(xmlDoc, columnStartNo - currentColumnNo - 1);
								tableElement.insertBefore(columnGap, child);
								currentColumnNo = columnStartNo - 1;
							}
							// if the currentposition is inbetween start and end position								
							TableTableColumnElement newColumnElement = newColumnElement(xmlDoc, columnEndNo - currentColumnNo);
							tableElement.insertBefore(newColumnElement, child);
							currentColumnNo++; // moved to the start
							if (change.effectsColumns()) {
								currentColumnNo = changeWithinRange(newColumnElement, (OdfElement) nextElement, columnStartNo, columnEndNo, currentColumnNo, columnStartNo, columnEndNo, change);
							} else {
								currentColumnNo += ((TableTableColumnElement) newColumnElement).getRepetition();
							}
						}
						// no start counting with zero
						maxColumnCount = currentColumnNo + 1;
						isColumnChangeFinished = true;
					}
				}
				if (child instanceof TableTableRowElement || child instanceof TableTableRowGroupElement
						|| child instanceof TableTableHeaderRowsElement
						|| child instanceof TableTableRowsElement) {
					if (child instanceof TableTableRowElement) {
						currentRowNo++;
						if (change.effectsRows() || change.effectsCells()) {
							currentRowNo = changeWithinRange((TableTableRowElement) child, (OdfElement) nextElement, rowStartNo, rowEndNo, currentRowNo, columnStartNo, columnEndNo, change);
						} else {
							if (columnsToExpand > 0) {
								if (xmlDoc == null) {
									xmlDoc = (OdfFileDom) child.getOwnerDocument();
								}
								child.appendChild(newCellElement(xmlDoc, columnsToExpand));
							}
							currentRowNo += ((TableTableRowElement) child).getRepetition();
						}
					} else {// traverses all row container
						traverseTable(tableElement, child.getFirstChild(), columnStartNo, columnEndNo, columnStartNo, rowStartNo, rowEndNo, currentRowNo, change);
					}
				}				
			}
			child = nextElement;
		}
		// if the end of table is reached, but not the rowStart, the gap has to be filled
		if (rowEndNo > 0 && currentRowNo >= 0 && currentRowNo < rowEndNo && (change.effectsRows() || change.effectsCells())) {
			// EXPAND TABLE ROWS
			if (xmlDoc == null) {
				xmlDoc = (OdfFileDom) tableElement.getOwnerDocument();
			}
			// if not even the start position was reached
			TableTableCellElement cellContent;
			if (currentRowNo + 1 < rowStartNo) {
				TableTableRowElement rowGap = newRowElement(xmlDoc, rowStartNo - currentRowNo - 1);
				Component.createComponent(tableElement.getComponent(), rowGap);
				cellContent = newCellElement(xmlDoc, maxColumnCount);
				Component.createComponent(rowGap.getComponent(), cellContent);
				rowGap.appendChild(cellContent);
				tableElement.appendChild(rowGap);
				currentRowNo = rowStartNo - 1;
			}
			// if the currentposition is inbetween start and end position								
			TableTableRowElement newRowElement = newRowElement(xmlDoc, rowEndNo - currentRowNo);
			Component.createComponent(tableElement.getComponent(), newRowElement);
			cellContent = newCellElement(xmlDoc, maxColumnCount);
			Component.createComponent(newRowElement.getComponent(), cellContent);
			newRowElement.appendChild(cellContent);
			tableElement.appendChild(newRowElement);
			currentRowNo++;
			changeWithinRange(newRowElement, null, rowStartNo, rowEndNo, currentRowNo, columnStartNo, columnEndNo, change);
		}
	}

	private static void traverseRow(Component rowComponent, Node child, int rowStartNo, int rowEndNo, int currentRowNo, int columnStartNo, int columnEndNo, int currentColumnNo, TableChange change) {
		OdfElement lastElement = null;
		while (child != null) {
			Node nextElement = child.getNextSibling();
			// get the next element
			while (nextElement != null && !(nextElement instanceof OdfElement)) {
				nextElement = nextElement.getNextSibling();
			}
			if (child instanceof OdfElement) {
				if (child instanceof TableTableCellElement) {
					currentColumnNo++;
					if (change.effectsCells()) {
						currentColumnNo = changeWithinRange((TableTableCellElement) child, (OdfElement) nextElement, columnStartNo, columnEndNo, currentColumnNo, columnStartNo, columnEndNo, change);
					}
				}
				lastElement = (OdfElement) child;
			}
			child = nextElement;
		}
		// if the end of table is reached, but not the cellStart, the gap has to be filled
		if (lastElement != null && currentColumnNo >= 0 && currentColumnNo < columnEndNo) {
			// EXPAND TABLE CELLS
			OdfFileDom xmlDoc = (OdfFileDom) lastElement.getOwnerDocument();
			// if not even the start position was reached
			if (currentColumnNo + 1 < columnStartNo) {
				TableTableCellElement cellGap = newCellElement(xmlDoc, columnStartNo - currentColumnNo - 1);
				lastElement.getParentNode().appendChild(cellGap);
				Component.createComponent(rowComponent, cellGap);
				currentColumnNo = columnStartNo - 1;
			}
			// if the currentposition is inbetween start and end position								
			TableTableCellElement newCellElement = newCellElement(xmlDoc, columnEndNo - currentColumnNo);
			lastElement.getParentNode().appendChild(newCellElement);
			Component.createComponent(rowComponent, newCellElement);
			currentColumnNo++;
			changeWithinRange(newCellElement, null, columnStartNo, columnEndNo, currentColumnNo, columnStartNo, columnEndNo, change);
		}
	}

	/**
	 * Considers if the element is in the given range and applies a split if
	 * required
	 */
	private static int changeWithinRange(OdfStylableElement tableChildElement, OdfElement nextElement, int startNo, int endNo, int currentNo, int columnStartNo, int columnEndNo, TableChange change) {
		int repetition = tableChildElement.getRepetition();
		// REACHED = the start range was reached (or exceeded)..
		if (startNo <= (currentNo + (repetition - 1)) && ((currentNo <= endNo) || endNo < 0))  {
			NamedNodeMap attrs = tableChildElement.getAttributes();
			// expand the current element by repetition if there is a gap between now and the end AND the current element has no attributes (NOR content!)
			if ((endNo - currentNo > 0) && (tableChildElement.getAttributes() == null && attrs == null || attrs.getLength() == 0) && !tableChildElement.hasChildNodes()) {
				if (nextElement == null || (!(nextElement instanceof TableTableColumnElement || nextElement instanceof TableTableColumnGroupElement
						|| nextElement instanceof TableTableHeaderColumnsElement
						|| nextElement instanceof TableTableColumnsElement))) {
					repetition = endNo - currentNo - repetition + 2;
					if (repetition > 1) {
						tableChildElement.setRepetition(repetition);
					}
				}
			}
			// if the start of the (repeating) column is the range start
			if (startNo <= currentNo && (currentNo <= endNo || endNo < 0)) {
				change.execute(tableChildElement);
				if (tableChildElement instanceof TableTableRowElement && change.effectsCells()) {
					traverseRow(((TableTableRowElement) tableChildElement).getComponent(), tableChildElement.getFirstChild(), startNo, endNo, currentNo, columnStartNo, columnEndNo, -1, change);
				}
				// if the repeated exceed the start..
			} else if (currentNo - startNo < 0) {
				// if the desired range does not cover all the repeated columns, we need to split the column
				OdfElement p = (OdfElement) tableChildElement.getParentNode();
				OdfStylableElement secondHalf = (OdfStylableElement) tableChildElement.split(startNo - currentNo);
				if (endNo > 0 && (currentNo + repetition >= endNo)) {
					// use the middle part (this piece starts at startNo and should end at endNo, but is at least 1 long
					secondHalf.split(endNo - startNo + 1);
					change.execute(secondHalf);
					if (secondHalf instanceof TableTableRowElement && change.effectsCells()) {
						traverseRow(((TableTableRowElement) secondHalf).getComponent(), secondHalf.getFirstChild(), startNo, endNo, currentNo, columnStartNo, columnEndNo, -1, change);
					}
				} else {
					// use the complete second part
					change.execute(secondHalf);
					if (secondHalf instanceof TableTableRowElement && change.effectsCells()) {
						traverseRow(((TableTableRowElement) secondHalf).getComponent(), secondHalf.getFirstChild(), startNo, endNo, currentNo, columnStartNo, columnEndNo, -1, change);
					}
				}
				// negative column indicates selection of all columns
			} else if (endNo < 0) {
				change.execute(tableChildElement);
				if (tableChildElement instanceof TableTableRowElement && change.effectsCells()) {
					traverseRow(((TableTableRowElement) tableChildElement).getComponent(), tableChildElement.getFirstChild(), startNo, endNo, currentNo, columnStartNo, columnEndNo, -1, change);
				}
			} else if (currentNo + (repetition - 1) >= endNo) {
				// if the repetition of the element is exceeded the desired range
				if (currentNo + (repetition - 1) != endNo) {
					tableChildElement.split(currentNo + (repetition - 1) - endNo);
				}
				// use the first part
				change.execute(tableChildElement);
				if (tableChildElement instanceof TableTableRowElement && change.effectsCells()) {
					traverseRow(((TableTableRowElement) tableChildElement).getComponent(), tableChildElement.getFirstChild(), startNo, endNo, currentNo, columnStartNo, columnEndNo, -1, change);
				}
			}
		}
		currentNo += repetition - 1;
		// if there had been a repetition add it (over the usual 1)
		return currentNo;
	}
}
