/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.json.actions;

import java.util.Collection;
import java.util.Iterator;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.apache.commons.logging.Log;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.ajax.container.ByteArrayFileHolder;
import com.openexchange.ajax.requesthandler.AJAXActionService;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.exception.OXException;
import com.openexchange.log.LogFactory;
import com.openexchange.realtime.dispatch.MessageDispatcher;
import com.openexchange.realtime.packet.ID;
import com.openexchange.realtime.packet.Message;
import com.openexchange.realtime.packet.Stanza;
import com.openexchange.realtime.payload.PayloadElement;
import com.openexchange.realtime.payload.PayloadTree;
import com.openexchange.realtime.payload.PayloadTreeNode;
import com.openexchange.realtime.util.ActionHandler;
import com.openexchange.realtime.util.ElementPath;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link DocumentFilterAction}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
/**
 * {@link DocumentFilterAction}
 * 
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @author <a href="mailto:malte.timmermann@open-xchange.com">Malte Timmermann</a>
 */
public abstract class DocumentFilterAction implements AJAXActionService {

    private static final Log LOG = LogFactory.getLog(DocumentFilterAction.class);

    static final String m_officeNamespace = "office";

    static final String m_operationsElement = "operations";

    static final String m_errorKey = "hasErrors";

    static final String m_eTagBase = "http://www.open-xchange.com/filestore/";

    /**
     * Initializes a new {@link DocumentFilterAction}.
     * 
     * @param services
     * @param oqm
     */
    public DocumentFilterAction(final ServiceLookup services) {
        this.m_services = services;
    }

    /**
     * @param inputStm
     * @param fileName
     * @param mimeType
     * @return
     * @throws OXException
     */
    static protected AJAXRequestResult createFileRequestResult(AJAXRequestData request, byte[] data, String mimeType, String extension) {
        final String fileId = request.getParameter("id");
        final String fileVersion = request.getParameter("version");
        final ByteArrayFileHolder fileHolder = new ByteArrayFileHolder(data);
        String fileName = request.getParameter("filename");

        if (null == fileName) {
            fileName = "file";
        }

        if (null != extension) {
            fileName = fileName + '.' + extension;
        }

        fileHolder.close();
        fileHolder.setName(fileName);
        fileHolder.setContentType(mimeType);

        AJAXRequestResult requestResult = new AJAXRequestResult(fileHolder, "file");

        setETag(getETag(fileId, fileVersion), 0, requestResult);

        return requestResult;
    }

    /** return JSON which contain information about current user session.
     * 
     *  @note	Retrieving session information is implemented very gracefully.
     *  		It's because the session API seems not well designed and show
     *  		different behaviour on different installations.
     *  		Especially on custom SSO (single sign on) installations the
     *  		source of context information is not (real) clear.
     * 
     * 	@param	serverSession [IN]
     * 			the server session object where we should retrieve all informations from.
     *
     *  @return a JSON object which contain all session data we need for RT.
     */
    static protected JSONObject getJSONObjectFromServerSession(final ServerSession session)
    {
        JSONObject jsonSession = new JSONObject();

        try
        {
            final String sessionId = session.getSessionID();
            final String login     = impl_getLoginFromSession   (session);
            final String context   = impl_getContextFromSession (session);

            Validate.notEmpty(sessionId, "Miss session ID.");
            Validate.notEmpty(login    , "Miss login."     );
            Validate.notEmpty(context  , "Miss context."   );

            final JSONObject jsonSessionDetails
            	= new JSONObject()
            		   .put("protocol" , "ox"           )
            		   .put("component", JSONObject.NULL)
            		   .put("user"     , login          )
            		   .put("context"  , context        )
            		   .put("resource" , sessionId      );

            jsonSession = new JSONObject().put("session", jsonSessionDetails);
        }
        catch (final Throwable ex)
        {
            LOG.error(ex.getMessage(), ex);
        	impl_dumpSession (session);
            throw new RuntimeException ("Not abe to retrieve needed information from session.", ex);
        }

        return jsonSession;
    }

    /// the separator used by session.getLogin() to separate login & context information (as it's documented)
    static private final String LOGIN_CONTEXT_SEPARATOR = "@";
    
    /** extract the login identifier from given session.
     * 
     *  @param	session [IN]
     *  		the session where the login has to be retrieved from.
     *
     *  @return the login identifier.
     *  
     *  @throw  an exception in case login could not be retrieved successfully.
     */
    static protected String impl_getLoginFromSession (final ServerSession session)
    {
        String login = null;

        // TODO find out : is toString(session.getUserId()) the real login ID ?
//        try
//        {
//        	login = Integer.toString(session.getUserId());
//        }
//        catch(final Throwable exIgnore)
//        {};

        if (StringUtils.isEmpty(login))
        {
	        try
	        {
	        	LOG.warn ("Miss login - try fallback 01 instead.");
	        	login = session.getUserlogin();
	        }
	        catch(final Throwable exIgnore)
	        {};
        }

        if (StringUtils.isEmpty(login))
        {
	        try
	        {
	        	LOG.warn ("Miss login - try fallback 02 instead.");
	        	login = StringUtils.substringBefore(session.getLogin(), LOGIN_CONTEXT_SEPARATOR);
	        }
	        catch(final Throwable exIgnore)
	        {};
        }
        
        if (StringUtils.isEmpty(login))
        {
        	impl_dumpSession (session);
        	throw new RuntimeException ("Could not extract login from session.");
        }

        return login;
    }
    
    /** extract the context identifier from given session.
     * 
     *  @param	session [IN]
     *  		the session where the context has to be retrieved from.
     *
     *  @return the context identifier.
     *  
     *  @throw  an exception in case context could not be retrieved successfully.
     */
    static protected String impl_getContextFromSession (final ServerSession session)
    {
        String context = null;

        try
        {
        	context = Integer.toString(session.getContextId());
        }
        catch(final Throwable exIgnore)
        {};

        if (StringUtils.isEmpty(context))
        {
        	LOG.warn ("Miss context - try fallback 01 instead.");
        	try
        	{
        		context = Integer.toString(session.getContext().getContextId());
        	}
        	catch(final Throwable exIgnore)
        	{};
        }

        if (StringUtils.isEmpty(context))
        {
        	LOG.warn ("Miss context - try fallback 02 instead.");
        	try
        	{
        		context = StringUtils.substringAfterLast(session.getLogin(), LOGIN_CONTEXT_SEPARATOR);
        	}
        	catch(final Throwable exIgnore)
        	{};
        }

        if (StringUtils.isEmpty(context))
        {
        	LOG.warn ("Miss context - try fallback 03 instead.");
        	try
        	{
        		context = session.getContext().getName();
        	}
        	catch(final Throwable exIgnore)
        	{};
        }

        if (StringUtils.isEmpty(context))
        {
        	impl_dumpSession (session);
        	context = "defaultcontext"; // THIS is wrong ... but the last-out of previous implementation :-)
//        	throw new RuntimeException ("Could not extract context from session.");
        }
        
        return context;
    }
    
    /** helper for debug purposes.
     *
     *  Dump a given session object to our LOG to analyze
     *  problems on customer installations directly.
     * 
     *  @param	session [IN]
     *  		the session to be analyzed here
     */
    static protected void impl_dumpSession (final ServerSession session)
    {
		final StringBuffer dump = new StringBuffer (256);
    	try
    	{
    		dump.append ("#### =================================================\n");
    		dump.append ("#### session dump :                                   \n");
    	try{dump.append ("#### session      : "+session                       +"\n");}catch(final Throwable ignore){}; // catch errors line by line ! we want to get info's as much as possible :-)
    	try{dump.append ("#### context      : "+session.getContext()          +"\n");}catch(final Throwable ignore){};
    	try{dump.append ("#### user         : "+session.getUser()             +"\n");}catch(final Throwable ignore){};
    	try{dump.append ("#### login        : "+session.getLogin()            +"\n");}catch(final Throwable ignore){};
    	try{dump.append ("#### login name   : "+session.getLoginName()        +"\n");}catch(final Throwable ignore){};
    	try{dump.append ("#### user login   : "+session.getUserlogin()        +"\n");}catch(final Throwable ignore){};
    	try{dump.append ("#### user id      : "+session.getUserId()           +"\n");}catch(final Throwable ignore){};
    	try{dump.append ("#### context id   : "+session.getContextId()        +"\n");}catch(final Throwable ignore){};
    	try{dump.append ("#### context name : "+session.getContext().getName()+"\n");}catch(final Throwable ignore){};
    	try{dump.append ("#### user config  : "+session.getUserConfiguration()+"\n");}catch(final Throwable ignore){};
    		dump.append ("#### =================================================\n");
    	}
    	catch (final Throwable ex)
    	{
    		LOG.error (ex.getMessage(), ex);
    	}
    	LOG.info(dump.toString ());
    }
    
    /**
     * @param request
     * @param session
     * @param methodName
     * @param elementName
     * @param jsonMessageData
     * @return
     */
    protected JSONObject sendConnectionMessageSynchronously(AJAXRequestData request, ServerSession session, String methodName, String elementName, JSONObject messageData) {
        final String fileId = request.getParameter("id");
        final String folderId = request.getParameter("folder_id");
        final JSONObject requestData = getJSONObjectFromServerSession(session);
        JSONObject resultData = null;
        Stanza resultStanza = null;

        if ( ! StringUtils.isEmpty(fileId) && ! StringUtils.isEmpty(folderId)) {
            try {
                final Message message = new Message();
                final String resource = (new StringBuilder(folderId)).append('.').append(fileId).toString();

                message.setTo(new ID(
                    "synthetic",
                    m_officeNamespace,
                    m_operationsElement,
                    impl_getContextFromSession(session),
                    resource));
                message.addPayload(ActionHandler.getMethodCall(methodName));

                if (null != messageData) {
                    for (final Iterator<String> keys = messageData.keys(); keys.hasNext();) {
                        final String curKey = keys.next();
                        requestData.put(curKey, messageData.get(curKey));
                    }
                }

                message.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                    new PayloadElement(requestData, "json", m_officeNamespace, elementName)).build()));

                resultStanza = m_services.getService(MessageDispatcher.class).sendSynchronously(message, 60, TimeUnit.SECONDS);
            } catch (Exception e) {
                //
            }

            // check result of the RT connection call and return appropriately
            if (null != resultStanza) {
                Collection<PayloadTree> payloads = resultStanza.getPayloadTrees(new ElementPath(m_officeNamespace, elementName));

                if (!payloads.isEmpty()) {
                    resultData = (JSONObject) payloads.iterator().next().getRoot().getData();
                }
            }
        }

        return resultData;
    }

    /**
     * @param resultData
     * @return
     */
    static final AJAXRequestResult getAjaxRequestResult(JSONObject resultData) {
        JSONObject ret = resultData;

        if ((null == ret) || ret.isEmpty()) {
            ret = new JSONObject();

            try {
                ret.put(m_errorKey, true);
            } catch (JSONException e) {
                //
            }
        }

        return new AJAXRequestResult(ret);
    }

    /**
     * @param fileId
     * @param fileVersion
     * @return
     */
    static protected String getETag(String fileId, String fileVersion) {
        return (new StringBuilder(m_eTagBase)).append(fileId).append('/').append(fileVersion).append(".pdf").toString();
    }

    /**
     * @param eTag
     * @param expires
     * @param result
     * @throws OXException
     */
    static protected void setETag(String eTag, long expires, AJAXRequestResult result) {
        result.setExpires(expires);

        if (eTag != null) {
            result.setHeader("ETag", eTag);
        }
    }

    /**
     * @param request
     * @param testETag
     * @return
     * @throws OXException
     */
    static protected boolean checkETag(String fileId, String fileVersion, String testETag) {
        return getETag(fileId, fileVersion).equals(testETag);
    }

    // - Members ---------------------------------------------------------------

    protected ServiceLookup m_services = null;
}
