/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.json.actions;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.documentation.RequestMethod;
import com.openexchange.documentation.annotations.Action;
import com.openexchange.documentation.annotations.Parameter;
import com.openexchange.documentconverter.Feature;
import com.openexchange.documentconverter.IManager;
import com.openexchange.documentconverter.Properties;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.DefaultFile;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.File.Field;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.folderstorage.FolderService;
import com.openexchange.office.tools.DocumentFormatHelper;
import com.openexchange.office.tools.DocumentType;
import com.openexchange.office.tools.ErrorCode;
import com.openexchange.office.tools.FileHelper;
import com.openexchange.office.tools.LocalFileMappingManager;
import com.openexchange.office.tools.MimeTypeHelper;
import com.openexchange.office.tools.monitoring.CreateEvent;
import com.openexchange.office.tools.monitoring.DocumentEvent;
import com.openexchange.office.tools.monitoring.ErrorType;
import com.openexchange.office.tools.monitoring.Statistics;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.file.external.QuotaFileStorageExceptionCodes;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.tx.TransactionAwares;

public class CreateFromTemplateAction extends DocumentFilterAction {

    private static final Log LOG = com.openexchange.log.Log.loggerFor(CreateFromTemplateAction.class);

    private LocalFileMappingManager m_localFileMapper = null;

    public CreateFromTemplateAction(ServiceLookup services, LocalFileMappingManager localFileMapper) {
        super(services);
        m_localFileMapper = localFileMapper;
    }

    @Action(method = RequestMethod.GET, name = "createfromtemplate", description = "Request to create a new file based on a specific template.", parameters = {
        @Parameter(name = "session", description = "A session ID previously obtained from the login module."),
        @Parameter(name = "id", description = "The id of the file to be used as a template."),
        @Parameter(name = "folder_id", description = "The folder id of the file to be used as a template."),
        @Parameter(name = "initial_filename", description = "The initial file name of the new file to be created"),
        @Parameter(name = "uid", description = "The unique id of the client application.")
    }, responseDescription = "Response with timestamp: A JSON object containing the id and folder_id of the newly created document.")

    @Override
    public AJAXRequestResult perform(AJAXRequestData request, ServerSession session) throws OXException {
        AJAXRequestResult requestResult = null;
        DocumentEvent documentEvent = null;

        if ((null != request) && (null != session)) {
            final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
            final IDBasedFileAccess fileAccess = (null != fileFactory) ? fileFactory.createAccess(session) : null;
            DocumentType eventDocumentType = DocumentType.NONE;
            InputStream inputStm = null;
            boolean rollback = false;

            try {
                final File file = new DefaultFile();
                final String fileId = request.getParameter("id");
                final String srcFolderId = request.getParameter("folder_id");
                final String folderId = FileHelper.getUserFolderId(m_services.getService(FolderService.class), session);

                if ((null != fileId) && (null != srcFolderId) && (null != fileAccess) && (null != folderId)) {
                    // Create document based on template file. It could be that we have to
                    // convert the template document to the related ODF document format.
                    boolean bLocalFile = fileId.startsWith("template:");
                    final String version = FileStorageFileAccess.CURRENT_VERSION;
                    String fullFilePath = null;
                    String extensionType = null;
                    String mimeType = null;
                    File templateFile = null;

                    if (bLocalFile) {
                        // a local file must be treated differently
                        fullFilePath = m_localFileMapper.getPathFromId(fileId);
                        if (null == fullFilePath) {
                            throw new OXException();
                        } else {
                            String fileName = FileHelper.getFileName(fullFilePath);
                            extensionType = FileHelper.getExtension(fileName);
                            mimeType = MimeTypeHelper.getMimeTypeFromTemplateExtension(extensionType);
                        }
                    } else {
                        templateFile = fileAccess.getFileMetadata(fileId, version);
                        extensionType = FileHelper.getExtension(templateFile.getFileName());
                        mimeType = templateFile.getFileMIMEType();
                    }

                    // we can continue if we have a path or not local file
                    if ((bLocalFile && (null != fullFilePath)) || !bLocalFile) {
                        HashMap<String, String> conversionFormat = DocumentFormatHelper.getConversionFormatInfo(mimeType, extensionType);

                        if (conversionFormat != null) {
                            // conversion already sets the correct target document format
                            final IManager dcManager = m_services.getService(IManager.class);

                            if ((null != dcManager) && dcManager.hasFeature(Feature.DOCUMENTCONVERTER)) {
                                InputStream documentInputStm = null;
                                if (bLocalFile) {
                                    // local file can directly read from the local file system
                                    documentInputStm = FileHelper.getDocumentStreamFromLocalFile(fullFilePath);
                                } else {
                                    // OX Drive file must be loaded using Files API
                                    AJAXRequestData loadRequest = new AJAXRequestData();
                                    loadRequest.putParameter("id", fileId);
                                    loadRequest.putParameter("version", version);
                                    loadRequest.putParameter("fileName", templateFile.getFileName());
                                    documentInputStm = FileHelper.getDocumentStream(m_services, session, loadRequest);
                                }

                                if (null != documentInputStm) {
                                    HashMap<String, Object> jobProperties = new HashMap<String, Object>(4);
                                    HashMap<String, Object> resultProperties = new HashMap<String, Object>(8);
                                    final String filterShortName = conversionFormat.get(Properties.PROP_FILTER_SHORT_NAME);
                                    final String convertedExtensionType = conversionFormat.get(Properties.PROP_INPUT_TYPE);

                                    jobProperties.put(Properties.PROP_INPUT_STREAM, documentInputStm);
                                    jobProperties.put(Properties.PROP_MIME_TYPE, convertedExtensionType);
                                    jobProperties.put(Properties.PROP_FILTER_SHORT_NAME, filterShortName);

                                    inputStm = dcManager.convert(filterShortName, jobProperties, resultProperties);
                                    IOUtils.closeQuietly(documentInputStm);

                                    // set new mime type and extensionType
                                    mimeType = conversionFormat.get(Properties.PROP_MIME_TYPE);
                                    extensionType = conversionFormat.get(Properties.PROP_INPUT_TYPE);
                                }
                            }

                            eventDocumentType = DocumentType.valueOf(conversionFormat.get("DocumentType"));
                        } else {
                            // we have to set the correct target document format (created from the template)
                            HashMap<String, String> targetFormat = DocumentFormatHelper.getTemplateTargetFormatInfo(mimeType, extensionType);
                            if (bLocalFile) {
                                inputStm = FileHelper.getDocumentStreamFromLocalFile(fullFilePath);
                            } else {
                                inputStm = fileAccess.getDocument(fileId, version);
                            }

                            // set new mime type and extensionType for the target document file
                            if (null != targetFormat) {
                                mimeType = targetFormat.get(Properties.PROP_MIME_TYPE);
                                extensionType = targetFormat.get(Properties.PROP_INPUT_TYPE);
                            }
                        }

                        // create a new file store item with default content
                        file.setId(FileStorageFileAccess.NEW);
                        file.setFolderId(folderId);

                        final String initialFileName = request.getParameter("initial_filename");
                        final String createFileName = ((null == initialFileName) || (initialFileName.length() < 1)) ? "unnamed" : initialFileName;
                        file.setFileName(createFileName + "." + extensionType);
                        file.setFileMIMEType(mimeType);

                        if (null != inputStm) {
	                    	if(extensionType.equals("docx")) {
	                    		final InputStream newInputStm = changeDocumentContentType(inputStm, "application/vnd.openxmlformats-officedocument.wordprocessingml.template",
	                    			"application/vnd.openxmlformats-officedocument.wordprocessingml.document");
	                    		IOUtils.closeQuietly(inputStm);
	                    		inputStm = newInputStm;
	                    	}
	                    	else if(extensionType.equals("xlsx")) {
	                    		final InputStream newInputStm = changeDocumentContentType(inputStm, "application/vnd.openxmlformats-officedocument.spreadsheetml.template",
	                    			"application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
	                    		IOUtils.closeQuietly(newInputStm);
	                    		inputStm = newInputStm;
	                    	}
                        }

                        if (null != inputStm) {
                            fileAccess.startTransaction();
                            rollback = true;
                            fileAccess.saveDocument(file, inputStm, FileStorageFileAccess.DISTANT_FUTURE, new ArrayList<Field>());
                            fileAccess.commit();
                            rollback = false;

                            final JSONObject jsonObj = new JSONObject(6);

                            // return actual parameters of new file
                            jsonObj.put("id", file.getId());
                            jsonObj.put("folder_id", file.getFolderId());
                            jsonObj.put("version", file.getVersion());
                            jsonObj.put("filename", file.getFileName());
                            jsonObj.put("file_mimetype", file.getFileMIMEType());
                            jsonObj.put("converted", (conversionFormat != null));

                            requestResult = new AJAXRequestResult(jsonObj);
                            documentEvent = new CreateEvent(eventDocumentType);
                        } else {
                            // error case: we don't have a input stream
                            ErrorCode errorCode = null;
                            if (null != conversionFormat) {
                                errorCode = ErrorCode.CREATEDOCUMENT_CONVERSION_FAILED_ERROR;
                            } else {
                                errorCode = ErrorCode.CREATEDOCUMENT_CANNOT_READ_TEMPLATEFILE_ERROR;
                            }
                            JSONObject jsonObj = errorCode.getAsJSON();
                            requestResult = new AJAXRequestResult(jsonObj);
                            documentEvent = new DocumentEvent(eventDocumentType, ErrorType.CREATE);
                            LOG.warn(errorCode.getDescription());
                        }
                    } else {
                        // no path found for the provided id
                        ErrorCode errorCode = ErrorCode.LOCALFILE_INVALID_ID_ERROR;
                        LOG.error(errorCode.getDescription());
                        try {
                            requestResult = new AJAXRequestResult(createErrorObject(errorCode));
                            documentEvent = new DocumentEvent(eventDocumentType, ErrorType.CREATE);
                        } catch (final JSONException e) {
                            LOG.warn("Couldn't create JSON object while creating new document from template");
                        }
                    }
                } else {
                    // required arguments are null or not provided
                    ErrorCode errorCode = ErrorCode.GENERAL_ARGUMENTS_ERROR;
                    LOG.error(errorCode.getDescription());
                    try {
                        requestResult = new AJAXRequestResult(createErrorObject(errorCode));
                        documentEvent = new DocumentEvent(eventDocumentType, ErrorType.CREATE);
                    } catch (final JSONException e) {
                        LOG.warn("Couldn't create JSON object while creating new document from template");
                    }
                }
            } catch (final Exception e) {
                ErrorCode errorCode = ErrorCode.CREATEDOCUMENT_PERMISSION_CREATEFILE_MISSING_ERROR;
                if (e instanceof OXException) {
                    // special handling for OXExceptions
                    OXException ox = (OXException) e;

                    // special handling for file storage exceptions
                    if (ox.getPrefix().equalsIgnoreCase("FLS")) {
                        if (ox.getCode() == QuotaFileStorageExceptionCodes.STORE_FULL.getNumber()) {
                            // set specific error code if quota reached
                            errorCode = ErrorCode.CREATEDOCUMENT_QUOTA_REACHED_ERROR;
                        }
                    }
                }

                LOG.error(errorCode.getDescription(), e);
                try {
                    requestResult = new AJAXRequestResult(createErrorObject(errorCode));
                    documentEvent = new DocumentEvent(eventDocumentType, ErrorType.CREATE);
                } catch (final JSONException je) {
                    LOG.warn("Couldn't create JSON object while creating new document from template");
                }
            } finally {
                // Roll-back (if needed) and finish
                if (rollback) {
                    TransactionAwares.rollbackSafe(fileAccess);
                }

                TransactionAwares.finishSafe(fileAccess);
                IOUtils.closeQuietly(inputStm);
            }
        }

        // update statistics with a possibly created event
        Statistics.handleDocumentEvent(documentEvent);

        return requestResult;
    }

    private InputStream changeDocumentContentType(InputStream inputStm, String sourceContentType, String destinationContentType) {

		InputStream retStm = null;

		final String warning = "CreateFromTemplateAction::changeContentType, couldn't change content type of the template document";

		final ByteArrayOutputStream outStream = new ByteArrayOutputStream();
        final ZipInputStream zipInputStream = new ZipInputStream(inputStm);
        final ZipOutputStream zipOutputStream = new ZipOutputStream(outStream);
        ZipEntry zipEntry = null;
        final byte[] buffer = new byte[0x10000];  // 64kb buffer

        try {
			while((zipEntry = zipInputStream.getNextEntry())!=null) {
		        final ZipEntry newEntry = new ZipEntry(zipEntry.getName());
		        zipOutputStream.putNextEntry(newEntry);

		        int read;

		        if(zipEntry.getName().equals("[Content_Types].xml")) {
		        	StringBuffer stringBuffer = new StringBuffer();
		        	while((read = zipInputStream.read(buffer))>=0) {
		        		stringBuffer.append(new String(buffer, 0, read, "UTF-8"));
		        	}
		        	int sourceIndex = stringBuffer.indexOf(sourceContentType);
		        	if(sourceIndex!=-1) {
		        		stringBuffer.replace(sourceIndex, sourceIndex + sourceContentType.length(), destinationContentType);
		        	}
		        	zipOutputStream.write(stringBuffer.toString().getBytes());
		        }
		        else {
			        while ((read = zipInputStream.read(buffer))>=0){
			            zipOutputStream.write(buffer,0,read);
			        }
		        }
		        zipOutputStream.closeEntry();
			    zipInputStream.closeEntry();
			}
		} catch (IOException e) {
            LOG.warn(warning);
		} finally {
			try {
				zipInputStream.close();
			} catch (IOException e) {
	            LOG.warn(warning);
			}
			try {
				zipOutputStream.close();
			} catch (IOException e) {
	            LOG.warn(warning);
			}
			retStm = new ByteArrayInputStream(outStream.toByteArray());
			IOUtils.closeQuietly(outStream);
		}
		return retStm;
	}

    
    
    /**
     * Create the necessary JSON error object in case of an error.
     *
     * @param errorCode
     *  A valid ErrorCode.
     *
     * @return
     *  A JSON object containing the error information to be sent to the
     *  client.
     *
     * @throws JSONException
     */
    private JSONObject createErrorObject(ErrorCode errorCode) throws JSONException {
        final JSONObject jsonObj = new JSONObject(3);
        jsonObj.put("errorCode", errorCode.getCode());
        jsonObj.put("error", errorCode.getCodeAsStringConstant());
        jsonObj.put("errorDescription", errorCode.getDescription());
        return jsonObj;
    }
}
