/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.json.actions;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.documentconverter.Feature;
import com.openexchange.documentconverter.IManager;
import com.openexchange.documentconverter.Properties;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.DefaultFile;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.File.Field;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.folderstorage.FolderService;
import com.openexchange.groupware.ldap.User;
import com.openexchange.office.DocumentProperties;
import com.openexchange.office.IImporter;
import com.openexchange.office.IResourceProvider;
import com.openexchange.office.tools.DocumentFormatHelper;
import com.openexchange.office.tools.DocumentFormatHelper.OfficeDocumentType;
import com.openexchange.office.tools.DocumentType;
import com.openexchange.office.tools.ErrorCode;
import com.openexchange.office.tools.FileHelper;
import com.openexchange.office.tools.MimeTypeHelper;
import com.openexchange.office.tools.monitoring.CreateEvent;
import com.openexchange.office.tools.monitoring.DocumentEvent;
import com.openexchange.office.tools.monitoring.ErrorType;
import com.openexchange.office.tools.monitoring.Statistics;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.file.external.QuotaFileStorageExceptionCodes;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.tx.TransactionAwares;

/**
 * {@link CreateDefaultDocumentAction}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */

/*
 * MH/KA comment this Action to make it compile
 * @Action(method = RequestMethod.GET, name = "createdefaultdocument", description =
 * "Create a default (empty) OOXML document based on given document type.", parameters = {
 * @Parameter(name = "folder_id", description = "Folder ID of the requested infoitem."),
 * @Parameter(name = "session", description = "A session ID previously obtained from the login module."),
 * @Parameter(name = "uid", description = "The unique id of the client application."),
 * @Parameter(name = "document_type", optional=true, description =
 * "Specifies the type of default document to be created. Possible values are: 'text' (default), 'spreadsheet' and 'presentation' ."),
 * responseDescription =
 * @Parameter(name = "as_template", optional = true, description = "Specifies that a template document should stay a template format, even if conversion is needed.") , },
 * "Response with timestamp: A JSON object containing the attributes of the created filestore item: 'id', 'folder_id', 'version' and 'filename'."
 * )
 */

public class CreateDefaultDocumentAction extends DocumentFilterAction {

    private static final Log LOG = com.openexchange.log.Log.loggerFor(CreateDefaultDocumentAction.class);

    /**
     * Initializes a new {@link CreateDefaultDocumentAction}.
     *
     * @param services
     * @param oqm
     */
    public CreateDefaultDocumentAction(ServiceLookup services) {

        super(services);
    }

    @Override
    public AJAXRequestResult perform(AJAXRequestData request, ServerSession session) {
        AJAXRequestResult requestResult = null;
        DocumentEvent documentEvent = null;

        if ((null != request) && (null != session)) {
            DocumentType eventDocumentType = DocumentType.NONE;
            IDBasedFileAccess fileAccess = null;
            InputStream inputStm = null;
            boolean rollback = false;

            try {
                final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
                fileAccess = fileFactory.createAccess(session);

                // default type for new documents is OOXML
                if (null != fileAccess) {
                    final File file = new DefaultFile();
                    final String docTypeName = request.getParameter("document_type").toLowerCase();
                    final String fileId = request.getParameter("template_id");
                    String folderId = request.getParameter("folder_id");
                    final String templateAccessId = FileHelper.getFileAccessId(fileId, folderId);
                    final String versionStr = request.getParameter("version");
                    final boolean preserveFileName = Boolean.parseBoolean(request.getParameter("preserve_filename"));
                    final boolean asTemplate = Boolean.parseBoolean(request.getParameter("as_template"));
                    String version = (null != versionStr) ? versionStr : "";

                    // determine requested document type and corresponding extensionType
                    OfficeDocumentType officeDocumentType = OfficeDocumentType.NONE;
                    String module;
                    String extensionType;
                    String mimeType;
                    HashMap<String, String> conversionFormat = null;

                    if (fileId == null) {
                        // create new document based on stored default document
                        if (docTypeName.equals("spreadsheet")) {
                            module = "spreadsheet";
                            officeDocumentType = OfficeDocumentType.MS_DOCUMENT_SPREADSHEET;
                            extensionType = "xlsx";
                            mimeType = MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET;
                            eventDocumentType = DocumentType.SPREADSHEET;
                        } else if (docTypeName.equals("presentation")) {
                            module = "presentation";
                            officeDocumentType = OfficeDocumentType.MS_DOCUMENT_PRESENTATION;
                            extensionType = "pptx";
                            mimeType = MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION;
                            eventDocumentType = DocumentType.PRESENTATION;
                        } else {
                            module = "text";
                            officeDocumentType = OfficeDocumentType.MS_DOCUMENT_WORD;
                            extensionType = "docx";
                            mimeType = MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING;
                            eventDocumentType = DocumentType.TEXT;
                        }

                        final IResourceProvider resourceProvider = m_services.getService(com.openexchange.office.IResourceProvider.class);
                        InputStream resourceInputStream = null;

                        // try to get default document from resource first
                        if (null != resourceProvider) {
                            // !!! resource leak warning has been checked:
                            // if resourceInputStream is returned unchanged as the result
                            // InputStream, it will be closed by the caller; otherwise it will
                            // be closed instantly after retrieving the real result InputStream from the
                            // resourceInputStream after calling importer.getDefaultDocument(...) !!!
                            resourceInputStream = resourceProvider.getResource(resourceProvider.getEntry(module, "template", "Default"));
                        }

                        // let the backend create the default document, if no default resource is available
                        IImporter importer = null;

                        switch (officeDocumentType) {
                        case MS_DOCUMENT_WORD: {
                            importer = m_services.getService(com.openexchange.office.ooxml.docx.Importer.class);
                            break;
                        }

                        case MS_DOCUMENT_SPREADSHEET: {
                            importer = m_services.getService(com.openexchange.office.ooxml.xlsx.Importer.class);
                            break;
                        }

                        case MS_DOCUMENT_PRESENTATION: {
                            importer = m_services.getService(com.openexchange.office.ooxml.pptx.Importer.class);
                            break;
                        }

                        default: {
                            break;
                        }
                        }

                        if (null != importer) {
                            final DocumentProperties docProps = new DocumentProperties();
                            final User user = session.getUser();

                            if (null != user) {
                                docProps.put(DocumentProperties.PROP_CREATOR, user.getDisplayName());
                                final String initialSheetname = request.getParameter("initial_sheetname");
                                if (initialSheetname != null) {
                                    docProps.put(DocumentProperties.PROP_INITIAL_SHEETNAME, initialSheetname);
                                }
                            }
                            inputStm = importer.getDefaultDocument(module, resourceInputStream, docProps);
                        }

                        if (resourceInputStream != inputStm) {
                            // Close resource input stream if default document stream
                            // was created by the import getDefaultDocument function. Don't
                            // do that if inputStm refers to resourceInputStream
                            IOUtils.closeQuietly(resourceInputStream);
                        }
                    } else {
                        // Create document based on template file. It could be that we have to
                        // convert the template document to the related ODF document format.
                        if (version.length() == 0) {
                            version = FileStorageFileAccess.CURRENT_VERSION;
                        }

                        final File templateFile = fileAccess.getFileMetadata(templateAccessId, version);

                        extensionType = FileHelper.getExtension(templateFile.getFileName());
                        mimeType = templateFile.getFileMIMEType();
                        conversionFormat = DocumentFormatHelper.getConversionFormatInfo(mimeType, extensionType);

                        if (conversionFormat != null) {
                            final IManager dcManager = m_services.getService(IManager.class);

                            if ((null != dcManager) && dcManager.hasFeature(Feature.DOCUMENTCONVERTER)) {
                                AJAXRequestData loadRequest = new AJAXRequestData();
                                loadRequest.putParameter("id", fileId);
                                loadRequest.putParameter("folder_id", folderId);
                                loadRequest.putParameter("version", version);
                                loadRequest.putParameter("fileName", templateFile.getFileName());

                                final InputStream documentInputStm = FileHelper.getDocumentStream(m_services, session, loadRequest);

                                if (null != documentInputStm) {
                                    HashMap<String, Object> jobProperties = new HashMap<String, Object>(4);
                                    HashMap<String, Object> resultProperties = new HashMap<String, Object>(8);
                                    final String filterShortName = conversionFormat.get(Properties.PROP_FILTER_SHORT_NAME);
                                    final String convertedExtensionType = conversionFormat.get(Properties.PROP_INPUT_TYPE);

                                    jobProperties.put(Properties.PROP_INPUT_STREAM, documentInputStm);
                                    jobProperties.put(Properties.PROP_MIME_TYPE, convertedExtensionType);
                                    jobProperties.put(Properties.PROP_FILTER_SHORT_NAME, filterShortName);

                                    inputStm = dcManager.convert(filterShortName, jobProperties, resultProperties);
                                    IOUtils.closeQuietly(documentInputStm);

                                    // set new mime type and extensionType
                                    mimeType = conversionFormat.get(Properties.PROP_MIME_TYPE);
                                    extensionType = conversionFormat.get(Properties.PROP_INPUT_TYPE);

                                    if (asTemplate) {
                                        // target format should be a template => retrieve mime & extension type
                                        HashMap<String, String> templateFormatInfo = DocumentFormatHelper.getTemplateFormatInfoForDocument(mimeType, convertedExtensionType);
                                        if (null != templateFormatInfo) {
                                            mimeType = templateFormatInfo.get(Properties.PROP_MIME_TYPE);
                                            extensionType = templateFormatInfo.get(Properties.PROP_INPUT_TYPE);
                                        }
                                    }
                                }
                            }

                            eventDocumentType = DocumentType.valueOf(conversionFormat.get("DocumentType"));
                        } else {
                            inputStm = fileAccess.getDocument(templateAccessId, version);
                        }
                    }

                    boolean canWriteToFolder = FileHelper.folderHasWriteAccess(fileAccess, folderId);
                    boolean canCreateFilesInFolder = FileHelper.folderHasCreateAccess(fileAccess, folderId);
                    if ((!canWriteToFolder || !canCreateFilesInFolder) && !preserveFileName) {
                        // We cannot create files/write to the folder which contains our source template document (Edit as New).
                        // Create the document in the user's default folder instead.
                        folderId = FileHelper.getUserFolderId(m_services.getService(FolderService.class), session);
                    }

                    // create a new file store item with default content
                    file.setId(FileStorageFileAccess.NEW);
                    file.setFolderId(folderId);

                    if (!preserveFileName) {
                        final String initialFileName = request.getParameter("initial_filename");
                        final String createFileName = ((null == initialFileName) || (initialFileName.length() < 1)) ? "unnamed" : initialFileName;
                        file.setFileName(createFileName + "." + extensionType);
                    } else {
                        final File templateFile = fileAccess.getFileMetadata(templateAccessId, version);
                        file.setFileName(FileHelper.getBaseName(templateFile.getFileName()) + "." + extensionType);
                    }

                    file.setFileMIMEType(mimeType);

                    if (null != inputStm) {
                        fileAccess.startTransaction();
                        rollback = true;
                        fileAccess.saveDocument(file, inputStm, FileStorageFileAccess.DISTANT_FUTURE, new ArrayList<Field>());
                        fileAccess.commit();
                        rollback = false;

                        final JSONObject jsonObj = new JSONObject(6);

                        // return actual parameters of new file
                        jsonObj.put("id", file.getId());
                        jsonObj.put("folder_id", file.getFolderId());
                        jsonObj.put("version", file.getVersion());
                        jsonObj.put("filename", file.getFileName());
                        jsonObj.put("file_mimetype", file.getFileMIMEType());

                        requestResult = new AJAXRequestResult(jsonObj);
                        documentEvent = new CreateEvent(eventDocumentType);
                    } else {
                        // error case: we don't have a input stream
                        ErrorCode errorCode = null;
                        if (null != conversionFormat) {
                            errorCode = ErrorCode.CREATEDOCUMENT_CONVERSION_FAILED_ERROR;
                        } else if (null != fileId) {
                            errorCode = ErrorCode.CREATEDOCUMENT_CANNOT_READ_TEMPLATEFILE_ERROR;
                        } else {
                            errorCode = ErrorCode.CREATEDOCUMENT_CANNOT_READ_DEFAULTTEMPLATEFILE_ERROR;
                        }
                        JSONObject jsonObj = errorCode.getAsJSON();
                        requestResult = new AJAXRequestResult(jsonObj);
                        documentEvent = new DocumentEvent(eventDocumentType, ErrorType.CREATE);
                        LOG.warn(errorCode.getDescription());
                    }
                }
            } catch (final Exception e) {
                ErrorCode errorCode = ErrorCode.CREATEDOCUMENT_PERMISSION_CREATEFILE_MISSING_ERROR;

                if (e instanceof OXException) {
                    // special handling for OXExceptions
                    OXException ox = (OXException) e;

                    // special handling for file storage exceptions
                    if (ox.getPrefix().equalsIgnoreCase("FLS")) {
                        if (ox.getCode() == QuotaFileStorageExceptionCodes.STORE_FULL.getNumber()) {
                            // set specific error code if quota reached
                            errorCode = ErrorCode.CREATEDOCUMENT_QUOTA_REACHED_ERROR;
                        }
                    }
                }

                LOG.error(errorCode.getDescription(), e);
                try {
                    final JSONObject jsonObj = new JSONObject(3);
                    jsonObj.put("errorCode", errorCode.getCode());
                    jsonObj.put("error", errorCode.getCodeAsStringConstant());
                    jsonObj.put("errorDescription", errorCode.getDescription());
                    requestResult = new AJAXRequestResult(jsonObj);
                    documentEvent = new DocumentEvent(eventDocumentType, ErrorType.CREATE);
                } catch (final JSONException je) {
                    LOG.warn("Couldn't create JSON object while creating new document");
                }
            } finally {
                // Roll-back (if needed) and finish
                if (rollback) {
                    TransactionAwares.rollbackSafe(fileAccess);
                }

                TransactionAwares.finishSafe(fileAccess);
                IOUtils.closeQuietly(inputStm);
            }
        }

        // update statistics with a possibly created event
        Statistics.handleDocumentEvent(documentEvent);

        return requestResult;
    }
}
