/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.calcengine.client.impl;

import java.io.IOException;
import java.io.OutputStream;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.slf4j.Logger;

import com.openexchange.office.tools.logging.ELogLevel;
import com.openexchange.office.tools.logging.LogFactory;

//=============================================================================
/** bridge to pipe streams like stdout/stderr into our logging environment
*
*  <p>
*  It buffer all incoming bytes till a line feed will be detected.
*  Those 'line' will be put into the log and the buffer will be reseted.
*  Default level for logging is INFO.
*  </p>
*  
*  <p>
*  As an extra feature we (try to) detect stacktraces and put them
*  as one log message to the underlying logging framework.
*  Note: this ignore the log level of this bridge ...
*  Even if Level!=SEVERE an incoming exception will be logged as SEVERE.
*  </p>
*/
public class CalcEngineWorkerLogSink extends OutputStream
{
	//-------------------------------------------------------------------------
	private static final Logger LOG = LogFactory.getSlfLog(CalcEngineWorkerLogSink.class);
	
	//-------------------------------------------------------------------------
	private CalcEngineWorkerLogSink ()
	{}

	//-------------------------------------------------------------------------
	public static CalcEngineWorkerLogSink create (final ELogLevel eLevel)
		throws Exception
	{
		// aJclLog is optional !
		Validate.notNull(eLevel, "Invalid argument 'level'.");
		
		final CalcEngineWorkerLogSink aLogSink = new CalcEngineWorkerLogSink ();
		aLogSink.m_eLevel = eLevel;

		return aLogSink;
	}

	//-------------------------------------------------------------------------
	@Override
	public synchronized void write(int nByte)
		throws IOException
	{
		try
		{
			final StringBuffer aBuffer = mem_LineBuffer ();
			final char aByte = (char)nByte;

			if (aByte != '\n')
			{
				aBuffer.append ((char)aByte);
				return;
			}
				
			final String sLastLine = aBuffer.toString ( );
			                         aBuffer.setLength(0);

			// a) did we detect the start of a stacktrace ?
			if (StringUtils.containsIgnoreCase(sLastLine, "exception"))
			{
				if (m_aExceptionBuffer == null)
					m_aExceptionBuffer = new StringBuffer (256);
				m_aExceptionBuffer.append(sLastLine+"\n");
			}
			else
		    // b) are we in 'stacktrace mode' ?
			if (m_aExceptionBuffer != null)
			{
				// b1) is it one line of a stacktrace
				if (StringUtils.contains(sLastLine, "\tat "))
				{
					m_aExceptionBuffer.append(sLastLine+"\n");
				}
				// b2) or is current line 'after' the stacktrace
				else
				{
					// flush the stacktrace
					// Note: Exceptions are errors (always) .-)
					impl_log (ELogLevel.E_ERROR, m_aExceptionBuffer.toString (), null);
					m_aExceptionBuffer= null;

					// don't forget the current line not part of the stacktrace before
					impl_log (m_eLevel, sLastLine, null);
				}
			}
			// c) or is it a 'normal' line ?
			else
				impl_log (m_eLevel, sLastLine, null);
		}
		catch(Throwable ex)
		{
			impl_log (m_eLevel, ex.getMessage(), ex);
		}
	}

	//-------------------------------------------------------------------------
	private void impl_log (final ELogLevel eLevel  ,
						   final String    sMessage,
						   final Throwable aEx     )
	{
		String sNormalizedMessage = sMessage;
		if ( ! StringUtils.endsWith(sNormalizedMessage, "\n"))
			sNormalizedMessage += "\n";
		
		if (eLevel.equals(ELogLevel.E_ERROR))
			LOG.error(sNormalizedMessage, aEx);
		else
		if (eLevel.equals(ELogLevel.E_WARNING))
			LOG.warn(sNormalizedMessage, aEx);
		else
		if (eLevel.equals(ELogLevel.E_INFO))
			LOG.info(sNormalizedMessage, aEx);
		else
		if (eLevel.equals(ELogLevel.E_DEBUG))
			LOG.debug(sNormalizedMessage, aEx);
		else
		if (eLevel.equals(ELogLevel.E_TRACE))
			LOG.trace(sNormalizedMessage, aEx);
		else
			throw new UnsupportedOperationException ("No support implemented yet for '"+eLevel+"'.");
	}

	//-------------------------------------------------------------------------
	private StringBuffer mem_LineBuffer ()
	    throws Exception
	{
		if (m_aLineBuffer == null)
			m_aLineBuffer = new StringBuffer (256);
		return m_aLineBuffer;
	}

	//-------------------------------------------------------------------------
	private ELogLevel m_eLevel = null;
	
	//-------------------------------------------------------------------------
	private StringBuffer m_aLineBuffer = null;

	//-------------------------------------------------------------------------
	private StringBuffer m_aExceptionBuffer = null;
}
