/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.calcengine.client.impl;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.apache.commons.logging.Log;

import com.openexchange.office.calcengine.CalcEngineConst;
import com.openexchange.office.calcengine.client.CalcEngineClipBoardEvent;
import com.openexchange.office.calcengine.client.CalcEngineConfig;
import com.openexchange.office.calcengine.client.CalcEngineDescriptor;
import com.openexchange.office.calcengine.client.CalcEngineLoadBalancer;
import com.openexchange.office.calcengine.client.ECalcEngineError;
import com.openexchange.office.calcengine.client.ICalcEngineClient;
import com.openexchange.office.tools.jobpool.IJob;
import com.openexchange.office.tools.jobpool.IJobPool;
import com.openexchange.office.tools.jobpool.JobArguments;
import com.openexchange.office.tools.jobpool.JobEnvironment;
import com.openexchange.office.tools.jobpool.JobPoolConfig;
import com.openexchange.office.tools.jobpool.JobPoolManager;
import com.openexchange.office.tools.jobpool.JobResults;
import com.openexchange.office.tools.logging.ELogLevel;
import com.openexchange.office.tools.logging.LogFactory;

//=============================================================================
/** Run the calc engine in 'local server' mode.
 */
public class CalcEngineClientLocal implements ICalcEngineClient
{
    //-------------------------------------------------------------------------
    private static final Log LOG = LogFactory.getJclLog(CalcEngineClientLocal.class);

    //-------------------------------------------------------------------------
    private static final boolean ENABLE_JOB_EXCHANGE = false;

    //-------------------------------------------------------------------------
    private CalcEngineClientLocal ()
        throws Exception
    {}
    
    //-------------------------------------------------------------------------
    public synchronized static CalcEngineClientLocal create()
        throws Exception
    {
    	if (m_gSingleton == null)
    		m_gSingleton = new CalcEngineClientLocal ();
    	return m_gSingleton;
    }

    //-------------------------------------------------------------------------
    public synchronized void describeEnvironment (final CalcEngineDescriptor aDesc)
        throws Exception
    {
    	m_aDescriptor = aDesc;
    }
    
    //-------------------------------------------------------------------------
    @Override
    public synchronized String createDocument()
        throws Exception
    {
    	final IJob   iJob    = impl_createStickyJob ();
    	      String sHandle = null;
    	
    	try
    	{
        	final JobArguments lArgs = JobArguments.EMPTY_ARGUMENTS();
        	lArgs.set(CalcEngineJob.ARG_OPERATION, CalcEngineJob.OP_CREATE_DOCUMENT);

	    	final JobResults lResults = iJob.executeSynchronous(lArgs);
	    	if ( ! lResults.wasSuccessfully())
	    		throw new Exception ("Creating new document was not successfully.");
	    	
	    	sHandle = lResults.get(CalcEngineJob.RESULT_DOCUMENT_HANDLE);
	    	Validate.notEmpty(sHandle, "Unexpected and missing doc handle.");
	
	    	impl_cacheStickyJob (sHandle, iJob);
	    	return sHandle;
    	}
    	catch (Throwable ex)
    	{
    		impl_freeStickyJob(sHandle, iJob);
    		throw new Exception(ex);
    	}
    };
    
    //-------------------------------------------------------------------------
    @Override
    public synchronized void destroyDocument(final String sHandle)
        throws Exception
    {
    	IJob iJob = null;
    	
    	try
    	{
	    	final JobArguments lArgs = JobArguments.EMPTY_ARGUMENTS();
	    	
	    	lArgs.set(CalcEngineJob.ARG_OPERATION      , CalcEngineJob.OP_DESTROY_DOCUMENT);
	    	lArgs.set(CalcEngineJob.ARG_DOCUMENT_HANDLE, sHandle                          );
	
            				 iJob     = impl_getStickyJob (sHandle);
	    	final JobResults lResults = iJob.executeSynchronous(lArgs);
	    	
	    	impl_countDocumentsAndExchangeChannelIfTresholdWasReached (sHandle);

	    	if ( ! lResults.wasSuccessfully())
	    		throw new Exception ("Destroy of document '"+sHandle+"' was not successfully.");
    	}
    	finally
    	{
	    	impl_freeStickyJob(sHandle, iJob);
    	}
    };

    //-------------------------------------------------------------------------
    @Override
    public synchronized ECalcEngineError executeOperation(final String       sHandle       ,
    													  final String       sOperationJSON,
    													  final StringBuffer sResultJSON   )
        throws Exception
    {
    	IJob iJob = null;

    	try
    	{
	    	final JobArguments lArgs = JobArguments.EMPTY_ARGUMENTS();
	    	
	    	lArgs.set(CalcEngineJob.ARG_OPERATION      , CalcEngineJob.OP_EXECUTE_OPERATION);
	    	lArgs.set(CalcEngineJob.ARG_DOCUMENT_HANDLE, sHandle                           );
	    	lArgs.set(CalcEngineJob.ARG_OPERATIONLIST  , sOperationJSON                    );
	
                             iJob     = impl_getStickyJob (sHandle);
	    	final JobResults lResults = iJob.executeSynchronous(lArgs);
	    	
			impl_countRequestAndExchangeChannelIfTresholdWasReached (sHandle);

	    	if ( ! lResults.wasSuccessfully())
	    		throw new Exception ("Execution of operations on document '"+sHandle+"' was not successfully.");
	    
	    	final ECalcEngineError eError = lResults.get(CalcEngineJob.RESULT_ERROR_STATE);
	    	if (eError != ECalcEngineError.E_NONE)
	    		return eError;
	    	
	    	final String sResult = lResults.get(CalcEngineJob.RESULT_4_EXECUTION);
	    	sResultJSON.setLength(0);
	    	sResultJSON.append   (sResult);

	    	return ECalcEngineError.E_NONE;
		}
		catch (Throwable ex)
		{
			impl_freeStickyJob(sHandle, iJob);
			throw new Exception(ex);
		}
    };

    //-------------------------------------------------------------------------
    @Override
    public synchronized ECalcEngineError restoreDocument (final String    sHandle             ,
			 				                              final String... lPagedOperationJSONs)
        throws Exception
    {
    	IJob iJob = null;

    	try
    	{
	    	final JobArguments lArgs = JobArguments.EMPTY_ARGUMENTS();
	    	
	    	lArgs.set(CalcEngineJob.ARG_OPERATION      , CalcEngineJob.OP_RESTORE_DOCUMENT);
	    	lArgs.set(CalcEngineJob.ARG_DOCUMENT_HANDLE, sHandle                          );
	    	lArgs.set(CalcEngineJob.ARG_OPERATIONLIST  , lPagedOperationJSONs             );
	
                             iJob     = impl_getStickyJob (sHandle);
	    	final JobResults lResults = iJob.executeSynchronous(lArgs);
	    	
			impl_countRequestAndExchangeChannelIfTresholdWasReached (sHandle);
			
	    	if ( ! lResults.wasSuccessfully())
	    		throw new Exception ("Restore of document '"+sHandle+"' was not successfully.");

	    	final ECalcEngineError eError = lResults.get(CalcEngineJob.RESULT_ERROR_STATE);
	    	return eError;
		}
		catch (Throwable ex)
		{
			impl_freeStickyJob(sHandle, iJob);
			throw new Exception(ex);
		}
    }

    //-------------------------------------------------------------------------
	@Override
	public ECalcEngineError copy(final String                   sSourceDocHandle,
								 final CalcEngineClipBoardEvent aEvent          )
		throws Exception
	{
    	IJob iJob = null;

    	try
    	{
	    	final JobArguments lArgs = JobArguments.EMPTY_ARGUMENTS();
	    	
	    	lArgs.set(CalcEngineJob.ARG_OPERATION      , CalcEngineJob.OP_COPY);
	    	lArgs.set(CalcEngineJob.ARG_DOCUMENT_HANDLE, sSourceDocHandle     );
	    	lArgs.set(CalcEngineJob.ARG_CLIPBOARD_EVENT, aEvent               );
	
                             iJob     = impl_getStickyJob (sSourceDocHandle);
	    	final JobResults lResults = iJob.executeSynchronous(lArgs);

	    	impl_countRequestAndExchangeChannelIfTresholdWasReached (sSourceDocHandle);
	    	
	    	if ( ! lResults.wasSuccessfully())
	    		throw new Exception ("Clipboard copy event on document '"+sSourceDocHandle+"' was not successfully.");
	    
	    	final ECalcEngineError eError = lResults.get(CalcEngineJob.RESULT_ERROR_STATE);
	    	if (eError != ECalcEngineError.E_NONE)
	    		return eError;
	    	
	    	// transfer results to OUT param !
	    	final CalcEngineClipBoardEvent aResult = lResults.get(CalcEngineJob.RESULT_CLIPBOARD_EVENT);
	    	aEvent.takeOver(aResult);

	    	return ECalcEngineError.E_NONE;
		}
		catch (Throwable ex)
		{
			impl_freeStickyJob(sSourceDocHandle, iJob);
			throw new Exception(ex);
		}
	}

    //-------------------------------------------------------------------------
	@Override
	public ECalcEngineError paste(final String                   sTargetDocHandle,
								  final CalcEngineClipBoardEvent aEvent          )
		throws Exception
	{
    	IJob iJob = null;

    	try
    	{
	    	final JobArguments lArgs = JobArguments.EMPTY_ARGUMENTS();
	    	
	    	lArgs.set(CalcEngineJob.ARG_OPERATION      , CalcEngineJob.OP_PASTE);
	    	lArgs.set(CalcEngineJob.ARG_DOCUMENT_HANDLE, sTargetDocHandle      );
	    	lArgs.set(CalcEngineJob.ARG_CLIPBOARD_EVENT, aEvent                );
	
                             iJob     = impl_getStickyJob (sTargetDocHandle);
	    	final JobResults lResults = iJob.executeSynchronous(lArgs);
	    	
			impl_countRequestAndExchangeChannelIfTresholdWasReached (sTargetDocHandle);
			
	    	if ( ! lResults.wasSuccessfully())
	    		throw new Exception ("Clipboard paste event on document '"+sTargetDocHandle+"' was not successfully.");
	    
	    	final ECalcEngineError eError = lResults.get(CalcEngineJob.RESULT_ERROR_STATE);
	    	if (eError != ECalcEngineError.E_NONE)
	    		return eError;

	    	// transfer results to OUT param !
	    	final CalcEngineClipBoardEvent aResult = lResults.get(CalcEngineJob.RESULT_CLIPBOARD_EVENT);
	    	aEvent.takeOver(aResult);

	    	return ECalcEngineError.E_NONE;
		}
		catch (Throwable ex)
		{
			impl_freeStickyJob(sTargetDocHandle, iJob);
			throw new Exception(ex);
		}
	}

    //-------------------------------------------------------------------------
    @Override
    public String getVersion()
        throws Exception
    {
        return "";
    }

    //-------------------------------------------------------------------------
    @Override
    public void setLogLevel (final ELogLevel eLevel) 
    	throws Exception
    {
    	throw new UnsupportedOperationException ("not useful within the LOCAL context ! WE define right log level internal !");
    }

    //-------------------------------------------------------------------------
    private void impl_countRequestAndExchangeChannelIfTresholdWasReached (final String sDocHandle)
    	throws Exception
    {
    	final CalcEngineLoadBalancer aLB      = mem_LoadBalancer ();
    	final String                 sChannel = aLB.getChannelForItem (sDocHandle);
    	
    	aLB.countRequest(sChannel, 1);

    	impl_checkForDirtyJobAndExchangeIt (sChannel);
    }
    
    //-------------------------------------------------------------------------
    private void impl_countDocumentsAndExchangeChannelIfTresholdWasReached (final String sDocHandle)
    	throws Exception
    {
    	final CalcEngineLoadBalancer aLB      = mem_LoadBalancer ();
    	final String                 sChannel = aLB.getChannelForItem (sDocHandle);
    	
    	aLB.countDocuments (sChannel, 1);

    	impl_checkForDirtyJobAndExchangeIt (sChannel);
    }

    //-------------------------------------------------------------------------
    private void impl_checkForDirtyJobAndExchangeIt (final String sChannel)
        throws Exception
    {
    	if ( ! ENABLE_JOB_EXCHANGE)
    		return;
    	
    	final CalcEngineLoadBalancer aLB            = mem_LoadBalancer ();
    	final boolean                bNeedsExchange = aLB.needsChannelExchange(sChannel);
    	if ( ! bNeedsExchange)
    		return;
    	
		if (m_bExchangeInProgress)
		{
			LOG.debug("channel '"+sChannel+"' needs to be deactivated - but another deactivation is still in progress ...");
			return;
		}
		m_bExchangeInProgress = true;
		
		LOG.info ("channel '"+sChannel+"' seems to be dirty and will be deactivated now.");
		final Object  aLock            = this;
		final boolean bExchangeStarted = aLB.exchangeChannel(sChannel, new CalcEngineLoadBalancer.INotify4OutboundChannel()
		{
			@Override
			public void notifyOutboundChannel(final String sDeactivatedChannel)
				throws Exception
			{
				LOG.info ("channel '"+sDeactivatedChannel+"' is not used any longer ... can be restarted.");
				synchronized(aLock)
				{
					final int      nJobId = impl_mapChannelId2JobId (sDeactivatedChannel);
					final IJobPool iPool    = mem_Pool ();
					iPool.restartJob (nJobId);
					aLB.resetRequestCountForChannel(sDeactivatedChannel);
					aLB.resetDocCountForChannel    (sDeactivatedChannel);
					m_bExchangeInProgress = false;
				}
			}
		});
		
		if ( ! bExchangeStarted)
			LOG.warn ("exchange of channel '"+sChannel+"' was not started successfully.");
    }

    //-------------------------------------------------------------------------
    private IJob impl_createStickyJob ()
        throws Exception
    {
    	final CalcEngineLoadBalancer aLB      = mem_LoadBalancer ();
    	final String                 sChannel = aLB.getBalancedChannel ();
    	final int                    nJobId   = impl_mapChannelId2JobId (sChannel);
    	final IJobPool               iPool    = mem_Pool ();
    	final IJob                   iJob     = iPool.leaseStickyJob(nJobId);

    	return iJob;
    }

    //-------------------------------------------------------------------------
    private void impl_cacheStickyJob (final String sDocHandle,
    								  final IJob   iJob      )
        throws Exception
    {
    	final int                    nJobId     = iJob.getId ();
    	final CalcEngineLoadBalancer aLB        = mem_LoadBalancer ();
    	final String                 sChannel   = impl_mapJobId2ChannelId (nJobId);

    	aLB.bindItemToChannel(sChannel, sDocHandle);
    }

    //-------------------------------------------------------------------------
    private IJob impl_getStickyJob (final String sDocHandle)
        throws Exception
    {
    	final CalcEngineLoadBalancer aLB      = mem_LoadBalancer ();
    	final String                 sChannel = aLB.getChannelForItem(sDocHandle);
    	
    	Validate.notEmpty (sChannel, "No calc engine for this handle '"+sDocHandle+"' ?!");

    	final int      nJobId = impl_mapChannelId2JobId (sChannel);
    	final IJobPool iPool  = mem_Pool ();
    	final IJob     iJob   = iPool.leaseStickyJob(nJobId);
    	
    	return iJob;
    }

    //-------------------------------------------------------------------------
    private void impl_freeStickyJob (final String sDocHandle,
    								 final IJob   iJob      )
        throws Exception
    {
    	// be carefully ...
    	// on 'dirty' document should not disturb all documents within same
    	// calc engine instance ...
    	// so we remove stick docs only here ...
    	// other 'dirty' documents will run into the same trouble ...
    	// and will be removed THEN !

    	// further ...
    	// be aware this method is called under might be during 'bootstrap' ...
    	// e.g. inside finally block ...
    	// where no doc handle nor a real job exists ...
    	// handle those cases carefully too .-)
    	
    	if (iJob != null)
    	{
	    	final IJobPool iPool = mem_Pool ();
	    	iPool.freeJob(iJob);
    	}
    	
    	if ( ! StringUtils.isEmpty(sDocHandle))
    	{
        	final CalcEngineLoadBalancer aLB = mem_LoadBalancer ();
        	aLB.unbindItemFromChannel(sDocHandle);
    	}
    }

    //-------------------------------------------------------------------------
    private void impl_initPoolAndLB ()
        throws Exception
    {
		final CalcEngineLoadBalancer aLB             = CalcEngineLoadBalancer.get(CalcEngineConst.LOADBALANCERID_4_CALCENGINE_WORKER);
		final IJobPool               iPool           = JobPoolManager.accessPool(CalcEngineJob.class);
		final JobPoolConfig          aPoolConfig     = JobPoolConfig.DEFAULT_CONFIG();
		final CalcEngineConfig       aCEConfig       = CalcEngineConfig.get();
		final int                    nPortMin        = aCEConfig.getWorkerPortRangeMin();
		final int                    nPortMax        = aCEConfig.getWorkerPortRangeMax();
		final String                 sWorkerJar      = aCEConfig.getWorkerBin();
		final long                   nWorkerMaxMemKB = aCEConfig.getWorkerMaxMemInKB();

		// check for invalid worker jar is already done within configuration !
		// check for port>0 is already done within configuration !
		// but check for port range is missing there .-)
		Validate.isTrue(nPortMax>=nPortMin, "Wrong or missing configuration for worker port range. Range must be (at least) 1 and max needs to be greater then min.");
		
		final boolean bBlockOnLease = Boolean.TRUE;
		final int     nPoolSize     = (nPortMax - nPortMin) + 1;
		final int     nJobTimeout   = aCEConfig.getRequestTimeout();
		
		aPoolConfig.set(JobPoolConfig.PROP_BLOCK_ON_LEASE       , bBlockOnLease);
		aPoolConfig.set(JobPoolConfig.PROP_POOL_SIZE            , nPoolSize    );
		aPoolConfig.set(JobPoolConfig.PROP_JOB_EXECUTION_TIMEOUT, nJobTimeout  );
		
		LOG.info("configure job pool ...");
		LOG.info("... block on lease = " + bBlockOnLease);
		LOG.info("... pool size      = " + nPoolSize    );
		LOG.info("... job timeout    = " + nJobTimeout  );
		
		iPool.configureJobPool(aPoolConfig);

        int nJob = 0;

		for (int nPort=nPortMin; nPort<=nPortMax; ++nPort, ++nJob)
		{
			LOG.info("configure job env ...");
			LOG.info("... worker bin   = "+sWorkerJar     );
			LOG.info("... port         = "+nPort          );
			LOG.info("... maxmemb [kb] = "+nWorkerMaxMemKB);
			
			final JobEnvironment aEnv = JobEnvironment.EMPTY_ENVIRONMENT();
			aEnv.set(CalcEngineJob.ENV_WORKERBIN, sWorkerJar     );
			aEnv.set(CalcEngineJob.ENV_PORT     , nPort          );
			aEnv.set(CalcEngineJob.ENV_MAXMEM   , nWorkerMaxMemKB);
			iPool.configureJobEnvironment(nJob, aEnv);
			
			final String sLBChannel = impl_mapJobId2ChannelId(nJob);
			aLB.registerTargetChannel(sLBChannel);
		}

		impl_configureRestartTrigger (aCEConfig, aLB, nPoolSize);
		
		m_iPool         = iPool;
		m_aLoadBalancer = aLB;
    }
    
    //-------------------------------------------------------------------------
    private void impl_configureRestartTrigger (final CalcEngineConfig       aCEConfig,
    										   final CalcEngineLoadBalancer aLB      ,
    										   final int                    nPoolSize)
        throws Exception
    {
		final int     nReserveCount         = aCEConfig.getWorkerRestartReserveCount ();
		final boolean bRestartFeatureActive = nReserveCount > 0;
		
		if ( ! bRestartFeatureActive)
			return;

	          int nRestartTriggerCount = 0;
		final int nMaxDocCountTrigger  = aCEConfig.getWorkerRestartTriggerMaxDocCount();

		if (nMaxDocCountTrigger > 0)
			nRestartTriggerCount++;

		// reserve count configured ...
		// but no idea which feature trigger a restart ...
		// disable the whole feature reasoned by inconsistent configuration.
		
		if (nRestartTriggerCount < 1)
		{
			LOG.warn("Worker restart feature disabled. Worker reserved - but no trigger defined. Please bring those values in sync.");
			return;
		}

		// a) validate configuration ... check for inconsistent states
		
		Validate.isTrue(nPoolSize     > 1        , "In case any restart trigger is defined we expect a worker pool size > 1. Please configure port range of worker right."         );
		Validate.isTrue(nReserveCount > 0        , "In case any restart trigger is defined we expect a configured reserve count also. Please configure worker reserve count right.");
		Validate.isTrue(nReserveCount < nPoolSize, "Count of reserved worker instances ["+nReserveCount+"] exceeds the configured pool size ["+nPoolSize+"]. Please adapt port range or reserve count of worker configuration.");

		// b) reserve some channels for exchange dirty worker
		
		final String[] lReservedChannels = new String[nReserveCount];
		for (int i=0; i<nReserveCount; ++i)
			lReservedChannels [i] = impl_mapJobId2ChannelId (i);

		final String sReservedChannels4Log = ArrayUtils.toString(lReservedChannels);
		LOG.info("list of reserved channels : "+sReservedChannels4Log);
		
		aLB.reserveChannelsForExchange(lReservedChannels);

		//  c) see which restart trigger is configured ...
		// c1) ... look for restart trigger : max doc count
		
		if (nMaxDocCountTrigger != CalcEngineConfig.INVALID_RESTARTTRIGGER_MAXDOC)
		{
			LOG.info("configure restart trigger for worker : max doc count = "+nMaxDocCountTrigger+" ...");
			aLB.setDocThreshold4ChannelExchange(nMaxDocCountTrigger);
		}

		// c2) ... look for restart trigger : ??? :-)
    }
    
    //-------------------------------------------------------------------------
    private static String impl_mapJobId2ChannelId (final int nJobId)
        throws Exception
    {
    	final String sChannelId = Integer.toString(nJobId);
    	return sChannelId;
    }
    
    //-------------------------------------------------------------------------
    private static int impl_mapChannelId2JobId (final String sChannelId)
        throws Exception
    {
    	final int nJobId = Integer.parseInt(sChannelId);
    	return nJobId;
    }

    //-------------------------------------------------------------------------
    private IJobPool mem_Pool ()
        throws Exception
    {
    	if (m_iPool == null)
    		impl_initPoolAndLB ();
    	return m_iPool;
    }

    //-------------------------------------------------------------------------
    private CalcEngineLoadBalancer mem_LoadBalancer ()
        throws Exception
    {
    	if (m_aLoadBalancer == null)
    		impl_initPoolAndLB ();
    	return m_aLoadBalancer;
    }

    //-------------------------------------------------------------------------
    private static CalcEngineClientLocal m_gSingleton = null;

    //-------------------------------------------------------------------------
    @SuppressWarnings("unused")
	private CalcEngineDescriptor m_aDescriptor = null;

    //-------------------------------------------------------------------------
    private IJobPool m_iPool = null;
    
    //-------------------------------------------------------------------------
    private CalcEngineLoadBalancer m_aLoadBalancer = null;

    //-------------------------------------------------------------------------
    private boolean m_bExchangeInProgress = false;
}
