
package com.openexchange.office.realtime.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import org.apache.commons.io.IOUtils;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.documentconverter.Properties;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.infostore.InfostoreExceptionCodes;
import com.openexchange.groupware.ldap.User;
import com.openexchange.log.Log;
import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.IExporter;
import com.openexchange.office.IImporter;
import com.openexchange.office.tools.ErrorCode;
import com.openexchange.office.tools.FileHelper;
import com.openexchange.office.tools.Resource;
import com.openexchange.office.tools.ResourceManager;
import com.openexchange.session.Session;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link OXDocument}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
class OXDocument {

    /**
     * Initializes a new {@link OXDocument}.
     *
     * @param session
     * @param fileHelper
     * @param importer
     * @param exporter
     */
    OXDocument(Session session, FileHelper fileHelper, ResourceManager resourceManager) {
        super();

        m_fileHelper = fileHelper;
        m_resourceManager = resourceManager;

        if (null != m_fileHelper) {
            final InputStream documentStm = m_fileHelper.getDocumentStream(session);

            if (null != documentStm) {
                try {
                    m_documentBuffer = IOUtils.toByteArray(documentStm);
                } catch (IOException e) {
                    //
                } finally {
                    IOUtils.closeQuietly(documentStm);
                }
            }

            if (null != m_documentBuffer) {
                final ZipInputStream zipInputStm = new ZipInputStream(new ByteArrayInputStream(m_documentBuffer));
                ZipEntry entry = null;
                byte[] documentBuffer = null;
                byte[] operationsBuffer = null;

                try {
                    // create a dummy resource manager, if no one is given
                    if (null == m_resourceManager) {
                        m_resourceManager = new ResourceManager(m_fileHelper.getServiceLookup());
                    }

                    while ((entry = zipInputStm.getNextEntry()) != null) {
                        final String entryName = entry.getName();

                        if (entryName.equals(FileHelper.OX_RESCUEDOCUMENT_DOCUMENTSTREAM_NAME)) {
                            try {
                                // read original document
                                documentBuffer = IOUtils.toByteArray(zipInputStm);
                            } catch (IOException e) {
                                //
                            }
                        } else if (entryName.equals(FileHelper.OX_RESCUEDOCUMENT_OPERATIONSSTREAM_NAME)) {
                            try {
                                // read original document
                                operationsBuffer = IOUtils.toByteArray(zipInputStm);
                            } catch (IOException e) {
                                //
                            }
                        } else if (entryName.startsWith(FileHelper.OX_RESCUEDOCUMENT_RESOURCESTREAM_NAME_PREFIX)) {
                            // read resource and put into the resource manager, if not already contained
                            final String resourceName = entryName.substring(FileHelper.OX_RESCUEDOCUMENT_RESOURCESTREAM_NAME_PREFIX.length());

                            if (!m_resourceManager.containsResource(Resource.getUIDFromResourceName(resourceName))) {
                                byte[] resourceBuffer = null;

                                try {
                                    resourceBuffer = IOUtils.toByteArray(zipInputStm);
                                } catch (IOException e) {
                                    //
                                }

                                m_resourceManager.addResource(resourceBuffer);
                            }
                        }

                        zipInputStm.closeEntry();
                    }
                } catch (IOException e) {
                    //
                } finally {
                    IOUtils.closeQuietly(zipInputStm);

                    // set content of original document
                    if (null != documentBuffer) {
                        m_documentBuffer = documentBuffer;
                    }

                    // set content of already available operations
                    if (null != operationsBuffer) {
                        try {
                            m_documentOperations = new JSONObject(new String(operationsBuffer));
                        } catch (JSONException e) {
                            //
                        }
                    }
                }
            }

            if ((null == m_documentBuffer) && LOG.isWarnEnabled()) {
                LOG.warn("OXDocument Ctor: could not retrieve a valid document buffer from given session.");
            }
        }
    }

    /**
     * @param appendOperations
     * @returnb The JSONObject containing all operations from the document, the saved document operations and the given operations.
     */
    JSONObject getOperations(Session session, IImporter importer, JSONObject appendOperations) throws FilterException {
        JSONObject jsonResult = null;

        if ((null != session) && (null != m_documentBuffer) && (null != importer)) {
            final InputStream documentStm = new ByteArrayInputStream(m_documentBuffer);
            try {
                String userLangCode = null;

                if (session instanceof ServerSession) {
                    final User user = ((ServerSession) session).getUser();

                    if (null != user) {
                        userLangCode = FileHelper.mapUserLanguageToLangCode(user.getPreferredLanguage());
                    }
                }
                long startTime = System.currentTimeMillis();

                DocumentProperties docProps = new DocumentProperties();
                docProps.put(DocumentProperties.PROP_USER_LANGUAGE, userLangCode);
                jsonResult = importer.createOperations(documentStm, docProps);

                if (LOG.isInfoEnabled()) {
                    LOG.info("TIME getOperations: " + (System.currentTimeMillis() - startTime));
                }

            } finally {
                IOUtils.closeQuietly(documentStm);
            }
        }

        // getting operations from the original document is mandatory
        if (null != jsonResult) {
            // append possible operations from the rescue document
            if (null != m_documentOperations) {
                ConnectionHelper.appendJSON(jsonResult, m_documentOperations);
            }

            // append given operations
            if (null != appendOperations) {
                ConnectionHelper.appendJSON(jsonResult, appendOperations);
            }
        }

        return jsonResult;
    }

    /**
     * @return the ResourceManager used to read the document
     */
    ResourceManager getResourceManager() {
        return m_resourceManager;
    }

    /**
     * @param session
     * @param exporter
     * @param messageChunkList
     * @param errorCode
     * @return
     */
    InputStream getResolvedDocumentStream(Session session, IExporter exporter, ResourceManager resourceManager, ArrayList<MessageChunk> messageChunkList, final String version, int[] errorCode, boolean finalSave) {
        InputStream resolvedDocumentStream = (null != m_documentBuffer) ? new ByteArrayInputStream(m_documentBuffer) : null;
        final JSONObject combinedOperations = new JSONObject();

        errorCode[0] = MessageData.SAVE_ERROR_NONE;

        // get combined operations
        if ((null != m_documentOperations) || !messageChunkList.isEmpty()) {
            // append possible operations from the rescue document
            if (null != m_documentOperations) {
                ConnectionHelper.appendJSON(combinedOperations, m_documentOperations);
            }

            // apply collected operations
            if (null != messageChunkList) {
                ConnectionHelper.appendOperationChunks(combinedOperations, messageChunkList, true);
            }
        }

        // if there are any operations, try to export them with the document exporter first
        if ((null != resolvedDocumentStream) && combinedOperations.has("operations") && (null != exporter) && (null != resourceManager)) {
            InputStream combinedStm = null;

            // set error code to worst case first
            errorCode[0] = MessageData.SAVE_ERROR_NOBACKUP;

            try {
                String userLangCode = null;
                String userName = "";

                if (session instanceof ServerSession) {
                    final User user = ((ServerSession) session).getUser();

                    if (null != user) {
                        userLangCode = FileHelper.mapUserLanguageToLangCode(user.getPreferredLanguage());
                        userName = user.getDisplayName();
                    }
                }

                long startTime = System.currentTimeMillis();

                DocumentProperties docProps = new DocumentProperties();
                docProps.put(DocumentProperties.PROP_USER_LANGUAGE, userLangCode);
                docProps.put(DocumentProperties.PROP_LAST_MODIFIED_BY, userName);
                if (null != version) {
                    // only use a valid version string
                    docProps.put(DocumentProperties.PROP_VERSION, version);
                }

                combinedStm = exporter.createDocument(
                    resolvedDocumentStream,
                    combinedOperations.getJSONArray("operations").toString(),
                    resourceManager,
                    docProps,
                    finalSave);

                if (LOG.isInfoEnabled()) {
                    LOG.info("TIME createDocument: " + (System.currentTimeMillis() - startTime));
                }

            } catch (FilterException e) {
                if (LOG.isWarnEnabled()) {
                    LOG.warn("Document export filter threw a filter exception while exporting the document", e);
                }
            } catch (Exception e) {
                if (LOG.isErrorEnabled()) {
                    LOG.error("Document export filter threw an exception while exporting the document", e);
                }
            } finally {
                IOUtils.closeQuietly(resolvedDocumentStream);

                if (null != (resolvedDocumentStream = combinedStm)) {
                    errorCode[0] = MessageData.SAVE_ERROR_NONE;
                }
            }
        } else if (combinedOperations.has("operations")) {
            errorCode[0] = MessageData.SAVE_ERROR_NOBACKUP;

            if (LOG.isErrorEnabled()) {
                if (null == exporter) {
                    LOG.error("Document export filter not available - not possible to provide resolved stream");
                }
                if (null == resourceManager) {
                    LOG.error("Resource manager not available - not possible to provide resolved stream");
                }
                if (null == resolvedDocumentStream) {
                    LOG.error("Document stream not available - not possible to provide resolved stream");
                }
            }
        }

// !!! DEBUG CODE to create OX rescue documents in every case => Don't forget to remove or comment for final release !!!
//        {
//            errorCode[0] = MessageData.SAVE_ERROR_OPERATIONS;
//        }
// !!! DEBUG CODE  => remove or comment for final release !!!

        // if the there was an error before, we write a zipped OXDocument with
        // the original document and the operations content in separate streams
        if (MessageData.SAVE_ERROR_NONE != errorCode[0]) {
            InputStream inputStm = null;
            final ByteArrayOutputStream byteArrayStm = new ByteArrayOutputStream(8192);
            final ZipOutputStream zipOutputStm = new ZipOutputStream(byteArrayStm);

            // set error code to worst case
            errorCode[0] = MessageData.SAVE_ERROR_NOBACKUP;

            // write original document as well as collected operations
            // and collected resources into different zip entry streams
            try {
                // write original document stream
                if (null != m_documentBuffer) {
                    inputStm = new ByteArrayInputStream(m_documentBuffer);

                    try {
                        zipOutputStm.putNextEntry(new ZipEntry(FileHelper.OX_RESCUEDOCUMENT_DOCUMENTSTREAM_NAME));
                        IOUtils.copy(inputStm, zipOutputStm);
                        zipOutputStm.flush();
                    } catch (IOException e) {
                        if (LOG.isErrorEnabled()) {
                            LOG.error("Exception while writing original document stream to backup document", e);
                        }
                    } finally {
                        zipOutputStm.closeEntry();
                        IOUtils.closeQuietly(inputStm);
                        inputStm = null;
                    }

                }

                // write collected operations, that were added
                // by editing users, into separate zip entry stream
                if (!combinedOperations.isEmpty()) {
                    inputStm = new ByteArrayInputStream(combinedOperations.toString().getBytes("UTF-8"));

                    try {
                        zipOutputStm.putNextEntry(new ZipEntry(FileHelper.OX_RESCUEDOCUMENT_OPERATIONSSTREAM_NAME));
                        IOUtils.copy(inputStm, zipOutputStm);
                        zipOutputStm.flush();
                    } catch (IOException e) {
                        if (LOG.isErrorEnabled()) {
                            LOG.error("Exception while writing operations stream to backup document", e);
                        }
                    } finally {
                        zipOutputStm.closeEntry();
                        IOUtils.closeQuietly(inputStm);
                        inputStm = null;
                    }

                }

                // write collected resources, that were added by
                // editing users, into separate zip entry streams
                if (null != resourceManager) {
                    final Hashtable<Long, byte[]> resourceBuffers = resourceManager.getByteArrayTable();

                    for (final Long curKey : resourceBuffers.keySet()) {
                        final byte[] curResourceBuffer = resourceBuffers.get(curKey);

                        if (null != curResourceBuffer) {
                            final String curResourceName = FileHelper.OX_RESCUEDOCUMENT_RESOURCESTREAM_NAME_PREFIX + Resource.getResourceNameFromUID(curKey.longValue());

                            inputStm = new ByteArrayInputStream(curResourceBuffer);

                            try {
                                zipOutputStm.putNextEntry(new ZipEntry(curResourceName));
                                IOUtils.copy(inputStm, zipOutputStm);
                                zipOutputStm.flush();
                            } catch (IOException e) {
                                if (LOG.isErrorEnabled()) {
                                    LOG.error("Exception while writing collected resources to backup document", e);
                                }
                            } finally {
                                zipOutputStm.closeEntry();
                                IOUtils.closeQuietly(inputStm);
                                inputStm = null;
                            }
                        }
                    }
                }

                zipOutputStm.flush();
            } catch (IOException e) {
                if (LOG.isErrorEnabled()) {
                    LOG.error("Exception while writing backup document", e);
                }
            } finally {
                IOUtils.closeQuietly(zipOutputStm);
                final byte[] byteArray = byteArrayStm.toByteArray();

                if (byteArray.length > 0) {
                    resolvedDocumentStream = new ByteArrayInputStream(byteArray);
                    errorCode[0] = MessageData.SAVE_ERROR_OPERATIONS;
                }
            }
        }

        return resolvedDocumentStream;
    }

    /**
     * @param session
     * @param exporter
     * @param messageChunkList
     * @param errorCode
     * @return
     */
    String save(Session session, IExporter exporter, ResourceManager resourceManager, ArrayList<MessageChunk> messageChunkList, int[] errorCode, boolean revisionless, boolean finalSave) {
        String currentVersion = null;
        String fileVersion = null;

        try {
            currentVersion = m_fileHelper.getVersionFromFile(session);
        } catch (OXException e) {
            if (e.getPrefix() == "IFO") {
                if (e.getCode() == InfostoreExceptionCodes.NOT_EXIST.getNumber()) {
                    errorCode[0] = MessageData.SAVE_ERROR_FILE_NOT_FOUND;
                    return fileVersion;
                }
            }
        }

        // revisions are incremented by one - determine current version and
        // provide an incremented version.
        if ((null != currentVersion) && !revisionless) {
            try {
                int currentRevision = Integer.parseInt(currentVersion);
                currentRevision++;
                currentVersion = Integer.toString(currentRevision);
            } catch (final NumberFormatException e) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: Exception while determining the new version", e);
                }
            }
        }

        final InputStream documentStm = getResolvedDocumentStream(
            session,
            exporter,
            resourceManager,
            messageChunkList,
            currentVersion,
            errorCode,
            finalSave);

        if ((MessageData.SAVE_ERROR_NOBACKUP != errorCode[0]) && (null != documentStm)) {
            final ErrorCode noError = ErrorCode.NO_ERROR;
            final ErrorCode[] writeDocErrorCode = { noError };

            fileVersion = m_fileHelper.writeDocumentStream(
                session,
                documentStm,
                Properties.OX_RESCUEDOCUMENT_EXTENSION_APPENDIX,
                MessageData.SAVE_ERROR_OPERATIONS == errorCode[0],
                writeDocErrorCode,
                revisionless);

            if (writeDocErrorCode[0].getCode() != 0) {
                // error evaluation
                if (writeDocErrorCode[0] == ErrorCode.FILTER_QUOTA_REACHED_ERROR) {
                    errorCode[0] = MessageData.SAVE_ERROR_QUOTA_REACHED;
                }
            }
        }

        IOUtils.closeQuietly(documentStm);

        return fileVersion;
    }

    // - Members ---------------------------------------------------------------

    static private final org.apache.commons.logging.Log LOG = Log.loggerFor(OXDocument.class);

    private FileHelper m_fileHelper = null;

    private byte[] m_documentBuffer = null;

    private JSONObject m_documentOperations = null;

    private ResourceManager m_resourceManager = null;
}
