/*
 *  Copyright 2010, Plutext Pty Ltd.
 *
 *  This file is part of docx4j.

    docx4j is licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.

    You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

    Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.

 */


package org.xlsx4j.sml;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import javax.xml.bind.Unmarshaller;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlTransient;
import javax.xml.bind.annotation.XmlType;
import org.apache.commons.logging.Log;
import org.docx4j.XmlUtils;
import org.docx4j.jaxb.Context;
import org.jvnet.jaxb2_commons.ppp.Child;
import org.xlsx4j.sml.Cell.CellRef;
import com.openexchange.log.LogFactory;
import com.openexchange.office.ooxml.tools.Commons;



/**
 * <p>Java class for CT_SheetData complex type.
 *
 * <p>The following schema fragment specifies the expected content contained within this class.
 *
 * <pre>
 * &lt;complexType name="CT_SheetData">
 *   &lt;complexContent>
 *     &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType">
 *       &lt;sequence>
 *         &lt;element name="row" type="{http://schemas.openxmlformats.org/spreadsheetml/2006/main}CT_Row" maxOccurs="unbounded" minOccurs="0"/>
 *       &lt;/sequence>
 *     &lt;/restriction>
 *   &lt;/complexContent>
 * &lt;/complexType>
 * </pre>
 *
 *
 */
@XmlAccessorType(XmlAccessType.FIELD)
@XmlType(name = "CT_SheetData", propOrder = {
    "row"
})
public class SheetData implements Child {

    @XmlTransient
    protected static Log log = LogFactory.getLog(SheetData.class);
    protected List<Row> row;
    @XmlTransient
    private Map<Integer, Row> rows_impl;
    @XmlTransient
    private Object parent;
    @XmlTransient
    private Long maxSi;

    /**
     * Gets the value of the row property.
     *
     * <p>
     * This accessor method returns a reference to the live list,
     * not a snapshot. Therefore any modification you make to the
     * returned list will be present inside the JAXB object.
     * This is why there is not a <CODE>set</CODE> method for the row property.
     *
     * <p>
     * For example, to add a new item, do as follows:
     * <pre>
     *    getRow().add(newItem);
     * </pre>
     *
     *
     * <p>
     * Objects of the following type(s) are allowed in the list
     * {@link Row }
     *
     *
     */

    // this method is now private to ensure that Map<Long, Row> stays up to date..
    private List<Row> getRow() {
        if (row == null) {
            row = new ArrayList<Row>();
        }
        return this.row;
    }

    // the List<Row> may include multiple Rows addressing the same row index, so our Map<Long, Row>
    // takes care only of the first (TODO merging of rows with the same index)
    // if a row is not having an explicit index its index is always previous rowIndex + 1..
    // if a rowIndex is greater than the index of the previous row we have an invalid document
    private Map<Integer, Row> getMap() {
        if(rows_impl==null) {
            rows_impl = new HashMap<Integer, Row>();
            Commons.RowIterator rowIterator = new Commons.RowIterator(getRow());
            while(rowIterator.hasNext()) {
                Row _row = rowIterator.next();
                rows_impl.put(rowIterator.getRowNumber(), _row);
            }
        }
        return rows_impl;
    }

    public Row getRow(int index, boolean forceCreate) {
        final Map<Integer, Row> hashRows = getMap();
        Row r = hashRows.get(index);
        if(forceCreate&&r==null) {

            // creating a new row...

            r = org.docx4j.jaxb.Context.getsmlObjectFactory().createRow();
            r.setParent(this);
            r.setR((long)(index + 1));
            hashRows.put(index, r);

            // applying changes to the list<row> which is
            // the only serialized member here

            int rowListIndex = getRow().size();
            Commons.RowIterator rowIterator = new Commons.RowIterator(getRow());
            while(rowIterator.hasNext()) {
                rowIterator.next();
                if(rowIterator.getRowNumber()>=index) {
                    rowListIndex = rowIterator.getListIndex();
                    break;
                }
            }
            getRow().add(rowListIndex, r);
        }
        return r;
    }

    private static boolean equals(CTBorderPr border1, CTBorderPr border2) {
        if(border1==border2)
            return true;
        if(border1==null||border2==null)
            return false;
        if(border1.getStyle()!=border2.getStyle())
            return false;
        CTColor col1 = border1.getColor();
        CTColor col2 = border2.getColor();
        if(col1==col2)
            return true;
        if(col1==null||col2==null)
            return false;
        if(col1.isAuto()!=null&&col2.isAuto()!=null) {
            if(col1.isAuto().booleanValue()!=col2.isAuto().booleanValue())
                return false;
        }
        else if (col1.isAuto()!=col2.isAuto())
            return false;
        if(col1.getIndexed()!=null&&col2.getIndexed()!=null) {
            if(col1.getIndexed().longValue()!=col2.getIndexed().longValue())
                return false;
        }
        else if(col1.getIndexed()!=col2.getIndexed())
            return false;
        if(!Arrays.equals(col1.getRgb(), col2.getRgb()))
            return false;
        if(col1.getTheme()!=null&&col2.getTheme()!=null) {
            if(col1.getTheme().longValue()!=col2.getTheme().longValue())
                return false;
        }
        else if(col1.getTheme()!=col2.getTheme())
            return false;

        if(col1.getTint()!=null&&col2.getTint()!=null) {
            if(col1.getTint().doubleValue()!=col2.getTint().doubleValue())
                return false;
        }
        else if(col1.getTint()!=col2.getTint())
            return false;
        return true;
    }

    private static CTBorderPr createClone(CTBorderPr border) {
        if(border==null)
            return null;
        CTBorderPr newBorder = Context.getsmlObjectFactory().createCTBorderPr();
        newBorder.setStyle(border.getStyle());

        CTColor color = border.getColor();
        if(color!=null) {
            CTColor newColor = Context.getsmlObjectFactory().createCTColor();
            newColor.setAuto(color.isAuto());
            newColor.setIndexed(color.getIndexed());
            if(color.getRgb()!=null) {
                newColor.setRgb(Arrays.copyOf(color.getRgb(), color.getRgb().length));
            }
            newColor.setTheme(color.getTheme());
            newColor.setTint(color.getTint());
        }
        return newBorder;
    }

    public static void ApplyBorderStyle(CTStylesheet stylesheet, Cell cellClone, Cell nextRowCell) {
        final CTXf ctXfPrev = stylesheet.getCellXfByIndex(cellClone.getS());
        // is prevRow using a hard border ?
        if(ctXfPrev!=null&&ctXfPrev.isApplyBorder()!=null&&ctXfPrev.isApplyBorder()&&ctXfPrev.getBorderId()!=null) {
            long borderId = ctXfPrev.getBorderId().longValue();
            long newBorderId = 0;   // this is our new borderId if the cell of the nextRow is not having hard border attributes
            if(nextRowCell!=null) {
                final CTXf ctXfNext = stylesheet.getCellXfByIndex(nextRowCell.getS());
                if(ctXfNext!=null&&ctXfNext.isApplyBorder()!=null&&ctXfNext.isApplyBorder()&&ctXfNext.getBorderId()!=null) {
                    long nextBorderId = ctXfNext.getBorderId().longValue();
                    if(nextBorderId==borderId) {
                        newBorderId = borderId;     // no special border handling...
                    }
                    else {
                        CTBorder prevBorder = stylesheet.getBorderByIndex(borderId);
                        CTBorder nextBorder = stylesheet.getBorderByIndex(nextBorderId);
                        if(prevBorder!=null&&nextBorder!=null) {
                            // we have to create a new border...
                            CTBorder newBorder = Context.getsmlObjectFactory().createCTBorder();
                            if(equals(prevBorder.getBottom(), nextBorder.getBottom()))
                                newBorder.setBottom(createClone(prevBorder.getBottom()));
                            if(equals(prevBorder.getLeft(), nextBorder.getLeft()))
                                newBorder.setLeft(createClone(prevBorder.getLeft()));
                            if(equals(prevBorder.getTop(), nextBorder.getTop()))
                                newBorder.setTop(createClone(prevBorder.getTop()));
                            if(equals(prevBorder.getRight(), nextBorder.getRight()))
                                newBorder.setRight(createClone(prevBorder.getRight()));
                            if(equals(prevBorder.getDiagonal(), nextBorder.getDiagonal()))
                                newBorder.setDiagonal(createClone(prevBorder.getDiagonal()));
                            if(equals(prevBorder.getHorizontal(), nextBorder.getHorizontal()))
                                newBorder.setHorizontal(createClone(prevBorder.getHorizontal()));
                            if(equals(prevBorder.getVertical(), nextBorder.getVertical()))
                                newBorder.setVertical(createClone(prevBorder.getVertical()));
                            newBorder.setDiagonalDown(prevBorder.isDiagonalDown());
                            newBorder.setDiagonalUp(prevBorder.isDiagonalUp());

                            // this is our new border id...
                            newBorderId = stylesheet.getOrApplyBorder(newBorder);
                        }
                    }
                }
            }
            if(borderId!=newBorderId) {
                CTXf detachedXf = XmlUtils.deepCopy(ctXfPrev, Context.getJcSML());
                detachedXf.setApplyBorder(newBorderId!=0);
                detachedXf.setBorderId(newBorderId);
                long index = stylesheet.getOrApplyCellXf(detachedXf);
                if(index>0) {
                    cellClone.setS(index);
                }
            }
        }
    }

    public void insertRows(CTStylesheet stylesheet, int start, int insertCount) {
        final List<Row> rows = getRow();
        final ObjectFactory objectFactory = Context.getsmlObjectFactory();

        int lastRowNumber = 0;
        int rowListIndex = rows.size();
        Commons.RowIterator rowIterator = new Commons.RowIterator(rows);
        while(rowIterator.hasNext()) {
            rowIterator.next();
            if(rowIterator.getRowNumber()>=start) {
                rowListIndex = rowIterator.getListIndex();
                break;
            }
            if(rowIterator.hasNext()) {
                lastRowNumber = rowIterator.getRowNumber();
            }
        }

        int i, insertIndex = rowListIndex;


        Row prevRow = null;
        // check if we can inherit row properties from the preceding row
        if(insertIndex>0&&lastRowNumber+1==start) {
            prevRow = rows.get(insertIndex-1);
        }
        Row nextRow = null;
        if(insertIndex<rows.size()&&rowIterator.getRowNumber()==start) {
            nextRow = rows.get(insertIndex);
        }

        // now each following row and cell ref behind our insert position
        // is to be increased by the insertCount
        for(i=insertIndex;i<rows.size();i++) {
            Row r = rows.get(i);
            Long rowRef = r.getR();
            if(rowRef!=null) {
                rowRef += insertCount;
                r.setR(rowRef);
            }
            Commons.CellIterator cellIterator = r.createCellIterator();
            while(cellIterator.hasNext()) {
                Cell cell = cellIterator.next();
                String stringRef = cell.getR();
                if(stringRef!=null) {
                    CellRef cellRef = Cell.createCellRef(stringRef);
                    cell.setR(Cell.getCellRef(cellRef.getColumn(), cellRef.getRow() + insertCount));
                }
            }
        }

        // now we create and insert our new rows
        for(i=0; i<insertCount; i++) {
            Row newRow = objectFactory.createRow();
            newRow.setR(new Long(start+1+i));
            rows.add(insertIndex+i, newRow);
            if(prevRow!=null) {
                newRow.setCollapsed(prevRow.isCollapsed());
                newRow.setCustomFormat(prevRow.isCustomFormat());
                newRow.setCustomHeight(prevRow.isCustomHeight());
                newRow.setHidden(false);
                newRow.setHt(prevRow.getHt());
                newRow.setOutlineLevel(prevRow.getOutlineLevel());
                newRow.setParent(prevRow.getParent());
                newRow.setPh(prevRow.isPh());
                newRow.setS(prevRow.getS());
                newRow.setThickBot(prevRow.isThickBot());
                newRow.setThickTop(prevRow.isThickTop());

                // now cloning cells
                Commons.CellIterator cellIterator  = prevRow.createCellIterator();
                while(cellIterator.hasNext()) {
                    Cell cell = cellIterator.next();
                    CellRef cellRef = Cell.createCellRef(cell.getR());
                    Cell cellClone = newRow.getCell(cellRef.getColumn(), true);
                    cellClone.setS(cell.getS());

                    // we need to take care of border attributes only if
                    // inserting the first row, second insertion is always
                    // a clone of the previous row
                    if(i==0) {
                        Cell nextRowCell = null;
                        if(nextRow!=null) {
                            nextRowCell = nextRow.getCell(cellRef.getColumn(), false);
                        }
                        ApplyBorderStyle(stylesheet, cellClone, nextRowCell);
                    }
                }
                prevRow = newRow;
            }
        }
        // resetting our hash map
        rows_impl = null;
    }

    public void deleteRows(int start, int deleteCount) {
        final List<Row> rows = getRow();
        final Commons.RowIterator rowIterator = new Commons.RowIterator(rows);
        while(rowIterator.hasNext()) {
            Row r = rowIterator.next();
            if(rowIterator.getRowNumber()>=start) {
                int lastRowNumber = rowIterator.getRowNumber() - 1;
                for(int i=rowIterator.getListIndex();i<rows.size();) {
                    r = rows.get(i);
                    int rowNumber = r.getR()!=null?r.getR().intValue()-1:lastRowNumber+1;
                    if(rowNumber<=start+deleteCount-1) {
                        rows.remove(i);
                    }
                    else {
                        if(r.getR()!=null) {
                            rowNumber -= deleteCount;
                            r.setR((long)(rowNumber+1));
                        }
                        final Iterator<Cell> cellIterator = r.createCellIterator();
                        while(cellIterator.hasNext()) {
                            final Cell cell = cellIterator.next();
                            final String stringRef = cell.getR();
                            if(stringRef!=null) {
                                final CellRef cellRef = Cell.createCellRef(stringRef);
                                cell.setR(Cell.getCellRef(cellRef.getColumn(), cellRef.getRow() - deleteCount));
                            }
                        }
                        i++;
                    }
                    lastRowNumber=rowNumber;
                }
                break;
            }
        }
        rows_impl = null;
    }

    public void insertColumns(CTStylesheet stylesheet, int start, int insertCount) {
        Commons.RowIterator rowIterator = new Commons.RowIterator(getRow());
        while(rowIterator.hasNext()) {
            Row r = rowIterator.next();
            r.insertCells(stylesheet, rowIterator.getRowNumber(), start, insertCount);
        }
    }

    public void deleteColumns(int start, int deleteCount) {
        for(Row _row:getRow()) {
            _row.deleteCells(start, deleteCount);
        }
    }

    public Commons.RowIterator createRowIterator() {
        return new Commons.RowIterator(getRow(), rows_impl);
    }

    @Override
    public Object getParent() {
        return this.parent;
    }

    @Override
    public void setParent(Object parent) {
        this.parent = parent;
    }

    public void setSi(long si) {
        if(maxSi==null) {
            maxSi = si;
        }
        else if(si>maxSi) {
            maxSi = si;
        }
    }

    public long getNextSi() {
        if(maxSi==null) {
            maxSi = 0L;
        }
        else {
            maxSi++;
        }
        return maxSi;
    }

    /**
     * This method is invoked by the JAXB implementation on each instance when unmarshalling completes.
     *
     * @param parent
     *     The parent object in the object tree.
     * @param unmarshaller
     *     The unmarshaller that generated the instance.
     */
    public void afterUnmarshal(Unmarshaller unmarshaller, Object _parent) {
        setParent(_parent);
    }
}
