/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.drawingml;

import java.util.HashMap;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.docx4j.dml.CTShapeProperties;
import org.docx4j.dml.chart.CTArea3DChart;
import org.docx4j.dml.chart.CTAreaChart;
import org.docx4j.dml.chart.CTAxDataSource;
import org.docx4j.dml.chart.CTAxPos;
import org.docx4j.dml.chart.CTBar3DChart;
import org.docx4j.dml.chart.CTBarChart;
import org.docx4j.dml.chart.CTBarDir;
import org.docx4j.dml.chart.CTBarGrouping;
import org.docx4j.dml.chart.CTBoolean;
import org.docx4j.dml.chart.CTBubbleChart;
import org.docx4j.dml.chart.CTBubbleSer;
import org.docx4j.dml.chart.CTCatAx;
import org.docx4j.dml.chart.CTChart;
import org.docx4j.dml.chart.CTChartSpace;
import org.docx4j.dml.chart.CTDoughnutChart;
import org.docx4j.dml.chart.CTGrouping;
import org.docx4j.dml.chart.CTLegend;
import org.docx4j.dml.chart.CTLegendPos;
import org.docx4j.dml.chart.CTLine3DChart;
import org.docx4j.dml.chart.CTLineChart;
import org.docx4j.dml.chart.CTLineSer;
import org.docx4j.dml.chart.CTMarker;
import org.docx4j.dml.chart.CTNumData;
import org.docx4j.dml.chart.CTNumDataSource;
import org.docx4j.dml.chart.CTNumRef;
import org.docx4j.dml.chart.CTNumVal;
import org.docx4j.dml.chart.CTOfPieChart;
import org.docx4j.dml.chart.CTOrientation;
import org.docx4j.dml.chart.CTPie3DChart;
import org.docx4j.dml.chart.CTPieChart;
import org.docx4j.dml.chart.CTPlotArea;
import org.docx4j.dml.chart.CTRadarChart;
import org.docx4j.dml.chart.CTScaling;
import org.docx4j.dml.chart.CTScatterChart;
import org.docx4j.dml.chart.CTSerTx;
import org.docx4j.dml.chart.CTStockChart;
import org.docx4j.dml.chart.CTStrData;
import org.docx4j.dml.chart.CTStrRef;
import org.docx4j.dml.chart.CTStrVal;
import org.docx4j.dml.chart.CTSurface3DChart;
import org.docx4j.dml.chart.CTSurfaceChart;
import org.docx4j.dml.chart.CTUnsignedInt;
import org.docx4j.dml.chart.CTValAx;
import org.docx4j.dml.chart.ListSer;
import org.docx4j.dml.chart.STAxPos;
import org.docx4j.dml.chart.STBarDir;
import org.docx4j.dml.chart.STBarGrouping;
import org.docx4j.dml.chart.STGrouping;
import org.docx4j.dml.chart.STLegendPos;
import org.docx4j.dml.chart.STOrientation;
import org.docx4j.dml.chart.SerContent;
import org.docx4j.jaxb.Context;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.office.ooxml.tools.Commons;

final public class DMLChartSpace {

    public static void createChartSpaceOperations(JSONArray operationsArray, JSONArray position, CTChartSpace chartSpace)
        throws JSONException {

        final CTChart chart = chartSpace.getChart();
        if(chart==null) {
            return;
        }
        final CTPlotArea plotArea = chart.getPlotArea();
        if(plotArea!=null) {
            final List<ListSer> chartList = plotArea.getAreaChartOrArea3DChartOrLineChart();
            // TODO: what if there is more than one chart in this list ?
            if(!chartList.isEmpty()) {
                final ListSer chartSer = chartList.get(0);
                
                final CTBoolean varyColors = chartSer.getVaryColors();
                final CTLegend legend = chart.getLegend();
                
                if (varyColors != null || legend != null) {
                    JSONObject insertDrawingOperation = operationsArray.getJSONObject(operationsArray.length() - 1);
                    JSONObject attrs = insertDrawingOperation.getJSONObject("attrs");
                    JSONObject chartProperties = attrs.getJSONObject("chart");                    
                    
                    if (chartSer instanceof CTPieChart) {
                        chartProperties.put("rotation", ((CTPieChart) chartSer).getFirstSliceAng().getVal());
                    }
                    
                    if (varyColors != null) {
                        chartProperties.put("varyColors", varyColors.isVal());
                    }
                    if (legend != null) {
                        if (legend.getLegendPos() != null && legend.getLegendPos().getVal() != null) {
                            chartProperties.put("legendPos", legend.getLegendPos().getVal().getRealName());
                        }
                    }
                }

                // we are taking only the first chart ...
                for(int seriesIndex = 0; seriesIndex < chartSer.getSer().size(); seriesIndex++) {
                    final SerContent ser = chartSer.getSer().get(seriesIndex);
                   
                    // title
                    final JSONObject series = new JSONObject();
                    final CTSerTx tx = ser.getTx();
                    Object title = null;
                    if(tx!=null) {
                        if(tx.getStrRef()!=null) {
                            title = tx.getStrRef().getF();
                        }
                        else {
                            title = new JSONArray();
                            ((JSONArray)title).put(tx.getV());
                        }
                    }
                    if(title!=null) {
                        series.put("title", title);
                    }

                    // names
                    Object names = null;
                    final CTAxDataSource cat = ser.getCat();
                    if(cat!=null) {
                        if(cat.getStrRef()!=null) {
                            names = cat.getStrRef().getF();
                        }
                        else if(cat.getStrLit()!=null) {
                            final CTStrData strLit = cat.getStrLit();
                            final List<CTStrVal> strLitPtList = strLit.getPt();
                            if(strLitPtList!=null&&!strLitPtList.isEmpty()) {
                                names = new JSONArray();
                                for(CTStrVal strLitPtEntry:strLitPtList) {
                                    if(strLitPtEntry!=null) {
                                        ((JSONArray)names).put(strLitPtEntry.getV());
                                    }
                                    else {
                                        ((JSONArray)names).put((String)null);
                                    }
                                }
                            }
                        }
                        else if(cat.getNumRef()!=null) {
                            names = cat.getNumRef().getF();
                        }
                        else if(cat.getNumLit()!=null) {
                            final CTNumData numLit = cat.getNumLit();
                            final List<CTNumVal> numLitPtList = numLit.getPt();
                            if(numLitPtList!=null&&!numLitPtList.isEmpty()) {
                                names = new JSONArray();
                                for(CTNumVal numLitPtEntry:numLitPtList) {
                                    if(numLitPtEntry!=null) {
                                        ((JSONArray)names).put(numLitPtEntry.getV());
                                    }
                                    else {
                                        ((JSONArray)names).put((String)null);
                                    }
                                }
                            }
                        }
                        else if(cat.getMultiLvlStrRef()!=null) {
                            names = cat.getMultiLvlStrRef().getF();
                        }
                    }
                    if(names!=null) {
                        series.put("names", names);
                    }

                    // values
                    final CTNumDataSource numDataSource = ser.getVal();
                    final Object values = getData(numDataSource);
                    
                    if(values!=null) {
                        series.put("values", values);
                    }
                    
                    if(ser instanceof CTBubbleSer) {
                        final CTBubbleSer serBB = (CTBubbleSer)ser;
                        
                        CTNumDataSource bubblesizes = serBB.getBubbleSize();
                        Object bubbles = getData(bubblesizes);
                        if(bubbles!=null) {
                            series.put("bubbles", bubbles);
                        }
                    }
                    final CTShapeProperties shape = ser.getSpPr();
                    if (shape != null) {
                        shape.addToJSON(series);    
                    }
                    
                    CTMarker marker = ser.getMarker();
                    if(marker!=null && marker.getSymbol()!=null){
                        series.put("marker", marker.getSymbol().getVal().value());
                    }
                    

                    if(series.length()>0) {
                        final JSONObject insertChartDataSeriesOperation = new JSONObject();
                        insertChartDataSeriesOperation.put("name", "insertChartDataSeries");
                        insertChartDataSeriesOperation.put("start", position);
                        insertChartDataSeriesOperation.put("series", seriesIndex);
                        insertChartDataSeriesOperation.put("attrs", Commons.surroundJSONObject("series", series));
                        operationsArray.put(insertChartDataSeriesOperation);
                    }
                    
                }
            }
            plotArea.addToOperations(operationsArray, position);            
        }
    }
    
    private static Object getData(CTNumDataSource numDataSource){
        if(numDataSource.getNumRef()!=null) {
            return numDataSource.getNumRef().getF();
        } else if(numDataSource.getNumLit()!=null) {
            final CTNumData numLit = numDataSource.getNumLit();
            if(numLit!=null) {
                final List<CTNumVal> numLitPtList = numLit.getPt();
                if(numLitPtList!=null&&!numLitPtList.isEmpty()) {
                    JSONArray values = new JSONArray();
                    for(CTNumVal numLitPt:numLitPtList) {
                        if(numLitPt!=null) {
                            values.put(numLitPt.getV());       // TODO: getIdx
                        } else {
                            values.put((String)null);
                        }
                    }
                    return values;
                }
            }
        }
        return null;
    }

    public static void applyChartSpaceProperties(JSONObject chartProperties, CTChartSpace chartSpace) {

        if(chartProperties.has("type")) {
            final ListSer newListSer = createChartType(chartProperties);
            if(newListSer!=null) {
                final ListSer oldListSer = chartSpace.getChart().getPlotArea().getAreaChartOrArea3DChartOrLineChart().get(0);
                chartSpace.getChart().getPlotArea().getAreaChartOrArea3DChartOrLineChart().set(0, newListSer);
                for(SerContent serContent:oldListSer.getSer()) {
                    newListSer.getSer().add(serContent);
                }
            }
        }
    }


    private static CTUnsignedInt getUnsignedInt(int ptCount) {
        final CTUnsignedInt unsignedInt = Context.getDmlChartObjectFactory().createCTUnsignedInt();
        unsignedInt.setVal(ptCount);
        return unsignedInt;
    }

    private static String getRef(String sheetName, String ref) {
        if(ref.indexOf('!')==-1) {
            return sheetName + '!' + ref;
        }
        return ref;
    }

    // Idx and Order are required elements, so at least we have to create this two elements for our new
    // inserted series... but we will correct indexes of other series, so that they are properly increasing
    private static void updateIndexAndOrder(List<SerContent> serContentList) {
        for(int i=0;i<serContentList.size();i++) {
            final SerContent serCont = serContentList.get(i);
            serCont.setIdx(getUnsignedInt(i));
            serCont.setOrder(getUnsignedInt(i));
        }
    }

    public static void insertChartSpace(CTChartSpace chartSpace, String sheetName, int series, JSONObject attrs)
        throws JSONException {

        final ListSer listSer = chartSpace.getChart().getPlotArea().getAreaChartOrArea3DChartOrLineChart().get(0);
        final List<SerContent> serContentList = listSer.getSer();
        final SerContent serContent = listSer.createSer();
        serContentList.add(series, serContent);
        final CTSerTx serTx = Context.getDmlChartObjectFactory().createCTSerTx();
        serContent.setTx(serTx);
        final CTAxDataSource cat = Context.getDmlChartObjectFactory().createCTAxDataSource();
        serContent.setCat(cat);
        final CTNumDataSource val = Context.getDmlChartObjectFactory().createCTNumDataSource();
        serContent.setVal(val);

        if(attrs!=null) {
            final JSONObject attrSeries = attrs.optJSONObject("series");
            if(attrSeries!=null) {

                // title
                final Object title = attrSeries.opt("title");
                if(title instanceof String) {
                    final CTStrRef strRef = Context.getDmlChartObjectFactory().createCTStrRef();
                    strRef.setF(getRef(sheetName, (String)title));
                    serTx.setStrRef(strRef);
                }
                else if(title instanceof JSONArray&&!((JSONArray)title).isEmpty()) {
                    final Object o = ((JSONArray)title).get(0);
                    if(o instanceof String) {
                        serTx.setV((String)o);
                    }
                }

                // names
                final Object names = attrSeries.opt("names");
                if(names instanceof String) {
                    final CTStrRef strRef = Context.getDmlChartObjectFactory().createCTStrRef();
                    strRef.setF(getRef(sheetName, (String)names));
                    cat.setStrRef(strRef);
                }
                else if(title instanceof JSONArray) {
                    final JSONArray titleArray = (JSONArray)title;
                    final CTStrData strData = Context.getDmlChartObjectFactory().createCTStrData();
                    cat.setStrLit(strData);
                    strData.setPtCount(getUnsignedInt(titleArray.length()));
                    for(int i=0; i<titleArray.length();i++) {
                        final CTStrVal strVal = Context.getDmlChartObjectFactory().createCTStrVal();
                        strVal.setV(titleArray.optString(i, ""));
                        strData.getPt().add(strVal);
                    }
                }

                // values
                final Object values = attrSeries.opt("values");
                if(values instanceof String) {
                    final CTNumRef numRef = Context.getDmlChartObjectFactory().createCTNumRef();
                    numRef.setF(getRef(sheetName, (String)values));
                    val.setNumRef(numRef);
                }
                else if(values instanceof JSONArray) {
                    final JSONArray valuesArray = (JSONArray)values;
                    final CTNumData numData = Context.getDmlChartObjectFactory().createCTNumData();
                    val.setNumLit(numData);
                    numData.setPtCount(getUnsignedInt(valuesArray.length()));
                    for(int i=0; i<valuesArray.length();i++) {
                        final CTNumVal numVal = Context.getDmlChartObjectFactory().createCTNumVal();
                        numVal.setV(valuesArray.optString(i, ""));
                        numData.getPt().add(numVal);
                    }
                }
            }
            updateIndexAndOrder(serContentList);
        }
    }

    public static void deleteChartSpace(CTChartSpace chartSpace, int series) {

        final ListSer listSer = chartSpace.getChart().getPlotArea().getAreaChartOrArea3DChartOrLineChart().get(0);
        final List<SerContent> serContentList = listSer.getSer();
        serContentList.remove(series);
        updateIndexAndOrder(serContentList);
    }

    public static CTChartSpace createChartSpace(JSONObject chartAttrs) {
        final org.docx4j.dml.chart.ObjectFactory chartObjectFactory = Context.getDmlChartObjectFactory();
        final CTChartSpace chartSpace = chartObjectFactory.createCTChartSpace();
        final CTChart chart = chartObjectFactory.createCTChart();
        chartSpace.setChart(chart);
        final CTPlotArea plotArea = chartObjectFactory.createCTPlotArea();
        chart.setPlotArea(plotArea);
        final CTLegend legend = chartObjectFactory.createCTLegend();
        chart.setLegend(legend);
        final CTLegendPos legendPos = chartObjectFactory.createCTLegendPos();
        legend.setLegendPos(legendPos);
        legendPos.setVal(STLegendPos.R);
        final CTBoolean booleanValue = chartObjectFactory.createCTBoolean();
        booleanValue.setVal(true);
        chart.setPlotVisOnly(booleanValue);
        plotArea.getAreaChartOrArea3DChartOrLineChart().add(createChartType(chartAttrs));

        /* CTCatAx
           CTSerAx
           CTValAx
           CTDateAx
        */
        List<Object> axList = plotArea.getValAxOrCatAxOrDateAx();
        axList.add(createCatAx());
        axList.add(createValAx());
        return chartSpace;
    }

    private static CTCatAx createCatAx() {
        final org.docx4j.dml.chart.ObjectFactory chartObjectFactory = Context.getDmlChartObjectFactory();
        final CTCatAx catAx = chartObjectFactory.createCTCatAx();
        final CTUnsignedInt axId = chartObjectFactory.createCTUnsignedInt();
        axId.setVal(0);
        catAx.setAxId(axId);
        final CTScaling scaling = chartObjectFactory.createCTScaling();
        final CTOrientation orientation = chartObjectFactory.createCTOrientation();
        orientation.setVal(STOrientation.MIN_MAX);
        scaling.setOrientation(orientation);
        catAx.setScaling(scaling);
        final CTAxPos axPos = chartObjectFactory.createCTAxPos();
        axPos.setVal(STAxPos.B);
        catAx.setAxPos(axPos);
        final CTUnsignedInt crossAx = chartObjectFactory.createCTUnsignedInt();
        crossAx.setVal(1);
        catAx.setCrossAx(crossAx);
        return catAx;
    }

    private static CTValAx createValAx() {
        final org.docx4j.dml.chart.ObjectFactory chartObjectFactory = Context.getDmlChartObjectFactory();
        final CTValAx valAx = chartObjectFactory.createCTValAx();
        final CTUnsignedInt axId = chartObjectFactory.createCTUnsignedInt();
        axId.setVal(1);
        valAx.setAxId(axId);
        final CTScaling scaling = chartObjectFactory.createCTScaling();
        final CTOrientation orientation = chartObjectFactory.createCTOrientation();
        orientation.setVal(STOrientation.MIN_MAX);
        scaling.setOrientation(orientation);
        valAx.setScaling(scaling);
        final CTAxPos axPos = chartObjectFactory.createCTAxPos();
        axPos.setVal(STAxPos.L);
        valAx.setAxPos(axPos);
        final CTUnsignedInt crossAx = chartObjectFactory.createCTUnsignedInt();
        crossAx.setVal(0);
        valAx.setCrossAx(crossAx);
        return valAx;
    }

    private static void createAxId(List<CTUnsignedInt> axId) {
        final org.docx4j.dml.chart.ObjectFactory chartObjectFactory = Context.getDmlChartObjectFactory();
        final CTUnsignedInt id0 = chartObjectFactory.createCTUnsignedInt();
        final CTUnsignedInt id1 = chartObjectFactory.createCTUnsignedInt();
        id0.setVal(0);
        id1.setVal(1);
        axId.add(id0);
        axId.add(id1);
    }

    // TODO: creator for pie2d, pie3d, bubble2d and donut2d needs to be implemented...

    private static ListSer createChartType(JSONObject chartAttrs) {

        ListSer listSer = null;
        final org.docx4j.dml.chart.ObjectFactory chartObjectFactory = Context.getDmlChartObjectFactory();
        String chartType = chartAttrs!=null ? chartAttrs.optString("type", "bar2d") : "bar2d";

        if(chartType.equals("bar3d")) {
            listSer = chartObjectFactory.createCTBar3DChart();
            final CTBarDir barDir = chartObjectFactory.createCTBarDir();
            ((CTBar3DChart)listSer).setBarDir(barDir);
            barDir.setVal(STBarDir.COL);
            createAxId(((CTBar3DChart)listSer).getAxId());
            final CTBarGrouping barGrouping = chartObjectFactory.createCTBarGrouping();
            barGrouping.setVal(STBarGrouping.CLUSTERED);
            ((CTBar3DChart)listSer).setGrouping(barGrouping);
        }
        else if(chartType.equals("line2d")) {
            listSer = chartObjectFactory.createCTLineChart();
            createAxId(((CTLineChart)listSer).getAxId());
            final CTGrouping grouping = chartObjectFactory.createCTGrouping();
            grouping.setVal(STGrouping.STANDARD);
            ((CTLineChart)listSer).setGrouping(grouping);
        }
        else if(chartType.equals("line3d")) {
            listSer = chartObjectFactory.createCTLine3DChart();
            createAxId(((CTLine3DChart)listSer).getAxId());
            final CTGrouping grouping = chartObjectFactory.createCTGrouping();
            grouping.setVal(STGrouping.STANDARD);
            ((CTLine3DChart)listSer).setGrouping(grouping);
        }
        else if(chartType.equals("area2d")) {
            listSer = chartObjectFactory.createCTAreaChart();
            createAxId(((CTAreaChart)listSer).getAxId());
            final CTGrouping grouping = chartObjectFactory.createCTGrouping();
            grouping.setVal(STGrouping.STANDARD);
            ((CTAreaChart)listSer).setGrouping(grouping);
        }
        else if(chartType.equals("area3d")) {
            listSer = chartObjectFactory.createCTArea3DChart();
            createAxId(((CTArea3DChart)listSer).getAxId());
            final CTGrouping grouping = chartObjectFactory.createCTGrouping();
            grouping.setVal(STGrouping.STANDARD);
            ((CTArea3DChart)listSer).setGrouping(grouping);
        }
        else if(chartType.equals("radar2d")) {
            listSer = chartObjectFactory.createCTRadarChart();
            createAxId(((CTRadarChart)listSer).getAxId());
        }
        else if(chartType.equals("scatter2d")) {
            listSer = chartObjectFactory.createCTScatterChart();
            createAxId(((CTScatterChart)listSer).getAxId());
        }
        else {  // bar2d -> default
            listSer = chartObjectFactory.createCTBarChart();
            createAxId(((CTBarChart)listSer).getAxId());
             final CTBarDir barDir = chartObjectFactory.createCTBarDir();
            ((CTBarChart)listSer).setBarDir(barDir);
            barDir.setVal(STBarDir.COL);
            final CTBarGrouping barGrouping = chartObjectFactory.createCTBarGrouping();
            barGrouping.setVal(STBarGrouping.CLUSTERED);
            ((CTBarChart)listSer).setGrouping(barGrouping);
        }
        return listSer;
    }

    public static void getChartType(CTChartSpace chartSpace, HashMap<String, Object> map) {

        if(chartSpace==null) {
            return;
        }
        CTChart chart = chartSpace.getChart();
        if(chart==null) {
            return;
        }
        CTPlotArea plotArea = chart.getPlotArea();
        if(plotArea==null) {
            return;
        }
        List<ListSer> chartList = plotArea.getAreaChartOrArea3DChartOrLineChart();
        if(chartList.isEmpty()) {
            return;
        }

        String chartType = null;
        ListSer listSer = chartList.get(0);
        if(listSer instanceof CTBubbleChart) {
            chartType = "bubble2d";
        }
        else if (listSer instanceof CTRadarChart) {
            chartType = "radar2d";
        }
        else if (listSer instanceof CTSurface3DChart) {
            chartType = "bar2d";
        }
        else if (listSer instanceof CTArea3DChart) {
            chartType = "area3d";
        }
        else if (listSer instanceof CTOfPieChart) {
            chartType = "bar2d";
        }
        else if (listSer instanceof CTAreaChart) {
            chartType = "area2d";
        }
        else if (listSer instanceof CTBarChart) {            
            CTBarChart bar = (CTBarChart)listSer;
            if(bar.getBarDir().getVal() == STBarDir.BAR){
                chartType = "bar2d";    
            }else{
                chartType = "column2d";
            }
        }
        else if (listSer instanceof CTSurfaceChart) {
            chartType = "bar2d";
        }
        else if (listSer instanceof CTLine3DChart) {
            chartType = handleLineType(listSer);
        }
        else if (listSer instanceof CTDoughnutChart) {
            chartType = "donut2d";
        }
        else if (listSer instanceof CTLineChart) {
            chartType = handleLineType(listSer);
        }
        else if (listSer instanceof CTScatterChart) {
            chartType = "scatter2d";
        }
        else if (listSer instanceof CTBar3DChart) {
            CTBar3DChart bar = (CTBar3DChart)listSer;
            if(bar.getBarDir().getVal() == STBarDir.BAR){
                chartType = "bar3d";    
            }else{
                chartType = "column3d";
            }
        }
        else if (listSer instanceof CTPieChart) {
            chartType = "pie2d";
        }
        else if (listSer instanceof CTPie3DChart) {
            chartType = "pie3d";
        }
        else if (listSer instanceof CTStockChart) {
            chartType = "bar2d";
            map.put("chartGroup", "clustered");
        }
        if(chartType!=null) {
            map.put("chartType", chartType);
        }
        String chartGroup = listSer.getGroupType();
        if(chartGroup != null){
            map.put("chartGroup", chartGroup);
        }      
    }

    private static String handleLineType(ListSer listSer) {
        String type = "line";
        try {
            final CTBoolean smooth = ((CTLineSer) listSer.getSer().get(0)).getSmooth();
            if(smooth!=null && smooth.isVal()){
                type = "spline";
            }
        } catch (Exception e) {
            Logger.getAnonymousLogger().log(Level.WARNING, e.getMessage());
        }
        if (listSer instanceof CTLine3DChart) {
            return type + "3d";
        } else {
            return type + "2d";
        }
    }
}
