/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.tools;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.xml.bind.DatatypeConverter;
import javax.xml.bind.JAXBElement;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.logging.Log;
import org.docx4j.dml.CTColor;
import org.docx4j.openpackaging.URIHelper;
import org.docx4j.openpackaging.contenttype.ContentTypeManager;
import org.docx4j.openpackaging.contenttype.ContentTypes;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.WordprocessingML.BinaryPart;
import org.docx4j.openpackaging.parts.WordprocessingML.BinaryPartAbstractImage;
import org.docx4j.openpackaging.parts.relationships.Namespaces;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.docx4j.relationships.Relationship;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.xlsx4j.sml.Cell;
import org.xlsx4j.sml.Row;
import com.openexchange.log.LogFactory;
import com.openexchange.office.ooxml.OperationDocument;
import com.openexchange.office.tools.FileHelper;
import com.openexchange.office.tools.Resource;
import com.openexchange.office.tools.ResourceManager;

final public class Commons {

    final private static Log log = LogFactory.getLog(Commons.class);

    final static String hexArray = "0123456789ABCDEF";

    public static String bytesToHexString(byte[] bytes, int offset, int size) {
        final StringBuilder hex = new StringBuilder(2 * size);
        for (int i=0; i<size; i++) {
            final byte b = bytes[offset + i];
            hex.append(hexArray.charAt((b & 0xF0) >> 4))
               .append(hexArray.charAt(b&0x0F));
        }
        return hex.toString();
    }

    public static  String ctColorToString( CTColor color ){
        final String fillString = "000000";
        byte[] rgbVal = color.getSrgbClr() != null ? color.getSrgbClr().getVal() : color.getSysClr().getLastClr();
        String ret = Integer.toHexString((((rgbVal[0]& 0xFF)<< 16) + ((rgbVal[1]& 0xFF)<< 8) + (rgbVal[2]& 0xFF)));
        if( ret.length() < 6 )
            ret = fillString.substring( ret.length() ) + ret;
        return ret;
    }

    public static byte[] ctColorToBytes(CTColor color){
        return color.getSrgbClr() != null ? color.getSrgbClr().getVal() : color.getSysClr().getLastClr();
    }

    public static byte[] hexStringToBytes(String color) {
        return DatatypeConverter.parseHexBinary(color);
    }

    public static JAXBElement<?> findElement(List<JAXBElement<?>> jaxbElements, String elementName) {
        for (int i=0; i<jaxbElements.size(); i++) {
            if (elementName.equals(jaxbElements.get(i).getName().getLocalPart())) {
                return jaxbElements.get(i);
            }
        }
        return null;
    }

    public static void removeElement(List<JAXBElement<?>> jaxbElements, String elementName) {
        for(int i=0; i<jaxbElements.size(); i++) {
            if(elementName.equals(jaxbElements.get(i).getName().getLocalPart())) {
                jaxbElements.remove(i);
            }
        }
    }

    public static void insertElement(List<JAXBElement<?>> jaxbElements, String elementName, JAXBElement<?> jaxbInsert) {
        JAXBElement<?> jaxb = findElement(jaxbElements, elementName);
        if(jaxb!=null)
            jaxb = jaxbInsert;
        else
            jaxbElements.add(jaxbInsert);
    }

    public static JSONObject surroundJSONObject(String type, JSONObject source) {
        if(source==null||source.length()==0)
            return null;
        JSONObject newJSON = new JSONObject();
        try {
            newJSON.put(type,  source);
        } catch (JSONException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        return newJSON;
    }

    public static void mergeJsonObjectIfUsed(String destinationPropertyName, JSONObject destination, JSONObject source)
        throws JSONException {

        if(source!=null&&source.length()>0) {
            if(destination.has(destinationPropertyName)) {
                Iterator<String> keys = source.keys();
                while(keys.hasNext()) {
                    String key = keys.next();
                    Object val = source.get(key);
                    destination.put(key, val);
                }
           } else
                destination.put(destinationPropertyName, source);
        }
    }

    public static void jsonPut(JSONObject destination, String destinationPropertyName, JSONObject source)
        throws JSONException {
        if(source!=null&&source.length()>0) {
            destination.put(destinationPropertyName, source);
        }
    }

    public static void jsonPut(JSONObject destination, String destinationPropertyName, JSONArray source)
        throws JSONException {
        if(source!=null&&source.length()>0) {
            destination.put(destinationPropertyName, source);
        }
    }

    public static void jsonPut(JSONObject destination, String destinationPropertyName, String source)
        throws JSONException {
        if(source!=null&&source.length()>0) {
            destination.put(destinationPropertyName, source);
        }
    }

    public static void jsonPut(JSONObject destination, String destinationPropertyName, Object source)
        throws JSONException {
        if(source!=null) {
            destination.put(destinationPropertyName, source);
        }
    }

    public static void jsonPut(JSONArray destination, String source) {
        if(source!=null&&source.length()>0) {
            destination.put(source);
        }
    }

    public static final Map<String, String> highlightColorToRgb = createHighlightColorsMap();
    public static final double themeTransformFactor = 100000.0 / 255.0;
    public static final double themeTransformDiv = 255.0 / 100000.0;

    private static Map<String, String> createHighlightColorsMap() {
        // highlight color map according to ooxml specification
        Map<String, String> result = new HashMap<String, String>();
        result.put( "black", "000000");
        result.put( "blue", "0000FF");
        result.put( "cyan", "00FFFF");
        result.put( "darkBlue", "00008B");
        result.put( "darkCyan", "008B8B");
        result.put( "darkGray", "A9A9A9");
        result.put( "darkGreen", "006400");
        result.put( "darkMagenta", "800080");
        result.put( "darkRed", "8B0000");
        result.put( "darkYellow", "808000");
        result.put( "green", "00FF00");
        result.put( "lightGray", "D3D3D3");
        result.put( "magenta", "FF00FF");
        result.put( "none", "");
        result.put( "red", "FF0000");
        result.put( "white", "FFFFFF");
        result.put( "yellow", "FFFF00");
        return Collections.unmodifiableMap(result);
    }

    private static int rgbStringToInt(String rgbColor) {
        return Integer.parseInt(rgbColor, 16);
    }

    public static String mapHightlightColorToRgb(String highlightColor) {
        String rgbColor = null;

        Map<String, String> map = Commons.highlightColorToRgb;
        rgbColor = map.get(highlightColor);

        return rgbColor;
    }

    private static int colorDiff(int rgbColor1, int rgbColor2) {
        int r1 = ((rgbColor1 & 0x00ff0000) >> 16);
        int g1 = ((rgbColor1 & 0x0000ff00) >> 8);
        int y1 = (rgbColor1 & 0x000000ff);
        int r2 = ((rgbColor2 & 0x00ff0000) >> 16);
        int g2 = ((rgbColor2 & 0x0000ff00) >> 8);
        int y2 = (rgbColor2 & 0x000000ff);

        return Math.abs(r1-r2) + Math.abs(g1-g2) + Math.abs(y1-y2);
    }

    public static String mapRgbToPredefined(String rgbColor, Map<String, String> mappings) {
        String predefinedColor = null;
        int diff = Integer.MAX_VALUE;;
        int rgbColorValue = Integer.parseInt(rgbColor);

        if (mappings != null) {
            Set<String> keys = mappings.keySet();
            for (String color : keys) {
                int colorValue = Commons.rgbStringToInt(color);
                int currDiff = Commons.colorDiff(rgbColorValue, colorValue);
                if (currDiff < diff) {
                    diff = currDiff;
                    predefinedColor = color;
                    if (diff == 0)
                        return predefinedColor;
                }
            }
        }

        return predefinedColor;
    }

    public static class CellRangeIterator implements Iterator<org.xlsx4j.sml.Cell> {

        public CellRangeIterator(int startColumn, int endColumn, CellIterator cellIterator) {
            this.startColumn = startColumn;
            this.endColumn = endColumn;
            this.cellIterator = cellIterator;
            this.nextCell = getNext();
        }

        @Override
        public org.xlsx4j.sml.Cell next() {
            final org.xlsx4j.sml.Cell cell = nextCell;
            nextCell = getNext();
            return cell;
        }

        @Override
        public boolean hasNext() {
            return nextCell!=null;
        }

        @Override
        public void remove() {
            cellIterator.remove();
        }

        private org.xlsx4j.sml.Cell getNext() {

            while(cellIterator.hasNext()) {
                final org.xlsx4j.sml.Cell cell = cellIterator.next();
                final int currentColumnNumber = cellIterator.getColumnNumber();
                if(currentColumnNumber>endColumn) {
                    return null;
                }
                else if (currentColumnNumber>=startColumn) {
                    return cell;
                }
            }
            return null;
        }

        private org.xlsx4j.sml.Cell nextCell;

        private final int startColumn;

        private final int endColumn;

        private final CellIterator cellIterator;
    }

    public static class CellIterator implements Iterator<org.xlsx4j.sml.Cell> {

        public CellIterator(List<org.xlsx4j.sml.Cell> list) {
            this.list = list;
        }
        public CellIterator(List<org.xlsx4j.sml.Cell> list, Map<Integer, Cell> cellMap) {
            this.list = list;
            this.cellMap = cellMap;
            removeSupported = true;
        }
        public int getColumnNumber() {
            return columnNumber;
        }
        public int getListIndex() {
            return listIndex;
        }
        @Override
        public org.xlsx4j.sml.Cell next() {
            cell = list.get(++listIndex);
            final String ref = cell.getR();
            if(ref==null) {
                columnNumber++;
            }
            else {
                org.xlsx4j.sml.Cell.CellRef cr = org.xlsx4j.sml.Cell.createCellRef(ref);
                int index = cr.getColumn();
                if(index<columnNumber) {
                    throw new RuntimeException("XLSX: CellIterator, invalid cell index found!!");
                }
                columnNumber = index;
            }
            return cell;
        }
        @Override
        public boolean hasNext() {
            return (listIndex+1) < list.size();
        }
        @Override
        public void remove() {
            if(!removeSupported||cell==null) {
                throw new UnsupportedOperationException();
            }
            if(cellMap!=null) {
                cellMap.remove(cell);
            }
            list.remove(listIndex--);
            columnNumber--;
        }

        private final List<org.xlsx4j.sml.Cell> list;

        private Map<Integer, Cell> cellMap = null;

        private org.xlsx4j.sml.Cell cell = null;

        private boolean removeSupported = false;

        private int columnNumber = -1;

        private int listIndex = -1;
    }

    public static class RowRangeIterator implements Iterator<org.xlsx4j.sml.Row> {

        public RowRangeIterator(int startRow, int endRow, RowIterator rowIterator) {
            this.startRow = startRow;
            this.endRow = endRow;
            this.rowIterator = rowIterator;
            this.nextRow = getNext();
        }

        @Override
        public org.xlsx4j.sml.Row next() {
            final org.xlsx4j.sml.Row row = nextRow;
            nextRow = getNext();
            return row;
        }

        @Override
        public boolean hasNext() {
            return nextRow!=null;
        }

        @Override
        public void remove() {
            rowIterator.remove();
        }

        private org.xlsx4j.sml.Row getNext() {

            while(rowIterator.hasNext()) {
                final org.xlsx4j.sml.Row row = rowIterator.next();
                final int currentRowNumber = rowIterator.getRowNumber();
                if(currentRowNumber>endRow) {
                    return null;
                }
                else if (currentRowNumber>=startRow) {
                    return row;
                }
            }
            return null;
        }

        private org.xlsx4j.sml.Row nextRow;

        private final int startRow;

        private final int endRow;

        private final RowIterator rowIterator;
    }

    public static class RowIterator implements Iterator<org.xlsx4j.sml.Row> {

        public RowIterator(List<org.xlsx4j.sml.Row> list) {
            this.list = list;
        }
        public RowIterator(List<org.xlsx4j.sml.Row> list, Map<Integer, Row> rowMap) {
            this.list = list;
            this.rowMap = rowMap;
            removeSupported = true;
        }
        public int getRowNumber() {
            return rowNumber;
        }
        public int getListIndex() {
            return listIndex;
        }
        @Override
        public org.xlsx4j.sml.Row next() {
            row = list.get(++listIndex);
            final Long r = row.getR();
            if(r==null) {
                rowNumber++;
            }
            else {
                int index = r.intValue() - 1;
                if(index<0||index<rowNumber) {
                    throw new RuntimeException("XLSX: RowIterator, invalid row index found!!");
                }
                rowNumber = index;
            }
            return row;
        }
        @Override
        public boolean hasNext() {
            return (listIndex+1) < list.size();
        }
        @Override
        public void remove() {
            if(!removeSupported||row==null) {
                throw new UnsupportedOperationException();
            }
            if(rowMap!=null) {
                rowMap.remove(row);
            }
            list.remove(listIndex--);
            rowNumber--;
        }

        private final List<org.xlsx4j.sml.Row> list;

        private Map<Integer, Row> rowMap = null;

        private org.xlsx4j.sml.Row row = null;

        boolean removeSupported = false;

        private int rowNumber = -1;

        private int listIndex = -1;
    }

    public static class Pair<T, E> {

        private T first;
        private E second;

        public Pair(T first, E second) {
            this.first = first;
            this.second = second;
        }
        public T getFirst() {
            return first;
        }
        public E getSecond() {
            return second;
        }
        public void setFirst(T first) {
            this.first = first;
        }
        public void setSecond(E second) {
            this.second = second;
        }
    }

    public static long coordinateTo100TH_MM(long coordinate) {
        return (coordinate + 180)/360;
    }
    public static long coordinateTo100TH_MM(String coordinate) {

        try {
            // ST_CoordinateUnqualified
            return (Long.parseLong(coordinate)+180)/360;
        }
        catch(NumberFormatException e) {
            // TODO: ST_UniversalMeasure -> 'cm', 'mm', 'in', 'pt', 'pc' and 'pi'
            return 0;
        }
    }

    public static long coordinateFrom100TH_MM(long mm100th) {
        return mm100th*360;
    }

    /**
     * @param part
     * @param rId
     * @return the targetUrl of the corresponding relationship
     */
    public static String getUrl(Part part, String rId) {

        String url = "";

        if(part!=null&&rId!=null&&rId.length()>0) {

            final RelationshipsPart relationshipsPart = part.getRelationshipsPart();
            if(relationshipsPart!=null) {

                final Relationship relationShip = relationshipsPart.getRelationshipByID(rId);
                if(relationShip!=null) {

                    try {

                        final String targetMode = relationShip.getTargetMode();
                        if(targetMode!=null&&targetMode.equals("External"))
                            url = relationShip.getTarget();
                        else
                            url = URIHelper. resolvePartUri(part.getPartName().getURI(), new URI(relationShip.getTarget())).getPath();

                        if(url.charAt(0)=='/') {
                            url = url.substring(1);
                        }

                    } catch(Exception e) {
                        //
                    }
                }
            }
        }
        return url;
    }

    /**
     * sets an external target url to the relationship. if rId is null or empty, a new hyperlink relationship
     * will be created, otherwise the existing relationship will be reused.
     *
     * @param part
     * @param rId
     * @param Url
     * @return rId
     */
    public static String setUrl(Part part, String rId, String Url) {

        if(part!=null) {

            final RelationshipsPart relationshipsPart = part.getRelationshipsPart();
            if(relationshipsPart!=null) {
                Relationship relationShip = null;
                if(rId==null||rId.length()==0) {
                    final org.docx4j.relationships.ObjectFactory factory =
                        new org.docx4j.relationships.ObjectFactory();
                    relationShip = factory.createRelationship();
                    relationShip.setType(Namespaces.HYPERLINK);
                    relationshipsPart.addRelationship(relationShip);
                    rId = relationShip.getId();
                }
                else
                    relationShip = relationshipsPart.getRelationshipByID(rId);

                if(relationShip!=null) {
                    relationShip.setTarget(Url);
                    relationShip.setTargetMode("External");
                }
            }
        }
        return rId;
    }

    // image Folder: "word/media/"
    public static String getImageUrlFromImageData(ResourceManager resourceManager, String imageData, String imageFolder) {
        String imageUrl = null;
        if(imageData != null) {
            long uid = resourceManager.addBase64(imageData);
            if(uid!=0) {
                imageUrl = imageFolder + "uid" + Long.toHexString(uid);
                final String basePattern = "base64,";
                final String dataPattern = "data:image/";

                int pos = imageData.indexOf(basePattern);
                if (imageData.startsWith(dataPattern)) {
                    imageUrl += '.' + imageData.substring(dataPattern.length(), pos - 1);
                }
            }
        }
        return imageUrl;
    }

    public static byte[] retrieveResource(String imageUrl, ResourceManager resourceManager) {

        byte[] buf = null;
        Resource resource = null;
        final String filename = FilenameUtils.getBaseName(imageUrl);
        if(filename.startsWith("uid")) {
            try {
                long uid = Long.parseLong(filename.substring(3), 16);
                resource = resourceManager.getResource(uid);
            }
            catch(NumberFormatException e) {
                //
            }
        }
        if(resource!=null) {
            buf = resource.getBuffer();

            if(buf==null) {
                log.error("ooxml export: could not get buffer of resource " + filename);
            }
        }
        return buf;
    }

    /**
    *
    * @param part (for docx the uri of the part is something like: word/document.xml)
    * @param url (should be like word/media/xxx.png)
    * @return the relationship of an url, the relationship is null if there is no relationship available
    */
   public static Relationship getRelationship(Part part, String _url) {

       if(part==null||_url==null||_url.isEmpty())
           return null;

       try {

           final boolean isAbsolute = new URI(_url).isAbsolute();
           final String url = isAbsolute ? _url : '/' + _url;

           final RelationshipsPart relationshipsPart = part.getRelationshipsPart();
           if(relationshipsPart!=null) {
               final List<Relationship> relationships = relationshipsPart.getRelationships().getRelationship();
               for(Relationship rel:relationships) {
                   try {
                       if(rel.getTarget()!=null) {
                           if(!isAbsolute&&(rel.getTargetMode()==null||rel.getTargetMode().equalsIgnoreCase("Internal"))) {
                               final String targetPath = URIHelper.resolvePartUri(part.getPartName().getURI(), new URI(rel.getTarget())).getPath();
                               if(targetPath!=null&&targetPath.equals(url)) {
                                   return rel;
                               }
                           }
                           else if(isAbsolute&&rel.getTarget().equals(url)) {
                               return rel;
                           }
                       }
                   } catch (Exception e) {
                   // TODO Auto-generated catch block
                   }
               }
           }
       } catch (URISyntaxException e1) {
           // TODO Auto-generated catch block
       }
       return null;
   }

   public static Relationship createAndApplyGraphic(OperationDocument operationDocument, Part part, String imageUrl)
       throws InvalidFormatException, PartUnrecognisedException {

       // check if the image is already part of the document
       Relationship rel = Commons.getRelationship(part, imageUrl);
       if(rel==null) {

           // Ensure the relationships part exists
           RelationshipsPart relationshipsPart = part.getRelationshipsPart();
           if(relationshipsPart==null) {
               relationshipsPart = RelationshipsPart.createRelationshipsPartForPart(part);
           }

           final String proposedRelId = part.getRelationshipsPart().getNextId();
           final ContentTypeManager ctm = operationDocument.getPackage().getContentTypeManager();

           String contentType = ContentTypes.IMAGE_JPEG;
           final String extension = FileHelper.getExtension(imageUrl);
           if(extension.length()>0) {
               if (extension.equals(ContentTypes.EXTENSION_JPG_1))
                   contentType = ContentTypes.IMAGE_JPEG;
               else if(extension.equals(ContentTypes.EXTENSION_JPG_2))
                   contentType = ContentTypes.IMAGE_JPEG;
               else if(extension.equals(ContentTypes.EXTENSION_PNG))
                   contentType = ContentTypes.IMAGE_PNG;
               else if(extension.equals(ContentTypes.EXTENSION_GIF))
                   contentType = ContentTypes.IMAGE_GIF;
               else if(extension.equals(ContentTypes.EXTENSION_BMP))
                   contentType = ContentTypes.IMAGE_BMP;
               else if(extension.equals(ContentTypes.EXTENSION_EMF))
                   contentType = ContentTypes.IMAGE_EMF;
               else if(extension.equals(ContentTypes.EXTENSION_WMF))
                   contentType = ContentTypes.IMAGE_WMF;
               else if(extension.equals(ContentTypes.EXTENSION_TIFF))
                   contentType = ContentTypes.IMAGE_TIFF;
               else if(extension.equals(ContentTypes.EXTENSION_PICT))
                   contentType = ContentTypes.IMAGE_PICT;
           }

           final byte[] imageData = Commons.retrieveResource(imageUrl, operationDocument.getResourceManager());
           if(imageData==null) {

               // we will create an external link to the graphic
               Part imagePart = ctm.newPartForContentType(contentType, BinaryPartAbstractImage.createImageName(operationDocument.getPackage(), part, proposedRelId, "ext"), null);
               rel = part.addTargetPart(imagePart);
               rel.setTargetMode("External");
               if(imagePart instanceof BinaryPart)
                   operationDocument.getPackage().getExternalResources().put(((BinaryPart)imagePart).getExternalTarget(), imagePart);
               rel.setTarget(imageUrl.toString());
           }
           else {

               // the graphic is embedded and will be inserted added to the zip archive
               Part imagePart = ctm.newPartForContentType(contentType, "/" + imageUrl, null);
               if(imagePart instanceof BinaryPart)
                   ((BinaryPart)imagePart).setBinaryData(imageData);
               if(imagePart!=null)
                   rel = part.addTargetPart(imagePart, proposedRelId);
           }
       }
       return rel;
   }
}
