/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.tools;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.docx4j.XmlUtils;
import org.docx4j.jaxb.Context;
import org.json.JSONException;
import org.json.JSONObject;
import org.xlsx4j.sml.CTCellStyleXfs;
import org.xlsx4j.sml.CTStylesheet;
import org.xlsx4j.sml.CTXf;
import org.xlsx4j.sml.Row;
import com.openexchange.office.ooxml.xlsx.OperationDocument;
import com.openexchange.office.ooxml.xlsx.operations.ApplyOperationHelper;

public class RowUtils {

    public static void applyRowProperties(OperationDocument operationDocument, JSONObject rowProperties, Row row, CTStylesheet stylesheet)
        throws  JSONException {

        if(rowProperties.has("styleId")) {
            Object value = rowProperties.get("styleId");
            if(value instanceof String) {
                final ApplyOperationHelper applyOperationHelper = operationDocument.getApplyOperationHelper();
                if(applyOperationHelper!=null) {
                    row.setS(applyOperationHelper.getStyleIndex((String)value));
                }
            }
        }
        // Row properties
        JSONObject jsonRowProperties = rowProperties.optJSONObject("row");
        if(jsonRowProperties!=null) {
            Iterator<String> keys = jsonRowProperties.keys();
            while(keys.hasNext()) {
                String attr = keys.next();
                Object value = jsonRowProperties.get(attr);
                if(attr.equals("height")) {
                    Double ht = null;
                    if(value instanceof Integer) {
                        ht = ((Integer)value).doubleValue()*72/2540.0;
                    }
                    row.setHt(ht);
                }
                else if(attr.equals("customHeight")) {
                    Boolean customHeight = null;
                    if(value instanceof Boolean) {
                        customHeight = (Boolean)value;
                    }
                    row.setCustomHeight(customHeight);
                }
                else if(attr.equals("visible")) {
                    Boolean hidden = null;
                    if(value instanceof Boolean) {
                        hidden = !(Boolean)value;
                    }
                    row.setHidden(hidden);
                }
                else if(attr.equals("customFormat")) {
                    Boolean customFormat = null;
                    if(value instanceof Boolean) {
                        customFormat = (Boolean)value;
                    }
                    row.setCustomFormat(customFormat);
                }
            }
        }

        // cell & character properties
        JSONObject jsonCellProperties = rowProperties.optJSONObject("cell");
        JSONObject jsonCharacterProperties = rowProperties.optJSONObject("character");

        if(jsonCellProperties!=null||jsonCharacterProperties!=null) {

            final CTXf xfSource = stylesheet.getCellXfByIndex(row.getS());
            final CTXf detachedXf = xfSource==null?Context.getsmlObjectFactory().createCTXf():XmlUtils.deepCopy(xfSource, Context.getJcSML());

            // character properties
            final Long xfId = detachedXf.getXfId();
            CTXf cellStyleXf = null;
            if(xfId!=null) {
                final CTCellStyleXfs cellStyleXfs = stylesheet.getCellStyleXfs();
                if(cellStyleXfs!=null) {
                    final List<CTXf> cellStyleXfsList = cellStyleXfs.getXf();
                    if(xfId>=0&&xfId<cellStyleXfsList.size()) {
                        cellStyleXf = cellStyleXfsList.get(xfId.intValue());
                    }
                }
            }

            if(jsonCellProperties!=null) {
                CellUtils.applyCellProperties(operationDocument, jsonCellProperties, detachedXf, stylesheet, cellStyleXf);
            }
            if(jsonCharacterProperties!=null) {
                CellUtils.applyCharacterProperties(operationDocument, null, null, jsonCharacterProperties, detachedXf, stylesheet, cellStyleXf);
            }

            long index = stylesheet.getOrApplyCellXf(detachedXf);
            if(index>0) {
                row.setS(index);
            }
        }
        return;
    }

    public static boolean compareRowProperties(Row r1, Row r2) {
        if(r1.getS()!=r2.getS()) {
            return false;
        }
        if(r1.getHt()!=null&&r2.getHt()!=null) {
            if(!r1.getHt().equals(r2.getHt())) {
                return false;
            }
        }
        else if(r1.getHt()!=null||r2.getHt()!=null) {
            return false;
        }
        if(r1.isCustomHeight()!=r2.isCustomHeight()) {
            return false;
        }
        if(r1.isHidden()!=r2.isHidden()) {
            return false;
        }
        if(r1.isCustomFormat()!=r2.isCustomFormat()) {
            return false;
        }
        return true;
    }

    public static JSONObject createRowProperties(Row row, CTStylesheet stylesheet, Map<Long, String> autoStyleToStyleMap)
        throws JSONException {

        JSONObject attrs = new JSONObject();

        // row
        JSONObject rowProperties = new JSONObject();
        Double rowHeight = row.getHt();
        if(rowHeight!=null) {
            rowProperties.put("height", (int)((rowHeight*2540.0/72)+0.5));
        }
        if(row.isCustomHeight()) {
            rowProperties.put("customHeight", true);
        }
        if(row.isHidden()) {
            rowProperties.put("visible", false);
        }
        if(row.isCustomFormat()) {

            rowProperties.put("customFormat", true);
        }
        if(rowProperties.length()>0) {
            attrs.put("row", rowProperties);
        }
        long s = row.getS();
        if(s>=0) {
            String style = autoStyleToStyleMap.get(s);
// TODO: if(style==null)
                style = "a" + Long.toString(s);
            attrs.put("styleId", style);
        }
/*
        if(s>0) {
            CTXf xf = stylesheet.getCellXfByIndex(s);
            if(xf!=null) {
                // creating style attribute (0 is default and though not created)
                Long xfId = xf.getXfId();
                if(xfId!=null&&xfId!=0) {
                    attrs.put("styleId", Long.toString(xfId));
                }
                Commons.mergeJsonObjectIfUsed("character", attrs, CellUtils.createCharacterProperties(xf, stylesheet, false));
                Commons.mergeJsonObjectIfUsed("cell", attrs, CellUtils.createCellProperties(xf, stylesheet, false));
            }
        }
*/
        if(attrs.length()==0) {
            return null;
        }
        return attrs;
    }
}
