/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.operations;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import javax.xml.bind.JAXBException;
import org.apache.commons.logging.Log;
import org.docx4j.dml.chart.CTChartSpace;
import org.docx4j.dml.spreadsheetdrawing.CellAnchor;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.packages.SpreadsheetMLPackage;
import org.docx4j.openpackaging.parts.PartName;
import org.docx4j.openpackaging.parts.DrawingML.Drawing;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorkbookPart;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorksheetPart;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart.AddPartBehaviour;
import org.docx4j.relationships.Relationship;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.xlsx4j.sml.CTCellFormula;
import org.xlsx4j.sml.CTCellStyle;
import org.xlsx4j.sml.CTCellStyleXfs;
import org.xlsx4j.sml.CTCellStyles;
import org.xlsx4j.sml.CTCellXfs;
import org.xlsx4j.sml.CTDefinedName;
import org.xlsx4j.sml.CTMergeCell;
import org.xlsx4j.sml.CTMergeCells;
import org.xlsx4j.sml.CTStylesheet;
import org.xlsx4j.sml.CTXf;
import org.xlsx4j.sml.Cell;
import org.xlsx4j.sml.Cell.CellRefRange;
import org.xlsx4j.sml.Col;
import org.xlsx4j.sml.Cols;
import org.xlsx4j.sml.DefinedNames;
import org.xlsx4j.sml.Row;
import org.xlsx4j.sml.STCellFormulaType;
import org.xlsx4j.sml.STCellType;
import org.xlsx4j.sml.Sheet;
import org.xlsx4j.sml.SheetData;
import org.xlsx4j.sml.Sheets;
import org.xlsx4j.sml.Workbook;
import org.xlsx4j.sml.Worksheet;
import com.openexchange.log.LogFactory;
import com.openexchange.office.ooxml.drawingml.DMLChartSpace;
import com.openexchange.office.ooxml.drawingml.DMLGraphic;
import com.openexchange.office.ooxml.tools.Commons;
import com.openexchange.office.ooxml.xlsx.OperationDocument;
import com.openexchange.office.ooxml.xlsx.tools.CellUtils;
import com.openexchange.office.ooxml.xlsx.tools.ChartUtils;
import com.openexchange.office.ooxml.xlsx.tools.ChartUtils.ColRow;
import com.openexchange.office.ooxml.xlsx.tools.ColumnUtils;
import com.openexchange.office.ooxml.xlsx.tools.Drawings;
import com.openexchange.office.ooxml.xlsx.tools.RowUtils;
import com.openexchange.office.ooxml.xlsx.tools.SheetUtils;
import com.openexchange.office.ooxml.xlsx.tools.Utils;

public class ApplyOperationHelper extends com.openexchange.office.ooxml.operations.ApplyOperationHelper {

    protected static Log log = LogFactory.getLog(ApplyOperationHelper.class);

    private final OperationDocument operationDocument;
    private final SpreadsheetMLPackage spreadsheetMLPackage;
    private final WorkbookPart workbookPart;
    private final Map<String, CTCellStyle> styleMap;

    public ApplyOperationHelper(OperationDocument _operationDocument) {
        super();

        operationDocument = _operationDocument;
        spreadsheetMLPackage = _operationDocument.getPackage();
        workbookPart = spreadsheetMLPackage.getWorkbookPart();
        styleMap = BuiltinIds.getUsedStyles(operationDocument.getStylesheet(false));
    }

    @Override
    public OperationDocument getOperationDocument() {
        return operationDocument;
    }

    public static void setCellContent(Cell cell, Object cellValue, String cellRef, Long cellSi) {
        if(cellValue instanceof String) {
            String v = (String)cellValue;
            if(v.length()>0&&v.startsWith("=")) {
                cell.setT(null);
                CTCellFormula cellFormula = Context.getsmlObjectFactory().createCTCellFormula();
                cell.setF(cellFormula);
                if(cellSi==null) {
                    cellFormula.setT(STCellFormulaType.NORMAL);
                    cellFormula.setValue(v.substring(1));
                }
                else {
                    cellFormula.setT(STCellFormulaType.SHARED);
                    if(cellRef!=null) {
                        cellFormula.setValue(v.substring(1));
                        cellFormula.setRef(cellRef);
                    }
                    cellFormula.setSi(cellSi);
                }
            }
            else if(v.length()>0&&v.startsWith("#")) {
                cell.setT(STCellType.E);
                cell.setV(v.substring(1));
                cell.setF(null);
            }
            else {
                cell.setT(STCellType.STR);
                cell.setV(v.startsWith("'") ? v.substring(1) : v);
                cell.setF(null);
            }
        }
        else if(cellValue instanceof Number) {
            cell.setT(STCellType.N);
            cell.setV(((Number)cellValue).toString());
            cell.setF(null);
        }
        else if(cellValue instanceof Boolean) {
            cell.setT(STCellType.B);
            cell.setV((Boolean)cellValue ? "1" : "0");
            cell.setF(null);
        }
        else if(cellValue==JSONObject.NULL) {
            // empty cell ->
            cell.setT(null);
            cell.setF(null);
            cell.setIs(null);
            cell.setV(null);
        }
    }

    private WorksheetPart getWorksheetPart(int sheetIndex) {
        Sheet sheet = SheetUtils.getSheet(spreadsheetMLPackage, sheetIndex);
        RelationshipsPart relationshipPart = spreadsheetMLPackage.getWorkbookPart().getRelationshipsPart();
        return (WorksheetPart)relationshipPart.getPart(sheet.getId());
    }

    private Worksheet getWorksheet(int sheetIndex) {
        return getWorksheetPart(sheetIndex).getJaxbElement();
    }

    public void setCellContents(int sheetIndex, JSONArray start, JSONArray rowArray)
        throws Exception {

        if(start.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::setCellContents: size of start parameter != 2");

        WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        Worksheet worksheet = worksheetPart.getJaxbElement();
        worksheet.setDimension(null);
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            sheetData = Context.getsmlObjectFactory().createSheetData();
            worksheet.setSheetData(sheetData);
        }
        int y = start.getInt(1);
        for(int i=0; i<rowArray.length(); i++) {
            JSONArray cellArray = rowArray.getJSONArray(i);
            int x = start.getInt(0);
            for(int j=0; j<cellArray.length(); j++) {

                Row row = sheetData.getRow(y, true);
                Cell cell = row.getCell(x, true);

                JSONObject cellObject = cellArray.getJSONObject(j);
                Iterator<String> keys = cellObject.keys();
                while(keys.hasNext()) {
                    String attr = keys.next();
                    if (attr.equals("value")) {
                        setCellContent(cell, cellObject.get(attr), null, null);
                    }
                    else if(attr.equals("attrs")) {
                        CellUtils.applyCellProperties(operationDocument, worksheetPart, cellObject.optJSONObject(attr), cell, getOperationDocument().getStylesheet(true));
                    }
/*
                    else if(attr.equals("shared")) {

                    }
                    else if(attr.equals("ref")) {

                    }
*/
                }
                x++;
            }
            y++;
        }
    }

    private static boolean isClearAttributes(JSONObject optAttrs) {
        if(optAttrs!=null) {
            if(optAttrs==JSONObject.NULL) {
                return true;
            }
            final JSONObject cellAttr = optAttrs.optJSONObject("cell");
            final JSONObject characterAttr = optAttrs.optJSONObject("character");
            if(cellAttr!=null&&cellAttr.length()>=12&&characterAttr!=null&&characterAttr.length()>=9) {
                final Set<Entry<String, Object>> cellEntries = cellAttr.entrySet();
                final Iterator<Entry<String, Object>> cellEntryIter = cellEntries.iterator();
                while(cellEntryIter.hasNext()) {
                    final Entry<String, Object> cellEntry = cellEntryIter.next();
                    if(cellEntry.getValue()!=JSONObject.NULL) {
                        return false;
                    }
                }
                final Set<Entry<String, Object>> characterEntries = characterAttr.entrySet();
                final Iterator<Entry<String, Object>> characterEntryIter = characterEntries.iterator();
                while(characterEntryIter.hasNext()) {
                    final Entry<String, Object> characterEntry = characterEntryIter.next();
                    if(characterEntry.getValue()!=JSONObject.NULL) {
                        return false;
                    }
                }
                return true;
            }
        }
        return false;
    }

    public void fillCellRange(int sheetIndex, JSONArray start, JSONArray optEnd, Object optValue, JSONObject optAttrs, int optShared, JSONArray optRef)
        throws Exception {

        if(start.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::fillCellContents: size of start parameter != 2");

        if(optEnd==null)
            optEnd = start;
        else if (optEnd.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::fillCellContents: size of end parameter != 2");

        WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        Worksheet worksheet = worksheetPart.getJaxbElement();
        worksheet.setDimension(null);
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            sheetData = Context.getsmlObjectFactory().createSheetData();
            worksheet.setSheetData(sheetData);
        }

        final boolean clearContent = optValue!=null&&optValue==JSONObject.NULL;
        final boolean clearAttributes = isClearAttributes(optAttrs);
        final boolean setContent = (optValue!=null)&&!clearContent;
        final boolean setAttributes = (optAttrs!=null)&&!clearAttributes;

        // check if we have to create cells for the fillRange..
        if(setContent||setAttributes) {

            final Long cellSi = optRef!=null ? sheetData.getNextSi() : null;
            for(int y=start.getInt(1); y<=optEnd.getInt(1); y++) {
                for(int x=start.getInt(0); x<=optEnd.getInt(0); x++) {

                    Row row = sheetData.getRow(y, true);
                    Cell cell = row.getCell(x, true);

                    if(setContent) {
                        if(optRef==null) {
                            setCellContent(cell, optValue, null, null);
                        }
                        else {
                            final String cellRef = optRef.getInt(0)==x&&optRef.getInt(1)==y ? Cell.getCellRefRange(start.getInt(0), start.getInt(1), optEnd.getInt(0), optEnd.getInt(1)) : null;
                            setCellContent(cell, optValue, cellRef, cellSi);
                        }
                    }

                    if(setAttributes) {
                        CellUtils.applyCellProperties(operationDocument, worksheetPart, optAttrs, cell, getOperationDocument().getStylesheet(true));
                    }
                }
            }
        }
        if(clearContent||clearAttributes) {
            // only deleting cell content or cell attrs or both ... we are using another iterator
            clearCellRange(sheetIndex, start, optEnd, clearContent, clearAttributes);
        }
    }

    public void clearCellRange(int sheetIndex, JSONArray start, JSONArray optEnd, boolean clearContent, boolean clearAttributes)
        throws Exception {

        if(!clearContent&&!clearAttributes) {
            return;
        }
        WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        Worksheet worksheet = worksheetPart.getJaxbElement();
        worksheet.setDimension(null);
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            return;
        }
        if(start.length()!=2)
            throw new RuntimeException("xlsx::ApplyOperationHelper::clearCellContents: size of start parameter != 2");

        if(optEnd==null)
            optEnd = start;
        else if (optEnd.length()!=2)
            throw new RuntimeException("xlsx::ApplyOperationHelper::clearCellContents: size of end parameter != 2");

        final Commons.RowRangeIterator rowRangeIterator = new Commons.RowRangeIterator(start.getInt(1), optEnd.getInt(1), sheetData.createRowIterator());
        while(rowRangeIterator.hasNext()) {
            final Row row = rowRangeIterator.next();
            final Commons.CellRangeIterator cellRangeIterator = new Commons.CellRangeIterator(start.getInt(0), optEnd.getInt(0), row.createCellIterator());
            while(cellRangeIterator.hasNext()) {
                final Cell cell = cellRangeIterator.next();
                if(clearContent&&clearAttributes) {
                    cellRangeIterator.remove();
                }
                else if(clearContent) {
                    setCellContent(cell, JSONObject.NULL, null, null);
                }
                else if(clearAttributes) {
                    // TODO: needs to be implemented
                }
            }
        }
    }

    public void setRowAttributes(int sheetIndex, int start, int optEnd, JSONObject attrs)
        throws Exception {

        if(optEnd==-1)
            optEnd = start;

        Worksheet worksheet = getWorksheet(sheetIndex);
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            sheetData = Context.getsmlObjectFactory().createSheetData();
            worksheet.setSheetData(sheetData);
        }
        for(int y=start; y<=optEnd; y++) {
            SheetUtils.applyRowProperties(getOperationDocument(), sheetData, start, attrs);
            RowUtils.applyRowProperties(getOperationDocument(), attrs, sheetData.getRow(y, true), getOperationDocument().getStylesheet(true));
        }
    }

    public void setColumnAttributes(int sheetIndex, int start, int optEnd, JSONObject attrs)
        throws Exception {

        if(optEnd==-1)
            optEnd = start;

        Worksheet worksheet = getWorksheet(sheetIndex);
        List<Cols> colsList = worksheet.getCols();
        if(colsList.size()==0) {
            colsList.add(Context.getsmlObjectFactory().createCols());
        }

        // TODO: we only apply our attributes to the first Cols within the cols list, so additional
        // entries have to be merged ...
        start++;
        optEnd++;
        Cols cols = colsList.get(0);
        Utils.createColumnAttributeRanges(cols, Long.valueOf(start), Long.valueOf((optEnd-start)+1));
        for(Col col:cols.getCol()) {
            if(col.getMin()>optEnd)
                return;
            if(col.getMax()<start)
                continue;
            SheetUtils.applyColumnProperties(getOperationDocument(), worksheet.getSheetData(), (int)(col.getMin()-1), (int)(col.getMax()-1), attrs);
            ColumnUtils.applyColumnProperties(getOperationDocument(), attrs, col, getOperationDocument().getStylesheet(true));
        }
    }

    public void insertRows(int sheetIndex, int start, int optEnd)
        throws Exception {

        if(start<0) {
            throw new RuntimeException("xlsx::insertRows: start<0");
        }
        int count = 1;
        if(optEnd!=-1) {
            if(optEnd<start) {
                throw new RuntimeException("xlsx::insertRows: optEnd<start");
            }
            count = (optEnd-start)+1;
        }
        WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        Worksheet worksheet = worksheetPart.getJaxbElement();
        worksheet.setDimension(null);
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            sheetData = Context.getsmlObjectFactory().createSheetData();
            worksheet.setSheetData(sheetData);
        }
        sheetData.insertRows(getOperationDocument().getStylesheet(true), start, count);

        // taking care of merged cells
        CTMergeCells mergeCells = worksheet.getMergeCells();
        if(mergeCells!=null) {
            List<CTMergeCell> mergeCellList = mergeCells.getMergeCell();
            for(CTMergeCell mergeCell:mergeCellList) {
                CellRefRange cellRefRange = Cell.createCellRefRange(mergeCell.getRef());
                int startRow = cellRefRange.getStart().getRow();
                int endRow = cellRefRange.getEnd().getRow();
                if(start<=startRow) {
                    cellRefRange.getStart().setRow(startRow+count);
                    cellRefRange.getEnd().setRow(endRow+count);
                    mergeCell.setRef(Cell.getCellRefRange(cellRefRange));
                }
                else if(start<=endRow) {
                    cellRefRange.getEnd().setRow(endRow+count);
                    mergeCell.setRef(Cell.getCellRefRange(cellRefRange));
                }
            }
        }
        handleInsertDeleteColRow(sheetIndex, start, count, ColRow.Row, 1);
    }

    public void deleteRows(int sheetIndex, int start, int optEnd) {

        if(start<0) {
            throw new RuntimeException("xlsx::deleteRows: start<0");
        }
        int count = 1;
        if(optEnd!=-1) {
            if(optEnd<start) {
                throw new RuntimeException("xlsx::deleteRows: optEnd<start");
            }
            count = (optEnd-start)+1;
        }
        else {
            optEnd = start;
        }
        final WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        worksheet.setDimension(null);
        final SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            log.warn("xlsx export: deleteRows, no sheetData available");
            return;
        }
        sheetData.deleteRows(start, count);

        // taking care of merged cells
        final CTMergeCells mergeCells = worksheet.getMergeCells();
        if(mergeCells!=null) {
            final List<CTMergeCell> mergeCellList = mergeCells.getMergeCell();
            for(int i=0;i<mergeCellList.size();i++) {
                CTMergeCell mergeCell = mergeCellList.get(i);
                final CellRefRange newMergeCell = Cell.CellRefRange.deleteRowRange(Cell.createCellRefRange(mergeCell.getRef()), start, optEnd);
                if(newMergeCell==null) {
                    mergeCellList.remove(i--);
                }
                else if(newMergeCell.getStart().equals(newMergeCell.getEnd())) {
                    mergeCellList.remove(i--);
                }
                else {
                    mergeCell.setRef(Cell.getCellRefRange(newMergeCell));
                }
            }
        }
        handleInsertDeleteColRow(sheetIndex, start, count, ColRow.Row, -1);
    }

    public void insertColumns(int sheetIndex, int start, int optEnd)
        throws Exception {

        if(start<0) {
            throw new RuntimeException("xlsx::insertColumns: start<0");
        }
        int count = 1;
        if(optEnd!=-1) {
            if(optEnd<start) {
                throw new RuntimeException("xlsx::insertColumns: optEnd<start");
            }
            count = (optEnd-start)+1;
        }
        WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        Worksheet worksheet = worksheetPart.getJaxbElement();
        worksheet.setDimension(null);
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            sheetData = Context.getsmlObjectFactory().createSheetData();
            worksheet.setSheetData(sheetData);
        }
        sheetData.insertColumns(getOperationDocument().getStylesheet(true), start, count);

        // taking care of merged cells
        CTMergeCells mergeCells = worksheet.getMergeCells();
        if(mergeCells!=null) {
            List<CTMergeCell> mergeCellList = mergeCells.getMergeCell();
            for(CTMergeCell mergeCell:mergeCellList) {
                CellRefRange cellRefRange = Cell.createCellRefRange(mergeCell.getRef());

                int startColumn = cellRefRange.getStart().getColumn();
                int endColumn = cellRefRange.getEnd().getColumn();
                if(start<=startColumn) {
                    cellRefRange.getStart().setColumn(startColumn+count);
                    cellRefRange.getEnd().setColumn(endColumn+count);
                    mergeCell.setRef(Cell.getCellRefRange(cellRefRange));
                }
                else if(start<=endColumn) {
                    cellRefRange.getEnd().setColumn(endColumn+count);
                    mergeCell.setRef(Cell.getCellRefRange(cellRefRange));
                }
            }
        }
        handleInsertDeleteColRow(sheetIndex, start, count, ColRow.Column, 1);
    }

    public void deleteColumns(int sheetIndex, int start, int optEnd) {

        if(start<0) {
            throw new RuntimeException("xlsx::deleteColumns: start<0");
        }
        int count = 1;
        if(optEnd!=-1) {
            if(optEnd<start) {
                throw new RuntimeException("xlsx::deleteColumns: optEnd<start");
            }
            count = (optEnd-start)+1;
        }
        else {
            optEnd = start;
        }
        final Worksheet worksheet = getWorksheet(sheetIndex);
        worksheet.setDimension(null);
        final SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            log.warn("xlsx export: deleteColumns, no sheetData available");
            return;
        }
        sheetData.deleteColumns(start, count);

        // taking care of merged cells
        final CTMergeCells mergeCells = worksheet.getMergeCells();
        if(mergeCells!=null) {
            final List<CTMergeCell> mergeCellList = mergeCells.getMergeCell();
            for(int i=0;i<mergeCellList.size();i++) {
                CTMergeCell mergeCell = mergeCellList.get(i);
                final CellRefRange newMergeCell = Cell.CellRefRange.deleteColumnRange(Cell.createCellRefRange(mergeCell.getRef()), start, optEnd);
                if(newMergeCell==null) {
                    mergeCellList.remove(i--);
                }
                else if(newMergeCell.getStart().equals(newMergeCell.getEnd())) {
                    mergeCellList.remove(i--);
                }
                else {
                    mergeCell.setRef(Cell.getCellRefRange(newMergeCell));
                }
            }
        }
        handleInsertDeleteColRow(sheetIndex, start, count, ColRow.Column, -1);
    }

    private void handleInsertDeleteColRow(int sheetIndex, int start, int count, ColRow colrow, int rel){

        final Workbook workbook = workbookPart.getJaxbElement();
        final List<Sheet> allSheets = workbook.getSheets().getSheet();
        final String sheetName = allSheets.get(sheetIndex).getName();
        final RelationshipsPart workbookRelationshipsPart = workbookPart.getRelationshipsPart();

        for (int i = 0; i < allSheets.size(); i++) {

            final Sheet sheet = allSheets.get(i);
            final WorksheetPart sheetPart = (WorksheetPart)workbookRelationshipsPart.getPart(sheet.getId());
            final RelationshipsPart sheetRelationshipsPart = sheetPart.getRelationshipsPart();
            if(sheetRelationshipsPart!=null) {
                final Worksheet worksheet = sheetPart.getJaxbElement();
                final org.xlsx4j.sml.CTDrawing ctDrawing = worksheet.getDrawing();
                if(ctDrawing!=null) {
                    final Drawing drawing = (Drawing)sheetRelationshipsPart.getPart(ctDrawing.getId());
                    if(drawing!=null) {
                        if(drawing.getRelationshipsPart().getRelationshipByType("http://schemas.openxmlformats.org/officeDocument/2006/relationships/chart")!=null) {

                            // if we have a drawing and if the drawing is containing a chart... we have to go deeper into detail
                            ChartUtils.handleInsertDeleteColRow(drawing, sheet.getName(), sheetName, start, count, colrow, rel);
                        }
                    }
                }
            }
        }
    }

    public void insertSheet(int sheetIndex, String sheetName, JSONObject attrs)
        throws InvalidFormatException, JAXBException {

        // first we have to get a free relationship that can be used for our new sheet
        int sheetNumber = 1;

        Relationship relShip = null;
        PartName partName = null;
        do {
            partName = new PartName("/xl/worksheets/sheet" + Integer.toString(sheetNumber++) + ".xml");
            relShip = workbookPart.getRelationshipsPart().getRel(partName);
        }
        while(relShip!=null);

        // now we have a free partName... and we will look for a free SheetId
        Workbook workbook = workbookPart.getJaxbElement();
        Sheets sheets = workbook.getSheets();
        List<Sheet> sheetList = sheets.getSheet();

        long sheetId = 1;
        do {
            boolean idFound = false;
            for(int i = 0; i < sheetList.size(); i++) {
                if(sheetList.get(i).getSheetId()==sheetId) {
                    idFound = true;
                    break;
                }
            }
            if(idFound==false)
                break;
            sheetId++;
        }
        while(true);

        final WorksheetPart worksheetPart = spreadsheetMLPackage.createWorksheetPart(partName, sheetName, sheetId);

        // setting the correct index position
        final Sheet sheet = sheetList.remove(sheetList.size() - 1);
        sheetList.add(sheetIndex, sheet);
        if(attrs!=null) {
            SheetUtils.applySheetProperties(workbook.getSheets().getSheet().get(sheetIndex), worksheetPart.getJaxbElement(), attrs.optJSONObject("sheet"));
        }

        // correcting definedName local sheet ids
        final DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames!=null) {
            for(final CTDefinedName definedName:definedNames.getDefinedName()) {
                final Long localSheetId = definedName.getLocalSheetId();
                if(localSheetId!=null) {
                    if(localSheetId>=sheetIndex) {
                        definedName.setLocalSheetId(localSheetId+1);
                    }
                }
            }
        }
    }

    public void moveSheet(int sheetIndex, int to) {
        Workbook workbook = workbookPart.getJaxbElement();
        Sheets sheets = workbook.getSheets();
        List<Sheet> sheetList = sheets.getSheet();
        sheetList.add(to, sheetList.remove(sheetIndex));

        // correcting definedName local sheet ids
        final DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames!=null) {
            final List<CTDefinedName> definedNameList = definedNames.getDefinedName();
            for(CTDefinedName definedName:definedNameList) {
                final Long localSheetId = definedName.getLocalSheetId();
                if(localSheetId!=null) {
                    long newSheetId;
                    if(localSheetId==sheetIndex) {
                        newSheetId = to;
                    }
                    else {
                        newSheetId = localSheetId;
                        if(sheetIndex<newSheetId) {
                            newSheetId--;
                        }
                        if(newSheetId>=to) {
                            newSheetId++;
                        }
                    }
                    definedName.setLocalSheetId(newSheetId);
                }
            }
        }
    }

    public void deleteSheet(int sheetIndex) {
        Workbook workbook = workbookPart.getJaxbElement();
        Sheets sheets = workbook.getSheets();
        List<Sheet> sheetList = sheets.getSheet();
        Sheet sheet = sheetList.remove(sheetIndex);
        String relId = sheet.getId();
        RelationshipsPart relationshipsPart = workbookPart.getRelationshipsPart();
        relationshipsPart.removeRelationship(relationshipsPart.getRelationshipByID(relId));

        // correcting definedName local sheet ids
        final DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames!=null) {
            final List<CTDefinedName> definedNameList = definedNames.getDefinedName();
            for(int i=0;i<definedNameList.size();i++) {
                final CTDefinedName definedName = definedNameList.get(i);
                final Long localSheetId = definedName.getLocalSheetId();
                if(localSheetId!=null) {
                    if(localSheetId==sheetIndex) {
                        definedNameList.remove(i--);
                    }
                    else if(localSheetId>sheetIndex) {
                        definedName.setLocalSheetId(localSheetId-1);
                    }
                }
            }
        }
    }

    public void setSheetName(int sheetIndex, String sheetName) {
        Workbook workbook = workbookPart.getJaxbElement();
        workbook.getSheets().getSheet().get(sheetIndex).setName(sheetName);
    }

    public void setSheetAttributes(int sheetIndex, JSONObject attrs) {
        if(attrs!=null) {
            final Workbook workbook = workbookPart.getJaxbElement();
            SheetUtils.applySheetProperties(workbook.getSheets().getSheet().get(sheetIndex), getWorksheet(sheetIndex), attrs.optJSONObject("sheet"));
        }
    }

    public void mergeCells(int sheetIndex, JSONArray start, JSONArray optEnd, boolean keepContent, String type)
        throws JSONException {

        if(start.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::mergeCells: size of start parameter != 2");

        if(optEnd==null)
            optEnd = start;
        else if (optEnd.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::mergeCells: size of end parameter != 2");

        Worksheet worksheet = getWorksheet(sheetIndex);
        CTMergeCells mergeCells = worksheet.getMergeCells();
        if(mergeCells==null) {
            mergeCells = Context.getsmlObjectFactory().createCTMergeCells();
            worksheet.setMergeCells(mergeCells);
        }
        // first we will remove each mergeCell that is covering our new mergeCellRange
        final CellRefRange cellRefRange = new CellRefRange(new Cell.CellRef(start.getInt(0), start.getInt(1)), new Cell.CellRef(optEnd.getInt(0), optEnd.getInt(1)));
        final List<CTMergeCell> mergeCellList = mergeCells.getMergeCell();
        for(int i=mergeCellList.size()-1;i>=0;i--) {
            final CTMergeCell mergeCell = mergeCellList.get(i);
            final CellRefRange mergedRefRange = Cell.createCellRefRange(mergeCell.getRef());
            if(mergedRefRange==null) {
                log.warn("xlsx export: mergeRange is invalid");
            }
            else {
                if(cellRefRange.intersects(mergedRefRange)) {
                    mergeCellList.remove(i);
                }
            }
        }
        if(type.equals("merge")) {
            CTMergeCell mergeCell = Context.getsmlObjectFactory().createCTMergeCell();
            mergeCell.setRef(Cell.getCellRefRange(start.getInt(0), start.getInt(1), optEnd.getInt(0), optEnd.getInt(1)));
            mergeCellList.add(mergeCell);
        }
        else if(type.equals("horizontal")) {
            for(int row=start.getInt(1);row<=optEnd.getInt(1);row++) {
                CTMergeCell mergeCell = Context.getsmlObjectFactory().createCTMergeCell();
                mergeCell.setRef(Cell.getCellRefRange(start.getInt(0), row, optEnd.getInt(0), row));
                mergeCellList.add(mergeCell);
            }
        }
        else if(type.equals("vertical")) {
            for(int column=start.getInt(0);column<=optEnd.getInt(0);column++) {
                CTMergeCell mergeCell = Context.getsmlObjectFactory().createCTMergeCell();
                mergeCell.setRef(Cell.getCellRefRange(column, start.getInt(1), column, optEnd.getInt(1)));
                mergeCellList.add(mergeCell);
            }
        }
        if(mergeCells.getMergeCell().isEmpty()) {
            worksheet.setMergeCells(null);
        }
    }

    public CTXf getStyleXf(String styleId) {

        if(styleId==null||styleId.isEmpty())
            return null;

        final CTStylesheet stylesheet = operationDocument.getStylesheet(false);
        if(stylesheet==null)
            return null;

        final CTCellStyleXfs styleXfs = stylesheet.getCellStyleXfs();
        if(styleXfs==null)
            return null;

        final List<CTXf> styleXfsList = styleXfs.getXf();
        if(styleXfsList.isEmpty())
            return null;

        // first check if the styleId is a buildIn styles...
        final CTCellStyle cellStyle = styleMap.get(styleId);
        if(cellStyle!=null) {
            long index = cellStyle.getXfId();
            return index<0||index>=styleXfsList.size() ? null : styleXfsList.get((int)index);
        }
        if(styleId.charAt(0)=='a') {    // autostyle
            try {
                // get the style from cellXf
                int index = Integer.parseInt(styleId.substring(1));
                final CTCellXfs cellXfs = stylesheet.getCellXfs();
                if(cellXfs==null)
                    return null;
                final List<CTXf> cellXfList = cellXfs.getXf();
                if(index<0||index>=cellXfList.size())
                    return null;
                Long xfId = cellXfList.get(index).getXfId();
                if(xfId==null)
                    return styleXfsList.get(0);
                return xfId<0||xfId>=styleXfsList.size() ? null : styleXfsList.get(xfId.intValue());
             }
            catch (NumberFormatException e) {
                return null;
            }
        }
        try {
            int index = Integer.parseInt(styleId);
            return index<0||index>=styleXfsList.size() ? null : styleXfsList.get(index);
        }
        catch (NumberFormatException e) {
            return null;
        }
    }

    public Long getStyleIndex(String styleId) {
        if(styleId==null||styleId.isEmpty()) {
            return null;
        }

        // first check if the styleId is a buildIn styles...
        final CTCellStyle cellStyle = styleMap.get(styleId);
        if(cellStyle!=null) {
            return cellStyle.getXfId();
        }

        // this styleId is to be interpreted as index into the cellStyleXfs
        final String styleIndex = styleId.charAt(0)=='a' ? styleId.substring(1) : styleId;
        try {
            return Long.parseLong(styleIndex);
        }
        catch (NumberFormatException e) {
            return null;
        }
    }

    public void insertStyleSheet(String type, String styleId, String styleName, JSONObject attrs, String parent, boolean hidden, int uiPriority, boolean defaultStyle, boolean autoStyle)
        throws Exception {

        final CTStylesheet stylesheet = operationDocument.getStylesheet(true);
        CTCellStyleXfs cellStyleXfs = stylesheet.getCellStyleXfs();
        if(cellStyleXfs==null) {
            cellStyleXfs = Context.getsmlObjectFactory().createCTCellStyleXfs();
        }
        CTCellStyle cellStyle = styleMap.get(styleId);
        if(cellStyle==null) {
            final CTXf ctXf = Context.getsmlObjectFactory().createCTXf();
            final List<CTXf> ctXfList = cellStyleXfs.getXf();
            final Long ctXfIndex = new Long(ctXfList.size());
            ctXfList.add(ctXf);

            CTCellStyles cellStyles = stylesheet.getCellStyles();
            if(cellStyles==null) {
                cellStyles = Context.getsmlObjectFactory().createCTCellStyles();
            }
            final List<CTCellStyle> cellStyleList = cellStyles.getCellStyle();
            cellStyle = Context.getsmlObjectFactory().createCTCellStyle();
            cellStyleList.add(cellStyle);
            cellStyle.setBuiltinId(BuiltinIds.getBuiltinId(styleName));
            cellStyle.setName(styleName);
            cellStyle.setHidden(hidden);
            cellStyle.setXfId(ctXfIndex);
            styleMap.put(styleId, cellStyle);
        }
        final CTXf ctXf = cellStyleXfs.getXf().get((int)cellStyle.getXfId());
        final JSONObject cellProperties = attrs.optJSONObject("cell");
        if(cellProperties!=null) {
            CellUtils.applyCellProperties(operationDocument, cellProperties, ctXf, stylesheet, null);
        }
        final JSONObject characterProperties = attrs.optJSONObject("character");
        if(characterProperties!=null) {
            CellUtils.applyCharacterProperties(operationDocument, null, null, characterProperties, ctXf, stylesheet, null);
        }
    }

    private Drawing getDrawing(int sheetIndex)
        throws Exception {

        final WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        final org.xlsx4j.sml.CTDrawing drawing = worksheet.getDrawing();
        if(drawing==null) {
            throw new RuntimeException("xlsx:getDrawing: could not get drawing for sheet");
        }
        final String drawingsId = drawing.getId();
        if(drawingsId==null||drawingsId.isEmpty()) {
            throw new RuntimeException("xlsx:getDrawing: could not get drawing for sheet");
        }
        final RelationshipsPart worksheetRelationships = worksheetPart.getRelationshipsPart();
        return (Drawing)worksheetRelationships.getPart(drawingsId);
    }

    public void insertDrawing(JSONArray start, String type, JSONObject attrs)
        throws Exception {

        Drawing drawingPart = null;
        final int sheetNumber = start.getInt(0);
        final WorksheetPart worksheetPart = getWorksheetPart(sheetNumber);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        org.xlsx4j.sml.CTDrawing drawing = worksheet.getDrawing();
        if(drawing==null) { // -> drawing part needs to be created
            drawing = Context.getsmlObjectFactory().createCTDrawing();
            worksheet.setDrawing(drawing);
            drawingPart = new Drawing(new PartName("/xl/drawings/drawing"+ Integer.toString(sheetNumber+1) + ".xml"));
            drawingPart.setJaxbElement(Context.getDmlSpreadsheetDrawingObjectFactory().createCTDrawing());
            Relationship dmlRelationship = worksheetPart.addTargetPart(drawingPart, AddPartBehaviour.RENAME_IF_NAME_EXISTS);
            drawing.setId(dmlRelationship.getId());
        }
        else {
            drawingPart = (Drawing)worksheetPart.getRelationshipsPart().getPart(drawing.getId());
        }
        Drawings.insertDrawing(getOperationDocument(), start.getInt(1), drawingPart, type, attrs);
    }

    public void applyDrawingAttributes(JSONArray start, JSONObject attrs)
        throws Exception {

        final org.docx4j.openpackaging.parts.DrawingML.Drawing drawingsPart = getDrawing(start.getInt(0));
        final JSONObject drawingProperties = attrs.optJSONObject("drawing");
        if(drawingProperties!=null) {
            Drawings.applyDrawingAttributes(Drawings.applyAnchorAttributes(start.getInt(1), drawingsPart, drawingProperties), drawingProperties);
        }

        final org.docx4j.dml.spreadsheetdrawing.CTDrawing drawings = drawingsPart.getJaxbElement();
        final CellAnchor cellAnchor = drawings.getEGAnchor().get(start.getInt(1));
        final HashMap<String, Object> drawingPropertyMap = DMLGraphic.getDrawingType(drawingsPart, cellAnchor);
        if(drawingPropertyMap.containsKey("type")) {
            final String drawingType = (String)drawingPropertyMap.get("type");
            if(drawingType.equals("chart")) {
                final JSONObject chartProperties = attrs.optJSONObject("chart");
                if(chartProperties!=null) {
                    CTChartSpace chartSpace = (CTChartSpace)drawingPropertyMap.get("chartObject");
                    DMLChartSpace.applyChartSpaceProperties(chartProperties, chartSpace);
                }
            }
        }
    }

    public void deleteDrawing(JSONArray start)
        throws Exception {

        Drawing drawing = getDrawing(start.getInt(0));
        drawing.getJaxbElement().getEGAnchor().remove(start.getInt(1));
    }

    public void insertChartDataSeries(JSONArray start, int series, JSONObject attrs)
        throws Exception {

        Workbook workbook = workbookPart.getJaxbElement();
        final String sheetName = workbook.getSheets().getSheet().get(start.getInt(0)).getName();
        final Drawing drawing = getDrawing(start.getInt(0));
        final HashMap<String, Object> drawingPropertyMap = new HashMap<String, Object>();
        DMLGraphic.getGraphicProperties(drawing, drawing.getJaxbElement().getEGAnchor().get(start.getInt(1)).getGraphicFrame().getGraphic(), drawingPropertyMap);
        DMLChartSpace.insertChartSpace((CTChartSpace)drawingPropertyMap.get("chartObject"), sheetName, series, attrs);
    }

    public void deleteChartDataSeries(JSONArray start, int series)
        throws Exception {

        final Drawing drawing = getDrawing(start.getInt(0));
        final HashMap<String, Object> drawingPropertyMap = new HashMap<String, Object>();
        DMLGraphic.getGraphicProperties(drawing, drawing.getJaxbElement().getEGAnchor().get(start.getInt(1)).getGraphicFrame().getGraphic(), drawingPropertyMap);
        DMLChartSpace.deleteChartSpace((CTChartSpace)drawingPropertyMap.get("chartObject"), series);
    }

    public void insertName(Long sheet, String name, String formula) {

        final Workbook workbook = workbookPart.getJaxbElement();
        DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames==null) {
            definedNames = Context.getsmlObjectFactory().createDefinedNames();
            workbook.setDefinedNames(definedNames);
        }
        final CTDefinedName newDefinedName = Context.getsmlObjectFactory().createCTDefinedName();
        newDefinedName.setName(name);
        newDefinedName.setValue(formula);
        newDefinedName.setLocalSheetId(sheet);
        final List<CTDefinedName> definedNameList = definedNames.getDefinedName();
        for(int i=0; i<definedNameList.size();i++) {
            final CTDefinedName definedName = definedNameList.get(i);

            // check if the name is already used..
            if(definedName.getName().equals(name)) {
                final Long localSheetId = definedName.getLocalSheetId();
                if((sheet==null&&localSheetId==null)||(sheet!=null&&localSheetId!=null&&sheet.longValue()==localSheetId.longValue())) {
                    definedNameList.set(i, newDefinedName);
                    return;
                }
            }
        }
        definedNameList.add(newDefinedName);
    }

    public void deleteName(Long sheet, String name) {

        final Workbook workbook = workbookPart.getJaxbElement();
        final DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames!=null) {
            final List<CTDefinedName> definedNameList = definedNames.getDefinedName();
            for(int i=0; i<definedNameList.size();i++) {
                final CTDefinedName definedName = definedNameList.get(i);

                // check if the name is already used..
                if(definedName.getName().equals(name)) {
                    final Long localSheetId = definedName.getLocalSheetId();
                    if((sheet==null&&localSheetId==null)||(sheet!=null&&localSheetId!=null&&sheet.longValue()==localSheetId.longValue())) {
                        definedNameList.remove(i);
                        return;
                    }
                }
            }
        }
        log.debug("xlsx:deleteName ... could not find name within definedNameList");
    }
}
