/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.operations;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.docx4j.openpackaging.packages.SpreadsheetMLPackage;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorkbookPart;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorksheetPart;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.xlsx4j.sml.CTCellFormula;
import org.xlsx4j.sml.CTCellStyle;
import org.xlsx4j.sml.CTCellStyleXfs;
import org.xlsx4j.sml.CTCellStyles;
import org.xlsx4j.sml.CTCellXfs;
import org.xlsx4j.sml.CTDefinedName;
import org.xlsx4j.sml.CTDrawing;
import org.xlsx4j.sml.CTHyperlink;
import org.xlsx4j.sml.CTHyperlinks;
import org.xlsx4j.sml.CTRst;
import org.xlsx4j.sml.CTStylesheet;
import org.xlsx4j.sml.CTXf;
import org.xlsx4j.sml.Cell;
import org.xlsx4j.sml.Col;
import org.xlsx4j.sml.Cols;
import org.xlsx4j.sml.DataValidation;
import org.xlsx4j.sml.DataValidations;
import org.xlsx4j.sml.DefinedNames;
import org.xlsx4j.sml.Row;
import org.xlsx4j.sml.STCellFormulaType;
import org.xlsx4j.sml.STCellType;
import org.xlsx4j.sml.STDataValidationErrorStyle;
import org.xlsx4j.sml.STDataValidationOperator;
import org.xlsx4j.sml.STDataValidationType;
import org.xlsx4j.sml.Sheet;
import org.xlsx4j.sml.SheetData;
import org.xlsx4j.sml.Sheets;
import org.xlsx4j.sml.Workbook;
import org.xlsx4j.sml.Worksheet;
import com.openexchange.config.ConfigurationService;
import com.openexchange.log.LogFactory;
import com.openexchange.office.FilterException;
import com.openexchange.office.FilterException.ErrorCode;
import com.openexchange.office.ooxml.tools.Commons;
import com.openexchange.office.ooxml.tools.Commons.RowIterator;
import com.openexchange.office.ooxml.xlsx.OperationDocument;
import com.openexchange.office.ooxml.xlsx.tools.CellUtils;
import com.openexchange.office.ooxml.xlsx.tools.ColumnUtils;
import com.openexchange.office.ooxml.xlsx.tools.Drawings;
import com.openexchange.office.ooxml.xlsx.tools.MergeCellHelper;
import com.openexchange.office.ooxml.xlsx.tools.RowUtils;
import com.openexchange.office.ooxml.xlsx.tools.SheetUtils;

public class CreateOperationHelper extends com.openexchange.office.ooxml.operations.CreateOperationHelper {

    private final static Log log = LogFactory.getLog(CreateOperationHelper.class);

    private final SpreadsheetMLPackage spreadsheetMLPackage;

    public CreateOperationHelper(OperationDocument _operationDocument, JSONArray mOperationsArray) {
        super(_operationDocument, mOperationsArray);
        spreadsheetMLPackage = _operationDocument.getPackage();
    }

    @Override
    public OperationDocument getOperationDocument() {
        return (OperationDocument)operationDocument;
    }

    @Override
    public void createDocumentDefaults(String userLanguage) {

    }

    public void createStyleOperations()
        throws Exception {

        CTStylesheet stylesheet = getOperationDocument().getStylesheet(false);
        if(stylesheet!=null) {

            final Map<Long, String> indexToStyleId = new HashMap<Long, String>();

            // creating styles
            final CTCellStyleXfs cellStyleXfs = stylesheet.getCellStyleXfs();
            if(cellStyleXfs!=null) {
                BuiltinIds.getUsedStyles(stylesheet, indexToStyleId, null);
                final List<CTXf> xfs = cellStyleXfs.getXf();
                for(int i=0;i<xfs.size();i++) {
                    final String styleId = indexToStyleId.get(new Long(i));
                    if(styleId!=null) {
                        String styleName = styleId;
                        JSONObject jsonProperties = new JSONObject();
                        Commons.mergeJsonObjectIfUsed("cell", jsonProperties, CellUtils.createCellProperties(xfs.get(i), stylesheet, true));
                        Commons.mergeJsonObjectIfUsed("character", jsonProperties, CellUtils.createCharacterProperties(xfs.get(i), stylesheet, true));
                        JSONObject attrs = jsonProperties.length()>0?jsonProperties:null;
                        String parentId = null;
                        Boolean hidden = null;
                        Integer uiPriority = null;
                        Boolean isDefault = null;

                        // check for proper UI styleName
                        CTCellStyles cellStyles = stylesheet.getCellStyles();
                        if(cellStyles!=null) {
                            for(CTCellStyle cellStyle:cellStyles.getCellStyle()) {
                                if(cellStyle.getXfId()==i) {
                                    final String name = cellStyle.getName();
                                    if(name!=null&&!name.isEmpty()) {
                                        styleName = name;
                                    }
                                    if(i==0) {
                                        isDefault = new Boolean(true);
                                    }
                                    hidden = cellStyle.isHidden();
                                    break;
                                }
                            }
                        }
                        addInsertStyleSheetOperation("cell", styleId, styleName, attrs, parentId, hidden, uiPriority, isDefault, false);
                    }
                }
            }

            // creating autostyles
            final CTCellXfs cellXfs = stylesheet.getCellXfs();
            if(cellXfs!=null) {
                final List<CTXf> cellXfsList = cellXfs.getXf();
                for(int i=0;i<cellXfsList.size();i++) {
                    final CTXf xf = cellXfsList.get(i);
                    final String styleId = "a" + Integer.toString(i);
                    String styleName = "";
                    JSONObject jsonProperties = new JSONObject();
                    Commons.mergeJsonObjectIfUsed("cell", jsonProperties, CellUtils.createCellProperties(xf, stylesheet, true));
                    Commons.mergeJsonObjectIfUsed("character", jsonProperties, CellUtils.createCharacterProperties(xf, stylesheet, true));
                    JSONObject attrs = jsonProperties.length()>0?jsonProperties:null;
                    String parentId = null;
                    final Long xfId = xf.getXfId();
                    if(xfId!=null) {
                        parentId = indexToStyleId.get(xfId);
                    }
                    if(parentId == null) {
                        parentId = "0";
                    }
                    Boolean hidden = true;
                    Integer uiPriority = null;
                    Boolean isDefault = false;
                    addInsertStyleSheetOperation("cell", styleId, styleName, attrs, parentId, hidden, uiPriority, isDefault, true);
                }
            }
        }
    }

    @Override
    public void createOperations()
        throws Exception {

        WorkbookPart workbookPart = spreadsheetMLPackage.getWorkbookPart();
        Workbook workbook = workbookPart.getJaxbElement();

        checkComplexity(workbook);

        createDocumentDefaults(getOperationDocument().getUserLanguage());
        createStyleOperations();
        CreateThemeOperations();
        createInsertNameOperations(workbook);

        Sheets sheets = workbook.getSheets();
        List<Sheet> sheetList = sheets.getSheet();
        for(int sheetIndex = 0; sheetIndex < sheetList.size(); sheetIndex++) {
            createSheetOperations(sheetList.get(sheetIndex), sheetIndex);
        }
    }

    private void checkComplexity(Workbook workbook)
        throws FilterException {

        final ConfigurationService configuration = getOperationDocument().getConfigurationService();

        int maxCellCount  =      configuration.getIntProperty("io.ox/office//spreadsheet/maxCells", 400000);
        int maxSheetCount =      configuration.getIntProperty("io.ox/office//spreadsheet/maxSheets", 200);

        final Sheets sheets = workbook.getSheets();
        if(sheets!=null) {
            final List<Sheet> sheetList = sheets.getSheet();
            if(maxSheetCount>=0) {
                if(sheetList.size()>maxSheetCount) {
                    throw new FilterException("", ErrorCode.COMPLEXITY_TOO_HIGH);
                }
            }
            if(maxCellCount>=0) {
                final RelationshipsPart relationshipPart = spreadsheetMLPackage.getWorkbookPart().getRelationshipsPart();
                for(Sheet sheet:sheetList) {
                    final Part part = relationshipPart.getPart(sheet.getId());
                    if(part instanceof WorksheetPart) {
                        final Worksheet worksheet = ((WorksheetPart)part).getJaxbElement();
                        final SheetData sheetData = worksheet.getSheetData();
                        if(sheetData!=null) {
                            final RowIterator rowIterator = sheetData.createRowIterator();
                            while(rowIterator.hasNext()) {
                                final Row row = rowIterator.next();
                                maxCellCount -= row.getCellCount();
                                if(maxCellCount<0) {
                                    throw new FilterException("", ErrorCode.COMPLEXITY_TOO_HIGH);
                                }
                            }
                        }
                    }
                }
            }
        }
    }


    private void createInsertNameOperations(Workbook workbook)
        throws JSONException {

        final DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames==null) {
            return;
        }
        final List<CTDefinedName> definedNameList = definedNames.getDefinedName();
        for(CTDefinedName definedName:definedNameList) {
            addInsertNameOperation(definedName.getLocalSheetId(), definedName.getName(), definedName.getValue()!=null ? definedName.getValue() : "");
        }
    }

    private JSONObject getCellObject(Cell cell)
        throws Exception {

        final JSONObject cellObject = new JSONObject();
        final CTCellFormula cellFormula = cell.getF();
        if(cellFormula!=null&&(cellFormula.getT()==STCellFormulaType.NORMAL||cellFormula.getT()==STCellFormulaType.SHARED)) {
            if(cellFormula.getT()==STCellFormulaType.SHARED) {
                final Long si = cellFormula.getSi();
                if(si!=null) {
                    cellObject.put("shared", si+1);
                }
                final Cell.CellRef cellRef = Cell.createCellRef(cell.getR());
                if(cellRef!=null) {
                    cellObject.put("ref", cellRef.getJSONArray());
                }
            }
            if(cellFormula.getValue()!=null&&cellFormula.getValue().length()>0)
                cellObject.put("value", "=" + cellFormula.getValue());
        }
        else if (cell.getT()==STCellType.INLINE_STR) {
            CTRst is = cell.getIs();
            if(is!=null) {
                // TODO: CTRst can include formatted text runs, but we will support simple t first.
                if(is.getT()!=null) {
                    cellObject.put("value",  is.getT());
                }
            }
        }
        else if(cell.getV()!=null) {
            if(cell.getT()==STCellType.N) {
                try {

                    double number = Double.parseDouble(cell.getV());
                    cellObject.put("value", number);
                }
                catch(NumberFormatException e) {
                    // TODO: ups
                }
            }
            else if(cell.getT()==STCellType.S) {
                String sharedString = getOperationDocument().getSharedString(cell.getV());
                if(sharedString!=null) {
                    cellObject.put("value", getOperationDocument().escapeString(sharedString));
                }
            }
            else if(cell.getT()==STCellType.B) {
                if(cell.getV().equals("0")||cell.getV().toLowerCase().equals("false")) {
                    cellObject.put("value", false);
                }
                else if(cell.getV().equals("1")||cell.getV().toLowerCase().equals("true")) {
                    cellObject.put("value", true);
                }
            }
            else {  // STCellType.STR, STCellType.E or STCellType.F (ARRAY or DATA_TABLE format)
                cellObject.put("value",  cell.getV());
            }
        }
       JSONObject jsonCellAttrs = CellUtils.createCellProperties(cell, getOperationDocument().getStylesheet(true));
       if(jsonCellAttrs!=null) {
           cellObject.put("attrs", jsonCellAttrs );
       }
        return cellObject;
    }

    private void createSheetOperations(final Sheet sheet, int sheetIndex)
        throws Exception {

        final RelationshipsPart relationshipPart = spreadsheetMLPackage.getWorkbookPart().getRelationshipsPart();
        final Part part = relationshipPart.getPart(sheet.getId());

        String sheetType = null;
        if(part.getRelationshipType().equals("http://schemas.openxmlformats.org/officeDocument/2006/relationships/worksheet")) {
            sheetType = "worksheet";
        }
        else if(part.getRelationshipType().equals("http://schemas.microsoft.com/office/2006/relationships/xlMacrosheet")) {
            sheetType = "macrosheet";
        }
        else if(part.getRelationshipType().equals("http://schemas.microsoft.com/office/2006/relationships/xlIntlMacrosheet")) {
            sheetType = "macrosheet";
        }
        else if(part.getRelationshipType().equals("http://schemas.openxmlformats.org/officeDocument/2006/relationships/dialogsheet")) {
            sheetType = "dialogsheet";
        }
        else if(part.getRelationshipType().equals("http://schemas.openxmlformats.org/officeDocument/2006/relationships/chartsheet")) {
            sheetType = "chartsheet";
        }

        if(sheetType==null) {
            throw new FilterException("xlsx import: unknown sheettype detected.", ErrorCode.CRITICAL_ERROR);
        }

        if(sheetType.equals("worksheet")) {
            final WorksheetPart worksheetPart = (WorksheetPart)part;
            final Worksheet worksheet = worksheetPart.getJaxbElement();

            final JSONObject sheetProperties = new JSONObject();
            sheetProperties.put("sheet", SheetUtils.createWorkSheetProperties(sheet, worksheet));
            sheetProperties.put("column", SheetUtils.createWorkSheetColumnProperties(getOperationDocument(), worksheet));
//          sheetProperties.put("row",  SheetUtils.createWorkSheetRowProperties(worksheet));        // the dafault row height of 0 is not working properly in the frontend
            addInsertSheetOperation(sheetIndex, sheet.getName(), sheetType, sheetProperties);

            // column attributes... row attributes ... cells...
            createColumnOperations(worksheet.getCols(), sheetIndex);
            final SheetData sheetData = worksheet.getSheetData();
            if(sheetData!=null) {
                createRowOperations(sheetData.createRowIterator(), sheetIndex);
                MergeCellHelper.createMergeCellOperations(operationsArray, sheetIndex, worksheet.getMergeCells());
            }
            // hyperlinks
            final CTHyperlinks hyperlinks = worksheet.getHyperlinks();
            if(hyperlinks!=null) {
                final List<CTHyperlink> hyperlinkList = hyperlinks.getHyperlink();
                for(int i=0; i<hyperlinkList.size();i++) {
                    final CTHyperlink hyperlink = hyperlinkList.get(i);
                    final Cell.CellRef cellRef = Cell.createCellRef(hyperlink.getRef());
                    if(cellRef!=null) {
                        final String hyperlinkUrl = Commons.getUrl(worksheetPart, hyperlink.getId());
                        if(hyperlinkUrl!=null&&!hyperlinkUrl.isEmpty()) {
                            final JSONObject attrs = new JSONObject();
                            final JSONObject characterAttributes = new JSONObject();
                            characterAttributes.put("url", hyperlinkUrl);
                            attrs.put("character", characterAttributes);
                            addSetCellAttributesOperation(sheetIndex, cellRef, attrs);
                        }
                    }
                }
            }
            //drawings
            final CTDrawing drawing = worksheet.getDrawing();
            if(drawing!=null) {
                final String drawingId = drawing.getId();
                if(drawingId!=null&&drawingId.length()>0) {
                    final RelationshipsPart worksheetRelationships = worksheetPart.getRelationshipsPart();
                    if(worksheetRelationships!=null) {
                        final Part drawingsPart = worksheetRelationships.getPart(drawingId);
                        if(drawingsPart.getRelationshipType().equals("http://schemas.openxmlformats.org/officeDocument/2006/relationships/drawing")&&drawingsPart instanceof org.docx4j.openpackaging.parts.DrawingML.Drawing) {
                            Drawings.createDrawingOperations(operationsArray, (org.docx4j.openpackaging.parts.DrawingML.Drawing)drawingsPart, sheetIndex);
                        }
                    }
                }
            }

            // validations
            final List<DataValidations> validations = getOperationDocument().getDataValidations(worksheet);
            for(DataValidations dataValidations:validations) {
                for(DataValidation dataValidation:dataValidations.getDataValidation()) {
                    createDataValidation(sheetIndex, dataValidation);
                }
            }
        }
        else {
            addInsertSheetOperation(sheetIndex, sheet.getName(), sheetType, Commons.surroundJSONObject("sheet", SheetUtils.createSheetProperties(sheet)));
        }
    }

    private void createDataValidation(int sheetIndex, DataValidation dataValidation)
        throws JSONException {

        final List<Cell.CellRefRange> cellRefRangeList = new ArrayList<Cell.CellRefRange>();
        final List<String> sqRefList = dataValidation.getsqref();
        for(String sqRef:sqRefList) {
            final Cell.CellRefRange cellRefRange = Cell.createCellRefRange(sqRef);
            if(cellRefRange!=null) {
                cellRefRangeList.add(cellRefRange);
            }
        }
        if(cellRefRangeList.size()>0) {
            addInsertValidationOperation(sheetIndex, cellRefRangeList, dataValidation.getType(), dataValidation.getOperator(), dataValidation.getFormula1(), dataValidation.getFormula2(), dataValidation.isShowInputMessage(),
                dataValidation.getPromptTitle(), dataValidation.getPrompt(), dataValidation.isShowErrorMessage(), dataValidation.getErrorTitle(), dataValidation.getError(), dataValidation.getErrorStyle(),
                    dataValidation.isShowDropDown(), dataValidation.isAllowBlank());
        }
    }

    private void createRowOperations(Commons.RowIterator rowIterator, int sheetIndex)
        throws Exception {

        Row lastRow = null;
        int lastRowNumber = 0;
        int rowCount = 0;

        while(rowIterator.hasNext()) {
            final Row row = rowIterator.next();

            if(lastRow!=null&&rowIterator.getRowNumber()==lastRowNumber+rowCount&&RowUtils.compareRowProperties(lastRow, row)) {
                rowCount++;
            }
            else {
                if(lastRow!=null) {
                    final JSONObject attrs = RowUtils.createRowProperties(lastRow, getOperationDocument().getStylesheet(true));
                    if(attrs!=null) {
                        addSetRowAttributes(sheetIndex, lastRowNumber, (lastRowNumber - 1) + rowCount, attrs);
                    }
                }
                lastRow = row;
                lastRowNumber = rowIterator.getRowNumber();
                rowCount = 1;
            }
            if(!rowIterator.hasNext()&&lastRow!=null) {
                final JSONObject attrs = RowUtils.createRowProperties(lastRow, getOperationDocument().getStylesheet(true));
                if(attrs!=null) {
                    addSetRowAttributes(sheetIndex, lastRowNumber, (lastRowNumber - 1) + rowCount, attrs);
                }
            }

            JSONArray rowArray = new JSONArray();
            Cell.CellRef topLeftCellRef = null;
            int lastCellX = -1;
            JSONArray cellArray = new JSONArray();
            Commons.CellIterator cellIterator = row.createCellIterator();
            while(cellIterator.hasNext()) {
                Cell cell = cellIterator.next();
                final Cell.CellRef cellRef = new Cell.CellRef(cellIterator.getColumnNumber(), rowIterator.getRowNumber());
                JSONObject cellObject = getCellObject(cell);
                if(topLeftCellRef==null) {
                    // first cell in our array
                    topLeftCellRef = cellRef;
                }
                else {
                    // check if empty cells are to be written and start with a new setCellContentOperation if needed
                    int emptyCells = cellRef.getColumn() - lastCellX;
                    if(emptyCells>3) {
                        // this are too many empty cells, we will create an additional setContent operation
                        if(cellArray.length()>0) {
                            // we have to write back our current content and start a new collection
                            rowArray.put(cellArray);
                            addSetCellContentsOperation(sheetIndex, topLeftCellRef, rowArray);
                            rowArray = new JSONArray();
                            cellArray = new JSONArray();
                        }
                        topLeftCellRef = cellRef;
                    }
                    else {
                        while(--emptyCells>0) {
                            cellArray.put((Object)null);
                        }
                    }
                }
                cellArray.put(cellObject);
                lastCellX = cellRef.getColumn();
            }
            rowArray.put(cellArray);
            if(topLeftCellRef!=null)
                addSetCellContentsOperation(sheetIndex, topLeftCellRef, rowArray);
        }
    }

    private void createColumnOperations(List<Cols> colsIter, int sheetIndex)
        throws Exception {

        if(colsIter==null)
            return;

        for(Cols cols:colsIter) {
            List<Col> colIter = cols.getCol();
            if(colIter!=null) {
                for(Col col:colIter) {
                    JSONObject attrs = ColumnUtils.createColumnProperties(getOperationDocument(), col);
                    if(attrs!=null) {
                        addSetColumnAttributes(sheetIndex, col.getMin()-1, col.getMax()-1, attrs);
                    }
                }
            }
        }
    }

    final static String[] enumValidationType = { "all", "integer", "number", "list", "date", "time", "length", "custom", "source" };
    final static String[] enumCompare        = { "between", "notBetween", "equal", "notEqual", "less", "lessEqual", "greater", "greaterEqual" };
    final static String[] enumErrorType      = { "error", "warning", "info" };

    public void addInsertValidationOperation(int sheetIndex, List<Cell.CellRefRange> rangeList, STDataValidationType type, STDataValidationOperator operator, String value1, String value2,
                                            boolean showInfo, String infoTitle, String infoText, boolean showError, String errorTitle, String errorText, STDataValidationErrorStyle errorStyle, boolean showDropDown, boolean ignoreBlank)
        throws JSONException {

        if(value1==null) {
            return;
        }
        final JSONObject insertValidationObject = new JSONObject();
        insertValidationObject.put("name", "insertValidation");
        insertValidationObject.put("sheet", sheetIndex);
        final JSONArray ranges = new JSONArray();
        for(Cell.CellRefRange cellRefRange:rangeList) {
            final JSONObject range = new JSONObject();
            range.put("start", cellRefRange.getStart().getJSONArray());
            if((cellRefRange.getEnd().getColumn()!=cellRefRange.getStart().getColumn())||(cellRefRange.getEnd().getRow()!=cellRefRange.getStart().getRow())) {
                range.put("end", cellRefRange.getEnd().getJSONArray());
            }
            ranges.put(range);
        }
        insertValidationObject.put("ranges", ranges);
        int val = type.ordinal();
        if(type==STDataValidationType.LIST) {
            if(!value1.isEmpty()&&value1.charAt(0)!='"') {
                val = enumValidationType.length - 1;    // using type source if value does not start with "
            }
        }
        if(val>=enumValidationType.length)
            val = 0;
        insertValidationObject.put("type", enumValidationType[val]);
        if(operator!=STDataValidationOperator.BETWEEN) {
            val = operator.ordinal();
            if(val>=enumCompare.length)
                val = 0;
            insertValidationObject.put("compare", enumCompare[val]);
        }
        insertValidationObject.put("value1", value1);
        if(value2!=null) {
            insertValidationObject.put("value2", value2);
        }
        if(showInfo==false) {
            insertValidationObject.put("showInfo", showInfo);
        }
        if(infoTitle!=null) {
            insertValidationObject.put("infoTitle", infoTitle);
        }
        if(infoText!=null) {
            insertValidationObject.put("infoText",  infoText);
        }
        if(showError==false) {
            insertValidationObject.put("showError", showError);
        }
        if(errorTitle!=null) {
            insertValidationObject.put("errorTitle", errorTitle);
        }
        if(errorText!=null) {
            insertValidationObject.put("errorText", errorText);
        }
        if(errorStyle!=STDataValidationErrorStyle.STOP) {
            val = errorStyle.ordinal();
            if(val>=enumErrorType.length)
                val = 0;
            insertValidationObject.put("errorType", enumErrorType[val]);
        }
        if(type==STDataValidationType.LIST&&showDropDown==false) {
            insertValidationObject.put("showDropDown", showDropDown);
        }
        if(ignoreBlank==false) {
            insertValidationObject.put("ignoreEmpty", ignoreBlank);
        }
        operationsArray.put(insertValidationObject);
    }

    public void addInsertSheetOperation(int sheetIndex, final String sheetName, final String type, final JSONObject attrs)
        throws JSONException {

        final JSONObject insertSheetObject = new JSONObject();
        insertSheetObject.put("name", "insertSheet");
        insertSheetObject.put("sheet", sheetIndex);
        if(type!=null&&!type.equals("worksheet")) {
            insertSheetObject.put("type", type);
        }
        insertSheetObject.put("sheetName", sheetName);
        if(attrs!=null)
            insertSheetObject.put("attrs", attrs);

        operationsArray.put(insertSheetObject);
    }

    final static List<Object> objectList = new ArrayList<Object>();

    public void addSetCellContentsOperation(int sheetIndex, Cell.CellRef cellRef, JSONArray cellContent)
        throws JSONException {

        final JSONObject insertCellsObject = new JSONObject();
        insertCellsObject.put("name", "setCellContents");
        insertCellsObject.put("sheet", sheetIndex);
        insertCellsObject.put("start", cellRef.getJSONArray());
        insertCellsObject.put("contents", cellContent);

        operationsArray.put(insertCellsObject);
    }

    public void addSetCellAttributesOperation(int sheetIndex, Cell.CellRef cellRef, JSONObject attrs)
        throws Exception {

        if(attrs!=null&&!attrs.isEmpty()) {
            final JSONObject fillCellRangeObject = new JSONObject();
            fillCellRangeObject.put("name", "fillCellRange");
            fillCellRangeObject.put("sheet", sheetIndex);
            fillCellRangeObject.put("start", cellRef.getJSONArray());
            fillCellRangeObject.put("attrs", attrs);

            operationsArray.put(fillCellRangeObject);
        }
    }

    public void addSetRowAttributes(int sheetIndex, int start, int end, JSONObject attrs)
        throws JSONException {

        if(attrs==null||attrs.length()==0)
            return;

        final JSONObject addSetRowAttributesObject = new JSONObject();
        addSetRowAttributesObject.put("name", "setRowAttributes");
        addSetRowAttributesObject.put("sheet", sheetIndex);
        addSetRowAttributesObject.put("start", start);
        if(end!=start) {
            addSetRowAttributesObject.put("end", end);
        }
        addSetRowAttributesObject.put("attrs", attrs);

        operationsArray.put(addSetRowAttributesObject);
    }

    public void addSetColumnAttributes(int sheetIndex, long start, long end, JSONObject attrs)
        throws JSONException {

        if(attrs==null||attrs.length()==0)
            return;

        final JSONObject addSetColumnAttributesObject = new JSONObject();
        addSetColumnAttributesObject.put("name", "setColumnAttributes");
        addSetColumnAttributesObject.put("sheet", sheetIndex);
        addSetColumnAttributesObject.put("start", start);
        if(end>start) {
            addSetColumnAttributesObject.put("end", end);
        }
        addSetColumnAttributesObject.put("attrs", attrs);

        operationsArray.put(addSetColumnAttributesObject);
    }

    public void addInsertNameOperation(Long sheetIndex, String name, String formula)
        throws JSONException {

        final JSONObject addInsertNameObject = new JSONObject();
        addInsertNameObject.put("name", "insertName");
        if(sheetIndex!=null) {
            addInsertNameObject.put("sheet", sheetIndex);
        }
        addInsertNameObject.put("exprName", name);
        addInsertNameObject.put("formula", formula);

        operationsArray.put(addInsertNameObject);
    }

}
