/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.tools;

import java.util.Iterator;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import org.docx4j.dml.CTBlipFillProperties;
import org.docx4j.dml.CTPositiveSize2D;
import org.docx4j.dml.CTShapeProperties;
import org.docx4j.dml.CTTransform2D;
import org.docx4j.dml.Graphic;
import org.docx4j.dml.picture.Pic;
import org.docx4j.dml.wordprocessingDrawing.Anchor;
import org.docx4j.dml.wordprocessingDrawing.CTPosH;
import org.docx4j.dml.wordprocessingDrawing.CTPosV;
import org.docx4j.dml.wordprocessingDrawing.CTWrapNone;
import org.docx4j.dml.wordprocessingDrawing.CTWrapSquare;
import org.docx4j.dml.wordprocessingDrawing.CTWrapThrough;
import org.docx4j.dml.wordprocessingDrawing.CTWrapTight;
import org.docx4j.dml.wordprocessingDrawing.CTWrapTopBottom;
import org.docx4j.dml.wordprocessingDrawing.Inline;
import org.docx4j.dml.wordprocessingDrawing.ObjectFactory;
import org.docx4j.dml.wordprocessingDrawing.STAlignH;
import org.docx4j.dml.wordprocessingDrawing.STAlignV;
import org.docx4j.dml.wordprocessingDrawing.STRelFromH;
import org.docx4j.dml.wordprocessingDrawing.STRelFromV;
import org.docx4j.dml.wordprocessingDrawing.STWrapText;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.relationships.Relationship;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.office.ooxml.docx.OperationDocument;
import com.openexchange.office.ooxml.docx.tools.Component.DrawingComponent;
import com.openexchange.office.ooxml.docx.tools.Component.Placeholder_Base.GraphicType;
import com.openexchange.office.ooxml.drawingml.DMLHelper;
import com.openexchange.office.ooxml.tools.Commons;

public class Drawings {

    final static String namespaces = " xmlns:w=\"http://schemas.openxmlformats.org/wordprocessingml/2006/main\" "
        + "xmlns:r=\"http://schemas.openxmlformats.org/officeDocument/2006/relationships\" "
        + "xmlns:wp=\"http://schemas.openxmlformats.org/drawingml/2006/wordprocessingDrawing\"";

    public static Inline createInline(boolean linked, String relId, int cx, int cy)
        throws JAXBException {

        String type = linked ? "r:link" : "r:embed";

        String ml =
            "<wp:inline distT=\"0\" distB=\"0\" distL=\"0\" distR=\"0\"" + namespaces + ">"
            + "<wp:extent cx=\"${cx}\" cy=\"${cy}\"/>"
            + "<wp:effectExtent l=\"0\" t=\"0\" r=\"0\" b=\"0\"/>" + //l=\"19050\"
            "<wp:docPr id=\"${id1}\" name=\"${filenameHint}\" descr=\"${altText}\"/><wp:cNvGraphicFramePr><a:graphicFrameLocks xmlns:a=\"http://schemas.openxmlformats.org/drawingml/2006/main\" noChangeAspect=\"1\"/></wp:cNvGraphicFramePr><a:graphic xmlns:a=\"http://schemas.openxmlformats.org/drawingml/2006/main\">"
            + "<a:graphicData uri=\"http://schemas.openxmlformats.org/drawingml/2006/picture\">"
            + "<pic:pic xmlns:pic=\"http://schemas.openxmlformats.org/drawingml/2006/picture\"><pic:nvPicPr><pic:cNvPr id=\"${id2}\" name=\"${filenameHint}\"/><pic:cNvPicPr/></pic:nvPicPr><pic:blipFill>"
            + "<a:blip " + type + "=\"${rEmbedId}\"/><a:stretch><a:fillRect/></a:stretch></pic:blipFill>"
            + "<pic:spPr><a:xfrm><a:off x=\"0\" y=\"0\"/><a:ext cx=\"${cx}\" cy=\"${cy}\"/></a:xfrm><a:prstGeom prst=\"rect\"><a:avLst/></a:prstGeom></pic:spPr></pic:pic></a:graphicData></a:graphic>"
            + "</wp:inline>";

        java.util.HashMap<String, String> mappings = new java.util.HashMap<String, String>();

        mappings.put("cx", Integer.toString(cx));
        mappings.put("cy", Integer.toString(cy));
        mappings.put("filenameHint", "");
        mappings.put("altText", "");
        mappings.put("rEmbedId", relId);
        mappings.put("id1", "0");           // it seem that this id does not need to be unique (samples/AddImage.java)...
        mappings.put("id2", "1");

        Inline inline = null;
        Object o = org.docx4j.XmlUtils.unmarshallFromTemplate(ml, mappings);
        inline = (Inline) ((JAXBElement<?>) o).getValue();
        return inline;
    }

    public static Anchor createAnchor(boolean linked, String relId, int cx, int cy)
        throws JAXBException {

        String type = linked ? "r:link" : "r:embed";

        String ml =
            "<wp:anchor simplePos=\"0\" distT=\"0\" distB=\"0\" distL=\"0\" distR=\"0\" layoutInCell=\"1\"" + namespaces + ">"
            + "<wp:simplePos x=\"0\" y=\"0\"/>"
            + "<wp:positionH relativeFrom=\"column\"><wp:posOffset>0</wp:posOffset></wp:positionH>"
            + "<wp:positionV relativeFrom=\"paragraph\"><wp:posOffset>0</wp:posOffset></wp:positionV>"
            + "<wp:extent cx=\"${cx}\" cy=\"${cy}\"/>"
            + "<wp:effectExtent l=\"0\" t=\"0\" r=\"0\" b=\"0\"/><wp:wrapSquare wrapText=\"bothSides\"/>"
            + "<wp:docPr id=\"${id1}\" name=\"${filenameHint}\" descr=\"${altText}\"/><wp:cNvGraphicFramePr><a:graphicFrameLocks xmlns:a=\"http://schemas.openxmlformats.org/drawingml/2006/main\" noChangeAspect=\"1\"/></wp:cNvGraphicFramePr><a:graphic xmlns:a=\"http://schemas.openxmlformats.org/drawingml/2006/main\">"
            + "<a:graphicData uri=\"http://schemas.openxmlformats.org/drawingml/2006/picture\">"
            + "<pic:pic xmlns:pic=\"http://schemas.openxmlformats.org/drawingml/2006/picture\"><pic:nvPicPr><pic:cNvPr id=\"${id2}\" name=\"${filenameHint}\"/><pic:cNvPicPr/></pic:nvPicPr><pic:blipFill>"
            + "<a:blip " + type + "=\"${rEmbedId}\"/><a:stretch><a:fillRect/></a:stretch></pic:blipFill>"
            + "<pic:spPr><a:xfrm><a:off x=\"0\" y=\"0\"/><a:ext cx=\"${cx}\" cy=\"${cy}\"/></a:xfrm><a:prstGeom prst=\"rect\"><a:avLst/></a:prstGeom></pic:spPr></pic:pic></a:graphicData></a:graphic>"
            + "</wp:anchor>";

        java.util.HashMap<String, String> mappings = new java.util.HashMap<String, String>();

        mappings.put("cx", Integer.toString(cx));
        mappings.put("cy", Integer.toString(cy));
        mappings.put("filenameHint", "");
        mappings.put("altText", "");
        mappings.put("rEmbedId", relId);
        mappings.put("id1", "0");           // it seem that this id does not need to be unique (samples/AddImage.java)...
        mappings.put("id2", "1");

        Anchor anchor = null;
        Object o = org.docx4j.XmlUtils.unmarshallFromTemplate(ml, mappings);
        anchor = (Anchor) ((JAXBElement<?>) o).getValue();
        return anchor;
    }

    private static Inline switchToInlineObject(DrawingComponent drawing)
        throws JAXBException {

        Anchor anchor = (Anchor)drawing.getAnchorOrInline();
        Graphic graphic = anchor.getGraphic();
        Inline inline = createInline(true, "", 0, 0);
        inline.setParent(anchor.getParent());
        inline.setCNvGraphicFramePr(anchor.getCNvGraphicFramePr());
        inline.setDocPr(anchor.getDocPr());
        inline.setGraphic(graphic);
        inline.setExtent(anchor.getExtent());
        inline.setDistB(anchor.getDistB());
        inline.setDistL(anchor.getDistL());
        inline.setDistR(anchor.getDistR());
        inline.setDistT(anchor.getDistT());
        drawing.setAnchorOrInline(inline);
        return inline;
    }

    private static Anchor switchToAnchorObject(DrawingComponent drawing)
        throws JAXBException {

        Inline inline = (Inline)drawing.getAnchorOrInline();
        Graphic graphic = inline.getGraphic();
        Anchor anchor = createAnchor(true, "", 0, 0);
        anchor.setParent(inline.getParent());
        anchor.setCNvGraphicFramePr(inline.getCNvGraphicFramePr());
        anchor.setDocPr(inline.getDocPr());
        anchor.setGraphic(graphic);
        anchor.setExtent(inline.getExtent());
        anchor.setDistB(inline.getDistB());
        anchor.setDistL(inline.getDistL());
        anchor.setDistR(inline.getDistR());
        anchor.setDistT(inline.getDistT());
        drawing.setAnchorOrInline(anchor);
        return anchor;
    }

    public static void applyProperties(OperationDocument operationDocument, JSONObject drawingProperties, DrawingComponent drawing)
        throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

        if(drawingProperties!=null) {
            ObjectFactory dmlObjectFactory = new ObjectFactory();

            String imageUrl = Commons.getImageUrlFromImageData(operationDocument.getResourceManager(), drawingProperties.optString("imageData"), "word/media/");
            if(imageUrl==null) {
                imageUrl = drawingProperties.optString("imageUrl", null);
            }
            if(imageUrl!=null) {
                final Relationship relationship = Commons.createAndApplyGraphic(operationDocument, operationDocument.getPackage().getMainDocumentPart(), imageUrl);
                if(relationship!=null) {

                    final boolean linked = relationship.getTargetMode() != null && relationship.getTargetMode().equals("External");
                    drawing.setAnchorOrInline(Drawings.createInline(linked, relationship.getId(), 500000, 500000));
                }
            }

            Object anchor = drawing.getAnchorOrInline();
            Object inlineObj = drawingProperties.opt("inline");
            if(inlineObj!=null) {
                boolean inline = inlineObj instanceof Boolean ? (Boolean)inlineObj : true;
                if((anchor instanceof Anchor)&&inline)
                    anchor = switchToInlineObject(drawing);
                else if((anchor instanceof Inline)&&(inline==false))
                    anchor = switchToAnchorObject(drawing);
            }
            Iterator<String> keys = drawingProperties.keys();
            while(keys.hasNext()) {
                String attr = keys.next();
                Object value = drawingProperties.get(attr);
                if(attr.equals("width")) {
                    if(value instanceof Integer) {
                        CTPositiveSize2D extend = drawing.getExtend();
                        if(extend==null)
                            extend = new CTPositiveSize2D();
                        extend.setCx(((Integer) value).intValue()*360);
                        CTShapeProperties shapeProperties = drawing.getSpPr();
                        if(shapeProperties!=null) {
                            CTTransform2D transform2D = shapeProperties.getXfrm();
                            if(transform2D!=null) {
                                extend = transform2D.getExt();
                                if(extend!=null)
                                    extend.setCx(((Integer)value).intValue()*360);
                            }
                        }
                    }
                }
                else if(attr.equals("height")) {
                    if(value instanceof Integer) {
                        CTPositiveSize2D extend = drawing.getExtend();
                        if(extend==null)
                            extend = new CTPositiveSize2D();
                        extend.setCy(((Integer) value).intValue()*360);
                        CTShapeProperties shapeProperties = drawing.getSpPr();
                        if(shapeProperties!=null) {
                            CTTransform2D transform2D = shapeProperties.getXfrm();
                            if(transform2D!=null) {
                                extend = transform2D.getExt();
                                if(extend!=null)
                                    extend.setCy(((Integer)value).intValue()*360);
                            }
                        }
                    }
                }
                else if(attr.equals("marginLeft")) {
                    if(value instanceof Integer) {
                        if(anchor instanceof Anchor)
                            ((Anchor)anchor).setDistL(((Integer)value).longValue()*360);
                        else
                            ((Inline)anchor).setDistL(((Integer)value).longValue()*360);
                    }
                }
                else if(attr.equals("marginTop")) {
                    if(value instanceof Integer) {
                        if(anchor instanceof Anchor)
                            ((Anchor)anchor).setDistT(((Integer)value).longValue()*360);
                        else
                            ((Inline)anchor).setDistT(((Integer)value).longValue()*360);
                    }
                }
                else if(attr.equals("marginRight")) {
                    if(value instanceof Integer) {
                        if(anchor instanceof Anchor)
                            ((Anchor)anchor).setDistR(((Integer)value).longValue()*360);
                        else
                            ((Inline)anchor).setDistR(((Integer)value).longValue()*360);
                    }
                }
                else if(attr.equals("marginBottom")) {
                    if(value instanceof Integer) {
                        if(anchor instanceof Anchor)
                            ((Anchor)anchor).setDistB(((Integer)value).longValue()*360);
                        else
                            ((Inline)anchor).setDistB(((Integer)value).longValue()*360);
                    }
                }
                else if(attr.equals("flipH")) {
                    boolean isFlipH = value instanceof Boolean ? (Boolean)value : false;
                    Pic pic = drawing.getPic();
                    if(pic!=null) {
                        CTShapeProperties shapeProperties = drawing.getSpPr();
                        if(shapeProperties==null)
                            shapeProperties = new CTShapeProperties();
                        CTTransform2D transform2D = shapeProperties.getXfrm();
                        if(transform2D==null)
                            transform2D = new CTTransform2D();
                        transform2D.setFlipH(isFlipH);
                    }
                }
                else if(attr.equals("flipV")) {
                    boolean isFlipV = value instanceof Boolean ? (Boolean)value : false;
                    Pic pic = drawing.getPic();
                    if(pic!=null) {
                        CTShapeProperties shapeProperties = drawing.getSpPr();
                        if(shapeProperties==null)
                            shapeProperties = new CTShapeProperties();
                        CTTransform2D transform2D = shapeProperties.getXfrm();
                        if(transform2D==null)
                            transform2D = new CTTransform2D();
                        transform2D.setFlipV(isFlipV);
                    }
                }
                // !!!!.... only attributes for anchored drawings are following now !!!!
                else if(anchor instanceof Anchor) {
                    if(attr.equals("anchorHorBase")) {
                        if(value instanceof String) {
                            CTPosH ctPosH = ((Anchor) anchor).getPositionH();
                            if(ctPosH==null)
                                ctPosH = dmlObjectFactory.createCTPosH();
                            ctPosH.setRelativeFrom(STRelFromH.fromValue((String)value));
                        }
                    }
                    else if(attr.equals("anchorHorAlign")) {
                        if(value instanceof String) {
                            CTPosH ctPosH = ((Anchor) anchor).getPositionH();
                            if(ctPosH==null)
                                ctPosH = dmlObjectFactory.createCTPosH();
                            if(((String)value).equals("offset"))
                                ctPosH.setAlign(null);
                            else
                                ctPosH.setAlign(STAlignH.fromValue((String)value));
                        }
                    }
                    else if(attr.equals("anchorHorOffset")) {
                        if(value instanceof Integer) {
                            CTPosH ctPosH = ((Anchor) anchor).getPositionH();
                            if(ctPosH==null)
                                ctPosH = dmlObjectFactory.createCTPosH();
                            ctPosH.setPosOffset((Integer)value * 360);
                        }
                    }
                    else if(attr.equals("anchorVertBase")) {
                        if(value instanceof String) {
                            CTPosV ctPosV = ((Anchor) anchor).getPositionV();
                            if(ctPosV==null)
                                ctPosV = dmlObjectFactory.createCTPosV();
                            ctPosV.setRelativeFrom(STRelFromV.fromValue((String)value));
                        }
                    }
                    else if(attr.equals("anchorVertAlign")) {
                        if(value instanceof String) {
                            CTPosV ctPosV = ((Anchor) anchor).getPositionV();
                            if(ctPosV==null)
                                ctPosV = dmlObjectFactory.createCTPosV();
                            if(((String)value).equals("offset"))
                                ctPosV.setAlign(null);
                            else
                                ctPosV.setAlign(STAlignV.fromValue((String)value));
                        }
                    }
                    else if(attr.equals("anchorVertOffset")) {
                        if(value instanceof Integer) {
                            CTPosV ctPosV = ((Anchor) anchor).getPositionV();
                            if(ctPosV==null)
                                ctPosV = dmlObjectFactory.createCTPosV();
                            ctPosV.setPosOffset((Integer)value * 360);
                        }
                    }
                    else if(attr.equals("textWrapMode")) {
                        if(value instanceof String) {
                            CTWrapNone      ctWrapNone = null;
                            CTWrapSquare    ctWrapSquare = null;
                            CTWrapTight     ctWrapTight = null;
                            CTWrapThrough   ctWrapThrough = null;
                            CTWrapTopBottom ctWrapTopBottom = null;
                            if(((String)value).equals("none")) {
                                ctWrapNone = ((Anchor)anchor).getWrapNone();
                                if(ctWrapNone==null)
                                    ctWrapNone = dmlObjectFactory.createCTWrapNone();
                            }
                            else if(((String)value).equals("square")) {
                                ctWrapSquare = ((Anchor)anchor).getWrapSquare();
                                if(ctWrapSquare==null) {
                                    ctWrapSquare = dmlObjectFactory.createCTWrapSquare();
                                    ctWrapSquare.setWrapText(STWrapText.BOTH_SIDES);
                                }
                            }
                            else if(((String)value).equals("tight")) {
                                ctWrapTight = ((Anchor)anchor).getWrapTight();
                                if(ctWrapTight==null) {
                                    ctWrapTight = dmlObjectFactory.createCTWrapTight();
                                    ctWrapTight.setWrapText(STWrapText.BOTH_SIDES);
                                }
                            }
                            else if(((String)value).equals("through")) {
                                ctWrapThrough = ((Anchor)anchor).getWrapThrough();
                                if(ctWrapThrough==null) {
                                    ctWrapThrough = dmlObjectFactory.createCTWrapThrough();
                                    ctWrapThrough.setWrapText(STWrapText.BOTH_SIDES);
                                }
                            }
                            else if(((String)value).equals("topAndBottom")) {
                                ctWrapTopBottom = ((Anchor)anchor).getWrapTopAndBottom();
                                if(ctWrapTopBottom==null)
                                    ctWrapTopBottom = dmlObjectFactory.createCTWrapTopBottom();
                            }
                            ((Anchor)anchor).setWrapNone(ctWrapNone);
                            ((Anchor)anchor).setWrapSquare(ctWrapSquare);
                            ((Anchor)anchor).setWrapTight(ctWrapTight);
                            ((Anchor)anchor).setWrapThrough(ctWrapThrough);
                            ((Anchor)anchor).setWrapTopAndBottom(ctWrapTopBottom);
                        }
                    }
                }
            }

            // either "align" or "position" is to be used, otherwise Word won't open the document
            if(anchor instanceof Anchor) {
                CTPosH ctPosH = ((Anchor) anchor).getPositionH();
                if(ctPosH!=null) {
                    if(ctPosH.getAlign()!=null)
                        ctPosH.setPosOffset(null);
                    else {
                        Integer pos = ctPosH.getPosOffset();
                        if(pos==null)
                            ctPosH.setPosOffset(0);
                    }
                }
                CTPosV ctPosV = ((Anchor) anchor).getPositionV();
                if(ctPosV!=null) {
                    if(ctPosV.getAlign()!=null)
                        ctPosV.setPosOffset(null);
                    else {
                        Integer pos = ctPosV.getPosOffset();
                        if(pos==null)
                            ctPosV.setPosOffset(0);
                    }
                }
                // textwrapside can't be set inside the above while loop, because it depends to the
                // wrapmode that is used. late applying preserves us from taking care of the property order.
                if (drawingProperties.has("textWrapSide")) {
                    Object value = drawingProperties.get("textWrapSide");
                    if(value instanceof String) {
                        String wrapText = ((String)value).equals("both") ? "bothSides" : (String)value;
                        if(((Anchor)anchor).getWrapSquare()!=null)
                            ((Anchor)anchor).getWrapSquare().setWrapText(STWrapText.fromValue(wrapText));
                        else if (((Anchor)anchor).getWrapTight()!=null)
                            ((Anchor)anchor).getWrapTight().setWrapText(STWrapText.fromValue(wrapText));
                        else if (((Anchor)anchor).getWrapThrough()!=null)
                            ((Anchor)anchor).getWrapThrough().setWrapText(STWrapText.fromValue(wrapText));
                    }
                }
            }
        }
    }

    final static String[] enumRelFromH = { "margin", "page", "column", "character", "leftMargin", "rightMargin", "insideMargin", "outsideMargin" };
    final static String[] enumSTAlignH = { "left", "right", "center", "inside", "outside" };
    final static String[] enumRelFromV = { "margin", "page", "paragraph", "line", "topmargin", "bottomMargin", "insideMargin", "outsideMargin" };
    final static String[] enumSTAlignV = { "top", "bottom", "center", "inside", "outside" };
    final static String[] enumSTWrapText = { "both", "left", "right", "largest" };

    public static JSONObject createProperties(Part mainDocumentPart, DrawingComponent drawing)
        throws JSONException {

        JSONObject jsonDrawingProperties = new JSONObject();

        CTBlipFillProperties blipFillProperties = drawing.getBlipFill();
        if(blipFillProperties!=null) {
            DMLHelper.createJsonFromBlipFillProperties(jsonDrawingProperties, drawing.getBlipFill());
            if(drawing.getType()==GraphicType.IMAGE || drawing.getType()==GraphicType.OLE) {
                jsonDrawingProperties.put("imageUrl", DMLHelper.getBlipUrl(mainDocumentPart, blipFillProperties.getBlip()));
            }
        }

        DMLHelper.createJsonFromNonVisualDrawingProperties(jsonDrawingProperties, drawing.getDocPr());
        DMLHelper.createJsonFromShapeProperties(jsonDrawingProperties, drawing.getSpPr());
        DMLHelper.createJsonFromPositiveSize2D(jsonDrawingProperties, drawing.getExtend());

        Object anchor = drawing.getAnchorOrInline();
        if(anchor!=null) {
            jsonDrawingProperties.put("inline", anchor instanceof Inline);

            // applying margin properties
            Long DistL = null;
            Long DistT = null;
            Long DistR = null;
            Long DistB = null;
            if(anchor instanceof Anchor) {
                DistL = ((Anchor)anchor).getDistL();
                DistT = ((Anchor)anchor).getDistT();
                DistB = ((Anchor)anchor).getDistB();
                DistR = ((Anchor)anchor).getDistR();
            } else {
                DistL = ((Inline)anchor).getDistL();
                DistT = ((Inline)anchor).getDistT();
                DistB = ((Inline)anchor).getDistB();
                DistR = ((Inline)anchor).getDistR();
            }
            if((DistL!=null) && (DistL.intValue() != 0))
                jsonDrawingProperties.put("marginLeft", DistL.intValue() / 360);
            if((DistT!=null) && (DistT.intValue() != 0))
                jsonDrawingProperties.put("marginTop", DistT.intValue() / 360);
            if((DistR!=null) && (DistR.intValue() != 0))
                jsonDrawingProperties.put("marginRight", DistR.intValue() / 360);
            if((DistB!=null) && (DistB.intValue() != 0))
                jsonDrawingProperties.put("marginBottom", DistB.intValue() / 360);

            // positioning in word
            if(anchor instanceof Anchor) {
                // horizontal
                CTPosH ctPosH = ((Anchor) anchor).getPositionH();
                if(ctPosH!=null) {
                    STRelFromH stRelFrom = ctPosH.getRelativeFrom();
                    if(stRelFrom!=null) {
                        int val = stRelFrom.ordinal();
                        if(val>=enumRelFromH.length)
                            val = 0;
                        jsonDrawingProperties.put("anchorHorBase", enumRelFromH[val] );
                    }
                    STAlignH stAlignH = ctPosH.getAlign();
                    if(stAlignH!=null) {
                        int val = stAlignH.ordinal();
                        if(val>=enumSTAlignH.length)
                            val = 0;
                        jsonDrawingProperties.put("anchorHorAlign", enumSTAlignH[val]);
                    }
                    else
                        jsonDrawingProperties.put("anchorHorAlign", "offset");
                    Integer val = ctPosH.getPosOffset();
                    if(val!=null) {
                        jsonDrawingProperties.put("anchorHorOffset", val.intValue() / 360);
                    }
                }
                // vertical
                CTPosV ctPosV = ((Anchor) anchor).getPositionV();
                if(ctPosV!=null) {
                    STRelFromV stRelFrom = ctPosV.getRelativeFrom();
                    if(stRelFrom!=null) {
                        int val = stRelFrom.ordinal();
                        if(val>=enumRelFromV.length)
                            val = 0;
                        jsonDrawingProperties.put("anchorVertBase", enumRelFromV[val] );
                    }
                    STAlignV stAlignV = ctPosV.getAlign();
                    if(stAlignV!=null) {
                        int val = stAlignV.ordinal();
                        if(val>=enumSTAlignV.length)
                            val = 0;
                        jsonDrawingProperties.put("anchorVertAlign", enumSTAlignV[val]);
                    }
                    else
                        jsonDrawingProperties.put("anchorVertAlign", "offset");
                    Integer val = ctPosV.getPosOffset();
                    if(val!=null) {
                        jsonDrawingProperties.put("anchorVertOffset",val.intValue() / 360);
                    }
                }
                // textwrapmode
                String textWrapMode = null;
                STWrapText textWrapSide = null;
                CTWrapNone wrapNone = ((Anchor)anchor).getWrapNone();
                if(wrapNone!=null)
                    textWrapMode = "none";
                else {
                    CTWrapSquare wrapSquare = ((Anchor)anchor).getWrapSquare();
                    if(wrapSquare!=null) {
                        textWrapMode = "square";
                        textWrapSide = wrapSquare.getWrapText();
                    }
                    else {
                        CTWrapTight wrapTight = ((Anchor)anchor).getWrapTight();
                        if(wrapTight!=null) {
                            textWrapMode = "tight";
                            textWrapSide = wrapTight.getWrapText();
                        }
                        else {
                            CTWrapThrough wrapThrough = ((Anchor)anchor).getWrapThrough();
                            if(wrapThrough!=null) {
                                textWrapMode = "through";
                                textWrapSide = wrapThrough.getWrapText();
                            }
                            else {
                                CTWrapTopBottom wrapTopAndBottom = ((Anchor)anchor).getWrapTopAndBottom();
                                if(wrapTopAndBottom!=null)
                                    textWrapMode = "topAndBottom";
                            }
                        }
                    }
                }
                if(textWrapMode!=null)
                    jsonDrawingProperties.put("textWrapMode", textWrapMode);
                if(textWrapSide!=null) {
                    int val = textWrapSide.ordinal();
                    if(val>=enumSTWrapText.length)
                        val = 0;
                    jsonDrawingProperties.put("textWrapSide", enumSTWrapText[val]);
                }
            }
        }
        return jsonDrawingProperties.length() > 0 ? jsonDrawingProperties : null;
    }
}
