/*
 * Copyright 2012 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.odftoolkit.odfdom.component;

import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.imageio.ImageIO;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.OdfSchemaDocument;
import org.odftoolkit.odfdom.dom.OdfStylesDom;
import org.odftoolkit.odfdom.dom.element.OdfStylableElement;
import org.odftoolkit.odfdom.dom.element.OdfStyleBase;
import org.odftoolkit.odfdom.dom.element.OdfStylePropertiesBase;
import org.odftoolkit.odfdom.dom.element.office.OfficeMasterStylesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleListLevelLabelAlignmentElement;
import org.odftoolkit.odfdom.dom.element.style.StyleListLevelPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleMasterPageElement;
import org.odftoolkit.odfdom.dom.element.style.StyleParagraphPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTabStopsElement;
import org.odftoolkit.odfdom.dom.element.text.TextListLevelStyleBulletElement;
import org.odftoolkit.odfdom.dom.element.text.TextListLevelStyleElementBase;
import org.odftoolkit.odfdom.dom.element.text.TextListLevelStyleImageElement;
import org.odftoolkit.odfdom.dom.element.text.TextListLevelStyleNumberElement;
import org.odftoolkit.odfdom.dom.element.text.TextListStyleElement;
import org.odftoolkit.odfdom.dom.element.text.TextParagraphElementBase;
import org.odftoolkit.odfdom.dom.style.OdfStyleFamily;
import org.odftoolkit.odfdom.dom.style.props.OdfStylePropertiesSet;
import org.odftoolkit.odfdom.dom.style.props.OdfStyleProperty;
import org.odftoolkit.odfdom.incubator.doc.office.OdfOfficeAutomaticStyles;
import org.odftoolkit.odfdom.incubator.doc.office.OdfOfficeStyles;
import org.odftoolkit.odfdom.incubator.doc.style.OdfDefaultStyle;
import org.odftoolkit.odfdom.incubator.doc.style.OdfStyle;
import org.odftoolkit.odfdom.incubator.doc.style.OdfStylePageLayout;
import org.odftoolkit.odfdom.pkg.OdfElement;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.odftoolkit.odfdom.type.Length;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.odftoolkit.odfdom.pkg.OdfNamespace;
import static org.odftoolkit.odfdom.component.OdfFileSaxHandler.*;
import org.odftoolkit.odfdom.doc.OdfTextDocument;
import org.odftoolkit.odfdom.dom.element.style.StyleTabStopElement;
import org.odftoolkit.odfdom.pkg.OdfPackage;

/**
 * ToDo: Is it more flexible to build a different queue for OperationQueue and
 * create an JSON exporter? Can a JSONArray / JSONObject be initialized with an
 * existing queue?
 *
 * @author svante.schubertATgmail.com
 */
public class JsonOperationProducer {

	private static final Logger LOG = Logger.getLogger(JsonOperationProducer.class.getName());
	static final String TRANSPARENT = "transparent";
	static final String HASH = "#";
	static final String BLACK = "#000000";
	// line widths constants
	static final String AUTO = "auto";
	static final String NORMAL = "normal";
	static final String BOLD = "bold";
	static final String THIN = "thin";
	static final String MEDIUM = "medium";
	static final String THICK = "thick";
	static final String OX_DEFAULT_LIST = "OX_DEFAULT_LIST";
	static final String OX_DEFAULT_STYLE_PREFIX = "default_";
	static final String OX_DEFAULT_STYLE_SUFFIX = "_style";
	private final JSONArray mOperationQueue = new JSONArray();
	private final JSONObject mOperations = new JSONObject();
	private static final char PATH_SEPARATOR = '/';
	private static final String PERCENT = "%";
	private static final double DOTS_PER_INCH = 72.0;
	/**
	 * The maximum empty cell number before starting a new operation
	 */
	// a color, which type is set to auto - adapting color to environment
	private static final Map<String, String> COLOR_MAP_AUTO = createColorMap(AUTO);
	/**
	 * Every knonwStyle does not have to be read
	 */
	Map knownStyles = new HashMap<String, Boolean>();
	// Added an own map for list styles as it is not 100% certain that the names between styles and list style might be overlapping.
	Map knownListStyles = new HashMap<String, Boolean>();
	/**
	 * There is a special style for the replacement table of too large tables,
	 * which have to be added only once to a document
	 */
	boolean mIsTableExceededStyleAdded = false;

	public JsonOperationProducer() {
		try {
			mOperations.put("operations", mOperationQueue);
		} catch (JSONException e) {
			Logger.getLogger(JsonOperationProducer.class.getName()).log(Level.SEVERE, null, e);
		}
	}

	public JSONObject getDocumentOperations() {
		return mOperations;
	}

	/**
	 * Used for repeated elements, repeats a set of operations.
	 *
	 * @param from where the repetition of operations should start.
	 * @param times the operations should be repeated.
	 */
	// ToDo Svante
	public void repeatOperations(int from, int times) {
		int initialInsert = mOperations.length();

		// repetition of the component operations (and its chilrden)
		for (int t = 1; t <= times; t++) {
			int startInsert = initialInsert * times;
			for (int i = from; i < initialInsert; i++) {
				mOperationQueue.optJSONObject(i);
				mOperationQueue.put(startInsert + i);
			}
		}
	}

	/**
	 * Used for repeated elements, repeats a set of operations
	 */
	public int getCurrentOperationIndex() {
		return mOperations.length() - 1;
	}

//	// -------------------------------------------------------------------------
//	/**
//	 * @param start: An array, that contains the number of that paragraph,
//	 * before which the new paragraph shall be inserted. Has the last paragraph
//	 * the number 2, so causes para=3, that the new paragraph will be inserted
//	 * at the end. para=4 is not allowed in this case.
//	 */
//	void addChild(String componentType, final List<Integer> start) {
//
//		final JSONObject insertComponentObject = new JSONObject();
//
//		try {
//			insertComponentObject.put("name", "insert" + componentType);
//			insertComponentObject.put("start", start);
//			mOperationQueue.put(insertComponentObject);
//			LOG.log(Level.FINEST, "insert" + componentType + " - component:{0}", insertComponentObject);
//
//		} catch (JSONException e) {
//			Logger.getLogger(JsonOperationProducer.class.getName()).log(Level.SEVERE, null, e);
//		}
//	}
	// -------------------------------------------------------------------------
	/**
	 * @param * @param start: An array, that contains the number of that
	 * paragraph, before which the new paragraph shall be inserted. Has the last
	 * paragraph the number 2, so causes para=3, that the new paragraph will be
	 * inserted at the end. para=4 is not allowed in this case.
	 */
	void add(String componentType, final List<Integer> start, final Map<String, Object> formattingProperties) {
// ToDo: Moving styleId into para list - 
		// final Map<String, Object> args, 

		final JSONObject insertComponentObject = new JSONObject();

		try {
			insertComponentObject.put("name", "insert" + componentType);
			insertComponentObject.put("start", start);
// ToDo: Moving styleId into para list - 			
//			if (args != null) {
//				for (String arg : args.keySet()) {
//					insertComponentObject.put(arg, args.get(arg));
//				}
//			}
			if (formattingProperties != null && !formattingProperties.isEmpty()) {
				JSONObject attrs = new JSONObject();
				for (String arg : formattingProperties.keySet()) {
					attrs.put(arg, formattingProperties.get(arg));
				}
				insertComponentObject.put("attrs", attrs);
			}
// IN CASE STYLE BECOMES AN ATTRIBUTE			
//			if (args != null || formattingProperties != null && !formattingProperties.isEmpty()) {
//				JSONObject attrs = new JSONObject();
//
//				if (args != null) {
//					for (String arg : args.keySet()) {
//						attrs.put(arg, args.get(arg));
//					}
//				}
//				if (formattingProperties != null && !formattingProperties.isEmpty()) {
//					for (String arg : formattingProperties.keySet()) {
//						attrs.put(arg, formattingProperties.get(arg));
//					}
//					insertComponentObject.put("attrs", attrs);
//				}
//			}			

			mOperationQueue.put(insertComponentObject);
			LOG.log(Level.FINEST, "insert" + componentType + " - component:{0}", insertComponentObject);

		} catch (JSONException e) {
			Logger.getLogger(JsonOperationProducer.class.getName()).log(Level.SEVERE, null, e);
		}
	}

	/**
	 * @param start: An array, that contains the number of that paragraph,
	 * before which the new paragraph shall be inserted. Has the last paragraph
	 * the number 2, so causes para=3, that the new paragraph will be inserted
	 * at the end. para=4 is not allowed in this case.
	 */
	void formatColumns(final List<Integer> start, final Map<String, Object> formattingProperties, Integer firstColumn, Integer lastColumn) {
		if (formattingProperties != null && !formattingProperties.isEmpty()) {
			final JSONObject insertComponentObject = new JSONObject();

			try {
				insertComponentObject.put("name", "setColumnAttributes");
				insertComponentObject.put("sheet", start.get(0));
				insertComponentObject.put("start", firstColumn);
				if (lastColumn != null && !firstColumn.equals(lastColumn)) {
					insertComponentObject.put("end", lastColumn);
				}
				if (formattingProperties != null && !formattingProperties.isEmpty()) {
					JSONObject attrs = new JSONObject();
					for (String arg : formattingProperties.keySet()) {
						attrs.put(arg, formattingProperties.get(arg));
					}
					insertComponentObject.put("attrs", attrs);
				}
				mOperationQueue.put(insertComponentObject);
				LOG.log(Level.FINEST, "setColumnAttributes - component:{0}", insertComponentObject);

			} catch (JSONException e) {
				Logger.getLogger(JsonOperationProducer.class.getName()).log(Level.SEVERE, null, e);
			}
		}
	}

	/**
	 * @param start: An array, that contains the number of that paragraph,
	 * before which the new paragraph shall be inserted. Has the last paragraph
	 * the number 2, so causes para=3, that the new paragraph will be inserted
	 * at the end. para=4 is not allowed in this case.
	 */
	void formatRows(final List<Integer> start, final Map<String, Object> formattingProperties, Integer firstRow, Integer lastRow, Integer previousRowRepeated) {
		if (formattingProperties != null && !formattingProperties.isEmpty()) {
			final JSONObject insertComponentObject = new JSONObject();

			try {
				insertComponentObject.put("name", "setRowAttributes");
				insertComponentObject.put("sheet", start.get(0));
				insertComponentObject.put("start", firstRow + previousRowRepeated);
				if (lastRow != null && !firstRow.equals(lastRow)) {
					insertComponentObject.put("end", lastRow + previousRowRepeated);
				}
				if (formattingProperties != null && !formattingProperties.isEmpty()) {
					JSONObject attrs = new JSONObject();
					for (String arg : formattingProperties.keySet()) {
						attrs.put(arg, formattingProperties.get(arg));
					}
					insertComponentObject.put("attrs", attrs);
				}
				mOperationQueue.put(insertComponentObject);
				LOG.log(Level.FINEST, "setRowAttributes - component:{0}", insertComponentObject);

			} catch (JSONException e) {
				Logger.getLogger(JsonOperationProducer.class.getName()).log(Level.SEVERE, null, e);
			}
		}
	}
	private static final String NUMBER_FORMAT_CODE = "code";
	private static final String NUMBER_FORMAT_CODE_STANDARD = "Standard";

	/**
	 * Maps the ODF office:value-type attribute to the OX Text numberFormat
	 * property
	 */
	JSONObject getCellNumberFormat(String valueType) {
		JSONObject cellNumberFormat = new JSONObject();
		try {
			if (valueType == null || valueType.isEmpty()) {
				cellNumberFormat.put(NUMBER_FORMAT_CODE, NUMBER_FORMAT_CODE_STANDARD);
//			} else if (valueType.equals("1")) {
//			} else if (valueType.equals("i")) {
//			} else if (valueType.equals("I")) {
//			} else if (valueType.equals("a")) {
//			} else if (valueType.equals("A")) {
			} else {
				cellNumberFormat.put(NUMBER_FORMAT_CODE, NUMBER_FORMAT_CODE_STANDARD);
			}
		} catch (JSONException ex) {
			Logger.getLogger(JsonOperationProducer.class.getName()).log(Level.SEVERE, null, ex);
		}
		return cellNumberFormat;
	}

	public void addRange(int sheet, Integer firstRow, Integer lastRow, int repeatedRowOffset, int firstContentCell, int horizontalRepetition, JSONObject repeatedCell, JSONArray singleRow, boolean hasHorizontalRepetition) {

		// the row start/end number 0 based, therefore - 1
		int rowStartNo = firstRow + repeatedRowOffset;
		List rangeStart = new LinkedList<Integer>();

		// if there is a repeated row, there will be repeated cells (at least vertical)
		if (hasHorizontalRepetition || lastRow != null && !firstRow.equals(lastRow)) {
			// first the start column position 
			rangeStart.add(firstContentCell);
			// second the start row position 
			rangeStart.add(rowStartNo);

			List rangeEnd = new LinkedList<Integer>();

			// first the end column position: StartPos of content plus any repetiton (including itself, therefore - 1)
			rangeEnd.add(firstContentCell + horizontalRepetition - 1);
			// second the end row position 
			int rowEndNo = lastRow + repeatedRowOffset;
			rangeEnd.add(rowEndNo);

			// create a operation for the given cell range with similar content
			fillCellRange(sheet, rangeStart, rangeEnd, repeatedCell);
		} else {
			// first the start column position
			rangeStart.add(firstContentCell);
			// second the start row position 
			rangeStart.add(rowStartNo);
			setCellContents(sheet, rangeStart, singleRow);
		}
	}

	/**
	 * Writes a range of the spreadsheet with various values. Will be called
	 * only indirectly after a spreadsheet row has checked for optimization.
	 *
	 * @param name	String	'setCellContents'
	 * @param sheet	Integer	The zero-based index of the sheet containing the
	 * cell range.
	 * @param firstRow	Integer	The next row number being written
	 * @param lastRow	Integer The last row number being written (different when
	 * the row was repeated)
	 * @param spreadsheetRange	Object[][]	The values and attribute sets to be
	 * written into the cell range. The outer array contains rows of cell
	 * contents, and the inner row arrays contain the cell contents for each
	 * single row. The lengths of the inner arrays may be different. Cells not
	 * covered by a row array will not be modified.
	 */
	private void setCellContents(Integer sheet, List rangeStart, JSONArray spreadsheetRange) {
		final JSONObject insertComponentObject = new JSONObject();

		try {
			insertComponentObject.put("name", "setCellContents");
			insertComponentObject.put("sheet", sheet);
			insertComponentObject.put("start", rangeStart);
			// Although we only deliver a single row, the range have to be two dimensional
			insertComponentObject.put("contents", new JSONArray().put(spreadsheetRange));
			mOperationQueue.put(insertComponentObject);
			LOG.log(Level.FINEST, "setCellContents - component:{0}", insertComponentObject);
		} catch (JSONException e) {
			Logger.getLogger(JsonOperationProducer.class
					.getName()).log(Level.SEVERE, null, e);
		}
	}

	/**
	 * Writes a range of the spreadsheet with various identical/repeating values
	 * and attributes
	 *
	 * Will be called only indirectly after a spreadsheet row has checked for
	 * optimization.
	 *
	 * @param name	String	'fillCellRange'
	 * @param sheet	Integer	The zero-based index of the sheet containing the
	 * cell range.
	 * @param start	Integer[2]	The logical cell position of the upper-left cell
	 * in the range.
	 * @param end	Integer[2]	(optional) The logical cell position of the
	 * bottom-right cell in the range. If omitted, the operation addresses a
	 * single cell.
	 * @param value	CellValue	(optional) The value used to fill the specified
	 * cell range. The value null will clear the cell range. If omitted, the
	 * current values will not change (e.g., to change the formatting only),
	 * except for shared formulas referred by the shared attribute of this
	 * operation. If the parse property is set in the operation, the value must
	 * be a string.
	 */
	private void fillCellRange(int sheet, final List<Integer> start, final List<Integer> end, JSONObject cell) {
		final JSONObject insertComponentObject = new JSONObject();

		try {
			if (cell != null && cell.length() != 0) {
				insertComponentObject.put("name", "fillCellRange");
				insertComponentObject.put("sheet", sheet);
				insertComponentObject.put("start", start);
				if (end != null) {
					insertComponentObject.put("end", end);
				}
				if (cell.has("value")) {
					insertComponentObject.put("value", cell.get("value"));
				}
				if (cell.has("attrs")) {
					insertComponentObject.put("attrs", cell.get("attrs"));
				}
				mOperationQueue.put(insertComponentObject);
				LOG.log(Level.FINEST, "fillCellRange - component:{0}", insertComponentObject);

			}
		} catch (JSONException e) {
			Logger.getLogger(JsonOperationProducer.class
					.getName()).log(Level.SEVERE, null, e);
		}
	}

	// -------------------------------------------------------------------------
	/**
	 * @param text: text to be inserted into the specified paragraph at the
	 * specified position
	 * @param start: an array that contains the position, where the new text
	 * shall be inserted.
	 */
	void addText(final List<Integer> start, final String text) {

		final JSONObject insertTextObject = new JSONObject();

		try {
			insertTextObject.put("text", text);
			insertTextObject.put("start", start);
			insertTextObject.put("name", "insertText");
			mOperationQueue.put(insertTextObject);

			LOG.log(Level.FINEST, "insertText - pos:{0} text:{1}", new Object[]{getComponentPath(start), text});



		} catch (JSONException e) {
			Logger.getLogger(JsonOperationProducer.class
					.getName()).log(Level.SEVERE, null, e);
		}
	}

	// -------------------------------------------------------------------------
	/**
	 * @param para
	 * @param start
	 * @param end
	 * @return
	 */
	@SuppressWarnings("unused")
	void setAttributes(final List<Integer> start, Map<String, Object> attrs) {
		setAttributes(start, null, attrs);
	}

	// -------------------------------------------------------------------------
	/**
	 * @param para
	 * @param start
	 * @param end
	 * @return
	 */
	@SuppressWarnings("unused")
	void setAttributes(final List<Integer> start, final List<Integer> end, Map<String, Object> attrs) {
		if (attrs != null && attrs.size() > 0) {
			// Not the next position, but the last character to be marked will be referenced
			final JSONObject newOp = new JSONObject();
			List<Integer> lastCharacterPos = new LinkedList<Integer>();
			// text position is usually -1 as we take the first and the last character to be styled
			if (end != null) {
				for (int i = 0; i < end.size(); i++) {
					Integer pos = end.get(i);
					if (i == end.size() - 1) {
						// Special case the span is empty, in this case it shall not be -1
						if (pos != 0) {
							pos--;
						}
					}
					lastCharacterPos.add(pos);
				}
			}

			try {
				newOp.put("name", "setAttributes");
				newOp.put("start", start);
				boolean isValidOperation = true;
				if (end != null) {
					newOp.put("end", lastCharacterPos);
					if (start.get(start.size() - 1) > lastCharacterPos.get(start.size() - 1)) {
						isValidOperation = false;
						LOG.fine("Neglecting 'setAttributes' with start:" + start.toString() + " and end:" + end.toString());
					}
				}
				newOp.put("attrs", attrs);
				if (isValidOperation) {
					mOperationQueue.put(newOp);
				}

				LOG.log(Level.FINEST, "New Operation 'setAttributes':" + newOp);


			} catch (JSONException e) {
				Logger.getLogger(JsonOperationProducer.class
						.getName()).log(Level.SEVERE, null, e);
			}
		}
	}

	// -------------------------------------------------------------------------
	/**
	 * @param para
	 * @param start
	 * @param end
	 * @return
	 */
	@SuppressWarnings("unused")
	void deleteText(final List<Integer> start, final List<Integer> end) {

		final JSONObject aDeleteTextObject = new JSONObject();

		try {
			aDeleteTextObject.put("name", "deleteText");
			aDeleteTextObject.put("start", start);
			aDeleteTextObject.put("end", end);

			mOperationQueue.put(aDeleteTextObject);


		} catch (JSONException e) {
			Logger.getLogger(JsonOperationProducer.class
					.getName()).log(Level.SEVERE, null, e);
		}
	}

	void addImage(final List<Integer> start, Map<String, Object> hardFormatations) {

		final JSONObject insertComponentObject = new JSONObject();
		try {
			insertComponentObject.put("name", "insertDrawing");
			insertComponentObject.put("type", "image");
			insertComponentObject.put("start", start);
			if (hardFormatations != null) {
//				scaleWidthToPage(hardFormatations);
				insertComponentObject.put("attrs", hardFormatations);
			}
			mOperationQueue.put(insertComponentObject);
			LOG.log(Level.FINEST, "insertDrawing (image)" + " - component:{0}", insertComponentObject);



		} catch (JSONException e) {
			Logger.getLogger(JsonOperationProducer.class
					.getName()).log(Level.SEVERE, null, e);
		}
	}

	void addShape(final List<Integer> start, Map<String, Object> hardFormatations) {

		final JSONObject insertComponentObject = new JSONObject();
		try {
			insertComponentObject.put("name", "insertDrawing");
			insertComponentObject.put("type", "shape");
			insertComponentObject.put("start", start);
			if (hardFormatations != null) {
				insertComponentObject.put("attrs", hardFormatations);
			}
			mOperationQueue.put(insertComponentObject);
			LOG.log(Level.FINEST, "insertDrawing (shape)" + " - component:{0}", insertComponentObject);



		} catch (JSONException e) {
			Logger.getLogger(JsonOperationProducer.class
					.getName()).log(Level.SEVERE, null, e);
		}
	}

	/**
	 * Special Table color mJsonOperationProducer.addChild("Table", position,
	 * props, mTableColor);
	 */
	// -------------------------------------------------------------------------
	/**
	 * @param * @param start: An array, that contains the number of that
	 * paragraph, before which the new paragraph shall be inserted. Has the last
	 * paragraph the number 2, so causes para=3, that the new paragraph will be
	 * inserted at the end. para=4 is not allowed in this case.
	 */
	void addTable(final List<Integer> start, Map<String, Object> hardFormatations, final List<Integer> tableGrid, String tableName, boolean isSpreadsheet) {

		final JSONObject insertComponentObject = new JSONObject();

		try {
			if (isSpreadsheet) {
				insertComponentObject.put("name", "insertSheet");
				insertComponentObject.put("sheet", start.get(0));
				if (tableName != null && !tableName.isEmpty()) {
					insertComponentObject.put("sheetName", tableName);
				}
//				// move element style attribute into style map
//				if (!isTableVisible) {
//					hardFormatations = setSheetVisibility(hardFormatations, isTableVisible);
//					insertComponentObject.put("attrs", hardFormatations);
//				}
				if (hardFormatations != null && !hardFormatations.isEmpty()) {
					JSONObject tableAttrs = (JSONObject) hardFormatations.get("table");
					if (tableAttrs != null) {
						// SPREADHEET SPECIAL HANDLING:
						// unfortunately the sheet properties are using a different key..						
						// renaming propertes to "sheet"..
						hardFormatations.put("sheet", tableAttrs);
						// ..from originnal "table"
						hardFormatations.remove("table");
						insertComponentObject.put("attrs", hardFormatations);
					}
				}
				mOperationQueue.put(insertComponentObject);
				LOG.log(Level.FINEST, "insertSheet" + " - component:{0}", insertComponentObject);
			} else {
				insertComponentObject.put("name", "insertTable");
				insertComponentObject.put("start", start);
				JSONObject tableAttrs = null;
				if (hardFormatations != null && !hardFormatations.isEmpty()) {
					tableAttrs = (JSONObject) hardFormatations.get("table");
				} else {
					if (hardFormatations == null) {
						hardFormatations = new HashMap<String, Object>();
					}
				}
				if (tableAttrs == null) {
					tableAttrs = new JSONObject();
				}
				if (tableGrid != null) {
					tableAttrs.put("tableGrid", tableGrid);
				}
				hardFormatations.put("table", tableAttrs);
				insertComponentObject.put("attrs", hardFormatations);
				mOperationQueue.put(insertComponentObject);
				LOG.log(Level.FINEST, "insertTable" + " - component:{0}", insertComponentObject);


			}
		} catch (JSONException e) {
			Logger.getLogger(JsonOperationProducer.class
					.getName()).log(Level.SEVERE, null, e);
		}
	}

	/* If the sheetVisiblity is not TRUE, it will be included into the given hardFormatations Map, within another Map referenced by the key "sheet", which will be created on demand if NULL. **/
	private static Map<String, Object> setSheetVisibility(Map<String, Object> hardFormatations, Boolean isTableVisible) {
		JSONObject tableAttrs = null;
		// set sheet visibility				
		if (!isTableVisible) {
			if (hardFormatations != null && !hardFormatations.isEmpty()) {
				tableAttrs = (JSONObject) hardFormatations.get("sheet");
			} else {
				if (hardFormatations == null) {
					hardFormatations = new HashMap<String, Object>();
				}
			}
			if (tableAttrs == null) {
				tableAttrs = new JSONObject();
			}
			try {
				tableAttrs.put("visible", Boolean.FALSE);
			} catch (JSONException ex) {
				Logger.getLogger(JsonOperationProducer.class.getName()).log(Level.SEVERE, null, ex);
			}
			hardFormatations.put("sheet", tableAttrs);
		}
		return hardFormatations;
	}

	private void addTableExceededStyle() {
		try {
			JSONArray operations = new JSONArray(new JSONTokener("[{\"styleName\":\"Light Shading Accent 1\",\"styleId\":\"LightShading-Accent1\",\"attrs\":{\"firstRow\":{\"paragraph\":{\"marginBottom\":0,\"lineHeight\":{\"value\":100,\"type\":\"percent\"},\"marginTop\":0},\"cell\":{\"borderInsideVert\":{\"style\":\"none\"},\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"accent1\",\"type\":\"scheme\"},\"width\":35},\"borderInsideHor\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"accent1\",\"type\":\"scheme\"},\"width\":35},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"}},\"character\":{\"bold\":true}},\"lastRow\":{\"paragraph\":{\"marginBottom\":0,\"lineHeight\":{\"value\":100,\"type\":\"percent\"},\"marginTop\":0},\"cell\":{\"borderInsideVert\":{\"style\":\"none\"},\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"accent1\",\"type\":\"scheme\"},\"width\":35},\"borderInsideHor\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"accent1\",\"type\":\"scheme\"},\"width\":35},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"}},\"character\":{\"bold\":true}},\"band1Hor\":{\"cell\":{\"borderInsideVert\":{\"style\":\"none\"},\"fillColor\":{\"value\":\"accent1\",\"type\":\"scheme\",\"transformations\":[{\"value\":24706,\"type\":\"tint\"}]},\"borderInsideHor\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"}}},\"lastCol\":{\"character\":{\"bold\":true}},\"wholeTable\":{\"paragraph\":{\"marginBottom\":0,\"lineHeight\":{\"value\":100,\"type\":\"percent\"}},\"table\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"accent1\",\"type\":\"scheme\"},\"width\":35},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"accent1\",\"type\":\"scheme\"},\"width\":35},\"paddingBottom\":0,\"paddingLeft\":190,\"paddingRight\":190},\"character\":{\"color\":{\"value\":\"accent1\",\"type\":\"scheme\",\"transformations\":[{\"value\":74902,\"type\":\"shade\"}]}}},\"band1Vert\":{\"cell\":{\"borderInsideVert\":{\"style\":\"none\"},\"fillColor\":{\"value\":\"accent1\",\"type\":\"scheme\",\"transformations\":[{\"value\":24706,\"type\":\"tint\"}]},\"borderInsideHor\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"}}},\"firstCol\":{\"character\":{\"bold\":true}}},\"parent\":\"TableNormal\",\"uiPriority\":60,\"type\":\"table\",\"name\":\"insertStyleSheet\"}]"));
			mOperationQueue.put(operations.get(0));


		} catch (JSONException ex) {
			Logger.getLogger(JsonOperationProducer.class
					.getName()).log(Level.SEVERE, null, ex);
		}
	}

	void addExceededTable(final List<Integer> start, int columns, int rows, final List<Integer> tableGrid) {
		final JSONObject insertComponentObject = new JSONObject();

		if (!mIsTableExceededStyleAdded) {
			// addChild once the table style
			addTableExceededStyle();
			mIsTableExceededStyleAdded = true;
		}

		try {
			insertComponentObject.put("name", "insertTable");
			insertComponentObject.put("start", start);
			Map<String, Integer> sizeExceeded = new HashMap<String, Integer>();
			sizeExceeded.put("columns", columns);
			sizeExceeded.put("rows", rows);
			insertComponentObject.put("sizeExceeded", sizeExceeded);
			Map<String, Object> hardFormatations = new HashMap<String, Object>();
			JSONObject tableAttrs = new JSONObject();
//			JSONObject tableAttrs = null;
//			if (hardFormatations != null && !hardFormatations.isEmpty()) {
//				tableAttrs = (JSONObject) hardFormatations.get("table");
//			} else {
//				if (hardFormatations == null) {
//					hardFormatations = new HashMap<String, Object>();
//				}
//			}
//			if (tableAttrs == null) {
//				tableAttrs = new JSONObject();
//			}
//			if (tableGrid != null) {
//				tableAttrs.put("tableGrid", tableGrid);
//			}
			tableAttrs.put("tableGrid", tableGrid);
			tableAttrs.put("style", "LightShading-Accent1");
			tableAttrs.put("width", "auto");
			List exclude = new ArrayList(3);
			exclude.add("lastRow");
			exclude.add("lastCol");
			exclude.add("bandsVert");
			tableAttrs.put("exclude", exclude);
			hardFormatations.put("table", tableAttrs);
			insertComponentObject.put("attrs", hardFormatations);
			mOperationQueue.put(insertComponentObject);


			LOG.log(Level.FINEST, "insertTable" + " - component:{0}", insertComponentObject);



		} catch (JSONException e) {
			Logger.getLogger(JsonOperationProducer.class
					.getName()).log(Level.SEVERE, null, e);
		}
	}

	void insertField(final List<Integer> start, String fieldType, String fieldContent) {
		final JSONObject insertComponentObject = new JSONObject();

		try {
			insertComponentObject.put("name", "insertField");
			insertComponentObject.put("start", start);
			insertComponentObject.put("type", fieldType);
			if (fieldContent != null) {
				insertComponentObject.put("representation", fieldContent);
			} else {
				insertComponentObject.put("representation", "");
			}
			mOperationQueue.put(insertComponentObject);
			LOG.log(Level.FINEST, "insertTable" + " - component:{0}", insertComponentObject);



		} catch (JSONException e) {
			Logger.getLogger(JsonOperationProducer.class
					.getName()).log(Level.SEVERE, null, e);
		}
	}

	/**
	 * *
	 * insertStylesheet name: 'insertStylesheet' type: "table" styleId: The
	 * identifier of this stylesheet (unique for the corresponding type).
	 * stylename: The readable name of this style sheet. attrs: JSONObject,
	 * contains formatting attributes as nested JSON objects, keyed by attribute
	 * family, as well as other properties specific to a family. Must support
	 * the attributes of the main attribute family, may support attributes of
	 * other families. See chapter "Style Sheet Properties" below for a list of
	 * supported values. parent: (string, optional) The identifier of the parent
	 * style sheet that derives formatting attributes to this style sheet
	 * (default: no parent). hidden: (boolean, optional) Whether the style sheet
	 * is hidden in the user interface (default: false). uipriority: (integer,
	 * optional) The priority of the style sheet used to order style sheet in
	 * the user interface. The lower the value the higher the priority (default:
	 * 0). default: (boolean, optional) Whether this style sheet is the default
	 * style sheet of the family. Only one style sheet per family can be the
	 * default style sheet (default: false). pooldefault: (boolean, optional)
	 * Whether this style sheet is the pool default style sheet. (default:
	 * false). OOXML may have on the table default style properties for table
	 * Object { type: table properties }	'' row	Object { type: row properties }
	 * '' cell	Object { type: cell properties }	'' paragraph	Object { type:
	 * paragraph properties }	'' character	Object { type: character properties }
	 * We need to split this table style into an additional row and cell style
	 * and addChild it as parent for those.
	 */
	// 
	void insertStyleSheet(String styleId, String familyID, String displayName, Map<String, Object> componentProps, String parentStyle, String nextStyleId, Integer outlineLevel, boolean isDefaultStyle, boolean isHidden) {
		//conditionalType: wholetable 
		final JSONObject insertComponentObject = new JSONObject();
		try {
			insertComponentObject.put("name", "insertStyleSheet");
			if (styleId != null && !styleId.isEmpty()) {
				insertComponentObject.put("styleId", styleId);
			}
			insertComponentObject.put("type", familyID);
			if (displayName != null && !displayName.isEmpty()) {
				insertComponentObject.put("styleName", displayName);
			}
			if (familyID.equals("table")) {
				final JSONObject tableStyleAttrs = new JSONObject();
				tableStyleAttrs.put("wholeTable", componentProps);
				insertComponentObject.put("attrs", tableStyleAttrs);

			} else {
				insertComponentObject.put("attrs", componentProps);
			}
			if (parentStyle != null && !parentStyle.isEmpty()) {
				insertComponentObject.put("parent", parentStyle);
			}
			if (isDefaultStyle) {
				insertComponentObject.put("default", isDefaultStyle);
			}
			if (isHidden) {
				insertComponentObject.put("hidden", isHidden);
			}
			if (outlineLevel != null || nextStyleId != null) {
				JSONObject paraProps;
				if (componentProps.containsKey("paragraph")) {
					paraProps = (JSONObject) componentProps.get("paragraph");
				} else {
					paraProps = new JSONObject();
					componentProps.put("paragraph", paraProps);
				}
				if (outlineLevel != null) {
					paraProps.put("outlineLevel", outlineLevel - 1);
				}
				if (nextStyleId != null && !nextStyleId.isEmpty()) {
					paraProps.put("nextStyleId", nextStyleId);
				}
				componentProps.put("paragraph", paraProps);
				insertComponentObject.put("attrs", componentProps);
			}
			mOperationQueue.put(insertComponentObject);
			LOG.log(Level.FINEST, "insertStylesheet" + " - component:{0}", insertComponentObject);


		} catch (JSONException e) {
			Logger.getLogger(JsonOperationProducer.class
					.getName()).log(Level.SEVERE, null, e);
		}
	}

	void insertFontDescription(String fontName, String[] altNames, String family, String familyGeneric, String pitch, String panose1) {
		List<Integer> panose1_Integers = null;
		if (panose1 != null && !panose1.isEmpty()) {
			String[] result = null;
			if (panose1.contains("[")) {
				panose1 = panose1.substring(1, panose1.length() - 1);
			}

			if (panose1.contains(",")) {
				result = panose1.split(",");
			} else {
				result = panose1.split("\\s");
			}
			panose1_Integers = new LinkedList<Integer>();
			for (String token : result) {
				try {
					panose1_Integers.add(Integer.parseInt(token));


				} catch (NumberFormatException e) {
					Logger.getLogger(JsonOperationProducer.class
							.getName()).log(Level.SEVERE, null, e);
				}
			}
		}
		insertFontDescription(fontName, altNames, family, familyGeneric, pitch, panose1_Integers);
	}

	/**
	 * Inserts an extended description for a specific font which an be used for
	 * font substitution algorithms. name	String	'insertFontDescription'
	 *
	 * @param fontName	The font name.
	 * @param altNames	String[]	NOTE: Will be ignored in ODF. A list of
	 * alternate names for the font.
	 * @param family	String	The font family.
	 * @param pitch	String	The font pitch. One of 'fixed', or 'variable'.
	 * @param panose1	Integer[10]	The font typeface classification number. See
	 * http://en.wikipedia.org/wiki/PANOSE.
	 *
	 */
	private void insertFontDescription(String fontName, String[] altNames, String family, String familyGeneric, String pitch, List<Integer> panose1) {
		final JSONObject insertComponentObject = new JSONObject();
		final JSONObject attrs = new JSONObject();
		try {
			insertComponentObject.put("name", "insertFontDescription");
			if (fontName != null && !fontName.isEmpty()) {
				insertComponentObject.put("fontName", fontName);
			} else {
				LOG.fine("The font name is mandatory!");
			}
			insertComponentObject.put("attrs", attrs);
			if (family != null && !family.isEmpty()) {
				attrs.put("family", family);
			}
			if (familyGeneric != null && !familyGeneric.isEmpty()) {
				attrs.put("familyGeneric", familyGeneric);
			}
			if (pitch != null && !pitch.isEmpty()) {
				attrs.put("pitch", pitch);
			}
			if (panose1 != null && !panose1.isEmpty()) {
				if (panose1.size() != 10) {
					LOG.fine("Panose1 is not 10 digits long: " + panose1.toString());
				}
				attrs.put("panose1", panose1);
			}

			mOperationQueue.put(insertComponentObject);
			LOG.log(Level.FINEST, "insertFontDescription" + " - component:{0}", insertComponentObject);


		} catch (JSONException e) {
			Logger.getLogger(JsonOperationProducer.class
					.getName()).log(Level.SEVERE, null, e);
		}
	}

	/**
	 * @param the componentProps is a JSONObject that has the family ID as key
	 * (e.g 'page') and the properties as JSON object as value
	 */
	void setDocumentAttributes(JSONObject componentProps) {
		//conditionalType: wholetable 
		final JSONObject insertComponentObject = new JSONObject();
		try {
			insertComponentObject.put("attrs", componentProps);
			insertComponentObject.put("name", "setDocumentAttributes");
			mOperationQueue.put(insertComponentObject);
			LOG.log(Level.FINEST, "setDocumentAttributes" + " - component:{0}", insertComponentObject);


		} catch (JSONException e) {
			Logger.getLogger(JsonOperationProducer.class
					.getName()).log(Level.SEVERE, null, e);
		}
	}

	private String getComponentPath(List<Integer> pathIntegers) {
		StringBuilder path = new StringBuilder();
		for (Integer pathInteger : pathIntegers) {
			path.append(PATH_SEPARATOR);
			path.append(pathInteger);
		}
		return path.toString();
	}

	/**
	 * ODF List Style inheritance will be resolved: <ol> <li> The
	 * paragraph/heading's list style reference from its closest text:list or
	 * text:list-item ancestor is taken. </li> text:list uses the attribute
	 *
	 * @text:style-name and text:list-item the attribute
	 * @text:style-override respectively. <li> If no list style was given by any
	 * text:list nor text:list-item ancestor the list-style reference of the
	 * paragraph style is being chosen, i.e. the
	 * @style:list-style-name attribute within the paragraph style style:style
	 * element.</li> </ol>
	 */
	public static String getListStyle(ArrayDeque<OdfFileSaxHandler.ParagraphListProperties> listStyleStack, TextParagraphElementBase p) {
		String listStyleId = null;
		Iterator<OdfFileSaxHandler.ParagraphListProperties> listStyles = listStyleStack.descendingIterator();
		// Choose the first style being set
		while (listStyles.hasNext()) {
			OdfFileSaxHandler.ParagraphListProperties paraListStyle = listStyles.next();
			listStyleId = paraListStyle.getListStyleName();
			if (listStyleId != null && !listStyleId.isEmpty()) {
				break;
			}
		}
		// if no style was previous set, use the style on the paragraph		
		if (listStyleId == null || listStyleId.isEmpty()) {
			OdfStyleBase style = null;
			if (p.hasAutomaticStyle()) {
				style = p.getAutomaticStyle();
			} else {
				style = p.getDocumentStyle();
			}
			if (style != null) {
				listStyleId = ((OdfStyle) style).getStyleListStyleNameAttribute();
			}
		}
		return listStyleId;
	}

	static Map<String, Object> getAutomaticStyleHierarchyProps(OdfStylableElement styleElement) {
		// Hard formatted properties (automatic styles)
		Map<String, Object> allHardFormatting = null;
		Map<String, Map<String, String>> allOdfProps = null;
		// AUTOMATIC STYLE HANDLING
		if (styleElement.hasAutomaticStyle()) {
			OdfStyleBase style = styleElement.getAutomaticStyle();

			// all ODF properties
			allOdfProps = new HashMap<String, Map<String, String>>();
			List<OdfStyleBase> parents = new LinkedList<OdfStyleBase>();
			parents.add(style);
			OdfStyleBase parent = style.getParentStyle();
			// if automatic style inheritance is possible 
			while (parent != null) {
				Node n = parent.getParentNode();
				// if it is no longer an automatic style (template or default style)
				if (n instanceof OdfOfficeStyles) {
					break;
				}
				parents.add(parent);
				parent = parent.getParentStyle();
			}
			// due to inheritance the top ancestor style have to be propagated first
			for (int i = parents.size() - 1; i >= 0; i--) {
				getStyleProperties(parents.get(i), styleElement, allOdfProps);
			}
			allHardFormatting = mapStyleProperties(styleElement, allOdfProps);
		}
		return allHardFormatting;
	}
//	
//	static Map<String, Object> addTemplateStyleName(OdfStylableElement styleElement, Map<String, Object> allHardFormatting) {
//		// Hard formatted properties (automatic styles)
//		
//		// TEMPLATE STYLE HANDLING (and DEFAULT STYLES HANDLING)
//		if (styleElement.hasDocumentStyle()) {
//			String groupingId = Component.getMainOxStyleGroupingId(styleElement);
//			OdfStyle style = styleElement.getDocumentStyle();			
//			JSONObject props = null;
//			if (allHardFormatting != null && !allHardFormatting.isEmpty()) {
//				props = (JSONObject) allHardFormatting.get(groupingId);
//			} else {
//				if (allHardFormatting == null) {
//					allHardFormatting = new HashMap<String, Object>();
//				}
//			}
//			try {
//				if (props == null) {
//					props = new JSONObject();
//				}
//				props.put("styleId", style.getStyleNameAttribute());
//			} catch (JSONException ex) {
//				Logger.getLogger(JsonOperationProducer.class.getName()).log(Level.SEVERE, null, ex);
//			}
//			allHardFormatting.put(groupingId, props);			
//		}
//
//		return allHardFormatting;
//	}	

	/**
	 * Maps the styles of the given stylable ODF element to operations. In case
	 * the ODF element uses automatic styles, all styles will be returned as
	 * property map. In case the ODF element uses template styles, an operation
	 * for each style is being triggered, which not have been triggered by now.
	 *
	 * @return the mapped automatic style grouped by property set
	 */
	Map<String, Object> getHardStyles(OdfStylableElement styleElement) {
		// Hard formatted properties (automatic styles)
		Map<String, Object> allHardFormatting = null;
		// AUTOMATIC STYLE HANDLING
		if (styleElement.hasAutomaticStyle()) {
			OdfStyleBase style = styleElement.getAutomaticStyle();
			// Handle master page attributes
			String pageStyleName = ((OdfStyle) style).getStyleMasterPageNameAttribute();
			if (pageStyleName != null && !pageStyleName.isEmpty()) {
				insertPageProperties(styleElement, pageStyleName);
			}
			allHardFormatting = getAutomaticStyleHierarchyProps(styleElement);
		}

		// TEMPLATE STYLE HANDLING (and DEFAULT STYLES HANDLING)
		if (styleElement.hasDocumentStyle()) {
//			allHardFormatting = addTemplateStyleName(styleElement, allHardFormatting);			
			triggerStyleHierarchyOps(styleElement.getStyleFamily(), styleElement.getDocumentStyle());
		}
		return allHardFormatting;
	}

	/**
	 * Creates the operation to insert a list style. All template list styles
	 * should be already set during initialization.
	 */
	void addListStyle(OdfSchemaDocument doc, Map<String, TextListStyleElement> autoListStyles, String styleId) {
		if (styleId != null & !styleId.isEmpty()) {
			if (!knownListStyles.containsKey(styleId)) {
				try {
					// Three locations to check for the list style
					//  - 1 -  Automatic Styles of content.xml 
					TextListStyleElement listStyle = autoListStyles.get(styleId);
					if (listStyle != null) {
						addListStyle(listStyle);
					} else {
						//  - 2 -  Automatic Styles of styles.xml 
						OdfOfficeAutomaticStyles autoStyles = doc.getStylesDom().getAutomaticStyles();
						listStyle = autoStyles.getListStyle(styleId);
						if (listStyle != null) {
							addListStyle(listStyle);
						} else {
							//  - 3 -  Template Styles of styles.xml -- third as already checked when initialized
							OdfOfficeStyles templateStyles = doc.getStylesDom().getOfficeStyles();
							listStyle = templateStyles.getListStyle(styleId);
							if (listStyle != null) {
								addListStyle(listStyle);


							}
						}
					}
				} catch (Exception ex) {
					Logger.getLogger(JsonOperationProducer.class
							.getName()).log(Level.SEVERE, null, ex);
				}
			}
		}
	}

	/**
	 * After checking if the given list style was not declared before as
	 * operation call. A new list style will be mapped to an operation.
	 *
	 *
	 * The <text:list-style> element has the following attributes:
	 * style:display-name, style:name and text:consecutive-numbering.
	 */
	void addListStyle(TextListStyleElement listStyle) {
		if (listStyle != null) {
			String styleId = listStyle.getStyleNameAttribute();
			if (!knownListStyles.containsKey(styleId)) {
				// addChild the given style to the known styles, so it will not provided again
				knownListStyles.put(styleId, Boolean.TRUE);
				insertListStyle(listStyle.getStyleNameAttribute(), listStyle.getTextConsecutiveNumberingAttribute(), getListLevelDefinitions(listStyle));
			}
		}
	}

	/**
	 * Receives the ten list definition
	 */
	private JSONObject getListLevelDefinitions(TextListStyleElement listStyle) {
		JSONObject listDefinition = new JSONObject(9);
		NodeList listStyleChildren = listStyle.getChildNodes();
		int size = listStyleChildren.getLength();
		for (int i = 0; i < size; i++) {
			Node child = listStyleChildren.item(i);
			if (!(child instanceof Element)) {
				// avoid line breaks, when XML is indented
				continue;
			} else {
				TextListLevelStyleElementBase listLevelStyle = (TextListLevelStyleElementBase) child;
				// Transform mandatory attribute to integer

				String textLevel = listLevelStyle.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "level");
				int listLevel = Integer.parseInt(textLevel) - 1;
				try {
					listDefinition.put("listLevel" + listLevel, createListLevelDefinition(listLevelStyle, listLevel));


				} catch (JSONException ex) {
					Logger.getLogger(JsonOperationProducer.class
							.getName()).log(Level.SEVERE, null, ex);
				}
			}
		}
		return listDefinition;
	}

	/*
	 justification		String	 One of 'left', 'right', or 'center'.
	 numberFormat		String	 One of 'none', 'bullet', 'decimal', 'lowerRoman', 'upperRoman', 'lowerLetter', or 'upperLetter'.
	 levelStart			Integer	 Start index of the level.
	 indentLeft			Integer	 Left indent of the numbered paragraph.
	 indentFirstLine		Integer	 First line indent, negative values represent hanging indents.
	 fontName			String	 Font name, typically used for bullet symbols.
	 levelText			String	 Formatting text of the label, can contain bullet symbol or level format like "%1."
	 levelRestartValue	Integer	 The number level that resets the current level back to its starting value.
	 OLD		paraStyle			String	 Identifier of the paragraph style that the current numbering level shall be applied to.
	 NEW		textStyle
	 NEW		HARD TEXT PROPERTIES
	 levelPicBulletUri	String	 (optional) URI of the bullet picture.
	 tabStopPosition		Integer	 Tabulator position, in 1/100 of millimeters.
	 OLD		color				Color	  
	 * 
	 * The <text:list-level-style-bullet> element has the following attributes: style:num-prefix 19.502, style:num-suffix 19.503, text:bullet-char 19.760, text:bullet-relative-size 19.761, text:level 19.828 and text:style-name 19.874.24
	 * The <text:list-level-style-number> element has the following attributes: style:num-format 19.500, style:num-letter-sync 19.501, style:num-prefix 19.502, style:num-suffix 19.503, text:display-levels 19.797, text:level 19.828, text:start-value 19.868.4 and text:style-name 19.874.23.
	 * The <text:list-level-style-image> element has the following attributes: text:level 19.828, xlink:actuate 19.909, xlink:href 19.910.35, xlink:show 19.911 and xlink:type 19.913.
	 */
	private JSONObject createListLevelDefinition(TextListLevelStyleElementBase listLevelStyle, int listLevel) throws JSONException {
		JSONObject listLevelDefinition = new JSONObject();

		// NUMBERED LISTS
		if (listLevelStyle instanceof TextListLevelStyleNumberElement) {
			TextListLevelStyleNumberElement listLevelNumberStyle = (TextListLevelStyleNumberElement) listLevelStyle;
			listLevelDefinition.put("levelText", getLabel(listLevelNumberStyle, listLevel));
			if (listLevelStyle.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "start-value")) {
				String listStartValue = getListStartValue(listLevelNumberStyle);
				listLevelDefinition.put("listStartValue", Integer.parseInt(listStartValue));
			}
			// There is always the number format set
			listLevelDefinition.put("numberFormat", getNumberFormat(listLevelNumberStyle));

			// BULLET LISTS		
		} else if (listLevelStyle instanceof TextListLevelStyleBulletElement) {
			TextListLevelStyleBulletElement listLevelBulletStyle = (TextListLevelStyleBulletElement) listLevelStyle;
			listLevelDefinition.put("levelText", getLabel(listLevelBulletStyle, listLevel));
			if (listLevelStyle.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "bullet-relative-size")) {
				listLevelDefinition.put("bulletRelativeSize", listLevelStyle.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "bullet-relative-size"));
			}
			listLevelDefinition.put("numberFormat", "bullet");


			// IMAGE LISTS
		} else if (listLevelStyle instanceof TextListLevelStyleImageElement) {
			listLevelDefinition.put("levelPicBulletUri", listLevelStyle.getAttributeNS(OdfDocumentNamespace.XLINK.getUri(), "href"));
			listLevelDefinition.put("numberFormat", "bullet");
		}

		// ALL THREE TYPES: number, bullet and image list
		if (listLevelStyle.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "style-name")) {
			listLevelDefinition.put("styleId", listLevelStyle.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "style-name"));
		}


		/*
		 The <style:list-level-properties> element has the following attributes: 
		 fo:height 20.187,
		 fo:text-align 20.216.2,
		 fo:width 20.222,
		 style:font-name 20.269,
		 style:vertical-pos 20.387,
		 style:vertical-rel 20.388,
		 svg:y 20.402.2,
		 text:list-level-position-and-space-mode 20.421,
		 text:min-label-distance 20.422,
		 text:min-label-width 20.423 and 
		 text:space-before 20.425.
		 */

		NodeList listLevelProps = listLevelStyle.getElementsByTagNameNS(OdfDocumentNamespace.STYLE.getUri(), "list-level-properties");
		if (listLevelProps != null) {
			StyleListLevelPropertiesElement styleListLevelProperties = (StyleListLevelPropertiesElement) listLevelProps.item(0);
			if (styleListLevelProperties != null) {
				//  fo:height
				if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.FO.getUri(), "height")) {
					String heightValue = styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.FO.getUri(), "height");
					if (heightValue != null) {
						int height = normalizeLength(heightValue);
						listLevelDefinition.put("height", height);
					}
				}

				// fo:text-align
				if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.FO.getUri(), "text-align")) {
					listLevelDefinition.put("textAlign", mapFoTextAlign(styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.FO.getUri(), "text-align")));
				}

				//  fo:width
				if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.FO.getUri(), "width")) {
					String widthValue = styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.FO.getUri(), "width");
					if (widthValue != null) {
						int width = normalizeLength(widthValue);
						listLevelDefinition.put("width", width);
					}
				}

				// style:font-name
				if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "font-name")) {
					listLevelDefinition.put("fontName", styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "font-name"));
				}

				// style:vertical-pos
				if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "vertical-pos")) {
					listLevelDefinition.put("verticalPos", styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "vertical-pos"));
				}

				// style:vertical-rel
				if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "vertical-rel")) {
					listLevelDefinition.put("verticalRel", styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "vertical-rel"));
				}

				// svg:y
				if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.SVG.getUri(), "y")) {
					listLevelDefinition.put("y", styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.SVG.getUri(), "y"));
				}

				// text:list-level-position-and-space-mode
				if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "list-level-position-and-space-mode")) {
					listLevelDefinition.put("listLevelPositionAndSpaceMode", styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "list-level-position-and-space-mode"));
				}

				// text:min-label-distance
				if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "min-label-distance")) {
					String minLabelDistanceValue = styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "min-label-distance");
					if (minLabelDistanceValue != null && !minLabelDistanceValue.isEmpty()) {
						int minLabelDistance = normalizeLength(minLabelDistanceValue);
						listLevelDefinition.put("minLabelDistance", minLabelDistance);
					}
				}

				// text:min-label-width
				String minLabelWidthValue = null;
				if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "min-label-width")) {
					minLabelWidthValue = styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "min-label-width");
					if (minLabelWidthValue != null && !minLabelWidthValue.isEmpty()) {
						int width = normalizeLength(minLabelWidthValue);
						listLevelDefinition.put("minLabelWidth", width);
					}
				}

				// text:space-before
				String spaceBeforeValue = null;
				if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "space-before")) {
					spaceBeforeValue = styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "space-before");
					if (spaceBeforeValue != null && !spaceBeforeValue.isEmpty()) {
						int spaceBefore = normalizeLength(spaceBeforeValue);
						listLevelDefinition.put("spaceBefore", spaceBefore);
					}
				}

				// Mapping list XML ODF 1.1 to ODF 1.2: Adding @text:min-label-width & @text:space-before to margin-left
				listLevelDefinition = mapIndent(minLabelWidthValue, spaceBeforeValue, listLevelDefinition);

				if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "list-level-position-and-space-mode")) {
					if ("label-alignment".equals(styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "list-level-position-and-space-mode"))) {
						NodeList nl = styleListLevelProperties.getElementsByTagNameNS(OdfDocumentNamespace.STYLE.getUri(), "list-level-label-alignment");
						if (nl != null && nl.getLength() == 1) {
							StyleListLevelLabelAlignmentElement labelAlignmentElement = (StyleListLevelLabelAlignmentElement) nl.item(0);
							String marginLeft = labelAlignmentElement.getAttributeNS(OdfDocumentNamespace.FO.getUri(), "margin-left");
							int margin = 0;
							if (marginLeft != null && !marginLeft.isEmpty()) {
								margin = normalizeLength(marginLeft);
								listLevelDefinition.put("indentLeft", margin);
							} else {
								listLevelDefinition = mapIndent(minLabelWidthValue, spaceBeforeValue, listLevelDefinition);
							}
							String textIndent = labelAlignmentElement.getAttributeNS(OdfDocumentNamespace.FO.getUri(), "text-indent");
							if (textIndent != null && !textIndent.isEmpty()) {
								int indent = normalizeLength(textIndent);
								listLevelDefinition.put("indentFirstLine", indent);
							}

							//			<optional>
							//				<attribute name="text:list-tab-stop-position">
							//					<ref name="length"/>
							//				</attribute>
							//			</optional>					
							if (labelAlignmentElement.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "list-tab-stop-position")) {
								String tabPosition = labelAlignmentElement.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "list-tab-stop-position");
								if (tabPosition != null && !tabPosition.isEmpty()) {
//									if(marginLeft != null && !marginLeft.isEmpty()){
//										listLevelDefinition.put("tabStopPosition", normalizeLength(tabPosition) + margin);
//									}else{
									listLevelDefinition.put("tabStopPosition", normalizeLength(tabPosition));
//									}									
								}
							}

							//			<attribute name="text:label-followed-by">
							//				<choice>
							//					<value>listtab</value>
							//					<value>space</value>
							//					<value>nothing</value>
							//				</choice>
							//			</attribute>
							if (labelAlignmentElement.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "label-followed-by")) {
								listLevelDefinition.put("labelFollowedBy", labelAlignmentElement.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "label-followed-by"));
							}
						}
					}
				}

			}
		}
		return listLevelDefinition;
	}

	private static JSONObject mapIndent(String minLabelWidthValue, String spaceBeforeValue, JSONObject listLevelDefinition) throws JSONException {
		int minLabelWidth = 0;
		boolean isValidMinLabelWidth = Length.isValid(minLabelWidthValue);
		if (isValidMinLabelWidth) {
			minLabelWidth = normalizeLength(minLabelWidthValue);
		}
		int spaceBefore = 0;
		boolean isValidSpaceBefore = Length.isValid(spaceBeforeValue);
		if (isValidSpaceBefore) {
			spaceBefore = normalizeLength(spaceBeforeValue);
		}
		if (isValidMinLabelWidth || isValidSpaceBefore) {			
			listLevelDefinition.put("indentLeft", minLabelWidth + spaceBefore);
		}
		return listLevelDefinition;
	}

	/**
	 * Handling the attributes of
	 *
	 * @style:num-format
	 *
	 * The style:num-format attribute specifies a numbering sequence. The
	 * defined ODF values for the style:num-format attribute are: <ul> <li>1:
	 * Hindu-Arabic number sequence starts with 1. </li> <li>a: number sequence
	 * of lowercase Modern Latin basic alphabet characters starts with "a".
	 * </li> <li>A: number sequence of uppercase Modern Latin basic alphabet
	 * characters starts with "A". </li> <li>i: number sequence of lowercase
	 * Roman numerals starts with "i". </li> <li>I: number sequence of uppercase
	 * Roman numerals start with "I". </li> <li>a value of type string 18.2.
	 * (COMPLEX NUMBERING ie. ASIAN oder ERSTENS..)</li> <li>an empty string: no
	 * number sequence displayed.</li> <li>If no value is given, no number
	 * sequence is displayed. </li> </ul> Our API: One of 'none', 'bullet',
	 * 'decimal', 'lowerRoman', 'upperRoman', 'lowerLetter', or 'upperLetter'.
	 */
	private String getNumberFormat(TextListLevelStyleElementBase listLevelStyle) {
		String numberFormat = listLevelStyle.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "num-format");
		String numFormat;
		if (numberFormat == null || numberFormat.isEmpty()) {
			numFormat = "none";
		} else if (numberFormat.equals("1")) {
			numFormat = "decimal";
		} else if (numberFormat.equals("i")) {
			numFormat = "lowerRoman";
		} else if (numberFormat.equals("I")) {
			numFormat = "upperRoman";
		} else if (numberFormat.equals("a")) {
			numFormat = "lowerLetter";
		} else if (numberFormat.equals("A")) {
			numFormat = "upperLetter";
		} else {
			// a value of type string 18.2. (COMPLEX NUMBERING ie. ASIAN oder ERSTENS..)
			numFormat = numberFormat;
		}
		return numFormat;
	}

	/**
	 * Handling the attributes of
	 *
	 * @text:start-value
	 */
	private String getListStartValue(TextListLevelStyleElementBase listLevelStyle) {
		return listLevelStyle.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "start-value");
	}

	/**
	 * Handling the attributes of
	 *
	 * @style:num-prefix
	 * @text:display-levels
	 * @style:num-suffix
	 */
	private String getLabel(TextListLevelStyleElementBase listLevelStyle, int listLevel) {
		StringBuilder levelText = new StringBuilder();

		// creating label prefix
		String labelPrefix = listLevelStyle.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "num-prefix");
		if (labelPrefix != null && !labelPrefix.isEmpty()) {
			levelText.append(labelPrefix);
		}


		// creating label number		
		if (listLevelStyle instanceof TextListLevelStyleNumberElement) {
			String displayLevels = listLevelStyle.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "display-levels");
			if (displayLevels != null && !displayLevels.isEmpty()) {
				int showLevels = Integer.parseInt(displayLevels);
				// Creating the label, in ODF always adding the low levelText first, adding each follow up level for display level
				// Custom string with one of the placeholders from '%1' to '%9') for numbered lists.
				for (int i = showLevels; i > 0; i--) {
					levelText.append("%").append(listLevel + 2 - i);
					// Although not commented in the specification a "." is being added to the text level
					if (i != 1) {
						levelText.append('.');
					}
				}
			} else {
				levelText.append("%").append(listLevel + 1);
			}
			// creating label suffix
			String labelSuffix = listLevelStyle.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "num-suffix");
			if (labelSuffix != null && !labelSuffix.isEmpty()) {
				levelText.append(labelSuffix);
			}
		} else {
			String bulletChar = listLevelStyle.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "bullet-char");
			if (bulletChar != null && !bulletChar.isEmpty()) {
				levelText.append(bulletChar);
			}
		}
		return levelText.toString();
	}
	// insertListStyle(knownListStyles.size(), listStyle.getStyleNameAttribute(), listStyle.getStyleDisplayNameAttribute(), listStyle.getTextConsecutiveNumberingAttribute(), getListLevelDefinitions(listStyle));

	private void insertListStyle(String styleName, boolean hasConsecutiveNumbering, JSONObject listDefinition) {
		//conditionalType: wholetable 
		final JSONObject insertComponentObject = new JSONObject();
		try {
			insertComponentObject.put("name", "insertListStyle");
			insertComponentObject.put("listStyleId", styleName);
//			insertComponentObject.put("displayName", displayName);
			if (hasConsecutiveNumbering) {
				insertComponentObject.put("listUnifiedNumbering", hasConsecutiveNumbering);
			}
			insertComponentObject.put("listDefinition", listDefinition);

			mOperationQueue.put(insertComponentObject);
			LOG.log(Level.FINEST, "insertListStyle" + " - component:{0}", insertComponentObject);


		} catch (JSONException e) {
			Logger.getLogger(JsonOperationProducer.class
					.getName()).log(Level.SEVERE, null, e);
		}
	}

	void triggerStyleHierarchyOps(OdfStyleFamily styleFamily, OdfStyleBase style) {
		if (style != null) {

			if (!(style instanceof OdfDefaultStyle)) {
				if (!knownStyles.containsKey(((OdfStyle) style).getStyleNameAttribute())) {
					List<OdfStyleBase> parents = new LinkedList<OdfStyleBase>();
					OdfStyleBase parent = style;

					// Collecting hierachy, to go back through the style hierarchy from the end, to be able to neglect empty styles and adjust parent style attribute
					while (parent != null
							&& (parent instanceof OdfDefaultStyle || !knownStyles.containsKey(((OdfStyle) parent).getStyleNameAttribute()))) {
						if (parent instanceof OdfDefaultStyle) {
							triggerDefaultStyleOp(styleFamily, parent);
							// NEXT: there is no style above a default in the style hierarchy
							break;
						} else if (parent != null) {
							parents.add(parent);

							// NEXT: get the next parent style and if the style parent name is the OX DEFAULT NAME remove it
							Attr parentStyleName = parent.getAttributeNodeNS(OdfDocumentNamespace.STYLE.getUri(), "parent-style-name");
							if (parentStyleName != null && parentStyleName.getValue().equals(OX_DEFAULT_STYLE_PREFIX + Component.getFamilyID(styleFamily) + OX_DEFAULT_STYLE_SUFFIX)) {
								parent.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "parent-style-name");
								triggerDefaultStyleOp(styleFamily, style.getParentStyle());
								break;
							} else {
								parent = parent.getParentStyle();
							}
						}

						// trigger operation only for those style not already existing
						// check if the named style already exists											
					}

					String lastWrittenStyleName = null; // Only write out parents with mapped styles
					boolean skippedEmptyParent = false;
					// Intermediate ODF properties
					Map<String, Map<String, String>> allOdfProps = new HashMap<String, Map<String, String>>();
					// The property groups for this component, e.g. cell, paragraph, text for a cell with properties
					Map<String, OdfStylePropertiesSet> familyPropertyGroups = Component.getAllOxStyleGroupingIdProperties(styleFamily);
					// Mapped properties
					Map<String, Object> mappedFormatting = null;

					// addChild named style to operation
					String styleName;

					// due to inheritance the top ancestor style have to be propagated first
					for (int i = parents.size() - 1; i >= 0; i--) {
						style = parents.get(i);

						// Handle master page properties first
						String pageStyleName = ((OdfStyle) style).getStyleMasterPageNameAttribute();
						if (pageStyleName != null && !pageStyleName.isEmpty()) {
							insertPageProperties(style, pageStyleName);
						}

						// get all ODF properties from this style
						getStyleProperties(style, familyPropertyGroups, allOdfProps);
						// mapping the ODF attribute style props to our component properties					
						mappedFormatting = mapStyleProperties(familyPropertyGroups, allOdfProps);
						styleName = ((OdfStyle) style).getStyleNameAttribute();
						// No OdfStyle, as the parent still might be a default style without name
						OdfStyleBase parentStyle = style.getParentStyle();
						String parentStyleName = null;

						// Default styles do not have a name
						if (parentStyle != null && !(parentStyle instanceof OdfDefaultStyle)) {
							parentStyleName = ((OdfStyle) parentStyle).getStyleNameAttribute();
						}
						String nextStyle = ((OdfStyle) style).getStyleNextStyleNameAttribute();
						Integer outlineLevel = ((OdfStyle) style).getStyleDefaultOutlineLevelAttribute();
						// Do not trigger operations to create empty styles
						if (skippedEmptyParent) {
							parentStyleName = lastWrittenStyleName;
						}
						String familyId = Component.getFamilyID(styleFamily);
						if (parentStyleName != null && !parentStyleName.isEmpty()) {
							insertStyleSheet(styleName, familyId, ((OdfStyle) style).getStyleDisplayNameAttribute(), mappedFormatting, parentStyleName, nextStyle, outlineLevel, false, false);
						} else {
							// the template style without parent is a root style and being used as a default style in the editor
							//insertStyleSheet(styleName, familyId, ((OdfStyle) style).getStyleDisplayNameAttribute(), mappedFormatting, OX_DEFAULT_STYLE_PREFIX + familyId + OX_DEFAULT_STYLE_SUFFIX, nextStyle, outlineLevel, true, false);
							insertStyleSheet(styleName, familyId, ((OdfStyle) style).getStyleDisplayNameAttribute(), mappedFormatting, OX_DEFAULT_STYLE_PREFIX + familyId + OX_DEFAULT_STYLE_SUFFIX, nextStyle, outlineLevel, false, false);
						}

						lastWrittenStyleName = styleName;
						mappedFormatting.clear();
						allOdfProps.clear();
						// addChild named style to known styles, so it will be only executed once				
						knownStyles.put(styleName, Boolean.TRUE);
					}
				}
			} else {
				// DEFAULT STYLE PARENT				
				// Default styles will receive a name and will be referenced by the root style (the default style for the web editor)
				triggerDefaultStyleOp(styleFamily, style);
			}
		}
	}

	/**
	 * Tests first if the default style was already added to the document, than
	 * triggers a insertStylesheet operation
	 */
	void triggerDefaultStyleOp(OdfStyleFamily styleFamily, OdfStyleBase style) {
		// Intermediate ODF properties
		Map<String, Map<String, String>> allOdfProps = new HashMap<String, Map<String, String>>();
		// The property groups for this component, e.g. cell, paragraph, text for a cell with properties
		Map<String, OdfStylePropertiesSet> familyPropertyGroups = Component.getAllOxStyleGroupingIdProperties(styleFamily);
		// Mapped properties
		Map<String, Object> mappedFormatting = null;

		// addChild named style to operation
		String styleName;

		if (!knownStyles.containsKey(OX_DEFAULT_STYLE_PREFIX + Component.getFamilyID(styleFamily) + OX_DEFAULT_STYLE_SUFFIX)) {
			// get all ODF properties from this style
			getStyleProperties(style, familyPropertyGroups, allOdfProps);
			// mapping the ODF attribute style props to our component properties					
			mappedFormatting = mapStyleProperties(familyPropertyGroups, allOdfProps);
			// Tabulator default size is an attribute in the default style, will be received from static mapping functions
			if (mappedFormatting.containsKey("paragraph")) {
				JSONObject paraProps = (JSONObject) mappedFormatting.get("paragraph");
				if (paraProps.has("document")) {
					try {
						JSONObject attrs = new JSONObject();
						attrs.put("document", paraProps.get("document"));
						setDocumentAttributes(attrs);
						paraProps.remove("document");
						if (paraProps.length() == 0) {
							mappedFormatting.remove("paragraph");


						}
					} catch (JSONException ex) {
						Logger.getLogger(JsonOperationProducer.class
								.getName()).log(Level.SEVERE, null, ex);
					}
				}
			}
			String familyId = Component.getFamilyID(styleFamily);
			// Do not trigger operations to create empty styles					
			if (!mappedFormatting.isEmpty()) {
				String displayName = "Default " + Component.getFamilyDisplayName(styleFamily) + " Style";
				insertStyleSheet(OX_DEFAULT_STYLE_PREFIX + familyId + OX_DEFAULT_STYLE_SUFFIX, familyId, displayName, mappedFormatting, null, null, null, true, true);
			}
			// addChild named style to known styles, so it will be only executed once
			styleName = OX_DEFAULT_STYLE_PREFIX + Component.getFamilyID(styleFamily) + OX_DEFAULT_STYLE_SUFFIX;
			knownStyles.put(styleName, Boolean.TRUE);
		}
	}

	private void insertPageProperties(OdfElement element, String pageStyleName) {
		OdfFileDom domFile = (OdfFileDom) element.getOwnerDocument();
		OdfStylesDom stylesDom = null;
		if (!(domFile instanceof OdfStylesDom)) {
			try {
				OdfSchemaDocument schemaDoc = (OdfSchemaDocument) domFile.getDocument();
				stylesDom = schemaDoc.getStylesDom();


			} catch (Exception ex) {
				Logger.getLogger(JsonOperationProducer.class
						.getName()).log(Level.SEVERE, null, ex);
			}
		} else {
			stylesDom = (OdfStylesDom) domFile;
		}
		insertDocumentProperties(stylesDom, pageStyleName);
	}

	void insertDocumentProperties(OdfStylesDom stylesDom, String pageStyleName) {
		try {
			JSONObject documentPropsObject = new JSONObject();
			NodeList masterStyles = stylesDom.getElementsByTagNameNS(OfficeMasterStylesElement.ELEMENT_NAME.getUri(), OfficeMasterStylesElement.ELEMENT_NAME.getLocalName());
			OfficeMasterStylesElement OfficeMasterStylesElement = (OfficeMasterStylesElement) masterStyles.item(0);
			if (OfficeMasterStylesElement != null) {
				NodeList masterPages = OfficeMasterStylesElement.getElementsByTagNameNS(StyleMasterPageElement.ELEMENT_NAME.getUri(), StyleMasterPageElement.ELEMENT_NAME.getLocalName());
				String pageLayoutName = null;
				for (int i = 0; i < masterPages.getLength(); i++) {
					StyleMasterPageElement masterPage = (StyleMasterPageElement) masterPages.item(i);
					String styleName = masterPage.getStyleNameAttribute();
					if (styleName != null && styleName.equals(pageStyleName)) {
						pageLayoutName = masterPage.getStylePageLayoutNameAttribute();
						break;
					}
				}
				JSONObject pagePropsJson = null;
				if (pageLayoutName != null && !pageLayoutName.isEmpty()) {
					OdfOfficeAutomaticStyles autoStyles = stylesDom.getAutomaticStyles();
					OdfStylePageLayout pageLayout = autoStyles.getPageLayout(pageLayoutName);
					Map<OdfStyleProperty, String> pageProperties = pageLayout.getStyleProperties();
					Map<String, String> pageProps = transformMap(pageProperties);
					pagePropsJson = mapProperties("page", pageProps);
				}
				if (pagePropsJson != null && pagePropsJson.length() != 0) {
					documentPropsObject.put("page", pagePropsJson);
				}
			}
			JSONObject docPropsJson = new JSONObject();
			docPropsJson.putOpt("fileFormat", "odf");
			documentPropsObject.put("document", docPropsJson);
			setDocumentAttributes(documentPropsObject);
		} catch (Exception ex) {
			Logger.getLogger(JsonOperationProducer.class
					.getName()).log(Level.SEVERE, null, ex);
		}
	}

	// ToDo: Wird noch immer vererbt?!??
	// Ersteinmal wird alles \u00fcberschrieben und vererbt
	// \u00dcberschreiben kann aber auch die gemappten werte, dann w\u00fcrde man ggf. mehrmals umsonst mappen..
	private Map<String, String> transformMap(Map<OdfStyleProperty, String> props) {
		Map<String, String> odfProps = new HashMap<String, String>();
		for (OdfStyleProperty styleProp : props.keySet()) {
			odfProps.put(styleProp.getName().getQName(), props.get(styleProp));
		}
		return odfProps;
	}

	static Map<String, Object> getMappedStyleProperties(OdfStyle style) {
		// Mapped properties
		Map<String, Object> mappedFormatting = null;
		if (style != null) {
			// Intermediate ODF properties
			Map<String, Map<String, String>> allOdfProps = new HashMap<String, Map<String, String>>();
			// The property groups for this component, e.g. cell, paragraph, text for a cell with properties
			Map<String, OdfStylePropertiesSet> familyPropertyGroups = Component.getAllOxStyleGroupingIdProperties(style.getFamily());
			// get all ODF properties from this style
			getStyleProperties(style, familyPropertyGroups, allOdfProps);
			// mapping the ODF attribute style props to our component properties					
			mappedFormatting = mapStyleProperties(familyPropertyGroups, allOdfProps);
		}
		return mappedFormatting;
	}

	static private void getStyleProperties(OdfStyleBase style, OdfStylableElement styleElement, Map<String, Map<String, String>> allOdfProps) {
		// returns the set of allowed properties, e.g. cell, paragraph, text for a cell with properties
		Map<String, OdfStylePropertiesSet> familyPropertyGroups = Component.getAllOxStyleGroupingIdProperties(styleElement);
		getStyleProperties(style, familyPropertyGroups, allOdfProps);
	}

	static private void getStyleProperties(OdfStyleBase style, Map<String, OdfStylePropertiesSet> familyPropertyGroups, Map<String, Map<String, String>> allOdfProps) {
		if (style != null) {
			for (String styleFamilyKey : familyPropertyGroups.keySet()) {
				// the ODF properties of one family group
				Map<String, String> odfProps = new HashMap<String, String>();
				OdfStylePropertiesSet key = familyPropertyGroups.get(styleFamilyKey);
				OdfStylePropertiesBase propsElement = style.getPropertiesElement(key);
				if (propsElement != null) {
					NamedNodeMap attrs = propsElement.getAttributes();
					String name = null;
					for (int i = 0; i < attrs.getLength(); i++) {
						name = null;
						Attr prop = (Attr) attrs.item(i);

						// normalize XML prefix of ODF attributes to the prefixes used in the specification
						name = OdfNamespace.getNamespace(prop.getNamespaceURI()).getPrefix();
						if (name == null) {
							name = prop.getPrefix();
						}
						if (name != null) {
							name = name + ":" + prop.getName();
						} else {
							name = prop.getName();
						}
						odfProps.put(name, prop.getValue());
					}
					if (propsElement instanceof StyleParagraphPropertiesElement) {
						StyleParagraphPropertiesElement paraPropsElement = (StyleParagraphPropertiesElement) propsElement;
						NodeList tabStops = paraPropsElement.getElementsByTagNameNS(OdfDocumentNamespace.STYLE.getUri(), "tab-stops");
						if (tabStops.getLength() > 0) {
							StyleTabStopsElement tabStopsElement = (StyleTabStopsElement) tabStops.item(0);
							NodeList tabStopList = tabStopsElement.getElementsByTagNameNS(OdfDocumentNamespace.STYLE.getUri(), "tab-stop");
							int size = tabStopList.getLength();
							int tabNumber = -1;
							for (int i = 0; i < size; i++) {
								Node child = tabStopList.item(i);
								if (!(child instanceof Element)) {
									// avoid line breaks, when XML is indented
									continue;
								} else {
									tabNumber++;
									extractTabulatorLeaderText((StyleTabStopElement) child, odfProps, tabNumber);
									extractTabulatorPosition((StyleTabStopElement) child, odfProps, tabNumber);
									// NOT SUPPORTED BY CLIENT: extractTabulatorType((StyleTabStopElement) child, odfProps, tabNumber);
								}
							}
						}
					}
				}
				if (!odfProps.isEmpty()) {
					allOdfProps.put(styleFamilyKey, odfProps);
				}
			}
		}
	}

	/**
	 * style:char fillChar	String	Type of fill character. 'dot'	Tab will be
	 * filled with dot chars: . . . . . . . . . . . 'hyphen'	Tab will be filled
	 * with hyphen chars: - - - - - - - - - 'underscore'	Tab will be filled with
	 * underscore chars _ _ _ _ _ _ _ none	Tab is just empty space.
	 */
	private static void extractTabulatorLeaderText(StyleTabStopElement tabStopElement, Map<String, String> odfProps, int tabNumber) {
		String tabLeaderText = tabStopElement.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "leader-text");

		if (tabLeaderText != null & !tabLeaderText.isEmpty()) {
			if (tabLeaderText.equals(DOT_CHAR)) {
				odfProps.put("tab_LeaderText" + tabNumber, DOT);
			} else if (tabLeaderText.equals(HYPHEN_CHAR)) {
				odfProps.put("tab_LeaderText" + tabNumber, HYPHEN);
			} else if (tabLeaderText.equals(UNDERSCORE_CHAR)) {
				odfProps.put("tab_LeaderText" + tabNumber, UNDERSCORE);
			} else if (tabLeaderText.equals(SPACE_CHAR)) {
				odfProps.put("tab_LeaderText" + tabNumber, NONE);
			}
		}
	}

	/**
	 * style:position pos	Integer	0	Tab stop position in 1/100th mm.
	 */
	private static void extractTabulatorPosition(StyleTabStopElement tabStopElement, Map<String, String> odfProps, int tabNumber) {
		String tabPos = tabStopElement.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "position");
		int tabIndent = 0;
		/**
		 * >20 years ago someone in the StarOffice Writer team thought it was a
		 * brilliant idea to align tabs from the real text start. Therefore to
		 * ODF from LO/AO we need to addChild the left-margin to the
		 * tab-position, to get the REAL tab-position. If the tabs are relative
		 * to indent (margin) we need to addChild the left margin. But there is
		 * a compatibility mode for the MS Office formats filter, where this can
		 * be disabled:
		 * <config:config-item config:name="TabsRelativeToIndent"
		 * config:type="boolean">false</config:config-item>
		 */
		if (odfProps.containsKey("fo:margin-left") && hasTabsRelativeToIndent(tabStopElement)) {
			// get the variable from the text document
			String propValue = odfProps.get("fo:margin-left");
			if (!propValue.contains(PERCENT)) {
				tabIndent = normalizeLength(propValue);
			}
		}
		if (tabPos.isEmpty()) {
			LOG.severe("There should be only be a length, but it has been: '" + tabPos + "'");
		} else {
			odfProps.put("tab_Pos" + tabNumber, Integer.toString(normalizeLength(tabPos) + tabIndent));
		}
	}
// NOT NEEDED YET	
//	/**
//	 * style:type fillChar	String	Type of fill character. 'dot'	Tab will be
//	 * filled with dot chars: . . . . . . . . . . . 'hyphen'	Tab will be filled
//	 * with hyphen chars: - - - - - - - - - 'underscore'	Tab will be filled with
//	 * underscore chars _ _ _ _ _ _ _ none	Tab is just empty space.
//	 */
//	private static void extractTabulatorType(StyleTabStopElement tabStopElement, Map<String, String> odfProps, int tabNumber) {
//		String tabType = tabStopElement.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "type");
//		
//		if (tabType != null && !tabType.isEmpty()) {
//			if (!tabType.equals("char")) {
//				odfProps.put("tab_Type" + tabNumber, tabType);
//			} else {
//				LOG.fine("What shall we do with the 'char' type?");
//			}
//		}
//	}

	/**
	 * In LO/AO Tabs are counted from the start of text and the fo:left-margin
	 * have to be added. Microsoft Office does not behave this way. For the
	 * filter a compatibility flag was added to the settings.xml
	 * <config:config-item config:name="TabsRelativeToIndent"
	 * config:type="boolean">false</config:config-item>
	 */
	static private boolean hasTabsRelativeToIndent(OdfElement element) {
		boolean isTabsRelativeToIndent = false;
		OdfSchemaDocument schemaDoc = (OdfSchemaDocument) ((OdfFileDom) element.getOwnerDocument()).getDocument();
		if (schemaDoc instanceof OdfTextDocument) {
			isTabsRelativeToIndent = ((OdfTextDocument) schemaDoc).hasTabsRelativeToIndent();
		}
		return isTabsRelativeToIndent;
	}

	static private Map<String, Object> mapStyleProperties(OdfStylableElement styleElement, Map<String, Map<String, String>> allOdfProps) {
		// returns the set of allowed properties, e.g. cell, paragraph, text for a cell with properties
		Map<String, OdfStylePropertiesSet> familyPropertyGroups = Component.getAllOxStyleGroupingIdProperties(styleElement);
		return mapStyleProperties(familyPropertyGroups, allOdfProps);
	}

	static private Map<String, Object> mapStyleProperties(Map<String, OdfStylePropertiesSet> familyPropertyGroups, Map<String, Map<String, String>> allOdfProps) {

		Map<String, Object> allProps = new HashMap<String, Object>();
		for (String styleFamilyKey : familyPropertyGroups.keySet()) {
			//NOTE: Perhaps we should first inherit everything from the parents and map afterwards
			// the ODF properties of one family group						
			JSONObject mappedProps = mapProperties(styleFamilyKey, allOdfProps.get(styleFamilyKey));
			if (mappedProps != null && mappedProps.length() != 0) {
				allProps.put(styleFamilyKey, mappedProps);
			}
		}
		return allProps;
	}

	/**
	 * Currently the normalized length is 100th millimeter (or 10 micrometer)
	 */
	static int normalizeLength(String value) {
		Length length = new Length(value);
		return (int) Math.round(length.getMicrometer() / 10.0);
	}

	private static JSONObject mapProperties(String styleFamilyGroup, Map<String, String> odfProps) {
		JSONObject newProps = null;
		if (odfProps != null) {

			// ToDo: Recycle the maps... (??)
			newProps = new JSONObject();
			String propValue;

			// *** TEXT STYLES ***
			if (styleFamilyGroup.equals("character")) {
				boolean borderToBeDone = true;
				boolean paddingToBeDone = true;
				boolean marginToBeDone = true;
				for (String propName : odfProps.keySet()) {
					try {
						if (propName.contains("margin")) {
							if (marginToBeDone) {
								mapMargin(newProps, odfProps);
								marginToBeDone = false;
							}
						} else if (propName.contains("padding")) {
							if (paddingToBeDone) {
								mapPadding(newProps, odfProps);
								paddingToBeDone = false;
							}
						} else if (propName.contains("border")) {
							if (borderToBeDone) {
								mapBorder(newProps, odfProps);
								borderToBeDone = false;
							}
						} else if (propName.contains("letter-spacing")) {
							propValue = odfProps.get("fo:letter-spacing");
							if (propValue.equals("normal")) {
								newProps.put("letterSpacing", propValue);
							} else {
								Integer spacing = normalizeLength(propValue);
								newProps.put("letterSpacing", spacing);
							}
						} else if (propName.equals("fo:font-size")) {
							propValue = odfProps.get("fo:font-size");
							if (propValue.contains("%")) {
								LOG.fine("fo:font-size does have a percentage value, which we do not support!");
							} else {
								Length length = new Length(propValue);
								int fontSize = (int) Math.round(length.getPoint());
								newProps.put("fontSize", fontSize);
							}
						} else if (propName.equals("style:font-size-asian")) {
							propValue = odfProps.get("style:font-size-asian");
							if (propValue.contains("%")) {
								LOG.fine("style:font-size-asia does have a percentage value!");
							} else {
								Length length = new Length(propValue);
								int fontSize = (int) Math.round(length.getPoint());
								newProps.put("fontSizeAsian", fontSize);
							}
						} else if (propName.equals("style:font-size-complex")) {
							propValue = odfProps.get("style:font-size-complex");
							if (propValue.contains("%")) {
								LOG.fine("style:font-size-complex does have a percentage value!");
							} else {
								Length length = new Length(propValue);
								int fontSize = (int) Math.round(length.getPoint());
								newProps.put("fontSizeAsian", fontSize);
							}
						} else if (propName.equals("style:font-name")) {
							propValue = odfProps.get("style:font-name");
							newProps.put("fontName", propValue);
						} else if (propName.equals("style:font-name-asian")) {
							propValue = odfProps.get("style:font-name-asian");
							newProps.put("fontNameAsian", propValue);
						} else if (propName.equals("style:font-name-complex")) {
							propValue = odfProps.get("style:font-name-complex");
							newProps.put("fontNameComplex", propValue);
						} else if (propName.equals("style:text-position")) {
							propValue = odfProps.get("style:text-position");
							if (propValue.contains("sub")) {
								propValue = "sub";
								newProps.put("vertAlign", propValue);
							} else if (propValue.contains("super")) {
								propValue = "super";
								newProps.put("vertAlign", propValue);
							} else if (propValue.equals("0%") || propValue.equals("0% 100%")) {
								propValue = "baseline";
								newProps.put("vertAlign", propValue);
							}
						} else if (propName.equals("fo:language")) {
							propValue = odfProps.get("fo:language");
							String country = odfProps.get("fo:country");
							if (propValue != null && !propValue.equals("none")) {
								if (country != null && !country.isEmpty() && !country.equals("none")) {
									propValue = propValue + '-' + country;
								}
								newProps.put("language", propValue);
							}
						}
						// TODO -- !!!!!!!!!!!!!!!!ROUNDTRIP WITH THESE VALUES!!!!!!!!!!!
						//	<define name="fontWeight">
						//		<choice>
						//			<value>normal</value>
						//			<value>bold</value>
						//			<value>100</value>
						//			<value>200</value>
						//			<value>300</value>
						//			<value>400</value>
						//			<value>500</value>
						//			<value>600</value>
						//			<value>700</value>
						//			<value>800</value>
						//			<value>900</value>
						//		</choice>
						//	</define>				
						if (propName.equals("fo:font-weight")) {
							propValue = odfProps.get("fo:font-weight");
							if (propValue.equals("normal")) {
								newProps.put("bold", Boolean.FALSE);
							} else {
								newProps.put("bold", Boolean.TRUE);
							}
						} else if (propName.equals("style:font-weight-asian")) {
							propValue = odfProps.get("style:font-weight-asian");
							if (propValue.equals("normal")) {
								newProps.put("boldAsian", Boolean.FALSE);
							} else {
								newProps.put("boldAsian", Boolean.TRUE);
							}
						} else if (propName.equals("style:font-weight-complex")) {
							propValue = odfProps.get("style:font-weight-complex");
							if (propValue.equals("normal")) {
								newProps.put("boldComplex", Boolean.FALSE);
							} else {
								newProps.put("boldComplex", Boolean.TRUE);
							}
						} //	<define name="lineStyle">
						//		<choice>
						//			<value>none</value>
						//			<value>solid</value>
						//			<value>dotted</value>
						//			<value>dash</value>
						//			<value>long-dash</value>
						//			<value>dot-dash</value>
						//			<value>dot-dot-dash</value>
						//			<value>wave</value>
						//		</choice>
						//	</define>				
						else if (propName.equals("style:text-underline-style")) {
							propValue = odfProps.get("style:text-underline-style");
							if (propValue.equals("none")) {
								newProps.put("underline", Boolean.FALSE);
							} else {
								newProps.put("underline", Boolean.TRUE);
							}


							//	<define name="fontStyle">
							//		<choice>
							//			<value>normal</value>
							//			<value>italic</value>
							//			<value>oblique</value>
							//		</choice>
							//	</define>					
						} else if (propName.equals("fo:font-style")) {
							propValue = odfProps.get("fo:font-style");
							if (propValue.equals("normal")) {
								newProps.put("italic", Boolean.FALSE);
							} else {
								newProps.put("italic", Boolean.TRUE);
							}
						} else if (propName.equals("style:font-style-asian")) {
							propValue = odfProps.get("style:font-style-asian");
							if (propValue.equals("normal")) {
								newProps.put("italicAsian", Boolean.FALSE);
							} else {
								newProps.put("italicAsian", Boolean.TRUE);
							}
						} else if (propName.equals("style:font-style-complex")) {
							propValue = odfProps.get("style:font-style-complex");
							if (propValue.equals("normal")) {
								newProps.put("italicComplex", Boolean.FALSE);
							} else {
								newProps.put("italicComplex", Boolean.TRUE);
							}
						} // fo:color - text props only
						else if (propName.equals("fo:color")) {
							// "auto" color type wins..
							if (!newProps.has("color")) {
								propValue = odfProps.get("fo:color");
								Map<String, String> color = createColorMap(propValue);
								newProps.put("color", color);
							}
						} else if (propName.equals("style:use-window-font-color")) {
							propValue = odfProps.get("style:use-window-font-color");
							if (propValue.equals("true")) {
								newProps.put("color", COLOR_MAP_AUTO);
							}
						} else if (propName.equals("fo:background-color")) {
							propValue = odfProps.get("fo:background-color");
							Map<String, String> color = createColorMap(propValue);
							newProps.put("fillColor", color);
						} else if (propName.equals("style:width")) {
							propValue = odfProps.get("style:width");
							if (!propValue.contains(PERCENT)) {
								newProps.put("width", normalizeLength(propValue));
							}
						} else if (propName.equals("style:writing-mode")) {
							propValue = odfProps.get("style:writing-mode");
							newProps.put("writingMode", propValue);

						} else if (propName.equals("style:text-line-through-style")) {
							/* the style might be 'none'*/
							propValue = odfProps.get("style:text-line-through-style");
							if (propValue.equals("none")) {
								newProps.put("strike", "none");
							} else {
								if (odfProps.containsKey("style:text-line-through-type")) {
									propValue = odfProps.get("style:text-line-through-type");
									/* The type is either "double" or different, than we are providing a "single" style */
									if (propValue.equals("double")) {
										newProps.put("strike", "double");
									} else {
										newProps.put("strike", "single");
									}
								} else {
									newProps.put("strike", "single");


								}
							}
						}
						//fo:letter-spacing="0.0104in" fo:hyphenate="false
//						"vertAlign":"super"}
					} catch (JSONException ex) {
						Logger.getLogger(JsonOperationProducer.class
								.getName()).log(Level.SEVERE, null, ex);
					}
				}
			} else if (styleFamilyGroup.equals("paragraph")) {
				boolean borderToBeDone = true;
				boolean paddingToBeDone = true;
				boolean marginToBeDone = true;
				JSONArray tabs = null;
				for (String propName : odfProps.keySet()) {
					try {
						if (propName.contains("margin")) {
							if (marginToBeDone) {
								mapMargin(newProps, odfProps);
								marginToBeDone = false;
							}
						} else if (propName.contains("padding")) {
							if (paddingToBeDone) {
								mapPadding(newProps, odfProps);
								paddingToBeDone = false;
							}
						} else if (propName.contains("border")) {
							if (borderToBeDone) {
								mapBorder(newProps, odfProps);
								borderToBeDone = false;
							}
						} else if (propName.equals("fo:background-color")) {
							propValue = odfProps.get("fo:background-color");
							Map<String, String> color = createColorMap(propValue);
							newProps.put("fillColor", color);
							//				<attribute name="fo:line-height">
							//					<choice>
							//						<value>normal</value>
							//						<ref name="nonNegativeLength"/>
							//						<ref name="percent"/>
							//					</choice>
							//				</attribute>
							// { type: 'percent', value: 100 }						
						} else if (propName.equals("fo:line-height")) {
							String lineHeightValue = odfProps.get("fo:line-height");
							if (lineHeightValue != null && !lineHeightValue.isEmpty()) {
								JSONObject lineHeight = createLineHeightMap(lineHeightValue);
								newProps.put("lineHeight", lineHeight);
							}
						} else if (propName.equals("style:line-spacing")) {
							String lineLeadingValue = odfProps.get("style:line-spacing");
							if (lineLeadingValue != null && !lineLeadingValue.isEmpty()) {
								JSONObject lineHeightLeadingMap = new JSONObject();
								lineHeightLeadingMap.put("type", "leading");
								lineHeightLeadingMap.put("value", normalizeLength(lineLeadingValue));
								newProps.put("lineHeight", lineHeightLeadingMap);
							}
						} else if (propName.equals("style:line-height-at-least")) {
							String lineHeightAtLeastValue = odfProps.get("style:line-height-at-least");
							if (lineHeightAtLeastValue != null && !lineHeightAtLeastValue.isEmpty()) {
								JSONObject lineHeightAtLeastMap = new JSONObject();
								lineHeightAtLeastMap.put("type", "atLeast");
								lineHeightAtLeastMap.put("value", normalizeLength(lineHeightAtLeastValue));
								newProps.put("lineHeight", lineHeightAtLeastMap);
							}
						} // see. https://dev-wiki-archiv.open-xchange.com/cgi-bin/twiki/view/Main/Operations#Paragraph_Formatting_Attributes
						// One of 'left', 'center', 'right', or 'justify'.
						// start, end, left, right, center or justify.
						else if (propName.equals("fo:text-align")) {
							propValue = (String) odfProps.get("fo:text-align");
							newProps.put("alignment", mapFoTextAlign(propValue));
						} else if (propName.equals("fo:text-indent")) {
							propValue = (String) odfProps.get("fo:text-indent");
							if (propValue.contains("%")) {
								LOG.fine("WARNING: Found a 'fo:text-indent' with percentage we are not yet supporting in our API: " + propValue);
							} else {
								newProps.put("indentFirstLine", normalizeLength(propValue));
							}
						} else if (propName.startsWith("tab_")) {
							int i = 0;
							boolean hasTabChar = false;
							boolean hasTabPos = false;
							boolean hasTabType = false;
							JSONObject tab = null;
							// addChild tab property only once
							if (!newProps.has("tabStops")) {
								while (true) {
									if (odfProps.containsKey("tab_LeaderText" + i)) {
										propValue = odfProps.get("tab_LeaderText" + i);
										if (tab == null) {
											tab = new JSONObject();
										}
										tab.put("fillChar", propValue);
										hasTabChar = true;
									} else {
										hasTabChar = false;
									}
									if (odfProps.containsKey("tab_Pos" + i)) {
										propValue = odfProps.get("tab_Pos" + i);
										if (tab == null) {
											tab = new JSONObject();
										}
										tab.put("pos", Integer.parseInt(propValue));
										hasTabPos = true;
									} else {
										hasTabPos = false;
									}
// NOT NEEDED YET									
//									if (odfProps.containsKey("tab_Type" + i)) {
//										propValue = odfProps.get("tab_Type" + i);
//										if (tab == null) {
//											tab = new JSONObject();
//										}
//										tab.put("value", (String) propValue);
//										hasTabType = true;
//									} else {
//										hasTabType = false;
//									}
									if (!hasTabChar && !hasTabType && !hasTabPos) {
										newProps.put("tabStops", tabs);
										break;
									} else {
										if (tabs == null) {
											tabs = new JSONArray();
										}
										tabs.put(tab);
										hasTabType = false;
										hasTabPos = false;
										hasTabChar = false;
										tab = null;
									}
									i++;
								}
							}
						} else if (propName.equals("style:tab-stop-distance")) {
							propValue = (String) odfProps.get("style:tab-stop-distance");
							if (propValue != null && !propValue.isEmpty()) {
								JSONObject documentProps;
								if (newProps.has("document")) {
									documentProps = newProps.getJSONObject("document");
								} else {
									documentProps = new JSONObject();
								}
								documentProps.put("defaultTabStop", normalizeLength(propValue));
								newProps.put("document", documentProps);
							}
						}
					} catch (JSONException ex) {
						Logger.getLogger(JsonOperationProducer.class
								.getName()).log(Level.SEVERE, null, ex);
					}
				}
			} else if (styleFamilyGroup.equals("cell")) {
				boolean borderToBeDone = true;
				boolean paddingToBeDone = true;
				for (String propName : odfProps.keySet()) {
					try {
						// No margin
						if (propName.contains("padding")) {
							if (paddingToBeDone) {
								mapPadding(newProps, odfProps);
								paddingToBeDone = false;
							}
						} else if (propName.contains("border")) {
							if (borderToBeDone) {
								mapBorder(newProps, odfProps);
								borderToBeDone = false;
							}
						} else if (propName.equals("fo:background-color")) {
							propValue = odfProps.get("fo:background-color");
							Map<String, String> color = createColorMap(propValue);
							newProps.put("fillColor", color);
						}
					} catch (JSONException ex) {
						Logger.getLogger(JsonOperationProducer.class
								.getName()).log(Level.SEVERE, null, ex);
					}
				}
			} else if (styleFamilyGroup.equals("column")) {
				for (String propName : odfProps.keySet()) {
					try {
						// No margin
						if (propName.equals("style:column-width")) {
							propValue = odfProps.get("style:column-width");
							newProps.put("width", normalizeLength(propValue));
						} else if (propName.contains("style:use-optimal-column-width")) {
							propValue = odfProps.get("style:use-optimal-column-width");
							newProps.put("customWidth", !Boolean.parseBoolean(propValue));
						}
					} catch (JSONException ex) {
						Logger.getLogger(JsonOperationProducer.class
								.getName()).log(Level.SEVERE, null, ex);
					}
				}
			} else if (styleFamilyGroup.equals("row")) {
				try {
					for (String propName : odfProps.keySet()) {
						if (propName.equals("fo:background-color")) {
							propValue = odfProps.get("fo:background-color");
							Map<String, String> color = createColorMap(propValue);
							newProps.put("fillColor", color);
						} else if (propName.equals("style:min-row-height")) {
							propValue = odfProps.get("style:min-row-height");
							newProps.put("height", normalizeLength(propValue));
						} else if (propName.equals("style:row-height")) {
							propValue = odfProps.get("style:row-height");
							newProps.put("height", normalizeLength(propValue));
						} else if (propName.contains("style:use-optimal-row-height")) {
							propValue = odfProps.get("style:use-optimal-row-height");
							newProps.put("customHeight", !Boolean.parseBoolean(propValue));
						}
					}
				} catch (JSONException ex) {
					Logger.getLogger(JsonOperationProducer.class
							.getName()).log(Level.SEVERE, null, ex);
				}
			} else if (styleFamilyGroup.equals("list")) {
				try {
					for (String propName : odfProps.keySet()) {
						if (propName.equals("style:font-name")) {
							propValue = odfProps.get("style:font-name");
							newProps.put("fontName", propValue);
						} else if (propName.equals("style:font-name-asian")) {
							propValue = odfProps.get("style:font-name-asian");
							newProps.put("fontNameAsian", propValue);
						} else if (propName.equals("style:font-name-complex")) {
							propValue = odfProps.get("style:font-name-complex");
							newProps.put("fontNameComplex", propValue);
						} else if (propName.equals("fo:text-align")) {
							propValue = (String) odfProps.get("fo:text-align");

							if (propValue.equals("start") || propValue.equals("left")) {
								//ToDo: I8N requires a correspondonce to the writing direction
								propValue = "left";
							} else if (propValue.equals("end") || propValue.equals("right")) {
								//ToDo: I8N requires a correspondonce to the writing direction
								propValue = "right";
							}
							newProps.put("alignment", propValue);
						}
					}
				} catch (JSONException ex) {
					Logger.getLogger(JsonOperationProducer.class
							.getName()).log(Level.SEVERE, null, ex);
				}
			} else if (styleFamilyGroup.equals("table")) {
				boolean marginToBeDone = true;
				for (String propName : odfProps.keySet()) {
					try {
						// no padding, no border
						if (propName.contains("margin")) {
							if (marginToBeDone) {
								mapMargin(newProps, odfProps);
								marginToBeDone = false;
							}
						} else if (propName.equals("fo:background-color")) {
							propValue = odfProps.get("fo:background-color");
							Map<String, String> color = createColorMap(propValue);
							newProps.put("fillColor", color);
						} else if (propName.equals("table:display")) {
							propValue = odfProps.get("table:display");
							newProps.put("visible", Boolean.parseBoolean(propValue));
						}
					} catch (JSONException ex) {
						Logger.getLogger(JsonOperationProducer.class
								.getName()).log(Level.SEVERE, null, ex);
					}
				}
			} else if (styleFamilyGroup.equals("page")) {
				boolean borderToBeDone = true;
				boolean paddingToBeDone = true;
				boolean marginToBeDone = true;
				for (String propName : odfProps.keySet()) {
					try {
						if (propName.contains("margin")) {
							if (marginToBeDone) {
								mapMargin(newProps, odfProps);
								marginToBeDone = false;
							}
						} else if (propName.contains("padding")) {
							if (paddingToBeDone) {
								mapPadding(newProps, odfProps);
								paddingToBeDone = false;
							}
						} else if (propName.contains("border")) {
							if (borderToBeDone) {
								mapBorder(newProps, odfProps);
								borderToBeDone = false;
							}
						} else if (propName.equals("fo:background-color")) {
							propValue = odfProps.get("fo:background-color");
							Map<String, String> color = createColorMap(propValue);
							newProps.put("fillColor", color);
						} else if (propName.equals("fo:page-width")) {
							propValue = odfProps.get("fo:page-width");
							newProps.put("width", normalizeLength(propValue));
						} else if (propName.equals("fo:page-height")) {
							propValue = odfProps.get("fo:page-height");
							newProps.put("height", normalizeLength(propValue));
						} else if (propName.equals("style:print-orientation")) {
							propValue = odfProps.get("style:print-orientation");
							newProps.put("printOrientation", propValue);
						} else if (propName.equals("style:num-format")) {
							propValue = odfProps.get("style:num-format");
							newProps.put("numberFormat", propValue);
						}
					} catch (JSONException ex) {
						Logger.getLogger(JsonOperationProducer.class
								.getName()).log(Level.SEVERE, null, ex);
					}
				}
			} else if (styleFamilyGroup.equals("drawing")) {
				boolean borderToBeDone = true;
				boolean paddingToBeDone = true;
				boolean marginToBeDone = true;
				for (String propName : odfProps.keySet()) {
					try {
						if (propName.contains("margin")) {
							if (marginToBeDone) {
								mapMargin(newProps, odfProps);
								marginToBeDone = false;
							}
						} else if (propName.contains("padding")) {
							if (paddingToBeDone) {
								mapPadding(newProps, odfProps);
								paddingToBeDone = false;
							}
						} else if (propName.contains("border")) {
							if (borderToBeDone) {
								mapBorder(newProps, odfProps);
								borderToBeDone = false;
							}
						} else if (propName.equals("fo:background-color")) {
							propValue = odfProps.get("fo:background-color");
							Map<String, String> color = createColorMap(propValue);
							newProps.put("fillColor", color);
							// http://docs.oasis-open.org/office/v1.2/os/OpenDocument-v1.2-os-part1.html#property-style_mirror
						} else if (propName.equals("style:mirror")) {
							String mirror = odfProps.get("style:mirror");
							if (mirror.contains("horizontal") && !mirror.contains("-on-")) {
								newProps.put("flipH", Boolean.TRUE);
							}
							if (mirror.contains("vertical")) {
								newProps.put("flipV", Boolean.TRUE);
							}
							if (mirror.equals("none")) {
								newProps.put("flipV", Boolean.FALSE);
								newProps.put("flipH", Boolean.FALSE);
							}
							/*    
							 @style:horizontal-rel:
							 The defined values for the style:horizontal-rel attribute are:
							 * char: horizontal position of a frame is positioned relative to a character.  
							 * page: horizontal position of a frame is positioned relative to a page.  
							 * page-content: horizontal position of a frame is positioned relative to page-content.  
							 * page-start-margin: horizontal position of a frame is positioned relative to a page start margin.  
							 * page-end-margin: horizontal position of a frame is positioned relative to a page end margin.  
							 * frame: horizontal position of a frame is positioned relative to another frame.  
							 * frame-content: horizontal position of a frame is positioned relative to frame content.  
							 * frame-end-margin: horizontal position of a frame is positioned relative to a frame end margin.  
							 * frame-start-margin: horizontal position of a frame is positioned relative to a frame start margin  
							 * paragraph: horizontal position of a frame is positioned relative to a paragraph.  
							 * paragraph-content: horizontal position of a frame is positioned relative to paragraph content.  
							 * paragraph-end-margin: horizontal position of a frame is positioned relative to a paragraph end margin.  
							 * paragraph-start-margin:horizontal position of a frame is positioned relative to a paragraph start margin.  					

							 @style:horizontal-pos:
							 The defined values for the style:horizontal-pos attribute are:
							 * center: horizontal alignment of a frame should be centered relative to the specified area. 
							 * anchorHorAlign=center
							 * from-inside: on pages with an odd page number the left edge of the specific area is taken as the horizontal alignment of a frame. On pages with an even page number the right edge of the specified area is taken. Attribute svg:x associated with the frame element specifies the horizontal position of the frame from the edge which is taken.  
							 * UNSUPPORTED
							 * from-left: the svg:x attribute associated with the frame element specifies the horizontal position of the frame from the left edge of the specified area. 
							 * anchorHorAlign=offset 
							 * inside: on pages with an odd page number the horizontal alignment of a frame is the same as for the attribute value left. On pages with an even page number the horizontal alignment of a frame is the same as for the attribute value right.  
							 * anchorHorAlign=inside
							 * left: horizontal alignment of a frame should be left aligned relative to the specified area. 
							 * anchorHorAlign=left
							 * outside: on pages with an odd page number the horizontal alignment of a frame is the same as for the attribute value right. On pages with an even page number the horizontal alignment of a frame is the same as for the attribute value left.  
							 * anchorHorAlign=outside 
							 * right: horizontal alignment of a frame should be right aligned relative to the specified area. 
							 * anchorHorAlign=right
							 If the attribute value is not from-left and not from-inside, the svg:x attribute associated with the frame element is ignored for text documents.						 */
							// OX API:
							// anchorHorAlign: Horizontal anchor position:	One of 'left', 'right', 'center', 'inside', 'outside', or 'offset'.   	
							// anchorHorOffset: Horizontal position offset (only used if anchorHorAlign is set to 'offset') 
						} else if (propName.contains("horizontal-pos")) {
							String horizontalPos = odfProps.get("style:horizontal-pos");
							if (horizontalPos.equals("center")) {
								newProps.put("anchorHorAlign", "center");
							} else if (horizontalPos.equals("from-left")) {
								newProps.put("anchorHorAlign", "offset");
							} else if (horizontalPos.equals("left")) {
								newProps.put("anchorHorAlign", "left");
							} else if (horizontalPos.equals("right")) {
								newProps.put("anchorHorAlign", "right");
							} else if (horizontalPos.equals("inside")) {
								newProps.put("anchorHorAlign", "inside");
							} else if (horizontalPos.equals("outside")) {
								newProps.put("anchorHorAlign", "outside");
							}
							// anchorVertAlign	 Vertical anchor position. One of 'top', 'bottom', 'center', 'inside', 'outside', 'offset'.
						} else if (propName.contains("vertical-pos")) {
							String verticalPos = odfProps.get("style:vertical-pos");
							if (verticalPos.equals("center")) {
								newProps.put("anchorVertAlign", "center");
							} else if (verticalPos.equals("from-top")) {
								newProps.put("anchorVertAlign", "offset");
							} else if (verticalPos.equals("top")) {
								newProps.put("anchorVertAlign", "top");
							} else if (verticalPos.equals("bottom")) {
								newProps.put("anchorVertAlign", "bottom");
							} else if (verticalPos.equals("inside")) {
								newProps.put("anchorVertAlign", "inside");
							} else if (verticalPos.equals("outside")) {
								newProps.put("anchorVertAlign", "outside");
							}
						} else if (propName.equals("svg:x")) {
							int x = normalizeLength(odfProps.get("svg:x"));
							if (x != 0) {
								newProps.put("anchorHorOffset", x);
							}
						} else if (propName.equals("svg:y")) {
							int y = normalizeLength(odfProps.get("svg:y"));
							if (y != 0) {
								newProps.put("anchorVertOffset", y);
							}

							/*
							 OX API:
							 cropLeft	Left cropping in percent	 
							 cropRight	Right cropping in percent	 
							 cropTop		Top cropping in percent	 	
							 cropBottom	Bottom cropping in percent
							 http://www.w3.org/TR/CSS2/visufx.html#propdef-clip */
							// <top>, <right>, <bottom>, <left>					
							// Value length or "auto"
							// e.g. @fo:clip="rect(0in, 0.07874in, 0in, 0.07874in)"
							// PROBLEM: The width & height is not accessible
						} else if (propName.equals("fo:clip")) {
							String clipping = odfProps.get("fo:clip");
							int start = clipping.indexOf("rect(");
							int end = clipping.indexOf(")");
							if (start > -1 && end > -1) {
								clipping = clipping.substring(start + 5, end);
							}
							String clips[] = clipping.split(", ");
							if (clips.length != 4) {
								// fallback as some documents to not behave as the standard explains
								clips = clipping.split(" ");
							}
							int clipTop = normalizeLength(clips[0]);
							newProps.put("cropTop", clipTop);
							int clipRight = normalizeLength(clips[1]);
							newProps.put("cropRight", clipRight);
							int clipBottom = normalizeLength(clips[2]);
							newProps.put("cropBottom", clipBottom);
							int clipLeft = normalizeLength(clips[3]);
							newProps.put("cropLeft", clipLeft);

							/* http://docs.oasis-open.org/office/v1.2/os/OpenDocument-v1.2-os-part1.html#property-style_wrap
							 The style:wrap attribute specifies how text is displayed around a frame or graphic object.
							 The defined values for the style:wrap attribute are:
							 * biggest: text may wrap around the shape where the difference to the left or right page or column border is largest. 
							 * Mode=square Side=largest
							 * dynamic: text may wrap around both sides of the shape. The space for wrapping is set by the style:wrap-dynamic-threshold attribute. 20.393 
							 * UNSUPPORTED
							 * left: text wraps around the left side of the shape. 
							 * Mode=square	Side=left
							 * none: text does not wrap around the shape. 
							 * Mode=topAndBottom 
							 * parallel: text wraps around both sides of the shape. 
							 * Mode=square	Side=both	 
							 * right: text wraps around the right side of the shape. 
							 * Mode=square	Side=right
							 * run-through: text runs through the shape.
							 * Mode=through	Side=both   */
							/* 
							 OX API: textWrapMode	One of 'none', 'square', 'tight', 'through', or 'topAndBottom'. (IMHO 'none' == 'topAndBottom', but the latter is implemented)
							 OX API: textWrapSide	Sides where text wraps around the image (only used if textWrapMode is set to 'square', 'tight', or 'through') (1). One of ['both', 'left', 'right', 'largest'	*/
						} else if (propName.equals("style:wrap")) {
							String wrap = (String) odfProps.get("style:wrap");

							if (wrap.equals("biggest")) {
								newProps.put("textWrapMode", "square");
								newProps.put("textWrapSide", "largest");
							} else if (wrap.equals("left")) {
								newProps.put("textWrapMode", "square");
								newProps.put("textWrapSide", "left");
							} else if (wrap.equals("none")) {
								newProps.put("textWrapMode", "topAndBottom");
							} else if (wrap.equals("parallel")) {
								newProps.put("textWrapMode", "square");
								newProps.put("textWrapSide", "both");
							} else if (wrap.equals("right")) {
								newProps.put("textWrapMode", "square");
								newProps.put("textWrapSide", "right");
							} else if (wrap.equals("run-through")) {
								newProps.put("textWrapMode", "through");
								newProps.put("textWrapSide", "both");


							}
						}
					} catch (JSONException ex) {
						Logger.getLogger(JsonOperationProducer.class
								.getName()).log(Level.SEVERE, null, ex);
					}
				}
			} // NOTE: HEADER FOOTER ARE SOMEHOW (ASYMETRIC) NESTED AMONG PAGE PROPERTIES.. (ToDo: Unused yet!)
			else if (styleFamilyGroup.equals("headerFooter")) {
				boolean borderToBeDone = true;
				boolean paddingToBeDone = true;
				boolean marginToBeDone = true;
				for (String propName : odfProps.keySet()) {
					try {
						if (propName.contains("margin")) {
							if (marginToBeDone) {
								mapMargin(newProps, odfProps);
								marginToBeDone = false;
							}
						} else if (propName.contains("padding")) {
							if (paddingToBeDone) {
								mapPadding(newProps, odfProps);
								paddingToBeDone = false;
							}
						} else if (propName.contains("border")) {
							if (borderToBeDone) {
								mapBorder(newProps, odfProps);
								borderToBeDone = false;
							}
						} else if (propName.equals("fo:background-color")) {
							propValue = odfProps.get("fo:background-color");
							Map<String, String> color = createColorMap(propValue);
							newProps.put("fillColor", color);


						}
					} catch (JSONException ex) {
						Logger.getLogger(JsonOperationProducer.class
								.getName()).log(Level.SEVERE, null, ex);
					}
				}
			}
		}
		return newProps;
	}

	/**
	 * LO/AO does not interpret clipping as in FO/CSS
	 * http://www.w3.org/TR/CSS2/visufx.html#propdef-clip Instead the clip
	 * values measure the distance from each border to the start of the viewed
	 * area. The clip vales are taking measure from the original size, which is
	 * not part of the OFD XML, therefore the image have to be loaded for
	 * receiving the size.
	 */
	static void calculateCrops(OdfElement image, String href, JSONObject drawingProps) {
		try {
			// ToDo: Although the streams are cached we might cache the clipping for known href, help if images occure more than once
			OdfPackage pkg = ((OdfFileDom) image.getOwnerDocument()).getDocument().getPackage();

			InputStream is = pkg.getInputStream(href);
			if (is != null) {
				BufferedImage bimg = ImageIO.read(is);
				if (bimg != null) {
					double width = JsonOperationProducer.normalizeLength((bimg.getWidth() / DOTS_PER_INCH) + "in");
					double height = JsonOperationProducer.normalizeLength((bimg.getHeight() / DOTS_PER_INCH) + "in");
					try {
						// 2nd half of absolute fo:clip to relative crop (OX API) mapping 
						if (drawingProps.has("cropRight")) {
							Number cropRight = (Number) drawingProps.get("cropRight");
							LOG.log(Level.FINEST, "The clipRight is {0}", cropRight);
							if (cropRight != null) {
								if (cropRight.doubleValue() != 0.0) {
									drawingProps.put("cropRight", cropRight.doubleValue() * 100.0 / width);
									LOG.log(Level.FINEST, "The cropRight is {0}", cropRight.doubleValue() * 100.0 / width);
								} else {
									// do not set explicitly with 0
									drawingProps.remove("cropRight");
								}
							}
						}
						if (drawingProps.has("cropLeft")) {
							Number cropLeft = (Number) drawingProps.get("cropLeft");
							LOG.log(Level.FINEST, "The clipLeft is {0}", cropLeft);
							if (cropLeft != null) {
								if (cropLeft.doubleValue() != 0.0) {
									drawingProps.put("cropLeft", cropLeft.doubleValue() * 100.0 / width);
									LOG.log(Level.FINEST, "The cropLeft is {0}", cropLeft.doubleValue() * 100.0 / width);
								} else {
									// do not set explicitly with 0
									drawingProps.remove("cropLeft");
								}
							}
						}
						// 2nd half of absolute fo:clip to relative crop (OX API) mapping 
						if (drawingProps.has("cropTop")) {
							Number cropTop = (Number) drawingProps.get("cropTop");
							LOG.log(Level.FINEST, "The clipTop is {0}", cropTop);
							double d = cropTop.doubleValue();
							if (cropTop != null) {
								if (cropTop.doubleValue() != 0.0) {
									drawingProps.put("cropTop", cropTop.doubleValue() * 100.0 / height);
									LOG.log(Level.FINEST, "The cropTop is {0}", cropTop.doubleValue() * 100.0 / height);
								} else {
									// do not set explicitly with 0
									drawingProps.remove("cropTop");
								}
							}
						}
						if (drawingProps.has("cropBottom")) {
							Number cropBottom = (Number) drawingProps.get("cropBottom");
							LOG.log(Level.FINEST, "The clipBottom is {0}", cropBottom);
							if (cropBottom != null) {
								if (cropBottom.doubleValue() != 0.0) {
									drawingProps.put("cropBottom", cropBottom.doubleValue() * 100.0 / height);
									LOG.log(Level.FINEST, "The cropBottom is {0}", cropBottom.doubleValue() * 100.0 / height);
								} else {
									// do not set explicitly with 0
									drawingProps.remove("cropBottom");


								}
							}
						}
					} catch (JSONException ex) {
						Logger.getLogger(OdfFileSaxHandler.class
								.getName()).log(Level.SEVERE, null, ex);
					}
					LOG.log(Level.FINEST, "Width: {0} Height: {1}", new Object[]{width, height});
				} else {
					LOG.log(Level.WARNING, "The image ''{0}'' could not be loaded!", href);
				}
			} else {
				LOG.log(Level.WARNING, "The image ''{0}'' could not be loaded!", href);


			}
		} catch (IOException ex) {
			Logger.getLogger(OdfFileSaxHandler.class
					.getName()).log(Level.SEVERE, "Image could not be found at " + href, ex);
		}
	}

	private static String mapFoTextAlign(String propValue) {
		if (propValue.equals("start") || propValue.equals("left")) {
			//ToDo: I8N requires a correspondonce to the writing direction
			propValue = "left";
		} else if (propValue.equals("end") || propValue.equals("right")) {
			//ToDo: I8N requires a correspondonce to the writing direction
			propValue = "right";
		}
		return propValue;
	}

	private static void mapMargin(JSONObject newProps, Map<String, String> odfProps) throws JSONException {
		String propValue;
		Integer defaultLength = null;
		if (odfProps.containsKey("fo:margin")) {
			propValue = odfProps.get("fo:margin");
			if (!propValue.contains(PERCENT)) {
				defaultLength = normalizeLength(propValue);
			}
		}
		if (odfProps.containsKey("fo:margin-left")) {
			propValue = odfProps.get("fo:margin-left");
			if (!propValue.contains(PERCENT)) {
				newProps.put("marginLeft", normalizeLength(propValue));
				newProps.put("indentLeft", normalizeLength(propValue)); //FIX API
			}
		} else {
			if (defaultLength != null) {
				newProps.put("marginLeft", defaultLength);
				newProps.put("indentLeft", defaultLength); //FIX API				
			}
		}
		if (odfProps.containsKey("fo:margin-top")) {
			propValue = odfProps.get("fo:margin-top");
			if (!propValue.contains(PERCENT)) {
				newProps.put("marginTop", normalizeLength(propValue));
			}
		} else {
			if (defaultLength != null) {
				newProps.put("marginTop", defaultLength);
			}
		}
		if (odfProps.containsKey("fo:margin-right")) {
			propValue = odfProps.get("fo:margin-right");

			if (!propValue.contains(PERCENT)) {
				newProps.put("marginRight", normalizeLength(propValue));
				newProps.put("indentRight", normalizeLength(propValue)); //FIX API
			}
		} else {
			if (defaultLength != null) {
				newProps.put("marginRight", defaultLength);
				newProps.put("indentRight", defaultLength); //FIX API
			}
		}
		if (odfProps.containsKey("fo:margin-bottom")) {
			propValue = odfProps.get("fo:margin-bottom");
			if (!propValue.contains(PERCENT)) {
				newProps.put("marginBottom", normalizeLength(propValue));
			}
		} else {
			if (defaultLength != null) {
				newProps.put("marginBottom", defaultLength);
			}
		}
	}

	private static void mapPadding(JSONObject newProps, Map<String, String> odfProps) throws JSONException {
		String propValue;
		Integer defaultLength = null;
		if (odfProps.containsKey("fo:padding")) {
			propValue = odfProps.get("fo:padding");
			if (!propValue.contains(PERCENT)) {
				defaultLength = normalizeLength(propValue);
			}
		}
		if (odfProps.containsKey("fo:padding-left")) {
			propValue = odfProps.get("fo:padding-left");
			if (!propValue.contains(PERCENT)) {
				newProps.put("paddingLeft", normalizeLength(propValue));
			}
		} else {
			if (defaultLength != null) {
				newProps.put("paddingLeft", defaultLength);
			}
		}
		if (odfProps.containsKey("fo:padding-top")) {
			propValue = odfProps.get("fo:padding-top");
			if (!propValue.contains(PERCENT)) {
				newProps.put("paddingTop", normalizeLength(propValue));
			}
		} else {
			if (defaultLength != null) {
				newProps.put("paddingTop", defaultLength);
			}
		}
		if (odfProps.containsKey("fo:padding-right")) {
			propValue = odfProps.get("fo:padding-right");

			if (!propValue.contains(PERCENT)) {
				newProps.put("paddingRight", normalizeLength(propValue));
			}
		} else {
			if (defaultLength != null) {
				newProps.put("paddingRight", defaultLength);
			}
		}
		if (odfProps.containsKey("fo:padding-bottom")) {
			propValue = odfProps.get("fo:padding-bottom");
			if (!propValue.contains(PERCENT)) {
				newProps.put("paddingBottom", normalizeLength(propValue));
			}
		} else {
			if (defaultLength != null) {
				newProps.put("paddingBottom", defaultLength);
			}
		}
	}

	private static void mapBorder(JSONObject newProps, Map<String, String> odfProps) throws JSONException {
		String propValue;
		JSONObject defaultBorder = null;
		Integer defaultSpace = null;
		// fo:border="0.5pt solid #000000" 
		if (odfProps.containsKey("fo:border")) {
			propValue = odfProps.get("fo:border");
			defaultBorder = createBorderMap(propValue);
			if (odfProps.containsKey("fo:padding")) {
				propValue = odfProps.get("fo:padding");
				if (!propValue.contains(PERCENT)) {
					defaultSpace = normalizeLength(propValue);
				}
				defaultBorder.put("space", defaultSpace);
			}
		}
		if (odfProps.containsKey("fo:border-left")) {
			propValue = odfProps.get("fo:border-left");
			JSONObject border = createBorderMap(propValue);
			if (odfProps.containsKey("fo:padding-left")) {
				propValue = odfProps.get("fo:padding-left");
				if (!propValue.contains(PERCENT)) {
					border.put("space", normalizeLength(propValue));
				}
			} else if (defaultSpace != null) {
				border.put("space", defaultSpace);
			}
			newProps.put("borderLeft", border);
		} else {
			newProps.put("borderLeft", defaultBorder);
		}
		if (odfProps.containsKey("fo:border-top")) {
			propValue = odfProps.get("fo:border-top");
			JSONObject border = createBorderMap(propValue);
			if (odfProps.containsKey("fo:padding-top")) {
				propValue = odfProps.get("fo:padding-top");
				if (!propValue.contains(PERCENT)) {
					border.put("space", normalizeLength(propValue));
				}
			} else if (defaultSpace != null) {
				border.put("space", defaultSpace);
			}
			newProps.put("borderTop", border);
		} else {
			newProps.put("borderTop", defaultBorder);
		}
		if (odfProps.containsKey("fo:border-right")) {
			propValue = odfProps.get("fo:border-right");
			JSONObject border = createBorderMap(propValue);
			if (odfProps.containsKey("fo:padding-right")) {
				propValue = odfProps.get("fo:padding-right");
				if (!propValue.contains(PERCENT)) {
					border.put("space", normalizeLength(propValue));
				}
			} else if (defaultSpace != null) {
				border.put("space", defaultSpace);
			}
			newProps.put("borderRight", border);
		} else {
			newProps.put("borderRight", defaultBorder);
		}
		if (odfProps.containsKey("fo:border-bottom")) {
			propValue = odfProps.get("fo:border-bottom");
			JSONObject border = createBorderMap(propValue);
			if (odfProps.containsKey("fo:padding-bottom")) {
				propValue = odfProps.get("fo:padding-bottom");
				if (!propValue.contains(PERCENT)) {
					border.put("space", normalizeLength(propValue));
				}
			} else if (defaultSpace != null) {
				border.put("space", defaultSpace);
			}
			newProps.put("borderBottom", border);
		} else {
			newProps.put("borderBottom", defaultBorder);
		}
	}

	/**
	 * see
	 * https://dev-wiki-archiv.open-xchange.com/cgi-bin/twiki/view/Main/Operations#Color
	 */
	private static Map<String, String> createColorMap(String rgbValue) {
		Map color = new HashMap<String, String>();
		if (rgbValue.contains(HASH)) {
			color.put("type", "rgb");
			rgbValue = rgbValue.subSequence(rgbValue.indexOf('#') + 1, rgbValue.length()).toString();
			color.put("value", rgbValue);
		} else if (rgbValue.equals(TRANSPARENT) || rgbValue.equals(AUTO)) {
			color.put("type", AUTO);
		}
		return color;
	}

	/**
	 * see
	 * https://dev-wiki-archiv.open-xchange.com/cgi-bin/twiki/view/Main/Operations#Border
	 */
	private static JSONObject createBorderMap(String borderValue) {
		JSONObject border = new JSONObject();
		try {
			if (borderValue.equals("none")) {
				border.put("style", "none");
			} else {
				String[] tokens = borderValue.split("\\s+");
				boolean checkedColor = false;
				boolean checkedStyle = false;
				boolean checkedWidth = false;
				for (int i = 0; i < tokens.length; i++) {
					String token = tokens[i];
					if (!token.isEmpty()) {
						boolean isTokenTaken = false;
						if (!checkedColor) {
							if (isColor(token)) {
								checkedColor = mapColor(border, token);
								isTokenTaken = checkedColor;
							}
						}
						if (!isTokenTaken && !checkedStyle) {
							checkedStyle = mapStyle(border, token);
							isTokenTaken = checkedStyle;
						}
						if (!isTokenTaken && !checkedWidth) {
							checkedWidth = mapWidth(border, token, tokens);


						}
					}
				}
			}
		} catch (JSONException ex) {
			Logger.getLogger(JsonOperationProducer.class
					.getName()).log(Level.SEVERE, null, ex);
		}
		return border;
	}

	private static boolean isColor(String color) {
		return color.startsWith(HASH) || color.equals(TRANSPARENT);
	}

	private static boolean mapColor(JSONObject border, String width) {
		boolean isColor = false;
		try {
			// try if the first token is a width
			border.put("color", createColorMap(width));


		} catch (JSONException ex) {
			Logger.getLogger(JsonOperationProducer.class
					.getName()).log(Level.SEVERE, null, ex);
		}
		isColor = true;

		return isColor;
	}

	/**
	 * <define name="lineWidth">
	 * <choice>
	 * <value>auto</value>
	 * <value>normal</value>
	 * <value>bold</value>
	 * <value>thin</value>
	 * <value>medium</value>
	 * <value>thick</value>
	 * <ref name="positiveInteger"/>
	 * <ref name="percent"/>
	 * <ref name="positiveLength"/>
	 * </choice>
	 * </define>
	 */
	private static boolean mapWidth(JSONObject border, String widthString, String[] tokens) {
		boolean isWidth = false;
		int width;
		try {
			if (widthString.equals(AUTO)) {
				width = 26;// my guess
			} else if (widthString.equals(NORMAL)) {
				width = 27;// my guess
			} else if (widthString.equals(BOLD)) {
				width = 54;// my guess
			} else if (widthString.equals(THIN)) {
				width = 26;
			} else if (widthString.equals(MEDIUM)) {
				width = 53;
			} else if (widthString.equals(THICK)) {
				width = 79;
			} else {
				width = normalizeLength(widthString);
			}
			// try if the first token is a width
			border.put("width", width);
			isWidth = true;
		} catch (Throwable t) {
			// there have to a a width with three tokens
			if (tokens.length == 3) {
				throw new RuntimeException(t);
			}
		}
		return isWidth;
	}

	private static boolean mapStyle(JSONObject border, String style) throws JSONException {
		boolean isStyle = false;
		//solid	The border is a solid line
		//groove	The border looks like it were carved into the canvas
		//ridge	The border "comes out" of the canvas
		if (style.equals("solid") || style.equals("groove") || style.equals("ridge")) {
			style = "single";
			border.put("style", style);
			isStyle = true;

			// hidden	A hidden border
		} else if (style.equals("hidden")) {
			style = "none";
			border.put("style", style);
			isStyle = true;

		} else if (style.equals("double") || style.equals("dotted") || style.equals("dashed") || style.equals("outset") || style.equals("inset")) {
			border.put("style", style);
			isStyle = true;
		}
		return isStyle;
	}

	/**
	 * see
	 * https://dev-wiki-archiv.open-xchange.com/cgi-bin/twiki/view/Main/Operations#LineHeight
	 */
	private static JSONObject createLineHeightMap(String lineHeightValue) {
		JSONObject lineHeight = new JSONObject();
		try {
			/**
			 * Usually normal is given by the font and 110% to 120% of the
			 * font-size (see
			 * http://www.w3.org/TR/CSS2/visudet.html#line-height), browser even
			 * show 1.1 to 1.3, but in office applicatioons normal is ALWAYS
			 * 100%, therefore it will be mapped see as well
			 * http://www.w3.org/TR/xsl/#line-height
			 */
			if (lineHeightValue.equals("normal")) {
				lineHeight.put("type", "percent");
				lineHeight.put("value", "100");
			} else if (lineHeightValue.contains("%")) {
				lineHeight.put("type", "percent");
				lineHeight.put("value", Integer.parseInt(lineHeightValue.subSequence(0, lineHeightValue.indexOf('%')).toString()));
			} else {
				lineHeight.put("type", "fixed");
				lineHeight.put("value", normalizeLength(lineHeightValue));


			}
		} catch (JSONException ex) {
			Logger.getLogger(JsonOperationProducer.class
					.getName()).log(Level.SEVERE, null, ex);
		}
		return lineHeight;
	}

	/*
	 Renames a sheet in the spreadsheet document.

	 Property	 Type	 Description
	 name	 String	 'setSheetName'
	 sheet	 Integer	 The zero-based index of the sheet to be renamed.
	 sheetName	 String	 The new visible name of the sheet. Must not be empty. Must not be equal to the name of a
	 */
	private static void setSheetName(int sheetNo, String sheetName) {
		throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
	}

	/*
	 Deletes entire rows from a sheet in the spreadsheet document.

	 Property	 Type	 Description
	 name	 String	 'deleteRows'
	 sheet	 Integer	 The zero-based index of the sheet.
	 start	 Integer	 The zero-based index of the first row to be deleted.
	 end	 Integer	 (optional) The zero-based index of the last row to be deleted. If omitted, a single row will be deleted. If specified, MUST be greater than or equal to start.
	 Drawing objects that are completely contained in the deleted rows will be deleted from the sheet too. Drawing objects that are partly contained in the deleted rows will be shortened accordingly.

	 Concerning the behavior for deleting rows containing merged cells and borders have a look at this description.
	 */
	private static void deleteSheetRows(int sheetNo, int start, int end) {
		throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
	}

	/*
	 Adds new columns to a sheet in the spreadsheet document.

	 Property	 Type	 Description
	 name	 String	 'insertColumns'
	 sheet	 Integer	 The zero-based index of the sheet.
	 start	 Integer	 The zero-based index of the first column to be moved aside in order to insert the new columns.
	 end	 Integer	 (optional) The zero-based index of the last column in the column range. If omitted, a single column will be inserted. If specified, MUST be greater than or equal to start.
	 All new columns will be formatted exactly like the column preceding the inserted columns (with column index start-1, also if that column is hidden), including the individual formatting of single cells in that column, with the exception that the new columns will always be visible. If the columns are inserted at the beginning of the sheet, they will not contain any cell formatting, and will get the default width as specified in the sheet attribute colWidth.

	 Example: New columns will be inserted into the column range B:C. All existing columns starting at column B will be moved aside by two columns. The new empty columns B and C will be formatted as column A and all the cells it contains. Columns B and C will be visible even if column A is hidden.

	 Concerning the behavior for inserting columns into merged cells and borders have a look at this description.
	 */
	private static void insertSheetColumns(int aInt, int aInt0, int optInt) {
		throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
	}

	/*
	 Deletes entire columns from a sheet in the spreadsheet document.

	 Property	 Type	 Description
	 name	 String	 'deleteColumns'
	 sheet	 Integer	 The zero-based index of the sheet.
	 start	 Integer	 The zero-based index of the first column to be deleted.
	 end	 Integer	 (optional) The zero-based index of the last column to be deleted. If omitted, a single column will be deleted. If specified, MUST be greater than or equal to start.
	 Drawing objects that are completely contained in the deleted columns will be deleted from the sheet too. Drawing objects that are partly contained in the deleted columns will be shortened accordingly.

	 Concerning the behavior for deleting columns containing merged cells and borders have a look at this description.
	 */
	private static void deleteSheetColumns(int aInt, int aInt0, int optInt) {
		throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
	}

	/*
	 Overwrites a range of cells with the specified values and/or formatting attributes.

	 Property	 Type	 Description	 Notes
	 name	 String	 'setCellContents'	
	 sheet	 Integer	 The zero-based index of the sheet containing the cell range.	
	 start	 Integer[2]	 The logical cell position of the upper-left cell in the range.	
	 contents	 Object[][]	 The values and attribute sets to be written into the cell range. MUST be a 2-dimensional array. The outer array contains rows of cell contents, and the inner row arrays contain the cell contents for each single row. The lengths of the inner arrays may be different. Cells not covered by a row array will not be modified. See next table for details about the format of the single cell content objects.	
	 parse	 String	 (optional) If set, the values in the passed contents property must be strings (if existing), and will be parsed to detect their actual contents (numbers, Booleans, formulas, etc.). The value of this property contains the locale to be used to parse the values.	 (1)
	 Each array element of the contents property describes the contents of a single cell. If set to the value null, the cell will not be modified. Otherwise, the array element represents an empty cell, a value cell, a formula cell, or a shared formula cell:

	 Property	 Type	 Description	 Notes
	 value	CellValue	 (optional) The new value of the cell. The value null will clear the cell. If omitted, the current value will not change (e.g., to change the formatting only), except for shared formulas referred by the shared attribute of this object. If the parse property is set in the operation, the value must be a string.	
	 shared	 Integer	 (optional) The identifier of the shared formula in the sheet addressed by this operation.	
	 ref	 Integer[2]	 (optional) The logical cell position of the reference cell, used to adjust relative references in a shared formula.	
	 result	CellValue	 (optional) The result of a formula cell. Intended to transport formula results to export filters which store the formula and last result in the cell.	 (2)
	 attrs	 Object	 (optional) The attribute set containing all attributes to be changed. Attributes with a value of null will be removed from the cell. Attributes not contained in the set remain unmodified.	
	 Notes

	 (1) Supported by application frontend only, must not be sent from or to document filters.

	 (2) Supported by document filters only, must not be sent from the application frontend.
	 */
	private static void setSheetCellContents(int aInt, JSONArray jsonArray, JSONArray jsonArray0) {
		throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
	}


	/*
	 Changes the formatting attributes of a range of columns in a sheet.

	 Property	 Type	 Description
	 name	 String	 'formatColumns'
	 sheet	 Integer	 The zero-based index of the sheet containing the columns.
	 start	 Integer	 The zero-based index of the first column in the column range.
	 end	 Integer	 (optional) The zero-based index of the last column in the column range. If omitted, the operation addresses a single column. If specified, MUST be greater than or equal to start.
	 attrs	 Object	 The attribute set for the columns, containing all attributes to be changed. Attributes with a value of null will be removed from the columns. Attributes not contained in the set remain unmodified.
	 rangeBorders	 Boolean	 (optional) If set to true, the border attributes in the passed attrs property must be assigned to the complete cell range (borderLeft, borderRight, borderTop, and borderBottom for the outer borders of the outer cells in the range, and borderInsideHor and borderInsideVert for all other inner borders). If not set, the outer border attributes must be assigned to each single cell of the selection, and values for inner borders will be ignored.
	 visibleBorders	 Boolean	 (optional) If set to true, the border attributes in the passed attrs property may be incomplete. Existing border properties ('style', 'width', or 'color') must only be assigned to visible borders (all borders with a current style property different from none).
	 The attribute set passed to the operation may contain cell attributes, character attributes, or a cell style reference which will be used to render undefined cells in the columns.

	 The following additional actions have to be performed for the columns addressed by this operation:

	 The attributes and/or cell style will be applied (changed or removed, according to the attribute values) at all defined cells in the columns.
	 Empty but formatted cells will be removed completely, if their explicit formatting attributes and style reference are completely equal to the new default column formatting (a new empty cell may be inserted afterwards for formatted rows, see next point).
	 New empty cells must be created and formatted accordingly for all undefined cells in formatted rows, as long as the default row formatting differs from the resulting default column formattign of this operation (this is needed because default row formatting overrides default column formatting).
	 Drawing objects that are completely contained in hidden columns will be invisible, but will NOT be deleted from the sheet. Showing the hidden columns will show the drawing objects according to their cell anchor attributes.

	 Example:

	 Current state of the sheet:
	 Cell B1 contains the value 1 and has default formatting.
	 Cell B2 contains red fill color.
	 Row 3 contains red fill color for all unspecified cells.
	 Row 4 contains yellow fill color for all unspecified cells.
	 Yellow fill color will be set to column B using the formatColumns operation. The following changes have to be made:
	 The color attribute will be stored at column B.
	 The color attribute will be stored at the non-empty cells B1 and B2.
	 Empty but formatted cell B2 will be removed (its new formatting equals the new default column formatting).
	 A new empty cell B3 with the color attribute will be created (overrides the default formatting of row 3 with red fill color).
	 Nothing happens for cell B4 (default formatting of row 4 is equal to new default column formatting).
	 */
	private static void setSheetColumnAttributes(int aInt, int aInt0, int optInt, JSONObject jsonObject) {
		throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
	}

	/*
	 Adds new rows to a sheet in the spreadsheet document.

	 Property	 Type	 Description
	 name	 String	 'insertRows'
	 sheet	 Integer	 The zero-based index of the sheet.
	 start	 Integer	 The zero-based index of the first row to be moved down in order to insert the new rows.
	 end	 Integer	 (optional) The zero-based index of the last row in the row range. If omitted, a single row will be inserted. If specified, MUST be greater than or equal to start.
	 All new rows will be formatted exactly like the row above the inserted rows (with row index start-1, also if that row is hidden), including the individual formatting of single cells in that row, with the exception that the new rows will always be visible. If the rows are inserted at the beginning of the sheet, they will not contain any cell formatting, and will get the default height as specified in the sheet attribute rowHeight.

	 Example: New rows will be inserted into the row range 2:3. All existing rows starting at row 2 will be moved down by two rows. The new empty rows 2 and 3 will be formatted as row 1 and all the cells it contains. Rows 2 and 3 will be visible even if row 1 is hidden.

	 Concerning the behavior for inserting rows into merged cells and borders have a look at this description.
	 */
	private static void insertSheetRows(int aInt, int aInt0, int optInt) {
		throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
	}

	/*
	 Merge cells in a sheet in the spreadsheet document. For additional information please look here.

	 Property	 Type	 Description
	 name	 String	 'mergeCells'
	 sheet	 Integer	 The zero-based index of the sheet.
	 start	 Integer[2]	 The logical cell position of the upper-left cell in the range.
	 end	 Integer[2]	 (optional) The logical cell position of the lower-right cell in the range.
	 keepContent	 Boolean	 (optional) Whether the content inside the cell shall be removed or not. This flag is only required for communication between calcEngine and filter. Default: false
	 type	 Enum		merge	 The cells in the specified cell range will be merged (default).
	 horizontal	 The cells in the specified cell range will be merged horizontally. This means that all cells inside a row will be merged.
	 vertical	 The cells in the specified cell range will be merged vertically. This means that all cells inside a column will be merged.
	 unmerge	 The cells in the specified cell range will be unmerged.
	 */
	private static void mergeSheetCells(int aInt, JSONArray jsonArray, JSONArray optJSONArray, boolean optBoolean, String optString) {
		throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
	}
}