/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.calcengine.client.impl;

import java.util.List;

import org.apache.commons.lang.Validate;
import org.apache.commons.logging.Log;
import org.json.JSONObject;

import com.openexchange.office.calcengine.JOperation;
import com.openexchange.office.calcengine.client.CalcEngineClipBoardEvent;
import com.openexchange.office.calcengine.client.CalcEngineHandleGenerator;
import com.openexchange.office.calcengine.client.ECalcEngineError;
import com.openexchange.office.calcengine.client.ICalcEngineClient;
import com.openexchange.office.tools.logging.ContextAwareLogHelp;
import com.openexchange.office.tools.logging.ELogLevel;
import com.openexchange.office.tools.logging.LogFactory;

//=============================================================================
/** Call the native (JNI) bridge directly within the current process.
 */
public class CalcEngineClientNative implements ICalcEngineClient
{
    //-------------------------------------------------------------------------
    private static final Log LOG = LogFactory.getJclLog(CalcEngineClientNative.class);

    //-------------------------------------------------------------------------
//	private static final boolean IGNORE_ERRORS = false;
	
    //-------------------------------------------------------------------------
    private CalcEngineClientNative ()
        throws Exception
    {}
    
    //-------------------------------------------------------------------------
    public synchronized static CalcEngineClientNative create()
        throws Exception
    {
    	final CalcEngineClientNative aClient = new CalcEngineClientNative ();
    	return aClient;
    }

    //-------------------------------------------------------------------------
    @Override
    public String createDocument()
        throws Exception
    {
        final String                  sHandle = CalcEngineHandleGenerator.newHandle();
        final CalcEngineNativeWrapper aEngine = mem_Engine ();
        final boolean                 bOK     = aEngine.createDocument(sHandle);
        
        if (! bOK)
        	impl_logAndThrowError ("Creating new calc engine document with handle '"+sHandle+"' failed.");

        return sHandle;
    };
    
    //-------------------------------------------------------------------------
    @Override
    public void destroyDocument(final String sHandle)
        throws Exception
    {
        Validate.isTrue(CalcEngineHandleGenerator.validateHandle(sHandle), "'"+sHandle+"' is an invalid handle.");

        if ( ! impl_isHandleKnown (sHandle))
        {
        	LOG.warn(mem_Log ().enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sHandle)
        			           .forLevel    (ELogLevel.E_WARNING                                  )
        			           .toLog       ("Handle is unknown."                                 ));
        	return;
        }
        
        final CalcEngineNativeWrapper aEngine = mem_Engine ();
        final boolean                 bOK     = aEngine.destroyDocument(sHandle);
        
        if (! bOK)
        	impl_logAndThrowError ("Destroy of calc engine document with handle '"+sHandle+"' failed.");
    };

    //-------------------------------------------------------------------------
    @Override
    public ECalcEngineError executeOperation(final String       sHandle       ,
                                   			 final String       sOperationJSON,
                                   			 final StringBuffer sResultJSON   )
        throws Exception
    {
        Validate.isTrue(CalcEngineHandleGenerator.validateHandle(sHandle), "'"+sHandle+"' is an invalid handle.");

        if ( ! impl_isHandleKnown (sHandle))
        {
        	LOG.error(mem_Log ().enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sHandle)
			                    .forLevel    (ELogLevel.E_ERROR                                    )
			                    .toLog       ("document lost"                                      ));
        	return ECalcEngineError.E_DOCUMENT_LOST;
        }

        final JOperation              aOp     = new JOperation (new JSONObject(sOperationJSON));
        final CalcEngineNativeWrapper aEngine = mem_Engine ();
        final boolean                 bOK     = aEngine.executeOperation(sHandle, aOp);
        
        if (! bOK)
        	impl_logAndThrowError ("Execute operation on calc engine document '"+sHandle+"' failed.");

        final String sResult = aOp.getResult().toString();
		sResultJSON.setLength(0      );
        sResultJSON.append   (sResult);

        return ECalcEngineError.E_NONE;
    };
    
    //-------------------------------------------------------------------------
    @Override
    public ECalcEngineError restoreDocument (final String    sHandle             ,
			 				                 final String... lPagedOperationJSONs)
        throws Exception
    {
        Validate.isTrue(CalcEngineHandleGenerator.validateHandle(sHandle), "'"+sHandle+"' is an invalid handle.");

        final CalcEngineNativeWrapper aEngine = mem_Engine ();
    	      boolean                 bOK     = false;

    	// a) create the document with given handle new
      	LOG.info(mem_Log ().enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sHandle)
      					   .forLevel    (ELogLevel.E_INFO                                     )
      					   .toLog       ("restore document - create document new"             ));

      	bOK = aEngine.createDocument(sHandle);
        if (! bOK)
        	impl_logAndThrowError ("Recreating calc engine document with handle '"+sHandle+"' failed.");

        // b) put all known operations to that new document for restore ...
      	LOG.info(mem_Log ().enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sHandle)
				   		   .forLevel    (ELogLevel.E_INFO                                     )
				   		   .toLog       ("restore document - apply operations"                ));
        
        for (String sOperationJSON : lPagedOperationJSONs)
        {
        	LOG.debug(mem_Log ().enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sHandle)
                    			.forLevel    (ELogLevel.E_DEBUG                                    )
                    			.toLog       ("apply operation ["+sOperationJSON+"]"               ));

        	JOperation aOp = new JOperation (new JSONObject(sOperationJSON));
                       bOK = aEngine.executeOperation(sHandle, aOp);

			if (! bOK)
				impl_logAndThrowError ("Restore of operation set failed on document '"+sHandle+"'.");
        }
        
        return ECalcEngineError.E_NONE;
    }
    
    //-------------------------------------------------------------------------
	@Override
	public ECalcEngineError copy(final String                   sDocHandle,
								 final CalcEngineClipBoardEvent aEvent    )
		throws Exception
	{
        Validate.isTrue(CalcEngineHandleGenerator.validateHandle(sDocHandle), "'"+sDocHandle+"' is an invalid handle.");

        if ( ! impl_isHandleKnown (sDocHandle))
        {
        	LOG.error(mem_Log ().enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sDocHandle)
			                    .forLevel    (ELogLevel.E_ERROR                                       )
			                    .toLog       ("document lost"                                         ));
        	return ECalcEngineError.E_DOCUMENT_LOST;
        }

        final String           sOperationJSON = CalcEngineClipBoardEvent.toJSON4CopyEvent (aEvent);
        final StringBuffer     sResultJSON    = new StringBuffer (256);
        final ECalcEngineError eError         = executeOperation (sDocHandle, sOperationJSON, sResultJSON);

        if (eError != ECalcEngineError.E_NONE)
        {
        	LOG.error(mem_Log ().enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sDocHandle)
                    			.forLevel    (ELogLevel.E_ERROR                                       )
                    			.toLog       ("copy operation failed with error code : "+eError       ));
        	return eError;
        }
        
    	aEvent.sCopyOperationsJSON = sResultJSON.toString();
        return ECalcEngineError.E_NONE;
	}

    //-------------------------------------------------------------------------
	@Override
	public ECalcEngineError paste(final String                   sDocHandle,
								  final CalcEngineClipBoardEvent aEvent    )
		throws Exception
	{
        Validate.isTrue(CalcEngineHandleGenerator.validateHandle(sDocHandle), "'"+sDocHandle+"' is an invalid handle.");

        if ( ! impl_isHandleKnown (sDocHandle))
        {
        	LOG.error(mem_Log ().enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sDocHandle)
			                    .forLevel    (ELogLevel.E_ERROR                                       )
			                    .toLog       ("document lost"                                         ));
        	return ECalcEngineError.E_DOCUMENT_LOST;
        }

        final String           sOperationJSON = CalcEngineClipBoardEvent.toJSON4PasteEvent (aEvent);
        final StringBuffer     sResultJSON    = new StringBuffer (256);
        final ECalcEngineError eError         = executeOperation (sDocHandle, sOperationJSON, sResultJSON);
        
        if (eError != ECalcEngineError.E_NONE)
        {
        	LOG.error(mem_Log ().enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sDocHandle)
        						.forLevel    (ELogLevel.E_ERROR                                       )
        						.toLog       ("paset operation failed with error code : "+eError      ));
        	return eError;
        }

    	aEvent.sPasteOperationsJSON = sResultJSON.toString();
        return ECalcEngineError.E_NONE;
	}

    //-------------------------------------------------------------------------
    @Override
    public String getVersion() 
        throws Exception
    {
        final CalcEngineNativeWrapper aEngine  = mem_Engine ();
        final String                  sVersion = aEngine.getVersion();
        return sVersion;
    }

    //-------------------------------------------------------------------------
    @Override
    public void setLogLevel (final ELogLevel eLevel) 
    	throws Exception
    {
        final CalcEngineNativeWrapper aEngine  = mem_Engine ();
        aEngine.setLogLevel(eLevel);
    }

	//-------------------------------------------------------------------------
    private boolean impl_isHandleKnown (final String sHandle)
       throws Exception
    {
    	// TODO implement me better ... more performance - please
    	
    	final CalcEngineNativeWrapper aEngine  = mem_Engine ();
    	final List< String >          lHandles = aEngine.getHandleList ();
    	final boolean                 bIsKnown = lHandles.contains (sHandle);
    	return bIsKnown;
    }
    
    //-------------------------------------------------------------------------
    private void impl_logAndThrowError (final String sMessage)
        throws Exception
    {
    	LOG.error (sMessage);
    	throw new Exception (sMessage);
    }
    
    //-------------------------------------------------------------------------
    private CalcEngineNativeWrapper mem_Engine ()
       throws Exception
    {
        if (m_aEngine == null)
            m_aEngine = new CalcEngineNativeWrapper ();
        return m_aEngine;
    }
    
    //-------------------------------------------------------------------------
    private ContextAwareLogHelp mem_Log ()
        throws Exception
    {
    	if (m_aContextLog == null)
    		m_aContextLog = new ContextAwareLogHelp (LOG);
    	return m_aContextLog;
    }
    
    //-------------------------------------------------------------------------
    private CalcEngineNativeWrapper m_aEngine = null;

    //-------------------------------------------------------------------------
    private ContextAwareLogHelp m_aContextLog = null;
}