/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.calcengine.client;

import com.openexchange.office.tools.logging.ELogLevel;


//=============================================================================
public interface ICalcEngineClient
{
    //-------------------------------------------------------------------------
    /** create a new (empty) calc document and return an unique handle
     *  which can be used as reference to those document for further requests.
     *  
     *  @return the unique handle of that document.
     *  
     *  @throws an exception in case document could not be created successfully.
     */
    public String createDocument ()
        throws Exception;

    //-------------------------------------------------------------------------
    /** close and delete the document referenced by the given handle.
     * 
     *  @param  sDocHandle [IN]
     *          the unique handle retrieved by any createDocument()
     *          request before.
     *          
     *  Does nothing in case document dose not exists.
     *  Throws an exception in case something goes wrong. 
     */
    public void destroyDocument (final String sDocHandle)
        throws Exception;

    //-------------------------------------------------------------------------
    /** execute the given operation (formated as JSON) on the referenced document
     *  and return the result as formated JSON too.
     * 
     *  @param  sDocHandle [IN]
     *          the unique handle retrieved by any createDocument()
     *          request before.
     *  
     *  @param  sOperationJSON [IN]
     *          the (might set of) operation to be executed on the
     *          requested document formated as JSON.
     *
     *  @param  sResultJSON [OUT]
     *          the result set of (view) operations formated as JSON.
     *
     *  @return	if an error seems to be 'recoverable' it return the detected error of type {@link ECalcEngineError}
     *  		or {@link ECalcEngineError#E_NONE} on success.
     *  
     *  @throws	an exception if an internal ('unrecoverable') error occurred.
     */
    public ECalcEngineError executeOperation (final String       sDocHandle    ,
                                              final String       sOperationJSON,
                                              final StringBuffer sResultJSON   )
        throws Exception;

    //-------------------------------------------------------------------------
    /** restore the whole document from the given set of operations.
     *  Of course document handle will stay the same after this operation ...
     *  so the outside session can handle further requests and even the destroy
     *  operation without switching the context.
     * 
     *  @param	sDocHandle [IN]
     *          the unique handle retrieved by any createDocument()
     *          request before.
     *
     *	@param	lPagedOperationJSONs [IN]
     *			a list of JSON operation sets ...
     *			paged ...
     *			so restoring the document can be optimized
     *			and wont fail by 'to big data sets' .-)
     *
     *  @return	{@link ECalcEngineError#E_NONE} on success ...
     *  		different return states makes no sense at this method .-)
     *  
     *  @throws an exception in case restoring the document was not successfully
     *  		even after a count of configured retries.
     */
    public ECalcEngineError restoreDocument (final String    sDocHandle          ,
            					             final String... lPagedOperationJSONs)
        throws Exception;

    //-------------------------------------------------------------------------
    /** initiate a new clip board transaction and define the new 'copy' event.
     * 
     *  It ...
     *  
     *  <ul>
     *  <li>generate a new clip-board handle for return.<br/>
     *      The caller of this method has to use it on every following 'paste'
     *      event to refer to THIS 'copy' event.
     *  </li>
     *  <li>generate the suitable set of document operations<br/>
     *  	(retrieved from the source document specified within this request)<br/>
     *      which can be applied later to the target document on 'paste'.
     *      Those operations are cached on the server side so a following 'paste'
     *      needs the clip-board-handle only to refer to those data.
     *  </li>
     *  <li>It will remove all cached data of any 'copy' event before<br/>
     *  	(bound to the same session / user).
     *  </li>
     *  </ul>
     *  
     *  @note	Those cached clip board data will live not for ever.
     *  		It depends from the current strategy implemented and some configuration entries.
     *  		Current solution : all data will live e.g. for 10 minutes ...
     *  		those time-to-live counter will be refreshed on every new 'paste' event.
     *  
     *  The given event is designed as an IN/OUT parameter and has different functions at the same time:
     *  
     *  <ul>
     *  <li>public : {@link CalcEngineClipBoardEvent#sSourceSelectionJSON} [IN] parameter<br/>
     *  	define the current selection of the source document where the copy operation should rely on.
     *  </li>
     *  <li>public : {@link CalcEngineClipBoardEvent#sClipboardHandle} [OUT] parameter<br/>
     *  	return the new generated (unique) handle for this clip-board operation.<br/>
     *  	Has to be used on any further 'paste' event as reference to these 'copy' operation.
     *  </li>
     *  <li>private : {@link CalcEngineClipBoardEvent#sCopyOperationsJSON} [OUT] parameter<br/>
     *  	return the generated set of operations to be used on a following 'paste' event
     *  	to generate suitable view operations for.<br/>
     *  	THIS IS AN INTERNAL API - NOT THOUGHT TO BE USED PUBLIC.
     *  </li>
     *  </ul>
     *  
     *  @param	sSourceDocHandle [IN]
     *          the unique document handle retrieved by any {@link #createDocument} request before<br/>
     *          which points to the source document for this operation.
     *
     *  @param	aEvent [IN/OUT]
     *  		describe the incoming event and is used to return some values also.<br/>
     *  		see description before ...
     *  
     *  @return	if an error seems to be 'recoverable' it return the detected error of type {@link ECalcEngineError}
     *  		or {@link ECalcEngineError#E_NONE} on success.
     *  
     *  @throws	an exception if an internal ('unrecoverable') error occurred.
     */
    public ECalcEngineError copy (final String                   sSourceDocHandle,
    							  final CalcEngineClipBoardEvent aEvent          ) 
        throws Exception;

    //-------------------------------------------------------------------------
    /** a 'paste' event ask for the result of a 'copy' event before and
     *  get a set of (view related) operations.
     *  
     *  It ...
     *  
     *  <ul>
     *  <li>read the set of document operations<br/>
     *  	(cached within the corresponding 'copy' event before)<br/>
     *  	and apply it on the defined target document.<br/>
     *  </li>
     *  <li>those applied document operations will create a suitable set of view operations<br/>
     *  	to be used for updating the UI.
     *  </li>
     *  <li>this game can be go on as long as no new 'copy' event occur.<br/>
     *  	A new 'copy' event will remove the current set of data ...
     *  	but
     *  </li>
     *  </ul>
     *  
     *  @param	sTargetDocHandle [IN]
     *          the unique document handle retrieved by any {@link #createDocument} request before<br/>
     *          which points to the target document for this operation.
     *
     *  @param	aEvent [IN/OUT]
     *  		describe the incoming event and is used to return some values also.<br/>
     *  		see description before ...
     *  
     *  @return	if an error seems to be 'recoverable' it return the detected error of type {@link ECalcEngineError}
     *  		or {@link ECalcEngineError#E_NONE} on success.
     *  
     *  @throws	an exception if an internal ('unrecoverable') error occurred.
     */
    public ECalcEngineError paste (final String                   sTargetDocHandle,
			  					   final CalcEngineClipBoardEvent aEvent          ) 
        throws Exception;

    //-------------------------------------------------------------------------
    /**@return the version of the calcengine implementation.
     */
    public String getVersion() 
        throws Exception;

    //-------------------------------------------------------------------------
    /** set a new log level on that calcengine client ...
     *  which means : 
     *  - forward set log level till you find the native C++ implementation
     *  - those log level has to be recognized there
     */
    public void setLogLevel (final ELogLevel eLevel) 
        throws Exception;
}
