#!/bin/bash

function help {
    cat << EOF
${0} [OPTIONS] --release=<RELEASE VERSION> <DIR 1> <Dir 2> ...

    OPTIONS:
        --release=<RELEASE VERSION> (mandatory)
        --OX6-frontend-release=<OX6_FRONTEND_RELEASE> (optional)
        --mobile-release=<MOBILE_RELEASe> (optional)
        --auto-push=yes/no (optional)

    Examples:
        The following command creates the release-7.4.0 branch and mst common changes
        automatically. Please take a close look to the auto commited stuff afterwards.

        ${0} --release=7.4.0 --OX6-frontend-release=6.22.4 --auto-push=YES oxcs oxio

EOF
}

NEEDED_TOOLs="\
    ant
    debian-builder
    git
"

# Check for needed tools
set +e
for tool in ${NEEDED_TOOLs}; do
    which ${tool} &> /dev/null
    if [ ! '0' -eq "${?}" ]; then
        echo "${0}: Error, needed tool not found: ${tool}" >&2
        exit 1
    fi
done
set -e

for argument in $@; do
    case ${argument} in
        --release=*)
            RELEASE=${argument##--release=}
        ;;
        --OX6-frontend-release=*)
            OX6_FRONTEND_RELEASE=${argument##--OX6-frontend-release=}
        ;;
        --mobile-release=*)
            MOBILE_RELEASE=${argument##--mobile-release=}
        ;;
        --auto-push=*)
            AUTO_PUSH=$(echo ${argument##--auto-push=} | tr [A-Z] [a-z])
            echo $AUTO_PUSH
        ;;
        --*|-*)
            help
        ;;
        *)
            if [ -d "${argument}" ]; then
                CUSTOMER_LIST="${CUSTOMER_LIST} ${argument}"
            else
                echo "Directory not found: ${argument}"
                help
                exit 1
            fi
    esac
done

if [ -z "${RELEASE}" ]; then
    echo "Error, parameter '--release' is not defined"
    help
    exit 1
fi

SNAPSHOT_IDENTIFIER=dependingProjectNames
SNAPSHOT_DEPENDENCIES="${SNAPSHOT_IDENTIFIER}=backend-release-${RELEASE},frontend-release-${RELEASE}"
###############################################################################
for customer in ${CUSTOMER_LIST}; do

    BACKEND_APPSUITE_PSF_FILE=${customer}-packages/${customer}-packages-appsuite-and-backend.psf
    OX6_PSF_FILE=${customer}-packages/${customer}-packages-OX6-frontend.psf
    MOBILE_PSF_FILE=${customer}-packages/${customer}-packages-mobile-web-interface.list

    if [ ! -f "${BACKEND_APPSUITE_PSF_FILE}" -a -n "${OX6_FRONTEND_RELEASE}" ]; then
        BRANCH=release-${OX6_FRONTEND_RELEASE}
    else
        BRANCH=release-${RELEASE}
    fi

    echo "Prepare customer ${customer} repo for ${BRANCH}"

    cd ${customer}
    echo "Update local copy of repository"
    git remote update

    # Check if release branch already exsist and skip
    if [ -n "$(git branch -a | grep remotes/origin/${BRANCH})" ]; then
        echo "Existing ${BRANCH} branch on remote repsository detected"
        echo "Skipping next steps for customer ${customer}"
    else
        # Drop uncommited changes
        git checkout .
        git checkout develop
        git reset --hard remotes/origin/develop
        git checkout -b ${BRANCH}

        # Update build tools
        echo "Update build tools"
        git submodule init
        git submodule update
        cd build
        git checkout master
        git pull
        cd ..

        # Commit build tools update
        if [ -n "$(git status | grep build)" ]; then
            git add build
            git commit -m "Update build tools to latest version"
        else
            echo "Build tools are already up-to-date"
        fi

        # Update snaptshot repositories
        sed -i "/${SNAPSHOT_IDENTIFIER}/d" ${customer}-packages/buildservice.properties
        echo ${SNAPSHOT_DEPENDENCIES} >> ${customer}-packages/buildservice.properties

        # Commit updated snapshot build settings
        git add ${customer}-packages/buildservice.properties
        git commit -m "Update snapshot build settings to ${RELEASE} build dependencies"

        # Prepare backend and appsuite custom plugins for new release
        if [ -f "${BACKEND_APPSUITE_PSF_FILE}" ]; then
            echo "Updating backend and appsuite custom plugins for new release"
            cd build
            ant -lib lib \
                      -f releaseNumbers.xml \
                      -DversionNumber=${RELEASE} \
                      "-DreleaseNotes=Prepare backend and appsuite custom plugins for ${RELEASE} release" \
                      -DprojectSets=${customer}-packages \
                      '-DpsfPattern=*${customer}-packages-appsuite-and-backend' \
                      loadProjectSets \
                      newVersion \
                      ;
            # Commit updated backend and appsuite custom plugins plugin changes
            cd ..
            git add .
            git commit -m "Prepare backend and appsuite custom plugins for ${RELEASE} release"
        else
            echo "Skipping, no backend and appsuite custom plugins found"
        fi

        # Prepare for OX6 frontend custom plugins for new release
        if [ -f "${OX6_PSF_FILE}" -a -n "${OX6_FRONTEND_RELEASE}" ]; then
            echo "Updating OX6 frontend custom plugins for new release"
            cd build
            ant -lib lib \
                      -f releaseNumbers.xml \
                      -DversionNumber=${OX6_FRONTEND_RELEASE} \
                      "-DreleaseNotes=Prepare OX6 frontend custom plugins for ${OX6_FRONTEND_RELEASE} release" \
                      -DprojectSets=${customer}-packages \
                      '-DpsfPattern=*${customer}-packages-OX6-frontend' \
                      loadProjectSets \
                      newVersion \
                      ;
            # Commit updated OX6 frontend custom plugins changes
            cd ..
            git add .
            git commit -m "Prepare OX6 frontend custom plugins for ${OX6_FRONTEND_RELEASE} release"
        else
            echo "Skipping, no OX6 frontend custom plugins found"
        fi

        # Prepare for mobile UI custom plugins for new release
        if [ -f "${MOBILE_PSF_FILE}" -a -n "${MOBILE_RELEASE}" ]; then
            echo "Updating mobile UI custom plugins for new release"
            cd build
            ant -lib lib \
                      -f releaseNumbers.xml \
                      -DversionNumber=${MOBILE_RELEASE} \
                      "-DreleaseNotes=Prepare mobile UI custom plugins for ${MOBILE_RELEASE} release" \
                      -DprojectSets=${customer}-packages \
                      '-DpsfPattern=*${customer}-packages-mobile-web-interface' \
                      loadProjectSets \
                      newVersion \
                      ;
            # Commit updated mobile UI custom plugin changes
            cd ..
            git add .
            git commit -m "Prepare mobile UI custom plugins for ${MOBILE_RELEASE} release"
        fi
        if [ 'yes' == "${AUTO_PUSH}" ]; then
            echo "Pushing ${customer} changes to remote ${BRANCH} branch"
            git push origin ${BRANCH}:${BRANCH}
            # Remove and recreate to point for pull on the correct branch
            git checkout develop
            git branch -D ${BRANCH}
            git checkout -b ${BRANCH} remotes/origin/${BRANCH}
        fi
    fi
    cd ..
done
