/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.drawingml;

import org.docx4j.dml.CTBlip;
import org.docx4j.dml.CTBlipFillProperties;
import org.docx4j.dml.CTNonVisualDrawingProps;
import org.docx4j.dml.CTPoint2D;
import org.docx4j.dml.CTPositiveSize2D;
import org.docx4j.dml.CTPresetGeometry2D;
import org.docx4j.dml.CTRelativeRect;
import org.docx4j.dml.CTShapeProperties;
import org.docx4j.dml.CTTransform2D;
import org.docx4j.dml.STShapeType;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.relationships.Relationship;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.office.ooxml.OperationDocument;
import com.openexchange.office.ooxml.tools.Commons;

final public class DMLHelper {

    public static void createJsonFromPositiveSize2D(JSONObject drawingProperties, CTPositiveSize2D positiveSize2D)
        throws JSONException {

        if(positiveSize2D!=null) {
            drawingProperties.put("width", Commons.coordinateTo100TH_MM(positiveSize2D.getCx()));
            drawingProperties.put("height", Commons.coordinateTo100TH_MM(positiveSize2D.getCy()));
        }
    }

    public static void createJsonFromBlipFillProperties(JSONObject drawingProperties, CTBlipFillProperties blipFillProperties)
        throws JSONException {

        if(blipFillProperties!=null) {
            final CTRelativeRect srcRect = blipFillProperties.getSrcRect();
            if (srcRect!=null) {
                int l = srcRect.getL() / 1000;
                int t = srcRect.getT() / 1000;
                int r = srcRect.getR() / 1000;
                int b = srcRect.getB() / 1000;

                // add possible crop properties
                if (l != 0)
                    drawingProperties.put("cropLeft", l);
                if (t != 0)
                    drawingProperties.put("cropTop", t);
                if (r != 0)
                    drawingProperties.put("cropRight", r);
                if (b != 0)
                    drawingProperties.put("cropBottom", b);
            }
        }
    }

    public static void createJsonFromShapeProperties(JSONObject drawingProperties, CTShapeProperties shapeProperties)
        throws JSONException {

        if(shapeProperties!=null) {
            final CTTransform2D xFrm = shapeProperties.getXfrm();
            if(xFrm!=null) {
                CTPoint2D point2D = xFrm.getOff();
                if(point2D!=null) {
                    if(point2D.getX()!=0) {
                        if(!drawingProperties.hasAndNotNull("left")) {
                            drawingProperties.put("left", Commons.coordinateTo100TH_MM(point2D.getX()));
                        }
                    }
                    if(point2D.getY()!=0) {
                        if(!drawingProperties.hasAndNotNull("top")) {
                            drawingProperties.put("top", Commons.coordinateTo100TH_MM(point2D.getX()));
                        }
                    }
                }
                CTPositiveSize2D positiveSize2D = xFrm.getExt();
                if(positiveSize2D!=null) {
                    if(!drawingProperties.hasAndNotNull("width")) {
                        drawingProperties.put("width", Commons.coordinateTo100TH_MM(positiveSize2D.getCx()));
                    }
                    if(!drawingProperties.hasAndNotNull("height")) {
                        drawingProperties.put("height", Commons.coordinateTo100TH_MM(positiveSize2D.getCy()));
                    }
                }
                if(xFrm.isFlipH()) {
                    if(!drawingProperties.hasAndNotNull("flipH")) {
                        drawingProperties.put("flipH", true );
                    }
                }
                if(xFrm.isFlipV()) {
                    if(!drawingProperties.hasAndNotNull("flipV")) {
                        drawingProperties.put("flipV", true );
                    }
                }
            }
        }
    }

    public static void createJsonFromNonVisualDrawingProperties(JSONObject drawingProperties, CTNonVisualDrawingProps nonVisualDrawingProps)
        throws JSONException {

        final String name = nonVisualDrawingProps.getName();
        if(name!=null&&!name.isEmpty()) {
            drawingProperties.put("name", name);
        }
        final String description = nonVisualDrawingProps.getDescr();
        if(description!=null&&!description.isEmpty()) {
            drawingProperties.put("description", description);
        }
    }

    public static void applyNonVisualDrawingProperties(JSONObject drawingProperties, CTNonVisualDrawingProps nonVisualDrawingProps) {

        if(drawingProperties.has("name")) {
            final Object optName = drawingProperties.opt("name");
            if(optName instanceof String) {
                nonVisualDrawingProps.setName((String)optName);
            }
            else {
                // name is a required xml attribute... so we are setting a dummy
                nonVisualDrawingProps.setName("...");
            }
        }
        if(drawingProperties.has("description")) {
            final Object optDescription = drawingProperties.opt("description");
            if(optDescription instanceof String) {
                nonVisualDrawingProps.setDescr((String)optDescription);
            }
            else {
                nonVisualDrawingProps.setDescr(null);
            }
        }
    }

    public static CTTransform2D createTransform2D(int x100thmm, int y100thmm, int width100thmm, int height100thmm) {
        final org.docx4j.dml.ObjectFactory dmlObjectFactory = Context.getDmlObjectFactory();
        final CTTransform2D transform2D = dmlObjectFactory.createCTTransform2D();
        final CTPoint2D point2D = dmlObjectFactory.createCTPoint2D();
        point2D.setX(Commons.coordinateFrom100TH_MM(x100thmm));
        point2D.setY(Commons.coordinateFrom100TH_MM(y100thmm));
        transform2D.setOff(point2D);
        final CTPositiveSize2D positiveSize2D = dmlObjectFactory.createCTPositiveSize2D();
        positiveSize2D.setCx(Commons.coordinateFrom100TH_MM(width100thmm));
        positiveSize2D.setCy(Commons.coordinateFrom100TH_MM(height100thmm));
        return transform2D;
    }

    public static CTPresetGeometry2D createPresetGeometry2D(STShapeType presetType) {
        CTPresetGeometry2D presetGeometry2D = Context.getDmlObjectFactory().createCTPresetGeometry2D();
        presetGeometry2D.setPrst(presetType);
        return presetGeometry2D;
    }

    public static CTBlipFillProperties createBlipFillProperties(OperationDocument operationDocument, Part part, String imageUrl)
        throws Exception {

        final org.docx4j.dml.ObjectFactory dmlObjectFactory = Context.getDmlObjectFactory();
        final CTBlipFillProperties blipFillProperties = dmlObjectFactory.createCTBlipFillProperties();
        final CTBlip blip = dmlObjectFactory.createCTBlip();
        blipFillProperties.setBlip(blip);
        blipFillProperties.setStretch(dmlObjectFactory.createCTStretchInfoProperties());
        final Relationship relationship = Commons.createAndApplyGraphic(operationDocument, part, imageUrl);
        if(relationship!=null) {
            final boolean linked = relationship.getTargetMode() != null && relationship.getTargetMode().equals("External");
            if(linked) {
                blip.setLink(relationship.getId());
            }
            else {
                blip.setEmbed(relationship.getId());
            }
        }
        return blipFillProperties;
    }

    public static CTNonVisualDrawingProps createNonVisualDrawingProps(String name) {
        CTNonVisualDrawingProps nonVisualDrawingProps = Context.getDmlObjectFactory().createCTNonVisualDrawingProps();
        nonVisualDrawingProps.setName(name);
        return nonVisualDrawingProps;
    }

    /**
     * @param part (partName Uri should be /word/document.xml)
     * @return the imageUrl in following form: word/media/xxxx.png
     */
    public static String getBlipUrl(Part part, CTBlip blip) {
        String imageUrl = "";
        if(blip!=null) {
            imageUrl = Commons.getUrl(part, blip.getEmbed().length() > 0 ? blip.getEmbed() : blip.getLink() );
        }
        return imageUrl;
    }
}
