/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.tools;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.xlsx4j.sml.CTMergeCell;
import org.xlsx4j.sml.CTMergeCells;
import org.xlsx4j.sml.Cell;
import org.xlsx4j.sml.Cell.CellRef;
import org.xlsx4j.sml.Cell.CellRefRange;


public class MergeCellHelper {

    private static class MergeCellWithDirectRefAccess implements Comparable<MergeCellWithDirectRefAccess> {

        MergeCellWithDirectRefAccess(CTMergeCell mergeCell) {
            final CellRefRange cellRefRange = Cell.createCellRefRange(mergeCell.getRef());
            topRow = cellRefRange.getStart().getRow();
            leftColumn = cellRefRange.getStart().getColumn();
            bottomRow = cellRefRange.getEnd().getRow();
            rightColumn = cellRefRange.getEnd().getColumn();
            alive = true;
        }

        int getTopRow() {
            return topRow;
        }
        int getLeftColumn() {
            return leftColumn;
        }
        int getBottomRow() {
            return bottomRow;
        }
        int getRightColumn() {
            return rightColumn;
        }
        CellRef getStart() {
            return new CellRef(leftColumn, topRow);
        }
        CellRef getEnd() {
            return new CellRef(rightColumn, bottomRow);
        }
        boolean isAlive() {
            return alive;
        }
        void kill() {
            alive = false;
        }
        @Override
        public int compareTo(MergeCellWithDirectRefAccess mergeCellWithDirectRefAccess) {
            final int diff = topRow-mergeCellWithDirectRefAccess.getTopRow();
            if (diff!=0)
                return diff;        // the rows are unequal... that is sufficient... otherwise if equal, we have to take care of the column position (next line)
            return leftColumn-mergeCellWithDirectRefAccess.getLeftColumn();
        }
        private final int topRow;
        private final int leftColumn;
        private final int bottomRow;
        private final int rightColumn;
        private boolean alive;
    }

    public static void createMergeCellOperations(JSONArray operationsArray, int sheetIndex, CTMergeCells mergeCells)
        throws JSONException {

        if(mergeCells==null)
            return;

        final List<CTMergeCell> mergeCellList = mergeCells.getMergeCell();
        if(mergeCellList==null||mergeCellList.isEmpty())
            return;

        // we create a sorted list to be able to properly iterate from top left to bottom right (we imply that there are no intersections between mergeCellRanges and topLeft is our Key)
        final List<MergeCellWithDirectRefAccess> mergeCellSortedList = new ArrayList<MergeCellWithDirectRefAccess>(mergeCellList.size());
        // next we create a hashMap to gain direct access to each topLeft mergeCellRange
        final HashMap<Integer, HashMap<Integer, MergeCellWithDirectRefAccess>> rowMap = new HashMap<Integer, HashMap<Integer, MergeCellWithDirectRefAccess>>();

        for(CTMergeCell mergeCell:mergeCellList) {
            final MergeCellWithDirectRefAccess mergeCellSorted = new MergeCellWithDirectRefAccess(mergeCell);
            mergeCellSortedList.add(mergeCellSorted);

            final int r = mergeCellSorted.getTopRow();
            final int c = mergeCellSorted.getLeftColumn();
            HashMap<Integer, MergeCellWithDirectRefAccess> columnMap = rowMap.get(r);
            if(columnMap==null) {
                columnMap = new HashMap<Integer, MergeCellWithDirectRefAccess>();
                rowMap.put(r,  columnMap);
            }
            columnMap.put(c, mergeCellSorted);
        }

        Collections.sort(mergeCellSortedList);

        // the sortedList and HashMap above is now initialized ...
        for(MergeCellWithDirectRefAccess mergeCellSorted:mergeCellSortedList) {

            // now iterating from top left to bottom right
            if(mergeCellSorted.isAlive()) { // maybe this cellRange has already been taken for an vertical/horizontal merge
                final CellRef start = mergeCellSorted.getStart();
                CellRef end = mergeCellSorted.getEnd();
                String mergeType = null;                                            // default for simple merge

                // first we will check for horizontal merges....
                if(mergeCellSorted.getTopRow()==mergeCellSorted.getBottomRow()) {                           // horizontal merge requires that top and bottom row is equal
                    int count = 1;
                    int r = mergeCellSorted.getTopRow();
                    while(true) {
                        final HashMap<Integer, MergeCellWithDirectRefAccess> columnMap = rowMap.get(r+1);
                        if(columnMap==null) {
                            break;
                        }
                        final MergeCellWithDirectRefAccess compare = columnMap.get(mergeCellSorted.getLeftColumn());
                        if(compare==null||!compare.isAlive()) {
                            break;
                        }
                        if(compare.getRightColumn()!=mergeCellSorted.getRightColumn()) {
                            break;
                        }
                        if(compare.getTopRow()!=compare.getBottomRow()) {
                            break;
                        }
                        end = compare.getEnd();
                        compare.kill();
                        count++;
                        r++;
                    }
                    if (count>1) {
                        mergeType = "horizontal";
                    }
                }
                // check for vertical merges... (only if not already a horizontal merge was found)
                if(mergeType==null&&mergeCellSorted.getLeftColumn()==mergeCellSorted.getRightColumn()) {    // vertical merge requires that left and right column is equal
                    int count = 1;
                    int c = mergeCellSorted.getLeftColumn();
                    while(true) {
                        final HashMap<Integer, MergeCellWithDirectRefAccess> columnMap = rowMap.get(mergeCellSorted.getTopRow());
                        // columnMap can't be zero as we are part of it
                        final MergeCellWithDirectRefAccess compare = columnMap.get(c+1);
                        if(compare==null||!compare.isAlive()) {
                            break;
                        }
                        if(compare.getBottomRow()!=mergeCellSorted.getBottomRow()) {
                            break;
                        }
                        if(compare.getLeftColumn()!=compare.getRightColumn()) {
                            break;
                        }
                        end = compare.getEnd();
                        compare.kill();
                        count++;
                        c++;
                    }
                    if (count>1) {
                        mergeType = "vertical";
                    }
                }
                addMergeCells(operationsArray, sheetIndex, mergeType, start, end);
            }
        }
    }

    public static void addMergeCells(JSONArray operationsArray, int sheetIndex, String mergeType, CellRef start, CellRef end)
        throws JSONException {

        final JSONObject addMergeCellsObject = new JSONObject();
        addMergeCellsObject.put("name", "mergeCells");
        addMergeCellsObject.put("sheet", sheetIndex);
        addMergeCellsObject.put("start", start.getJSONArray());
        addMergeCellsObject.put("end", end.getJSONArray());
        addMergeCellsObject.putOpt("type", mergeType);
        operationsArray.put(addMergeCellsObject);
    }
}
