/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx;

import java.io.InputStream;
import java.math.BigInteger;
import java.util.List;
import org.docx4j.dml.Theme;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.Docx4JException;
import org.docx4j.openpackaging.packages.WordprocessingMLPackage;
import org.docx4j.openpackaging.parts.PartName;
import org.docx4j.openpackaging.parts.ThemePart;
import org.docx4j.openpackaging.parts.WordprocessingML.FontTablePart;
import org.docx4j.openpackaging.parts.WordprocessingML.MainDocumentPart;
import org.docx4j.openpackaging.parts.WordprocessingML.StyleDefinitionsPart;
import org.docx4j.wml.CTLanguage;
import org.docx4j.wml.DocDefaults;
import org.docx4j.wml.DocDefaults.RPrDefault;
import org.docx4j.wml.Fonts;
import org.docx4j.wml.Fonts.Font;
import org.docx4j.wml.ObjectFactory;
import org.docx4j.wml.RPr;
import org.docx4j.wml.Styles;
import org.docx4j.wml.Styles.LatentStyles;
import org.docx4j.wml.Styles.LatentStyles.LsdException;
import org.json.JSONArray;
import org.json.JSONObject;
import org.json.JSONTokener;
import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.FilterException.ErrorCode;
import com.openexchange.office.ooxml.docx.operations.ApplyOperationHelper;
import com.openexchange.office.ooxml.docx.operations.CreateOperationHelper;
import com.openexchange.office.tools.ResourceManager;
import com.openexchange.server.ServiceLookup;

public class OperationDocument extends com.openexchange.office.ooxml.OperationDocument {

    final private WordprocessingMLPackage opcPackage;

    public OperationDocument(ServiceLookup _services, InputStream _inputDocumentStream, ResourceManager _resourceManager, DocumentProperties _documentProperties)
        throws FilterException {

        super(_services, _inputDocumentStream, _resourceManager, _documentProperties);
        try {
            opcPackage = WordprocessingMLPackage.load(_inputDocumentStream);
        }
        catch(Docx4JException e) {
            ErrorCode errorCode = ErrorCode.CRITICAL_ERROR;
            if(e.getCause() instanceof java.security.InvalidKeyException) {
                errorCode = ErrorCode.WRONG_PASSWORD;
            }
            else if(e.getCause() instanceof org.apache.poi.EncryptedDocumentException) {
                errorCode = ErrorCode.UNSUPPORTED_ENCRYPTION_USED;
            }
            throw new FilterException(e, errorCode);
        }
    }

    // the constructor without inputStream creates an empty document
    public OperationDocument(ServiceLookup _services, DocumentProperties _documentProperties)
        throws FilterException {

        super(_services, null, null, _documentProperties);
        try {
            opcPackage = WordprocessingMLPackage.createPackage();

            MainDocumentPart mainDocumentPart = getPackage().getMainDocumentPart();
            Styles styles = mainDocumentPart.getStyleDefinitionsPart().getJaxbElement();
            resetDocumentLanguage(styles);
            insertPredefinedLatentStyles(styles);
            ThemePart themePart = getDefaultThemePart(new PartName("/word/theme/theme1.xml"));
            mainDocumentPart.addTargetPart(themePart);

            // creating a fontlist (QuickOffice won't load documents without this part)
            getFontList(true);
        }
        catch(Exception e) {
            throw new FilterException(e, ErrorCode.CRITICAL_ERROR);
        }
    }

    @Override
    public void applyOperations(String applyOperations) throws FilterException {

        if (applyOperations != null){
            int i = 0;
            JSONObject op = null;
            try {
                ApplyOperationHelper applyOperationHelper = new ApplyOperationHelper(this);
                final JSONArray aOperations = new JSONArray(new JSONTokener(applyOperations));
                for (i = 0; i < aOperations.length(); i++) {
                    op = (JSONObject) aOperations.get(i);
                    String opName = op.getString("name");
                    if (opName.equals("insertParagraph"))
                        applyOperationHelper.insertParagraph(op.getJSONArray("start"), op.optJSONObject("attrs"));
                    else if (opName.equals("delete"))
                        applyOperationHelper.delete(op.getJSONArray("start"), op.optJSONArray("end"));
                    else if (opName.equals("move"))
                        applyOperationHelper.move(op.getJSONArray("start"), op.optJSONArray("end"), op.getJSONArray("to"));
                    else if (opName.equals("splitParagraph"))
                        applyOperationHelper.splitParagraph(op.getJSONArray("start"));
                    else if (opName.equals("mergeParagraph"))
                        applyOperationHelper.mergeParagraph(op.getJSONArray("start"));
                    else if (opName.equals("insertText"))
                        applyOperationHelper.insertText(op.getJSONArray("start"), op.getString("text").replaceAll("\\p{C}", " "), op.optJSONObject("attrs"));
                    else if (opName.equals("insertTab"))
                        applyOperationHelper.insertTab(op.getJSONArray("start"), op.optJSONObject("attrs"));
                    else if (opName.equals("insertHardBreak"))
                        applyOperationHelper.insertHardBreak(op.getJSONArray("start"), op.optJSONObject("attrs"));
                    else if (opName.equals("insertTable"))
                        applyOperationHelper.insertTable(op.getJSONArray("start"), op.optJSONObject("attrs"));
                    else if (opName.equals("insertRows"))
                        applyOperationHelper.insertRows(op.getJSONArray("start"), op.optInt("count", 1), op.optBoolean("insertDefaultCells", false), op.optInt("referenceRow", -1), op.optJSONObject("attrs"));
                    else if (opName.equals("insertCells"))
                        applyOperationHelper.insertCells(op.getJSONArray("start"), op.optInt("count", 1), op.optJSONObject("attrs"));
                    else if (opName.equals("insertColumn"))
                        applyOperationHelper.insertColumn(op.getJSONArray("start"), op.getJSONArray("tableGrid"), op.getInt("gridPosition"), op.optString("insertMode", "before"));
                    else if (opName.equals("deleteColumns"))
                        applyOperationHelper.deleteColumns(op.getJSONArray("start"), op.getInt("startGrid"), op.optInt("endGrid", op.getInt("startGrid")));
                    else if (opName.equals("setAttributes"))
                        applyOperationHelper.setAttributes(op.getJSONObject("attrs"), op.getJSONArray("start"), op.optJSONArray("end"));
                    else if (opName.equals("insertField"))
                        applyOperationHelper.insertField(op.getJSONArray("start"), op.getString("type"), op.getString("representation"), op.optJSONObject("attrs"));
                    else if (opName.equals("insertDrawing"))
                        applyOperationHelper.insertDrawing(op.getJSONArray("start"), op.getString("type").toUpperCase(), op.optJSONObject("attrs"));
                    else if (opName.equals("insertStyleSheet"))
                        applyOperationHelper.insertStyleSheet(op.getString("type"), op.getString("styleId"), op.getString("styleName"), op.getJSONObject("attrs"), op.optString("parent"), op.optBoolean("hidden"), op.getInt("uiPriority"), op.optBoolean("default"));
                    else if (opName.equals("deleteStylesheet"))
                        applyOperationHelper.deleteStylesheet(op.getString("type"), op.getString("styleId"));
                    else if (opName.equals("insertListStyle"))
                        applyOperationHelper.insertListStyle(op.getString("listStyleId"), op.getJSONObject("listDefinition"));
                }

                // creating fontListPart ...for QuickOffice
                getFontList(true);

                // set user language if not already set in the document
                applyOperationHelper.setDocumentLanguageIfNotSet(getUserLanguage());
                applyOperationHelper.postFixNonConformance();
                applyOperationHelper.removePreviewThumbnail();
            }
            catch(Exception e) {
                String message = e.getMessage();
                if(op!=null) {
                    message += ", operation:" + Integer.toString(i) + " " + op.toString();
                }
                throw new FilterException(message, e, ErrorCode.CRITICAL_ERROR);
            }
        }
    }

    @Override
    public JSONObject getOperations()
        throws Exception {

        final JSONObject aOperations = new JSONObject();
        final JSONArray operationsArray = new JSONArray();
        CreateOperationHelper createOperationHelper = new CreateOperationHelper(this, operationsArray);
        createOperationHelper.createDocumentDefaults(getUserLanguage());
        createOperationHelper.CreateFontDescriptions();
        createOperationHelper.CreateStyleOperations();
        createOperationHelper.CreateThemeOperations();
        createOperationHelper.CreateListOperations();
        createOperationHelper.createOperations();
        aOperations.put("operations", operationsArray);

        return aOperations;
    }

    @Override
    public WordprocessingMLPackage getPackage() {
        return opcPackage;
    }

    @Override
    public Theme getTheme() {
        ThemePart themePart = getPackage().getMainDocumentPart().getThemePart();
        if( themePart == null )
            return null;
        return themePart.getJaxbElement();
    }

    public Styles getStyles(boolean createIfMissing)
        throws Exception {

        StyleDefinitionsPart styleDefinitionsPart = getPackage().getMainDocumentPart().getStyleDefinitionsPart();
        if(styleDefinitionsPart==null&&createIfMissing) {
            styleDefinitionsPart = new StyleDefinitionsPart(new PartName("/word/styles.xml"));
            styleDefinitionsPart.setJaxbElement(Context.getWmlObjectFactory().createStyles());
            getPackage().getMainDocumentPart().addTargetPart(styleDefinitionsPart);
        }
        return styleDefinitionsPart!=null?styleDefinitionsPart.getJaxbElement() : null;
    }

    public List<Font> getFontList(boolean createIfMissing)
        throws Exception {

        FontTablePart fontTablePart = getPackage().getMainDocumentPart().getFontTablePart();
        if(fontTablePart==null&&createIfMissing) {
            fontTablePart = new FontTablePart(new PartName("/word/fontTable.xml"));
            fontTablePart.setJaxbElement(Context.getWmlObjectFactory().createFonts());
            getPackage().getMainDocumentPart().addTargetPart(fontTablePart);
        }
        if(fontTablePart!=null) {
            final Fonts fonts = fontTablePart.getJaxbElement();
            if(fonts!=null) {
                final List<Font> fontList = fonts.getFont();
                if(createIfMissing) {
                    int i;
                    // we put at least one dummy font into this list
                    for(i=0;i<fontList.size();i++) {
                        final String fontName = fontList.get(i).getName();
                        if(fontName!=null&&fontName.equals("Times New Roman")) {
                            break;
                        }
                    }
                    if(i==fontList.size()) {
                        final Font timesNewRoman = Context.getWmlObjectFactory().createFontsFont();
                        timesNewRoman.setName("Times New Roman");
                        fontList.add(timesNewRoman);
                    }
                }
            }
        }
        return null;
    }


    /**
     * Reset document language.
     *
     * @param styles {Styles}
     *  The WordML styles with or without language settings
     */
    private void resetDocumentLanguage(Styles styles) {

        if (styles != null) {
            final ObjectFactory objectFactory = new ObjectFactory();
            DocDefaults docDefaults = styles.getDocDefaults();
            if(docDefaults==null) {
                docDefaults = objectFactory.createDocDefaults();
                docDefaults.setParent(styles);
            }
            RPrDefault rDefault = docDefaults.getRPrDefault();
            if(rDefault==null) {
                rDefault = objectFactory.createDocDefaultsRPrDefault();
                rDefault.setParent(docDefaults);
            }
            RPr rPrDefault = rDefault.getRPr();
            if(rPrDefault==null) {
                rPrDefault = objectFactory.createRPr();
                rPrDefault.setParent(rDefault);
            }
            CTLanguage ctLanguage = rPrDefault.getLang();
            if(ctLanguage==null) {
                ctLanguage = objectFactory.createCTLanguage();
                ctLanguage.setParent(rPrDefault);
            }
            // reset language
            ctLanguage.setVal("");
            ctLanguage.setEastAsia("");
            ctLanguage.setBidi("");
        }
    }

    private void insertPredefinedLatentStyles(Styles styles) {

        final ObjectFactory objectFactory = new ObjectFactory();
        // name, semiHidden, uiPriority, unhideWhenUsed, qFormat
        final String[] predefinedLatentStyles = {
            "Normal, 0, 0, 0, 1",
            "heading 1, 0, 9, 0, 1",
            "heading 2,, 9,, 1",
            "heading 3,, 9,, 1",
            "heading 4,, 9,, 1",
            "heading 5,, 9,, 1",
            "heading 6,, 9,, 1",
            "heading 7,, 9,, 1",
            "heading 8,, 9,, 1",
            "heading 9,, 9,, 1",
            "toc 1,, 39",
            "toc 2,, 39",
            "toc 3,, 39",
            "toc 4,, 39",
            "toc 5,, 39",
            "toc 6,, 39",
            "toc 7,, 39",
            "toc 8,, 39",
            "toc 9,, 39",
            "caption,, 35,, 1",
            "Title, 0, 10, 0, 1",
            "Default Paragraph Font, , 1",
            "Subtitle, 0, 11, 0, 1",
            "Strong, 0, 22,, 0, 1",
            "Emphasis, 0, 20,, 0, 1",
            "Table Grid, 0, 59, 0",
            "Placeholder Text,,,0",
            "No Spacing, 0, 1, 0, 1",
            "Light Shading, 0, 60,0",
            "Light List, 0 ,61, 0",
            "Light Grid, 0 ,62, 0",
            "Medium Shading 1, 0 ,63, 0",
            "Medium Shading 2, 0 ,64, 0",
            "Medium List 1, 0 ,65, 0",
            "Medium List 2, 0 ,66, 0",
            "Medium Grid 1, 0 ,67, 0",
            "Medium Grid 2, 0 ,68, 0",
            "Medium Grid 3, 0 ,69, 0",
            "Dark List, 0 ,70, 0",
            "Colorful Shading, 0 ,71, 0",
            "Colorful List, 0 ,72, 0",
            "Colorful Grid, 0 ,73, 0",
            "Light Shading Accent 1, 0 ,60, 0",
            "Light List Accent 1, 0 ,61, 0",
            "Light Grid Accent 1, 0 ,62, 0",
            "Medium Shading 1 Accent 1, 0 ,63, 0",
            "Medium Shading 2 Accent 1, 0 ,64, 0",
            "Medium List 1 Accent 1, 0 ,65, 0",
            "Revision, 0",
            "List Paragraph, 0 ,34, 0, 1",
            "Quote, 0 ,29, 0, 1",
            "Intense Quote, 0 ,30, 0, 1",
            "Medium List 2 Accent 1, 0 ,66, 0",
            "Medium Grid 1 Accent 1, 0 ,67, 0",
            "Medium Grid 2 Accent 1, 0 ,68, 0",
            "Medium Grid 3 Accent 1, 0 ,69, 0",
            "Dark List Accent 1, 0 ,70, 0",
            "Colorful Shading Accent 1, 0 ,71, 0",
            "Colorful List Accent 1, 0 ,72, 0",
            "Colorful Grid Accent 1, 0 ,73, 0",
            "Light Shading Accent 2, 0 ,60, 0",
            "Light List Accent 2, 0 ,61, 0",
            "Light Grid Accent 2, 0 ,62, 0",
            "Medium Shading 1 Accent 2, 0 ,63, 0",
            "Medium Shading 2 Accent 2, 0 ,64, 0",
            "Medium List 1 Accent 2, 0 ,65, 0",
            "Medium List 2 Accent 2, 0 ,66, 0",
            "Medium Grid 1 Accent 2, 0 ,67, 0",
            "Medium Grid 2 Accent 2, 0 ,68, 0",
            "Medium Grid 3 Accent 2, 0 ,69, 0",
            "Dark List Accent 2, 0 ,70, 0",
            "Colorful Shading Accent 2, 0 ,71, 0",
            "Colorful List Accent 2, 0 ,72, 0",
            "Colorful Grid Accent 2, 0 ,73, 0",
            "Light Shading Accent 3, 0 ,60, 0",
            "Light List Accent 3, 0 ,61, 0",
            "Light Grid Accent 3, 0 ,62, 0",
            "Medium Shading 1 Accent 3, 0 ,63, 0",
            "Medium Shading 2 Accent 3, 0 ,64, 0",
            "Medium List 1 Accent 3, 0 ,65, 0",
            "Medium List 2 Accent 3, 0 ,66, 0",
            "Medium Grid 1 Accent 3, 0 ,67, 0",
            "Medium Grid 2 Accent 3, 0 ,68, 0",
            "Medium Grid 3 Accent 3, 0 ,69, 0",
            "Dark List Accent 3, 0 ,70, 0",
            "Colorful Shading Accent 3, 0 ,71, 0",
            "Colorful List Accent 3, 0 ,72, 0",
            "Colorful Grid Accent 3, 0 ,73, 0",
            "Light Shading Accent 4, 0 ,60, 0",
            "Light List Accent 4, 0 ,61, 0",
            "Light Grid Accent 4, 0 ,62, 0",
            "Medium Shading 1 Accent 4, 0 ,63, 0",
            "Medium Shading 2 Accent 4, 0 ,64, 0",
            "Medium List 1 Accent 4, 0 ,65, 0",
            "Medium List 2 Accent 4, 0 ,66, 0",
            "Medium Grid 1 Accent 4, 0 ,67, 0",
            "Medium Grid 2 Accent 4, 0 ,68, 0",
            "Medium Grid 3 Accent 4, 0 ,69, 0",
            "Dark List Accent 4, 0 ,70, 0",
            "Colorful Shading Accent 4, 0 ,71, 0",
            "Colorful List Accent 4, 0 ,72, 0",
            "Colorful Grid Accent 4, 0 ,73, 0",
            "Light Shading Accent 5, 0 ,60, 0",
            "Light List Accent 5, 0 ,61, 0",
            "Light Grid Accent 5, 0 ,62, 0",
            "Medium Shading 1 Accent 5, 0 ,63, 0",
            "Medium Shading 2 Accent 5, 0 ,64, 0",
            "Medium List 1 Accent 5, 0 ,65, 0",
            "Medium List 2 Accent 5, 0 ,66, 0",
            "Medium Grid 1 Accent 5, 0 ,67, 0",
            "Medium Grid 2 Accent 5, 0 ,68, 0",
            "Medium Grid 3 Accent 5, 0 ,69, 0",
            "Dark List Accent 5, 0 ,70, 0",
            "Colorful Shading Accent 5, 0 ,71, 0",
            "Colorful List Accent 5, 0 ,72, 0",
            "Colorful Grid Accent 5, 0 ,73, 0",
            "Light Shading Accent 6, 0 ,60, 0",
            "Light List Accent 6, 0 ,61, 0",
            "Light Grid Accent 6, 0 ,62, 0",
            "Medium Shading 1 Accent 6, 0 ,63, 0",
            "Medium Shading 2 Accent 6, 0 ,64, 0",
            "Medium List 1 Accent 6, 0 ,65, 0",
            "Medium List 2 Accent 6, 0 ,66, 0",
            "Medium Grid 1 Accent 6, 0 ,67, 0",
            "Medium Grid 2 Accent 6, 0 ,68, 0",
            "Medium Grid 3 Accent 6, 0 ,69, 0",
            "Dark List Accent 6, 0 ,70, 0",
            "Colorful Shading Accent 6, 0 ,71, 0",
            "Colorful List Accent 6, 0 ,72, 0",
            "Colorful Grid Accent 6, 0 ,73, 0",
            "Subtle Emphasis, 0 ,19, 0, 1",
            "Intense Emphasis, 0 ,21, 0, 1",
            "Subtle Reference, 0 ,31, 0, 1",
            "Intense Reference, 0 ,32, 0, 1",
            "Book Title, 0 ,33, 0, 1",
            "Bibliography,, 37",
            "TOC Heading,, 39,, 1"
        };

        if (styles != null) {
            // set attributes for the latent styles part
            LatentStyles latentStyles = styles.getLatentStyles();
            latentStyles.setDefLockedState(false);
            latentStyles.setDefUIPriority(BigInteger.valueOf(99));
            latentStyles.setDefSemiHidden(true);
            latentStyles.setDefUnhideWhenUsed(true);
            latentStyles.setDefQFormat(false);

            // Extract predefined latent styles and put into the styles parts
            List<LsdException> styleList = latentStyles.getLsdException();
            // We will add all known latent MS styles therefore remove preset
            // to prevent duplicates
            styleList.clear();

            for (String style: predefinedLatentStyles) {
                String[] styleParts = style.split(",");

                LsdException lsdException = objectFactory.createStylesLatentStylesLsdException();
                lsdException.setParent(styleList);
                for (int i = 0; i < styleParts.length; i++) {

                    String value = styleParts[i].trim();
                    if (value.length() > 0) {
                        switch (i) {
                            case 0: lsdException.setName(value); break;
                            case 1: lsdException.setSemiHidden(Boolean.parseBoolean(value)); break;
                            case 2: lsdException.setUiPriority(BigInteger.valueOf(Long.parseLong(value))); break;
                            case 3: lsdException.setUnhideWhenUsed(Boolean.parseBoolean(value)); break;
                            case 4: lsdException.setQFormat(Boolean.parseBoolean(value)); break;
                        }
                    }
                }
                styleList.add(lsdException);
            }
        }
    }
}
