/*
 * Copyright 2014 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.odftoolkit.odfdom.component;

import org.json.JSONArray;
import org.json.JSONObject;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.element.OdfStylableElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableCellElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableRowElement;
import org.odftoolkit.odfdom.dom.element.text.TextPElement;
import org.odftoolkit.odfdom.pkg.OdfFileDom;

/**
 *
 * @author svante.schubert AT gmail DOT com
 */
class TableRangeChange implements TableChange {

	/** @param formatProps the format/style properties to adjust the range with. */
	public TableRangeChange(JSONObject formatProps) {
		determineRange(formatProps);
		mFormatProps = formatProps;		
	}
	
	/** @param changeRange a two dimensional row of changes. */ 
	public TableRangeChange(JSONArray changeRange) {
		// only cells are being changed in this call
		hasColumnEffect = Boolean.FALSE;
		hasRowEffect = Boolean.FALSE;
		hasCellEffect = Boolean.TRUE;
		// alternative we migth need to
		// Loop over every changed cell in the range, considering the change
		//determineRange(formatProps);

		mChangeRange = changeRange;		
	}
	
	private JSONArray mChangeRange;
	private JSONObject mFormatProps;
	
	private OdfFileDom mXMLDoc;
	private Boolean hasColumnEffect;
	private Boolean hasRowEffect;
	private Boolean hasCellEffect;

	public static TableRangeChange newFormatation(JSONObject formatProps) {		
		return new TableRangeChange(formatProps);
	}
	
	public static TableRangeChange newChange(JSONArray changeProps) {
		return new TableRangeChange(changeProps);
	}	
	
	private void determineRange(JSONObject formatProps){
		if (formatProps != null && formatProps.hasAndNotNull("column")) {
			hasColumnEffect = Boolean.TRUE;
		} else {
			hasColumnEffect = Boolean.FALSE;
		}
		if (formatProps != null && formatProps.hasAndNotNull("row")) {
			hasRowEffect = Boolean.TRUE;
		} else {
			hasRowEffect = Boolean.FALSE;
		}
		if (formatProps != null && (formatProps.hasAndNotNull("cell") || formatProps.hasAndNotNull("character") || formatProps.hasAndNotNull("paragraph"))) {
			hasCellEffect = Boolean.TRUE;
		} else {
			hasCellEffect = Boolean.FALSE;
		}		
	}

	public Object execute(OdfStylableElement tableStyleableElement, Object... args) {
		if (tableStyleableElement != null) {
			if (tableStyleableElement instanceof TableTableCellElement) {
				formatCell((TableTableCellElement) tableStyleableElement);
			}
			if (tableStyleableElement instanceof TableTableRowElement) {
				formatRow((TableTableRowElement) tableStyleableElement);
			}
			if (tableStyleableElement instanceof TableTableColumnElement) {
				formatColumn((TableTableColumnElement) tableStyleableElement);
			}
		}
		return Boolean.TRUE;
	}

	/**
	 * ToDo: PossibleRefactoring: This format Method might be moved to
	 * TableTableCell element
	 */
	public void formatCell(TableTableCellElement cellElement) {
		if (mXMLDoc == null) {
			mXMLDoc = (OdfFileDom) cellElement.getOwnerDocument();
		}
		// An empty cell element will not show formatting
		ensureContent(cellElement, mXMLDoc);
		JsonOperationConsumer.addStyle(mFormatProps, (OdfStylableElement) cellElement, mXMLDoc);

	}
	
	/**
	 * ToDo: PossibleRefactoring: This format Method might be moved to
	 * TableTableCell element
	 */
	public void changeCell(TableTableCellElement cellElement) {

	}	
	
	/** An empty cell element will not show formatting. At least an empty paragraph will be ensured. */
	private static void ensureContent(TableTableCellElement cellElement, OdfFileDom document){
		if(!cellElement.hasChildNodes()){
			cellElement.appendChild(document.createElementNS(TextPElement.ELEMENT_NAME));
		}
	}

	/**
	 * ToDo: PossibleRefactoring: This format Method might be moved to
	 * TableTableRow element
	 */
	public void formatRow(TableTableRowElement rowElement) {
		if (mXMLDoc == null) {
			mXMLDoc = (OdfFileDom) rowElement.getOwnerDocument();
		}
		JsonOperationConsumer.addStyle(mFormatProps, (OdfStylableElement) rowElement, mXMLDoc);
		JSONObject rowProps = mFormatProps.optJSONObject("row");
		if (rowProps != null) {
			Boolean changeVisibility = rowProps.optBoolean("visible", Boolean.TRUE);
			rowElement.setVisiblity(changeVisibility);
		}
	}

	/**
	 * ToDo: PossibleRefactoring: This format Method might be moved to
	 * TableTableColumn element
	 */
	public void formatColumn(TableTableColumnElement columnElement) {
		if (mXMLDoc == null) {
			mXMLDoc = (OdfFileDom) columnElement.getOwnerDocument();
		}
		JsonOperationConsumer.addStyle(mFormatProps, (OdfStylableElement) columnElement, mXMLDoc);
		JSONObject columnProps = mFormatProps.optJSONObject("column");
		if (columnProps != null) {
			Boolean changeVisibility = columnProps.optBoolean("visible", Boolean.TRUE);
			columnElement.setVisiblity(changeVisibility);
		}
	}

	public boolean effectsColumns() {
		return hasColumnEffect;
	}

	public boolean effectsRows() {
		return hasRowEffect;
	}

	public boolean effectsCells() {
		return hasCellEffect;
	}

	@Override
	public String toString() {
		return "formatChange: " + mFormatProps.toString();
	}
}
