
package com.openexchange.office.json.actions;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.documentconverter.Feature;
import com.openexchange.documentconverter.IManager;
import com.openexchange.documentconverter.Properties;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.DefaultFile;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.File.Field;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.folderstorage.FolderService;
import com.openexchange.groupware.ldap.User;
import com.openexchange.office.DocumentProperties;
import com.openexchange.office.IImporter;
import com.openexchange.office.IResourceProvider;
import com.openexchange.office.tools.DocumentType;
import com.openexchange.office.tools.ErrorCode;
import com.openexchange.office.tools.FileHelper;
import com.openexchange.office.tools.monitoring.CreateEvent;
import com.openexchange.office.tools.monitoring.DocumentEvent;
import com.openexchange.office.tools.monitoring.ErrorType;
import com.openexchange.office.tools.monitoring.Statistics;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.file.external.QuotaFileStorageExceptionCodes;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.tx.TransactionAwares;

/**
 * {@link CreateDefaultDocumentAction}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */

/*
 * MH/KA comment this Action to make it compile
 * @Action(method = RequestMethod.GET, name = "createdefaultdocument", description =
 * "Create a default (empty) OOXML document based on given document type.", parameters = {
 * @Parameter(name = "folder_id", description = "Folder ID of the requested infoitem."),
 * @Parameter(name = "session", description = "A session ID previously obtained from the login module."),
 * @Parameter(name = "uid", description = "The unique id of the client application."),
 * @Parameter(name = "document_type", optional=true, description =
 * "Specifies the type of default document to be created. Possible values are: 'text' (default), 'spreadsheet' and 'presentation' ."), },
 * responseDescription =
 * "Response with timestamp: A JSON object containing the attributes of the created filestore item: 'id', 'folder_id', 'version' and 'filename'."
 * )
 */

public class CreateDefaultDocumentAction extends DocumentFilterAction {

    private static final Log LOG = com.openexchange.log.Log.loggerFor(CreateDefaultDocumentAction.class);

    private static String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET = "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet";

    private static String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION = "application/vnd.openxmlformats-officedocument.presentationml.presentation";

    private static String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING = "application/vnd.openxmlformats-officedocument.wordprocessingml.document";

    private static String MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT = "application/vnd.oasis.opendocument.text";

    private static String MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET = "application/vnd.oasis.opendocument.spreadsheet";

    private static String MIMETYPE_VND_OASIS_OPENDOCUMENT_PRESENTATION = "application/vnd.oasis.opendocument.presentation";

    private enum OfficeDocumentType {
        NONE,
        MS_DOCUMENT_WORD,
        MS_DOCUMENT_SPREADSHEET,
        MS_DOCUMENT_PRESENTATION,
        ODF_DOCUMENT_WRITER,
        ODF_DOCUMENT_CALC,
        ODF_DOCUMENT_IMPRESS
    }

    /**
     * Initializes a new {@link CreateDefaultDocumentAction}.
     *
     * @param services
     * @param oqm
     */
    public CreateDefaultDocumentAction(ServiceLookup services) {

        super(services);
    }

    /**
     * @param extensionType
     * @param mimeType
     * @return
     */
    static public HashMap<String, String> getConversionFormatInfo(String mimeType, String extensionType) {
        HashMap<String, String> result = null;

        // get default conversion for mime type first
        if (mimeType != null) {
            if (mimeType.startsWith("application/msword") || mimeType.startsWith("application/vnd.ms-word") || mimeType.equals("application/rtf") || mimeType.equals("text/rtf")) {
                result = getFormatData(OfficeDocumentType.ODF_DOCUMENT_WRITER);
            }
            else if (mimeType.startsWith("application/vnd.ms-excel")) {
                // xlsm files have a binary mime type set, although they
                // are xml files => do not convert xlsm files (see #31245)
                if ((null == extensionType) || !extensionType.equals("xlsm")) {
                    result = getFormatData(OfficeDocumentType.MS_DOCUMENT_SPREADSHEET);
                }
            }
            else if (mimeType.startsWith("application/vnd.ms-powerpoint")) {
                result = getFormatData(OfficeDocumentType.ODF_DOCUMENT_IMPRESS);
            }
        }

        // get default conversion for extension type, if no valid mime type was given before
        if ((null == result) && (null != extensionType)) {
            if (extensionType.equals("doc") || extensionType.equals("dot") || extensionType.equals("rtf")) {
                result = getFormatData(OfficeDocumentType.ODF_DOCUMENT_WRITER);
            }
            else if (extensionType.equals("xls") || extensionType.equals("xlt") || extensionType.equals("xlsb")) {
                result = getFormatData(OfficeDocumentType.MS_DOCUMENT_SPREADSHEET);
            }
            else if (extensionType.equals("ppt") || extensionType.equals("pot")) {
                result = getFormatData(OfficeDocumentType.ODF_DOCUMENT_IMPRESS);
            }
        }

        return result;
    }

    /**
     * @param type
     * @return
     */
    static public HashMap<String, String> getFormatData(OfficeDocumentType officeDocumentType) {
        HashMap<String, String> data = new HashMap<String, String>();

        switch (officeDocumentType) {
        case MS_DOCUMENT_WORD: {
            data.put(Properties.PROP_MIME_TYPE, MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING);
            data.put(Properties.PROP_INPUT_TYPE, "docx");
            data.put(Properties.PROP_FILTER_SHORT_NAME, "ooxml");
            data.put("DocumentType", DocumentType.TEXT.toString());
            break;
        }

        case MS_DOCUMENT_SPREADSHEET: {
            data.put(Properties.PROP_MIME_TYPE, MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET);
            data.put(Properties.PROP_INPUT_TYPE, "xlsx");
            data.put(Properties.PROP_FILTER_SHORT_NAME, "ooxml");
            data.put("DocumentType", DocumentType.SPREADSHEET.toString());
            break;
        }

        case MS_DOCUMENT_PRESENTATION: {
            data.put(Properties.PROP_MIME_TYPE, MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION);
            data.put(Properties.PROP_INPUT_TYPE, "pptx");
            data.put(Properties.PROP_FILTER_SHORT_NAME, "ooxml");
            data.put("DocumentType", DocumentType.PRESENTATION.toString());
            break;
        }

        case ODF_DOCUMENT_WRITER: {
            data.put(Properties.PROP_MIME_TYPE, MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT);
            data.put(Properties.PROP_INPUT_TYPE, "odt");
            data.put(Properties.PROP_FILTER_SHORT_NAME, "odf");
            data.put("DocumentType", DocumentType.TEXT.toString());
            break;
        }

        case ODF_DOCUMENT_CALC: {
            data.put(Properties.PROP_INPUT_TYPE, "ods");
            data.put(Properties.PROP_MIME_TYPE, MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET);
            data.put(Properties.PROP_FILTER_SHORT_NAME, "odf");
            data.put("DocumentType", DocumentType.SPREADSHEET.toString());
            break;
        }

        case ODF_DOCUMENT_IMPRESS: {
            data.put(Properties.PROP_INPUT_TYPE, "odp");
            data.put(Properties.PROP_MIME_TYPE, MIMETYPE_VND_OASIS_OPENDOCUMENT_PRESENTATION);
            data.put(Properties.PROP_FILTER_SHORT_NAME, "odf");
            data.put("DocumentType", DocumentType.PRESENTATION.toString());
            break;
        }

        default: {
            break;
        }
        }

        return data;
    }

    @Override
    public AJAXRequestResult perform(AJAXRequestData request, ServerSession session) {
        AJAXRequestResult requestResult = null;
        DocumentEvent documentEvent = null;

        if ((null != request) && (null != session)) {
            DocumentType eventDocumentType = DocumentType.NONE;
            IDBasedFileAccess fileAccess = null;
            InputStream inputStm = null;
            boolean rollback = false;

            try {
                final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
                fileAccess = fileFactory.createAccess(session);

                // default type for new documents is OOXML
                if (null != fileAccess) {
                    final File file = new DefaultFile();
                    final String docTypeName = request.getParameter("document_type").toLowerCase();
                    final String fileId = request.getParameter("template_id");
                    final String versionStr = request.getParameter("version");
                    final boolean preserveFileName = Boolean.parseBoolean(request.getParameter("preserve_filename"));
                    String version = (null != versionStr) ? versionStr : "";

                    // determine requested document type and corresponding extensionType
                    OfficeDocumentType officeDocumentType = OfficeDocumentType.NONE;
                    String module;
                    String extensionType;
                    String mimeType;
                    HashMap<String, String> conversionFormat = null;

                    if (fileId == null) {
                        // create new document based on stored default document
                        if (docTypeName.equals("spreadsheet")) {
                            module = "spreadsheet";
                            officeDocumentType = OfficeDocumentType.MS_DOCUMENT_SPREADSHEET;
                            extensionType = "xlsx";
                            mimeType = MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET;
                            eventDocumentType = DocumentType.SPREADSHEET;
                        } else if (docTypeName.equals("presentation")) {
                            module = "presentation";
                            officeDocumentType = OfficeDocumentType.MS_DOCUMENT_PRESENTATION;
                            extensionType = "pptx";
                            mimeType = MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION;
                            eventDocumentType = DocumentType.PRESENTATION;
                        } else {
                            module = "text";
                            officeDocumentType = OfficeDocumentType.MS_DOCUMENT_WORD;
                            extensionType = "docx";
                            mimeType = MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING;
                            eventDocumentType = DocumentType.TEXT;
                        }

                        final IResourceProvider resourceProvider = m_services.getService(com.openexchange.office.IResourceProvider.class);
                        InputStream resourceInputStream = null;

                        // try to get default document from resource first
                        if (null != resourceProvider) {
                            // !!! resource leak warning has been checked:
                            // if resourceInputStream is returned unchanged as the result
                            // InputStream, it will be closed by the caller; otherwise it will
                            // be closed instantly after retrieving the real result InputStream from the
                            // resourceInputStream after calling importer.getDefaultDocument(...) !!!
                            resourceInputStream = resourceProvider.getResource(resourceProvider.getEntry(module, "template", "Default"));
                        }

                        // let the backend create the default document, if no default resource is available
                        IImporter importer = null;

                        switch (officeDocumentType) {
                        case MS_DOCUMENT_WORD: {
                            importer = m_services.getService(com.openexchange.office.ooxml.docx.Importer.class);
                            break;
                        }

                        case MS_DOCUMENT_SPREADSHEET: {
                            importer = m_services.getService(com.openexchange.office.ooxml.xlsx.Importer.class);
                            break;
                        }

                        case MS_DOCUMENT_PRESENTATION: {
                            importer = m_services.getService(com.openexchange.office.ooxml.pptx.Importer.class);
                            break;
                        }

                        default: {
                            break;
                        }
                        }

                        if (null != importer) {
                            final DocumentProperties docProps = new DocumentProperties();
                            final User user = session.getUser();

                            if (null != user) {
                                docProps.put(DocumentProperties.PROP_CREATOR, user.getDisplayName());
                                final String initialSheetname = request.getParameter("initial_sheetname");
                                if (initialSheetname != null) {
                                    docProps.put(DocumentProperties.PROP_INITIAL_SHEETNAME, initialSheetname);
                                }
                            }
                            inputStm = importer.getDefaultDocument(module, resourceInputStream, docProps);
                        }

                        if (resourceInputStream != inputStm) {
                            // Close resource input stream if default document stream
                            // was created by the import getDefaultDocument function. Don't
                            // do that if inputStm refers to resourceInputStream
                            IOUtils.closeQuietly(resourceInputStream);
                        }
                    } else {
                        // Create document based on template file. It could be that we have to
                        // convert the template document to the related ODF document format.
                        if (version.length() == 0) {
                            version = FileStorageFileAccess.CURRENT_VERSION;
                        }

                        File templateFile = fileAccess.getFileMetadata(fileId, version);
                        extensionType = FileHelper.getExtension(templateFile.getFileName());
                        mimeType = templateFile.getFileMIMEType();
                        conversionFormat = CreateDefaultDocumentAction.getConversionFormatInfo(mimeType, extensionType);

                        if (conversionFormat != null) {
                            final IManager dcManager = m_services.getService(IManager.class);

                            if ((null != dcManager) && dcManager.hasFeature(Feature.DOCUMENTCONVERTER)) {
                                AJAXRequestData loadRequest = new AJAXRequestData();
                                loadRequest.putParameter("id", fileId);
                                loadRequest.putParameter("version", version);
                                loadRequest.putParameter("fileName", templateFile.getFileName());
                                InputStream documentInputStm = FileHelper.getDocumentStream(m_services, session, loadRequest);

                                if (null != documentInputStm) {
                                    HashMap<String, Object> jobProperties = new HashMap<String, Object>(4);
                                    HashMap<String, Object> resultProperties = new HashMap<String, Object>(8);
                                    final String filterShortName = conversionFormat.get(Properties.PROP_FILTER_SHORT_NAME);
                                    final String convertedExtensionType = conversionFormat.get(Properties.PROP_INPUT_TYPE);

                                    jobProperties.put(Properties.PROP_INPUT_STREAM, documentInputStm);
                                    jobProperties.put(Properties.PROP_MIME_TYPE, convertedExtensionType);
                                    jobProperties.put(Properties.PROP_FILTER_SHORT_NAME, filterShortName);

                                    inputStm = dcManager.convert(filterShortName, jobProperties, resultProperties);
                                    IOUtils.closeQuietly(documentInputStm);

                                    // set new mime type and extensionType
                                    mimeType = conversionFormat.get(Properties.PROP_MIME_TYPE);
                                    extensionType = conversionFormat.get(Properties.PROP_INPUT_TYPE);
                                }
                            }

                            eventDocumentType = DocumentType.valueOf(conversionFormat.get("DocumentType"));
                        } else {
                            inputStm = fileAccess.getDocument(fileId, version);
                        }
                    }

                    String folderId = request.getParameter("folder_id");
                    boolean canWriteToFolder = FileHelper.folderHasWriteAccess(fileAccess, folderId);
                    boolean canCreateFilesInFolder = FileHelper.folderHasCreateAccess(fileAccess, folderId);
                    if ((!canWriteToFolder || !canCreateFilesInFolder) && !preserveFileName) {
                        // We cannot create files/write to the folder which contains our source template document (Edit as New).
                        // Create the document in the user's default folder instead.
                        folderId = FileHelper.getUserFolderId(m_services.getService(FolderService.class), session);
                    }

                    // create a new file store item with default content
                    file.setId(FileStorageFileAccess.NEW);
                    file.setFolderId(folderId);

                    if (!preserveFileName) {
                        final String initialFileName = request.getParameter("initial_filename");
                        final String createFileName = ((null == initialFileName) || (initialFileName.length() < 1)) ? "unnamed" : initialFileName;
                        file.setFileName(createFileName + "." + extensionType);
                    } else {
                        File templateFile = fileAccess.getFileMetadata(fileId, version);
                        file.setFileName(FileHelper.getBaseName(templateFile.getFileName()) + "." + extensionType);
                    }

                    file.setFileMIMEType(mimeType);

                    if (null != inputStm) {
                        fileAccess.startTransaction();
                        rollback = true;
                        fileAccess.saveDocument(file, inputStm, FileStorageFileAccess.DISTANT_FUTURE, new ArrayList<Field>());
                        fileAccess.commit();
                        rollback = false;

                        final JSONObject jsonObj = new JSONObject(6);

                        // return actual parameters of new file
                        jsonObj.put("id", file.getId());
                        jsonObj.put("folder_id", file.getFolderId());
                        jsonObj.put("version", file.getVersion());
                        jsonObj.put("filename", file.getFileName());
                        jsonObj.put("file_mimetype", file.getFileMIMEType());

                        requestResult = new AJAXRequestResult(jsonObj);
                        documentEvent = new CreateEvent(eventDocumentType);
                    } else {
                        // error case: we don't have a input stream
                        ErrorCode errorCode = null;
                        if (null != conversionFormat) {
                            errorCode = ErrorCode.CREATEDOCUMENT_CONVERSION_FAILED_ERROR;
                        } else if (null != fileId) {
                            errorCode = ErrorCode.CREATEDOCUMENT_CANNOT_READ_TEMPLATEFILE_ERROR;
                        } else {
                            errorCode = ErrorCode.CREATEDOCUMENT_CANNOT_READ_DEFAULTTEMPLATEFILE_ERROR;
                        }
                        JSONObject jsonObj = errorCode.getAsJSON();
                        requestResult = new AJAXRequestResult(jsonObj);
                        documentEvent = new DocumentEvent(eventDocumentType, ErrorType.CREATE);
                        LOG.warn(errorCode.getDescription());
                    }
                }
            } catch (final Exception e) {
                ErrorCode errorCode = ErrorCode.CREATEDOCUMENT_PERMISSION_CREATEFILE_MISSING_ERROR;

                if (e instanceof OXException) {
                    // special handling for OXExceptions
                    OXException ox = (OXException) e;

                    // special handling for file storage exceptions
                    if (ox.getPrefix().equalsIgnoreCase("FLS")) {
                        if (ox.getCode() == QuotaFileStorageExceptionCodes.STORE_FULL.getNumber()) {
                            // set specific error code if quota reached
                            errorCode = ErrorCode.CREATEDOCUMENT_QUOTA_REACHED_ERROR;
                        }
                    }
                }

                LOG.error(errorCode.getDescription(), e);
                try {
                    final JSONObject jsonObj = new JSONObject(3);
                    jsonObj.put("errorCode", errorCode.getCode());
                    jsonObj.put("error", errorCode.getCodeAsStringConstant());
                    jsonObj.put("errorDescription", errorCode.getDescription());
                    requestResult = new AJAXRequestResult(jsonObj);
                    documentEvent = new DocumentEvent(eventDocumentType, ErrorType.CREATE);
                } catch (final JSONException je) {
                    LOG.warn("Couldn't create JSON object while creating new document");
                }
            } finally {
                // Roll-back (if needed) and finish
                if (rollback) {
                    TransactionAwares.rollbackSafe(fileAccess);
                }

                TransactionAwares.finishSafe(fileAccess);
                IOUtils.closeQuietly(inputStm);
            }
        }

        // update statistics with a possibly created event
        Statistics.handleDocumentEvent(documentEvent);

        return requestResult;
    }
}
