/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.tools.jobpool.impl;

import java.lang.ref.WeakReference;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import com.openexchange.office.tools.jobpool.IJob;
import com.openexchange.office.tools.jobpool.IJobResultSink;
import com.openexchange.office.tools.jobpool.JobArguments;
import com.openexchange.office.tools.jobpool.JobBase;
import com.openexchange.office.tools.jobpool.JobPoolConfig;
import com.openexchange.office.tools.jobpool.JobResults;

//=============================================================================
public class Job implements IJob
{
	//-------------------------------------------------------------------------
	public static final int DEFAULT_ASYNC_TIMEOUT_IN_MS = 30000;
	
	//-------------------------------------------------------------------------
	public Job ()
		throws Exception
	{}

	//-------------------------------------------------------------------------
	public synchronized void bindToPool (final JobPool aPool)
	    throws Exception
	{
		m_aPool = new WeakReference< JobPool >(aPool);
	}

	//-------------------------------------------------------------------------
	public synchronized void bindToJob (final JobBase aJob)
	    throws Exception
	{
		m_aJob = aJob;
	}
	
	//-------------------------------------------------------------------------
	public synchronized JobBase getJob ()
	    throws Exception
	{
		return m_aJob;
	}

	//-------------------------------------------------------------------------
	@Override
	public synchronized int getId ()
	    throws Exception
	{
		return m_aJob.getId();
	}

	//-------------------------------------------------------------------------
	@Override
	public /* NO SYNCHRONIZE ! */ JobResults executeSynchronous(final JobArguments iArguments)
		throws Exception
	{
		// reuse code ... and implement synchronous execution on top of asynchronous execution .-)
		// the other way around wont be possible.
		//
		// note some details ...
		// - sink is a local instance only - not known by others
		//   so we can be sure there is a 1:1 relation between
		//   our request and it's response
		//
		// - using the asynchronous way in combination with "sink.wait()"
		//   provide the possibility to implement a timeout behaviour ...
		//   which is not (real) possible with a synchronous call

		long nTimeoutInMs = JobPoolConfig.DEFAULT_JOB_EXECUTION_TIMEOUT_IN_MS;
		synchronized(this)
		{
			nTimeoutInMs = impl_getConfiguredTimeout ();
		}
		
		JobResultSink  aSink = new JobResultSink  ( );
		CountDownLatch aSync = new CountDownLatch (1);
		aSink.bind2Latch(aSync);
		
		try
		{
			executeAsynchronoues (iArguments, aSink);
		}
		catch (Throwable exT)
		{
			System.err.println(exT.getMessage ());
			exT.printStackTrace(System.err);
		}
		
		System.out.println ("wait for job "+nTimeoutInMs+" ms...");
		boolean bFinished = aSync.await(nTimeoutInMs, TimeUnit.MILLISECONDS);
		boolean bTimedOut = ! bFinished; // :-)
		System.out.println ("===== job finished ...");
		
		JobResults aResult = null;
		if ( bTimedOut )
		{
			System.out.println ("##### ... with timeout");
			aResult = JobResults.TIMEOUT_RESULT();
		}
		else
		{
			System.out.println ("##### ... with result");
			aResult = aSink.getLastResults();
		}
		
		System.out.println ("##### RESULTS = "+aResult);
		return aResult;
	}

	//-------------------------------------------------------------------------
	@Override
	public synchronized void executeAsynchronoues(final JobArguments   iArguments,
									 			  final IJobResultSink iSink     )
		throws Exception
	{
		impl_checkAndRejectReusingSameJob ();

		// job died under the hood ? (disabled by anywhere else ... !)
		// -> return 'nothing' .. because we have nothing

		final JobBase aJob = getJob ();
		if (aJob == null)
		{
			if (iSink != null)
			{
				System.out.println ("NO JOB - return empty results");
				iSink.setResults(JobResults.EMPTY_RESULT());
			}
			return;
		}

		// execute job asynchronous
		
		new Thread ()
		{
			@Override
			public void run ()
			{
				Throwable aError = null;
				try
				{
					aJob.setArguments(iArguments);
					aJob.execute();
					
					if (iSink != null)
					{
						final JobResults iResults = aJob.getResults();
						iResults.setSuccessfully();
						System.out.println ("JOB FINISHED - return real results");
						iSink.setResults(iResults);
					}
					
					return;
				}
				catch (Throwable ex)
				{
					aError = ex;
				}
				
				try
				{
					impl_handleJobError (aError);

					if (iSink != null)
					{
						System.out.println ("JOB ERROR - return error results");
						iSink.setResults(JobResults.ERROR_RESULT(aError));
					}
				}
				catch (Throwable ex)
				{
					System.err.println (ex.getMessage ());
					ex.printStackTrace(System.err);

					// ignore that "inner" error !
				}
			}
			
		}.start ();
	}
	
	//-------------------------------------------------------------------------
	private void impl_handleJobError (final Throwable aError)
	{
		try
		{
			// TODO refactor me : use real logging instead !

			System.err.println (aError);
			aError.printStackTrace(System.err);
			
			JobPool aPool = null;
			synchronized(this)
			{
				aPool = m_aPool.get();
			}
			
			if (aPool != null)
				aPool.deactivateJob (this);
		}
		catch (Exception ex)
		{
			System.err.println (ex.getMessage ());
			ex.printStackTrace(System.err);

			// ignore errors inside errors !
			// TODO implement me : log error ?
		}
	}
	
	//-------------------------------------------------------------------------
	private void impl_checkAndRejectReusingSameJob ()
	    throws Exception
	{
		if (m_bWasUsed)
			throw new Exception ("Job has not to be reused for multiple requests ! Please lease new job always ...");
		m_bWasUsed = true;
	}
	
	//-------------------------------------------------------------------------
	private long impl_getConfiguredTimeout ()
	    throws Exception
	{
		JobPool aPool = m_aPool.get ();
		if (aPool == null)
			return DEFAULT_ASYNC_TIMEOUT_IN_MS;
		
		final JobPoolConfig aConfig  = aPool.getConfig();
		final Long          nTimeout = aConfig.get(JobPoolConfig.PROP_JOB_EXECUTION_TIMEOUT, JobPoolConfig.DEFAULT_JOB_EXECUTION_TIMEOUT_IN_MS);
		
		return nTimeout;
	}

	//-------------------------------------------------------------------------
	private WeakReference< JobPool > m_aPool = null;

	//-------------------------------------------------------------------------
	private JobBase m_aJob = null;

	//-------------------------------------------------------------------------
	private boolean m_bWasUsed = false;
}
