
package com.openexchange.office.json.actions;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import org.apache.commons.logging.Log;
import org.json.JSONObject;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.documentconverter.Feature;
import com.openexchange.documentconverter.IManager;
import com.openexchange.file.storage.DefaultFile;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.File.Field;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.folderstorage.FolderService;
import com.openexchange.groupware.ldap.User;
import com.openexchange.java.Streams;
import com.openexchange.office.DocumentProperties;
import com.openexchange.office.DocumentType;
import com.openexchange.office.IImporter;
import com.openexchange.office.IResourceProvider;
import com.openexchange.office.tools.ErrorCode;
import com.openexchange.office.tools.FileHelper;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.tx.TransactionAwares;

/**
 * {@link CreateDefaultDocumentAction}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */

/*
 *  MH/KA comment this Action to make it compile

@Action(method = RequestMethod.GET, name = "createdefaultdocument", description = "Create a default (empty) OOXML document based on given document type.", parameters = {
    @Parameter(name = "folder_id", description = "Folder ID of the requested infoitem."),
    @Parameter(name = "session", description = "A session ID previously obtained from the login module."),
    @Parameter(name = "uid", description = "The unique id of the client application."),
    @Parameter(name = "document_type", optional=true, description = "Specifies the type of default document to be created. Possible values are: 'text' (default), 'spreadsheet' and 'presentation' ."),
}, responseDescription = "Response with timestamp: A JSON object containing the attributes of the created filestore item: 'id', 'folder_id', 'version' and 'filename'.")
 *
 */

public class CreateDefaultDocumentAction extends DocumentFilterAction {

    private static final Log LOG = com.openexchange.log.Log.loggerFor(CreateDefaultDocumentAction.class);

    private static String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET = "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet";
    private static String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION = "application/vnd.openxmlformats-officedocument.presentationml.presentation";
    private static String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING = "application/vnd.openxmlformats-officedocument.wordprocessingml.document";
    private static String MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT = "application/vnd.oasis.opendocument.text";
    private static String MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET = "application/vnd.oasis.opendocument.spreadsheet";
    private static String MIMETYPE_VND_OASIS_OPENDOCUMENT_PRESENTATION = "application/vnd.oasis.opendocument.presentation";

    private enum MsDocumentType
    {
        MS_DOCUMENT_WORD,
        MS_DOCUMENT_SPREADSHEET,
        MS_DOCUMENT_PRESENTATION
    }

    private enum OdfDocumentType
    {
        ODF_DOCUMENT_WRITER,
        ODF_DOCUMENT_CALC,
        ODF_DOCUMENT_IMPRESS
    }

    /**
     * Initializes a new {@link CreateDefaultDocumentAction}.
     *
     * @param services
     * @param oqm
     */
    public CreateDefaultDocumentAction(ServiceLookup services) {

        super(services);
    }

    static public HashMap<String, String> getFormatData(MsDocumentType type) {
        HashMap<String, String> data = new HashMap<String, String>();

        if (type == MsDocumentType.MS_DOCUMENT_WORD) {
            data.put("extension", "docx");
            data.put("mimetype", MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING);
        }
        else if (type == MsDocumentType.MS_DOCUMENT_SPREADSHEET) {
            data.put("extension", "xlsx");
            data.put("mimetype", MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET);
        }
        else if (type == MsDocumentType.MS_DOCUMENT_PRESENTATION) {
            data.put("extension", "pptx");
            data.put("mimetype", MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION);
        }

        return data;
    }

    static public HashMap<String, String> getFormatData(OdfDocumentType type) {
        HashMap<String, String> data = new HashMap<String, String>();

        if (type == OdfDocumentType.ODF_DOCUMENT_WRITER) {
            data.put("extension", "odt");
            data.put("mimetype", MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT);
        }
        else if (type == OdfDocumentType.ODF_DOCUMENT_CALC) {
            data.put("extension", "ods");
            data.put("mimetype", MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET);
        }
        else if (type == OdfDocumentType.ODF_DOCUMENT_IMPRESS) {
            data.put("extension", "odp");
            data.put("mimetype", MIMETYPE_VND_OASIS_OPENDOCUMENT_PRESENTATION);
        }

        return data;
    }

    static public HashMap<String, String> getConvertedFormatInfo(String extension, String mimeType) {
        HashMap<String, String> result = null;

        if (extension != null || mimeType != null) {
            if (extension != null) {
                if (extension.equals("doc") || extension.equals("dot")) {
                    result = CreateDefaultDocumentAction.getFormatData(OdfDocumentType.ODF_DOCUMENT_WRITER);
                }
                else if (extension.equals("xls") || extension.equals("xlt")) {
                    result = CreateDefaultDocumentAction.getFormatData(OdfDocumentType.ODF_DOCUMENT_CALC);
                }
                else if (extension.equals("ppt") || extension.equals("pot")) {
                    result = CreateDefaultDocumentAction.getFormatData(OdfDocumentType.ODF_DOCUMENT_IMPRESS);
                }
                else if (extension.equals("rtf")) {
                    result = CreateDefaultDocumentAction.getFormatData(OdfDocumentType.ODF_DOCUMENT_WRITER);
                }
            }
            else if (mimeType != null) {
                if (mimeType.equals("application/msword")) {
                    result = CreateDefaultDocumentAction.getFormatData(OdfDocumentType.ODF_DOCUMENT_WRITER);
                }
                else if (mimeType.equals("application/vnd.ms-excel")) {
                    result = CreateDefaultDocumentAction.getFormatData(OdfDocumentType.ODF_DOCUMENT_CALC);
                }
                else if (mimeType.equals("application/vnd.ms-powerpoint")) {
                    result = CreateDefaultDocumentAction.getFormatData(OdfDocumentType.ODF_DOCUMENT_IMPRESS);
                }
                else if (mimeType.equals("application/rtf") || mimeType.equals("text/rtf")) {
                    result = CreateDefaultDocumentAction.getFormatData(OdfDocumentType.ODF_DOCUMENT_WRITER);
                }
            }
        }

        return result;
    }

    @SuppressWarnings("resource")
    @Override
    public AJAXRequestResult perform(AJAXRequestData request, ServerSession session) {
        AJAXRequestResult requestResult = null;

        if ((null != request) && (null != session)) {
            IDBasedFileAccess fileAccess = null;
            InputStream inputStm = null;
            boolean rollback = false;
            try {
                final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
                fileAccess = fileFactory.createAccess(session);

                // default type for new documents is OOXML
                if (null != fileAccess) {
                    final File file = new DefaultFile();
                    final String docTypeName = request.getParameter("document_type").toLowerCase();
                    final String fileId = request.getParameter("template_id");
                    final String versionStr = request.getParameter("version");
                    final boolean preserveFileName = Boolean.parseBoolean(request.getParameter("preserve_filename"));
                    String version = (null != versionStr) ? versionStr : "";


                    // determine requested document type and corresponding extension
                    DocumentType docType;
                    String module;
                    String extension;
                    String mimeType;
                    HashMap<String, String> convertedFormat = null;

                    if (fileId == null) {
                        // create new document based on stored default document
                        if (docTypeName.equals("spreadsheet")) {
                            docType = DocumentType.SPREADSHEET;
                            module = "spreadsheet";
                            extension = "xlsx";
                            mimeType = MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET;
                        } else if (docTypeName.equals("presentation")) {
                            docType = DocumentType.PRESENTATION;
                            module = "presentation";
                            extension = "pptx";
                            mimeType = MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION;
                        }
                        else {
                            docType = DocumentType.TEXT;
                            module = "text";
                            extension = "docx";
                            mimeType = MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING;
                        }

                        final IResourceProvider resourceProvider = m_services.getService(com.openexchange.office.IResourceProvider.class);

                        // try to get default document from resource first
                        if (null != resourceProvider) {
                            inputStm = resourceProvider.getResource(resourceProvider.getEntry(module, "template", "Default"));
                        }

                        // let the backend create the default document, if no default resource is available
                        IImporter importer = null;
                        if(docType == DocumentType.TEXT) {
                            importer = m_services.getService(com.openexchange.office.ooxml.docx.Importer.class);
                        }
                        else if(docType == DocumentType.PRESENTATION) {
                            importer = m_services.getService(com.openexchange.office.ooxml.pptx.Importer.class);
                        }
                        else if(docType == DocumentType.SPREADSHEET) {
                            importer = m_services.getService(com.openexchange.office.ooxml.xlsx.Importer.class);
                        }
                        if(importer!=null) {
                            final DocumentProperties docProps = new DocumentProperties();
                            final User user = session.getUser();

                            if (null != user) {
                                docProps.put(DocumentProperties.PROP_CREATOR, user.getDisplayName());
                            }

                            inputStm = importer.getDefaultDocument(docType, inputStm, docProps);
                        }
                    }
                    else {
                        // Create document based on template file. It could be that we have to
                        // convert the template document to the related ODF document format.
                        if (version.length() == 0) {
                            version = FileStorageFileAccess.CURRENT_VERSION;
                        }

                        File templateFile = fileAccess.getFileMetadata(fileId, version);
                        mimeType = templateFile.getFileMIMEType();
                        extension = FileHelper.getExtension(templateFile.getFileName());

                        convertedFormat = CreateDefaultDocumentAction.getConvertedFormatInfo(extension, mimeType);

                        if (convertedFormat != null) {
                            final IManager dcManager = m_services.getService(IManager.class);

                            if ((null != dcManager) && dcManager.hasFeature(Feature.DOCUMENTCONVERTER)) {
                                AJAXRequestData loadRequest = new AJAXRequestData();
                                loadRequest.putParameter("id", fileId);
                                loadRequest.putParameter("version", version);
                                loadRequest.putParameter("fileName", templateFile.getFileName());
                                InputStream documentStm = FileHelper.getDocumentStream(m_services, session, loadRequest);

                                if (null != documentStm) {
                                    HashMap<String, Object> jobProperties = new HashMap<String, Object>(8);
                                    HashMap<String, Object> resultProperties = new HashMap<String, Object>(8);

                                    jobProperties.put("InputStream", documentStm);
                                    inputStm = dcManager.convert("odf", jobProperties, resultProperties);

                                    // set new mime type and extension
                                    mimeType = convertedFormat.get("mimetype");
                                    extension = convertedFormat.get("extension");

                                    documentStm.close();
                                }
                            }
                        }
                        else {
                            inputStm = fileAccess.getDocument(fileId, version);
                        }
                    }

                    String folderId = request.getParameter("folder_id");
                    boolean canWriteToFolder = FileHelper.folderHasWriteAccess(fileAccess, folderId);
                    boolean canCreateFilesInFolder = FileHelper.folderHasCreateAccess(fileAccess, folderId);
                    if ((!canWriteToFolder || !canCreateFilesInFolder) && !preserveFileName) {
                        // We cannot create files/write to the folder which contains our source template document (Edit as New).
                        // Create the document in the user's default folder instead.
                        folderId = FileHelper.getUserFolderId(m_services.getService(FolderService.class), session);
                    }

                    // create a new file store item with default content
                    file.setId(FileStorageFileAccess.NEW);
                    file.setFolderId(folderId);

                    if (!preserveFileName) {
                        final String initialFileName = request.getParameter("initial_filename");
                        final String createFileName = ((null == initialFileName) || (initialFileName.length() < 1)) ? "unnamed" : initialFileName;
                        file.setFileName( createFileName + "." + extension);
                    }
                    else {
                        File templateFile = fileAccess.getFileMetadata(fileId, version);
                        file.setFileName(FileHelper.getBaseName(templateFile.getFileName()) + "." + extension);
                    }

                    file.setFileMIMEType(mimeType);

                    if (null != inputStm) {
                        fileAccess.startTransaction();
                        rollback = true;
                        fileAccess.saveDocument(file, inputStm, FileStorageFileAccess.DISTANT_FUTURE, new ArrayList<Field>());
                        fileAccess.commit();
                        rollback = false;

                        final JSONObject jsonObj = new JSONObject(6);
                        // return actual parameters of new file
                        jsonObj.put("id", file.getId());
                        jsonObj.put("folder_id", file.getFolderId());
                        jsonObj.put("version", file.getVersion());
                        jsonObj.put("filename", file.getFileName());

                        requestResult = new AJAXRequestResult(jsonObj);
                    } else {
                        // error case: we don't have a input stream
                        ErrorCode errorCode = null;
                        if (null != convertedFormat) {
                            errorCode = ErrorCode.CREATEDOCUMENT_CONVERSION_FAILED_ERROR;
                        } else if (null != fileId) {
                            errorCode = ErrorCode.CREATEDOCUMENT_CANNOT_READ_TEMPLATEFILE_ERROR;
                        } else {
                            errorCode = ErrorCode.CREATEDOCUMENT_CANNOT_READ_DEFAULTTEMPLATEFILE_ERROR;
                        }
                        JSONObject jsonObj = errorCode.getAsJSON();
                        requestResult = new AJAXRequestResult(jsonObj);
                        LOG.warn(errorCode.getDescription());
                    }
                }
            } catch (final Exception e) {
                final ErrorCode errorCode = ErrorCode.CREATEDOCUMENT_PERMISSION_CREATEFILE_MISSING_ERROR;
                LOG.error(errorCode.getDescription(), e);

                try {
                    final JSONObject jsonObj = new JSONObject(3);
                    jsonObj.put("errorCode", errorCode.getCode());
                    jsonObj.put("error", errorCode.getCodeAsStringConstant());
                    jsonObj.put("errorDescription", errorCode.getDescription());
                    requestResult = new AJAXRequestResult(jsonObj);
                } catch (final Exception je) {
                    LOG.warn("Couldn't create JSON object while creating new document");
                }
            } finally {
                // Roll-back (if needed) and finish
                if (rollback) {
                    TransactionAwares.rollbackSafe(fileAccess);
                }
                TransactionAwares.finishSafe(fileAccess);
                // Close stream
                Streams.close(inputStm);
            }
        }

        return requestResult;
    }
}
