/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.operations;

import java.util.Iterator;
import java.util.List;
import javax.xml.bind.JAXBException;
import org.apache.commons.logging.Log;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.packages.SpreadsheetMLPackage;
import org.docx4j.openpackaging.parts.PartName;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorkbookPart;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorksheetPart;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.docx4j.relationships.Relationship;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.xlsx4j.sml.CTCellFormula;
import org.xlsx4j.sml.CTMergeCell;
import org.xlsx4j.sml.CTMergeCells;
import org.xlsx4j.sml.Cell;
import org.xlsx4j.sml.Cell.CellRefRange;
import org.xlsx4j.sml.Col;
import org.xlsx4j.sml.Cols;
import org.xlsx4j.sml.Row;
import org.xlsx4j.sml.STCellFormulaType;
import org.xlsx4j.sml.STCellType;
import org.xlsx4j.sml.Sheet;
import org.xlsx4j.sml.SheetData;
import org.xlsx4j.sml.Sheets;
import org.xlsx4j.sml.Workbook;
import org.xlsx4j.sml.Worksheet;
import com.openexchange.log.LogFactory;
import com.openexchange.office.ooxml.xlsx.OperationDocument;
import com.openexchange.office.ooxml.xlsx.tools.CellUtils;
import com.openexchange.office.ooxml.xlsx.tools.ColumnUtils;
import com.openexchange.office.ooxml.xlsx.tools.RowUtils;
import com.openexchange.office.ooxml.xlsx.tools.SheetUtils;
import com.openexchange.office.ooxml.xlsx.tools.Utils;

public class ApplyOperationHelper extends com.openexchange.office.ooxml.operations.ApplyOperationHelper {

    protected static Log log = LogFactory.getLog(ApplyOperationHelper.class);

    private final OperationDocument operationDocument;
    private final SpreadsheetMLPackage spreadsheetMLPackage;
    private final WorkbookPart workbookPart;

    public ApplyOperationHelper(OperationDocument _operationDocument) {
        super();

        operationDocument = _operationDocument;
        spreadsheetMLPackage = _operationDocument.getPackage();
        workbookPart = spreadsheetMLPackage.getWorkbookPart();
    }

    @Override
    public OperationDocument getOperationDocument() {
        return operationDocument;
    }

    public static void setCellContent(Cell cell, Object cellValue) {
        if(cellValue instanceof String) {
            String v = (String)cellValue;
            if(v.length()>0&&v.startsWith("=")) {
                cell.setT(null);;
                CTCellFormula cellFormula = Context.getsmlObjectFactory().createCTCellFormula();
                cellFormula.setT(STCellFormulaType.NORMAL);
                cellFormula.setValue(v.substring(1));
                cell.setF(cellFormula);
            }
            else if(v.length()>0&&v.startsWith("#")) {
                cell.setT(STCellType.E);
                cell.setV(v.substring(1));
                cell.setF(null);
            }
            else {
                cell.setT(STCellType.STR);
                cell.setV(v.startsWith("'") ? v.substring(1) : v);
                cell.setF(null);
            }
        }
        else if(cellValue instanceof Number) {
            cell.setT(STCellType.N);
            cell.setV(((Number)cellValue).toString());
            cell.setF(null);
        }
        else if(cellValue instanceof Boolean) {
            cell.setT(STCellType.B);
            cell.setV((Boolean)cellValue ? "1" : "0");
            cell.setF(null);
        }
        else if(cellValue==null) {
            // empty cell ->
            cell.setT(null);
            cell.setF(null);
            cell.setIs(null);
            cell.setV(null);
        }
    }

    private Worksheet getWorksheet(int sheetIndex) {
        Sheet sheet = SheetUtils.getSheet(spreadsheetMLPackage, sheetIndex);

        RelationshipsPart relationshipPart = spreadsheetMLPackage.getWorkbookPart().getRelationshipsPart();
        WorksheetPart worksheetPart = (WorksheetPart)relationshipPart.getPart(sheet.getId());
        return worksheetPart.getJaxbElement();
    }

    public void setCellContents(int sheetIndex, JSONArray start, JSONArray rowArray)
        throws Exception {

        if(start.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::setCellContents: size of start parameter != 2");

        Worksheet worksheet = getWorksheet(sheetIndex);
        worksheet.setDimension(null);
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            sheetData = Context.getsmlObjectFactory().createSheetData();
            worksheet.setSheetData(sheetData);
        }
        int y = start.getInt(1);
        for(int i=0; i<rowArray.length(); i++) {
            JSONArray cellArray = rowArray.getJSONArray(i);
            int x = start.getInt(0);
            for(int j=0; j<cellArray.length(); j++) {

                Row row = sheetData.getRow(y, true);
                Cell cell = row.getCell(x, true);

                JSONObject cellObject = cellArray.getJSONObject(j);
                Iterator<String> keys = cellObject.keys();
                while(keys.hasNext()) {
                    String attr = keys.next();
                    if (attr.equals("value")) {
                        setCellContent(cell, cellObject.get(attr));
                    }
                    else if(attr.equals("attrs")) {
                        CellUtils.applyCellProperties(operationDocument, cellObject.optJSONObject(attr), cell, getOperationDocument().getStylesheet(true));
                    }
                    else if(attr.equals("shared")) {

                    }
                    else if(attr.equals("ref")) {

                    }
                }
                x++;
            }
            y++;
        }
    }

    public void fillCellRange(int sheetIndex, JSONArray start, JSONArray optEnd, boolean hasValue, Object optValue, JSONObject optAttrs, int optShared, JSONArray optRef)
        throws Exception {

        if(start.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::fillCellContents: size of start parameter != 2");

        if(optEnd==null)
            optEnd = start;
        else if (optEnd.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::fillCellContents: size of end parameter != 2");

        Worksheet worksheet = getWorksheet(sheetIndex);
        worksheet.setDimension(null);
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            sheetData = Context.getsmlObjectFactory().createSheetData();
            worksheet.setSheetData(sheetData);
        }
        for(int y=start.getInt(1); y<=optEnd.getInt(1); y++) {
            for(int x=start.getInt(0); x<=optEnd.getInt(0); x++) {

                Row row = sheetData.getRow(y, true);
                Cell cell = row.getCell(x, true);

                // value is tripleState (omitted, null and !=null)
                if(hasValue) {
                    setCellContent(cell, optValue);
                }

                if(optAttrs!=null) {
                    CellUtils.applyCellProperties(operationDocument, optAttrs, cell, getOperationDocument().getStylesheet(true));
                }
            }
        }
    }

    public void setRowAttributes(int sheetIndex, int start, int optEnd, JSONObject attrs)
        throws Exception {

        if(optEnd==-1)
            optEnd = start;

        Worksheet worksheet = getWorksheet(sheetIndex);
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            sheetData = Context.getsmlObjectFactory().createSheetData();
            worksheet.setSheetData(sheetData);
        }
        for(int y=start; y<=optEnd; y++) {
            RowUtils.applyRowProperties(operationDocument, attrs, sheetData.getRow(y, true), getOperationDocument().getStylesheet(true));
        }
    }

    public void setColumnAttributes(int sheetIndex, int start, int optEnd, JSONObject attrs)
        throws Exception {

        if(optEnd==-1)
            optEnd = start;

        Worksheet worksheet = getWorksheet(sheetIndex);
        List<Cols> colsList = worksheet.getCols();
        if(colsList.size()==0) {
            colsList.add(Context.getsmlObjectFactory().createCols());
        }

        // TODO: we only apply our attributes to the first Cols within the cols list, so additional
        // entries have to be merged ...
        start++;
        optEnd++;
        Cols cols = colsList.get(0);
        Utils.createColumnAttributeRanges(cols, Long.valueOf(start), Long.valueOf((optEnd-start)+1));
        for(Col col:cols.getCol()) {
            if(col.getMin()>optEnd)
                return;
            if(col.getMax()<start)
                continue;
            ColumnUtils.applyColumnProperties(operationDocument, attrs, col, getOperationDocument().getStylesheet(true));
        }
    }

    public void insertRows(int sheetIndex, int start, int optEnd) {

        Worksheet worksheet = getWorksheet(sheetIndex);
        worksheet.setDimension(null);
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            sheetData = Context.getsmlObjectFactory().createSheetData();
            worksheet.setSheetData(sheetData);
        }
        sheetData.insertRows(start, optEnd < start ? 1 : (optEnd - start) + 1);

        // taking care of merged cells
        CTMergeCells mergeCells = worksheet.getMergeCells();
        if(mergeCells!=null) {
            List<CTMergeCell> mergeCellList = mergeCells.getMergeCell();
            for(CTMergeCell mergeCell:mergeCellList) {
                CellRefRange cellRefRange = Cell.createCellRefRange(mergeCell.getRef());
                if(start<=cellRefRange.getEnd().getRow()) {
//                    cellRefRange.getEnd().

                }

                // TODO: taking care of mergedCells
            }
        }
        // TODO: taking care of drawings
    }

    public void deleteRows(int sheetIndex, int start, int optEnd) {

        Worksheet worksheet = getWorksheet(sheetIndex);
        worksheet.setDimension(null);
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            log.warn("xlsx export: deleteRows, no sheetData available");
            return;
        }
        sheetData.deleteRows(start, optEnd < start ? 1 : (optEnd - start) + 1);

        // taking care of merged cells
        CTMergeCells mergeCells = worksheet.getMergeCells();
        if(mergeCells!=null) {
            List<CTMergeCell> mergeCellList = mergeCells.getMergeCell();
            for(CTMergeCell mergeCell:mergeCellList) {
                // TODO: taking care of mergedCells
            }
        }
        // TODO: taking care of drawings
    }

    public void insertSheet(int sheetIndex, String sheetName, JSONObject attrs)
        throws InvalidFormatException, JAXBException {

        // first we have to get a free relationship that can be used for our new sheet
        int sheetNumber = 1;

        Relationship relShip = null;
        PartName partName = null;
        do {
            partName = new PartName("/xl/worksheets/sheet" + Integer.toString(sheetNumber++) + ".xml");
            relShip = workbookPart.getRelationshipsPart().getRel(partName);
        }
        while(relShip!=null);

        // now we have a free partName... and we will look for a free SheetId
        Workbook workbook = workbookPart.getJaxbElement();
        Sheets sheets = workbook.getSheets();
        List<Sheet> sheetList = sheets.getSheet();

        long sheetId = 1;
        do {
            boolean idFound = false;
            for(int i = 0; i < sheetList.size(); i++) {
                if(sheetList.get(i).getSheetId()==sheetId) {
                    idFound = true;
                    break;
                }
            }
            if(idFound==false)
                break;
            sheetId++;
        }
        while(true);
        spreadsheetMLPackage.createWorksheetPart(partName, sheetName, sheetId);

        // setting the correct index position
        Sheet sheet = sheetList.remove(sheetList.size() - 1);
        sheetList.add(sheetIndex, sheet);
        if(attrs!=null)
            SheetUtils.applySheetProperties(workbook.getSheets().getSheet().get(sheetIndex), attrs.optJSONObject("sheet"));
    }

    public void moveSheet(int sheetIndex, int to) {
        Workbook workbook = workbookPart.getJaxbElement();
        Sheets sheets = workbook.getSheets();
        List<Sheet> sheetList = sheets.getSheet();
        sheetList.add(to, sheetList.remove(sheetIndex));
    }

    public void deleteSheet(int sheetIndex) {
        Workbook workbook = workbookPart.getJaxbElement();
        Sheets sheets = workbook.getSheets();
        List<Sheet> sheetList = sheets.getSheet();
        Sheet sheet = sheetList.remove(sheetIndex);
        String relId = sheet.getId();
        RelationshipsPart relationshipsPart = workbookPart.getRelationshipsPart();
        relationshipsPart.removeRelationship(relationshipsPart.getRelationshipByID(relId));
    }

    public void setSheetName(int sheetIndex, String sheetName) {
        Workbook workbook = workbookPart.getJaxbElement();
        workbook.getSheets().getSheet().get(sheetIndex).setName(sheetName);
    }

    public void setSheetAttributes(int sheetIndex, JSONObject attrs) {
        Workbook workbook = workbookPart.getJaxbElement();
        if(attrs!=null)
            SheetUtils.applySheetProperties(workbook.getSheets().getSheet().get(sheetIndex), attrs.optJSONObject("sheet"));
    }

    public void mergeCells(int sheetIndex, JSONArray start, JSONArray optEnd, boolean keepContent, String type)
        throws JSONException {

        if(start.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::mergeCells: size of start parameter != 2");

        if(optEnd==null)
            optEnd = start;
        else if (optEnd.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::mergeCells: size of end parameter != 2");

        Worksheet worksheet = getWorksheet(sheetIndex);
        CTMergeCells mergeCells = worksheet.getMergeCells();
        if(mergeCells==null) {
            mergeCells = Context.getsmlObjectFactory().createCTMergeCells();
            worksheet.setMergeCells(mergeCells);
        }
        // first we will remove each mergeCell that is covering our new mergeCellRange
        List<CTMergeCell> mergeCellList = mergeCells.getMergeCell();
        for(int i=mergeCellList.size()-1;i>=0;i--) {
            CTMergeCell mergeCell = mergeCellList.get(i);
            final CellRefRange cellRefRange = Cell.createCellRefRange(mergeCell.getRef());
            if(cellRefRange==null) {
                log.warn("xlsx export: mergeRange is invalid");
            }
            else {
                if(cellRefRange.isInside(start.getInt(0), start.getInt(1))||cellRefRange.isInside(optEnd.getInt(0), optEnd.getInt(1))) {
                    mergeCellList.remove(i);
                }
            }
        }
        if(type.equals("merge")) {
            CTMergeCell mergeCell = Context.getsmlObjectFactory().createCTMergeCell();
            mergeCell.setRef(Cell.getCellRefRange(start.getInt(0), start.getInt(1), optEnd.getInt(0), optEnd.getInt(1)));
            mergeCellList.add(mergeCell);
        }
        else if(type.equals("horizontal")) {
            for(int row=start.getInt(1);row<=optEnd.getInt(1);row++) {
                CTMergeCell mergeCell = Context.getsmlObjectFactory().createCTMergeCell();
                mergeCell.setRef(Cell.getCellRefRange(start.getInt(0), row, optEnd.getInt(0), row));
                mergeCellList.add(mergeCell);
            }
        }
        else if(type.equals("vertical")) {
            for(int column=start.getInt(0);column<=optEnd.getInt(0);column++) {
                CTMergeCell mergeCell = Context.getsmlObjectFactory().createCTMergeCell();
                mergeCell.setRef(Cell.getCellRefRange(column, start.getInt(1), column, optEnd.getInt(1)));
                mergeCellList.add(mergeCell);
            }
        }
    }
}
