/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.tools;

import java.util.List;
import java.util.Vector;
import javax.xml.bind.JAXBException;
import org.docx4j.XmlUtils;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.wml.ContentAccessor;
import org.docx4j.wml.ObjectFactory;
import org.docx4j.wml.P;
import org.docx4j.wml.PPr;
import org.docx4j.wml.ParaRPr;
import org.docx4j.wml.R;
import org.docx4j.wml.RPr;
import org.docx4j.wml.RStyle;
import org.docx4j.wml.Text;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.jvnet.jaxb2_commons.ppp.Child;
import com.openexchange.office.ooxml.docx.OperationDocument;
import com.openexchange.office.ooxml.docx.tools.Component.ComponentContext;
import com.openexchange.office.ooxml.docx.tools.Component.ComponentContextWrapper;
import com.openexchange.office.ooxml.docx.tools.Component.DrawingComponent;
import com.openexchange.office.ooxml.docx.tools.Component.PictVMLComponent;
import com.openexchange.office.ooxml.docx.tools.Component.Placeholder_Base;
import com.openexchange.office.ooxml.docx.tools.Component.TextComponent;
import com.openexchange.office.ooxml.docx.tools.Component.TextRun_Base;

public class TextUtils{

    public static void preserveSpace(Text t) {
        if(t!=null) {
            String s = t.getValue();
            if(s!=null&&s.length()>0) {
                if(s.charAt(0)==' '||s.charAt(s.length()-1)==' ') {
                    t.setSpace("preserve");
                    return;
                }
            }
            t.setSpace(null);
        }
    }

    // just splitting text... the parent must be a textRun, the index into the textRun (textRunIndex) is allowed to be -1 (optional)
    public static void splitText(ObjectFactory objectFactory, Text text, int textSplitPosition, int textRunIndex) {
        if(text.getParent() instanceof R) {
            if(textSplitPosition>0&&textSplitPosition<text.getValue().length()) {
                Text newText = objectFactory.createText();
                newText.setParent(text.getParent());
                List<Object> runContent = ((R)text.getParent()).getContent();
                if(textRunIndex==-1)
                    textRunIndex = runContent.indexOf(text);
                if(textRunIndex!=-1) {
                    runContent.add(textRunIndex+1, newText);
                    StringBuffer s = new StringBuffer(text.getValue());
                    text.setValue(s.substring(0, textSplitPosition));
                    newText.setValue(s.substring(textSplitPosition));
                    preserveSpace(newText);
                    preserveSpace(text);
                }
            }
        }
    }

    // splitTextRun is needed to assure that the textSplitPosition is starting with a new textRun.
    public static void splitTextRun(ObjectFactory objectFactory, P paragraph, int textSplitPosition){

        Component.ParagraphContext paragraphContext = new Component.ParagraphContext(paragraph);
        while(paragraphContext.hasNext()) {
            Component component = paragraphContext.next();
            if(textSplitPosition>=component.getComponentNumber()&&textSplitPosition<component.getNextComponentNumber()) {
                int textRunIndex = ((TextRun_Base)component).getTextRunIndex();
                int split = textSplitPosition - component.getComponentNumber();
                if (split>0) {
                    // this must be a text component, its the onliest component where nextComponentNumber - componentNumber is greater than 1
                    if(component instanceof TextComponent) {
                        splitText(objectFactory, (Text)component.getObject(), split, textRunIndex);
                        textRunIndex++;     // the split text is now also inserted into the textRun -> textRun++
                    }
                }

                // the component is not first within the textRun, so we will split the run
                if(textRunIndex>0) {
                    R textRun = ((TextRun_Base)component).getTextRun();
                    RPr rPr = textRun.getRPr();
                    R textRunNew = objectFactory.createR();
                    textRunNew.setParent(textRun.getParent());
                    List<Object> l = Utils.getContent(textRun.getParent());
                    l.add(((TextRun_Base)component).getTextRunParentIndex()+1,textRunNew);
                    if(rPr!=null){
                        RPr rPrNew = XmlUtils.deepCopy(rPr);
                        rPrNew.setParent(textRunNew);
                        textRunNew.setRPr(rPrNew);
                    }
                    List<Object> rContent = Utils.getContent(textRun);
                    List<Object> rContentNew = Utils.getContent(textRunNew);
                    while(rContent.size()>textRunIndex) {
                        Object o = rContent.remove(textRunIndex);
                        if(o instanceof Child)
                            ((Child)o).setParent(textRunNew);
                        rContentNew.add(rContentNew.size(), o);
                    }
                    if(textRun.getParent() instanceof P.Hyperlink) {
                        P.Hyperlink hyper = (P.Hyperlink)textRun.getParent();
                        int hyperIndex = paragraph.getContent().indexOf(hyper);
                        if(hyperIndex>-1) {
                            P.Hyperlink hyperNew = objectFactory.createPHyperlink();
                            hyperNew.setAnchor(hyper.getAnchor());
                            hyperNew.setDocLocation(hyper.getDocLocation());
                            hyperNew.setHistory(hyper.isHistory());
                            hyperNew.setId(hyper.getId());
                            hyperNew.setParent(hyper.getParent());
                            hyperNew.setTgtFrame(hyper.getTgtFrame());
                            hyperNew.setTooltip(hyper.getTooltip());
                            paragraph.getContent().add(hyperIndex+1, hyperNew);
                            Object o = hyper.getContent().remove(((TextRun_Base)component).getTextRunParentIndex()+1);
                            hyperNew.getContent().add(o);
                            if(o instanceof Child) {
                                ((Child)o).setParent(hyperNew);
                            }
                        }
                    }
                }
                break;
            }
        }
    }

    public static void insertParagraph(Object parent, int index, P paragraph) {
        List<Object> content = Utils.getContent(parent);
        if (content!=null) {
            paragraph.setParent(parent);
            if (index<0)
                content.add(paragraph);
            else
                content.add(index, paragraph);
        }
    }

    public static void splitParagraph(ObjectFactory objectFactory, P paragraph, int textPosition) {

        // checking needed split for textPosition
        TextUtils.splitTextRun(objectFactory, paragraph, textPosition);

        // creating our new paragraph
        P destParagraph = objectFactory.createP();
        insertParagraph(paragraph.getParent(), Utils.getContent(paragraph.getParent()).indexOf(paragraph)+1, destParagraph);
        PPr sourceParagraphProperties = paragraph.getPPr();
        if(sourceParagraphProperties!=null){
            PPr destParagraphProperties = XmlUtils.deepCopy(sourceParagraphProperties);
            destParagraphProperties.setParent(destParagraph);
            destParagraph.setPPr(destParagraphProperties);
        }

        // moving textRuns from behind textPosition into our new paragraph
        Component.ParagraphContext paragraphContext = new Component.ParagraphContext(paragraph);
        RPr lastRPr = null;
        while(paragraphContext.hasNext()) {
            Component component = paragraphContext.next();
            if(textPosition>=component.getComponentNumber()&&textPosition<component.getNextComponentNumber()) {
                while(component.getContextIndex()<paragraph.getContent().size()) {
                    Object pElement = paragraph.getContent().get(component.getContextIndex());
                    if(pElement instanceof Child)
                        ((Child)pElement).setParent(destParagraph);
                    destParagraph.getContent().add(pElement);
                    paragraph.getContent().remove(component.getContextIndex());
                }
                break;
            } else if (component instanceof TextRun_Base) {
                lastRPr = ((TextRun_Base)component).getTextRun().getRPr();
            }
        }

        // taking care of paragraph attributes
        if(lastRPr!=null) {

            // if available, we have to get the character attributes from the last textrun
            PPr paraProps = destParagraph.getPPr();
            if(paraProps==null) {
                paraProps = objectFactory.createPPr();
                paraProps.setParent(destParagraph);
                destParagraph.setPPr(paraProps);
            }
            ParaRPr paraRPr = cloneRPrToParaRPr(objectFactory, lastRPr);
            if(paraRPr!=null) {
                paraRPr.setParent(paraProps);
                paraProps.setRPr(paraRPr);
            }
        }
    }

    public static RPr cloneParaRPrToRPr(ObjectFactory objectFactory, ParaRPr _paraRPr) {
        RPr rPr = null;
        if(_paraRPr!=null) {
            ParaRPr destParaProperties = XmlUtils.deepCopy(_paraRPr);
            if(destParaProperties!=null) {
                rPr = objectFactory.createRPr();
                rPr.setB(destParaProperties.getB());
                rPr.setBCs(destParaProperties.getBCs());
                rPr.setBdr(destParaProperties.getBdr());
                rPr.setCaps(destParaProperties.getCaps());
                rPr.setColor(destParaProperties.getColor());
                rPr.setCs(destParaProperties.getCs());
                rPr.setDstrike(destParaProperties.getDstrike());
                rPr.setEastAsianLayout(destParaProperties.getEastAsianLayout());
                rPr.setEffect(destParaProperties.getEffect());
                rPr.setEm(destParaProperties.getEm());
                rPr.setEmboss(destParaProperties.getEmboss());
                rPr.setFitText(destParaProperties.getFitText());
                rPr.setHighlight(destParaProperties.getHighlight());
                rPr.setI(destParaProperties.getI());
                rPr.setICs(destParaProperties.getICs());
                rPr.setImprint(destParaProperties.getImprint());
                rPr.setKern(destParaProperties.getKern());
                rPr.setLang(destParaProperties.getLang());
                rPr.setNoProof(destParaProperties.getNoProof());
                rPr.setOMath(destParaProperties.getOMath());
                rPr.setOutline(destParaProperties.getOutline());
                rPr.setPosition(destParaProperties.getPosition());
                rPr.setRFonts(destParaProperties.getRFonts());
                rPr.setRStyle(destParaProperties.getRStyle());
                rPr.setRtl(destParaProperties.getRtl());
                rPr.setShadow(destParaProperties.getShadow());
                rPr.setShd(destParaProperties.getShd());
                rPr.setSmallCaps(destParaProperties.getSmallCaps());
                rPr.setSnapToGrid(destParaProperties.getSnapToGrid());
                rPr.setSpacing(destParaProperties.getSpacing());
                rPr.setSpecVanish(destParaProperties.getSpecVanish());
                rPr.setStrike(destParaProperties.getStrike());
                rPr.setSz(destParaProperties.getSz());
                rPr.setSzCs(destParaProperties.getSzCs());
                rPr.setU(destParaProperties.getU());
                rPr.setVanish(destParaProperties.getVanish());
                rPr.setVertAlign(destParaProperties.getVertAlign());
                rPr.setW(destParaProperties.getW());
                rPr.setWebHidden(destParaProperties.getWebHidden());
            }
        }
        return rPr;
    }

    public static ParaRPr cloneRPrToParaRPr(ObjectFactory objectFactory, RPr _rPr) {
        ParaRPr rPr = null;
        if(_rPr!=null) {
            RPr destParaProperties = XmlUtils.deepCopy(_rPr);
            if(destParaProperties!=null) {
                rPr = objectFactory.createParaRPr();
                rPr.setB(destParaProperties.getB());
                rPr.setBCs(destParaProperties.getBCs());
                rPr.setBdr(destParaProperties.getBdr());
                rPr.setCaps(destParaProperties.getCaps());
                rPr.setColor(destParaProperties.getColor());
                rPr.setCs(destParaProperties.getCs());
                rPr.setDstrike(destParaProperties.getDstrike());
                rPr.setEastAsianLayout(destParaProperties.getEastAsianLayout());
                rPr.setEffect(destParaProperties.getEffect());
                rPr.setEm(destParaProperties.getEm());
                rPr.setEmboss(destParaProperties.getEmboss());
                rPr.setFitText(destParaProperties.getFitText());
                rPr.setHighlight(destParaProperties.getHighlight());
                rPr.setI(destParaProperties.getI());
                rPr.setICs(destParaProperties.getICs());
                rPr.setImprint(destParaProperties.getImprint());
                rPr.setKern(destParaProperties.getKern());
                rPr.setLang(destParaProperties.getLang());
                rPr.setNoProof(destParaProperties.getNoProof());
                rPr.setOMath(destParaProperties.getOMath());
                rPr.setOutline(destParaProperties.getOutline());
                rPr.setPosition(destParaProperties.getPosition());
                rPr.setRFonts(destParaProperties.getRFonts());
                rPr.setRStyle(destParaProperties.getRStyle());
                rPr.setRtl(destParaProperties.getRtl());
                rPr.setShadow(destParaProperties.getShadow());
                rPr.setShd(destParaProperties.getShd());
                rPr.setSmallCaps(destParaProperties.getSmallCaps());
                rPr.setSnapToGrid(destParaProperties.getSnapToGrid());
                rPr.setSpacing(destParaProperties.getSpacing());
                rPr.setSpecVanish(destParaProperties.getSpecVanish());
                rPr.setStrike(destParaProperties.getStrike());
                rPr.setSz(destParaProperties.getSz());
                rPr.setSzCs(destParaProperties.getSzCs());
                rPr.setU(destParaProperties.getU());
                rPr.setVanish(destParaProperties.getVanish());
                rPr.setVertAlign(destParaProperties.getVertAlign());
                rPr.setW(destParaProperties.getW());
                rPr.setWebHidden(destParaProperties.getWebHidden());
            }
        }
        return rPr;
    }

    public static void mergeParagraph(P paragraph) {
        Object parent = paragraph.getParent();
        List<Object> parentContent = Utils.getContent(parent);
        int sourceParagraphIndex = parentContent.indexOf(paragraph);
        for (int i = sourceParagraphIndex+1; i<parentContent.size();i++) {
            Object o = parentContent.get(i);
            if (o instanceof P) {
                List<Object> sourceContent = ((P)o).getContent();
                List<Object> destContent = paragraph.getContent();
                for(Object so:sourceContent){
                    if(so instanceof Child)
                        ((Child)so).setParent(paragraph);
                    destContent.add(so);
                }
                parentContent.remove(i);
                break;
            }
        }
    }

    public static void insertText(ObjectFactory objectFactory, P paragraph, int textPosition, String text) {

        Text t = null;

        Component.ParagraphContext paragraphContext = new Component.ParagraphContext(paragraph);
        if(!paragraphContext.hasNext()) {

            // the paragraph is empty, we have to create R and its text
            R newRun = objectFactory.createR();
            newRun.setParent(paragraph);
            if(paragraph.getPPr()!=null) {
                RPr rPr = cloneParaRPrToRPr(objectFactory, paragraph.getPPr().getRPr());
                if(rPr!=null) {
                    rPr.setParent(newRun);
                    newRun.setRPr(rPr);
                }
            }
            paragraph.getContent().add(newRun);
            t = objectFactory.createText();
            t.setParent(newRun);
            t.setValue(text);
            newRun.getContent().add(t);
        }
        else {

            while(paragraphContext.hasNext()) {
                Component component = paragraphContext.next();
                if(textPosition>=component.getComponentNumber()&&textPosition<=component.getNextComponentNumber()) {

                    // check if the character could be inserted into an existing text:
                    if(component instanceof TextComponent) {
                        t = (Text)component.getObject();
                        StringBuffer s = new StringBuffer(t.getValue());
                        s.insert(textPosition-((TextComponent)component).getComponentNumber(), text);
                        t.setValue(s.toString());
                    }
                    else {

                        TextRun_Base runBase = (TextRun_Base)component;
                        if(runBase.isSingleComponentRun()) {
                            // new run needs to be created ... (simpleField...)
                            R newRun = objectFactory.createR();
                            newRun.setParent(paragraph);
                            RPr sourceRPr = runBase.getTextRun().getRPr();
                            if(sourceRPr!=null) {
                                RPr newRPr = XmlUtils.deepCopy(sourceRPr);
                                newRPr.setParent(newRun);
                                newRun.setRPr(newRPr);
                            }
                            t = objectFactory.createText();
                            t.setParent(newRun);
                            t.setValue(text);
                            newRun.getContent().add(t);
                            ContentAccessor contextObject = (ContentAccessor)runBase.getContextObject();
                            newRun.setParent(contextObject);
                            contextObject.getContent().add(textPosition == component.getComponentNumber() ? runBase.getContextIndex() : runBase.getContextIndex() + 1, newRun);
                        }
                        else {

                            // we can insert the text into a existing run
                            R oldRun = runBase.getTextRun();
                            t = objectFactory.createText();
                            t.setParent(oldRun);
                            t.setValue(text);
                            oldRun.getContent().add(textPosition == component.getComponentNumber() ? runBase.getTextRunIndex() : runBase.getTextRunIndex() + 1, t);
                        }
                    }
                    break;
                }
            }
        }
        preserveSpace(t);
    }

    public static void deleteText(ObjectFactory objectFactory, P paragraph, int textStartPosition, int textEndPosition) {

        // TODO: there is no need to split TextRuns if deleting only one component, for this we should have an optimized version of deleteText.
        // the current version is complex and is deleting also objects in textRuns we are not supporting and which are positioned between textStart
        // and textEndPosition...

        // checking needed split for textStartPosition
        splitTextRun(objectFactory, paragraph, textStartPosition);

        // checking needed split for textEndPosition
        splitTextRun(objectFactory, paragraph, textEndPosition+1);

        Vector<Component> delList = new Vector<Component>();
        Component.ParagraphContext paragraphContext = new Component.ParagraphContext( paragraph );
        while (paragraphContext.hasNext()) {
            Component component = paragraphContext.next();
            if ((component.getComponentNumber()>=textStartPosition)&&
                (component.getNextComponentNumber()-1<=textEndPosition))
                delList.add(component);
        }
        // removing textRuns from the back, so indexes remain valid.
        for (int i=delList.size()-1; i>=0; i--) {
            Component component = delList.get(i);
            if(component instanceof TextRun_Base) {
                // TODO: removing single TextComponents (simpleField)
                R textRun = ((TextRun_Base)component).getTextRun();
                List<Object> textRunContent = textRun.getContent();
                textRunContent.remove(((TextRun_Base)component).getTextRunIndex());
                if(textRunContent.size()==0) {                                              // is R completely to be removed from the parent ?
                    Object textRunParent = textRun.getParent();
                    if(textRunParent instanceof P) {
                        paragraph.getContent().remove(component.getContextIndex());         // removing R from paragraph
                    }
                    else if(textRunParent instanceof P.Hyperlink) {
                        ((P.Hyperlink)textRunParent).getContent().remove(((TextRun_Base)component).getTextRunParentIndex());    // removing R from P.Hyperlink
                        if(((P.Hyperlink)textRunParent).getContent().size()==0)
                            paragraph.getContent().remove(component.getContextIndex());     // removing P.Hyperlink from the paragraph
                    }
                }
            }
        }
    }

    private static P.Hyperlink cloneHyperlinkWithoutContent(ObjectFactory objectFactory, P.Hyperlink source) {

        P.Hyperlink dest = objectFactory.createPHyperlink();
        dest.setAnchor(source.getAnchor());
        dest.setDocLocation(source.getDocLocation());
        dest.setId(source.getId());
        dest.setTooltip(source.getTooltip());
        dest.setHistory(source.isHistory());
        dest.setParent(source.getParent());
        return dest;
    }

    public static void setTextRunAttributes(OperationDocument operationDocument, JSONObject attrs, ObjectFactory objectFactory, P paragraph, int textStartPosition, int textEndPosition)
        throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

        JSONObject characterAttributes = attrs.optJSONObject("character");
        Object styleId = attrs.opt("styleId");

        Vector<TextRun_Base> urlComponentList = characterAttributes!=null && characterAttributes.has("url") ? new Vector<TextRun_Base>() : null;

        // checking needed split for textStartPosition
        splitTextRun(objectFactory, paragraph, textStartPosition);

        // checking needed split for textEndPosition
        splitTextRun(objectFactory, paragraph, textEndPosition+1);

        // now we can traverse over the textruns
        Component.ParagraphContext paragraphContext = new Component.ParagraphContext(paragraph);
        while(paragraphContext.hasNext()) {
            Component component = paragraphContext.next();
            if(component.getComponentNumber()<textStartPosition)
                continue;
            if(component.getComponentNumber()>textEndPosition)
                break;

            if(component instanceof TextRun_Base) {
                if(component instanceof Placeholder_Base) {
                    if(component instanceof DrawingComponent)
                        Drawings.applyProperties(attrs.optJSONObject("drawing"), (DrawingComponent)component, operationDocument.getPackage(), operationDocument.getResourceManager());
                    else if(component instanceof PictVMLComponent)
                        PictVML.applyProperties(attrs.optJSONObject("drawing"), (PictVMLComponent)component);
                }
                R textRun = ((TextRun_Base)component).getTextRun();
                RPr rPr = textRun.getRPr();
                if (rPr==null) {
                    rPr = objectFactory.createRPr();
                    rPr.setParent(textRun);
                    textRun.setRPr(rPr);
                }
                if(styleId!=null) {
                    if (styleId instanceof String) {
                        RStyle rStyle = rPr.getRStyle();
                        if(rStyle==null) {
                            rStyle = objectFactory.createRStyle();
                            rStyle.setParent(rPr);
                        }
                        rStyle.setVal((String)styleId);
                        rPr.setRStyle(rStyle);
                    }
                    else {
                        rPr.setRStyle(null);
                    }
                }
                Character.applyCharacterProperties(operationDocument, objectFactory, characterAttributes, rPr);
                if(urlComponentList!=null)
                    urlComponentList.add((TextRun_Base)component);
            }
        }

        if(urlComponentList!=null&&urlComponentList.size()>0) {

            // first step...we will remove each P.Hyperlink that is addressed via textStart->textEndPosition
            // iterating from back, so indexes within components in urlComponentList remains valid
            for(int i=urlComponentList.size()-1;i>=0;i--) {
                TextRun_Base textRunComponent = urlComponentList.get(i);
                R textRun = textRunComponent.getTextRun();
                Object textRunParent = textRun.getParent();
                if(textRunParent instanceof P.Hyperlink) {
                    List<Object> hyperlinkContent = ((P.Hyperlink)textRunParent).getContent();
                    int hyperlinkIndex = textRunComponent.getTextRunParentIndex();

                    // check if Hyperlink Split is needed
                    int textRunsBehind = hyperlinkContent.size()-(hyperlinkIndex+1);
                    if(textRunsBehind>0) {
                        // splitting old hyperlink
                        P.Hyperlink splitAfterHyperlink = cloneHyperlinkWithoutContent(objectFactory, (P.Hyperlink)textRunParent);
                        paragraph.getContent().add(textRunComponent.getContextIndex()+1, splitAfterHyperlink);
                        for(int nCount=0;nCount<textRunsBehind;nCount++) {
                            Object o = hyperlinkContent.remove(hyperlinkIndex+1);
                            if(o instanceof Child)
                                ((Child) o).setParent(splitAfterHyperlink);
                            splitAfterHyperlink.getContent().add(nCount, o);
                        }
                    }
                    // moving TextRun
                    paragraph.getContent().add(textRunComponent.getContextIndex()+1, hyperlinkContent.remove(hyperlinkIndex));
                    textRun.setParent(paragraph);

                    // check if P.Hyperlink still has content...
                    if(hyperlinkContent.size()==0) {
                        paragraph.getContent().remove(textRunComponent.getContextIndex());
                    }
                }
            }

            // second step... only necessary if url!=null... then we will create a new P.Hyperlink for each component in urlComponentList
            @SuppressWarnings("null")
            Object value = characterAttributes.opt("url");
            if(value instanceof String) {
                P.Hyperlink hyperlink = objectFactory.createPHyperlink();
                hyperlink.setParent(paragraph);
                hyperlink.setId(Utils.setUrl(operationDocument.getPackage().getMainDocumentPart(), hyperlink.getId(), (String)value));
                int startIndex = urlComponentList.get(0).getContextIndex();
                List<Object> paragraphContent = paragraph.getContent();
                if(paragraphContent.get(startIndex) instanceof P.Hyperlink)
                    startIndex++;
                for(int i=0; i<urlComponentList.size();i++) {
                    TextRun_Base component = urlComponentList.get(i);
                    R textRun = component.getTextRun();
                    if(textRun.getParent() instanceof P) {
                        int x = paragraphContent.indexOf(textRun);
                        if(x!=-1) {
                            paragraphContent.remove(x);
                            textRun.setParent(hyperlink);
                            hyperlink.getContent().add(hyperlink.getContent().size(), textRun);
                        }
                    }
                }
                paragraphContent.add(startIndex, hyperlink);
            }
        }
    }

    /**
     * @param contextObject (ContentAccessor) up from where components are taken
     * @param position component number that should be returned
     * @return Component or null if position could not be found
     */
    public static Component getPosition(ComponentContext componentContext, int position) {

        Component ret = null;
        while(componentContext.hasNext()) {
            Component component = componentContext.next();
            if(component.getNextComponentNumber()<=position)
                continue;
            ret = component;
            break;
        }
        return ret;
    }

    /**
     * @param contextObject (ContentAccessor) up from where components are taken (in general this is the documentPart)
     * @param position holds the position of the component that should be returned
     * @param positionCount the number of positions that should be used from the position array (in general position.length())
     * @return Component or null if position could not be found
     */
    public static Component getPosition(ComponentContext componentContext, JSONArray position, int positionCount) {
        return getPosition(new ComponentContextWrapper(componentContext), position, positionCount);
    }

    public static Component getPosition(ComponentContextWrapper contextWrapper, JSONArray position, int positionCount) {

        Component o = null;
        try {
            for(int i=0; i<positionCount;i++) {
                if(o!=null) {
                    contextWrapper.setComponentContext(o.createComponentContext());
                }
                o = getPosition(contextWrapper.getComponentContext(), position.getInt(i));
                if (o==null) {
                    return null;
                }
            }
            return o;
        }
        catch(JSONException e) {
            // ups
        }
        return null;
    }



    /**
     * @param contextObject contextObject (ContentAccessor) up from where components are taken (in general this is the documentPart)
     * @param component number that should be returned
     * @return Component or null if position could not be found. A special DummyPlaceholderComponent is returned if position
     *         describes a position at the end of the list where no component could be found, but new components can be appended.
     */
    public static Component getInsertPosition(ComponentContext componentContext, int insertPosition) {

        Component ret = null;
        int position = 0;
        while(componentContext.hasNext()) {
            Component component = componentContext.next();
            position = component.getNextComponentNumber();
            if(position<=insertPosition)
                continue;
            ret = component;
            break;
        }
        if(ret==null&&position==insertPosition) {
            Object contextObject = componentContext.getContextObject();
            List<Object> content = Utils.getContent(contextObject);
            if(content!=null) {
                ret = new Component.DummyPlaceholderComponent(contextObject, content.size(), insertPosition);
            }
        }
        return ret;
    }

    /**
     * @param contextObject (ContentAccessor) up from where components are taken (in general this is the documentPart)
     * @param position holds the position of the component that should be returned
     * @return Component or null if position could not be found. A special DummyPlaceholderComponent is returned if position
     *         describes a position at the end of the list where no component could be found, but new components can be appended.
     */

    public static Component getInsertPosition(ComponentContext componentContext, JSONArray insertPosition) {
        return getInsertPosition(new ComponentContextWrapper(componentContext), insertPosition);
    }

    public static Component getInsertPosition(ComponentContextWrapper contextWrapper, JSONArray insertPosition) {
        Component o = null;
        try {
            for(int i=0; i<insertPosition.length();i++) {
                if(o!=null) {
                    contextWrapper.setComponentContext(o.createComponentContext());
                }
                o = getInsertPosition(contextWrapper.getComponentContext(), insertPosition.getInt(i));
                if (o==null) {
                    return null;
                }
            }
            return o;
        }
        catch(JSONException e) {
            // ups
        }
        return null;
    }
}
