/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.operations;

import java.util.ArrayList;
import java.util.List;
import org.docx4j.openpackaging.packages.SpreadsheetMLPackage;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorkbookPart;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorksheetPart;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.xlsx4j.sml.CTCellFormula;
import org.xlsx4j.sml.CTRst;
import org.xlsx4j.sml.Cell;
import org.xlsx4j.sml.Row;
import org.xlsx4j.sml.STCellFormulaType;
import org.xlsx4j.sml.STCellType;
import org.xlsx4j.sml.Sheet;
import org.xlsx4j.sml.SheetData;
import org.xlsx4j.sml.Sheets;
import org.xlsx4j.sml.Workbook;
import org.xlsx4j.sml.Worksheet;
import com.openexchange.office.ooxml.tools.Commons;
import com.openexchange.office.ooxml.xlsx.OperationDocument;
import com.openexchange.office.ooxml.xlsx.tools.SheetUtils;

public class CreateOperationHelper extends com.openexchange.office.ooxml.operations.CreateOperationHelper {

    private final SpreadsheetMLPackage spreadsheetMLPackage;

    public CreateOperationHelper(OperationDocument _operationDocument, JSONArray mOperationsArray) {
        super(_operationDocument, mOperationsArray);
        spreadsheetMLPackage = _operationDocument.getPackage();
    }

    @Override
    public OperationDocument getOperationDocument() {
        return (OperationDocument)operationDocument;
    }

    @Override
    public void createDocumentDefaults(String userLanguage) {

    }

    @Override
    public void createOperations()
        throws JSONException {

        WorkbookPart workbookPart = spreadsheetMLPackage.getWorkbookPart();
        Workbook workbook = workbookPart.getJaxbElement();
        Sheets sheets = workbook.getSheets();
        List<Sheet> sheetList = sheets.getSheet();
        for(int sheetIndex = 0; sheetIndex < sheetList.size(); sheetIndex++) {
            createSheetOperations(sheetList.get(sheetIndex), sheetIndex);
        }
    }

    static List<Integer> getCellRef(String cellRef) {

        if(cellRef==null)
            return null;

        List<Integer> cellPosition = null;

        int characters = 0;
        int characterStartIndex = 0;

        while((characters+characterStartIndex)<cellRef.length()) {
            char character = cellRef.charAt(characterStartIndex+characters);
            if((character<'A')||(character>'Z')) {
                break;
            }
            characters++;
        }
        if(characters>0&&characters<cellRef.length()&&characters<6) {

            final int[] exMap = { 1, 26, 676, 17576, 456976 };

            int ex = characters - 1;
            int x = 0;

            for(int i = 0; i < characters; i++) {
                x +=((cellRef.charAt(i+characterStartIndex)-'A')+1)*exMap[ex-i];
            }

            int numbers = 0;
            int numberStartIndex = characterStartIndex + characters;
            while(numbers+numberStartIndex<cellRef.length()) {
                char character = cellRef.charAt(numberStartIndex+numbers);
                if((character<'0')||(character>'9')) {
                    break;
                }
                numbers++;
            }
            if(numbers>0) {
                int y = 0;
                for(int i = 0; i < numbers; i++) {
                    y*=10;
                    y +=cellRef.charAt(i+numberStartIndex)-'0';
                }
                if(y>0) {
                    cellPosition = new ArrayList<Integer>();
                    cellPosition.add(x-1);
                    cellPosition.add(y-1);
                }
            }
        }
        return cellPosition;
    }

    private JSONObject getCellObject(Cell cell) throws JSONException {

        JSONObject cellObject = new JSONObject();
        CTCellFormula cellFormula = cell.getF();
        if(cellFormula!=null&&(cellFormula.getT()==STCellFormulaType.NORMAL||cellFormula.getT()==STCellFormulaType.SHARED)) {
            if(cellFormula.getT()==STCellFormulaType.SHARED) {
                cellObject.put("shared", cellFormula.getSi());
                List<Integer> cellRef = getCellRef(cellFormula.getRef());
                if(cellRef!=null) {
                    cellObject.put("ref", cellRef);
                }
            }
            if(cellFormula.getValue()!=null&&cellFormula.getValue().length()>0)
                cellObject.put("value", "=" + cellFormula.getValue());
        }
        else if (cell.getT()==STCellType.INLINE_STR) {
            CTRst is = cell.getIs();
            if(is!=null) {
                // TODO: CTRst can include formatted text runs, but we will support simple t first.
                if(is.getT()!=null) {
                    cellObject.put("value",  is.getT());
                }
            }
        }
        else if(cell.getV()!=null) {
            if(cell.getT()==STCellType.N) {
                try {

                    double number = Double.parseDouble(cell.getV());
                    cellObject.put("value", number);
                }
                catch(NumberFormatException e) {
                    // TODO: ups
                }
            }
            else if(cell.getT()==STCellType.S) {
                String sharedString = getOperationDocument().getSharedString(cell.getV());
                if(sharedString!=null) {
                    cellObject.put("value", getOperationDocument().escapeString(sharedString));
                }
            }
            else if(cell.getT()==STCellType.B) {
                if(cell.getV().equals("0")||cell.getV().toLowerCase().equals("false")) {
                    cellObject.put("value", false);
                }
                else if(cell.getV().equals("1")||cell.getV().toLowerCase().equals("true")) {
                    cellObject.put("value", true);
                }
            }
            else {  // STCellType.STR, STCellType.E or STCellType.F (ARRAY or DATA_TABLE format)
                cellObject.put("value",  cell.getV());
            }
        }
        return cellObject;
    }

    private void createSheetOperations(final Sheet sheet, int sheetIndex)
        throws JSONException {

        addInsertSheetOperation(sheetIndex, sheet.getName(), Commons.surroundJSONObject("sheet", SheetUtils.createSheetProperties(sheet)));

        RelationshipsPart relationshipPart = spreadsheetMLPackage.getWorkbookPart().getRelationshipsPart();
        WorksheetPart worksheetPart = (WorksheetPart)relationshipPart.getPart(sheet.getId());
        Worksheet worksheet = worksheetPart.getJaxbElement();
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData!=null) {
            createRowOperations(sheetData.getRow(), sheetIndex);
        }
    }

    private void createRowOperations(List<Row> rows, int sheetIndex)
        throws JSONException {

        if(rows!=null) {
            for(Row row:rows) {
                JSONArray rowArray = new JSONArray();
                List<Integer> topLeftCellPosition = null;
                List<Cell> cells = row.getC();
                if(cells!=null) {
                    int lastCellX = -1;
                    JSONArray cellArray = new JSONArray();
                    for(Cell cell:cells) {
                        List<Integer> cellPosition = getCellRef(cell.getR().toUpperCase());     // TODO: cellRef is optional...
                        if(cellPosition!=null) {
                            JSONObject cellObject = getCellObject(cell);
                            if(topLeftCellPosition==null) {
                                // first cell in our array
                                topLeftCellPosition = cellPosition;
                            }
                            else {
                                // check if empty cells are to be written and start with a new setCellContentOperation if needed
                                int emptyCells = cellPosition.get(0) - lastCellX;
                                if(emptyCells>3) {
                                    // this are too many empty cells, we will create an additional setContent operation
                                    if(cellArray.length()>0) {
                                        // we have to write back our current content and start a new collection
                                        rowArray.put(cellArray);
                                        addSetCellContentsOperation(sheetIndex, topLeftCellPosition, rowArray);
                                        rowArray = new JSONArray();
                                        cellArray = new JSONArray();
                                    }
                                    topLeftCellPosition = cellPosition;
                                }
                                else {
                                    while(--emptyCells>0) {
                                        cellArray.put((Object)null);
                                    }
                                }
                            }
                            cellArray.put(cellObject);
                            lastCellX = cellPosition.get(0);
                        }
                    }
                    rowArray.put(cellArray);
                    if(topLeftCellPosition!=null)
                        addSetCellContentsOperation(sheetIndex, topLeftCellPosition, rowArray);
                }
            }
        }
    }

    public void addInsertSheetOperation(int sheetIndex, final String sheetName, final JSONObject attrs)
        throws JSONException {

        final JSONObject insertSheetObject = new JSONObject();
        insertSheetObject.put("name", "insertSheet");
        insertSheetObject.put("sheet", sheetIndex);
        insertSheetObject.put("sheetName", sheetName);
        if(attrs!=null)
            insertSheetObject.put("attrs", attrs);

        operationsArray.put(insertSheetObject);
    }

    public void addSetCellContentsOperation(int sheetIndex, List<Integer> cellPosition, JSONArray cellContent)
        throws JSONException {

        final JSONObject insertCellsObject = new JSONObject();
        insertCellsObject.put("name", "setCellContents");
        insertCellsObject.put("sheet", sheetIndex);
        insertCellsObject.put("start", cellPosition);
        insertCellsObject.put("contents", cellContent);

        operationsArray.put(insertCellsObject);
    }
}
