/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.tools;

import java.math.BigInteger;
import java.util.Iterator;
import java.util.List;
import javax.xml.bind.JAXBElement;
import org.docx4j.openpackaging.packages.WordprocessingMLPackage;
import org.docx4j.wml.CTBorder;
import org.docx4j.wml.CTHeight;
import org.docx4j.wml.CTShd;
import org.docx4j.wml.CTShortHexNumber;
import org.docx4j.wml.CTTblCellMar;
import org.docx4j.wml.CTTblLayoutType;
import org.docx4j.wml.CTTblPrBase;
import org.docx4j.wml.CTTblPrBase.TblStyle;
import org.docx4j.wml.CTTblPrBase.TblStyleColBandSize;
import org.docx4j.wml.CTTblPrBase.TblStyleRowBandSize;
import org.docx4j.wml.ContentAccessor;
import org.docx4j.wml.ObjectFactory;
import org.docx4j.wml.STBorder;
import org.docx4j.wml.STTblLayoutType;
import org.docx4j.wml.SectPr;
import org.docx4j.wml.SectPr.PgMar;
import org.docx4j.wml.SectPr.PgSz;
import org.docx4j.wml.Tbl;
import org.docx4j.wml.TblBorders;
import org.docx4j.wml.TblGrid;
import org.docx4j.wml.TblGridCol;
import org.docx4j.wml.TblPr;
import org.docx4j.wml.TblWidth;
import org.docx4j.wml.Tc;
import org.docx4j.wml.TcMar;
import org.docx4j.wml.TcPr;
import org.docx4j.wml.TcPrInner;
import org.docx4j.wml.TcPrInner.GridSpan;
import org.docx4j.wml.TcPrInner.TcBorders;
import org.docx4j.wml.TrPr;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.office.ooxml.docx.OperationDocument;
import com.openexchange.office.ooxml.docx.tools.Component.TableContext;
import com.openexchange.office.ooxml.docx.tools.Component.TableRowContext;
import com.openexchange.office.ooxml.docx.tools.Component.TcComponent;
import com.openexchange.office.ooxml.tools.Commons;

public class Table {

    public static void insertTable(OperationDocument operationDocument, Object parent, int index, ObjectFactory objectFactory, JSONObject attrs)
        throws JSONException {

        if (parent instanceof ContentAccessor) {
            Tbl tbl = objectFactory.createTbl();
            tbl.setParent(parent);
            TblPr tblPr = getTableProperties(objectFactory, tbl);

            // turning on each conditional Style, each of our conditional styles has to be used by default,
            // (they can be deactivated by using the exclude property)
            CTShortHexNumber hexNumber = objectFactory.createCTShortHexNumber();
            hexNumber.setParent(tblPr);
            hexNumber.setVal("1E0");
            tblPr.setTblLook(hexNumber);

            // RowBandSize and ColBandSize needs to be set to 1, otherwise the
            // conditional table styles "band1Vert, band2Vert..." will be ignored
            // by word.
            TblStyleRowBandSize rowBandSize = objectFactory.createCTTblPrBaseTblStyleRowBandSize();
            rowBandSize.setParent(tblPr);
            rowBandSize.setVal(BigInteger.valueOf(1));
            tblPr.setTblStyleRowBandSize(rowBandSize);

            TblStyleColBandSize colBandSize = objectFactory.createCTTblPrBaseTblStyleColBandSize();
            colBandSize.setParent(tblPr);
            colBandSize.setVal(BigInteger.valueOf(1));
            tblPr.setTblStyleColBandSize(colBandSize);

            if (index<0)
                ((ContentAccessor)parent).getContent().add(tbl);
            else
                ((ContentAccessor)parent).getContent().add(index, tbl);

            if(attrs!=null) {
                if(attrs.has("styleId")) {
                    Table.applyTableStyle(objectFactory, attrs.getString("styleId"), tbl);
                }
                Table.applyTableProperties(operationDocument, objectFactory, attrs.optJSONObject("table"), tbl, null);
            }
        }
    }


    /**
     * @param objectFactory
     * @param tbl
     * @return TblPr are returned (they are created if not existing)
     */
    public static TblPr getTableProperties(ObjectFactory objectFactory, Tbl tbl) {
        TblPr tblPr = tbl.getTblPr();
        if(tblPr==null) {
            tblPr = objectFactory.createTblPr();
            tblPr.setParent(tbl);
            tbl.setTblPr(tblPr);
        }
        return tblPr;
    }

    public static JSONArray getTableGrid(Tbl tbl) {

        JSONArray tableGrid = null;

        TblGrid tblGrid = tbl.getTblGrid();
        if (tblGrid!=null) {
            List<TblGridCol> gridCols = tblGrid.getGridCol();
            if(gridCols!=null) {
                tableGrid = new JSONArray();
                for(TblGridCol gridCol:gridCols) {
                    BigInteger w = gridCol.getW();
                    tableGrid.put(w!=null?w.longValue()*254/144:1);
                }
            }
        }
        return tableGrid;
    }

    public static void setTableGrid(ObjectFactory objectFactory, Tbl tbl, JSONArray tableGrid)
        throws JSONException {

        if (tbl == null)
            return;

        TblGrid tblGrid = tbl.getTblGrid();
        if (tblGrid==null) {
            tblGrid = objectFactory.createTblGrid();
            tblGrid.setParent(tbl);
            tbl.setTblGrid(tblGrid);
        }
        List<TblGridCol> gridCols = tblGrid.getGridCol();
        gridCols.clear();
        if(tableGrid!=null) {
            for(int i=0; i<tableGrid.length(); i++) {
                TblGridCol tblGridCol = objectFactory.createTblGridCol();
                tblGridCol.setW(BigInteger.valueOf(tableGrid.getInt(i)*144/254));
                gridCols.add(tblGridCol);
            }
        }
    }

    public static void applyTableStyle(ObjectFactory objectFactory, String styleId, Tbl tbl) {
        TblPr tblPr = getTableProperties(objectFactory, tbl);
        if (styleId!=null) {
            TblStyle tblStyle = tblPr.getTblStyle();
            if (tblStyle==null) {
                tblStyle = objectFactory.createCTTblPrBaseTblStyle();
                tblStyle.setParent(tblPr);
            }
            tblStyle.setVal(styleId);
            tblPr.setTblStyle(tblStyle);
        }
        else {
            tblPr.setTblStyle(null);
        }
    }

    public static void applyTableProperties(OperationDocument operationDocument, ObjectFactory objectFactory, JSONObject tableProperties, Tbl tbl, CTTblPrBase tblPr)
        throws JSONException {

        if(tableProperties==null)
            return;

        // we must have a tbl or tblPr
        if ((tbl == null) && (tblPr == null))
            return;

        if (tbl != null)
            tblPr = getTableProperties(objectFactory, tbl);

        Iterator<String> keys = tableProperties.keys();
        while(keys.hasNext()) {
            String attr = keys.next();
            Object value = tableProperties.get(attr);

            if(attr.equals("tableGrid")) {
                if (value instanceof JSONArray) {
                    setTableGrid(objectFactory, tbl, (JSONArray)value);
                }
            } else if(attr.equals("width")) {
                tblPr.setTblW(createTblWidthFromJSON(objectFactory, tblPr, value));

                CTTblLayoutType tblLayout = tblPr.getTblLayout();
                if(tblLayout==null) {
                    tblLayout = objectFactory.createCTTblLayoutType();
                    tblLayout.setParent(tblPr);
                    tblPr.setTblLayout(tblLayout);
                }
                tblLayout.setType(STTblLayoutType.FIXED);
            }
            else if(attr.equals("borderLeft"))
                getTblBorders(objectFactory, tblPr).setLeft(createCTBorderFromJSON(operationDocument, objectFactory, tblPr, value));
            else if(attr.equals("borderTop"))
                getTblBorders(objectFactory, tblPr).setTop(createCTBorderFromJSON(operationDocument, objectFactory, tblPr, value));
            else if(attr.equals("borderRight"))
                getTblBorders(objectFactory, tblPr).setRight(createCTBorderFromJSON(operationDocument, objectFactory, tblPr, value));
            else if(attr.equals("borderBottom"))
                getTblBorders(objectFactory, tblPr).setBottom(createCTBorderFromJSON(operationDocument, objectFactory, tblPr, value));
            else if(attr.equals("borderInsideHor"))
                getTblBorders(objectFactory, tblPr).setInsideH(createCTBorderFromJSON(operationDocument, objectFactory, tblPr, value));
            else if(attr.equals("borderInsideVert"))
                getTblBorders(objectFactory, tblPr).setInsideV(createCTBorderFromJSON(operationDocument, objectFactory, tblPr, value));
            else if(attr.equals("paddingLeft"))
                getTblCellMar(objectFactory, tblPr).setLeft(createTblWidthFromJSON(objectFactory, tblPr, value));
            else if(attr.equals("paddingTop"))
                getTblCellMar(objectFactory, tblPr).setTop(createTblWidthFromJSON(objectFactory, tblPr, value));
            else if(attr.equals("paddingRight"))
                getTblCellMar(objectFactory, tblPr).setRight(createTblWidthFromJSON(objectFactory, tblPr, value));
            else if(attr.equals("paddingBottom"))
                getTblCellMar(objectFactory, tblPr).setBottom(createTblWidthFromJSON(objectFactory, tblPr, value));
            else if(attr.equals("fillColor")) {
                if(value instanceof JSONObject) {
                    CTShd shd = tblPr.getShd();
                    if(shd==null) {
                        shd = objectFactory.createCTShd();
                        shd.setParent(tblPr);
                        tblPr.setShd(shd);
                    }
                    Utils.initShdFromJSONColor(operationDocument, (JSONObject)value, shd);

                    // TableFillColor is not preserved (this is a bug)... we have to set this
                    // color at each cell that is not having its own hard color value,
                    TableContext tableContext = new TableContext(tbl);
                    while(tableContext.hasNext()) {
                        Component tableRowComponent = tableContext.next();
                        TableRowContext tableRowContext = (TableRowContext)tableRowComponent.getComponentContext();
                        while(tableRowContext.hasNext()) {
                            Tc tc = ((Tc)((TcComponent)tableRowContext.next()).getObject());
                            TcPr tcPr = tc.getTcPr();
                            if(tcPr==null) {
                                tcPr = objectFactory.createTcPr();
                                tcPr.setParent(tc);
                                tc.setTcPr(tcPr);
                            }
                            shd = tcPr.getShd();
                            if(shd==null) {
                                shd = objectFactory.createCTShd();
                                shd.setParent(tcPr);
                                tcPr.setShd(shd);
                            }
                            Utils.initShdFromJSONColor(operationDocument, (JSONObject)value, shd);
                        }
                    }
                }
                else
                    tblPr.setShd(null);
            }
            else if(attr.equals("exclude")) {
                CTShortHexNumber tblLook = objectFactory.createCTShortHexNumber();
                tblLook.setParent(tblPr);
                if(value instanceof JSONArray) {
                    short val = 0x1e0;
                    Iterator<Object> jsonArray = ((JSONArray)value).iterator();
                    while(jsonArray.hasNext()) {
                        Object o = jsonArray.next();
                        if(o instanceof String) {
                            if(o.equals("firstRow")) {
                                val&=~0x20;
                            } else if(o.equals("lastRow")) {
                                val&=~0x40;
                            } else if(o.equals("firstCol")) {
                                val&=~0x80;
                            } else if(o.equals("lastCol")) {
                                val&=~0x100;
                            } else if(o.equals("bandsHor")) {
                                val|=0x200;
                            } else if(o.equals("bandsVert")) {
                                val|=0x400;
                            }
                        }
                    }
                    tblLook.setVal(Integer.toHexString(val));
                }
                else
                    tblLook.setVal("01E0");
                tblPr.setTblLook(tblLook);
            }
        }
    }

    public static JSONObject createTableProperties(WordprocessingMLPackage document, CTTblPrBase tableProperties, JSONArray tableGrid, boolean isRootTable, int firstRowWidth)
        throws JSONException {

        JSONObject jsonTableProperties = new JSONObject();
        if(tableGrid!=null)
            jsonTableProperties.put("tableGrid", tableGrid);
        if(tableProperties!=null) {
            CTTblLayoutType ctLayoutType = tableProperties.getTblLayout();
            @SuppressWarnings("unused")
            STTblLayoutType tableLayoutType = ctLayoutType!=null ? ctLayoutType.getType() : STTblLayoutType.AUTOFIT;    // default = AUTOFIT
                                                                                                                        //           FIXED
            long referenceWidth = 0;

            if(isRootTable) {

                referenceWidth = 19000;
                // retrieving document size
                SectPr sectPr = Utils.getDocumentProperties(document.getMainDocumentPart());
                if(sectPr!=null) {
                    PgMar pgMar = sectPr.getPgMar();
                    PgSz pgSz = sectPr.getPgSz();
                    if(pgSz!=null&&pgMar!=null) {
                        long sz = pgSz.getW().longValue();
                        sz -= pgMar.getLeft().longValue();
                        sz -= pgMar.getRight().longValue();
                        referenceWidth = sz * 2540 / 72 / 20;
                    }
                }
            }
            Utils.jsonPut(jsonTableProperties, "width", createJSONFromTblWidth(tableProperties.getTblW(), referenceWidth, firstRowWidth));

            // creating table borders
            TblBorders tblBorders = tableProperties.getTblBorders();
            if(tblBorders!=null) {
                Utils.jsonPut(jsonTableProperties, "borderLeft", createJSONfromCTBorder(tblBorders.getLeft()));
                Utils.jsonPut(jsonTableProperties, "borderTop", createJSONfromCTBorder(tblBorders.getTop()));
                Utils.jsonPut(jsonTableProperties, "borderRight", createJSONfromCTBorder(tblBorders.getRight()));
                Utils.jsonPut(jsonTableProperties, "borderBottom", createJSONfromCTBorder(tblBorders.getBottom()));
                Utils.jsonPut(jsonTableProperties, "borderInsideHor", createJSONfromCTBorder(tblBorders.getInsideH()));
                Utils.jsonPut(jsonTableProperties, "borderInsideVert", createJSONfromCTBorder(tblBorders.getInsideV()));
            }

            CTTblCellMar tblCellMar= tableProperties.getTblCellMar();
            if(tblCellMar!=null) {
                Utils.jsonPut(jsonTableProperties, "paddingLeft", createJSONFromTblWidth(tblCellMar.getLeft(), 0, 0));
                Utils.jsonPut(jsonTableProperties, "paddingTop", createJSONFromTblWidth(tblCellMar.getTop(), 0, 0));
                Utils.jsonPut(jsonTableProperties, "paddingRight", createJSONFromTblWidth(tblCellMar.getRight(), 0, 0));
                Utils.jsonPut(jsonTableProperties, "paddingBottom", createJSONFromTblWidth(tblCellMar.getBottom(), 0, 0));
            }

            CTShd ctShd = tableProperties.getShd();
            Utils.jsonPut(jsonTableProperties, "fillColor", Utils.createFillColor(ctShd));

            // taking care of used conditional styles..:
            CTShortHexNumber tblLook = tableProperties.getTblLook();
            if(tblLook!=null) {
                String val = tblLook.getVal();
                if(val!=null) {
                    JSONArray jsonLook = new JSONArray();
                    int look = Integer.parseInt(val, 16);
                    if((look&0x20)==0)
                        jsonLook.put("firstRow");
                    if((look&0x40)==0)
                        jsonLook.put("lastRow");
                    if((look&0x80)==0)
                        jsonLook.put("firstCol");
                    if((look&0x100)==0)
                        jsonLook.put("lastCol");
                    if((look&0x200)>0) {
                        jsonLook.put("bandsHor");
                    }
                    if((look&0x400)>0) {
                        jsonLook.put("bandsVert");
                    }
                    if(jsonLook.length()>0)
                        jsonTableProperties.put("exclude", jsonLook);
                }
            }
        }
        return jsonTableProperties.length() > 0 ? jsonTableProperties : null;
    }

    public static void applyRowProperties(ObjectFactory objectFactory, JSONObject rowProperties, TrPr trPr)
        throws JSONException {

        if(rowProperties==null)
            return;
        Iterator<String> keys = rowProperties.keys();
        while(keys.hasNext()) {
            String attr = keys.next();
            Object value = rowProperties.get(attr);
            if(attr.equals("height")) {
                if (value instanceof Integer) {
                    List<JAXBElement<?>> jaxbElements = trPr.getCnfStyleOrDivIdOrGridBefore();
                    JAXBElement<?> jaxbElement = Commons.findElement(jaxbElements, "trHeight");
                    CTHeight ctHeight;
                    if(jaxbElement!=null)
                        ctHeight = (CTHeight)jaxbElement.getValue();
                    else {
                        ctHeight = objectFactory.createCTHeight();
                        ctHeight.setParent(trPr);
                    }
                    ctHeight.setVal(BigInteger.valueOf((Integer)value*144/254));
                    if(jaxbElement!=null)
                        jaxbElement = objectFactory.createCTTrPrBaseTrHeight(ctHeight);
                    else
                        jaxbElements.add(objectFactory.createCTTrPrBaseTrHeight(ctHeight));
                }
            }
        }
    }

    public static JSONObject createRowProperties(TrPr rowProperties)
        throws JSONException {

        JSONObject jsonRowProperties = new JSONObject();
        if(rowProperties!=null) {
            List<JAXBElement<?>> jaxbElements = rowProperties.getCnfStyleOrDivIdOrGridBefore();
            for(JAXBElement<?> jaxbElement:jaxbElements) {
                if(jaxbElement.getDeclaredType().getName().equals("org.docx4j.wml.CTHeight")) {
                    Object o = jaxbElement.getValue();
                    if(o instanceof CTHeight) {
                        BigInteger height = ((CTHeight)o).getVal();
                        if(height!=null)
                            jsonRowProperties.put("height", height.longValue()*254/144);
                    }
                    break;
                }
            }
        }
        return jsonRowProperties.length() > 0 ? jsonRowProperties : null;
    }

    public static void applyCellProperties(OperationDocument operationDocument, ObjectFactory objectFactory, JSONObject cellProperties, TcPr tcPr)
        throws JSONException {

        if(cellProperties==null)
            return;
        Iterator<String> keys = cellProperties.keys();
        while(keys.hasNext()) {
            String attr = keys.next();
            Object value = cellProperties.get(attr);
            if(attr.equals("gridSpan")) {
                if (value instanceof Integer) {
                    TcPrInner.GridSpan span = tcPr.getGridSpan();
                    if (span==null) {
                        span = objectFactory.createTcPrInnerGridSpan();
                        span.setParent(tcPr);
                    }
                    span.setVal(BigInteger.valueOf((Integer)value));
                    tcPr.setGridSpan(span);
                }
                else
                    tcPr.setGridSpan(null);
            }
            else if(attr.equals("borderLeft"))
                getTblBorders(objectFactory, tcPr).setLeft(createCTBorderFromJSON(operationDocument, objectFactory, tcPr, value));
            else if(attr.equals("borderTop"))
                getTblBorders(objectFactory, tcPr).setTop(createCTBorderFromJSON(operationDocument, objectFactory, tcPr, value));
            else if(attr.equals("borderRight"))
                getTblBorders(objectFactory, tcPr).setRight(createCTBorderFromJSON(operationDocument, objectFactory, tcPr, value));
            else if(attr.equals("borderBottom"))
                getTblBorders(objectFactory, tcPr).setBottom(createCTBorderFromJSON(operationDocument, objectFactory, tcPr, value));
            else if(attr.equals("borderInsideHor"))
                getTblBorders(objectFactory, tcPr).setInsideH(createCTBorderFromJSON(operationDocument, objectFactory, tcPr, value));
            else if(attr.equals("borderInsideVert"))
                getTblBorders(objectFactory, tcPr).setInsideV(createCTBorderFromJSON(operationDocument, objectFactory, tcPr, value));
            else if(attr.equals("paddingLeft"))
                getTblCellMar(objectFactory, tcPr).setLeft(createTblWidthFromJSON(objectFactory, tcPr, value));
            else if(attr.equals("paddingTop"))
                getTblCellMar(objectFactory, tcPr).setTop(createTblWidthFromJSON(objectFactory, tcPr, value));
            else if(attr.equals("paddingRight"))
                getTblCellMar(objectFactory, tcPr).setRight(createTblWidthFromJSON(objectFactory, tcPr, value));
            else if(attr.equals("paddingBottom"))
                getTblCellMar(objectFactory, tcPr).setBottom(createTblWidthFromJSON(objectFactory, tcPr, value));
            else if(attr.equals("fillColor")) {
                if(value instanceof JSONObject) {
                    CTShd shd = tcPr.getShd();
                    if(shd==null) {
                        shd = objectFactory.createCTShd();
                        shd.setParent(tcPr);
                        tcPr.setShd(shd);
                    }
                    Utils.initShdFromJSONColor(operationDocument, (JSONObject)value, shd);
                }
                else
                    tcPr.setShd(null);
            }
        }
    }

    public static JSONObject createCellProperties(TcPr cellProperties, Tbl table)
        throws JSONException {

        JSONObject jsonCellProperties = new JSONObject();
        if(cellProperties!=null) {

            GridSpan gridSpan = cellProperties.getGridSpan();
            if(gridSpan!=null) {
                jsonCellProperties.put("gridSpan", gridSpan.getVal().intValue());
            }

            // creating cell borders
            TcBorders tcBorder = cellProperties.getTcBorders();
            if(tcBorder!=null) {
                Utils.jsonPut(jsonCellProperties, "borderLeft", createJSONfromCTBorder(tcBorder.getLeft()));
                Utils.jsonPut(jsonCellProperties, "borderTop", createJSONfromCTBorder(tcBorder.getTop()));
                Utils.jsonPut(jsonCellProperties, "borderRight", createJSONfromCTBorder(tcBorder.getRight()));
                Utils.jsonPut(jsonCellProperties, "borderBottom", createJSONfromCTBorder(tcBorder.getBottom()));
                Utils.jsonPut(jsonCellProperties, "borderInsideHor", createJSONfromCTBorder(tcBorder.getInsideH()));
                Utils.jsonPut(jsonCellProperties, "borderInsideVert", createJSONfromCTBorder(tcBorder.getInsideV()));
            }

            // only create cell fill color if it differ from the table fill color
            CTShd shd = cellProperties.getShd();
            boolean createCellFill = true;
            if(table!=null) {
                TblPr tblPr = table.getTblPr();
                if(tblPr!=null) {
                    CTShd tblShd = tblPr.getShd();
                    if(tblShd!=null) {
                        if((tblShd.getVal() == shd.getVal())
                            && Utils.equals(tblShd.getFill(), shd.getFill())
                            && (tblShd.getThemeColor() == shd.getThemeColor())
                            && (tblShd.getThemeFill() == shd.getThemeFill())
                            && Utils.equals(tblShd.getThemeFillShade(), shd.getThemeFillShade())
                            && Utils.equals(tblShd.getThemeFillTint(), shd.getThemeFillTint())
                            && Utils.equals(tblShd.getThemeShade(), shd.getThemeShade())
                            && Utils.equals(tblShd.getThemeTint(), shd.getThemeTint()))

                            createCellFill = false;
                    }
                }
            }
            if(createCellFill)
                Utils.jsonPut(jsonCellProperties, "fillColor", Utils.createFillColor(shd));

            TcMar tcMar = cellProperties.getTcMar();
            if(tcMar!=null) {
                Utils.jsonPut(jsonCellProperties, "paddingLeft", createJSONFromTblWidth(tcMar.getLeft(), 0, 0));
                Utils.jsonPut(jsonCellProperties, "paddingTop", createJSONFromTblWidth(tcMar.getTop(), 0, 0));
                Utils.jsonPut(jsonCellProperties, "paddingRight", createJSONFromTblWidth(tcMar.getRight(), 0, 0));
                Utils.jsonPut(jsonCellProperties, "paddingBottom", createJSONFromTblWidth(tcMar.getBottom(), 0, 0));
            }
        }
        return jsonCellProperties.length() > 0 ? jsonCellProperties : null;
    }

    private static TcBorders getTblBorders(ObjectFactory objectFactory, TcPr tcPr) {
        TcBorders tcBorders = tcPr.getTcBorders();
        if(tcBorders==null) {
            tcBorders = objectFactory.createTcPrInnerTcBorders();
            tcBorders.setParent(tcPr);
            tcPr.setTcBorders(tcBorders);
        }
        return tcBorders;
    }

    private static TblBorders getTblBorders(ObjectFactory objectFactory, CTTblPrBase tblPr) {
        TblBorders tblBorders = tblPr.getTblBorders();
        if(tblBorders==null) {
            tblBorders = objectFactory.createTblBorders();
            tblBorders.setParent(tblPr);
            tblPr.setTblBorders(tblBorders);
        }
        return tblBorders;
    }

    private static TcMar getTblCellMar(ObjectFactory objectFactory, TcPr tcPr) {
        TcMar tcMar = tcPr.getTcMar();
        if(tcMar==null) {
            tcMar = objectFactory.createTcMar();
            tcMar.setParent(tcPr);
            tcPr.setTcMar(tcMar);
        }
        return tcMar;
    }

    private static CTTblCellMar getTblCellMar(ObjectFactory objectFactory, CTTblPrBase tblPr) {
        CTTblCellMar tblCellM = tblPr.getTblCellMar();
        if(tblCellM==null) {
            tblCellM = objectFactory.createCTTblCellMar();
            tblCellM.setParent(tblPr);
            tblPr.setTblCellMar(tblCellM);
        }
        return tblCellM;
    }

    private static CTBorder createCTBorderFromJSON(OperationDocument operationDocument, ObjectFactory objectFactory, Object parent, Object borderProperties )
        throws JSONException {

        CTBorder ctBorder = null;
        if(borderProperties instanceof JSONObject) {
             ctBorder = objectFactory.createCTBorder();
            ctBorder.setParent(parent);
            Iterator<String> keys = ((JSONObject)borderProperties).keys();
            while(keys.hasNext()) {
                String attr = keys.next();
                Object value = ((JSONObject)borderProperties).get(attr);
                if(attr.equals("style")) {
                    if(value instanceof String) {
                        STBorder borderStyle = STBorder.SINGLE;
                        if(((String)value).equals("none"))
                            borderStyle = STBorder.NONE;
                        else if (((String)value).equals("single"))
                            borderStyle = STBorder.SINGLE;
                        else if (((String)value).equals("none"))
                            borderStyle = STBorder.NONE;
                        else if (((String)value).equals("double"))
                            borderStyle = STBorder.DOUBLE;
                        else if (((String)value).equals("dotted"))
                            borderStyle = STBorder.DOTTED;
                        else if (((String)value).equals("dashed"))
                            borderStyle = STBorder.DASHED;
                        else if (((String)value).equals("outset"))
                            borderStyle = STBorder.OUTSET;
                        else if (((String)value).equals("inset"))
                            borderStyle = STBorder.INSET;
                        ctBorder.setVal(borderStyle);
                    }
                    else
                        ctBorder.setVal(STBorder.NONE);
                } else if(attr.equals("width")) {
                    if(value instanceof Integer) {
                        int val = ((Integer)value + 2 )* 576 / 2540;
                        ctBorder.setSz(BigInteger.valueOf(val));
                    }
                    else
                        ctBorder.setSz(null);
                } else if(attr.equals("color")) {
                    if(value instanceof JSONObject) {
                        Utils.initBorderColorFromJSONColor(operationDocument, (JSONObject)value, ctBorder);
                    }
                    else
                        ctBorder.setColor(null);
                }
            }
        }
        return ctBorder;
    }

    private static JSONObject createJSONfromCTBorder(CTBorder tblBorder)
        throws JSONException {

        if(tblBorder==null)
            return null;
        JSONObject jsonBorder = new JSONObject();
        STBorder borderStyle = tblBorder.getVal();
        if(borderStyle!=null) {
            String style = "single";
            if(borderStyle==STBorder.NONE)
                style = "none";
            else if(borderStyle==STBorder.NIL)
                style = "none";
            else if(borderStyle==STBorder.SINGLE)
                style = "single";
            else if(borderStyle==STBorder.DOUBLE)
                style = "double";
            else if(borderStyle==STBorder.DOTTED)
                style = "dotted";
            else if(borderStyle==STBorder.DASHED)
                style = "dashed";
            else if(borderStyle==STBorder.OUTSET)
                style = "outset";
            else if(borderStyle==STBorder.INSET)
                style = "inset";

            // TODO... each ooxml border style needs to be mapped to a corresponding html style... now they are all mapped to single border

            jsonBorder.put("style", style);
        }

        // creating border color
        Utils.jsonPut(jsonBorder, "color", Utils.createColor(tblBorder.getThemeColor(), tblBorder.getColor()));

        // creating border with
        BigInteger sz = tblBorder.getSz();
        if(sz!=null){
            int dv = (int)((sz.doubleValue() * 2540 + 288.)/ 576);
            jsonBorder.put("width", dv );    // 1/8 Pt to 1/100mm
        }
        return jsonBorder;
    }

    private static TblWidth createTblWidthFromJSON(ObjectFactory objectFactory, Object parent, Object jsonWidth) {

        TblWidth tblWidth = null;
        if (jsonWidth instanceof Integer||jsonWidth instanceof String) {

            tblWidth = objectFactory.createTblWidth();
            tblWidth.setParent(parent);

            if(jsonWidth instanceof Integer) {
                tblWidth.setType("dxa");
                tblWidth.setW(BigInteger.valueOf((Integer)jsonWidth * 72 * 20 / 2540));
            }
            else if (((String)jsonWidth).equals("auto")) {
                tblWidth.setType("pct");
                tblWidth.setW(BigInteger.valueOf(5000));
            }
        }
        return tblWidth;
    }

    private static Object createJSONFromTblWidth(TblWidth tblWidth, long referenceWidth, int firstRowWidth) {

        if(tblWidth==null)
            return null;
        Object jsonObject = null;

        String tableWidthType=null;     // default = "dxa"; (1/20 point)
                                        //           "auto"
                                        //           "pct"  (if percent sign is present... then val W represents whole percent points,
                                        //                   otherwise ????)
                                        //           "nil"  (zero width)
        int tableWidth = 0;             // default = 0;

        tableWidthType = tblWidth.getType();
        BigInteger bigInt = tblWidth.getW();
        if(bigInt!=null)
            tableWidth = bigInt.intValue();

        if ((tableWidthType != null) && (tableWidthType.equals("auto")) && (tableWidth == 0) && (firstRowWidth > 0)) {
            if(referenceWidth < firstRowWidth)
                jsonObject = new String("auto");
            else
                jsonObject = new Integer(firstRowWidth);  // using the width, that was calculated from all cells in the first row
        }
        else if ((tableWidthType!=null&&tableWidthType.equals("auto"))) {
            jsonObject = new String("auto");
        }
        else if (tableWidthType!=null&&tableWidthType.equals("pct")) {   // relative to page width excluding margins ... 5000==100%
            if(referenceWidth!=0) {
                tableWidth = (int)(referenceWidth * tableWidth / 5000);
                jsonObject = new Integer(tableWidth);
            }
            else {
                // we do not have a referenceWidth, percentage to absolute width can't be calculated, so we set width = auto here...
                jsonObject = new String("auto");
            }
        }
        else { // default: if(tableWidthType==null||tableWidthType.equals("dxa"))
            tableWidth = tableWidth * 2540 / 72 / 20;
            jsonObject = new Integer(tableWidth);
        }
        return jsonObject;
    }
}
