/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.tools;

import java.lang.reflect.InvocationTargetException;
import java.math.BigDecimal;
import java.text.NumberFormat;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.xml.bind.JAXBElement;

import org.docx4j.openpackaging.packages.WordprocessingMLPackage;
import org.docx4j.vml.CTArc;
import org.docx4j.vml.CTCurve;
import org.docx4j.vml.CTGroup;
import org.docx4j.vml.CTImage;
import org.docx4j.vml.CTLine;
import org.docx4j.vml.CTOval;
import org.docx4j.vml.CTPolyLine;
import org.docx4j.vml.CTRect;
import org.docx4j.vml.CTRoundRect;
import org.docx4j.vml.CTShape;
import org.docx4j.vml.CTShapetype;
import org.docx4j.vml.officedrawing.STHrAlign;
import org.docx4j.vml.wordprocessingDrawing.CTWrap;
import org.docx4j.vml.wordprocessingDrawing.STWrapType;
import org.docx4j.wml.Color;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.office.ooxml.docx.tools.Component.PictVMLComponent;
import com.openexchange.office.ooxml.docx.tools.Component.Placeholder_Base.GraphicType;
import com.openexchange.office.ooxml.tools.Commons;

public class PictVML {

    public static void applyProperties(JSONObject pictProperties, PictVMLComponent pictComponent)
        throws JSONException {

        if(pictProperties!=null) {
            VMLShape vmlShape = new VMLShape(pictComponent);
            VML_Shape_base vmlShapeImpl = vmlShape.getVMLShapeImpl();
            if(vmlShapeImpl!=null) {
                VMLStyle vmlStyle = new VMLStyle(vmlShapeImpl);

                int marginLeft = 0;
                int marginTop = 0;
                int marginRight = 0;
                int marginBottom = 0;
                if(pictProperties.hasAndNotNull("marginLeft")) {
                    marginLeft = pictProperties.getInt("marginLeft");
                }
                if(pictProperties.hasAndNotNull("marginTop")) {
                    marginTop = pictProperties.getInt("marginTop");
                }
                if(pictProperties.hasAndNotNull("marginRight")) {
                    marginRight = pictProperties.getInt("marginRight");
                }
                if(pictProperties.hasAndNotNull("marginBottom")) {
                    marginBottom = pictProperties.getInt("marginBottom");
                }

                boolean isInline = true;
                final String pos = vmlStyle.get("position");
                if(pos!=null)
                    isInline = !pos.equals("absolute");
                if(pictProperties.has("inline"))
                    isInline = pictProperties.getBoolean("inline");
                if (isInline) {
                    vmlStyle.set("position", "static");
                    vmlStyle.remove("mso-position-horizontal");
                    vmlStyle.remove("mso-position-horizontal-relative");
                    vmlStyle.remove("mso-position-vertical");
                    vmlStyle.remove("mso-position-vertical-relative");
                } else {
                    vmlStyle.set("position",  "absolute");
                    if(pictProperties.hasAndNotNull("anchorHorBase")) {
                        vmlStyle.setHorzBase(pictProperties.getString("anchorHorBase"));
                    }
                    String anchorHorAlign = null;
                    if(pictProperties.hasAndNotNull("anchorHorAlign")) {
                        anchorHorAlign = pictProperties.getString("anchorHorAlign");
                        vmlStyle.setHorzAlign(anchorHorAlign);
                    }

                    if(pictProperties.hasAndNotNull("anchorVertBase")) {
                        vmlStyle.setVertBase(pictProperties.getString("anchorVertBase"));
                    }
                    String anchorVertAlign = null;
                    if(pictProperties.hasAndNotNull("anchorVertAlign")) {
                        anchorVertAlign = pictProperties.getString("anchorVertAlign");
                        vmlStyle.setVertAlign(anchorVertAlign);
                    }

                    if(anchorHorAlign!=null&&anchorHorAlign.equals("offset")) {
                        if(pictProperties.hasAndNotNull("anchorHorOffset"))
                            marginLeft += pictProperties.getInt("anchorHorOffset");
                    }
                    if(anchorVertAlign!=null&&anchorVertAlign.equals("offset")) {
                        if(pictProperties.hasAndNotNull("anchorVertOffset"))
                            marginTop += pictProperties.getInt("anchorVertOffset");
                    }
                    if(pictProperties.hasAndNotNull("textWrapMode")) {
                        CTWrap ctWrap = vmlShapeImpl.getCTWrap(true);
                        String wrapMode = pictProperties.getString("textWrapMode");
                        if(wrapMode.equals("square"))
                            ctWrap.setType(STWrapType.SQUARE);
                        else if(wrapMode.equals("tight"))
                            ctWrap.setType(STWrapType.TIGHT);
                        else if(wrapMode.equals("topAndBottom"))
                            ctWrap.setType(STWrapType.TOP_AND_BOTTOM);
                        else if(wrapMode.equals("through"))
                            ctWrap.setType(STWrapType.THROUGH);
                        else
                            ctWrap.setType(STWrapType.NONE);
                        vmlShapeImpl.setCTWrap(ctWrap);
                    }
                }

                if(marginLeft!=0)
                    vmlStyle.setLength("margin-left", marginLeft);
                else
                    vmlStyle.remove("margin-left");
                if(marginTop!=0)
                    vmlStyle.setLength("margin-top", marginTop);
                else
                    vmlStyle.remove("margin-top");
                if(marginRight!=0)
                    vmlStyle.setLength("margin-right", marginRight);
                else
                    vmlStyle.remove("margin-right");
                if(marginBottom!=0)
                    vmlStyle.setLength("margin-bottom", marginBottom);
                else
                    vmlStyle.remove("margin-bottom");

                Iterator<String> keys = pictProperties.keys();
                while(keys.hasNext()) {
                    String attr = keys.next();
                    Object value = pictProperties.get(attr);
                    if(attr.equals("width")) {
                        vmlStyle.setLength("width", (Integer)value);
                    }
                    if(attr.equals("height")) {
                        vmlStyle.setLength("height", (Integer)value);
                    }
                    if(pictComponent.getType() == GraphicType.HORIZONTAL_LINE && (attr.equals("anchorHorAlign") || attr.equals("width"))) {
                        List<Object> content = pictComponent.getContent();
                        if(content!=null) {
                            for(Object o : content) {
                                if(o instanceof JAXBElement){
                                    o = ((JAXBElement<?>)o).getValue();
                                    if(o.getClass() == CTRect.class){
                                        CTRect rect = (CTRect)o;
                                        try{
                                            if( attr.equals("anchorHorAlign") ){
                                                STHrAlign hrAlign = STHrAlign.fromValue((String)value);
                                                rect.setHralign(hrAlign);
                                            } else {
                                                rect.setHrpct(new Float(0.));
                                            }
                                        } catch( IllegalArgumentException i ){
                                            //
                                        }
                                    }
                                    break;
                                }
                            }
                        }
                    }
                }
                vmlStyle.updateStyle();
            }
        }
    }

    public static JSONObject createProperties(WordprocessingMLPackage wordMLPackage, PictVMLComponent pictComponent)
        throws JSONException, ParseException {

        JSONObject jsonPictProperties = new JSONObject();
        VMLShape vmlShape = new VMLShape(pictComponent);
        VML_Shape_base vmlShapeImpl = vmlShape.getVMLShapeImpl();
        if(vmlShapeImpl!=null) {
            VMLStyle vmlStyle = new VMLStyle(vmlShapeImpl);

            String width = vmlStyle.get("width");
            if(vmlStyle.getDimensionType(width)==DimensionType.LENGTH) {
                jsonPictProperties.put("width", (int)vmlStyle.getLength(width));
            }
            String height = vmlStyle.get("height");
            if(vmlStyle.getDimensionType(height)==DimensionType.LENGTH) {
                jsonPictProperties.put("height", (int)vmlStyle.getLength(height));
            }

            boolean inline = true;
            String position = vmlStyle.get("position");
            if(position!=null) {
                if(position.equals("absolute"))
                    inline = false;
            }
            jsonPictProperties.put("inline", inline);

            if(!inline) {
                jsonPictProperties.put("anchorHorBase", vmlStyle.getHorzBase() );
                String anchorHAlign = vmlStyle.getHorzAlign();
                jsonPictProperties.put("anchorHorAlign", anchorHAlign);
                if(anchorHAlign.equals("offset")) {
                    int horzOffset = 0;
                    String marginLeft = vmlStyle.get("margin-left");
                    if(vmlStyle.getDimensionType(marginLeft)==DimensionType.LENGTH) {
                        horzOffset = (int)vmlStyle.getLength(marginLeft);
                    }
                    jsonPictProperties.put("anchorHorOffset", (Integer)horzOffset);
                }
                jsonPictProperties.put("anchorVertBase", vmlStyle.getVertBase() );
                String anchorVAlign = vmlStyle.getVertAlign();
                jsonPictProperties.put("anchorVertAlign", anchorVAlign);
                if(anchorVAlign.equals("offset")) {
                    int vertOffset = 0;
                    String marginTop = vmlStyle.get("margin-top");
                    if(vmlStyle.getDimensionType(marginTop)==DimensionType.LENGTH) {
                        vertOffset = (int)vmlStyle.getLength(marginTop);
                    }
                    jsonPictProperties.put("anchorVertOffset", (Integer)vertOffset);
                }
                CTWrap ctWrap = vmlShapeImpl.getCTWrap(false);
                if(ctWrap!=null) {
                    String wrapMode = "none";
                    if(ctWrap.getType()==STWrapType.SQUARE)
                        wrapMode = "square";
                    else if(ctWrap.getType()==STWrapType.THROUGH)
                        wrapMode = "through";
                    else if(ctWrap.getType()==STWrapType.TIGHT)
                        wrapMode = "tight";
                    else if(ctWrap.getType()==STWrapType.TOP_AND_BOTTOM)
                        wrapMode = "top-and-bottom";
                    jsonPictProperties.put("textWrapMode", wrapMode);
                }
            }
            if(pictComponent.getType() == GraphicType.OLE) {
                jsonPictProperties.put("imageUrl", pictComponent.getImageUrl(wordMLPackage.getMainDocumentPart()));
            }

            if(pictComponent.getType() == GraphicType.HORIZONTAL_LINE) {
                //convert o:hr-properties to shape properties
                //o:hralign, o:hrpct, o:hrnoshade, fillcolor, stroked
                if(!jsonPictProperties.has("width"))
                    jsonPictProperties.put("width", 0);

                jsonPictProperties.put( "textWrapMode", "square" );
                jsonPictProperties.put( "anchorHorBase", "column" );
                jsonPictProperties.put( "anchorHorOffset", 0 );
                jsonPictProperties.put( "inline", false );

                JSONObject topBorder = new JSONObject();
                topBorder.put("width", 10);
                topBorder.put("style", "solid");

                List<Object> content = pictComponent.getContent();
                if(content!=null) {
                    for(Object o : content) {
                        if(o instanceof JAXBElement){
                            o = ((JAXBElement<?>)o).getValue();
                            if(o.getClass() == CTRect.class){
                                CTRect rect = (CTRect)o;
                                STHrAlign hrAlign = rect.getHralign();
                                if(hrAlign != null){
                                    jsonPictProperties.put("anchorHorAlign", hrAlign.value());
                                    if( hrAlign.value().equals("right") )
                                        jsonPictProperties.put( "textWrapSide", "left" );
                                }
                                String fillColor = rect.getFillcolor();
                                if(fillColor != null){
                                    Color color = new Color();
                                    if(fillColor.startsWith("#"))
                                        color.setVal(fillColor.substring(1, 7).toUpperCase());
                                    else
                                        color.setVal(fillColor);
                                    Utils.jsonPut(topBorder, "color", Utils.createColor(color));
                                    jsonPictProperties.put("borderTop", topBorder);
                                }

                            }
                            break;
                        }
                    }
                }
                jsonPictProperties.put("borderTop", topBorder);
            }
        }
        return jsonPictProperties.length() > 0 ? jsonPictProperties : null;
    }

    public static class VMLShape {

        private final Map<String, VML_Shape_type> shapeTypes;
        private VML_Shape_base shape;

        VMLShape(PictVMLComponent vmlComponent) {

            shapeTypes = new HashMap<String, VML_Shape_type>();
            shape = null;

            List<Object> content = vmlComponent.getContent();
            if(content!=null) {
                for(Object o : content) {
                    if(o instanceof JAXBElement)
                        o = ((JAXBElement<?>)o).getValue();

                    VML_Shape_base vml_Shape_base = createPictVMLShape(this, o);
                    if(vml_Shape_base instanceof VML_Shape_type) {
                        String Id = ((VML_Shape_type)vml_Shape_base).getId();
                        if(Id!=null&&Id.length()>0)
                            getShapeTypes().put(Id, (VML_Shape_type)vml_Shape_base);
                    }else if(vml_Shape_base!=null) {
                        if(shape!=null)
                            System.out.println("OOXML - VMLShape, not assuming group shapes...");
                        else
                            shape = vml_Shape_base;
                    }
                }
            }
        }
        Map<String, VML_Shape_type> getShapeTypes() {
            return shapeTypes;
        }
        VML_Shape_base getVMLShapeImpl() {
            return shape;
        }
    }

    protected static abstract class VML_Shape_base {

        final Object  o;
        final VML_Shape_type referenceShape;
        final List<JAXBElement<?>> shapeElements;

        protected VML_Shape_base(Object _o, List<JAXBElement<?>> _shapeElements, VML_Shape_type _referenceShape) {
            o = _o;
            referenceShape = _referenceShape;
            shapeElements = _shapeElements;
        }
        public Object getObject() {
            return o;
        }

        // returns the wrap mode, can be null (not a live object)
        public CTWrap getCTWrap(boolean createIfNotAvailable) {
            CTWrap ctWrap = null;
            JAXBElement<?> jaxbElement = Commons.findElement(shapeElements, "wrap");
            if(jaxbElement!=null)
                ctWrap = (CTWrap)jaxbElement.getValue();
            else
                ctWrap = org.docx4j.jaxb.Context.getVmlWordprocessingDrawingObjectFactory().createCTWrap();
            return ctWrap;
        }

        // applies wrap mode to the vml object (null is allowed and removes the property)
        public void setCTWrap(CTWrap ctWrap) {
            if(ctWrap==null)
                Commons.removeElement(shapeElements, "wrap");
            else
                Commons.insertElement(shapeElements,  "wrap", org.docx4j.jaxb.Context.getVmlWordprocessingDrawingObjectFactory().createWrap(ctWrap));
        }
    }
    public static class VML_Shape_group extends VML_Shape_base {

        final List<VML_Shape_base> shapes;

        public VML_Shape_group(CTGroup o) {
            super(o, o.getPathOrFormulasOrHandles(), null);
            shapes = new ArrayList<VML_Shape_base>();
        }
        public void addShape(VML_Shape_base shape) {
            shapes.add(shape);
        }
    }

    // VML_Shape_type may be referenced by the VML_Shape_shape via type attribute
    public static class VML_Shape_type extends VML_Shape_base {

        final String Id;

        VML_Shape_type(CTShapetype o) {
            super(o, o.getEGShapeElements(), null);
            Id = o.getVmlId();
        }
        String getId() {
            return Id;
        }
    }
    public static class VML_Shape_shape extends VML_Shape_base {

        public VML_Shape_shape(CTShape o, VML_Shape_type vml_Shape_type) {
            super(o, o.getPathOrFormulasOrHandles(), vml_Shape_type);
        }
    }
    public static class VML_Shape_line extends VML_Shape_base {

        public VML_Shape_line(CTLine o) {
            super(o, o.getEGShapeElements(), null);
        }
    }
    public static class VML_Shape_polyline extends VML_Shape_base {

        public VML_Shape_polyline(CTPolyLine o) {
            super(o, o.getPathOrFormulasOrHandles(), null);
        }
    }
    public static class VML_Shape_curve extends VML_Shape_base {

        public VML_Shape_curve(CTCurve o) {
            super(o, o.getEGShapeElements(), null);
        }
    }
    public static class VML_Shape_rect extends VML_Shape_base {

        public VML_Shape_rect(CTRect o) {
            super(o, o.getPathOrFormulasOrHandles(), null);
        }
    }
    public static class VML_Shape_roundrect extends VML_Shape_base {

        public VML_Shape_roundrect(CTRoundRect o) {
            super(o, o.getPathOrFormulasOrHandles(), null);
        }
    }
    public static class VML_Shape_oval extends VML_Shape_base {

        public VML_Shape_oval(CTOval o) {
            super(o, o.getPathOrFormulasOrHandles(), null);
        }
    }
    public static class VML_Shape_arc extends VML_Shape_base {

        public VML_Shape_arc(CTArc o) {
            super(o, o.getEGShapeElements(), null);
        }
    }
    public static class VML_Shape_image extends VML_Shape_base {

        public VML_Shape_image(CTImage o) {
            super(o, o.getEGShapeElements(), null);
        }
    }

    protected static VML_Shape_base createPictVMLShape(VMLShape vmlShape, Object o) {
        VML_Shape_base vml_Shape_base = null;
        if(o instanceof CTShapetype)
            vml_Shape_base = new VML_Shape_type((CTShapetype)o);
        else if(o instanceof CTGroup) {
            List<JAXBElement<?>> content = ((CTGroup)o).getPathOrFormulasOrHandles();
            vml_Shape_base = new VML_Shape_group((CTGroup)o);
            for(Object group_o : content) {
                if(group_o instanceof JAXBElement)
                    group_o = ((JAXBElement<?>)group_o).getValue();
                VML_Shape_base group_vml = createPictVMLShape(vmlShape, group_o);
                if(group_vml instanceof VML_Shape_type) {
                    String Id = ((VML_Shape_type)group_vml).getId();
                    if(Id.length()>0)
                        vmlShape.getShapeTypes().put(Id, (VML_Shape_type)group_vml);
                }
                else if (group_vml!=null)
                    ((VML_Shape_group)vml_Shape_base).addShape(group_vml);
            }
        }
        else if(o instanceof CTShape) {
            VML_Shape_type vmlShapetype = null;
            String type = ((CTShape)o).getType();
            if(type!=null&&type.length()>1) {
                type = type.substring(1);
                vmlShapetype = vmlShape.getShapeTypes().get(type);
            }
            vml_Shape_base = new VML_Shape_shape((CTShape)o, vmlShapetype);
        }
        else if(o instanceof CTLine)
            vml_Shape_base = new VML_Shape_line((CTLine)o);
        else if(o instanceof CTPolyLine)
            vml_Shape_base = new VML_Shape_polyline((CTPolyLine)o);
        else if(o instanceof CTCurve)
            vml_Shape_base = new VML_Shape_curve((CTCurve)o);
        else if(o instanceof CTRect)
            vml_Shape_base = new VML_Shape_rect((CTRect)o);
        else if(o instanceof CTRoundRect)
            vml_Shape_base = new VML_Shape_roundrect((CTRoundRect)o);
        else if(o instanceof CTOval)
            vml_Shape_base = new VML_Shape_oval((CTOval)o);
        else if(o instanceof CTArc)
            vml_Shape_base = new VML_Shape_arc((CTArc)o);
        else if(o instanceof CTImage)
            vml_Shape_base = new VML_Shape_image((CTImage)o);
        return vml_Shape_base;
    }

    enum DimensionType {
        UNDEFINED,
        AUTO,
        LENGTH,
        PERCENT,
        INHERIT
    }

    enum UnitType {
        UNDEFINED,
        PT,
        PC,
        IN,
        MM,
        CM,
        PX,
        EM,
        EX,
        PERCENT
    }

    public static class VMLStyle {

        Object shape;
        Map<String, String> values;

        VMLStyle(VML_Shape_base o) {
            shape = o.getObject();
            values = new HashMap<String, String>();

            String style = null;
            try {
                style = (String)shape.getClass().getMethod("getStyle").invoke(shape);
            } catch (IllegalArgumentException e) {
                // TODO Auto-generated catch block
            } catch (SecurityException e) {
                // TODO Auto-generated catch block
            } catch (IllegalAccessException e) {
                // TODO Auto-generated catch block
            } catch (InvocationTargetException e) {
                // TODO Auto-generated catch block
            } catch (NoSuchMethodException e) {
                // TODO Auto-generated catch block
            }
            // parsing the style string
            if(style!=null&&style.length()>0) {
                String[] styleTokens = style.split(" *?; *?");
                for(String token : styleTokens) {
                    int index = token.indexOf(":");
                    if(index>0&&index<(token.length()+1)) {
                        String key = token.substring(0, index).trim().toLowerCase();
                        String value = token.substring(index+1).trim().toLowerCase();
                        if(key.length()>0&&value.length()>0) {
                            values.put(key,  value);
                        }
                    }
                }
            }
        }
        String get(String key) {
            return values.get(key);
        }
        void set(String key, String value) {
            values.put(key, value);
        }
        void remove(String key) {
            values.remove(key);
        }

        // writes back the style information to shape
        void updateStyle() {
            if(values.size()>0) {
                StringBuffer styleBuf = new StringBuffer();
                boolean first = true;
                for(String key : values.keySet()) {
                    if(!first)
                        styleBuf.append(";");
                    first = false;
                    styleBuf.append(key);
                    styleBuf.append(":");
                    styleBuf.append(values.get(key));
                }
                String style = styleBuf.toString();
                try {
                    shape.getClass().getMethod("setStyle", new Class[] {String.class}).invoke(shape, new Object[]{style});
                } catch (IllegalArgumentException e) {
                    // TODO Auto-generated catch block
                } catch (SecurityException e) {
                    // TODO Auto-generated catch block
                } catch (IllegalAccessException e) {
                    // TODO Auto-generated catch block
                } catch (InvocationTargetException e) {
                    // TODO Auto-generated catch block
                } catch (NoSuchMethodException e) {
                    // TODO Auto-generated catch block
                }
            }
        }

        String getHorzBase() {
            String value = get("mso-position-horizontal-relative"); // margin, page, text or char possible
            if(value==null)
                return "margin";
            else if(value.equals("page"))
                return "page";
            else if(value.equals("text"))
                return "column";
            else if(value.equals("char"))
                return "character";
            return "margin";
        }

        String getHorzAlign() {
            String value = get("mso-position-horizontal");          // absolute, left, center, right, inside or outside possible
            if(value==null)
                return "offset";
            else if(value.equals("absolute"))
                return "offset";
            else if(value.equals("center"))
                return "center";
            else if(value.equals("right"))
                return "right";
            else if(value.equals("inside"))
                return "inside";
            else if(value.equals("outside"))
                return "outside";
            return "left";
        }

        void setHorzBase(final String horzBase) {                   // page, column, character, leftMargin, insideMargin or outsideMargin
            if(horzBase.equals("page")) {
                set("mso-position-horizontal-relative", "page");
            }
            else if(horzBase.equals("column")) {
                set("mso-position-horizontal-relative", "text");
            }
            else if(horzBase.equals("character")) {
                set("mso-position-horizontal-relative", "char");
            }
            else if(horzBase.equals("leftMargin")) {
                set("mso-position-horizontal-relative", "margin");
            }
            else if(horzBase.equals("insideMargin")) {
                set("mso-position-horizontal-relative", "text");
            }
            else if(horzBase.equals("outsideMargin")) {
                set("mso-position-horizontal-relative", "text");
            }
        }

        void setHorzAlign(final String horzAlign) {                 // left, right, center, inside, outside or offset
            if(horzAlign.equals("left")) {
                set("mso-position-horizontal", "left");
            }
            if(horzAlign.equals("center")) {
                set("mso-position-horizontal", "center");
            }
            if(horzAlign.equals("right")) {
                set("mso-position-horizontal", "right");
            }
            if(horzAlign.equals("inside")) {
                set("mso-position-horizontal", "inside");
            }
            if(horzAlign.equals("outside")) {
                set("mso-position-horizontal", "outside");
            }
            if(horzAlign.equals("offset")) {                        // offset, we have to add the offset to the left margin..
                remove("mso-position-horizontal");
            }
        }

        String getVertBase() {
            String value = get("mso-position-vertical-relative");   // margin, page, text or line possible
            if(value==null)
                return "margin";
            else if(value.equals("page"))
                return "page";
            else if(value.equals("text"))
                return "paragraph";
            else if(value.equals("line"))
                return "line";
            return "margin";
        }

        String getVertAlign() {

            String value = get("mso-position-vertical");            // absolute, top, center, bottom, inside or outside possible
            if(value==null)
                return "offset";
            else if(value.equals("absolute"))
                return "offset";
            else if(value.equals("center"))
                return "center";
            else if(value.equals("bottom"))
                return "right";
            else if(value.equals("inside"))
                return "inside";
            else if(value.equals("outside"))
                return "outside";
            return "top";
        }

        void setVertBase(final String vertBase) {                   // margin, page, paragraph, line, topMargin, bottomMargin, insideMargin or outsideMargin
            if(vertBase.equals("margin")) {
                set("mso-position-vertical-relative", "margin");
            }
            else if(vertBase.equals("page")) {
                set("mso-position-vertical-relative", "page");
            }
            else if(vertBase.equals("paragraph")) {
                set("mso-position-vertical-relative", "text");
            }
            else if(vertBase.equals("line")) {
                set("mso-position-vertical-relative", "line");
            }
            else if(vertBase.equals("topMargin")) {
                set("mso-position-vertical-relative", "margin");
            }
            else if(vertBase.equals("bottomMargin")) {
                set("mso-position-vertical-relative", "margin");    // TODO: offset has to be added to the margin
            }
            else if(vertBase.equals("insideMargin")) {
                set("mso-position-vertical-relative", "inside");
            }
            else if(vertBase.equals("outsideMargin")) {
                set("mso-position-vertical-relative", "outside");
            }
        }

        void setVertAlign(final String vertAlign) {                 // top, bottom, center, inside, outside or offset
            if(vertAlign.equals("top")) {
                set("mso-position-vertical", "top");
            }
            else if(vertAlign.equals("bottom")) {
                set("mso-position-vertical", "bottom");
            }
            else if(vertAlign.equals("center")) {
                set("mso-position-vertical", "center");
            }
            else if(vertAlign.equals("inside")) {
                set("mso-position-vertical", "inside");
            }
            else if(vertAlign.equals("outside")) {
                set("mso-position-vertical", "outside");
            }
            else if(vertAlign.equals("offset")) {
                remove("mso-position-vertical");
            }
        }

        DimensionType getDimensionType(String value) {
            if(value==null)
                return DimensionType.UNDEFINED;
            else if(value.equals("auto"))
                return DimensionType.AUTO;
            else if (value.equals("inherit"))
                return DimensionType.INHERIT;
            else if (value.contains("%"))
                return DimensionType.PERCENT;
            else if(value.matches(".*?pt|.*?pc|.*?in|.*?mm|.*?cm|.*?px"))
                return DimensionType.LENGTH;
            return DimensionType.UNDEFINED;
        }
        UnitType getUnitType(String value) {
            if(value==null)
                return UnitType.UNDEFINED;
            else if(value.contains("pt"))
                return UnitType.PT;
            else if(value.contains("pc"))
                return UnitType.PC;
            else if(value.contains("in"))
                return UnitType.IN;
            else if(value.contains("mm"))
                return UnitType.MM;
            else if(value.contains("cm"))
                return UnitType.CM;
            else if(value.contains("px"))
                return UnitType.PX;
            else if(value.contains("em"))
                return UnitType.EM;
            else if(value.contains("ex"))
                return UnitType.EX;
            else if(value.contains("%"))
                return UnitType.PERCENT;
            else
                return UnitType.UNDEFINED;
        }
        // returns the length in 1/100mm
        double getLength(String value) throws ParseException {
            double length = 0;

            length = NumberFormat.getInstance(Locale.US).parse(value).doubleValue();
            switch(getUnitType(value)) {
                case UNDEFINED :
                case EM :
                case EX :
                case PERCENT :
                    throw new ParseException("", 0);
                case PT : length = (length/72)*2540; break;
                case PC : length = (length/6)*2540; break;
                case IN : length = length*2540; break;
                case MM : length = length*100; break;
                case CM : length = length*1000; break;
                case PX : length = (length/96)*2540; break;
            }
            return length;
        }

        // sets length which is assumed to be in 1/100mm
        void setLength(String key, int length) {
            BigDecimal dec = new BigDecimal(((double)length*72/2540));
            dec = dec.setScale(2, BigDecimal.ROUND_HALF_UP);
            values.put(key, dec.toString() + "pt");
        }
    }
}
