/*
 * Copyright 2012 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.openexchange.office.odf;

import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.json.JSONObject;
import org.odftoolkit.odfdom.component.TextOperationDocument;
import org.odftoolkit.odfdom.doc.OdfTextDocument;
import com.openexchange.config.ConfigurationService;
import com.openexchange.office.FilterException;
import com.openexchange.office.FilterException.ErrorCode;
import com.openexchange.office.tools.ResourceManager;
import com.openexchange.server.ServiceLookup;

/**
 * The access point to the ODF Toolkit, obfuscating all OX Office dependencies from the Toolkit.
 * 
 */
public class OdtOperationDocument {

    private TextOperationDocument mDocument;
    private static final Logger LOG = Logger.getLogger(OdtOperationDocument.class.getName());
    private ServiceLookup mServiceLookup;
    private boolean mSaveDebugOperations;
    private int mMaxTableColumns;
    private int mMaxTableCells;
    private int mMaxTableRows;
    
    /**
     * Creates an empty ODF text document.
     */
    public OdtOperationDocument() throws Exception {
        mDocument = new TextOperationDocument();
    }

    /**
     * Creates an OdtOperationDocument from the OpenDocument provided by a resource Stream.
     * <p>
     * Since an InputStream does not provide the arbitrary (non sequential) read access needed by OdtOperationDocument, the InputStream is
     * cached. This usually takes more time compared to the other createInternalDocument methods. An advantage of caching is that there are
     * no problems overwriting an input file.
     * </p>
     * 
     * @param inputStream - the InputStream of the ODF text document.
     * @return the text document created from the given InputStream
     */
    public OdtOperationDocument(InputStream aInputDocumentStm) throws Exception {
        mDocument = new TextOperationDocument(aInputDocumentStm);            
    }

    public OdtOperationDocument(ServiceLookup _services, InputStream inputDocumentStream, ResourceManager resourceManager, String userLanguage) throws Exception {
        mServiceLookup = _services;
        mSaveDebugOperations = getConfigurationService().getBoolProperty("io.ox/office//module/debugoperations", false);
        // adding one over the current default to be certain during testing to have an access to the config file
        mMaxTableColumns = getConfigurationService().getIntProperty("io.ox/office//module/maxTableColumns", 16);
        mMaxTableRows = getConfigurationService().getIntProperty("io.ox/office//module/maxTableRows", 1501);
        mMaxTableCells = getConfigurationService().getIntProperty("io.ox/office//module/maxTableCells", 1501);        

        Map<String, Object> configuration = new HashMap<String, Object>(5);
        configuration.put("debugoperations", Boolean.FALSE);
        configuration.put("maxTableColumns", mMaxTableColumns);
        configuration.put("maxTableRows", mMaxTableRows);
        configuration.put("maxTableCells", mMaxTableCells);
        mDocument = new TextOperationDocument(inputDocumentStream, (null != resourceManager) ? resourceManager.getByteArrayTable() : null, userLanguage, configuration);        
    }
    
    private ConfigurationService getConfigurationService() {
        return mServiceLookup.getService(com.openexchange.config.ConfigurationService.class);
    }    

    /**
     * Receives the (known) operations of the ODF text document
     * 
     * @return the operations as JSON
     */
    // ToDo OX - JSONObject is to be considered..
    public JSONObject getOperations() {
        JSONObject ops = mDocument.getOperations();
        if (ops != null) {
            LOG.log(Level.FINER, "\n\n*** ALL OPERATIONS:\n{0}", mDocument.getOperations().toString());
        } else {
            LOG.finer("\n\n*** ALL OPERATIONS:\n" + null);
        }
        return ops;
    }

    /**
     * Applies the (known) operations to upon the latest state of the ODF text document
     * 
     * @param operationString ODF operations as String
     * @return the number of operations being accepted
     */
    public int applyOperations(String operationString) {
        LOG.log(Level.FINER, "\n*** EDIT OPERATIONS:\n{0}", operationString);
        int operationsCount = 0;
        operationsCount = mDocument.applyOperations(operationString);
        return operationsCount;
    }

    /**
     * Applies the (known) operations to upon the latest state of the ODF text document
     * 
     * @param operations ODF operations as JSONArray within an JSONObject with "operations" key.
     * @return the number of operations being accepted
     */
    public int applyOperations(JSONObject operations) {
        LOG.log(Level.FINER, "\n*** EDIT OPERATIONS:\n{0}", operations.toString());
        int operationsCount = 0;
        operationsCount = mDocument.applyOperations(operations);        
        return operationsCount;
    }

    public long getContentSize() {
        return mDocument.getContentSize();
    }

    /**
     * Returns the TextOperationDocument encapsulating the DOM view
     * 
     * @return ODF text document
     */
    public OdfTextDocument getDocument() {
        return mDocument.getDocument();
    }
    
    /**
     * Close the OdfPackage and release all temporary created data. After execution of this method, this class is no longer usable. Do this
     * as the last action to free resources. Closing an already closed document has no effect.
     */
    public void close() {
        mDocument.close();
    }
}
