/*
 * Copyright 2012 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.odftoolkit.odfdom.component;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;
import org.odftoolkit.odfdom.pkg.OdfPackage;
import org.odftoolkit.odfdom.utils.ResourceUtilities;

/**
 * Loads a document with tables and gathers its operations. Gathered operations
 * will be applied to an empty text document. The changed text document will be
 * saved and reloaded. New gathered operations will be compared with the
 * original ones, expected to be identical!
 *
 * @author svanteschubert
 */
public class RoundtripEditingTest extends RoundtripTestTemplate {

	private static final Logger LOG = Logger.getLogger(RoundtripEditingTest.class.getName());

	public RoundtripEditingTest() {
	}
	private static final String OUTPUT_DIRECTORY = "regression-tests" + File.separatorChar;

	@BeforeClass
	public static void setUpBeforeClass() throws Exception {
		// Creating the output directory for the tests
		File outputDir = ResourceUtilities.newTestOutputFile(OUTPUT_DIRECTORY);
		outputDir.mkdir();
	}

	@Test
	/**
	 * ODF space XML element that contains more than two spaces and was not
	 * completely deleted, returned the removed spaces as remaining element
	 * https://bugs.open-xchange.com/show_bug.cgi?id=28869
	 */
	public void spaceElementSplitTest() {
		final String SOURCE_FILE_NAME_TRUNC = "hyperlinkSpacesNoUnderline";
		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"start\":[5,34],\"name\":\"delete\",\"opl\":1,\"end\":[5,34],\"osn\":48}"
				+ ",{\"start\":[5,33],\"name\":\"delete\",\"opl\":1,\"end\":[5,33],\"osn\":49}"
				+ ",{\"start\":[5,32],\"name\":\"delete\",\"opl\":1,\"end\":[5,32],\"osn\":50}"
				+ ",{\"start\":[5,31],\"name\":\"delete\",\"opl\":1,\"end\":[5,31],\"osn\":51}"
				+ ",{\"start\":[5,30],\"name\":\"delete\",\"opl\":1,\"end\":[5,30],\"osn\":52}"
				+ ",{\"start\":[5,29],\"name\":\"delete\",\"opl\":1,\"end\":[5,29],\"osn\":53}"
				+ ",{\"start\":[5,28],\"name\":\"delete\",\"opl\":1,\"end\":[5,28],\"osn\":54}"
				+ ",{\"start\":[5,27],\"name\":\"delete\",\"opl\":1,\"end\":[5,27],\"osn\":55}"
				+ ",{\"start\":[5,26],\"name\":\"delete\",\"opl\":1,\"end\":[5,26],\"osn\":56}"
				+ ",{\"start\":[5,25],\"name\":\"delete\",\"opl\":1,\"end\":[5,25],\"osn\":57}"
				+ ",{\"start\":[5,24],\"name\":\"delete\",\"opl\":1,\"end\":[5,24],\"osn\":58}"
				+ ",{\"start\":[5,23],\"name\":\"delete\",\"opl\":1,\"end\":[5,23],\"osn\":59}"
				+ ",{\"start\":[5,22],\"name\":\"delete\",\"opl\":1,\"end\":[5,22],\"osn\":60}"
				+ ",{\"start\":[5,21],\"name\":\"delete\",\"opl\":1,\"end\":[5,21],\"osn\":61}"
				+ ",{\"start\":[5,20],\"name\":\"delete\",\"opl\":1,\"end\":[5,20],\"osn\":62}"
				+ ",{\"start\":[5,19],\"name\":\"delete\",\"opl\":1,\"end\":[5,19],\"osn\":63}"
				+ ",{\"start\":[5,18],\"name\":\"delete\",\"opl\":1,\"end\":[5,18],\"osn\":64}"
				+ ",{\"start\":[5,17],\"name\":\"delete\",\"opl\":1,\"end\":[5,17],\"osn\":65}"
				+ ",{\"start\":[5,16],\"name\":\"delete\",\"opl\":1,\"end\":[5,16],\"osn\":66}"
				+ ",{\"start\":[5,15],\"name\":\"delete\",\"opl\":1,\"end\":[5,15],\"osn\":67}"
				+ ",{\"start\":[5,14],\"name\":\"delete\",\"opl\":1,\"end\":[5,14],\"osn\":68}"
				+ ",{\"start\":[5,13],\"name\":\"delete\",\"opl\":1,\"end\":[5,13],\"osn\":69}"
				+ ",{\"start\":[5,12],\"name\":\"delete\",\"opl\":1,\"end\":[5,12],\"osn\":70}"
				+ ",{\"start\":[5,11],\"name\":\"delete\",\"opl\":1,\"end\":[5,11],\"osn\":71}"
				+ ",{\"start\":[5,10],\"name\":\"delete\",\"opl\":1,\"end\":[5,10],\"osn\":72}"
				+ ",{\"start\":[5,9],\"name\":\"delete\",\"opl\":1,\"end\":[5,9],\"osn\":73}"
				+ ",{\"start\":[5,8],\"name\":\"delete\",\"opl\":1,\"end\":[5,8],\"osn\":74}"
				+ ",{\"start\":[5,7],\"name\":\"delete\",\"opl\":1,\"end\":[5,7],\"osn\":75}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}
	
	@Test
	/**
	 * ODF space XML element that contains more than two spaces and was not
	 * completely deleted, returned the removed spaces as remaining element
	 * https://bugs.open-xchange.com/show_bug.cgi?id=28869
	 */
	public void hyperlinkUnderscoreRemovalTest() {
		final String SOURCE_FILE_NAME_TRUNC = "hyperlinkSpaces";
		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"start\":[5,7],\"attrs\":{\"character\":{\"underline\":false}},\"name\":\"setAttributes\",\"opl\":1,\"end\":[5,34],\"osn\":48}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}
	

	@Test
	/**
	 * Foreign ODF elements could not be cloned. Throwable are now catched and
	 * rethrown as filterexception to be found by Admin logging -- adding
	 * regression test references
	 * https://bugs.open-xchange.com/show_bug.cgi?id=28841
	 */
	public void foreignElementSplitTest() {
		final String SOURCE_FILE_NAME_TRUNC = "foreignElementSplit";
		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"start\":[1,9,1,1,2],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":495}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 28817 - Preceding nested paragraph in paragraph within unknown
	 * content parent was erroneously counted as component
	 * https://bugs.open-xchange.com/show_bug.cgi?id=28817
	 */
	public void insertCharacterBehindLineShapeWithParagraphTest() {
		final String SOURCE_FILE_NAME_TRUNC = "sample";
		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\"123456789\",\"start\":[4,22],\"opl\":9,\"name\":\"insertText\",\"osn\":85}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.importOnlyRegressionTest(SOURCE_FILE_NAME_TRUNC, ".odt", getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 28752 - ODT - Hyperlinks over full unformatted text looses first
	 * character. https://bugs.open-xchange.com/show_bug.cgi?id=28752
	 */
	public void anchorOverFullUnformattedTextTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\"eins  zwei\",\"start\":[0,0],\"opl\":1,\"name\":\"insertText\",\"osn\":16},{\"styleId\":\"Hyperlink\",\"styleName\":\"Hyperlink\",\"attrs\":{\"character\":{\"color\":{\"value\":\"hyperlink\",\"fallbackValue\":\"0080C0\",\"type\":\"scheme\"},\"underline\":true}},\"name\":\"insertStyleSheet\",\"opl\":1,\"parent\":null,\"uiPriority\":99,\"type\":\"character\",\"osn\":17},{\"start\":[0,0],\"attrs\":{\"styleId\":\"Hyperlink\",\"character\":{\"url\":\"http://www.heise.de\"}},\"name\":\"setAttributes\",\"opl\":1,\"end\":[0,10],\"osn\":18}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 28752 - ODT - Hyperlinks with spaces >1: Hyperlink/First letter is
	 * lost after reediting https://bugs.open-xchange.com/show_bug.cgi?id=28752
	 */
	public void anchorNoneBreakableTest() {
		final String SOURCE_FILE_NAME_TRUNC = "liste2";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"start\":[0,0],\"attrs\":{\"styleId\":\"Hyperlink\",\"character\":{\"url\":\"http://www.heise.de\"}},\"name\":\"setAttributes\",\"opl\":1,\"end\":[0,11],\"osn\":19}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 28722 - ODT - Remove attributte does not works
	 * https://bugs.open-xchange.com/show_bug.cgi?id=28722
	 */
	public void removeUnderlineTest() {
		final String SOURCE_FILE_NAME_TRUNC = "UNDERLINE";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"start\":[0,0],\"attrs\":{\"character\":{\"vertAlign\":null,\"fontName\":null,\"bold\":null,\"strike\":null,\"fillColor\":null,\"color\":null,\"underline\":null,\"italic\":null,\"language\":null,\"fontSize\":null}},\"name\":\"setAttributes\",\"opl\":1,\"end\":[0,8],\"osn\":18}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.importOnlyRegressionTest(SOURCE_FILE_NAME_TRUNC, ".odt", getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 28722 - ODT - Remove attributte does not works
	 * https://bugs.open-xchange.com/show_bug.cgi?id=28722
	 */
	public void removeUnderlineBoldAddColorTest() {
		final String SOURCE_FILE_NAME_TRUNC = "UNDERLINE";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"start\":[0,0],\"attrs\":{\"character\":{\"vertAlign\":null,\"fontName\":null,\"bold\":null,\"strike\":null,\"fillColor\":{\"value\":\"FF0000\",\"type\":\"rgb\"},\"color\":null,\"underline\":null,\"italic\":null,\"language\":null,\"fontSize\":null}},\"name\":\"setAttributes\",\"opl\":1,\"end\":[0,8],\"osn\":18}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.importOnlyRegressionTest(SOURCE_FILE_NAME_TRUNC, ".odt", getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 26437 - ODF toolkit: hyperlink is not properly overwritten with new
	 * hyperlink https://bugs.open-xchange.com/show_bug.cgi?id=26437
	 */
	public void hyperlinkOverridenByHyperlinkTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\"Empty Document\",\"start\":[0,0],\"opl\":1,\"name\":\"insertText\",\"osn\":21}"
				+ ",{\"start\":[0,2],\"attrs\":{\"styleId\":\"Hyperlink\",\"character\":{\"url\":\"http://www.heise.de\"}},\"name\":\"setAttributes\",\"opl\":1,\"end\":[0,8],\"osn\":22}"
				+ ",{\"start\":[0,1],\"attrs\":{\"styleId\":\"Hyperlink\",\"character\":{\"url\":\"http://www.mopo.de\"}},\"name\":\"setAttributes\",\"opl\":1,\"end\":[0,9],\"osn\":23}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 28392 - Split Paragraph at end of hyperlink clones hyperlink
	 * https://bugs.open-xchange.com/show_bug.cgi?id=28392
	 */
	public void listHyperlinkTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\"link\",\"start\":[0,0],\"opl\":4,\"name\":\"insertText\",\"osn\":16}"
				+ ",{\"styleId\":\"Hyperlink\",\"styleName\":\"Hyperlink\",\"attrs\":{\"character\":{\"color\":{\"value\":\"hyperlink\",\"fallbackValue\":\"0080C0\",\"type\":\"scheme\"},\"underline\":true}},\"name\":\"insertStyleSheet\",\"opl\":1,\"parent\":null,\"uiPriority\":99,\"type\":\"character\",\"osn\":20}"
				+ ",{\"start\":[0,2],\"attrs\":{\"styleId\":\"Hyperlink\",\"character\":{\"url\":\"http://www.heise.de\"}},\"name\":\"setAttributes\",\"opl\":1,\"end\":[0,2],\"osn\":21}"
				+ ",{\"start\":[0,4],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":22}"
				+ ",{\"text\":\"link2\",\"start\":[1,0],\"opl\":5,\"name\":\"insertText\",\"osn\":23}"
				+ ",{\"start\":[1,0],\"attrs\":{\"styleId\":\"Hyperlink\",\"character\":{\"url\":\"http://www.spiegel.de\"}},\"name\":\"setAttributes\",\"opl\":1,\"end\":[1,4],\"osn\":28}"
				+ ",{\"start\":[1,5],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":29}"
				+ ",{\"text\":\"link3\",\"start\":[2,0],\"attrs\":{\"styleId\":null,\"character\":{\"url\":null}},\"opl\":5,\"name\":\"insertText\",\"osn\":30}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 28645 - ODT - mixed list (numbered/ bullet list): numbering is
	 * different in preview /editor
	 * https://bugs.open-xchange.com/show_bug.cgi?id=28645
	 */
	public void listLevelContinousTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\"1. eins\",\"start\":[0,0],\"opl\":9,\"name\":\"insertText\",\"osn\":16}"
				+ ",{\"start\":[0,7],\"name\":\"insertTab\",\"opl\":1,\"osn\":25}"
				+ ",{\"start\":[0,7],\"name\":\"delete\",\"opl\":1,\"end\":[0,7],\"osn\":26}"
				+ ",{\"start\":[0,7],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":27}"
				+ ",{\"start\":[0,0],\"name\":\"delete\",\"opl\":1,\"end\":[0,2],\"osn\":28}"
				+ ",{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"upperRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L1\",\"osn\":29}"
				+ ",{\"styleId\":\"ListParagraph\",\"styleName\":\"List Paragraph\",\"attrs\":{\"paragraph\":{\"indentLeft\":1270,\"nextStyleId\":\"ListParagraph\",\"contextualSpacing\":true}},\"name\":\"insertStyleSheet\",\"opl\":1,\"parent\":\"Standard\",\"uiPriority\":34,\"type\":\"paragraph\",\"osn\":30}"
				+ ",{\"start\":[0],\"attrs\":{\"styleId\":\"ListParagraph\",\"paragraph\":{\"listStyleId\":\"L1\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":31}"
				+ ",{\"start\":[1],\"attrs\":{\"styleId\":\"ListParagraph\",\"paragraph\":{\"listStyleId\":\"L1\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":32}"
				+ ",{\"text\":\"zwei\",\"start\":[1,0],\"opl\":4,\"name\":\"insertText\",\"osn\":33}"
				+ ",{\"start\":[1,4],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":37}"
				+ ",{\"text\":\"drei\",\"start\":[2,0],\"opl\":4,\"name\":\"insertText\",\"osn\":38}"
				+ ",{\"start\":[2,4],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":42}"
				+ ",{\"text\":\"vier\",\"start\":[3,0],\"opl\":4,\"name\":\"insertText\",\"osn\":43}"
				+ ",{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"■\",\"indentLeft\":11430,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel6\":{\"listStartValue\":1,\"fontName\":\"Symbol\",\"levelText\":\"\",\"indentLeft\":8890,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel7\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"○\",\"indentLeft\":10160,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"defaultlist\":\"bullet\",\"listLevel5\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"■\",\"indentLeft\":7620,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel4\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"○\",\"indentLeft\":6350,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel3\":{\"listStartValue\":1,\"fontName\":\"Symbol\",\"levelText\":\"\",\"indentLeft\":5080,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel2\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"■\",\"indentLeft\":3810,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel1\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"○\",\"indentLeft\":2540,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel0\":{\"listStartValue\":1,\"fontName\":\"Symbol\",\"levelText\":\"\",\"indentLeft\":1270,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L2\",\"osn\":47}"
				+ ",{\"start\":[1],\"attrs\":{\"styleId\":\"ListParagraph\",\"paragraph\":{\"listStyleId\":\"L2\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":48}"
				+ ",{\"start\":[2],\"attrs\":{\"styleId\":\"ListParagraph\",\"paragraph\":{\"listStyleId\":\"L2\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":49}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Big table size limit Bug 26493 - ODT -Edit: check for table limitation is
	 * missing https://bugs.open-xchange.com/show_bug.cgi?id=26493
	 */
	public void listAnomalityTest() {
		final String SOURCE_FILE_NAME_TRUNC = "bigtable";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "[" + "]";

		editOperations.add(firstEditOperations);
		super.importOnlyRegressionTest(SOURCE_FILE_NAME_TRUNC, ".odt", getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 28570 - ODT numbered Lists after saving: Labels and the sublist
	 * entries are wrong https://bugs.open-xchange.com/show_bug.cgi?id=28570
	 */
	public void listLevelComplexTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"attrs\":{\"document\":{\"defaultTabStop\":1251}},\"name\":\"setDocumentAttributes\"},{\"styleName\":\"Default Paragraph Style\",\"styleId\":\"default_paragraph_style\",\"default\":true,\"hidden\":true,\"attrs\":{\"character\":{\"fontSizeAsian\":12,\"fontName\":\"Times New Roman\",\"fontNameComplex\":\"Mangal\",\"language\":\"de-DE\",\"fontSize\":12,\"fontNameAsian\":\"SimSun\"}},\"name\":\"insertStyleSheet\",\"type\":\"paragraph\"},{\"styleId\":\"Standard\",\"attrs\":{},\"name\":\"insertStyleSheet\",\"parent\":\"default_paragraph_style\",\"type\":\"paragraph\"},{\"styleId\":\"Caption\",\"attrs\":{\"paragraph\":{\"marginBottom\":212,\"marginTop\":212},\"character\":{\"fontSizeAsian\":12,\"italicComplex\":true,\"fontNameComplex\":\"Mangal1\",\"italicAsian\":true,\"italic\":true,\"fontSize\":12}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},{\"styleId\":\"Heading\",\"attrs\":{\"paragraph\":{\"nextStyleId\":\"Text_20_body\",\"marginTop\":423,\"marginBottom\":212},\"character\":{\"fontSizeAsian\":14,\"fontNameComplex\":\"Mangal\",\"fontName\":\"Arial\",\"fontSize\":14,\"fontNameAsian\":\"Microsoft YaHei\"}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},{\"styleId\":\"Index\",\"attrs\":{\"character\":{\"fontNameComplex\":\"Mangal1\"}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},{\"styleName\":\"List Paragraph\",\"styleId\":\"ListParagraph\",\"attrs\":{\"paragraph\":{\"nextStyleId\":\"ListParagraph\",\"indentLeft\":1270,\"marginLeft\":1270}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},{\"styleName\":\"Text body\",\"styleId\":\"Text_20_body\",\"attrs\":{\"paragraph\":{\"marginBottom\":212,\"marginTop\":0}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},{\"styleId\":\"List\",\"attrs\":{\"character\":{\"fontNameComplex\":\"Mangal1\"}},\"name\":\"insertStyleSheet\",\"parent\":\"Text_20_body\",\"type\":\"paragraph\"},{\"attrs\":{\"page\":{\"printOrientation\":\"portrait\",\"marginBottom\":2000,\"indentRight\":2000,\"height\":29700,\"indentLeft\":2000,\"marginTop\":2000,\"marginLeft\":2000,\"numberFormat\":\"1\",\"width\":21001,\"marginRight\":2000}},\"name\":\"setDocumentAttributes\"},{\"fontName\":\"Mangal1\",\"attrs\":{\"family\":\"Mangal\"},\"name\":\"insertFontDescription\"},{\"fontName\":\"Times New Roman\",\"attrs\":{\"family\":\"'Times New Roman'\",\"familyGeneric\":\"roman\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},{\"fontName\":\"Arial\",\"attrs\":{\"family\":\"Arial\",\"familyGeneric\":\"swiss\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},{\"fontName\":\"Mangal\",\"attrs\":{\"family\":\"Mangal\",\"familyGeneric\":\"system\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},{\"fontName\":\"Microsoft YaHei\",\"attrs\":{\"family\":\"'Microsoft YaHei'\",\"familyGeneric\":\"system\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},{\"fontName\":\"SimSun\",\"attrs\":{\"family\":\"SimSun\",\"familyGeneric\":\"system\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":\"1\",\"levelText\":\"%9.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":11430,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel6\":{\"listStartValue\":\"1\",\"levelText\":\"%7.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":8890,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel7\":{\"listStartValue\":\"1\",\"levelText\":\"%8.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":10160,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel5\":{\"listStartValue\":\"1\",\"levelText\":\"%6.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel4\":{\"listStartValue\":\"1\",\"levelText\":\"%5.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":\"1\",\"levelText\":\"%4.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5080,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":\"1\",\"levelText\":\"%3.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3810,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel1\":{\"listStartValue\":\"1\",\"levelText\":\"%2.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":2540,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"listStartValue\":\"1\",\"levelText\":\"%1.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"listStyleId\":\"L30001\"},{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L30001\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},{\"text\":\"eins\",\"start\":[0,0],\"name\":\"insertText\"},{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L30001\",\"listLevel\":1}},\"name\":\"insertParagraph\"},{\"text\":\"zwei\",\"start\":[1,0],\"name\":\"insertText\"},{\"start\":[2],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L30001\",\"listLevel\":2}},\"name\":\"insertParagraph\"},{\"text\":\"drei\",\"start\":[2,0],\"name\":\"insertText\"},{\"listDefinition\":{\"listLevel8\":{\"levelText\":\" \",\"labelFollowedBy\":\"listtab\",\"indentLeft\":11430,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel6\":{\"levelText\":\" \",\"labelFollowedBy\":\"listtab\",\"indentLeft\":8890,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel7\":{\"levelText\":\" \",\"labelFollowedBy\":\"listtab\",\"indentLeft\":10160,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel5\":{\"levelText\":\" \",\"labelFollowedBy\":\"listtab\",\"indentLeft\":7620,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel4\":{\"levelText\":\" \",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6350,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"levelText\":\" \",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5080,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"levelText\":\" \",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3810,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel1\":{\"levelText\":\" \",\"labelFollowedBy\":\"listtab\",\"indentLeft\":2540,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"levelText\":\" \",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1270,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"listStyleId\":\"L20014\"},{\"start\":[3],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L20014\",\"listLevel\":3}},\"name\":\"insertParagraph\"},{\"text\":\"vier\",\"start\":[3,0],\"name\":\"insertText\"},{\"start\":[4],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L20014\",\"listLevel\":3}},\"name\":\"insertParagraph\"},{\"text\":\"fünf\",\"start\":[4,0],\"name\":\"insertText\"},{\"start\":[5],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L20014\",\"listLevel\":3}},\"name\":\"insertParagraph\"},{\"text\":\"sechs\",\"start\":[5,0],\"name\":\"insertText\"},{\"listDefinition\":{\"listLevel8\":{\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":11430,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel6\":{\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":8890,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel7\":{\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":10160,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel5\":{\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":7620,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel4\":{\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6350,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5080,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3810,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel1\":{\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":2540,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1270,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"listStyleId\":\"L20001\"},{\"start\":[6],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L20001\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},{\"text\":\"bullet\",\"start\":[6,0],\"name\":\"insertText\"},{\"start\":[7],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L20001\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},{\"text\":\"bullet2\",\"start\":[7,0],\"name\":\"insertText\"},{\"start\":[8],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L20001\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},{\"text\":\"bullet3\",\"start\":[8,0],\"name\":\"insertText\"},"
				+ "{\"start\":[3],\"attrs\":{\"paragraph\":{\"listLevel\":2}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":37},{\"start\":[3],\"attrs\":{\"paragraph\":{\"listLevel\":3}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":38},{\"start\":[3],\"attrs\":{\"paragraph\":{\"listLevel\":2}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":39},{\"start\":[3],\"attrs\":{\"paragraph\":{\"listLevel\":3}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":40},{\"start\":[3],\"attrs\":{\"paragraph\":{\"listLevel\":2}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":41},{\"start\":[3],\"attrs\":{\"paragraph\":{\"listLevel\":1}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":42},{\"start\":[3],\"attrs\":{\"paragraph\":{\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":43},{\"start\":[3],\"attrs\":{\"paragraph\":{\"listLevel\":1}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":44},{\"start\":[3],\"attrs\":{\"paragraph\":{\"listLevel\":2}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":45},{\"start\":[3],\"attrs\":{\"paragraph\":{\"listLevel\":3}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":46},{\"start\":[3],\"attrs\":{\"paragraph\":{\"listLevel\":2}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":47},{\"start\":[3],\"attrs\":{\"paragraph\":{\"listLevel\":1}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":48},{\"start\":[3],\"attrs\":{\"paragraph\":{\"listLevel\":2}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":49},{\"start\":[3],\"attrs\":{\"paragraph\":{\"listLevel\":3}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":50},{\"start\":[2],\"name\":\"mergeParagraph\",\"opl\":1,\"osn\":51},{\"start\":[2,4],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":52},{\"start\":[5],\"attrs\":{\"paragraph\":{\"listLevel\":2}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":53},{\"start\":[5],\"attrs\":{\"paragraph\":{\"listLevel\":1}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":54},{\"start\":[5],\"attrs\":{\"paragraph\":{\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":55},{\"start\":[5],\"attrs\":{\"paragraph\":{\"listLevel\":-1}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":56},{\"start\":[5],\"attrs\":{\"paragraph\":{\"listStyleId\":null}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":57},{\"start\":[5],\"attrs\":{\"paragraph\":{\"style\":\"Standard\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":58},{\"start\":[4],\"name\":\"mergeParagraph\",\"opl\":1,\"osn\":59},{\"start\":[4,3],\"name\":\"delete\",\"opl\":1,\"end\":[4,3],\"osn\":60},{\"start\":[4,3],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":61},{\"start\":[4],\"attrs\":{\"paragraph\":{\"listLevel\":2}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":62},{\"start\":[4],\"attrs\":{\"paragraph\":{\"listLevel\":1}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":63},{\"start\":[4],\"attrs\":{\"paragraph\":{\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":64},{\"start\":[4],\"attrs\":{\"paragraph\":{\"listLevel\":-1}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":65},{\"start\":[4],\"attrs\":{\"paragraph\":{\"listStyleId\":null}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":66},{\"start\":[4],\"attrs\":{\"paragraph\":{\"style\":\"Standard\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":67},{\"start\":[3],\"name\":\"mergeParagraph\",\"opl\":1,\"osn\":68},{\"start\":[3,3],\"name\":\"delete\",\"opl\":1,\"end\":[3,3],\"osn\":69},{\"text\":\"r\",\"start\":[3,3],\"opl\":1,\"name\":\"insertText\",\"osn\":70},{\"start\":[3,4],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":71},{\"text\":\"f\",\"start\":[4,3],\"opl\":1,\"name\":\"insertText\",\"osn\":72},{\"start\":[4],\"name\":\"mergeParagraph\",\"opl\":1,\"osn\":73},{\"start\":[4,4],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":74},"
				+ "{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":\"1\",\"levelText\":\"%9.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":11430,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listIdentifier\":\"L30001\",\"listLevel6\":{\"listStartValue\":\"1\",\"levelText\":\"%7.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":8890,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevels\":[{\"listStartValue\":\"1\",\"levelText\":\"%1.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":\"1\",\"levelText\":\"%2.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":2540,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":\"1\",\"levelText\":\"%3.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3810,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":\"1\",\"levelText\":\"%4.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5080,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":\"1\",\"levelText\":\"%5.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":\"1\",\"levelText\":\"%6.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":\"1\",\"levelText\":\"%7.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":8890,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":\"1\",\"levelText\":\"%8.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":10160,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":\"1\",\"levelText\":\"%9.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":11430,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}],\"listLevel7\":{\"listStartValue\":\"1\",\"levelText\":\"%8.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":10160,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel5\":{\"listStartValue\":\"1\",\"levelText\":\"%6.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel4\":{\"listStartValue\":\"1\",\"levelText\":\"%5.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":\"1\",\"levelText\":\"%4.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5080,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"(%3)\",\"indentLeft\":3810,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel1\":{\"listStartValue\":\"1\",\"levelText\":\"%2.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":2540,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"listStartValue\":\"1\",\"levelText\":\"%1.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L4\",\"osn\":75}"
				+ ",{\"start\":[2],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L4\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":76}"
				+ ",{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L4\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":77}"
				+ ",{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L4\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":78}"
				+ ",{\"start\":[4],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L4\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":79}"
				+ ",{\"start\":[5],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L4\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":80}"
				+ ",{\"start\":[3],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L4\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":81}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 28570 - ODT numbered Lists after saving: Labels and the sublist
	 * entries are wrong https://bugs.open-xchange.com/show_bug.cgi?id=28570
	 */
	public void listMultipleSimularChangeTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";
		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\"1.\",\"start\":[0,0],\"opl\":2,\"name\":\"insertText\",\"osn\":16}"
				+ ",{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"defaultlist\":\"numbering\",\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"upperRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L1\",\"osn\":18}"
				+ ",{\"styleId\":\"ListParagraph\",\"styleName\":\"List Paragraph\",\"attrs\":{\"paragraph\":{\"indentLeft\":1270,\"nextStyleId\":\"ListParagraph\",\"contextualSpacing\":true}},\"name\":\"insertStyleSheet\",\"opl\":1,\"parent\":\"Standard\",\"uiPriority\":34,\"type\":\"paragraph\",\"osn\":19}"
				+ ",{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L1\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":20}"
				+ ",{\"start\":[0,2],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":21}"
				+ ",{\"start\":[1],\"attrs\":{\"paragraph\":{\"listLevel\":1}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":22}"
				+ ",{\"text\":\"a.\",\"start\":[1,0],\"opl\":2,\"name\":\"insertText\",\"osn\":23}"
				+ ",{\"start\":[1,2],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":25}"
				+ ",{\"text\":\"b.\",\"start\":[2,0],\"opl\":2,\"name\":\"insertText\",\"osn\":26}"
				+ ",{\"start\":[2,2],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":28}"
				+ ",{\"text\":\"c.\",\"start\":[3,0],\"opl\":2,\"name\":\"insertText\",\"osn\":29}"
				+ ",{\"start\":[3,2],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":31}"
				+ ",{\"text\":\"d.\",\"start\":[4,0],\"opl\":2,\"name\":\"insertText\",\"osn\":32}"
				+ ",{\"start\":[4,2],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":34}"
				+ ",{\"text\":\"2.\",\"start\":[5,0],\"opl\":2,\"name\":\"insertText\",\"osn\":35}"
				+ ",{\"start\":[5],\"attrs\":{\"paragraph\":{\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":39}"
				+ ",{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listIdentifier\":\"L1\",\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevels\":[{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"right\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"upperRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"right\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"right\"}],\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"upperRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2)\",\"indentLeft\":2540,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L2\",\"osn\":40}"
				+ ",{\"start\":[2],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":41}"
				+ ",{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":42}"
				+ ",{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":43}"
				+ ",{\"start\":[4],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":44}"
				+ ",{\"start\":[5],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":45}"
				+ ",{\"start\":[3],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":46}"
				+ "]";
		editOperations.add(firstEditOperations);
		super.importOnlyRegressionTest(SOURCE_FILE_NAME_TRUNC, ".odt", getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * https://bugs.open-xchange.com/show_bug.cgi?id=26660
	 *
	 * FIXED: 1) Changing an automatic styles, requires to rename it, if it is
	 * being used by others! FIXME: 2) outlineLevel:0 check, copy all attributes
	 * and descendent from text:p to text:h and vice versa..
	 */
	public void changeParagraphToHeaderTest() {
		final String SOURCE_FILE_NAME_TRUNC = "field";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"start\":[1,0],\"name\":\"splitParagraph\"},"
				+ "{\"text\":\"Level 1 Heading!\",\"start\":[2,0],\"name\":\"insertText\"},"
				+ "{\"styleName\":\"heading 1\",\"styleId\":\"heading1\",\"attrs\":{\"paragraph\":{\"outlineLevel\":0,\"nextStyleId\":\"Standard\"},\"character\":{\"bold\":true,\"color\":{\"value\":\"accent1\",\"fallbackValue\":\"376092\",\"transformations\":[{\"value\":74902,\"type\":\"shade\"}],\"type\":\"scheme\"},\"fontSize\":14}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"uiPriority\":9,\"type\":\"paragraph\"},"
				+ "{\"start\":[2],\"attrs\":{\"paragraph\":{\"alignment\":null,\"lineHeight\":null,\"fillColor\":null,\"marginTop\":null,\"outlineLevel\":null,\"nextStyleId\":null,\"listStyleId\":null,\"borderBottom\":null,\"listLevel\":null,\"contextualSpacing\":null,\"borderLeft\":null,\"showListLabel\":null,\"marginBottom\":null,\"tabStops\":null,\"listStartValue\":null,\"indentRight\":null,\"style\":\"heading1\",\"indentLeft\":null,\"borderTop\":null,\"borderInside\":null,\"indentFirstLine\":null,\"borderRight\":null},\"character\":{\"vertAlign\":null,\"strike\":null,\"bold\":null,\"fontName\":null,\"fillColor\":null,\"color\":null,\"underline\":null,\"language\":null,\"italic\":null,\"fontSize\":null,\"url\":null}},\"name\":\"setAttributes\"},"
				+ "{\"start\":[2,16],\"name\":\"splitParagraph\"},"
				+ "{\"start\":[2,16],\"name\":\"splitParagraph\"},"
				+ "{\"text\":\"No Heading!\",\"start\":[3,0],\"name\":\"insertText\"},"
				+ "{\"text\":\"Heading by Parent!\",\"start\":[4,0],\"name\":\"insertText\"},"
				+ "{\"styleName\":\"No Heading\",\"styleId\":\"noheading\",\"attrs\":{\"paragraph\":{\"nextStyleId\":\"Standard\"},\"character\":{\"bold\":true,\"color\":{\"value\":\"accent1\",\"fallbackValue\":\"376092\",\"transformations\":[{\"value\":74902,\"type\":\"shade\"}],\"type\":\"scheme\"},\"fontSize\":14}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"uiPriority\":9,\"type\":\"paragraph\"},"
				+ "{\"start\":[3],\"attrs\":{\"paragraph\":{\"alignment\":null,\"lineHeight\":null,\"fillColor\":null,\"marginTop\":null,\"outlineLevel\":null,\"nextStyleId\":null,\"listStyleId\":null,\"borderBottom\":null,\"listLevel\":null,\"contextualSpacing\":null,\"borderLeft\":null,\"showListLabel\":null,\"marginBottom\":null,\"tabStops\":null,\"listStartValue\":null,\"indentRight\":null,\"style\":\"noheading\",\"indentLeft\":null,\"borderTop\":null,\"borderInside\":null,\"indentFirstLine\":null,\"borderRight\":null},\"character\":{\"vertAlign\":null,\"strike\":null,\"bold\":null,\"fontName\":null,\"fillColor\":null,\"color\":null,\"underline\":null,\"language\":null,\"italic\":null,\"fontSize\":null,\"url\":null}},\"name\":\"setAttributes\"},"
				+ "{\"styleName\":\"No Heading But Parent\",\"styleId\":\"noheadingbutParent\",\"attrs\":{\"paragraph\":{\"nextStyleId\":\"Standard\"},\"character\":{\"bold\":true,\"color\":{\"value\":\"accent1\",\"fallbackValue\":\"376092\",\"transformations\":[{\"value\":74902,\"type\":\"shade\"}],\"type\":\"scheme\"},\"fontSize\":14}},\"name\":\"insertStyleSheet\",\"parent\":\"headingParent\",\"uiPriority\":9,\"type\":\"paragraph\"},"
				+ "{\"styleName\":\"Heading Parent\",\"styleId\":\"headingParent\",\"attrs\":{\"paragraph\":{\"outlineLevel\":3,\"nextStyleId\":\"Standard\"},\"character\":{\"bold\":true,\"color\":{\"value\":\"accent1\",\"fallbackValue\":\"376092\",\"transformations\":[{\"value\":74902,\"type\":\"shade\"}],\"type\":\"scheme\"},\"fontSize\":14}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"uiPriority\":9,\"type\":\"paragraph\"},"
				+ "{\"start\":[4],\"attrs\":{\"paragraph\":{\"alignment\":null,\"lineHeight\":null,\"fillColor\":null,\"marginTop\":null,\"outlineLevel\":null,\"nextStyleId\":null,\"listStyleId\":null,\"borderBottom\":null,\"listLevel\":null,\"contextualSpacing\":null,\"borderLeft\":null,\"showListLabel\":null,\"marginBottom\":null,\"tabStops\":null,\"listStartValue\":null,\"indentRight\":null,\"style\":\"headingParent\",\"indentLeft\":null,\"borderTop\":null,\"borderInside\":null,\"indentFirstLine\":null,\"borderRight\":null},\"character\":{\"vertAlign\":null,\"strike\":null,\"bold\":null,\"fontName\":null,\"fillColor\":null,\"color\":null,\"underline\":null,\"language\":null,\"italic\":null,\"fontSize\":null,\"url\":null}},\"name\":\"setAttributes\"},"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 26872 - List items across the document with the same style do not
	 * influence each other in ODF
	 * https://bugs.open-xchange.com/show_bug.cgi?id=26872
	 */
	public void listWithContinuousNumberingTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9)\",\"indentLeft\":11430,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7)\",\"indentLeft\":8890,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8)\",\"indentLeft\":10160,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6)\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5)\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4)\",\"indentLeft\":5080,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3)\",\"indentLeft\":3810,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2)\",\"indentLeft\":2540,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1)\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L30002\",\"osn\":16}"
				+ ",{\"styleId\":\"ListParagraph\",\"styleName\":\"List Paragraph\",\"attrs\":{\"paragraph\":{\"indentLeft\":1270,\"nextStyleId\":\"ListParagraph\",\"contextualSpacing\":true}},\"name\":\"insertStyleSheet\",\"opl\":1,\"parent\":\"Standard\",\"uiPriority\":34,\"type\":\"paragraph\",\"osn\":17}"
				+ ",{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L30002\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":18}"
				+ ",{\"text\":\"1)\",\"start\":[0,0],\"opl\":2,\"name\":\"insertText\",\"osn\":19}"
				+ ",{\"start\":[0,2],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":21}"
				+ ",{\"start\":[1],\"attrs\":{\"paragraph\":{\"listLevel\":1}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":22}"
				+ ",{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9)\",\"indentLeft\":11430,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listIdentifier\":\"L30002\",\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7)\",\"indentLeft\":8890,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevels\":[{\"listStartValue\":1,\"levelText\":\"%1)\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%2)\",\"indentLeft\":2540,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%3)\",\"indentLeft\":3810,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%4)\",\"indentLeft\":5080,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%5)\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%6)\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%7)\",\"indentLeft\":8890,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%8)\",\"indentLeft\":10160,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%9)\",\"indentLeft\":11430,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}],\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8)\",\"indentLeft\":10160,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6)\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5)\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4)\",\"indentLeft\":5080,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3)\",\"indentLeft\":3810,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2)\",\"indentLeft\":2540,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1)\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L2\",\"osn\":23}"
				+ ",{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":24}"
				+ ",{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":25}"
				+ ",{\"start\":[1],\"attrs\":{\"paragraph\":{\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":26}"
				+ ",{\"start\":[1],\"attrs\":{\"paragraph\":{\"listLevel\":1}},\"name\":\"setAttributes\",\"opl\":1,\"end\":[1],\"osn\":27}"
				+ ",{\"text\":\"a)\",\"start\":[1,0],\"opl\":2,\"name\":\"insertText\",\"osn\":28}"
				+ ",{\"start\":[1,2],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":30}"
				+ ",{\"text\":\"b)\",\"start\":[2,0],\"opl\":2,\"name\":\"insertText\",\"osn\":31}"
				+ ",{\"start\":[2,2],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":33}"
				+ ",{\"text\":\"c)\",\"start\":[3,0],\"opl\":2,\"name\":\"insertText\",\"osn\":34}"
				+ ",{\"start\":[3,2],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":36}"
				+ ",{\"start\":[4],\"attrs\":{\"paragraph\":{\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":37}"
				+ ",{\"text\":\"2)\",\"start\":[4,0],\"opl\":2,\"name\":\"insertText\",\"osn\":38}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.importOnlyRegressionTest(SOURCE_FILE_NAME_TRUNC, ".odt", getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 28400 - ODT - Reedit odt with List entries: list entries are
	 * displayed on the wrong position
	 * https://bugs.open-xchange.com/show_bug.cgi?id=28400
	 */
	public void listMixedFirstLevelFullTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\"1.\",\"start\":[0,0],\"opl\":2,\"name\":\"insertText\",\"osn\":16}"
				+ ",{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L30001\",\"osn\":18}"
				+ ",{\"styleId\":\"ListParagraph\",\"styleName\":\"List Paragraph\",\"attrs\":{\"paragraph\":{\"indentLeft\":1270,\"nextStyleId\":\"ListParagraph\",\"contextualSpacing\":true}},\"name\":\"insertStyleSheet\",\"opl\":1,\"parent\":\"Standard\",\"uiPriority\":34,\"type\":\"paragraph\",\"osn\":19}"
				+ ",{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L30001\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":20}"
				+ ",{\"start\":[0,2],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":21}"
				+ ",{\"text\":\"2.\",\"start\":[1,0],\"opl\":2,\"name\":\"insertText\",\"osn\":22}"
				+ ",{\"start\":[1,2],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":24}"
				+ ",{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listIdentifier\":\"L30001\",\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevels\":[{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}],\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L2\",\"osn\":25}"
				+ ",{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":26}"
				+ ",{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":27}"
				+ ",{\"start\":[2],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":28}"
				+ ",{\"start\":[2],\"attrs\":{\"paragraph\":{\"listStyleId\":\"L30001\"}},\"name\":\"setAttributes\",\"opl\":1,\"end\":[2],\"osn\":29}"
				+ ",{\"start\":[0],\"attrs\":{\"paragraph\":{\"listStyleId\":\"L30001\"}},\"name\":\"setAttributes\",\"opl\":1,\"end\":[0],\"osn\":30}"
				+ ",{\"start\":[1],\"attrs\":{\"paragraph\":{\"listStyleId\":\"L30001\"}},\"name\":\"setAttributes\",\"opl\":1,\"end\":[1],\"osn\":31}"
				+ ",{\"name\":\"deleteListStyle\",\"opl\":1,\"listStyleId\":\"L2\",\"osn\":32}"
				+ ",{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L30011\",\"osn\":35}"
				+ ",{\"start\":[2],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L30011\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":36}"
				+ ",{\"start\":[2,2],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":37}"
				+ ",{\"text\":\"b.\",\"start\":[3,0],\"opl\":2,\"name\":\"insertText\",\"osn\":38}"
				+ ",{\"start\":[3,2],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":40}"
				+ ",{\"text\":\"1.\",\"start\":[4,0],\"opl\":2,\"name\":\"insertText\",\"osn\":41}"
				+ ",{\"start\":[4],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L30001\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":43}"
				+ ",{\"start\":[4,0],\"name\":\"delete\",\"opl\":1,\"end\":[4,0],\"osn\":44}"
				+ ",{\"text\":\"3\",\"start\":[4,0],\"opl\":1,\"name\":\"insertText\",\"osn\":45}"
				+ ",{\"start\":[4,2],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":46}"
				+ ",{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listIdentifier\":\"L30001\",\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevels\":[{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}],\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1)\",\"indentLeft\":1270,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L3\",\"osn\":47}"
				+ ",{\"start\":[4],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L3\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":48}"
				+ ",{\"start\":[5],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L3\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":49}"
				+ ",{\"start\":[5],\"attrs\":{\"paragraph\":{\"listStyleId\":\"L30001\"}},\"name\":\"setAttributes\",\"opl\":1,\"end\":[5],\"osn\":50}"
				+ ",{\"start\":[4],\"attrs\":{\"paragraph\":{\"listStyleId\":\"L30001\"}},\"name\":\"setAttributes\",\"opl\":1,\"end\":[4],\"osn\":51}"
				+ ",{\"name\":\"deleteListStyle\",\"opl\":1,\"listStyleId\":\"L3\",\"osn\":52}"
				+ ",{\"text\":\"4\",\"start\":[5,0],\"opl\":1,\"name\":\"insertText\",\"osn\":53}"
				+ ",{\"start\":[5],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":null,\"listLevel\":-1}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":54}"
				+ ",{\"start\":[5],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L30001\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"end\":[5],\"osn\":55}"
				+ ",{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"■\",\"indentLeft\":11430,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel6\":{\"listStartValue\":1,\"fontName\":\"Symbol\",\"levelText\":\"\",\"indentLeft\":8890,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel7\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"○\",\"indentLeft\":10160,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"defaultlist\":\"bullet\",\"listLevel5\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"■\",\"indentLeft\":7620,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel4\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"○\",\"indentLeft\":6350,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel3\":{\"listStartValue\":1,\"fontName\":\"Symbol\",\"levelText\":\"\",\"indentLeft\":5080,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel2\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"■\",\"indentLeft\":3810,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel1\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"○\",\"indentLeft\":2540,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel0\":{\"listStartValue\":1,\"fontName\":\"Symbol\",\"levelText\":\"\",\"indentLeft\":1270,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L3\",\"osn\":56}"
				+ ",{\"start\":[5],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L3\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":57}"
				+ ",{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"defaultlist\":\"numbering\",\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"upperRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L4\",\"osn\":58}"
				+ ",{\"start\":[5],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L4\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":59}"
				+ ",{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listIdentifier\":\"L4\",\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevels\":[{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"right\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"upperRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"right\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"right\"}],\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"upperRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L5\",\"osn\":60}"
				+ ",{\"start\":[5],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L5\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":61}"
				+ ",{\"start\":[5],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":null,\"listLevel\":-1}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":62}"
				+ ",{\"start\":[5],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L4\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":63}"
				+ ",{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9)\",\"indentLeft\":11430,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7)\",\"indentLeft\":8890,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8)\",\"indentLeft\":10160,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6)\",\"indentLeft\":7620,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5)\",\"indentLeft\":6350,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4)\",\"indentLeft\":5080,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3)\",\"indentLeft\":3810,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2)\",\"indentLeft\":2540,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1)\",\"indentLeft\":1270,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L30012\",\"osn\":64}"
				+ ",{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L30012\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":65}"
				+ ",{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L30012\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":66}"
				+ ",{\"start\":[4],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L30012\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":67}"
				+ ",{\"start\":[5],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L30012\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":68}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.importOnlyRegressionTest(SOURCE_FILE_NAME_TRUNC, ".odt", getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 28400 - ODT - Reedit odt with List entries: list entries are
	 * displayed on the wrong position
	 * https://bugs.open-xchange.com/show_bug.cgi?id=28400
	 */
	public void listMixedFirstLevelTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\"1.\",\"start\":[0,0],\"opl\":2,\"name\":\"insertText\",\"osn\":16}"
				+ ",{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},"
				+ "\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,"
				+ "\"listStyleId\":\"L30001\",\"osn\":18}"
				+ ",{\"styleId\":\"ListParagraph\",\"styleName\":\"List Paragraph\",\"attrs\":{\"paragraph\":{\"indentLeft\":1270,\"nextStyleId\":\"ListParagraph\",\"contextualSpacing\":true}},\"name\":\"insertStyleSheet\",\"opl\":1,\"parent\":\"Standard\",\"uiPriority\":34,\"type\":\"paragraph\",\"osn\":19}"
				+ ",{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L30001\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":20}"
				+ ",{\"start\":[0,2],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":21}"
				+ ",{\"text\":\"2.\",\"start\":[1,0],\"opl\":2,\"name\":\"insertText\",\"osn\":22}"
				+ ",{\"start\":[1,2],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":24}"
				+ ",{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listIdentifier\":\"L30001\",\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevels\":[{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}],"
				+ "\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},"
				+ "\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L2\",\"osn\":25}"
				+ ",{\"text\":\"a.\",\"start\":[2,0],\"opl\":2,\"name\":\"insertText\",\"osn\":33}"
				+ ",{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":26}"
				+ ",{\"text\":\"b.\",\"start\":[2,0],\"opl\":2,\"name\":\"insertText\",\"osn\":33}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.importOnlyRegressionTest(SOURCE_FILE_NAME_TRUNC, ".odt", getTestMethodName(), editOperations);

		//super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 28400 - ODT - Reedit odt with List entries: list entries are
	 * displayed on the wrong position
	 * https://bugs.open-xchange.com/show_bug.cgi?id=28400
	 */
	public void list6LevelTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L30001\",\"osn\":16}"
				+ ",{\"styleId\":\"ListParagraph\",\"styleName\":\"List Paragraph\",\"attrs\":{\"paragraph\":{\"indentLeft\":1270,\"nextStyleId\":\"ListParagraph\",\"contextualSpacing\":true}},\"name\":\"insertStyleSheet\",\"opl\":1,\"parent\":\"Standard\",\"uiPriority\":34,\"type\":\"paragraph\",\"osn\":17}"
				+ ",{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L30001\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":18}"
				+ ",{\"text\":\"eins\",\"start\":[0,0],\"opl\":4,\"name\":\"insertText\",\"osn\":19}"
				+ ",{\"start\":[0,4],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":23}"
				+ ",{\"start\":[1],\"attrs\":{\"paragraph\":{\"listLevel\":1}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":24}"
				+ ",{\"text\":\"zwei\",\"start\":[1,0],\"opl\":4,\"name\":\"insertText\",\"osn\":31}"
				+ ",{\"start\":[1,4],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":35}"
				+ ",{\"start\":[2],\"attrs\":{\"paragraph\":{\"listLevel\":2}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":36}"
				+ ",{\"text\":\"drei\",\"start\":[2,0],\"opl\":4,\"name\":\"insertText\",\"osn\":37}"
				+ ",{\"start\":[2,4],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":41}"
				+ ",{\"start\":[3],\"attrs\":{\"paragraph\":{\"listLevel\":3}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":42}"
				+ ",{\"text\":\"vier\",\"start\":[3,0],\"opl\":4,\"name\":\"insertText\",\"osn\":43}"
				+ ",{\"start\":[3,4],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":47}"
				+ ",{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listIdentifier\":\"L30001\",\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevels\":[{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}],\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4)\",\"indentLeft\":5080,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L2\",\"osn\":48}"
				+ ",{\"start\":[3],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":49}"
				+ ",{\"start\":[2],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":50}"
				+ ",{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":51}"
				+ ",{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":52}"
				+ ",{\"start\":[4],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":53}"
				+ ",{\"start\":[4],\"attrs\":{\"paragraph\":{\"listLevel\":4}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":54}"
				+ ",{\"text\":\"fuenf\",\"start\":[4,0],\"opl\":5,\"name\":\"insertText\",\"osn\":55}"
				+ ",{\"start\":[4,5],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":60}"
				+ ",{\"start\":[5],\"attrs\":{\"paragraph\":{\"listLevel\":5}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":61}"
				+ ",{\"text\":\"sechs\",\"start\":[5,0],\"opl\":5,\"name\":\"insertText\",\"osn\":62}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Create a complex bullet list!
	 *
	 */
	public void bulletListTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"attrs\":{\"document\":{\"defaultTabStop\":1251}},\"name\":\"setDocumentAttributes\"},"
				+ "{\"styleId\":\"default_paragraph_style\",\"default\":true,\"hidden\":true,\"attrs\":{\"character\":{\"fontSizeAsian\":12,\"fontName\":\"Times New Roman\",\"fontNameComplex\":\"Mangal\",\"language\":\"de-DE\",\"fontSize\":12,\"fontNameAsian\":\"SimSun\"}},\"name\":\"insertStyleSheet\",\"type\":\"paragraph\"},"
				+ "{\"styleId\":\"Standard\",\"attrs\":{},\"name\":\"insertStyleSheet\",\"parent\":\"default_paragraph_style\",\"type\":\"paragraph\"},"
				+ "{\"styleId\":\"Caption\",\"attrs\":{\"paragraph\":{\"marginBottom\":212,\"marginTop\":212},\"character\":{\"fontSizeAsian\":12,\"italicComplex\":true,\"fontNameComplex\":\"Mangal1\",\"italicAsian\":true,\"italic\":true,\"fontSize\":12}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},"
				+ "{\"styleId\":\"Heading\",\"attrs\":{\"paragraph\":{\"nextStyleId\":\"Text_20_body\",\"marginTop\":423,\"marginBottom\":212},\"character\":{\"fontSizeAsian\":14,\"fontNameComplex\":\"Mangal\",\"fontName\":\"Arial\",\"fontSize\":14,\"fontNameAsian\":\"Microsoft YaHei\"}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},"
				+ "{\"styleId\":\"Index\",\"attrs\":{\"character\":{\"fontNameComplex\":\"Mangal1\"}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},"
				+ "{\"styleName\":\"List Paragraph\",\"styleId\":\"ListParagraph\",\"attrs\":{\"paragraph\":{\"nextStyleId\":\"ListParagraph\",\"indentLeft\":1270,\"marginLeft\":1270}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},"
				+ "{\"styleName\":\"Text body\",\"styleId\":\"Text_20_body\",\"attrs\":{\"paragraph\":{\"marginBottom\":212,\"marginTop\":0}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},"
				+ "{\"styleId\":\"List\",\"attrs\":{\"character\":{\"fontNameComplex\":\"Mangal1\"}},\"name\":\"insertStyleSheet\",\"parent\":\"Text_20_body\",\"type\":\"paragraph\"},"
				+ "{\"attrs\":{\"page\":{\"printOrientation\":\"portrait\",\"marginBottom\":2000,\"indentRight\":2000,\"height\":29700,\"indentLeft\":2000,\"marginTop\":2000,\"marginLeft\":2000,\"numberFormat\":\"1\",\"width\":21001,\"marginRight\":2000}},\"name\":\"setDocumentAttributes\"},"
				+ "{\"fontName\":\"Mangal1\",\"attrs\":{\"family\":\"Mangal\"},\"name\":\"insertFontDescription\"},"
				+ "{\"fontName\":\"Times New Roman\",\"attrs\":{\"family\":\"'Times New Roman'\",\"familyGeneric\":\"roman\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},"
				+ "{\"fontName\":\"Arial\",\"attrs\":{\"family\":\"Arial\",\"familyGeneric\":\"swiss\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},"
				+ "{\"fontName\":\"Mangal\",\"attrs\":{\"family\":\"Mangal\",\"familyGeneric\":\"system\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},"
				+ "{\"fontName\":\"Microsoft YaHei\",\"attrs\":{\"family\":\"'Microsoft YaHei'\",\"familyGeneric\":\"system\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},"
				+ "{\"fontName\":\"SimSun\",\"attrs\":{\"family\":\"SimSun\",\"familyGeneric\":\"system\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},"
				+ "{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":\"1\",\"levelText\":\"%9\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":11430,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},"
				+ "	\"listLevel6\":{\"listStartValue\":\"1\",\"levelText\":\"%7\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":8890,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},"
				+ " \"listLevel7\":{\"listStartValue\":\"1\",\"levelText\":\"%8\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":10160,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},"
				+ " \"listLevel5\":{\"listStartValue\":\"1\",\"levelText\":\"%6\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel4\":{\"listStartValue\":\"1\",\"levelText\":\"%5\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":\"1\",\"levelText\":\"%4\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5080,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":\"1\",\"levelText\":\"%3\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3810,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},"
				+ "\"listLevel1\":{\"listStartValue\":\"1\",\"levelText\":\"%2\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":2540,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},"
				+ "\"listLevel0\":{\"listStartValue\":\"1\",\"levelText\":\"(%1\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1270,\"numberFormat\":\"lowerRoman\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"listStyleId\":\"L2\"},"
				+ "{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"1)\",\"start\":[0,0],\"name\":\"insertText\"},"
				+ "{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"(i)\",\"start\":[1,0],\"name\":\"insertText\"},"
				+ "{\"start\":[2],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"(ii)\",\"start\":[2,0],\"name\":\"insertText\"},"
				+ "{\"start\":[3],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"(iii)\",\"start\":[3,0],\"name\":\"insertText\"},"
				+ "{\"start\":[4],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"2)\",\"start\":[4,0],\"name\":\"insertText\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.importOnlyRegressionTest(SOURCE_FILE_NAME_TRUNC, ".odt", getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 26723 - ODT: Changing bullet for one list item changes whole bullet
	 * list https://bugs.open-xchange.com/show_bug.cgi?id=26723
	 */
	public void changeListStyleOfFirstItemTest() {
		final String SOURCE_FILE_NAME_TRUNC = "simple bullet list";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"listDefinition\":{\"listLevel8\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":6350,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6350,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listIdentifier\":\"L1\",\"listLevel6\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":5080,\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5080,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevels\":[{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":1270,\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1270,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":1905,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1905,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":2540,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":2540,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":3175,\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3175,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":3810,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3810,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":4445,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":4445,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":5080,\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5080,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":5715,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5715,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":6350,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6350,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635}],\"listLevel7\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":5715,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5715,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel5\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":4445,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":4445,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel4\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":3810,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3810,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel3\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":3175,\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3175,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel2\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":2540,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":2540,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel1\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":1905,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1905,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L3\",\"osn\":30},"
				+ "{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L3\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":31},"
				+ "{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L3\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":32},"
				+ "{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"L1\"}},\"name\":\"setAttributes\",\"opl\":1,\"end\":[1],\"osn\":33},"
				+ "{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"L1\"}},\"name\":\"setAttributes\",\"opl\":1,\"end\":[0],\"osn\":34},"
				+ "{\"name\":\"deleteListStyle\",\"opl\":1,\"listStyleId\":\"L3\",\"osn\":35},"
				+ "{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listIdentifier\":\"L2\",\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevels\":[{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},"
				+ "{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},"
				+ "{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},"
				+ "{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},"
				+ "{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"upperRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},"
				+ "{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},"
				+ "{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},"
				+ "{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},"
				+ "{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"right\"}],\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"upperRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1)\",\"indentLeft\":1270,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L3\",\"osn\":36},"
				+ "{\"start\":[2],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L3\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":37},"
				+ "{\"listDefinition\":{\"listLevel8\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":6350,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6350,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listIdentifier\":\"L1\",\"listLevel6\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":5080,\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5080,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevels\":[{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":1270,\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1270,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":1905,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1905,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":2540,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":2540,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":3175,\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3175,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":3810,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3810,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":4445,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":4445,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":5080,\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5080,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":5715,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5715,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":6350,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6350,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635}],\"listLevel7\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":5715,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5715,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel5\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":4445,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":4445,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel4\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":3810,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3810,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel3\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":3175,\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3175,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel2\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":2540,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":2540,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel1\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":1905,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1905,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1)\",\"indentLeft\":1270,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L4\",\"osn\":38},"
				+ "{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L4\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":39},"
				+ "{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L4\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":40},"
				+ "{\"listDefinition\":{\"listLevel8\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":6350,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6350,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listIdentifier\":\"L4\",\"listLevel6\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":5080,\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5080,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevels\":[{\"listStartValue\":1,\"levelText\":\"%1)\",\"indentLeft\":1270,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":1905,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1905,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":2540,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":2540,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":3175,\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3175,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":3810,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3810,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":4445,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":4445,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":5080,\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5080,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":5715,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5715,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},"
				+ "{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":6350,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6350,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635}],\"listLevel7\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":5715,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5715,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel5\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":4445,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":4445,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel4\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":3810,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3810,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel3\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":3175,\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3175,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel2\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":2540,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":2540,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel1\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":1905,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1905,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1)\",\"indentLeft\":1270,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L5\",\"osn\":41},"
				+ "{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L5\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":42},"
				+ "{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L5\"}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":43}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.importOnlyRegressionTest(SOURCE_FILE_NAME_TRUNC, ".odt", getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 27821 - ODT - Numbered list: After saving the numbering list contains
	 * wrong labels https://bugs.open-xchange.com/show_bug.cgi?id=27821
	 */
	public void editSubListTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\"a\",\"start\":[0,0],\"opl\":1,\"name\":\"insertText\",\"osn\":16},"
				+ "{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"■\",\"indentLeft\":11430,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel6\":{\"listStartValue\":1,\"fontName\":\"Symbol\",\"levelText\":\"\",\"indentLeft\":8890,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel7\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"○\",\"indentLeft\":10160,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"defaultlist\":\"bullet\",\"listLevel5\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"■\",\"indentLeft\":7620,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel4\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"○\",\"indentLeft\":6350,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel3\":{\"listStartValue\":1,\"fontName\":\"Symbol\",\"levelText\":\"\",\"indentLeft\":5080,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel2\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"■\",\"indentLeft\":3810,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel1\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"○\",\"indentLeft\":2540,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel0\":{\"listStartValue\":1,\"fontName\":\"Symbol\",\"levelText\":\"\",\"indentLeft\":1270,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L1\",\"osn\":17},"
				+ "{\"styleId\":\"ListParagraph\",\"styleName\":\"List Paragraph\",\"attrs\":{\"paragraph\":{\"indentLeft\":1270,\"nextStyleId\":\"ListParagraph\",\"contextualSpacing\":true}},\"name\":\"insertStyleSheet\",\"opl\":1,\"parent\":\"Standard\",\"uiPriority\":34,\"type\":\"paragraph\",\"osn\":18},"
				+ "{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L1\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":19},"
				+ "{\"start\":[0,1],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":20},"
				+ "{\"text\":\"b\",\"start\":[1,0],\"opl\":1,\"name\":\"insertText\",\"osn\":21},"
				+ "{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"defaultlist\":\"numbering\",\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"upperRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L2\",\"osn\":22},"
				+ "{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":23},"
				+ "{\"start\":[1],\"attrs\":{\"paragraph\":{\"listLevel\":1}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":24},"
				+ "{\"start\":[1,1],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":25},"
				+ "{\"text\":\"b\",\"start\":[2,0],\"opl\":1,\"name\":\"insertText\",\"osn\":26},"
				+ "{\"start\":[2,1],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":27},"
				+ "{\"text\":\"c\",\"start\":[3,0],\"opl\":1,\"name\":\"insertText\",\"osn\":28},"
				+ "{\"start\":[3,1],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":29},"
				+ "{\"text\":\"d\",\"start\":[4,0],\"opl\":1,\"name\":\"insertText\",\"osn\":30},"
				+ "{\"start\":[2],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L1\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":31},"
				+ "{\"start\":[2],\"attrs\":{\"paragraph\":{\"listLevel\":1}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":32}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 27821 - ODT - Numbered list: After saving the numbering list contains
	 * wrong labels https://bugs.open-xchange.com/show_bug.cgi?id=27821
	 */
	public void listStyleChangeTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\"A\",\"start\":[0,0],\"opl\":1,\"name\":\"insertText\",\"osn\":16},"
				+ "{\"listDefinition\":{\""
				+ "listLevel8\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"■\",\"indentLeft\":11430,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\""
				+ "listLevel6\":{\"listStartValue\":1,\"fontName\":\"Symbol\",\"levelText\":\"\",\"indentLeft\":8890,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\""
				+ "listLevel7\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"○\",\"indentLeft\":10160,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"defaultlist\":\"bullet\",\""
				+ "listLevel5\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"■\",\"indentLeft\":7620,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\""
				+ "listLevel4\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"○\",\"indentLeft\":6350,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\""
				+ "listLevel3\":{\"listStartValue\":1,\"fontName\":\"Symbol\",\"levelText\":\"\",\"indentLeft\":5080,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\""
				+ "listLevel2\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"■\",\"indentLeft\":3810,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\""
				+ "listLevel1\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"○\",\"indentLeft\":2540,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\""
				+ "listLevel0\":{\"listStartValue\":1,\"fontName\":\"Symbol\",\"levelText\":\"\",\"indentLeft\":1270,\"numberFormat\":\"bullet\",\"textAlign\":\"left\",\"indentFirstLine\":-635}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L1\",\"osn\":17},"
				+ "{\"styleId\":\"ListParagraph\",\"styleName\":\"List Paragraph\",\"attrs\":{\"paragraph\":{\"indentLeft\":1270,\"nextStyleId\":\"ListParagraph\",\"contextualSpacing\":true}},\"name\":\"insertStyleSheet\",\"opl\":1,\"parent\":\"Standard\",\"uiPriority\":34,\"type\":\"paragraph\",\"osn\":18},"
				+ "{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L1\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":19},"
				+ "{\"start\":[0,1],\"name\":\"splitParagraph\",\"opl\":1,\"osn\":20},"
				+ "{\"text\":\"B\",\"start\":[1,0],\"opl\":1,\"name\":\"insertText\",\"osn\":21},"
				+ "{\"listDefinition\":{\""
				+ "listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\""
				+ "listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\""
				+ "listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"defaultlist\":\"numbering\",\""
				+ "listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\""
				+ "listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"upperRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\""
				+ "listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\""
				+ "listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\""
				+ "listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\""
				+ "listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"opl\":1,\"listStyleId\":\"L2\",\"osn\":22},"
				+ "{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L2\",\"listLevel\":0}},\"name\":\"setAttributes\",\"opl\":1,\"osn\":23}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	/**
	 * Bug 26743 - ODT: inserting table after image doubles image
	 * https://bugs.open-xchange.com/show_bug.cgi?id=26743
	 */
	@Test
	public void insertTableBehindImage() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"start\":[0,0],\"attrs\":{\"drawing\":{\"marginBottom\":317,\"height\":16933,\"imageUrl\":\"http://no-real-server/skyline.png\",\"marginTop\":317,\"marginLeft\":317,\"width\":12700,\"marginRight\":317}},\"name\":\"insertDrawing\",\"type\":\"image\"},"
				+ "{\"start\":[0,1],\"name\":\"splitParagraph\"},"
				+ "{\"styleId\":\"TableGrid\",\"styleName\":\"Table Grid\",\"attrs\":{\"wholeTable\":{\"paragraph\":{\"lineHeight\":{\"value\":100,\"type\":\"percent\"}},\"table\":{\"borderInsideVert\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17},\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17},\"borderInsideHor\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17},\"borderBottom\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17},\"borderLeft\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17},\"borderRight\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17},\"paddingBottom\":0,\"paddingLeft\":190,\"paddingRight\":190}}},\"name\":\"insertStyleSheet\",\"uiPriority\":59,\"type\":\"table\"},"
				+ "{\"start\":[2],\"attrs\":{\"table\":{\"style\":\"TableGrid\",\"tableGrid\":[1000,1000,1000],\"width\":\"auto\",\"exclude\":[\"lastRow\",\"lastCol\",\"bandsVert\"]}},\"name\":\"insertTable\"},{\"count\":3,\"start\":[2,0],\"name\":\"insertRows\",\"insertDefaultCells\":true}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.importOnlyRegressionTest(SOURCE_FILE_NAME_TRUNC, ".odt", getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 26721 - ODT: Bullets in bullet list not saved
	 * https://bugs.open-xchange.com/show_bug.cgi?id=26721
	 */
	public void splitListParagraph2() {
		final String SOURCE_FILE_NAME_TRUNC = "simple bullet list 1_pre OX";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"start\":[2,8],\"name\":\"splitParagraph\"},"
				+ "{\"text\":\"s\",\"start\":[3,0],\"name\":\"insertText\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.importOnlyRegressionTest(SOURCE_FILE_NAME_TRUNC, ".odt", getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 26872 - ODT - splitParagraph does only split the paragraph and not
	 * create a new list item
	 * https://bugs.open-xchange.com/show_bug.cgi?id=26872
	 */
	public void splitListParagraph() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\"ab\",\"start\":[0,0],\"name\":\"insertText\"},"
				+ "{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"defaultlist\":\"numbering\",\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"upperRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"lowerRoman\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"upperLetter\",\"indentFirstLine\":-635,\"textAlign\":\"right\"},\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"listStyleId\":\"L1\"},"
				+ "{\"styleId\":\"ListParagraph\",\"styleName\":\"List Paragraph\",\"attrs\":{\"paragraph\":{\"indentLeft\":1270,\"nextStyleId\":\"ListParagraph\",\"contextualSpacing\":true}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"uiPriority\":34,\"type\":\"paragraph\"},{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L1\",\"listLevel\":0}},\"name\":\"setAttributes\"},"
				+ "{\"start\":[0,1],\"name\":\"splitParagraph\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.importOnlyRegressionTest(SOURCE_FILE_NAME_TRUNC, ".odt", getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 27057 - ODT - bulletlist: placed text behind a bulletlist is lost,
	 * after closing the odt https://bugs.open-xchange.com/show_bug.cgi?id=27057
	 *
	 * The deletion of one of two paragraphs within a list, resulted into the
	 * deletion of both, as the routine believed the second was empty after
	 * deletion within.
	 */
	public void textAfterListTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations1 = "["
				+ "{\"text\":\"a\",\"start\":[0,0],\"name\":\"insertText\"},"
				+ "{\"start\":[0,1],\"name\":\"splitParagraph\"},"
				+ "{\"listDefinition\":{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"lowerRoman\",\"textAlign\":\"right\",\"indentFirstLine\":-635},\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"lowerLetter\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"defaultlist\":\"numbering\",\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"upperLetter\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"decimal\",\"textAlign\":\"right\",\"indentFirstLine\":-635},\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"%5.\",\"indentLeft\":6350,\"numberFormat\":\"upperRoman\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"lowerRoman\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"upperLetter\",\"textAlign\":\"right\",\"indentFirstLine\":-635},\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"lowerLetter\",\"textAlign\":\"left\",\"indentFirstLine\":-635},\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"%1.\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"textAlign\":\"left\",\"indentFirstLine\":-635}},\"name\":\"insertListStyle\",\"listStyleId\":\"L1\"},{\"styleName\":\"List Paragraph\",\"styleId\":\"ListParagraph\",\"attrs\":{\"paragraph\":{\"indentLeft\":1270,\"nextStyleId\":\"ListParagraph\",\"contextualSpacing\":true}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"uiPriority\":34,\"type\":\"paragraph\"},{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listStyleId\":\"L1\",\"listLevel\":0}},\"name\":\"setAttributes\"},"
				+ "{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":null,\"listLevel\":-1}},\"name\":\"setAttributes\"},"
				+ "{\"text\":\"b\",\"start\":[1,0],\"name\":\"insertText\"}"
				+ "]";

		editOperations.add(firstEditOperations1);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, ".odt", getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * This test if the debug output does work: 1) Applies operations to a test
	 * document 2) Checks if operation and original document exist 3) Applies
	 * again operation to the test document 4) Checks if two operation files and
	 * same original document as previously exist
	 */
	public void debugOperationTest2() {
		final String SOURCE_FILE_NAME_TRUNC = "feature_bullets_numbering";

		List<String> editOperations = new ArrayList();
		String firstEditOperations1 = "["
				+ "{\"name\":\"insertText\",\"start\":[0,10],\"text\":\"Svante & \"},"
				+ "]";
		editOperations.add(firstEditOperations1);
		String savedDocumentPath = super.roundtripRegressionWithResourcesTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations, null, true);
		OdfPackage pkg = null;
		try {
			// return the absolute path to the test directory
			savedDocumentPath = ResourceUtilities.getTestOutputFolder() + savedDocumentPath;
			pkg = OdfPackage.loadPackage(savedDocumentPath);
		} catch (Exception ex) {
			Assert.fail("The saved document '" + savedDocumentPath + "' could not be found!");
			Logger.getLogger(LatestOngoingTest.class.getName()).log(Level.SEVERE, null, ex);
		}
		if (pkg == null || !pkg.contains(TextOperationDocument.ORIGNAL_ODT_FILE)) {
			Assert.fail("The original document '" + TextOperationDocument.ORIGNAL_ODT_FILE + "' could not be found in the saved document '" + savedDocumentPath + "'!");
		}

		String firstEditOperations2 = "["
				+ "{\"start\":[0],\"name\":\"delete\"},"
				+ "]";
		editOperations.clear();
		editOperations.add(firstEditOperations2);

		pkg = null;
		try {
			String previousOutputOdfPath = OUTPUT_DIRECTORY + SOURCE_FILE_NAME_TRUNC + ".odt" + HYPEN + getTestMethodName() + HYPEN + "OUT" + HYPEN + "org" + ".odt";
			Map<String, Object> configuration = new HashMap<String, Object>();
			configuration.put("debugoperations", Boolean.TRUE);
			TextOperationDocument doc = new TextOperationDocument(new FileInputStream(ResourceUtilities.newTestOutputFile(previousOutputOdfPath)), configuration);
			savedDocumentPath = super.roundtripOperationTest(doc, SOURCE_FILE_NAME_TRUNC, getTestMethodName() + HYPEN + "OUT" + HYPEN + "org", ".odt", editOperations, null, false, true, false, null, true);
			pkg = OdfPackage.loadPackage(ResourceUtilities.getTestOutputFolder() + savedDocumentPath);
		} catch (Exception ex) {
			Assert.fail("The saved document '" + savedDocumentPath + "' could not be found!");
			Logger.getLogger(LatestOngoingTest.class.getName()).log(Level.SEVERE, null, ex);
		}
		if (pkg == null || !pkg.contains(TextOperationDocument.ORIGNAL_ODT_FILE)) {
			Assert.fail("The original document '" + TextOperationDocument.ORIGNAL_ODT_FILE + "' could not be found in the saved document '" + savedDocumentPath + "'!");
		}
		if (!pkg.contains(TextOperationDocument.OPERATION_TEXT_FILE_PREFIX + "1.txt")) {
			Assert.fail("The original document '" + TextOperationDocument.OPERATION_TEXT_FILE_PREFIX + "1.txt" + "' could not be found in the saved document '" + savedDocumentPath + "'!");
		}
		if (!pkg.contains(TextOperationDocument.OPERATION_TEXT_FILE_PREFIX + "2.txt")) {
			Assert.fail("The original document '" + TextOperationDocument.OPERATION_TEXT_FILE_PREFIX + "2.txt" + "' could not be found in the saved document '" + savedDocumentPath + "'!");
		}
		if (!pkg.contains(TextOperationDocument.OPERATION_REVISON_FILE)) {
			Assert.fail("The original document '" + TextOperationDocument.OPERATION_REVISON_FILE + "' could not be found in the saved document '" + savedDocumentPath + "'!");
		}
	}

	@Test
	/**
	 * Bug 26831 - ODT: insert hyperlink via clipboard not saved
	 * https://bugs.open-xchange.com/show_bug.cgi?id=26831
	 *
	 *
	 * The width of an image should be limited to its page width (page width -
	 * left&rigth margin) QUESTION: Is the height adapted to keep the ratio?
	 */
	public void markHyperlinkAllTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\"http://www.bild.de/\",\"start\":[0,0],\"name\":\"insertText\"},"
				+ "{\"styleName\":\"Hyperlink\",\"styleId\":\"Hyperlink\",\"attrs\":{\"character\":{\"color\":{\"value\":\"hyperlink\",\"fallbackValue\":\"0080C0\",\"type\":\"scheme\"},\"underline\":true}},\"name\":\"insertStyleSheet\",\"parent\":null,\"uiPriority\":99,\"type\":\"character\"},"
				+ "{\"start\":[0,0],\"attrs\":{\"styleId\":\"Hyperlink\",\"character\":{\"url\":\"http://www.bild.de/\"}},\"name\":\"setAttributes\",\"end\":[0,19]}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 26831 - ODT: insert hyperlink via clipboard not saved
	 * https://bugs.open-xchange.com/show_bug.cgi?id=26831
	 *
	 *
	 * The width of an image should be limited to its page width (page width -
	 * left&rigth margin) QUESTION: Is the height adapted to keep the ratio?
	 */
	public void markHyperlinkFirstCharacterTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\"http://www.bild.de/\",\"start\":[0,0],\"name\":\"insertText\"},"
				+ "{\"styleName\":\"Hyperlink\",\"styleId\":\"Hyperlink\",\"attrs\":{\"character\":{\"color\":{\"value\":\"hyperlink\",\"fallbackValue\":\"0080C0\",\"type\":\"scheme\"},\"underline\":true}},\"name\":\"insertStyleSheet\",\"parent\":null,\"uiPriority\":99,\"type\":\"character\"},"
				+ "{\"start\":[0,0],\"attrs\":{\"character\":{\"style\":\"Hyperlink\",\"url\":\"http://www.bild.de/\"}},\"name\":\"setAttributes\",\"end\":[0,0]}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	@Ignore // Have to disable the randomization of automatic styles first
	/**
	 * Bug 26793 - ODT: Resized cell size not saved
	 * https://bugs.open-xchange.com/show_bug.cgi?id=26793
	 */
	public void changeCellSizeTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"styleName\":\"Table Grid\",\"styleId\":\"TableGrid\",\"attrs\":{\"wholeTable\":{\"paragraph\":{\"lineHeight\":{\"value\":100,\"type\":\"percent\"}},\"table\":{\"borderInsideVert\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17},\"borderTop\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17},\"borderInsideHor\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17},\"borderBottom\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17},\"borderRight\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17},\"borderLeft\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17}}}},\"name\":\"insertStyleSheet\",\"uiPriority\":59,\"type\":\"table\"},"
				+ "{\"start\":[1],\"attrs\":{\"table\":{\"style\":\"TableGrid\",\"width\":\"auto\",\"tableGrid\":[1000,1000,1000,1000,1000,1000],\"exclude\":[\"lastRow\",\"lastCol\",\"bandsVert\"]}},\"name\":\"insertTable\"},"
				+ "{\"count\":1,\"start\":[1,0],\"name\":\"insertRows\",\"insertDefaultCells\":true},"
				+ "{\"start\":[1],\"attrs\":{\"table\":{\"width\":\"auto\",\"tableGrid\":[1002,1437,566,566,33,33]}},\"name\":\"setAttributes\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Verified - https://bugs.open-xchange.com/show_bug.cgi?id=26748
	 */
	public void insertLargeImageTest() {
		final String SOURCE_FILE_NAME_TRUNC = "image-attributes";
		final String UID = "d03f7d7218eb";
		final String INTERNAL_IMAGE_PATH = "Pictures/uid" + UID + ".jpg";
		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"start\":[0,14],\"attrs\":{\"drawing\":{\"marginBottom\":317,\"height\":5982,\"marginLeft\":317,\"marginTop\":317,\"imageUrl\":\"" + INTERNAL_IMAGE_PATH + "\",\"marginRight\":317,\"width\":17013}},\"name\":\"insertDrawing\",\"type\":\"image\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		Map<Long, byte[]> resourceMap = new HashMap<Long, byte[]>();
		byte[] imageBytes = null;
		try {
			imageBytes = ResourceUtilities.loadFileAsBytes(ResourceUtilities.getReferenceFile("Herschel-Horsehead-Nebula.jpeg"));
		} catch (IOException ex) {
			Logger.getLogger(LatestOngoingTest.class.getName()).log(Level.SEVERE, null, ex);
			Assert.fail(ex.getMessage());
		}
		long uid = Long.parseLong(UID, 16);
		resourceMap.put(uid, imageBytes);
		String savedDocumentPath = super.roundtripRegressionWithResourcesTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations, resourceMap, true);
		OdfPackage pkg = null;
		try {
			// return the absolute path to the test directory
			savedDocumentPath = ResourceUtilities.getTestOutputFolder() + savedDocumentPath;
			pkg = OdfPackage.loadPackage(savedDocumentPath);
		} catch (Exception ex) {
			Assert.fail("The saved document '" + savedDocumentPath + "' could not be found!");
			Logger.getLogger(LatestOngoingTest.class.getName()).log(Level.SEVERE, null, ex);
		}
		if (pkg == null || !pkg.contains(INTERNAL_IMAGE_PATH)) {
			Assert.fail("The image '" + INTERNAL_IMAGE_PATH + "' could not be found in the saved document '" + savedDocumentPath + "'!");
		}
	}

	@Test
	/**
	 * Bug 25255 - ODT: Table borders expected to be on table style (extend ODF)
	 * https://bugs.open-xchange.com/show_bug.cgi?id=25255
	 */
	public void paddingRoundtripTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"attrs\":{\"document\":{\"defaultTabStop\":2401}},\"name\":\"setDocumentAttributes\"},{\"styleId\":\"default_paragraph_style\",\"default\":true,\"hidden\":true,\"attrs\":{\"character\":{\"fontSizeAsian\":12,\"fontName\":\"Times New Roman\",\"fontNameComplex\":\"Mangal\",\"language\":\"de-DE\",\"fontSize\":12,\"fontNameAsian\":\"SimSun\"}},\"name\":\"insertStyleSheet\",\"type\":\"paragraph\"},{\"styleId\":\"Standard\",\"attrs\":{},\"name\":\"insertStyleSheet\",\"parent\":\"default_paragraph_style\",\"type\":\"paragraph\"},{\"styleName\":\"Table Contents\",\"styleId\":\"Table_20_Contents\",\"attrs\":{},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},{\"styleName\":\"Text body\",\"styleId\":\"Text_20_body\",\"attrs\":{\"paragraph\":{\"marginBottom\":212,\"marginTop\":0}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},{\"styleId\":\"Caption\",\"attrs\":{\"paragraph\":{\"marginBottom\":212,\"marginTop\":212},\"character\":{\"fontSizeAsian\":12,\"italicComplex\":true,\"fontNameComplex\":\"Mangal1\",\"italicAsian\":true,\"italic\":true,\"fontSize\":12}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},{\"styleId\":\"Index\",\"attrs\":{\"character\":{\"fontNameComplex\":\"Mangal1\"}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},{\"styleId\":\"Heading\",\"attrs\":{\"paragraph\":{\"nextStyleId\":\"Text_20_body\",\"marginTop\":423,\"marginBottom\":212},\"character\":{\"fontSizeAsian\":14,\"fontNameComplex\":\"Mangal\",\"fontName\":\"Arial\",\"fontSize\":14,\"fontNameAsian\":\"Microsoft YaHei\"}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},{\"styleId\":\"List\",\"attrs\":{\"character\":{\"fontNameComplex\":\"Mangal1\"}},\"name\":\"insertStyleSheet\",\"parent\":\"Text_20_body\",\"type\":\"paragraph\"},{\"attrs\":{\"page\":{\"printOrientation\":\"portrait\",\"marginBottom\":2000,\"indentRight\":2000,\"height\":29700,\"indentLeft\":2000,\"marginTop\":2000,\"marginLeft\":2000,\"numberFormat\":\"1\",\"width\":21001,\"marginRight\":2000}},\"name\":\"setDocumentAttributes\"},{\"fontName\":\"Mangal1\",\"attrs\":{\"family\":\"Mangal\"},\"name\":\"insertFontDescription\"},{\"fontName\":\"Times New Roman\",\"attrs\":{\"family\":\"'Times New Roman'\",\"familyGeneric\":\"roman\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},{\"fontName\":\"Arial\",\"attrs\":{\"family\":\"Arial\",\"familyGeneric\":\"swiss\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},{\"fontName\":\"Mangal\",\"attrs\":{\"family\":\"Mangal\",\"familyGeneric\":\"system\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},{\"fontName\":\"Microsoft YaHei\",\"attrs\":{\"family\":\"'Microsoft YaHei'\",\"familyGeneric\":\"system\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},{\"fontName\":\"SimSun\",\"attrs\":{\"family\":\"SimSun\",\"familyGeneric\":\"system\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},{\"start\":[0],\"attrs\":{\"table\":{\"tableGrid\":[32767,32768]}},\"name\":\"insertTable\"},{\"start\":[0,0],\"name\":\"insertRows\"},{\"start\":[0,0,0],\"attrs\":{\"cell\":{\"paddingTop\":1300,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":2,\"space\":1300},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":2,\"space\":1401},\"borderRight\":{\"style\":\"none\",\"space\":1199},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":2,\"space\":1101},\"paddingBottom\":1401,\"paddingLeft\":1101,\"paddingRight\":1199}},\"name\":\"insertCells\"},{\"start\":[0,0,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Table_20_Contents\"}},\"name\":\"insertParagraph\"},{\"text\":\"x\",\"start\":[0,0,0,0,0],\"name\":\"insertText\"},{\"start\":[0,0,1],\"attrs\":{\"cell\":{\"paddingTop\":1300,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":2},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":2},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":2},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":2},\"paddingBottom\":1401,\"paddingLeft\":1101,\"paddingRight\":1199}},\"name\":\"insertCells\"},{\"start\":[0,0,1,0],\"attrs\":{\"paragraph\":{\"style\":\"Table_20_Contents\"}},\"name\":\"insertParagraph\"},{\"text\":\"y\",\"start\":[0,0,1,0,0],\"name\":\"insertText\"},{\"start\":[0,1],\"name\":\"insertRows\"},{\"start\":[0,1,0],\"attrs\":{\"cell\":{\"paddingTop\":1300,\"borderTop\":{\"style\":\"none\",\"space\":1300},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":2,\"space\":1401},\"borderRight\":{\"style\":\"none\",\"space\":1199},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":2,\"space\":1101},\"paddingBottom\":1401,\"paddingLeft\":1101,\"paddingRight\":1199}},\"name\":\"insertCells\"},{\"start\":[0,1,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Table_20_Contents\"}},\"name\":\"insertParagraph\"},{\"text\":\"z\",\"start\":[0,1,0,0,0],\"name\":\"insertText\"},{\"start\":[0,1,1],\"attrs\":{\"cell\":{\"paddingTop\":1300,\"borderTop\":{\"style\":\"none\",\"space\":1300},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":2,\"space\":1401},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":2,\"space\":1199},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":2,\"space\":1101},\"paddingBottom\":1401,\"paddingLeft\":1101,\"paddingRight\":1199}},\"name\":\"insertCells\"},{\"start\":[0,1,1,0],\"attrs\":{\"paragraph\":{\"style\":\"Table_20_Contents\"}},\"name\":\"insertParagraph\"},{\"text\":\"1\",\"start\":[0,1,1,0,0],\"name\":\"insertText\"},{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"Standard\"}},\"name\":\"insertParagraph\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 26784 - ODT: Paragraph style 'List Paragraph' not saved
	 * https://bugs.open-xchange.com/show_bug.cgi?id=26784
	 */
	public void leftIndentStyleTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\"List\",\"start\":[0,0],\"name\":\"insertText\"},"
				+ "{\"styleName\":\"List Paragraph\",\"styleId\":\"ListParagraph\",\"attrs\":{\"paragraph\":{\"indentLeft\":1270,\"nextStyleId\":\"ListParagraph\",\"contextualSpacing\":true}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"uiPriority\":34,\"type\":\"paragraph\"},"
				+ "{\"start\":[0],\"attrs\":{\"paragraph\":{\"alignment\":null,\"lineHeight\":null,\"fillColor\":null,\"marginTop\":null,\"outlineLevel\":null,\"nextStyleId\":null,\"borderBottom\":null,\"contextualSpacing\":null,\"borderLeft\":null,\"showListLabel\":null,\"marginBottom\":null,\"tabStops\":null,\"listStartValue\":null,\"indentRight\":null,\"style\":\"ListParagraph\",\"indentLeft\":null,\"borderTop\":null,\"borderInside\":null,\"indentFirstLine\":null,\"borderRight\":null},\"character\":{\"vertAlign\":null,\"strike\":null,\"bold\":null,\"fontName\":null,\"fillColor\":null,\"color\":null,\"underline\":null,\"language\":null,\"italic\":null,\"fontSize\":null,\"url\":null}},\"name\":\"setAttributes\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 26780 - ODT: Table border to close to the text
	 * https://bugs.open-xchange.com/show_bug.cgi?id=26780
	 */
	public void tableBorderTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\"peng\",\"start\":[0,0],\"name\":\"insertText\"},"
				+ "{\"start\":[0],\"attrs\":{\"paragraph\":{\"borderTop\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17,\"space\":140},\"borderInside\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17,\"space\":140},\"borderRight\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17,\"space\":140},\"borderLeft\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17,\"space\":140}}},\"name\":\"setAttributes\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * FIXED - https://bugs.open-xchange.com/show_bug.cgi?id=26678
	 */
	public void simpleList3CopyPasteTest() {
		final String SOURCE_FILE_NAME_TRUNC = "simpleList3";
		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"start\":[16,0],\"name\":\"splitParagraph\"},"
				+ "{\"start\":[16],\"attrs\":{\"paragraph\":{\"alignment\":null,\"lineHeight\":null,\"fillColor\":null,\"marginTop\":null,\"outlineLevel\":null,\"nextStyleId\":null,\"listStyleId\":\"L1\",\"borderBottom\":null,\"listLevel\":0,\"contextualSpacing\":null,\"borderLeft\":null,\"showListLabel\":null,\"marginBottom\":null,\"tabStops\":null,\"listStartValue\":null,\"indentRight\":null,\"style\":\"Standard\",\"indentLeft\":null,\"borderTop\":null,\"borderInside\":null,\"indentFirstLine\":null,\"borderRight\":null},\"character\":{\"vertAlign\":null,\"strike\":null,\"bold\":null,\"fontName\":null,\"fillColor\":null,\"color\":null,\"underline\":null,\"language\":null,\"italic\":null,\"fontSize\":null,\"url\":null}},\"name\":\"setAttributes\"},"
				+ "{\"text\":\"First-followed-by-space\",\"start\":[16,0],\"attrs\":{\"character\":{\"vertAlign\":null,\"strike\":null,\"bold\":null,\"fontName\":null,\"style\":null,\"fillColor\":null,\"color\":null,\"underline\":null,\"language\":null,\"italic\":null,\"fontSize\":null,\"url\":null}},\"name\":\"insertText\"},"
				+ "{\"start\":[17],\"attrs\":{\"paragraph\":{\"indentRight\":0,\"indentLeft\":2401,\"style\":\"Standard\",\"listStyleId\":\"L1\",\"indentFirstLine\":-635,\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"Second\",\"start\":[17,0],\"name\":\"insertText\"},"
				+ "{\"start\":[18],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"L1\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"Third\",\"start\":[18,0],\"name\":\"insertText\"},"
				+ "{\"start\":[19],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"L2\",\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"start\":[20],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"L2\",\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"Cross-First-followed-by-space\",\"start\":[20,0],\"name\":\"insertText\"},"
				+ "{\"start\":[21],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"L2\",\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"Cross-Second\",\"start\":[21,0],\"name\":\"insertText\"},"
				+ "{\"start\":[22],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"L2\",\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"Cross-Third\",\"start\":[22,0],\"name\":\"insertText\"},"
				+ "{\"start\":[23],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"L2\",\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"start\":[24],\"attrs\":{\"paragraph\":{\"style\":\"Standard\"}},\"name\":\"insertParagraph\"},"
				+ "{\"start\":[25],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"L3\",\"listLevel\":0},\"character\":{\"underline\":false}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"Star-First-followed-by-space\",\"start\":[25,0],\"name\":\"insertText\"},"
				+ "{\"start\":[26],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"L3\",\"listLevel\":0},\"character\":{\"underline\":false}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"Star-Second\",\"start\":[26,0],\"name\":\"insertText\"},"
				+ "{\"start\":[27],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"L3\",\"listLevel\":0},\"character\":{\"underline\":false}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"Star-Third\",\"start\":[27,0],\"name\":\"insertText\"},"
				+ "{\"start\":[28],\"attrs\":{\"paragraph\":{\"style\":\"Standard\"},\"character\":{\"underline\":false}},\"name\":\"insertParagraph\"},"
				+ "{\"start\":[29],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"L4\",\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"LabelTest1\",\"start\":[29,0],\"name\":\"insertText\"},"
				+ "{\"start\":[29,0],\"attrs\":{\"character\":{\"underline\":false}},\"name\":\"setAttributes\",\"end\":[29,9]},"
				+ "{\"start\":[30],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"L4\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"LabelTest2\",\"start\":[30,0],\"name\":\"insertText\"},"
				+ "{\"start\":[30,0],\"attrs\":{\"character\":{\"underline\":false}},\"name\":\"setAttributes\",\"end\":[30,9]},"
				+ "{\"start\":[31],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"L4\",\"listLevel\":2}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"LabelTest3\",\"start\":[31,0],\"name\":\"insertText\"},"
				+ "{\"start\":[31,0],\"attrs\":{\"character\":{\"underline\":false}},\"name\":\"setAttributes\",\"end\":[31,9]}"
				+ "]";
		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Moving root level images (usually page aligned) within a paragraph
	 *
	 * Verified: https://bugs.open-xchange.com/show_bug.cgi?id=26653
	 */
	public void loadPageAlignedImageTest() {
		final String SOURCE_FILE_NAME_TRUNC = "odf-test-images";
		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "]";
		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Bug 26806 - ODT: insertTab not saved
	 * https://bugs.open-xchange.com/show_bug.cgi?id=26806
	 */
	public void insertTabTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\"Hallo\",\"start\":[0,0],\"name\":\"insertText\"},"
				+ "{\"start\":[0,0],\"name\":\"insertTab\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	public void simpleListDocTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty_as_can_be";
		String firstEditOperations = "["
				+ "{\"attrs\":{\"document\":{\"defaultTabStop\":2401}},\"name\":\"setDocumentAttributes\"},"
				+ "{\"styleId\":\"default_paragraph_style\",\"default\":true,\"hidden\":true,\"attrs\":{\"character\":{\"fontSizeAsian\":12,\"fontName\":\"Times New Roman\",\"fontNameComplex\":\"Mangal\",\"language\":\"de-DE\",\"fontSize\":12,\"fontNameAsian\":\"SimSun\"}},\"name\":\"insertStyleSheet\",\"type\":\"paragraph\"},"
				+ "{\"styleId\":\"Standard\",\"attrs\":{},\"name\":\"insertStyleSheet\",\"parent\":\"default_paragraph_style\",\"type\":\"paragraph\"},"
				+ "{\"styleName\":\"Text body\",\"styleId\":\"Text_20_body\",\"attrs\":{\"paragraph\":{\"marginBottom\":212,\"marginTop\":0}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},"
				+ "{\"styleId\":\"Caption\",\"attrs\":{\"paragraph\":{\"marginBottom\":212,\"marginTop\":212},\"character\":{\"fontSizeAsian\":12,\"italicComplex\":true,\"fontNameComplex\":\"Mangal1\",\"italicAsian\":true,\"italic\":true,\"fontSize\":12}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},"
				+ "{\"styleId\":\"Index\",\"attrs\":{\"character\":{\"fontNameComplex\":\"Mangal1\"}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},"
				+ "{\"styleId\":\"Heading\",\"attrs\":{\"paragraph\":{\"nextStyleId\":\"Text_20_body\",\"marginTop\":423,\"marginBottom\":212},\"character\":{\"fontSizeAsian\":14,\"fontNameComplex\":\"Mangal\",\"fontName\":\"Arial\",\"fontSize\":14,\"fontNameAsian\":\"Microsoft YaHei\"}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},"
				+ "{\"styleId\":\"List\",\"attrs\":{\"character\":{\"fontNameComplex\":\"Mangal1\"}},\"name\":\"insertStyleSheet\",\"parent\":\"Text_20_body\",\"type\":\"paragraph\"},"
				+ "{\"styleName\":\"Bullet Symbols\",\"styleId\":\"Bullet_20_Symbols\",\"attrs\":{\"character\":{\"fontNameAsian\":\"OpenSymbol\",\"fontNameComplex\":\"OpenSymbol\",\"fontName\":\"OpenSymbol\"}},\"name\":\"insertStyleSheet\",\"parent\":\"default_character_style\",\"type\":\"character\"},"
				+ "{\"styleName\":\"Numbering Symbols\",\"styleId\":\"Numbering_20_Symbols\",\"attrs\":{},\"name\":\"insertStyleSheet\",\"parent\":\"default_character_style\",\"type\":\"character\"},"
				+ "{\"attrs\":{\"page\":{\"printOrientation\":\"portrait\",\"marginBottom\":2000,\"indentRight\":2000,\"height\":29700,\"indentLeft\":2000,\"marginTop\":2000,\"marginLeft\":2000,\"numberFormat\":\"1\",\"width\":21001,\"marginRight\":2000}},\"name\":\"setDocumentAttributes\"},"
				+ "{\"fontName\":\"StarSymbol\",\"attrs\":{\"family\":\"StarSymbol\"},\"name\":\"insertFontDescription\"},"
				+ "{\"fontName\":\"Mangal1\",\"attrs\":{\"family\":\"Mangal\"},\"name\":\"insertFontDescription\"},"
				+ "{\"fontName\":\"OpenSymbol\",\"attrs\":{\"family\":\"OpenSymbol\"},\"name\":\"insertFontDescription\"},"
				+ "{\"fontName\":\"Times New Roman\",\"attrs\":{\"family\":\"'Times New Roman'\",\"familyGeneric\":\"roman\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},"
				+ "{\"fontName\":\"Arial\",\"attrs\":{\"family\":\"Arial\",\"familyGeneric\":\"swiss\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},"
				+ "{\"fontName\":\"Mangal\",\"attrs\":{\"family\":\"Mangal\",\"familyGeneric\":\"system\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},"
				+ "{\"fontName\":\"Microsoft YaHei\",\"attrs\":{\"family\":\"'Microsoft YaHei'\",\"familyGeneric\":\"system\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},"
				+ "{\"fontName\":\"SimSun\",\"attrs\":{\"family\":\"SimSun\",\"familyGeneric\":\"system\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},"
				+ "{\"listDefinition\":{\"listLevel8\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":6350,\"listStartValue\":\"7\",\"levelText\":\"%1.%2.%3.%4.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel9\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":6985,\"listStartValue\":\"7\",\"levelText\":\"%1.%2.%3.%4.%5.%6.%7.%8.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6985,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel6\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":5080,\"listStartValue\":\"7\",\"levelText\":\"%1.%2.%3.%4.%5.%6.%7.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5080,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel7\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":5715,\"listStartValue\":\"7\",\"levelText\":\"%1.%2.%3.%4.%5.%6.%7.%8.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5715,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel5\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":4445,\"listStartValue\":\"7\",\"levelText\":\"%1.%2.%3.%4.%5.%6.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":4445,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel4\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":3810,\"listStartValue\":\"7\",\"levelText\":\"%1.%2.%3.%4.%5.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3810,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel3\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":3175,\"listStartValue\":\"7\",\"levelText\":\"%1.%2.%3.%4.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3175,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel2\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":2540,\"listStartValue\":\"7\",\"levelText\":\"%1.%2.%3.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":2540,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel1\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":1905,\"listStartValue\":\"7\",\"levelText\":\"b%1.%2a\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1905,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel0\":{\"styleId\":\"Numbering_20_Symbols\",\"listStartValue\":\"7\",\"levelText\":\"b%1a\",\"labelFollowedBy\":\"space\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635}},\"name\":\"insertListStyle\",\"listStyleId\":\"L1\"},"
				+ "{\"start\":[0],\"attrs\":{\"paragraph\":{\"listXmlId\":\"list2505945715557893221\",\"style\":\"Standard\",\"listId\":\"list2505945715557893221\",\"listStyleId\":\"L1\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"First-followed-by-space\",\"start\":[0,0],\"name\":\"insertText\"},"
				+ "{\"start\":[1],\"attrs\":{\"paragraph\":{\"listXmlId\":\"list2505945715557893221\",\"indentRight\":0,\"style\":\"Standard\",\"indentLeft\":2401,\"listId\":\"list2505945715557893221\",\"marginLeft\":2401,\"marginRight\":0,\"listStyleId\":\"L1\",\"listStart\":true,\"listLevel\":0,\"indentFirstLine\":-635}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"Second\",\"start\":[1,0],\"name\":\"insertText\"},"
				+ "{\"start\":[2],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"L1\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"Third\",\"start\":[2,0],\"name\":\"insertText\"},"
				+ "{\"listDefinition\":{\"listLevel8\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":6350,\"levelText\":\"✗\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6350,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel9\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":6985,\"levelText\":\"✗\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6985,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel6\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":5080,\"levelText\":\"✗\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5080,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel7\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":5715,\"levelText\":\"✗\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5715,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel5\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":4445,\"levelText\":\"✗\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":4445,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel4\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":3810,\"levelText\":\"✗\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3810,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel3\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":3175,\"levelText\":\"✗\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3175,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel2\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":2540,\"levelText\":\"✗\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":2540,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel1\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":1905,\"levelText\":\"✗\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1905,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel0\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":1270,\"levelText\":\"✗\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1270,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635}},\"name\":\"insertListStyle\",\"listStyleId\":\"L2\"},"
				+ "{\"start\":[3],\"attrs\":{\"paragraph\":{\"listXmlId\":\"list6626444102102110991\",\"listLabelHidden\":true,\"style\":\"Standard\",\"listId\":\"list6626444102102110991\",\"listStyleId\":\"L2\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"start\":[4],\"attrs\":{\"paragraph\":{\"listXmlId\":\"list6626444102102110991\",\"style\":\"Standard\",\"listId\":\"list6626444102102110991\",\"listStyleId\":\"L2\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"Cross-First-followed-by-space\",\"start\":[4,0],\"name\":\"insertText\"},"
				+ "{\"start\":[5],\"attrs\":{\"paragraph\":{\"listXmlId\":\"list6626444102102110991\",\"style\":\"Standard\",\"listId\":\"list6626444102102110991\",\"listStyleId\":\"L2\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"Cross-Second\",\"start\":[5,0],\"name\":\"insertText\"},"
				+ "{\"start\":[6],\"attrs\":{\"paragraph\":{\"listXmlId\":\"list6626444102102110991\",\"style\":\"Standard\",\"listId\":\"list6626444102102110991\",\"listStyleId\":\"L2\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"Cross-Third\",\"start\":[6,0],\"name\":\"insertText\"},"
				+ "{\"start\":[7],\"attrs\":{\"paragraph\":{\"listXmlId\":\"list6626444102102110991\",\"listLabelHidden\":true,\"style\":\"Standard\",\"listId\":\"list6626444102102110991\",\"listStyleId\":\"L2\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"start\":[8],\"attrs\":{\"paragraph\":{\"style\":\"Standard\"}},\"name\":\"insertParagraph\"},"
				+ "{\"listDefinition\":{\"listLevel8\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":6350,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6350,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel9\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":6985,\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6985,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel6\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":5080,\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5080,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel7\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":5715,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5715,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel5\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":4445,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":4445,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel4\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":3810,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3810,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel3\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":3175,\"levelText\":\"•\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3175,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel2\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":2540,\"levelText\":\"▪\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":2540,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel1\":{\"styleId\":\"Bullet_20_Symbols\",\"tabStopPosition\":1905,\"levelText\":\"◦\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1905,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel0\":{\"verticalRel\":\"line\",\"verticalPos\":\"middle\",\"tabStopPosition\":1270,\"height\":339,\"labelFollowedBy\":\"listtab\",\"indentLeft\":1270,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"width\":339,\"levelPicBulletUri\":\"Pictures/1000020000000010000000104191C4BD.gif\",\"indentFirstLine\":-635}},\"name\":\"insertListStyle\",\"listStyleId\":\"L3\"},"
				+ "{\"start\":[9],\"attrs\":{\"paragraph\":{\"listXmlId\":\"list4610980543945444999\",\"style\":\"Standard\",\"listId\":\"list4610980543945444999\",\"listStyleId\":\"L3\",\"listStart\":true,\"listLevel\":0},\"character\":{\"underline\":false}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"Star-First-followed-by-space\",\"start\":[9,0],\"name\":\"insertText\"},"
				+ "{\"start\":[10],\"attrs\":{\"paragraph\":{\"listXmlId\":\"list4610980543945444999\",\"style\":\"Standard\",\"listId\":\"list4610980543945444999\",\"listStyleId\":\"L3\",\"listStart\":true,\"listLevel\":0},\"character\":{\"underline\":false}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"Star-Second\",\"start\":[10,0],\"name\":\"insertText\"},"
				+ "{\"start\":[11],\"attrs\":{\"paragraph\":{\"listXmlId\":\"list4610980543945444999\",\"style\":\"Standard\",\"listId\":\"list4610980543945444999\",\"listStyleId\":\"L3\",\"listStart\":true,\"listLevel\":0},\"character\":{\"underline\":false}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"Star-Third\",\"start\":[11,0],\"name\":\"insertText\"},"
				+ "{\"start\":[12],\"attrs\":{\"paragraph\":{\"style\":\"Standard\"},\"character\":{\"underline\":false}},\"name\":\"insertParagraph\"},"
				+ "{\"listDefinition\":{\"listLevel8\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":6350,\"levelText\":\"(%1.%2.%3.%4.%5.%6.%7)\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel9\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":6985,\"levelText\":\"(%1.%2.%3.%4.%5.%6.%7)\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6985,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel6\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":5080,\"levelText\":\"(%1.%2.%3.%4.%5.%6.%7)\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5080,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel7\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":5715,\"levelText\":\"(%1.%2.%3.%4.%5.%6.%7)\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5715,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel5\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":4445,\"levelText\":\"(%1.%2.%3.%4.%5.%6)\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":4445,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel4\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":3810,\"levelText\":\"(%1.%2.%3.%4.%5)\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3810,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel3\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":3175,\"levelText\":\"(%1.%2.%3.%4)\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3175,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel2\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":2540,\"levelText\":\"(%1.%2.%3)\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":2540,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel1\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":1905,\"levelText\":\"(%1.%2)\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1905,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel0\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":1270,\"levelText\":\"(%1)\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1270,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635}},\"name\":\"insertListStyle\",\"listStyleId\":\"L4\"},"
				+ "{\"start\":[13],\"attrs\":{\"paragraph\":{\"listXmlId\":\"list4053051152566041198\",\"style\":\"Standard\",\"listId\":\"list4053051152566041198\",\"listStyleId\":\"L4\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"LabelTest1\",\"start\":[13,0],\"name\":\"insertText\"},"
				+ "{\"start\":[13,0],\"attrs\":{\"character\":{\"underline\":false}},\"name\":\"setAttributes\",\"end\":[13,9]},"
				+ "{\"start\":[14],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"L4\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"LabelTest2\",\"start\":[14,0],\"name\":\"insertText\"},"
				+ "{\"start\":[14,0],\"attrs\":{\"character\":{\"underline\":false}},\"name\":\"setAttributes\",\"end\":[14,9]},"
				+ "{\"start\":[15],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"L4\",\"listLevel\":2}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"LabelTest3\",\"start\":[15,0],\"name\":\"insertText\"},"
				+ "{\"start\":[15,0],\"attrs\":{\"character\":{\"underline\":false}},\"name\":\"setAttributes\",\"end\":[15,9]},"
				+ "{\"start\":[16],\"attrs\":{\"paragraph\":{\"style\":\"Standard\"},\"character\":{\"underline\":false}},\"name\":\"insertParagraph\"}"
				+ "]";
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), firstEditOperations);
	}

	@Test
	public void loadListStyleResolutionDocFullTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"attrs\":{\"document\":{\"defaultTabStop\":1251}},\"name\":\"setDocumentAttributes\"},"
				+ "{\"styleId\":\"default_paragraph_style\",\"default\":true,\"hidden\":true,\"attrs\":{\"character\":{\"fontSizeAsian\":12,\"fontName\":\"Times New Roman\",\"fontNameComplex\":\"Mangal\",\"language\":\"de-DE\",\"fontSize\":12,\"fontNameAsian\":\"SimSun\"}},\"name\":\"insertStyleSheet\",\"type\":\"paragraph\"},"
				+ "{\"styleId\":\"Standard\",\"attrs\":{},\"name\":\"insertStyleSheet\",\"parent\":\"default_paragraph_style\",\"type\":\"paragraph\"},"
				+ "{\"styleName\":\"Text body\",\"styleId\":\"Text_20_body\",\"attrs\":{\"paragraph\":{\"fillColor\":{\"value\":\"ff0000\",\"type\":\"rgb\"},\"marginTop\":0,\"marginBottom\":212}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},"
				+ "{\"styleId\":\"Caption\",\"attrs\":{\"paragraph\":{\"marginBottom\":212,\"marginTop\":212},\"character\":{\"fontSizeAsian\":12,\"italicComplex\":true,\"fontNameComplex\":\"Mangal1\",\"italicAsian\":true,\"italic\":true,\"fontSize\":12}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},"
				+ "{\"styleId\":\"Index\",\"attrs\":{\"character\":{\"fontSizeAsian\":12,\"fontNameComplex\":\"Mangal1\"}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},"
				+ "{\"styleId\":\"Heading\",\"attrs\":{\"paragraph\":{\"nextStyleId\":\"Text_20_body\",\"marginTop\":423,\"marginBottom\":212},\"character\":{\"fontSizeAsian\":14,\"fontNameComplex\":\"Mangal\",\"fontName\":\"Arial\",\"fontSize\":14,\"fontNameAsian\":\"Microsoft YaHei\"}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},"
				+ "{\"styleName\":\"Heading 2\",\"styleId\":\"Heading_20_2\",\"attrs\":{\"paragraph\":{\"nextStyleId\":\"Text_20_body\",\"outlineLevel\":1},\"character\":{\"italicAsian\":true,\"bold\":true,\"boldComplex\":true,\"fontSizeAsian\":14,\"italicComplex\":true,\"italic\":true,\"fontSize\":14,\"boldAsian\":true}},\"name\":\"insertStyleSheet\",\"parent\":\"Heading\",\"type\":\"paragraph\"},"
				+ "{\"styleName\":\"Heading 1\",\"styleId\":\"Heading_20_1\",\"attrs\":{\"paragraph\":{\"nextStyleId\":\"Text_20_body\",\"fillColor\":{\"value\":\"ffff00\",\"type\":\"rgb\"},\"outlineLevel\":0},\"character\":{\"boldAsian\":true,\"bold\":true,\"boldComplex\":true}},\"name\":\"insertStyleSheet\",\"parent\":\"Heading\",\"type\":\"paragraph\"},"
				+ "{\"styleId\":\"List\",\"attrs\":{\"character\":{\"fontSizeAsian\":12,\"fontNameComplex\":\"Mangal1\"}},\"name\":\"insertStyleSheet\",\"parent\":\"Text_20_body\",\"type\":\"paragraph\"},"
				+ "{\"styleId\":\"Signature\",\"attrs\":{\"paragraph\":{\"fillColor\":{\"value\":\"3deb3d\",\"type\":\"rgb\"}}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"type\":\"paragraph\"},"
				+ "{\"styleName\":\"Bullet Symbols\",\"styleId\":\"Bullet_20_Symbols\",\"attrs\":{\"character\":{\"fontNameAsian\":\"OpenSymbol\",\"fontNameComplex\":\"OpenSymbol\",\"fontName\":\"OpenSymbol\"}},\"name\":\"insertStyleSheet\",\"parent\":\"default_character_style\",\"type\":\"character\"},"
				+ "{\"styleName\":\"Numbering Symbols\",\"styleId\":\"Numbering_20_Symbols\",\"attrs\":{},\"name\":\"insertStyleSheet\",\"parent\":\"default_character_style\",\"type\":\"character\"},"
				+ "{\"attrs\":{\"page\":{\"printOrientation\":\"portrait\",\"marginBottom\":2000,\"indentRight\":2000,\"height\":29700,\"indentLeft\":2000,\"marginTop\":2000,\"marginLeft\":2000,\"numberFormat\":\"1\",\"width\":21001,\"marginRight\":2000}},\"name\":\"setDocumentAttributes\"},"
				+ "{\"fontName\":\"Mangal1\",\"attrs\":{\"family\":\"Mangal\"},\"name\":\"insertFontDescription\"},"
				+ "{\"fontName\":\"OpenSymbol\",\"attrs\":{\"family\":\"OpenSymbol\"},\"name\":\"insertFontDescription\"},"
				+ "{\"fontName\":\"Times New Roman\",\"attrs\":{\"family\":\"'Times New Roman'\",\"familyGeneric\":\"roman\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},"
				+ "{\"fontName\":\"Arial\",\"attrs\":{\"family\":\"Arial\",\"familyGeneric\":\"swiss\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},"
				+ "{\"fontName\":\"Mangal\",\"attrs\":{\"family\":\"Mangal\",\"familyGeneric\":\"system\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},"
				+ "{\"fontName\":\"Microsoft YaHei\",\"attrs\":{\"family\":\"'Microsoft YaHei'\",\"familyGeneric\":\"system\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},"
				+ "{\"fontName\":\"SimSun\",\"attrs\":{\"family\":\"SimSun\",\"familyGeneric\":\"system\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},"
				+ "{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"Standard\"}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"This is a test for resolving the list style being used for a list label. \",\"start\":[0,0],\"name\":\"insertText\"},"
				+ "{\"listDefinition\":{\"listLevel8\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":6350,\"levelText\":\"%9.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6350,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel9\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":6985,\"levelText\":\"%10.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":6985,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel6\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":5080,\"levelText\":\"%7.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5080,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel7\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":5715,\"levelText\":\"%8.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":5715,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel5\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":4445,\"levelText\":\"%6.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":4445,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel4\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":3810,\"levelText\":\"%5.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3810,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel3\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":3175,\"levelText\":\"%4.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":3175,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel2\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":2540,\"levelText\":\"%3.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":2540,\"numberFormat\":\"decimal\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel1\":{\"verticalRel\":\"baseline\",\"verticalPos\":\"bottom\",\"tabStopPosition\":1905,\"height\":212,\"labelFollowedBy\":\"listtab\",\"indentLeft\":1905,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"width\":212,\"levelPicBulletUri\":\"Pictures/100002000000000A0000000A1F4FBE6C.gif\",\"indentFirstLine\":-635},\"listLevel0\":{\"verticalRel\":\"baseline\",\"verticalPos\":\"bottom\",\"tabStopPosition\":1270,\"height\":212,\"labelFollowedBy\":\"listtab\",\"indentLeft\":1270,\"numberFormat\":\"bullet\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"width\":212,\"levelPicBulletUri\":\"Pictures/100002000000000A0000000A1F4FBE6C.gif\",\"indentFirstLine\":-635}},\"name\":\"insertListStyle\",\"listStyleId\":\"ListStyle-ImageLabel\"},"
				+ "{\"start\":[1],\"attrs\":{\"paragraph\":{\"listXmlId\":\"list6357203421921001764\",\"listId\":\"list6357203421921001764\",\"listStyleId\":\"ListStyle-ImageLabel\",\"listItemXmlId\":\"listItem1\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"The first level list uses a upper case roman numbering.\",\"start\":[1,0],\"name\":\"insertText\"},"
				+ "{\"start\":[2],\"attrs\":{\"paragraph\":{\"listXmlId\":\"list6357203421921001764\",\"listId\":\"list6357203421921001764\",\"listStyleId\":\"ListStyle-ImageLabel\",\"listItemXmlId\":\"listItem2\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"The first level list-item overrides with a upper case alphabetic numbering\",\"start\":[2,0],\"name\":\"insertText\"},"
				+ "{\"start\":[3],\"attrs\":{\"paragraph\":{\"listXmlId\":\"list6357203421921001764\",\"listId\":\"list6357203421921001764\",\"listStyleId\":\"ListStyle-ImageLabel\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"The second level list has a lower case roman numbering\",\"start\":[3,0],\"name\":\"insertText\"},"
				+ "{\"start\":[4],\"attrs\":{\"paragraph\":{\"listXmlId\":\"list6357203421921001764\",\"listId\":\"list6357203421921001764\",\"listStyleId\":\"ListStyle-ImageLabel\",\"listItemXmlId\":\"listItem4\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"The second level list-item overrides with a lower-case alphabetic numbering\",\"start\":[4,0],\"name\":\"insertText\"},"
				+ "{\"start\":[5],\"attrs\":{\"paragraph\":{\"listXmlId\":\"list6357203421921001764\",\"listId\":\"list6357203421921001764\",\"listStyleId\":\"ListStyle-ImageLabel\",\"listItemXmlId\":\"listItem5\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"Finally the paragraph within the list item is referencing to a list label using an image\",\"start\":[5,0],\"name\":\"insertText\"},"
				+ "{\"listDefinition\":{\"listLevel1\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":1905,\"levelText\":\"%2.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1905,\"numberFormat\":\"upperLetter\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel0\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":1270,\"levelText\":\"%1.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1270,\"numberFormat\":\"upperLetter\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635}},\"name\":\"insertListStyle\",\"listStyleId\":\"ListStyle-LetterUpperCaseLabel\"},"
				+ "{\"start\":[6],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"ListStyle-LetterUpperCaseLabel\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:3 Roman, Alpha\",\"start\":[6,0],\"name\":\"insertText\"},"
				+ "{\"start\":[6,21],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: A.\",\"start\":[6,22],\"name\":\"insertText\"},"
				+ "{\"start\":[6,9],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[6,13]},"
				+ "{\"start\":[6,16],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[6,20]},"
				+ "{\"listDefinition\":{\"listLevel1\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":1905,\"levelText\":\"%2.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1905,\"numberFormat\":\"upperRoman\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel0\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":1270,\"levelText\":\"%1.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1270,\"numberFormat\":\"upperRoman\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635}},\"name\":\"insertListStyle\",\"listStyleId\":\"ListStyle-RomanUpperCaseLabel\"},"
				+ "{\"start\":[7],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"ListStyle-RomanUpperCaseLabel\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:2 Roman, Alpha\",\"start\":[7,0],\"name\":\"insertText\"},"
				+ "{\"start\":[7,21],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: I.\",\"start\":[7,22],\"name\":\"insertText\"},"
				+ "{\"start\":[7,9],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[7,13]},"
				+ "{\"start\":[8],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"ListStyle-LetterUpperCaseLabel\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:1 Roman, Alpha\",\"start\":[8,0],\"name\":\"insertText\"},"
				+ "{\"start\":[8,21],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: A.\",\"start\":[8,22],\"name\":\"insertText\"},"
				+ "{\"start\":[8,16],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[8,20]},"
				+ "{\"start\":[9],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"ListStyle-ImageLabel\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:0 Roman, Alpha, but list style referenced by paragraph style\",\"start\":[9,0],\"name\":\"insertText\"},"
				+ "{\"start\":[9,67],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: image label with red bullet\",\"start\":[9,68],\"name\":\"insertText\"},"
				+ "{\"start\":[10],\"attrs\":{\"paragraph\":{\"listStyleId\":\"OX_DEFAULT_LIST\",\"listStart\":true,\"listLevel\":0}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:0 Roman, Alpha no paragraph style\",\"start\":[10,0],\"name\":\"insertText\"},"
				+ "{\"start\":[10,40],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: Implementation defined - AO/LO: bullet, MSO: no list, Calligra: number\",\"start\":[10,41],\"name\":\"insertText\"},"
				+ "{\"listDefinition\":{\"listLevel1\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":1905,\"levelText\":\"%2.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1905,\"numberFormat\":\"lowerLetter\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel0\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":1270,\"levelText\":\"%1.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1270,\"numberFormat\":\"lowerLetter\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635}},\"name\":\"insertListStyle\",\"listStyleId\":\"ListStyle-LetterLowerCaseLabel\"},"
				+ "{\"start\":[11],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"ListStyle-LetterLowerCaseLabel\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:15 Roman, Alpha, roman, alpha\",\"start\":[11,0],\"name\":\"insertText\"},"
				+ "{\"start\":[11,36],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: a.\",\"start\":[11,37],\"name\":\"insertText\"},"
				+ "{\"start\":[11,10],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[11,14]},"
				+ "{\"start\":[11,17],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[11,21]},"
				+ "{\"start\":[11,24],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[11,28]},"
				+ "{\"start\":[11,31],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[11,35]},"
				+ "{\"listDefinition\":{\"listLevel1\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":1905,\"levelText\":\"%2.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1905,\"numberFormat\":\"lowerRoman\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635},\"listLevel0\":{\"styleId\":\"Numbering_20_Symbols\",\"tabStopPosition\":1270,\"levelText\":\"%1.\",\"labelFollowedBy\":\"listtab\",\"indentLeft\":1270,\"numberFormat\":\"lowerRoman\",\"listLevelPositionAndSpaceMode\":\"label-alignment\",\"indentFirstLine\":-635}},\"name\":\"insertListStyle\",\"listStyleId\":\"ListStyle-RomanLowerCaseLabel\"},"
				+ "{\"start\":[12],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"ListStyle-RomanLowerCaseLabel\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:14 Roman, Alpha, roman, alpha\",\"start\":[12,0],\"name\":\"insertText\"},"
				+ "{\"start\":[12,36],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: i.\",\"start\":[12,37],\"name\":\"insertText\"},"
				+ "{\"start\":[12,10],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[12,14]},"
				+ "{\"start\":[12,17],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[12,21]},"
				+ "{\"start\":[12,24],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[12,28]},"
				+ "{\"start\":[13],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"ListStyle-LetterLowerCaseLabel\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:13 Roman, Alpha, roman, alpha\",\"start\":[13,0],\"name\":\"insertText\"},"
				+ "{\"start\":[13,36],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: a.\",\"start\":[13,37],\"name\":\"insertText\"},"
				+ "{\"start\":[13,10],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[13,14]},"
				+ "{\"start\":[13,17],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[13,21]},"
				+ "{\"start\":[13,31],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[13,35]},"
				+ "{\"start\":[14],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"ListStyle-LetterUpperCaseLabel\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:12 Roman, Alpha, roman, alpha\",\"start\":[14,0],\"name\":\"insertText\"},"
				+ "{\"start\":[14,36],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: A.\",\"start\":[14,37],\"name\":\"insertText\"},"
				+ "{\"start\":[14,10],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[14,14]},"
				+ "{\"start\":[14,17],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[14,21]},"
				+ "{\"start\":[15],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"ListStyle-LetterLowerCaseLabel\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:11 Roman, Alpha, roman, alpha\",\"start\":[15,0],\"name\":\"insertText\"},"
				+ "{\"start\":[15,36],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: a.\",\"start\":[15,37],\"name\":\"insertText\"},"
				+ "{\"start\":[15,10],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[15,14]},"
				+ "{\"start\":[15,24],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[15,28]},"
				+ "{\"start\":[15,31],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[15,35]},"
				+ "{\"start\":[16],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"ListStyle-RomanLowerCaseLabel\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:10 Roman, Alpha, roman, alpha\",\"start\":[16,0],\"name\":\"insertText\"},"
				+ "{\"start\":[16,36],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: i.\",\"start\":[16,37],\"name\":\"insertText\"},"
				+ "{\"start\":[16,10],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[16,14]},"
				+ "{\"start\":[16,24],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[16,28]},"
				+ "{\"start\":[17],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"ListStyle-LetterLowerCaseLabel\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:09 Roman, Alpha, roman, alpha\",\"start\":[17,0],\"name\":\"insertText\"},"
				+ "{\"start\":[17,36],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: a.\",\"start\":[17,37],\"name\":\"insertText\"},"
				+ "{\"start\":[17,10],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[17,14]},"
				+ "{\"start\":[17,31],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[17,35]},"
				+ "{\"start\":[18],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"ListStyle-RomanUpperCaseLabel\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:08 Roman, Alpha, roman, alpha\",\"start\":[18,0],\"name\":\"insertText\"},"
				+ "{\"start\":[18,36],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: I.\",\"start\":[18,37],\"name\":\"insertText\"},"
				+ "{\"start\":[18,10],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[18,14]},"
				+ "{\"start\":[19],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"ListStyle-LetterLowerCaseLabel\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:07 Roman, Alpha, roman, alpha\",\"start\":[19,0],\"name\":\"insertText\"},"
				+ "{\"start\":[19,36],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: a.\",\"start\":[19,37],\"name\":\"insertText\"},"
				+ "{\"start\":[19,17],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[19,21]},"
				+ "{\"start\":[19,24],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[19,28]},"
				+ "{\"start\":[19,31],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[19,35]},"
				+ "{\"start\":[20],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"ListStyle-RomanLowerCaseLabel\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:06 Roman, Alpha, roman, alpha\",\"start\":[20,0],\"name\":\"insertText\"},"
				+ "{\"start\":[20,36],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: i.\",\"start\":[20,37],\"name\":\"insertText\"},"
				+ "{\"start\":[20,17],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[20,21]},"
				+ "{\"start\":[20,24],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[20,28]},"
				+ "{\"start\":[21],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"ListStyle-LetterLowerCaseLabel\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:05 Roman, Alpha, roman, alpha\",\"start\":[21,0],\"name\":\"insertText\"},"
				+ "{\"start\":[21,36],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: a.\",\"start\":[21,37],\"name\":\"insertText\"},"
				+ "{\"start\":[21,17],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[21,21]},"
				+ "{\"start\":[21,31],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[21,35]},"
				+ "{\"start\":[22],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"ListStyle-LetterUpperCaseLabel\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:04 Roman, Alpha, roman, alpha\",\"start\":[22,0],\"name\":\"insertText\"},"
				+ "{\"start\":[22,36],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: A.\",\"start\":[22,37],\"name\":\"insertText\"},"
				+ "{\"start\":[22,17],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[22,21]},"
				+ "{\"start\":[23],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"ListStyle-LetterLowerCaseLabel\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:03 Roman, Alpha, roman, alpha\",\"start\":[23,0],\"name\":\"insertText\"},"
				+ "{\"start\":[23,36],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: a.\",\"start\":[23,37],\"name\":\"insertText\"},"
				+ "{\"start\":[23,24],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[23,28]},"
				+ "{\"start\":[23,31],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[23,35]},"
				+ "{\"start\":[24],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"ListStyle-RomanLowerCaseLabel\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:02 Roman, Alpha, roman, alpha\",\"start\":[24,0],\"name\":\"insertText\"},"
				+ "{\"start\":[24,36],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: i.\",\"start\":[24,37],\"name\":\"insertText\"},"
				+ "{\"start\":[24,24],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[24,28]},"
				+ "{\"start\":[25],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"ListStyle-LetterLowerCaseLabel\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:01 Roman, Alpha, roman, alpha\",\"start\":[25,0],\"name\":\"insertText\"},"
				+ "{\"start\":[25,36],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: a.\",\"start\":[25,37],\"name\":\"insertText\"},"
				+ "{\"start\":[25,31],\"attrs\":{\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"3deb3d\",\"type\":\"rgb\"},\"boldAsian\":true}},\"name\":\"setAttributes\",\"end\":[25,35]},"
				+ "{\"start\":[26],\"attrs\":{\"paragraph\":{\"style\":\"Standard\",\"listStyleId\":\"ListStyle-ImageLabel\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:00 Roman, Alpha, roman, alpha, but image list style referenced by paragraph style\",\"start\":[26,0],\"name\":\"insertText\"},"
				+ "{\"start\":[26,88],\"name\":\"insertHardBreak\"},"
				+ "{\"text\":\"Expected: image label with red bullet\",\"start\":[26,89],\"name\":\"insertText\"},"
				+ "{\"start\":[27],\"attrs\":{\"paragraph\":{\"listStyleId\":\"OX_DEFAULT_LIST\",\"listLevel\":1}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"binary:00 Roman, Alpha, roman, alpha\",\"start\":[27,0],\"name\":\"insertText\"},"
				+ "{\"start\":[28],\"attrs\":{\"paragraph\":{\"listLabelHidden\":true,\"listStyleId\":\"OX_DEFAULT_LIST\",\"listLevel\":1}},\"name\":\"insertParagraph\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	public void newListDefinitionTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty_as_can_be";
		String firstEditOperations = "["
				+ "{\"start\":[0],\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"1\",\"start\":[0,0],\"name\":\"insertText\"},"
				+ "{\"start\":[1],\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"2\",\"start\":[1,0],\"name\":\"insertText\"},"
				+ "{\"listDefinition\":"
				+ "{\"listLevel8\":{\"listStartValue\":1,\"levelText\":\"%9.\",\"indentLeft\":11430,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",\"listLevel6\":{\"listStartValue\":1,\"levelText\":\"%7.\",\"indentLeft\":8890,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",\"listLevel7\":{\"listStartValue\":1,\"levelText\":\"%8.\",\"indentLeft\":10160,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",\"listLevel5\":{\"listStartValue\":1,\"levelText\":\"%6.\",\"indentLeft\":7620,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",\"listLevel4\":{\"listStartValue\":1,\"levelText\":\"myprefixN4%0%1%2%3.mysuffixN4\",\"indentLeft\":6350,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",\"listLevel3\":{\"listStartValue\":1,\"levelText\":\"%4.\",\"indentLeft\":5080,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",\"listLevel2\":{\"listStartValue\":1,\"levelText\":\"%3.\",\"indentLeft\":3810,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",\"listLevel1\":{\"listStartValue\":1,\"levelText\":\"%2.\",\"indentLeft\":2540,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",\"listLevel0\":{\"listStartValue\":1,\"levelText\":\"myprefixN0%0%1%2%3.mysuffixN0\",\"indentLeft\":1270,\"numberFormat\":\"lowerLetter\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"listStyleId\":\"L10000\"},"
				+ "{\"listDefinition\":"
				+ "{\"listLevel8\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"●suffixB8\",\"indentLeft\":11430,\"numberFormat\":\"bullet\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",\"listLevel6\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"●\",\"indentLeft\":8890,\"numberFormat\":\"bullet\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",\"listLevel7\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"●\",\"indentLeft\":10160,\"numberFormat\":\"bullet\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",\"listLevel5\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"●\",\"indentLeft\":7620,\"numberFormat\":\"bullet\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",\"listLevel4\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"●\",\"indentLeft\":6350,\"numberFormat\":\"bullet\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",\"listLevel3\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"●\",\"indentLeft\":5080,\"numberFormat\":\"bullet\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",\"listLevel2\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"●\",\"indentLeft\":3810,\"numberFormat\":\"bullet\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",\"listLevel1\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"●\",\"indentLeft\":2540,\"numberFormat\":\"bullet\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}"
				+ ",\"listLevel0\":{\"listStartValue\":1,\"fontName\":\"Times New Roman\",\"levelText\":\"●suffixB1\",\"indentLeft\":1270,\"numberFormat\":\"bullet\",\"indentFirstLine\":-635,\"textAlign\":\"left\"}},\"name\":\"insertListStyle\",\"listStyleId\":\"L20000\"},"
				+ "{\"styleId\":\"ListParagraph\",\"styleName\":\"List Paragraph\",\"attrs\":{\"paragraph\":{\"indentLeft\":1270,\"nextStyleId\":\"ListParagraph\",\"contextualSpacing\":true}},\"name\":\"insertStyleSheet\",\"parent\":\"Standard\",\"uiPriority\":34,\"type\":\"paragraph\"},"
				+ "{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listXmlId\":\"someID1a\",\"listStyleId\":\"L1x000\""
				+ ",\"listLevel\":0}},\"name\":\"setAttributes\"},"
				+ "{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listXmlId\":\"someID1b\",\"listStyleId\":\"L10000\""
				+ ",\"listLevel\":1}},\"name\":\"setAttributes\"},"
				+ "{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listXmlId\":\"someID2a\",\"listStyleId\":\"L2x000\""
				+ ",\"listLevel\":2}},\"name\":\"setAttributes\"},"
				+ "{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"ListParagraph\",\"listXmlId\":\"someID2b\",\"listStyleId\":\"L20000\""
				+ ",\"listLevel\":1}},\"name\":\"setAttributes\"},"
				+ "]";
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), firstEditOperations);
	}

	@Test
	/**
	 * When a listLevel is changed on a paragraph the ancestor elements
	 * <text:list> and <text:list-item> have to be inserted/deleted.
	 */
	public void listLevelTest() {
		final String SOURCE_FILE_NAME_TRUNC = "ListHeading";
		String firstEditOperations = "["
				+ "{\"start\":[1],\"attrs\":{\"paragraph\":{\"listLevel\":0}},\"name\":\"setAttributes\"},"
				+ "]";
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), firstEditOperations);
	}

	@Test
	/**
	 * When a listLevel is changed on a paragraph the ancestor elements
	 * <text:list> and <text:list-item> have to be inserted/deleted.
	 */
	public void listCreationTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty_as_can_be";
		String firstEditOperations = "["
				+ "{\"start\":[0],\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"1\",\"start\":[0,0],\"name\":\"insertText\"},"
				+ "{\"start\":[0],\"attrs\":{\"paragraph\":{\"listLevel\":3}},\"name\":\"setAttributes\"},"
				+ "]";
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), firstEditOperations);
	}

	@Test
	/**
	 * Text will be inserted to a paragraph component/element, which has
	 * children, but none component child
	 */
	public void insertToBoilerplateElementsOnly() {
		final String SOURCE_FILE_NAME_TRUNC = "boilerplate";
		final String SOURCE_FILE_NAME_SUFFIX = ".odt";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\"Something!\",\"start\":[0,0],\"name\":\"insertText\"}"
				+ "]";
		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, SOURCE_FILE_NAME_SUFFIX, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Split should delete as well boilerplate from the first part!
	 */
	public void splitBeforeBoilerplateContent() {
		final String SOURCE_FILE_NAME_TRUNC = "footnote";
		final String SOURCE_FILE_NAME_SUFFIX = ".odt";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"start\":[1,22],\"name\":\"splitParagraph\"}"
				+ "]";
		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, SOURCE_FILE_NAME_SUFFIX, getTestMethodName(), editOperations);
	}

	@Test
	public void deleteListByDeletingAllParagraphsFromWithinTest() {
		final String SOURCE_FILE_NAME_TRUNC = "ListStyleResolution";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"start\":[28],\"name\":\"delete\"},"
				+ "{\"start\":[27],\"name\":\"delete\"},"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Verified: https://bugs.open-xchange.com/show_bug.cgi?id=26440
	 */
	public void TabsInHyperlinkTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\"Empty Document\",\"start\":[0,0],\"name\":\"insertText\"},{\"start\":[0,7],\"name\":\"insertTab\"},{\"start\":[0,8],\"name\":\"insertTab\"},{\"styleName\":\"Hyperlink\",\"styleId\":\"Hyperlink\",\"attrs\":{\"character\":{\"color\":{\"value\":\"hyperlink\",\"fallbackValue\":\"0080C0\",\"type\":\"scheme\"},\"underline\":true}},\"name\":\"insertStyleSheet\",\"parent\":null,\"uiPriority\":99,\"type\":\"character\"},{\"start\":[0,6],\"attrs\":{\"character\":{\"style\":\"Hyperlink\",\"url\":\"http://www.heise.de\"}},\"name\":\"setAttributes\",\"end\":[0,9]}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Verified: https://bugs.open-xchange.com/show_bug.cgi?id=26448
	 */
	public void textBeforeEmptyBookmarkTest() {
		final String SOURCE_FILE_NAME_TRUNC = "field";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\"hello\",\"start\":[0,0],\"name\":\"insertText\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Verified: https://bugs.open-xchange.com/show_bug.cgi?id=26447
	 */
	public void whitespaceAroundFieldTest() {
		final String SOURCE_FILE_NAME_TRUNC = "field";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"start\":[0,0],\"name\":\"insertTab\"},"
				+ "{\"start\":[0,3],\"attrs\":{},\"name\":\"insertTab\"},"
				+ "{\"text\":\"   \",\"start\":[0,4],\"name\":\"insertText\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Verified - https://bugs.open-xchange.com/show_bug.cgi?id=26391
	 */
	public void moveComponentTest() {
		final String SOURCE_FILE_NAME_TRUNC = "image-attributes";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"to\":[2,0],\"start\":[1,0],\"name\":\"move\",\"end\":[1,0]}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.importOnlyRegressionTest(SOURCE_FILE_NAME_TRUNC, ".odt", getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Verified - https://bugs.open-xchange.com/show_bug.cgi?id=26294
	 */
	public void insertNewImageTest() {
		final String SOURCE_FILE_NAME_TRUNC = "image-attributes";
		final String UID = "d03f7d7218eb";
		final String INTERNAL_IMAGE_PATH = "Pictures/uid" + UID + ".jpg";
		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"start\":[0,14],\"attrs\":{\"drawing\":{\"marginBottom\":317,\"height\":5982,\"marginLeft\":317,\"marginTop\":317,\"imageUrl\":\"" + INTERNAL_IMAGE_PATH + "\",\"marginRight\":317,\"width\":17013}},\"name\":\"insertDrawing\",\"type\":\"image\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		Map<Long, byte[]> resourceMap = new HashMap<Long, byte[]>();
		byte[] imageBytes = null;
		try {
			imageBytes = ResourceUtilities.loadFileAsBytes(ResourceUtilities.getReferenceFile("testA.jpg"));
		} catch (IOException ex) {
			Logger.getLogger(LatestOngoingTest.class.getName()).log(Level.SEVERE, null, ex);
			Assert.fail(ex.getMessage());
		}
		long uid = Long.parseLong(UID, 16);
		resourceMap.put(uid, imageBytes);
		String savedDocumentPath = super.roundtripRegressionWithResourcesTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations, resourceMap, true);
		OdfPackage pkg = null;
		try {
			// return the absolute path to the test directory
			savedDocumentPath = ResourceUtilities.getTestOutputFolder() + savedDocumentPath;
			pkg = OdfPackage.loadPackage(savedDocumentPath);
		} catch (Exception ex) {
			Assert.fail("The saved document '" + savedDocumentPath + "' could not be found!");
			Logger.getLogger(LatestOngoingTest.class.getName()).log(Level.SEVERE, null, ex);
		}
		if (pkg == null || !pkg.contains(INTERNAL_IMAGE_PATH)) {
			Assert.fail("The image '" + INTERNAL_IMAGE_PATH + "' could not be found in the saved document '" + savedDocumentPath + "'!");
		}
	}

	@Test
	/**
	 * Verified: https://www.pivotaltracker.com/story/show/35754805
	 */
	public void exportImageAttributesTest() {
		final String SOURCE_FILE_NAME_TRUNC = "odt-images-linked";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"start\":[2,0],\"attrs\":{\"drawing\":{\"anchorHorOffset\":0,\"textWrapMode\":\"square\",\"anchorHorBase\":\"column\",\"inline\":false,\"anchorHorAlign\":\"left\",\"textWrapSide\":\"right\"}},\"name\":\"setAttributes\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Verified: https://bugs.open-xchange.com/show_bug.cgi?id=26055
	 */
	public void changeListAlignmentTest() {
		final String SOURCE_FILE_NAME_TRUNC = "images";


		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"start\":[10,16],\"attrs\":{\"drawing\":{\"anchorHorBase\":\"column\",\"textWrapMode\":\"square\",\"anchorHorOffset\":0,\"inline\":false,\"anchorHorAlign\":\"left\",\"textWrapSide\":\"right\"}},\"name\":\"setAttributes\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	public void deleteUndoTest() {
		final String SOURCE_FILE_NAME_TRUNC = "odf-fields";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"start\":[1,17],\"name\":\"delete\",\"end\":[1,17]},"
				+ "{\"start\":[1,17],\"attrs\":{\"character\":{\"vertAlign\":null,\"strike\":null,\"bold\":null,\"fontName\":null,\"style\":null,\"fillColor\":null,\"color\":null,\"underline\":null,\"language\":null,\"italic\":null,\"fontSize\":null,\"url\":null}},\"representation\":\"14:58:45\",\"name\":\"insertField\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	public void imageChangeHorizontalAlignTest() {
		final String SOURCE_FILE_NAME_TRUNC = "images";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				// Moving Charlie Brown from left alinged to right aligned
				+ "{\"start\":[8,0],\"attrs\":{\"drawing\":{\"anchorHorBase\":\"column\",\"textWrapMode\":\"square\",\"anchorHorOffset\":0,\"inline\":false,\"anchorHorAlign\":\"right\",\"textWrapSide\":\"left\"}},\"name\":\"setAttributes\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	public void columnWidthTest() {
		final String SOURCE_FILE_NAME_TRUNC = "feature_attributes_tables";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"insertMode\":\"behind\",\"start\":[4],\"name\":\"insertColumn\",\"tableGrid\":[20,20,20,20,20],\"gridPosition\":2},"
				+ "{\"start\":[4],\"attrs\":{\"table\":{\"tableGrid\":[20,20,20,20,20]}},\"name\":\"setAttributes\"},"
				+ "{\"start\":[4,0,3,0],\"name\":\"insertParagraph\"},{\"text\":\"b\",\"start\":[4,0,3,0,0],\"name\":\"insertText\"},"
				+ "{\"start\":[4,1,3,0],\"name\":\"insertParagraph\"},{\"text\":\"c\",\"start\":[4,1,3,0,0],\"name\":\"insertText\"},"
				+ "{\"start\":[4,2,3,0],\"name\":\"insertParagraph\"},{\"text\":\"d\",\"start\":[4,2,3,0,0],\"name\":\"insertText\"},"
				+ "{\"start\":[4,3,3,0],\"name\":\"insertParagraph\"},{\"text\":\"e\",\"start\":[4,3,3,0,0],\"name\":\"insertText\"},"
				+ "{\"text\":\"b\",\"start\":[4,0,3,0,0],\"name\":\"insertText\"},"
				+ "{\"insertMode\":\"behind\",\"start\":[4],\"name\":\"insertColumn\",\"tableGrid\":[17,17,17,17,17,17],\"gridPosition\":1},"
				+ "{\"start\":[4],\"attrs\":{\"table\":{\"tableGrid\":[17,17,17,17,17,17]}},\"name\":\"setAttributes\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	public void hyperlinkInsertionTest() {

		final String SOURCE_FILE_NAME_TRUNC = "hyperlink_destination";
		String firstEditOperations = "["
				+ "{\"name\":\"setAttributes\",\"start\":[0,2],\"end\":[0,2],\"attrs\":{\"character\":{\"color\":{\"value\":\"C00000\",\"type\":\"rgb\"},\"fillColor\":{\"value\":\"FFF000\",\"type\":\"rgb\"},\"url\":\"http://www.heise.de/\"}}},"
				+ "]";
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), firstEditOperations);
	}

	@Test
	public void insertSingleCharacterTest() {
		final String SOURCE_FILE_NAME_TRUNC = "feature_attributes_character_MSO15";
		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\" \",\"start\":[3,5],\"name\":\"insertText\"},"
				+ "{\"text\":\"a\",\"start\":[3,6],\"name\":\"insertText\"},"
				+ "{\"text\":\"b\",\"start\":[3,7],\"name\":\"insertText\"},"
				+ "{\"text\":\"c\",\"start\":[3,8],\"name\":\"insertText\"},"
				+ "{\"text\":\"d\",\"start\":[3,9],\"name\":\"insertText\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	@Ignore // as the file name is randomized so the reference will always fail
	public void imageBase64Test() {
		final String SOURCE_FILE_NAME_TRUNC = "image";

		List<String> editOperations = new ArrayList();
		String inputOpsFromFile = null;
		String referenceOpsTextFilePath = OUTPUT_DIRECTORY + "image-base64.ops";
		File referenceReloadedOpsFile = ResourceUtilities.getReferenceFile(referenceOpsTextFilePath);
		if (referenceReloadedOpsFile.exists()) {
			inputOpsFromFile = ResourceUtilities.loadFileAsString(referenceReloadedOpsFile);
		}
		editOperations.add(inputOpsFromFile);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Verified: https://bugs.open-xchange.com/show_bug.cgi?id=25250
	 */
	public void insertColumnsWithStyleWithoutXmlId() {
		final String SOURCE_FILE_NAME_TRUNC = "feature_attributes_tables_FunnyTable_With_xmlid";
		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"insertMode\":\"behind\",\"start\":[2],\"name\":\"insertColumn\",\"tableGrid\":[20,20,20,20,20],\"gridPosition\":0},{\"start\":[2],\"attrs\":{\"table\":{\"tableGrid\":[20,20,20,20,20]}},\"name\":\"setAttributes\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Extended: https://bugs.open-xchange.com/show_bug.cgi?id=25250
	 */
	public void insertColumnWithRepeatedCells() {
		final String SOURCE_FILE_NAME_TRUNC = "feature_attributes_tables";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"insertMode\":\"behind\",\"start\":[21],\"name\":\"insertColumn\",\"tableGrid\":[20,20,20,20,20],\"gridPosition\":0},"
				+ "{\"start\":[21],\"attrs\":{\"table\":{\"tableGrid\":[20,20,20,20,20]}},\"name\":\"setAttributes\"}"
				+ "]";

		editOperations.add(firstEditOperations);

		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	@Ignore // generated automatic styles differs from test to test, therefore the reference is not usable and test always fails..
	public void insertTableTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty_as_can_be";
		final String SOURCE_FILE_NAME_SUFFIX = ".odt";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"styleId\":\"TableGrid\",\"styleName\":\"Table Grid\",\"attrs\":{\"wholeTable\":{\"paragraph\":{\"lineHeight\":{\"value\":100,\"type\":\"percent\"}},\"table\":{\"borderInsideVert\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17},\"borderTop\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17},\"borderInsideHor\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17},\"borderBottom\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17},\"borderLeft\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17},\"borderRight\":{\"style\":\"single\",\"color\":{\"type\":\"auto\"},\"width\":17}}},\"next\":null},\"name\":\"insertStyleSheet\",\"uiPriority\":59,\"type\":\"table\"},{\"start\":[0],\"attrs\":{\"table\":{\"style\":\"TableGrid\",\"tableGrid\":[1000,1000,1000,1000],\"width\":\"auto\",\"exclude\":[\"lastRow\",\"lastCol\",\"bandsVert\"]}},\"name\":\"insertTable\"},{\"count\":2,\"start\":[0,0],\"name\":\"insertRows\",\"insertDefaultCells\":true}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, SOURCE_FILE_NAME_SUFFIX, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Verified: https://bugs.open-xchange.com/show_bug.cgi?id=25248
	 */
	public void insertRowAtSecondPosition() {
		final String SOURCE_FILE_NAME_TRUNC = "feature_attributes_tables-backgroundTableOnly";

		String firstEditOperations = "["
				+ "{\"count\":1,\"start\":[4,1],\"name\":\"insertRows\",\"insertDefaultCells\":false,\"referenceRow\":0}"
				+ "]";

		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), firstEditOperations);
	}

	@Test
	public void spanStyleInheritanceTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty_as_can_be";
		final String SOURCE_FILE_NAME_SUFFIX = ".odt";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"name\":\"insertStyleSheet\",\"styleId\":\"BLUE\",\"type\":\"character\",\"attrs\":{\"character\":{\"fillColor\":{\"value\":\"0000FF\",\"type\":\"rgb\"}}},\"parent\":\"default_character_style\"},{\"name\":\"insertStyleSheet\",\"styleId\":\"ITALIC\",\"type\":\"character\",\"attrs\":{\"character\":{\"italic\":true,\"italicAsian\":true}},\"parent\":\"default_character_style\"},{\"name\":\"insertStyleSheet\",\"styleId\":\"BOLD\",\"type\":\"character\",\"attrs\":{\"character\":{\"boldAsian\":true,\"bold\":true}},\"parent\":\"default_character_style\"},{\"name\":\"insertStyleSheet\",\"styleId\":\"BACKGROUND_YELLOW\",\"type\":\"character\",\"attrs\":{\"character\":{\"fillColor\":{\"value\":\"FFFF00\",\"type\":\"rgb\"}}},\"parent\":\"default_character_style\"},{\"name\":\"insertFontDescription\",\"fontName\":\"Courier\",\"attrs\":{\"family\":\"Courier\",\"familyGeneric\":\"modern\",\"pitch\":\"fixed\"}},{\"name\":\"insertParagraph\",\"start\":[0],\"attrs\":{\"paragraph\":{}}},{\"text\":\"TEST ALL INNER AND OUTER SPAN COMBINATIONS\",\"start\":[0,0],\"name\":\"insertText\"},{\"name\":\"insertParagraph\",\"start\":[1],\"attrs\":{\"paragraph\":{}}},{\"text\":\"There are the binar flags: OuterHasTemplate OuterHasAuto InnerHasTemplate InnerHasAuto\",\"start\":[1,0],\"name\":\"insertText\"},{\"name\":\"insertParagraph\",\"start\":[2],\"attrs\":{\"paragraph\":{}}},{\"text\":\"00: \\\"outer:parent italic\\\" \\\"outer:auto super\\\" \\\"inner:parent backg.\\\" \\\"inner:auto red\\\"\",\"start\":[2,0],\"name\":\"insertText\"},{\"name\":\"setAttributes\",\"start\":[2,4],\"end\":[2,24],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[2,26],\"end\":[2,43],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[2,45],\"end\":[2,65],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[2,67],\"end\":[2,82],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"insertParagraph\",\"start\":[3],\"attrs\":{\"paragraph\":{}}},{\"text\":\" normal \",\"start\":[3,0],\"name\":\"insertText\"},{\"name\":\"insertHardBreak\",\"start\":[3,8]},{\"name\":\"insertParagraph\",\"start\":[4],\"attrs\":{\"paragraph\":{}}},{\"text\":\"01: \\\"outer:parent italic\\\" \\\"outer:auto super\\\" \\\"inner:parent backg.\\\" \\\"inner:auto red\\\"\",\"start\":[4,0],\"name\":\"insertText\"},{\"name\":\"setAttributes\",\"start\":[4,4],\"end\":[4,24],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[4,26],\"end\":[4,43],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[4,45],\"end\":[4,65],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[4,67],\"end\":[4,82],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"insertParagraph\",\"start\":[5],\"attrs\":{\"paragraph\":{}}},{\"text\":\" red \",\"start\":[5,0],\"name\":\"insertText\"},{\"name\":\"insertHardBreak\",\"start\":[5,5]},{\"name\":\"setAttributes\",\"start\":[5,0],\"end\":[5,4],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"insertParagraph\",\"start\":[6],\"attrs\":{\"paragraph\":{}}},{\"text\":\"02: \\\"outer:parent italic\\\" \\\"outer:auto super\\\" \\\"inner:parent backg.\\\" \\\"inner:auto red\\\"\",\"start\":[6,0],\"name\":\"insertText\"},{\"name\":\"setAttributes\",\"start\":[6,4],\"end\":[6,24],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[6,26],\"end\":[6,43],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[6,45],\"end\":[6,65],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[6,67],\"end\":[6,82],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"insertParagraph\",\"start\":[7],\"attrs\":{\"paragraph\":{}}},{\"text\":\" yellow background \",\"start\":[7,0],\"name\":\"insertText\"},{\"name\":\"insertHardBreak\",\"start\":[7,19]},{\"name\":\"setAttributes\",\"start\":[7,0],\"end\":[7,18],\"attrs\":{\"character\":{\"style\":\"BACKGROUND_YELLOW\"}}},{\"name\":\"insertParagraph\",\"start\":[8],\"attrs\":{\"paragraph\":{}}},{\"text\":\"03: \\\"outer:parent italic\\\" \\\"outer:auto super\\\" \\\"inner:parent backg.\\\" \\\"inner:auto red\\\"\",\"start\":[8,0],\"name\":\"insertText\"},{\"name\":\"setAttributes\",\"start\":[8,4],\"end\":[8,24],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[8,26],\"end\":[8,43],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[8,45],\"end\":[8,65],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[8,67],\"end\":[8,82],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"insertParagraph\",\"start\":[9],\"attrs\":{\"paragraph\":{}}},{\"text\":\" yellow background and red color \",\"start\":[9,0],\"name\":\"insertText\"},{\"name\":\"insertHardBreak\",\"start\":[9,33]},{\"name\":\"setAttributes\",\"start\":[9,0],\"end\":[9,32],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"},\"style\":\"BACKGROUND_YELLOW\"}}},{\"name\":\"insertParagraph\",\"start\":[10],\"attrs\":{\"paragraph\":{}}},{\"text\":\"04: \\\"outer:parent italic\\\" \\\"outer:auto super\\\" \\\"inner:parent backg.\\\" \\\"inner:auto red\\\"\",\"start\":[10,0],\"name\":\"insertText\"},{\"name\":\"setAttributes\",\"start\":[10,4],\"end\":[10,24],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[10,26],\"end\":[10,43],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[10,45],\"end\":[10,65],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[10,67],\"end\":[10,82],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"insertParagraph\",\"start\":[11],\"attrs\":{\"paragraph\":{}}},{\"text\":\" super \",\"start\":[11,0],\"name\":\"insertText\"},{\"name\":\"insertHardBreak\",\"start\":[11,7]},{\"name\":\"setAttributes\",\"start\":[11,0],\"end\":[11,6],\"attrs\":{\"character\":{\"vertAlign\":\"super\"}}},{\"name\":\"insertParagraph\",\"start\":[12],\"attrs\":{\"paragraph\":{}}},{\"text\":\"05: \\\"outer:parent italic\\\" \\\"outer:auto super\\\" \\\"inner:parent backg.\\\" \\\"inner:auto red\\\"\",\"start\":[12,0],\"name\":\"insertText\"},{\"name\":\"setAttributes\",\"start\":[12,4],\"end\":[12,24],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[12,26],\"end\":[12,43],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[12,45],\"end\":[12,65],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[12,67],\"end\":[12,82],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"insertParagraph\",\"start\":[13],\"attrs\":{\"paragraph\":{}}},{\"text\":\" super and red \",\"start\":[13,0],\"name\":\"insertText\"},{\"name\":\"insertHardBreak\",\"start\":[13,15]},{\"name\":\"setAttributes\",\"start\":[13,0],\"end\":[13,14],\"attrs\":{\"character\":{\"vertAlign\":\"super\",\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"insertParagraph\",\"start\":[14],\"attrs\":{\"paragraph\":{}}},{\"text\":\"06: \\\"outer:parent italic\\\" \\\"outer:auto super\\\" \\\"inner:parent backg.\\\" \\\"inner:auto red\\\"\",\"start\":[14,0],\"name\":\"insertText\"},{\"name\":\"setAttributes\",\"start\":[14,4],\"end\":[14,24],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[14,26],\"end\":[14,43],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[14,45],\"end\":[14,65],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[14,67],\"end\":[14,82],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"insertParagraph\",\"start\":[15],\"attrs\":{\"paragraph\":{}}},{\"text\":\" super and yellow background \",\"start\":[15,0],\"name\":\"insertText\"},{\"name\":\"insertHardBreak\",\"start\":[15,29]},{\"name\":\"setAttributes\",\"start\":[15,0],\"end\":[15,28],\"attrs\":{\"character\":{\"vertAlign\":\"super\",\"style\":\"BACKGROUND_YELLOW\"}}},{\"name\":\"insertParagraph\",\"start\":[16],\"attrs\":{\"paragraph\":{}}},{\"text\":\"07: \\\"outer:parent italic\\\" \\\"outer:auto super\\\" \\\"inner:parent backg.\\\" \\\"inner:auto red\\\"\",\"start\":[16,0],\"name\":\"insertText\"},{\"name\":\"setAttributes\",\"start\":[16,4],\"end\":[16,24],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[16,26],\"end\":[16,43],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[16,45],\"end\":[16,65],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[16,67],\"end\":[16,82],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"insertParagraph\",\"start\":[17],\"attrs\":{\"paragraph\":{}}},{\"text\":\" super, yellow background and red \",\"start\":[17,0],\"name\":\"insertText\"},{\"name\":\"insertHardBreak\",\"start\":[17,34]},{\"name\":\"setAttributes\",\"start\":[17,0],\"end\":[17,33],\"attrs\":{\"character\":{\"vertAlign\":\"super\",\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"},\"style\":\"BACKGROUND_YELLOW\"}}},{\"name\":\"insertParagraph\",\"start\":[18],\"attrs\":{\"paragraph\":{}}},{\"text\":\"08: \\\"outer:parent italic\\\" \\\"outer:auto super\\\" \\\"inner:parent backg.\\\" \\\"inner:auto red\\\"\",\"start\":[18,0],\"name\":\"insertText\"},{\"name\":\"setAttributes\",\"start\":[18,4],\"end\":[18,24],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[18,26],\"end\":[18,43],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[18,45],\"end\":[18,65],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[18,67],\"end\":[18,82],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"insertParagraph\",\"start\":[19],\"attrs\":{\"paragraph\":{}}},{\"text\":\" italic \",\"start\":[19,0],\"name\":\"insertText\"},{\"name\":\"insertHardBreak\",\"start\":[19,8]},{\"name\":\"setAttributes\",\"start\":[19,0],\"end\":[19,7],\"attrs\":{\"character\":{\"style\":\"ITALIC\"}}},{\"name\":\"insertParagraph\",\"start\":[20],\"attrs\":{\"paragraph\":{}}},{\"text\":\"09: \\\"outer:parent italic\\\" \\\"outer:auto super\\\" \\\"inner:parent backg.\\\" \\\"inner:auto red\\\"\",\"start\":[20,0],\"name\":\"insertText\"},{\"name\":\"setAttributes\",\"start\":[20,4],\"end\":[20,24],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[20,26],\"end\":[20,43],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[20,45],\"end\":[20,65],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[20,67],\"end\":[20,82],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"insertParagraph\",\"start\":[21],\"attrs\":{\"paragraph\":{}}},{\"text\":\" italic and red \",\"start\":[21,0],\"name\":\"insertText\"},{\"name\":\"insertHardBreak\",\"start\":[21,16]},{\"name\":\"setAttributes\",\"start\":[21,0],\"end\":[21,15],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"},\"style\":\"ITALIC\"}}},{\"name\":\"insertParagraph\",\"start\":[22],\"attrs\":{\"paragraph\":{}}},{\"text\":\"10: \\\"outer:parent italic\\\" \\\"outer:auto super\\\" \\\"inner:parent backg.\\\" \\\"inner:auto red\\\"\",\"start\":[22,0],\"name\":\"insertText\"},{\"name\":\"setAttributes\",\"start\":[22,4],\"end\":[22,24],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[22,26],\"end\":[22,43],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[22,45],\"end\":[22,65],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[22,67],\"end\":[22,82],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"insertParagraph\",\"start\":[23],\"attrs\":{\"paragraph\":{}}},{\"text\":\" italic and yellow background \",\"start\":[23,0],\"name\":\"insertText\"},{\"name\":\"insertHardBreak\",\"start\":[23,30]},{\"name\":\"setAttributes\",\"start\":[23,0],\"end\":[23,29],\"attrs\":{\"character\":{\"style\":\"BACKGROUND_YELLOW\"}}},{\"name\":\"insertParagraph\",\"start\":[24],\"attrs\":{\"paragraph\":{}}},{\"text\":\"11: \\\"outer:parent italic\\\" \\\"outer:auto super\\\" \\\"inner:parent backg.\\\" \\\"inner:auto red\\\"\",\"start\":[24,0],\"name\":\"insertText\"},{\"name\":\"setAttributes\",\"start\":[24,4],\"end\":[24,24],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[24,26],\"end\":[24,43],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[24,45],\"end\":[24,65],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[24,67],\"end\":[24,82],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"insertParagraph\",\"start\":[25],\"attrs\":{\"paragraph\":{}}},{\"text\":\" italic, yellow background and red \",\"start\":[25,0],\"name\":\"insertText\"},{\"name\":\"insertHardBreak\",\"start\":[25,35]},{\"name\":\"setAttributes\",\"start\":[25,0],\"end\":[25,34],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"},\"style\":\"BACKGROUND_YELLOW\"}}},{\"name\":\"insertParagraph\",\"start\":[26],\"attrs\":{\"paragraph\":{}}},{\"text\":\"12: \\\"outer:parent italic\\\" \\\"outer:auto super\\\" \\\"inner:parent backg.\\\" \\\"inner:auto red\\\"\",\"start\":[26,0],\"name\":\"insertText\"},{\"name\":\"setAttributes\",\"start\":[26,4],\"end\":[26,24],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[26,26],\"end\":[26,43],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[26,45],\"end\":[26,65],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[26,67],\"end\":[26,82],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"insertParagraph\",\"start\":[27],\"attrs\":{\"paragraph\":{}}},{\"text\":\" italic and super \",\"start\":[27,0],\"name\":\"insertText\"},{\"name\":\"insertHardBreak\",\"start\":[27,18]},{\"name\":\"setAttributes\",\"start\":[27,0],\"end\":[27,17],\"attrs\":{\"character\":{\"vertAlign\":\"super\",\"style\":\"ITALIC\"}}},{\"name\":\"insertParagraph\",\"start\":[28],\"attrs\":{\"paragraph\":{}}},{\"text\":\"13: \\\"outer:parent italic\\\" \\\"outer:auto super\\\" \\\"inner:parent backg.\\\" \\\"inner:auto red\\\"\",\"start\":[28,0],\"name\":\"insertText\"},{\"name\":\"setAttributes\",\"start\":[28,4],\"end\":[28,24],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[28,26],\"end\":[28,43],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[28,45],\"end\":[28,65],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[28,67],\"end\":[28,82],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"insertParagraph\",\"start\":[29],\"attrs\":{\"paragraph\":{}}},{\"text\":\" italic, super and red \",\"start\":[29,0],\"name\":\"insertText\"},{\"name\":\"insertHardBreak\",\"start\":[29,23]},{\"name\":\"setAttributes\",\"start\":[29,0],\"end\":[29,22],\"attrs\":{\"character\":{\"vertAlign\":\"super\",\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"insertParagraph\",\"start\":[30],\"attrs\":{\"paragraph\":{}}},{\"text\":\"14: \\\"outer:parent italic\\\" \\\"outer:auto super\\\" \\\"inner:parent backg.\\\" \\\"inner:auto red\\\"\",\"start\":[30,0],\"name\":\"insertText\"},{\"name\":\"setAttributes\",\"start\":[30,4],\"end\":[30,24],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[30,26],\"end\":[30,43],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[30,45],\"end\":[30,65],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[30,67],\"end\":[30,82],\"attrs\":{\"character\":{\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"}}}},{\"name\":\"insertParagraph\",\"start\":[31],\"attrs\":{\"paragraph\":{}}},{\"text\":\" italic, super and yellow background \",\"start\":[31,0],\"name\":\"insertText\"},{\"name\":\"insertHardBreak\",\"start\":[31,37]},{\"name\":\"setAttributes\",\"start\":[31,0],\"end\":[31,36],\"attrs\":{\"character\":{\"vertAlign\":\"super\",\"style\":\"BACKGROUND_YELLOW\"}}},{\"name\":\"insertParagraph\",\"start\":[32],\"attrs\":{\"paragraph\":{}}},{\"text\":\"15: \\\"outer:parent italic\\\" \\\"outer:auto super\\\" \\\"inner:parent backg.\\\" \\\"inner:auto red\\\"\",\"start\":[32,0],\"name\":\"insertText\"},{\"name\":\"setAttributes\",\"start\":[32,4],\"end\":[32,24],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[32,26],\"end\":[32,43],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[32,45],\"end\":[32,65],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"setAttributes\",\"start\":[32,67],\"end\":[32,82],\"attrs\":{\"character\":{\"color\":{\"value\":\"00FF00\",\"type\":\"rgb\"}}}},{\"name\":\"insertParagraph\",\"start\":[33],\"attrs\":{\"paragraph\":{}}},{\"text\":\" italic, super, yellow background and red \",\"start\":[33,0],\"name\":\"insertText\"},{\"name\":\"insertHardBreak\",\"start\":[33,42]},{\"name\":\"setAttributes\",\"start\":[33,0],\"end\":[33,41],\"attrs\":{\"character\":{\"vertAlign\":\"super\",\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"},\"style\":\"BACKGROUND_YELLOW\"}}},{\"name\":\"insertParagraph\",\"start\":[34],\"attrs\":{\"paragraph\":{}}},{\"text\":\"Three styles: Courier, sub, super (outer to inner):superscript\",\"start\":[34,0],\"name\":\"insertText\"},{\"name\":\"setAttributes\",\"start\":[34,51],\"end\":[34,61],\"attrs\":{\"character\":{\"vertAlign\":\"super\"}}}"
				+ "]";
		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, SOURCE_FILE_NAME_SUFFIX, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Split position marks the first component of the second half, therefore
	 * Character is Moving the unknown content DANGER: These tests are unable to
	 * detect unknown content, like if it is in the first or second half!
	 */
	public void markOverUnknownContentPlusSpaceElement() {

		final String SOURCE_FILE_NAME_TRUNC = "DUMMY";
		final String SOURCE_FILE_NAME_SUFFIX = ".odt";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "]";
		// {"start":[1,22],"name":"splitParagraph"},{"text":"Svante","start":[2,0],"name":"insertText"},{"start":[0,6],"attrs":{"character":{"color":{"value":"C00000","type":"rgb"}}},"name":"setAttributes","end":[0,22]},{"start":[1,0],"attrs":{"character":{"color":{"value":"C00000","type":"rgb"}}},"name":"setAttributes","end":[1,21]},{"start":[2,0],"attrs":{"character":{"color":{"value":"C00000","type":"rgb"}}},"name":"setAttributes","end":[2,4]},{"start":[2,0],"name":"delete","end":[2,3]},{"start":[1],"name":"delete"},{"start":[0,8],"name":"delete","end":[0,22]},{"start":[0],"name":"mergeParagraph"}
		String secondEditOperations = "["
				+ "{\"start\":[1,22],\"name\":\"splitParagraph\"},"
				+ "{\"text\":\"svante\",\"start\":[2,0],\"name\":\"insertText\"},"
				+ "{\"start\":[0,6],\"attrs\":{\"character\":{\"color\":{\"value\":\"C00000\",\"type\":\"rgb\"}}},\"name\":\"setAttributes\",\"end\":[0,22]},"
				+ "{\"start\":[1,0],\"attrs\":{\"character\":{\"color\":{\"value\":\"C00000\",\"type\":\"rgb\"}}},\"name\":\"setAttributes\",\"end\":[1,21]},"
				+ "{\"start\":[2,0],\"attrs\":{\"character\":{\"color\":{\"value\":\"C00000\",\"type\":\"rgb\"}}},\"name\":\"setAttributes\",\"end\":[2,4]}"
				+ ","
				+ "{\"start\":[0,6],\"name\":\"delete\",\"end\":[0,21]}"
				+ "]";
		editOperations.add(firstEditOperations);
		editOperations.add(secondEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, SOURCE_FILE_NAME_SUFFIX, getTestMethodName(), editOperations);
	}

	@Test
	public void mergeOverUnknownContent() {

		final String SOURCE_FILE_NAME_TRUNC = "footnote";
		final String SOURCE_FILE_NAME_SUFFIX = ".odt";

		String firstEditOperations = "["
				+ " {\"start\":[1,22],\"name\":\"splitParagraph\"},"
				+ "{\"text\":\"Svante\",\"start\":[2,0],\"name\":\"insertText\"},"
				+ "{\"start\":[0,6],\"attrs\":{\"character\":{\"color\":{\"value\":\"C00000\",\"type\":\"rgb\"}}},\"name\":\"setAttributes\",\"end\":[0,22]},"
				+ "{\"start\":[1,0],\"attrs\":{\"character\":{\"color\":{\"value\":\"C00000\",\"type\":\"rgb\"}}},\"name\":\"setAttributes\",\"end\":[1,21]},"
				+ "{\"start\":[2,0],\"attrs\":{\"character\":{\"color\":{\"value\":\"C00000\",\"type\":\"rgb\"}}},\"name\":\"setAttributes\",\"end\":[2,4]}"
				+ ","
				+ "{\"start\":[2,0],\"name\":\"delete\",\"end\":[2,3]},"
				+ "{\"start\":[1],\"name\":\"delete\"},"
				+ "{\"start\":[0,8],\"name\":\"delete\",\"end\":[0,22]}"
				+ ","
				+ "{\"start\":[0],\"name\":\"mergeParagraph\"}"
				+ "]";
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, SOURCE_FILE_NAME_SUFFIX, getTestMethodName(), firstEditOperations);
	}

	@Test
	public void markOverUnknownContent() {

		final String SOURCE_FILE_NAME_TRUNC = "footnote";
		final String SOURCE_FILE_NAME_SUFFIX = ".odt";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "]";

		String secondEditOperations = "["
				+ "{\"start\":[1,22],\"name\":\"splitParagraph\"},"
				+ "{\"text\":\"Svante\",\"start\":[2,0],\"name\":\"insertText\"},"
				+ "{\"start\":[0,6],\"attrs\":{\"character\":{\"color\":{\"value\":\"C00000\",\"type\":\"rgb\"}}},\"name\":\"setAttributes\",\"end\":[0,22]},"
				+ "{\"start\":[1,0],\"attrs\":{\"character\":{\"color\":{\"value\":\"C00000\",\"type\":\"rgb\"}}},\"name\":\"setAttributes\",\"end\":[1,21]},"
				+ "{\"start\":[2,0],\"attrs\":{\"character\":{\"color\":{\"value\":\"C00000\",\"type\":\"rgb\"}}},\"name\":\"setAttributes\",\"end\":[2,4]}"
				+ "]";
		editOperations.add(firstEditOperations);
		editOperations.add(secondEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, SOURCE_FILE_NAME_SUFFIX, getTestMethodName(), editOperations);
	}

	@Test
	public void deleteOverUnknownContent() {
		final String SOURCE_FILE_NAME_TRUNC = "footnote";
		final String SOURCE_FILE_NAME_SUFFIX = ".odt";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "]";

		String secondEditOperations = "["
				+ "{\"start\":[1,22],\"name\":\"splitParagraph\"}"
				+ ","
				+ "{\"text\":\"Something!\",\"start\":[2,0],\"name\":\"insertText\"}"
				+ "]";
		editOperations.add(firstEditOperations);
		editOperations.add(secondEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, SOURCE_FILE_NAME_SUFFIX, getTestMethodName(), editOperations);
	}

	@Test
	public void hyperlinkTest() {
		final String SOURCE_FILE_NAME_TRUNC = "hyperlink";
		String firstEditOperations = "["
				+ "{\"styleId\":\"Hyperlink\",\"styleName\":\"Hyperlink\",\"attrs\":{\"character\":{\"color\":{\"value\":\"hyperlink\",\"fallbackValue\":\"0080C0\",\"type\":\"scheme\"},\"underline\":true}},\"name\":\"insertStyleSheet\",\"parent\":null,\"uiPriority\":99,\"type\":\"character\"},{\"start\":[0,4],\"attrs\":{\"character\":{\"style\":\"Hyperlink\",\"url\":\"http://www.alternate.de\"}},\"name\":\"setAttributes\",\"end\":[0,4]}"
				+ "]";
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), firstEditOperations);
	}

	@Test
	public void multipleSplitMergeTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty_as_can_be";
		final String SOURCE_FILE_NAME_SUFFIX = ".odt";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"heading1\"}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"first\",\"start\":[0,0],\"name\":\"insertText\"},"
				+ "{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"heading1\"}},\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"second\",\"start\":[1,0],\"name\":\"insertText\"}"
				+ "]";

		String secondEditOperations = "["
				+ "{\"start\":[0],\"name\":\"mergeParagraph\"},"
				+ "{\"start\":[0,5],\"name\":\"splitParagraph\"},"
				+ "{\"start\":[0],\"name\":\"mergeParagraph\"},"
				+ "{\"start\":[0,5],\"name\":\"splitParagraph\"},"
				+ "{\"start\":[0],\"name\":\"mergeParagraph\"},"
				+ "{\"start\":[0,5],\"name\":\"splitParagraph\"},"
				+ "{\"start\":[0],\"name\":\"mergeParagraph\"},"
				+ "{\"start\":[0,5],\"name\":\"splitParagraph\"},"
				+ "{\"start\":[0],\"name\":\"mergeParagraph\"},"
				+ "{\"start\":[0,5],\"name\":\"splitParagraph\"},"
				+ "{\"start\":[0],\"name\":\"mergeParagraph\"},"
				+ "{\"start\":[0,5],\"name\":\"splitParagraph\"},"
				+ "{\"start\":[0],\"name\":\"mergeParagraph\"},"
				+ "{\"start\":[0,5],\"name\":\"splitParagraph\"},"
				+ "{\"start\":[0],\"name\":\"mergeParagraph\"},"
				+ "{\"start\":[0,5],\"name\":\"splitParagraph\"},"
				+ "{\"start\":[0],\"name\":\"mergeParagraph\"},"
				+ "{\"start\":[0,5],\"name\":\"splitParagraph\"},"
				+ "{\"start\":[0],\"name\":\"mergeParagraph\"},"
				+ "{\"start\":[0,5],\"name\":\"splitParagraph\"},"
				+ "{\"start\":[0],\"name\":\"mergeParagraph\"},"
				+ "{\"start\":[0,5],\"name\":\"splitParagraph\"}"
				+ "]";
		editOperations.add(firstEditOperations);
		editOperations.add(secondEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, SOURCE_FILE_NAME_SUFFIX, getTestMethodName(), editOperations);
	}

	@Test
	public void MergeSplitOverUnknownContent() {
		final String SOURCE_FILE_NAME_TRUNC = "footnote";
		final String SOURCE_FILE_NAME_SUFFIX = ".odt";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "]";

		String secondEditOperations = "["
				+ "{\"start\":[0,0],\"name\":\"splitParagraph\"},"
				+ "{\"start\":[0],\"name\":\"mergeParagraph\"},"
				+ "{\"start\":[1,22],\"name\":\"splitParagraph\"},"
				+ "{\"text\":\"This text was lost!!\",\"start\":[2,0],\"name\":\"insertText\"}"
				+ "]";
		editOperations.add(firstEditOperations);
		editOperations.add(secondEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, SOURCE_FILE_NAME_SUFFIX, getTestMethodName(), editOperations);
	}

	@Test
	public void splitParagraphAtBeginTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty_as_can_be";
		final String SOURCE_FILE_NAME_SUFFIX = ".odt";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"name\":\"insertParagraph\",\"start\":[0],\"attrs\":{\"paragraph\":{}}},"
				+ "{\"text\":\"heading\",\"start\":[0,0],\"name\":\"insertText\"},"
				+ "{\"styleId\":\"heading1\",\"styleName\":\"heading 1\",\"attrs\":{\"paragraph\":{\"outlineLevel\":0},\"next\":\"default_paragraph_style\",\"character\":{\"bold\":true,\"color\":{\"value\":\"accent1\",\"fallbackValue\":\"376092\",\"type\":\"scheme\",\"transformations\":[{\"value\":74902,\"type\":\"shade\"}]},\"fontSize\":14}},\"name\":\"insertStyleSheet\",\"parent\":\"default_paragraph_style\",\"uiPriority\":9,\"type\":\"paragraph\"},"
				+ "{\"start\":[0],\"attrs\":{\"paragraph\":{\"alignment\":null,\"lineHeight\":null,\"fillColor\":null,\"marginTop\":null,\"outlineLevel\":null,\"listStyleId\":null,\"listLevel\":null,\"borderBottom\":null,\"contextualSpacing\":null,\"borderLeft\":null,\"showListLabel\":null,\"marginBottom\":null,\"tabStops\":null,\"listStartValue\":null,\"indentRight\":null,\"indentLeft\":null,\"style\":\"heading1\",\"borderTop\":null,\"borderInside\":null,\"indentFirstLine\":null,\"borderRight\":null},\"character\":{\"vertAlign\":null,\"fontName\":null,\"bold\":null,\"strike\":null,\"fillColor\":null,\"color\":null,\"underline\":null,\"italic\":null,\"language\":null,\"fontSize\":null,\"url\":null}},\"name\":\"setAttributes\"},"
				+ "{\"start\":[0,0],\"name\":\"splitParagraph\"}"
				+ "]";
		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, SOURCE_FILE_NAME_SUFFIX, getTestMethodName(), editOperations);
	}

	@Test
	public void insertWithSpace() {
		final String SOURCE_FILE_NAME_TRUNC = "feature_attributes_character_MSO15";
		final String SOURCE_FILE_NAME_SUFFIX = ".odt";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\" svante\",\"start\":[5,5],\"name\":\"insertText\"}"
				+ "]";
		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, SOURCE_FILE_NAME_SUFFIX, getTestMethodName(), editOperations);
	}

	@Test
	/**
	 * Verified: https://bugs.open-xchange.com/show_bug.cgi?id=25245
	 */
	public void editingUsingMultipleSpans() {
		final String SOURCE_FILE_NAME_TRUNC = "empty_as_can_be";
		final String SOURCE_FILE_NAME_SUFFIX = ".odt";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"name\":\"insertParagraph\",\"start\":[0],\"attrs\":{\"paragraph\":{}}},"
				+ "{\"text\":\"Courier\",\"start\":[0,0],\"name\":\"insertText\"},"
				+ "{\"start\":[0,0],\"attrs\":{\"character\":{\"fontName\":\"Courier\"}},\"name\":\"setAttributes\",\"end\":[0,6]},"
				+ "{\"text\":\" subscript\",\"start\":[0,7],\"name\":\"insertText\"},"
				+ "{\"start\":[0,8],\"attrs\":{\"character\":{\"vertAlign\":\"sub\"}},\"name\":\"setAttributes\",\"end\":[0,16]},"
				+ "{\"text\":\" \",\"start\":[0,17],\"name\":\"insertText\"},"
				+ "{\"text\":\"superscript\",\"start\":[0,18],\"attrs\":{\"character\":{\"vertAlign\":\"super\"}},\"name\":\"insertText\"}"
				+ "]";
		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, SOURCE_FILE_NAME_SUFFIX, getTestMethodName(), editOperations);
	}

	@Test
	public void insertTextTest() {
		final String SOURCE_FILE_NAME_TRUNC = "ListStyleResolution";
		final String SOURCE_FILE_NAME_SUFFIX = ".odt";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"text\":\"Svante\",\"start\":[11,7],\"name\":\"insertText\"},"
				+ "]";
		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, SOURCE_FILE_NAME_SUFFIX, getTestMethodName(), editOperations);
	}

	@Test
	// Verified
	public void splitSpaceElementTest() {
		final String SOURCE_FILE_NAME_TRUNC = "compdocfileformat_shortened";

		List<String> firstEditOperations = new ArrayList();
		String firstEditOperation = "["
				+ "{\"name\":\"insertParagraph\",\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"Standard\"}}},"
				+ "{\"text\":\"2  3 4 \",\"start\":[0,0],\"name\":\"insertText\"},"
				+ "{\"name\":\"setAttributes\",\"start\":[0,0],\"end\":[0,0],\"attrs\":{\"character\":{\"boldComplex\":true,\"boldAsian\":true,\"bold\":true,\"fillColor\":{\"value\":\"3deb3d\",\"type\":\"rgb\"}}}},"
				+ "{\"name\":\"setAttributes\",\"start\":[0,2],\"end\":[0,2],\"attrs\":{\"character\":{\"boldComplex\":true,\"boldAsian\":true,\"bold\":true,\"fillColor\":{\"value\":\"3deb3d\",\"type\":\"rgb\"}}}},"
				+ "]";

		firstEditOperations.add(firstEditOperation);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), firstEditOperations);
	}

	@Test
	// Verified: https://bugs.open-xchange.com/show_bug.cgi?id=25251 as below
	public void deletionSpannedColumnTest() {
		final String SOURCE_FILE_NAME_TRUNC = "feature_attributes_tables_SMALL";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"startGrid\":2,\"start\":[2],\"name\":\"deleteColumns\",\"endGrid\":2},"
				+ "{\"start\":[2],\"attrs\":{\"table\":{\"tableGrid\":[25,25,25]}},\"name\":\"setAttributes\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), firstEditOperations);
	}

	@Test
	// Verified: https://bugs.open-xchange.com/show_bug.cgi?id=25251 as below & above
	public void deletionColumnTest() {
		final String SOURCE_FILE_NAME_TRUNC = "feature_attributes_tables-backgroundTableOnly";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"startGrid\":0,\"start\":[4],\"name\":\"deleteColumns\",\"endGrid\":0},"
				+ "{\"start\":[4],\"attrs\":{\"table\":{\"tableGrid\":[25,25,25]}},\"name\":\"setAttributes\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), firstEditOperations);
	}

	@Test
	// Verified: https://bugs.open-xchange.com/show_bug.cgi?id=25251 as above
	public void columnDelete() {
		final String SOURCE_FILE_NAME_TRUNC = "empty_as_can_be";

		List<String> editOperations = new ArrayList();
		String firstEditOperations = "["
				+ "{\"styleId\":\"default_paragraph_style\",\"default\":true,\"hidden\":true,\"attrs\":{\"paragraph\":{\"alignment\":\"left\",\"lineHeight\":{\"value\":\"100\",\"type\":\"percent\"},\"paddingTop\":0,\"fillColor\":{\"value\":\"transparent\",\"type\":\"auto\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"paddingLeft\":0,\"paddingBottom\":0,\"paddingRight\":0},\"character\":{\"fontNameComplex\":\"Times New Roman\",\"italicAsian\":false,\"fontName\":\"Cambria\",\"boldComplex\":false,\"fillColor\":{\"value\":\"transparent\",\"type\":\"auto\"},\"letterSpacing\":\"normal\",\"fontSizeAsian\":12,\"italicComplex\":false,\"italic\":false,\"fontSize\":12,\"vertAlign\":\"baseline\",\"bold\":false,\"language\":\"en-US\",\"boldAsian\":false,\"fontNameAsian\":\"MS Mincho\"}},\"name\":\"insertStyleSheet\",\"type\":\"paragraph\"},{\"styleName\":\"Normal\",\"styleId\":\"Normal\",\"attrs\":{},\"name\":\"insertStyleSheet\",\"parent\":\"default_paragraph_style\",\"type\":\"paragraph\"},{\"styleName\":\"Subtitle\",\"styleId\":\"Subtitle\",\"attrs\":{\"character\":{\"italicAsian\":true,\"fontName\":\"Calibri\",\"fontNameComplex\":\"Times New Roman\",\"letterSpacing\":26,\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"italicComplex\":true,\"italic\":true,\"fontNameAsian\":\"MS Gothic\"}},\"next\":\"Normal\",\"name\":\"insertStyleSheet\",\"parent\":\"Normal\",\"type\":\"paragraph\"},{\"styleName\":\"Heading 1\",\"styleId\":\"Heading1\",\"attrs\":{\"paragraph\":{\"marginTop\":847,\"outlineLevel\":0},\"character\":{\"fontNameComplex\":\"Times New Roman\",\"fontName\":\"Calibri\",\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"345A8A\",\"type\":\"rgb\"},\"fontSizeAsian\":16,\"fontSize\":16,\"fontNameAsian\":\"MS Gothic\",\"boldAsian\":true}},\"next\":\"Normal\",\"name\":\"insertStyleSheet\",\"parent\":\"Normal\",\"type\":\"paragraph\"},{\"styleName\":\"Title\",\"styleId\":\"Title\",\"attrs\":{\"paragraph\":{\"marginBottom\":529,\"paddingTop\":0,\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"width\":35},\"borderLeft\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"paddingLeft\":0,\"paddingBottom\":141,\"paddingRight\":0},\"character\":{\"fontName\":\"Calibri\",\"fontNameComplex\":\"Times New Roman\",\"letterSpacing\":9,\"color\":{\"value\":\"17365D\",\"type\":\"rgb\"},\"fontSizeAsian\":26,\"fontSize\":26,\"fontNameAsian\":\"MS Gothic\"}},\"next\":\"Normal\",\"name\":\"insertStyleSheet\",\"parent\":\"Normal\",\"type\":\"paragraph\"},{\"styleName\":\"Heading 2\",\"styleId\":\"Heading2\",\"attrs\":{\"paragraph\":{\"marginTop\":353,\"outlineLevel\":1},\"character\":{\"fontNameComplex\":\"Times New Roman\",\"fontName\":\"Calibri\",\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"fontSizeAsian\":13,\"fontSize\":13,\"fontNameAsian\":\"MS Gothic\",\"boldAsian\":true}},\"next\":\"Normal\",\"name\":\"insertStyleSheet\",\"parent\":\"Normal\",\"type\":\"paragraph\"},{\"styleName\":\"Default Paragraph Font\",\"styleId\":\"DefaultParagraphFont\",\"attrs\":{},\"name\":\"insertStyleSheet\",\"parent\":\"default_character_style\",\"type\":\"character\"},{\"styleName\":\"Heading 2 Char\",\"styleId\":\"Heading2Char\",\"attrs\":{\"character\":{\"fontNameComplex\":\"Times New Roman\",\"fontName\":\"Calibri\",\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"fontSizeAsian\":13,\"fontSize\":13,\"fontNameAsian\":\"MS Gothic\",\"boldAsian\":true}},\"name\":\"insertStyleSheet\",\"parent\":\"DefaultParagraphFont\",\"type\":\"character\"},{\"styleName\":\"Title Char\",\"styleId\":\"TitleChar\",\"attrs\":{\"character\":{\"fontName\":\"Calibri\",\"fontNameComplex\":\"Times New Roman\",\"letterSpacing\":9,\"color\":{\"value\":\"17365D\",\"type\":\"rgb\"},\"fontSizeAsian\":26,\"fontSize\":26,\"fontNameAsian\":\"MS Gothic\"}},\"name\":\"insertStyleSheet\",\"parent\":\"DefaultParagraphFont\",\"type\":\"character\"},{\"styleName\":\"Subtitle Char\",\"styleId\":\"SubtitleChar\",\"attrs\":{\"character\":{\"italicAsian\":true,\"fontName\":\"Calibri\",\"fontNameComplex\":\"Times New Roman\",\"letterSpacing\":26,\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"italicComplex\":true,\"italic\":true,\"fontNameAsian\":\"MS Gothic\"}},\"name\":\"insertStyleSheet\",\"parent\":\"DefaultParagraphFont\",\"type\":\"character\"},{\"styleName\":\"Heading 1 Char\",\"styleId\":\"Heading1Char\",\"attrs\":{\"character\":{\"fontNameComplex\":\"Times New Roman\",\"fontName\":\"Calibri\",\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"345A8A\",\"type\":\"rgb\"},\"fontSizeAsian\":16,\"fontSize\":16,\"fontNameAsian\":\"MS Gothic\",\"boldAsian\":true}},\"name\":\"insertStyleSheet\",\"parent\":\"DefaultParagraphFont\",\"type\":\"character\"},{\"fontName\":\"Cambria\",\"attrs\":{\"family\":\"Cambria\",\"panose1\":[2,4,5,3,5,4,6,3,2,4],\"familyGeneric\":\"roman\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},{\"fontName\":\"MS Mincho\",\"attrs\":{\"family\":\"MS Mincho\",\"panose1\":[2,2,6,9,4,2,5,8,3,4],\"familyGeneric\":\"modern\",\"pitch\":\"fixed\"},\"name\":\"insertFontDescription\"},{\"fontName\":\"Times New Roman\",\"attrs\":{\"family\":\"Times New Roman\",\"panose1\":[2,2,6,3,5,4,5,2,3,4],\"familyGeneric\":\"roman\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},{\"fontName\":\"Calibri\",\"attrs\":{\"family\":\"Calibri\",\"panose1\":[2,15,5,2,2,2,4,3,2,4],\"familyGeneric\":\"swiss\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},{\"fontName\":\"MS Gothic\",\"attrs\":{\"family\":\"MS Gothic\",\"panose1\":[2,11,6,9,7,2,5,8,2,4],\"familyGeneric\":\"modern\",\"pitch\":\"fixed\"},\"name\":\"insertFontDescription\"},{\"fontName\":\"Adobe Caslon Pro\",\"attrs\":{\"family\":\"Adobe Caslon Pro\",\"panose1\":[0,0,0,0,0,0,0,0,0,0],\"familyGeneric\":\"roman\",\"pitch\":\"variable\"},\"name\":\"insertFontDescription\"},{\"attrs\":{\"page\":{\"printOrientation\":\"portrait\",\"marginBottom\":2000,\"indentRight\":2499,\"height\":29704,\"indentLeft\":2499,\"marginTop\":2499,\"marginLeft\":2499,\"numberFormat\":\"1\",\"width\":20990,\"marginRight\":2499}},\"name\":\"setDocumentAttributes\"},{\"start\":[0],\"attrs\":{\"paragraph\":{\"style\":\"Title\"}},\"name\":\"insertParagraph\"},{\"text\":\"Tables\",\"start\":[0,0],\"name\":\"insertText\"},{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"Subtitle\"}},\"name\":\"insertParagraph\"},{\"text\":\"Examples\",\"start\":[1,0],\"name\":\"insertText\"},{\"start\":[2],\"attrs\":{\"paragraph\":{\"style\":\"Heading2\",\"outlineLevel\":1}},\"name\":\"insertParagraph\"},{\"text\":\"Table with different background colors (hard coded)\",\"start\":[2,0],\"name\":\"insertText\"},{\"start\":[3],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"}},\"name\":\"insertParagraph\"},{\"start\":[4],\"attrs\":{\"table\":{\"indentLeft\":0,\"marginLeft\":0,\"tableGrid\":[25,25,25,25]}},\"name\":\"insertTable\"},{\"start\":[4,0],\"attrs\":{\"row\":{\"height\":1000}},\"name\":\"insertRows\"},{\"start\":[4,0,0],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"FFFF00\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[4,0,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"1\",\"start\":[4,0,0,0,0],\"name\":\"insertText\"},{\"start\":[4,0,1],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"FFFF00\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[4,0,1,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"2\",\"start\":[4,0,1,0,0],\"name\":\"insertText\"},{\"start\":[4,0,2],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"FFFF00\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[4,0,2,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"3\",\"start\":[4,0,2,0,0],\"name\":\"insertText\"},{\"start\":[4,0,3],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"FFFF00\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[4,0,3,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"4\",\"start\":[4,0,3,0,0],\"name\":\"insertText\"},{\"start\":[4,1],\"attrs\":{\"row\":{\"height\":2000}},\"name\":\"insertRows\"},{\"start\":[4,1,0],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"92D050\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[4,1,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"5\",\"start\":[4,1,0,0,0],\"name\":\"insertText\"},{\"start\":[4,1,1],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"92D050\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[4,1,1,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"6\",\"start\":[4,1,1,0,0],\"name\":\"insertText\"},{\"start\":[4,1,2],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"FFC000\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[4,1,2,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"7\",\"start\":[4,1,2,0,0],\"name\":\"insertText\"},{\"start\":[4,1,3],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"92D050\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[4,1,3,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"8\",\"start\":[4,1,3,0,0],\"name\":\"insertText\"},{\"start\":[4,2],\"attrs\":{\"row\":{\"height\":1000}},\"name\":\"insertRows\"},{\"start\":[4,2,0],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"92D050\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[4,2,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"9\",\"start\":[4,2,0,0,0],\"name\":\"insertText\"},{\"start\":[4,2,1],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"00B0F0\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[4,2,1,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"10\",\"start\":[4,2,1,0,0],\"name\":\"insertText\"},{\"start\":[4,2,2],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"92D050\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[4,2,2,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"11\",\"start\":[4,2,2,0,0],\"name\":\"insertText\"},{\"start\":[4,2,3],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"92D050\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[4,2,3,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"12\",\"start\":[4,2,3,0,0],\"name\":\"insertText\"},{\"start\":[4,3],\"attrs\":{\"row\":{\"height\":1000}},\"name\":\"insertRows\"},{\"start\":[4,3,0],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"92D050\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[4,3,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"13\",\"start\":[4,3,0,0,0],\"name\":\"insertText\"},{\"start\":[4,3,1],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"92D050\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[4,3,1,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"14\",\"start\":[4,3,1,0,0],\"name\":\"insertText\"},{\"start\":[4,3,2],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"92D050\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[4,3,2,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"15\",\"start\":[4,3,2,0,0],\"name\":\"insertText\"},{\"start\":[4,3,3],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"92D050\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[4,3,3,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"16\",\"start\":[4,3,3,0,0],\"name\":\"insertText\"},{\"start\":[5],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"}},\"name\":\"insertParagraph\"},{\"start\":[6],\"attrs\":{\"paragraph\":{\"style\":\"Heading2\",\"outlineLevel\":1}},\"name\":\"insertParagraph\"},{\"text\":\"Table with different background colors (theme, toggle background)\",\"start\":[6,0],\"name\":\"insertText\"},{\"start\":[7],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"}},\"name\":\"insertParagraph\"},{\"start\":[8],\"attrs\":{\"table\":{\"indentLeft\":0,\"marginLeft\":0,\"tableGrid\":[16,17,17,17,17,16]}},\"name\":\"insertTable\"},{\"start\":[8,0],\"attrs\":{\"row\":{\"height\":677}},\"name\":\"insertRows\"},{\"start\":[8,0,0],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"FFFFFF\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"width\":106},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,0,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"Point\",\"start\":[8,0,0,0,0],\"name\":\"insertText\"},{\"start\":[8,0,1],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"FFFFFF\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"width\":106},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,0,1,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"Point A\",\"start\":[8,0,1,0,0],\"name\":\"insertText\"},{\"start\":[8,0,2],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"FFFFFF\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"width\":106},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,0,2,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"Point B\",\"start\":[8,0,2,0,0],\"name\":\"insertText\"},{\"start\":[8,0,3],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"FFFFFF\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"width\":106},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,0,3,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"Point C\",\"start\":[8,0,3,0,0],\"name\":\"insertText\"},{\"start\":[8,0,4],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"FFFFFF\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"width\":106},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,0,4,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"Point D\",\"start\":[8,0,4,0,0],\"name\":\"insertText\"},{\"start\":[8,0,5],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"FFFFFF\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"width\":106},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,0,5,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"Point E\",\"start\":[8,0,5,0,0],\"name\":\"insertText\"},{\"start\":[8,1],\"attrs\":{\"row\":{\"height\":677}},\"name\":\"insertRows\"},{\"start\":[8,1,0],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"FFFFFF\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"width\":35},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,1,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"Point A\",\"start\":[8,1,0,0,0],\"name\":\"insertText\"},{\"start\":[8,1,1],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"D3DFEE\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,1,1,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"—\",\"start\":[8,1,1,0,0],\"name\":\"insertText\"},{\"start\":[8,1,2],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"D3DFEE\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,1,2,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"start\":[8,1,3],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"D3DFEE\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,1,3,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"start\":[8,1,4],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"D3DFEE\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,1,4,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"start\":[8,1,5],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"D3DFEE\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"width\":35},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,1,5,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"start\":[8,2],\"attrs\":{\"row\":{\"height\":677}},\"name\":\"insertRows\"},{\"start\":[8,2,0],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"FFFFFF\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"width\":35},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,2,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"Point B\",\"start\":[8,2,0,0,0],\"name\":\"insertText\"},{\"start\":[8,2,1],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,2,1,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"87\",\"start\":[8,2,1,0,0],\"name\":\"insertText\"},{\"start\":[8,2,2],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,2,2,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"—\",\"start\":[8,2,2,0,0],\"name\":\"insertText\"},{\"start\":[8,2,3],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,2,3,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"start\":[8,2,4],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,2,4,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"start\":[8,2,5],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"width\":35},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,2,5,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"start\":[8,3],\"attrs\":{\"row\":{\"height\":677}},\"name\":\"insertRows\"},{\"start\":[8,3,0],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"FFFFFF\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"width\":35},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,3,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"Point C\",\"start\":[8,3,0,0,0],\"name\":\"insertText\"},{\"start\":[8,3,1],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"D3DFEE\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,3,1,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"64\",\"start\":[8,3,1,0,0],\"name\":\"insertText\"},{\"start\":[8,3,2],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"D3DFEE\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,3,2,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"56\",\"start\":[8,3,2,0,0],\"name\":\"insertText\"},{\"start\":[8,3,3],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"D3DFEE\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,3,3,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"—\",\"start\":[8,3,3,0,0],\"name\":\"insertText\"},{\"start\":[8,3,4],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"D3DFEE\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,3,4,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"start\":[8,3,5],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"D3DFEE\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"width\":35},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,3,5,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"start\":[8,4],\"attrs\":{\"row\":{\"height\":677}},\"name\":\"insertRows\"},{\"start\":[8,4,0],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"FFFFFF\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"width\":35},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,4,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"Point D\",\"start\":[8,4,0,0,0],\"name\":\"insertText\"},{\"start\":[8,4,1],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,4,1,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"37\",\"start\":[8,4,1,0,0],\"name\":\"insertText\"},{\"start\":[8,4,2],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,4,2,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"32\",\"start\":[8,4,2,0,0],\"name\":\"insertText\"},{\"start\":[8,4,3],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,4,3,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"91\",\"start\":[8,4,3,0,0],\"name\":\"insertText\"},{\"start\":[8,4,4],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,4,4,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"—\",\"start\":[8,4,4,0,0],\"name\":\"insertText\"},{\"start\":[8,4,5],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"width\":35},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,4,5,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"start\":[8,5],\"attrs\":{\"row\":{\"height\":677}},\"name\":\"insertRows\"},{\"start\":[8,5,0],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"FFFFFF\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"width\":35},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,5,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"Point E\",\"start\":[8,5,0,0,0],\"name\":\"insertText\"},{\"start\":[8,5,1],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"D3DFEE\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,5,1,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"93\",\"start\":[8,5,1,0,0],\"name\":\"insertText\"},{\"start\":[8,5,2],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"D3DFEE\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,5,2,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"35\",\"start\":[8,5,2,0,0],\"name\":\"insertText\"},{\"start\":[8,5,3],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"D3DFEE\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,5,3,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"54\",\"start\":[8,5,3,0,0],\"name\":\"insertText\"},{\"start\":[8,5,4],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"D3DFEE\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,5,4,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"43\",\"start\":[8,5,4,0,0],\"name\":\"insertText\"},{\"start\":[8,5,5],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"D3DFEE\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"4F81BD\",\"type\":\"rgb\"},\"width\":35},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[8,5,5,0],\"attrs\":{\"paragraph\":{\"alignment\":\"center\",\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11}},\"name\":\"insertParagraph\"},{\"text\":\"—\",\"start\":[8,5,5,0,0],\"name\":\"insertText\"},{\"start\":[9],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"}},\"name\":\"insertParagraph\"},{\"start\":[10],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"}},\"name\":\"insertParagraph\"},{\"start\":[11],\"attrs\":{\"paragraph\":{\"style\":\"Heading2\",\"outlineLevel\":1}},\"name\":\"insertParagraph\"},{\"text\":\"Thick table border, but cell can overrule table border\",\"start\":[11,0],\"name\":\"insertText\"},{\"start\":[12],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"}},\"name\":\"insertParagraph\"},{\"start\":[13],\"attrs\":{\"table\":{\"indentLeft\":0,\"marginLeft\":0,\"tableGrid\":[33,33,33]}},\"name\":\"insertTable\"},{\"start\":[13,0],\"attrs\":{\"row\":{\"height\":1000}},\"name\":\"insertRows\"},{\"start\":[13,0,0],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":212},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":26},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":26},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":212},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[13,0,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"start\":[13,0,1],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":212},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":26},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":26},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":26},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[13,0,1,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"start\":[13,0,2],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":212},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":26},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":212},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":26},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[13,0,2,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"start\":[13,1],\"attrs\":{\"row\":{\"height\":1000}},\"name\":\"insertRows\"},{\"start\":[13,1,0],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":212},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":212},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":212},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[13,1,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"start\":[13,1,1],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":26},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":26},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":26},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":26},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[13,1,1,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"start\":[13,1,2],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":26},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":26},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":212},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":26},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[13,1,2,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"start\":[13,2],\"attrs\":{\"row\":{\"height\":1000}},\"name\":\"insertRows\"},{\"start\":[13,2,0],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":212},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":212},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":26},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":212},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[13,2,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"start\":[13,2,1],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":26},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":212},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":26},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":26},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[13,2,1,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"start\":[13,2,2],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":26},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":212},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":212},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":26},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[13,2,2,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"start\":[14],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"}},\"name\":\"insertParagraph\"},{\"start\":[15],\"attrs\":{\"paragraph\":{\"style\":\"Heading2\",\"outlineLevel\":1}},\"name\":\"insertParagraph\"},{\"text\":\"Table with background color for top-left, top-right, bottom-left and bottom-right cell\",\"start\":[15,0],\"name\":\"insertText\"},{\"start\":[16],\"attrs\":{\"table\":{\"indentLeft\":0,\"marginLeft\":0,\"tableGrid\":[33,33,33]}},\"name\":\"insertTable\"},{\"start\":[16,0],\"attrs\":{\"row\":{\"height\":1501}},\"name\":\"insertRows\"},{\"start\":[16,0,0],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"548DD4\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":35},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":35},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[16,0,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"bold\":true,\"fontName\":\"Adobe Caslon Pro\",\"boldComplex\":true,\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"boldAsian\":true,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"Erste Ecke oben links\",\"start\":[16,0,0,0,0],\"name\":\"insertText\"},{\"start\":[16,0,1],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":35},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":35},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[16,0,1,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"boldAsian\":true,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"1\",\"start\":[16,0,1,0,0],\"name\":\"insertText\"},{\"start\":[16,0,2],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":35},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":35},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[16,0,2,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"boldAsian\":true,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"2\",\"start\":[16,0,2,0,0],\"name\":\"insertText\"},{\"start\":[16,1],\"attrs\":{\"row\":{\"height\":1501}},\"name\":\"insertRows\"},{\"start\":[16,1,0],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"C0C0C0\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[16,1,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"boldAsian\":true,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"Zweite Ecke oben links\",\"start\":[16,1,0,0,0],\"name\":\"insertText\"},{\"start\":[16,1,1],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"C0C0C0\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[16,1,1,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"3\",\"start\":[16,1,1,0,0],\"name\":\"insertText\"},{\"start\":[16,1,2],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"C0C0C0\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"none\"},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[16,1,2,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"4\",\"start\":[16,1,2,0,0],\"name\":\"insertText\"},{\"start\":[16,2],\"attrs\":{\"row\":{\"height\":1501}},\"name\":\"insertRows\"},{\"start\":[16,2,0],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":35},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[16,2,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"bold\":true,\"boldComplex\":true,\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"boldAsian\":true,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"Dritte Ecke oben links\",\"start\":[16,2,0,0,0],\"name\":\"insertText\"},{\"start\":[16,2,1],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":35},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[16,2,1,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"5\",\"start\":[16,2,1,0,0],\"name\":\"insertText\"},{\"start\":[16,2,2],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"none\"},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":35},\"borderRight\":{\"style\":\"none\"},\"borderLeft\":{\"style\":\"none\"},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[16,2,2,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"6\",\"start\":[16,2,2,0,0],\"name\":\"insertText\"},{\"start\":[17],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"}},\"name\":\"insertParagraph\"},{\"start\":[18],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"}},\"name\":\"insertParagraph\"},{\"start\":[19],\"attrs\":{\"paragraph\":{\"style\":\"Heading2\",\"outlineLevel\":1}},\"name\":\"insertParagraph\"},{\"text\":\"Funny Table\",\"start\":[19,0],\"name\":\"insertText\"},{\"start\":[20],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"}},\"name\":\"insertParagraph\"},{\"start\":[21],\"attrs\":{\"table\":{\"indentLeft\":0,\"marginLeft\":0,\"tableGrid\":[25,25,25,25]}},\"name\":\"insertTable\"},{\"start\":[21,0],\"attrs\":{\"row\":{\"height\":4001}},\"name\":\"insertRows\"},{\"start\":[21,0,0],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":159},\"gridSpan\":\"2\",\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":159},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":159},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":159},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[21,0,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"A big cell with thick border\",\"start\":[21,0,0,0,0],\"name\":\"insertText\"},{\"start\":[21,0,1],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":159},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[21,0,1,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fillColor\":{\"value\":\"FF0000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"Para1\",\"start\":[21,0,1,0,0],\"name\":\"insertText\"},{\"start\":[21,0,1,1],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fillColor\":{\"value\":\"FF0000\",\"type\":\"rgb\"},\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"Para2\",\"start\":[21,0,1,1,0],\"name\":\"insertText\"},{\"start\":[21,0,2],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":159},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[21,0,2,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"Hello world!\",\"start\":[21,0,2,0,0],\"name\":\"insertText\"},{\"start\":[21,1],\"attrs\":{\"row\":{\"height\":2501}},\"name\":\"insertRows\"},{\"start\":[21,1,0],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[21,1,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"start\":[21,1,1],\"attrs\":{\"cell\":{\"paddingTop\":0,\"fillColor\":{\"value\":\"00B050\",\"type\":\"rgb\"},\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"gridSpan\":\"2\",\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":159},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[21,1,1,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"bold\":true,\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"boldAsian\":true,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"Cell background is green.\",\"start\":[21,1,1,0,0],\"name\":\"insertText\"},{\"start\":[21,1,1,1],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"start\":[21,1,2],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":159},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":159},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":159},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[21,1,2,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"bold\":true,\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"boldAsian\":true,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"start\":[21,2],\"name\":\"insertRows\"},{\"start\":[21,2,0],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"},\"width\":212},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[21,2,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"start\":[21,2,1],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"},\"width\":212},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"},\"width\":212},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"},\"width\":212},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"},\"width\":212},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[21,2,1,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"bold\":true,\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"boldAsian\":true,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"Thick red border\",\"start\":[21,2,1,0,0],\"name\":\"insertText\"},{\"start\":[21,2,1,1],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"bold\":true,\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"boldAsian\":true,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"start\":[21,2,2],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":159},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"},\"width\":212},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[21,2,2,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"start\":[21,2,3],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":159},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":159},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":159},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[21,2,3,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"bold\":true,\"fontSizeAsian\":11,\"language\":\"de-DE\",\"fontSize\":11,\"boldAsian\":true,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"start\":[21,3],\"name\":\"insertRows\"},{\"start\":[21,3,0],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":212},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[21,3,0,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"Thick border at bottom of 3 cells\",\"start\":[21,3,0,0,0],\"name\":\"insertText\"},{\"start\":[21,3,1],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"FF0000\",\"type\":\"rgb\"},\"width\":212},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":212},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[21,3,1,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"start\":[21,3,2],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":212},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[21,3,2,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"fontSizeAsian\":11,\"fontSize\":11,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"start\":[21,3,3],\"attrs\":{\"cell\":{\"paddingTop\":0,\"borderTop\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":159},\"borderBottom\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderRight\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"borderLeft\":{\"style\":\"single\",\"color\":{\"value\":\"000000\",\"type\":\"rgb\"},\"width\":18},\"paddingBottom\":0,\"paddingLeft\":191,\"paddingRight\":191}},\"name\":\"insertCells\"},{\"start\":[21,3,3,0],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"},\"character\":{\"bold\":true,\"fontSizeAsian\":11,\"fontSize\":11,\"boldAsian\":true,\"fontNameAsian\":\"Cambria\"}},\"name\":\"insertParagraph\"},{\"text\":\"Normal border here...\",\"start\":[21,3,3,0,0],\"name\":\"insertText\"},{\"start\":[22],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"}},\"name\":\"insertParagraph\"},{\"start\":[23],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"}},\"name\":\"insertParagraph\"},{\"start\":[24],\"attrs\":{\"paragraph\":{\"style\":\"Normal\"}},\"name\":\"insertParagraph\"}"
				+ "]";

		String changeOps = "["
				+ "{\"startGrid\":0,\"start\":[4],\"name\":\"deleteColumns\",\"endGrid\":0},{\"start\":[4],\"attrs\":{\"table\":{\"tableGrid\":[25,25,25]}},\"name\":\"setAttributes\"}"
				+ "]";

		editOperations.add(firstEditOperations);
		editOperations.add(changeOps);

		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), editOperations);
	}

	@Test
	// Verfied: Does it works to delete a hard break within a text container?
	public void editingDeletionTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty_as_can_be";
		String firstEditOperations = "["
				+ "{\"start\":[0],\"name\":\"insertParagraph\"},"
				+ "{\"start\":[0],\"name\":\"delete\",\"end\":[0]},"
				+ "{\"start\":[0],\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"heading1\",\"start\":[0,0],\"name\":\"insertText\"},"
				+ "{\"styleId\":\"heading1\",\"styleName\":\"heading 1\",\"attrs\":{\"paragraph\":{\"outlineLevel\":0},\"next\":\"default_paragraph_style\",\"character\":{\"bold\":true,\"color\":{\"value\":\"accent1\",\"fallbackValue\":\"376092\",\"type\":\"scheme\",\"transformations\":[{\"value\":74902,\"type\":\"shade\"}]},\"fontSize\":14}},\"name\":\"insertStyleSheet\",\"parent\":\"default_paragraph_style\",\"uiPriority\":9,\"type\":\"paragraph\"},"
				+ "{\"start\":[0],\"attrs\":{\"paragraph\":{\"alignment\":null,\"lineHeight\":null,\"fillColor\":null,\"marginTop\":null,\"outlineLevel\":null,\"listStyleId\":null,\"listLevel\":null,\"borderBottom\":null,\"contextualSpacing\":null,\"borderLeft\":null,\"showListLabel\":null,\"marginBottom\":null,\"tabStops\":null,\"listStartValue\":null,\"indentRight\":null,\"indentLeft\":null,\"style\":\"heading1\",\"borderTop\":null,\"borderInside\":null,\"indentFirstLine\":null,\"borderRight\":null},\"character\":{\"vertAlign\":null,\"fontName\":null,\"bold\":null,\"strike\":null,\"fillColor\":null,\"color\":null,\"underline\":null,\"italic\":null,\"language\":null,\"fontSize\":null,\"url\":null}},\"name\":\"setAttributes\"},"
				+ "{\"start\":[0,8],\"name\":\"splitParagraph\"},"
				+ "{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"default_paragraph_style\"}},\"name\":\"setAttributes\"},"
				+ "{\"text\":\"text\",\"start\":[1,0],\"name\":\"insertText\"}"
				+ "]";
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), firstEditOperations);
	}

	@Test
	// Verfied: https://bugs.open-xchange.com/show_bug.cgi?id=25236
	public void editingTextAfterHeadingTestTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty_as_can_be";
		String firstEditOperations = "["
				+ "{\"start\":[0],\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"heading1\",\"start\":[0,0],\"name\":\"insertText\"},"
				+ "{\"styleId\":\"heading1\",\"styleName\":\"heading 1\",\"attrs\":{\"paragraph\":{\"outlineLevel\":0},\"next\":\"default_paragraph_style\",\"character\":{\"bold\":true,\"color\":{\"value\":\"accent1\",\"fallbackValue\":\"376092\",\"type\":\"scheme\",\"transformations\":[{\"value\":74902,\"type\":\"shade\"}]},\"fontSize\":14}},\"name\":\"insertStyleSheet\",\"parent\":\"default_paragraph_style\",\"uiPriority\":9,\"type\":\"paragraph\"},"
				+ "{\"start\":[0],\"attrs\":{\"paragraph\":{\"alignment\":null,\"lineHeight\":null,\"fillColor\":null,\"marginTop\":null,\"outlineLevel\":null,\"listStyleId\":null,\"listLevel\":null,\"borderBottom\":null,\"contextualSpacing\":null,\"borderLeft\":null,\"showListLabel\":null,\"marginBottom\":null,\"tabStops\":null,\"listStartValue\":null,\"indentRight\":null,\"indentLeft\":null,\"style\":\"heading1\",\"borderTop\":null,\"borderInside\":null,\"indentFirstLine\":null,\"borderRight\":null},\"character\":{\"vertAlign\":null,\"fontName\":null,\"bold\":null,\"strike\":null,\"fillColor\":null,\"color\":null,\"underline\":null,\"italic\":null,\"language\":null,\"fontSize\":null,\"url\":null}},\"name\":\"setAttributes\"},"
				+ "{\"start\":[0,8],\"name\":\"splitParagraph\"},"
				+ "{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"default_paragraph_style\"}},\"name\":\"setAttributes\"},"
				+ "{\"start\":[1,0],\"name\":\"splitParagraph\"},"
				+ "{\"text\":\"text\",\"start\":[2,0],\"name\":\"insertText\"}]";
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), firstEditOperations);
	}

	@Test
	public void addingNullTest() {
		final String SOURCE_FILE_NAME_TRUNC = "empty_as_can_be";
		String firstEditOperations = "["
				+ "{\"start\":[0],\"name\":\"insertParagraph\"},"
				+ "{\"text\":\"Überschrift\",\"start\":[0,0],\"name\":\"insertText\"},"
				+ "{\"styleId\":\"heading1\",\"styleName\":\"heading 1\",\"attrs\":{\"paragraph\":{\"outlineLevel\":0},\"next\":\"default_paragraph_style\",\"character\":{\"bold\":true,\"color\":{\"value\":\"accent1\",\"fallbackValue\":\"376092\",\"type\":\"scheme\",\"transformations\":[{\"value\":74902,\"type\":\"shade\"}]},\"fontSize\":14}},\"name\":\"insertStyleSheet\",\"parent\":\"default_paragraph_style\",\"uiPriority\":9,\"type\":\"paragraph\"},"
				+ "{\"start\":[0],\"attrs\":{\"paragraph\":{\"alignment\":null,\"lineHeight\":null,\"fillColor\":null,\"marginTop\":null,\"outlineLevel\":null,\"listStyleId\":null,\"listLevel\":null,\"borderBottom\":null,\"contextualSpacing\":null,\"borderLeft\":null,\"showListLabel\":null,\"marginBottom\":null,\"tabStops\":null,\"listStartValue\":null,\"indentRight\":null,\"indentLeft\":null,\"style\":\"heading1\",\"borderTop\":null,\"borderInside\":null,\"indentFirstLine\":null,\"borderRight\":null},\"character\":{\"vertAlign\":null,\"fontName\":null,\"bold\":null,\"strike\":null,\"fillColor\":null,\"color\":null,\"underline\":null,\"italic\":null,\"language\":null,\"fontSize\":null,\"url\":null}},\"name\":\"setAttributes\"},"
				+ "{\"start\":[0,11],\"name\":\"splitParagraph\"},"
				+ "{\"start\":[1],\"attrs\":{\"paragraph\":{\"style\":\"default_paragraph_style\"}},\"name\":\"setAttributes\"},"
				+ "{\"start\":[1,0],\"name\":\"splitParagraph\"},"
				+ "{\"text\":\"Standard\",\"start\":[2,0],\"name\":\"insertText\"},"
				+ "{\"start\":[2],\"attrs\":{\"paragraph\":{\"alignment\":null,\"lineHeight\":null,\"fillColor\":null,\"marginTop\":null,\"outlineLevel\":null,\"listStyleId\":null,\"listLevel\":null,\"borderBottom\":null,\"contextualSpacing\":null,\"borderLeft\":null,\"showListLabel\":null,\"marginBottom\":null,\"tabStops\":null,\"listStartValue\":null,\"indentRight\":null,\"indentLeft\":null,\"style\":\"Standard\",\"borderTop\":null,\"borderInside\":null,\"indentFirstLine\":null,\"borderRight\":null},\"character\":{\"vertAlign\":null,\"fontName\":null,\"bold\":null,\"strike\":null,\"fillColor\":null,\"color\":null,\"underline\":null,\"italic\":null,\"language\":null,\"fontSize\":null,\"url\":null}},\"name\":\"setAttributes\"},"
				+ "{\"start\":[2,8],\"name\":\"splitParagraph\"},"
				+ "{\"start\":[3,0],\"name\":\"splitParagraph\"},"
				+ "{\"text\":\"Überschrift2\",\"start\":[4,0],\"name\":\"insertText\"},"
				+ "{\"styleId\":\"heading2\",\"styleName\":\"heading 2\",\"attrs\":{\"paragraph\":{\"outlineLevel\":1},\"next\":\"default_paragraph_style\",\"character\":{\"bold\":true,\"color\":{\"value\":\"accent1\",\"fallbackValue\":\"4F81BD\",\"type\":\"scheme\"},\"fontSize\":13}},\"name\":\"insertStyleSheet\",\"parent\":\"default_paragraph_style\",\"uiPriority\":9,\"type\":\"paragraph\"},"
				+ "{\"start\":[4],\"attrs\":{\"paragraph\":{\"alignment\":null,\"lineHeight\":null,\"fillColor\":null,\"marginTop\":null,\"outlineLevel\":null,\"listStyleId\":null,\"listLevel\":null,\"borderBottom\":null,\"contextualSpacing\":null,\"borderLeft\":null,\"showListLabel\":null,\"marginBottom\":null,\"tabStops\":null,\"listStartValue\":null,\"indentRight\":null,\"indentLeft\":null,\"style\":\"heading2\",\"borderTop\":null,\"borderInside\":null,\"indentFirstLine\":null,\"borderRight\":null},\"character\":{\"vertAlign\":null,\"fontName\":null,\"bold\":null,\"strike\":null,\"fillColor\":null,\"color\":null,\"underline\":null,\"italic\":null,\"language\":null,\"fontSize\":null,\"url\":null}},\"name\":\"setAttributes\"},"
				+ "{\"text\":\"1\",\"start\":[0,11],\"name\":\"insertText\"}"
				+ "]";
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), firstEditOperations);
	}

	@Test
	public void borderImportTest() {
		final String SOURCE_FILE_NAME_TRUNC = "Tabelle1";
		final String firstEditOperations =
				"["
				+ "{\"name\":\"insertText\",\"text\":\"NewText\",\"start\":[0,0]},"
				+ "{\"name\":\"setAttributes\",\"start\":[0],\"attrs\":{"
				+ "\"paragraph\":{"
				+ "\"borderLeft\":{\"style\":\"single\",\"width\":17,\"space\":140,\"color\":{\"type\":\"auto\"}},"
				+ "\"borderRight\":{\"style\":\"single\",\"width\":17,\"space\":140,\"color\":{\"type\":\"auto\"}},"
				+ "\"borderTop\":{\"style\":\"single\"},"
				+ "\"borderBottom\":{\"style\":\"none\"},"
				+ "\"borderInside\":{\"style\":\"none\"}},"
				+ "\"character\":{"
				+ "\"color\":{\"type\":\"auto\"},"
				+ "\"fillColor\":{\"type\":\"rgb\",\"value\":\"FF00FF\"}}}"
				+ "},"
				//				+ "{\"name\":\"setAttributes\",\"start\":[0],\"attrs\":{" 
				//				+"\"character\":{"
				//					+"\"color\":{\"type\":\"rgb\",\"value\":\"000000\"}}}" 
				//				+ "}"
				+ "]";
		super.roundtripRegressionTest(SOURCE_FILE_NAME_TRUNC, getTestMethodName(), firstEditOperations);

//			//**********RELOAD
//			TextOperationDocument doc3 = new TextOperationDocument(new FileInputStream(ResourceUtilities.newTestOutputFile(SOURCE_FILE_NAME_TRUNC + UNDER_SCORE + getTestMethodName() + TARGET_FILE_NAME)));
//			// read the (known) operation from the test document			
//			JSONObject ops2 = doc3.getOperations();
//			LOG.log(Level.INFO, "The reloaded Operations in ASCII are:{0}", ops2.toString(true).replace("},{", "},\n{"));
//			// Test the knew read operations with the above one - using ASCII-compare, e.g. UTF-8 character for list bullets 
//			Assert.assertTrue(ops2.toString(true).replace("},{", "},\n{").equals(EXPECTED_OPS.toString()));			


//			//**********GET OPS
//			JSONObject ops = doc.getOperations();
//			System.out.println("ops:" + ops);
//			//**********APPLY
////			TextOperationDocument doc2 = new TextOperationDocument();
////			int opCount = doc2.applyOperations(ops);
////			LOG.log(Level.INFO, "opCount: {0}", opCount);
////
//////   UUID uniqueKey = UUID.randomUUID(); 			
////			
////			//**********SAVE		
////			File testFile = ResourceUtilities.newTestOutputFile(SOURCE_FILE_NAME_TRUNC + UNDER_SCORE + getTestMethodName() + TARGET_FILE_NAME_1);
////			doc2.getDocument().save(testFile);
////
////			LOG.log(Level.INFO, "***Saved:\n\t{0}", testFile.getAbsolutePath());
//
//			// read the (known) operation from the test document			
//			OdfContentDom contentDom2 = doc2.getDocument().getContentDom();
//			XPath xpath = contentDom2.getXPath();
//			Node rootNode = contentDom2.getRootElement();
//
//			// heading with no automatic properties should only bear a style reference
//			TextHElement header1 = (TextHElement) xpath.evaluate(".//text:h[1]", rootNode, XPathConstants.NODE);
//			if (header1 == null || !header1.getTextStyleNameAttribute().equals("Heading2")) {
//				Assert.fail("The first heading does not have the expected style 'Heading2'!");
//			}
//
//			TextHElement header = (TextHElement) xpath.evaluate(".//text:h[contains(text(), 'Paragraph Borders Outside')]", rootNode, XPathConstants.NODE);
//			if (header == null) {
//				Assert.fail("The test heading with 'Paragraph Borders Outside' could not be found!");
//			}
//			Node nextSibling = header.getNextSibling();
//			while (!(nextSibling instanceof OdfElement)) {
//				nextSibling = header.getNextSibling();
//			}
//			TextPElement testParagraph = (TextPElement) nextSibling;
//			OdfStyleProperty borderProp = OdfParagraphProperties.Border;
//			OdfStyleProperty borderBottomProp = OdfParagraphProperties.BorderBottom;
//			OdfStyleProperty borderLeftProp = OdfParagraphProperties.BorderLeft;
//			OdfStyleProperty borderRightProp = OdfParagraphProperties.BorderRight;
//			OdfStyleProperty borderTopProp = OdfParagraphProperties.BorderTop;
////			String border = testParagraph.getProperty(borderProp);
//			String borderBottom = testParagraph.getProperty(borderBottomProp);
//			String borderLeft = testParagraph.getProperty(borderLeftProp);
//			String borderRight = testParagraph.getProperty(borderRightProp);
//			String borderTop = testParagraph.getProperty(borderTopProp);
//
//			System.out.println("STYLENAME: " + testParagraph.getStyleName());
//			
////			System.out.println("******************************CONTENT:" + nextSibling.getTextContent());
//			System.out.println("******************************border:" + border);
//			System.out.println("******************************borderBottom:" + borderBottom);
//			System.out.println("******************************borderLeft:" + borderLeft);
//			System.out.println("******************************borderRight:" + borderRight);
//			System.out.println("******************************borderTop:" + borderTop);
//
//
//			NodeList header = odfContent2.getElementsByTagNameNS(OdfDocumentNamespace.TEXT.getUri(), "h");
//			Node node2 = header.
//			String paragraphText2 = node2.getTextContent();			

	}
//	
//	@Test
//	@Ignore
//	public void borderImportTestTest() {
//		final String SOURCE_FILE_NAME_TRUNC = "Tabelle1";
//		final String SOURCE_FILE_NAME_SUFFIX = ".odt";
//		final String TARGET_FILE_NAME_1 = "_OUT_1.odt";
//
//
////		final String OPERATIONS = "[{\"name\":\"insertText\",\"text\":\"NewText4\",\"start\":[0,0]}," 
////				+ "{\"name\":\"setAttributes\",\"start\":[1],\"attrs\":{\"paragraph\":{"
////				+ "\"borderLeft\":{\"style\":\"single\",\"width\":17,\"space\":140,\"color\":{\"type\":\"rgb\",\"value\":\"000000\"}},"
////				+ "\"borderRight\":{\"style\":\"single\",\"width\":17,\"space\":140,\"color\":{\"type\":\"rgb\",\"value\":\"000000\"}},"
////				+ "\"borderTop\":{\"style\":\"single\",\"width\":17,\"space\":140,\"color\":{\"type\":\"rgb\",\"value\":\"000000\"}},"
////				+ "\"borderBottom\":{\"style\":\"single\",\"width\":17,\"space\":140,\"color\":{\"type\":\"rgb\",\"value\":\"000000\"}},"
////				+ "\"borderInside\":{\"style\":\"single\",\"width\":17,\"space\":140,\"color\":{\"type\":\"rgb\",\"value\":\"000000\"}}}}}]";
//
//		try {
//			//**********LOAD
//			// read the (known) operation from the test document
//			TextOperationDocument doc = new TextOperationDocument(new FileInputStream(ResourceUtilities.newTestOutputFile(SOURCE_FILE_NAME_TRUNC + SOURCE_FILE_NAME_SUFFIX)));
//
//
//			//**********APPLY
//			int opCount = doc.applyOperations(OPERATIONS);
//			LOG.log(Level.INFO, "opCount: {0}", opCount);
//
//			//**********SAVE		
//			File testFile = ResourceUtilities.newTestOutputFile(SOURCE_FILE_NAME_TRUNC + UNDER_SCORE + getTestMethodName() + TARGET_FILE_NAME_1);
//			doc.getDocument().save(testFile);
//
//			LOG.log(Level.INFO, "***Saved:\n\t{0}", testFile.getAbsolutePath());
//
//
//		} catch (Throwable e) {
//			Logger.getLogger(LoadSaveTest.class.getName()).log(Level.SEVERE, e.getMessage(), e);
//			Assert.fail(e.getMessage());
//		}
//	}	
}