/*
 * Copyright 2012 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.odftoolkit.odfdom.component;

import java.util.AbstractList;
import java.util.List;
import java.util.logging.Logger;
import org.odftoolkit.odfdom.pkg.OdfElement;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

/**
 * This container can be used for components including a mix of text and
 * elements, where each character and element are components. Instead of a list
 * of this mixed components only a list of the elements is being held
 *
 * @author svante.schubertATgmail.com
 */
public class TextComponent<T> extends Component {

	private static final Logger LOG = Logger.getLogger(Component.class.getName());

	public TextComponent(OdfElement componentElement, Component parent) {
		super(componentElement, parent);
	}

	public void appendText(Text text) {
		mRootElement.appendChild(text);
	}

	public void removeText(int start, int end) {
		((TextContainingElement) mRootElement).delete(start, end);
	}

	public Node appendChild(Node node) {
		Node newNode = mRootElement.appendChild(node);
		return newNode;
	}
	

	@Override
	// Svante REMOVE THIS LATER AS THIS IT IS ONLY USED BY TABLES
	public List getChildren() {
		return list(this);
	}

	// Svante ToDo: After all the refactoring this looks like something to change after the release as well.
	private List<Component> list(final TextComponent c) {
		return new AbstractList<Component>() {
			public int size() {
				return c.componentLength();
			}

			public Component get(int index) {
				return c.getChild(index);				
			}
		};
	}

	/**
	 * Adds the given component to the root element
	 */
	public void add(int index, Component c) {
		OdfElement rootElement = c.getRootElement();
		// SVANTE: WARUM SOLLTE ICH NOCHGEINMAL VOM SAXPARSER EIN ELEMENT EINF\u00dcGEN		
		if (index >= 0) {
			mRootElement.insertBefore(rootElement, ((TextContainingElement) mRootElement).receiveNode(index));
		} else {
			mRootElement.appendChild(rootElement);
		}

	}

	/**
	 * @return either a text node of size 1 or an element being the root element
	 * of a component
	 */
	// SVANTE: ToDo: Alles muss NODES zur\u00fcckgeben? Gibt es noch bessere Methode??!?
	public Component getChild(int index) {
		Component c = null;
		Node node = ((TextContainingElement) mRootElement).receiveNode(index);
		if(node != null){
			if (node instanceof Element) {
				c = ((OdfElement) node).getComponent();
			}
//			else{
//				LOG.finest("Only text found!");
//			}			
		}
//		else{
//			LOG.finest("No Child to be returned.");			
//		}	
		return c;
	}
	
	/**
	 * @return the node (text or element) representing the component or null
	 */
	public Node getComponentNode(int index) {
		return ((TextContainingElement) mRootElement).receiveNode(index);
	}	

	/**
	 * Removes a component from the text element container. Removes either an
	 * element representing a component or text node of size 1
	 */
	@Override
	public Component remove(int index) {
		Component c = null;
		Node node = (Node) this.getChild(index);
		node = mRootElement.removeChild(node);
		if (node instanceof Element) {
			c = ((OdfElement) node).getComponent();
		}
		return c;
	}

	/**
	 * All children of the root element will be traversed. If it is a text node
	 * the size is added, if it is an element and a component a size of one is
	 * added, if it is a marker, for known text marker elements (text:span,
	 * text:bookmark) the children are recursive checked
	 * 
	 * @return the number of child components
	 */
	public int componentLength() {
		return mRootElement.componentLength();
	}

	/**
	 * Recursive traverse the text container and count the children
	 */
	private int childCount(Node parent, int size, Node targetNode) {
		NodeList children = parent.getChildNodes();
		Node child;
		for (int i = 0; i < children.getLength(); i++) {
			child = children.item(i);
			if (child != targetNode) {
				if (child instanceof Text) {
					size += ((Text) child).getLength();
				} else if (child instanceof Element) {
					if (OdfFileSaxHandler.isTextSelection(child)) {
						size = childCount(child, size, targetNode);
					} else if (child instanceof OdfElement && ((OdfElement) child).isComponentRoot()) {
						size++;
					} 
//					else if (child instanceof TextSElement) {
//						 Integer spaceCount = ((TextSElement) child).getTextCAttribute();
//						if(spaceCount == null){
//							size++;
//						}else{
//							size += spaceCount;
//						}
//					}
				}
			} else {
				break;
			}
		}
		return size;
	}

	public int indexOf(Object o) {
		Node targetNode = null;
		if (o instanceof Component) {
			targetNode = ((Component) o).getRootElement();
		} else if (o instanceof Node) {
			targetNode = (Node) o;
		}
		int position = 0;
		if (targetNode != null) {
			position += childCount(mRootElement, position, targetNode);
		}
		return position;
	}
}
