/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.tools;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.logging.Log;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.DefaultFile;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.FileStorageFolder;
import com.openexchange.file.storage.FileStoragePermission;
import com.openexchange.file.storage.composition.FolderAware;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.folderstorage.ContentType;
import com.openexchange.folderstorage.FolderService;
import com.openexchange.folderstorage.UserizedFolder;
import com.openexchange.folderstorage.database.contentType.InfostoreContentType;
import com.openexchange.groupware.ldap.User;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.tx.TransactionAwares;
import com.openexchange.user.UserService;

/**
 * {@link FileHelper}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public class FileHelper {
	private static final Log LOG = com.openexchange.log.Log.loggerFor(FileHelper.class);

    static public final int DEFAULT_BUFFER_SIZE = 8192;

    static public final String OX_DOCUMENT_DOCUMENTSTREAM_NAME = "oxoffice/documentStream";

    static public final String OX_DOCUMENT_OPERATIONSSTREAM_NAME = "oxoffice/operationUpdates";

    static public final String OX_DOCUMENT_RESOURCESTREAM_NAME_PREFIX = "oxoffice/resource/";

    static public final String OX_DOCUMENT_EXTENSION = "_ox";

    // ------------------------------------------------------

    static public final String IMAGE_JPEG = "image/jpeg";

    static public final String EXTENSION_JPG_1 = "jpg";

    static public final String EXTENSION_JPG_2 = "jpeg";

    static public final String IMAGE_EMF = "image/x-emf";

    static public final String IMAGE_EMF2 = "image/emf";

    static public final String EXTENSION_EMF = "emf";

    static public final String IMAGE_WMF = "image/x-wmf";

    static public final String EXTENSION_WMF = "wmf";

    static public final String IMAGE_GIF = "image/gif";

    static public final String EXTENSION_GIF = "gif";

    static public final String IMAGE_PICT = "image/pict";

    static public final String EXTENSION_PICT = "pct";

    static public final String IMAGE_PNG = "image/png";

    static public final String EXTENSION_PNG = "png";

    public static final String IMAGE_TIFF = "image/tiff";

    public static final String EXTENSION_TIFF = "tiff";

    public static final String EXTENSION_TIFF2 = "tif";

    public static final String IMAGE_EPS = "application/postscript"; // as reported by XmlGraphics

    public static final String EXTENSION_EPS = "eps";

    public static final String IMAGE_BMP = "image/bmp";

    public static final String EXTENSION_BMP = "bmp";

    public static final String EXTENSION_ODFTEXT = "odt";

    public static final String EXTENSION_ODT = "odt";

    public static final String DOC_ODF_TEXT = "application/vnd.oasis.opendocument.text";

    public static final String EXTENSION_ODS = "ods";

    public static final String DOC_ODF_SPREADSHEET = "application/vnd.oasis.opendocument.spreadsheet";

    public static final String EXTENSION_ODP = "odp";

    public static final String DOC_ODF_PRESENTATION = "application/vnd.oasis.opendocument.presentation";

    public static final String EXTENSION_ODG = "odg";

    public static final String DOC_ODF_GRAPHIC = "application/vnd.oasis.opendocument.graphics";

    public static final String EXTENSION_DOCX = "docx";

    public static final String DOC_DOCX_TEXT = "application/vnd.openxmlformats-officedocument.wordprocessingml.document";

    public static final String EXTENSION_XLSX = "xlsx";

    public static final String DOC_DOCX_SPREADSHEET = "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet";

    public static final String EXTENSION_PPTX = "pptx";

    public static final String DOC_DOCX_PRESENTATION = "application/vnd.openxmlformats-officedocument.presentationml.presentation";

    public static final String EXTENSION_XML = "xml";

    public static final String TEXT_XML = "text/xml";

    public static final String OCTET_STREAM = "application/octet-stream";

    /**
     * {@link DocumentType}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     */
    public enum DocumentType {
        UNDEFINED, DOCX, ODF, PPTX, XLSX
    }

    /**
     * Initializes a new {@link FileHelper}.
     */
    public FileHelper(ServiceLookup services, String folderId, String fileId) {
        super();

        m_services = services;
        m_folderId = folderId;
        m_fileId = fileId;
    }

    /**
     * Initializes a new {@link FileHelper}.
     */
    public FileHelper(ServiceLookup services, String folderId, String fileId, String version) {
        super();

        m_services = services;
        m_folderId = folderId;
        m_fileId = fileId;
        m_version = version;
    }

    /**
     * @return
     */
    public ServiceLookup getServiceLookup() {
        return m_services;
    }

    /**
     * @return
     */
    public String getFolderId() {
        return m_folderId;
    }

    /**
     * @return
     */
    public String getFileId() {
        return m_fileId;
    }

    /**
     * @return
     */
    public DocumentType getDocumentType() {
        return m_documentType;
    }

    /**
     * @return write protect state of the file
     */
    public boolean isWriteProtected() {
        return m_writeProtected;
    }

    /**
     *
     * @return Date when the lock is released or
     *  null if no lock has been set.
     */
    public Date lockedUntil() {
        return m_lockedUntil;
    }

    /**
     *
     */
    public String getLockedByUser() {
        return m_lockedByUser;
    }

    /**
     * @param session
     * @return
     */
    public InputStream getDocumentStream(Session session) {
        final boolean[] writeProteced = { false };
        final Date[] lockedUntil = { null };
        final String[] lockedByUser = { "" };
        final DocumentType[] documentType = { DocumentType.UNDEFINED };
        final InputStream resultStm = getDocumentStream(
            m_services,
            session,
            m_fileId,
            m_folderId,
            FileStorageFileAccess.CURRENT_VERSION,
            writeProteced,
            lockedUntil,
            lockedByUser,
            documentType);

        if (null != resultStm) {
            m_writeProtected = writeProteced[0];
            m_lockedUntil = lockedUntil[0];
            m_lockedByUser = lockedByUser[0];

            if (DocumentType.UNDEFINED == m_documentType) {
                m_documentType = documentType[0];
            }
        }

        return resultStm;
    }

    /**
     * @param session
     * @param requestData
     * @return
     */
    static public InputStream getDocumentStream(ServiceLookup services, Session session, AJAXRequestData requestData) {
        final String fileId = requestData.getParameter("id");
        final String folderId = requestData.getParameter("folder_id");
        String version = requestData.getParameter("version");

        if ((null == version) || (0 == version.length()) || (version.equals("0"))) {
            version = FileStorageFileAccess.CURRENT_VERSION;
        }

        return getDocumentStream(services, session, fileId, folderId, version, null, null, null, null);
    }

    /**
     * @param session
     * @param fileId
     * @param folderId
     * @param version
     * @param writeProtected
     * @param documentType
     * @return
     */
    static protected InputStream getDocumentStream(ServiceLookup services, Session session, String fileId, String folderId, String version, boolean[] writeProtected, Date[] lockedUntil, String[] lockedByUser, DocumentType[] documentType) {
        InputStream documentStm = null;
        final IDBasedFileAccessFactory fileFactory = services.getService(IDBasedFileAccessFactory.class);
        final UserService userService = services.getService(UserService.class);
        final IDBasedFileAccess fileAccess = ((null != fileFactory) && (null != session)) ? fileFactory.createAccess(session) : null;

        if ((null != fileId) && (fileId.length() > 0) && (null != fileAccess)) {
            try {
                if ((null != writeProtected) && (writeProtected.length > 0)) {
                    writeProtected[0] = !folderHasWriteAccess(fileAccess, folderId);
                }

                documentStm = fileAccess.getDocument(fileId, version);

                if ((null != documentStm) && (null != documentType) && (documentType.length > 0)) {
                    com.openexchange.file.storage.File metadata = null;

                    try {
                        metadata = fileAccess.getFileMetadata(fileId, version);
                    } catch (OXException e) {
                        ;
                    }

                    if (null != metadata) {
                        documentType[0] = getDocumentType(metadata.getFileName());
                        lockedUntil[0] = metadata.getLockedUntil();

                        if (null != lockedUntil[0] && null != userService) {
                            try {
                                User user = userService.getUser(metadata.getModifiedBy(), userService.getContext(session.getContextId()));
                                lockedByUser[0] = user.getDisplayName();
                            } catch (OXException e) {
                                // set user name to empty for unkown user name
                                lockedByUser[0] = "";
                            }
                        }
                    }
                }
            } catch (Exception e) {
                documentStm = null;
            } finally {
                try {
                    fileAccess.finish();
                } catch (OXException e) {
                    ;
                }
            }
        }

        return documentStm;
    }

    /**
     * @param session
     * @param documentStream
     * @param userExtension
     * @return
     */
    public String writeDocumentStream(Session session, InputStream documentStream, String userExtension, boolean setUserExtension, boolean revisionless) {
        String newFileVersion = ""; // empty string means error, version can never be "".
        final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
        final IDBasedFileAccess fileAccess = ((null != fileFactory) && (null != session)) ? fileFactory.createAccess(session) : null;

        if ((null != documentStream) && (null != m_fileId) && (m_fileId.length() > 0) && (null != fileAccess)) {
            final String version = FileStorageFileAccess.CURRENT_VERSION;
            boolean rollback = false;

            try {
                final com.openexchange.file.storage.File metadata = fileAccess.getFileMetadata(m_fileId, version);
                final com.openexchange.file.storage.DefaultFile newMetadata = new DefaultFile();
                final String fileName = metadata.getFileName();
                final ArrayList<com.openexchange.file.storage.File.Field> modifiedColumns = new ArrayList<com.openexchange.file.storage.File.Field>();

                newMetadata.setId(m_fileId);
                newMetadata.setFolderId(metadata.getFolderId());
                newMetadata.setFileMIMEType(metadata.getFileMIMEType());

                // ensure that the filename has the given user extension, if set as call parameter
                if (null != userExtension) {
                    final boolean hasUserExtension = fileName.endsWith(userExtension);

                    if (!hasUserExtension && setUserExtension) {
                        // add user extension
                        newMetadata.setFileName(fileName + userExtension);
                        modifiedColumns.add(com.openexchange.file.storage.File.Field.FILENAME);
                    } else if (hasUserExtension && !setUserExtension) {
                        // remove user extension
                        newMetadata.setFileName(fileName.substring(0, fileName.length() - userExtension.length()));
                        modifiedColumns.add(com.openexchange.file.storage.File.Field.FILENAME);
                    }
                }

                fileAccess.startTransaction();
                rollback = true;
                if (revisionless) {
                    // try to save this document using the non-revision save available at IDBasedIgnorableVersionFileAccess
                    if (fileAccess instanceof com.openexchange.file.storage.composition.IDBasedIgnorableVersionFileAccess) {
                        ((com.openexchange.file.storage.composition.IDBasedIgnorableVersionFileAccess)fileAccess).saveDocument(
                            newMetadata, documentStream, FileStorageFileAccess.DISTANT_FUTURE, modifiedColumns, true);
                    }
                } else {
                    fileAccess.saveDocument(newMetadata, documentStream, FileStorageFileAccess.DISTANT_FUTURE, modifiedColumns);
                }
                fileAccess.commit();
                rollback = false;

                newFileVersion = newMetadata.getVersion();
            } catch (final Exception e) {
                LOG.error("Couldn't write document stream", e);
            } finally {
                // Roll-back (if needed) and finish
                if (rollback) {
                    TransactionAwares.rollbackSafe(fileAccess);
                }
                TransactionAwares.finishSafe(fileAccess);
            }
        }

        return newFileVersion;
    }

    /**
     * @param session
     * @param newFileName
     * @param newFileVersion
     * @return
     */
    public String renameDocument(Session session, String newFileName, String[] newFileVersion) {
        String resultFileName = null;
        final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
        final IDBasedFileAccess fileAccess = ((null != fileFactory) && (null != session)) ? fileFactory.createAccess(session) : null;
        final String currentVersion = FileStorageFileAccess.CURRENT_VERSION;

        if (null != fileAccess) {
            try {
                if ((null != newFileName) && (newFileName.length() > 0)) {
                    final ArrayList<com.openexchange.file.storage.File.Field> modifiedColumns = new ArrayList<com.openexchange.file.storage.File.Field>();
                    final DefaultFile metadata = new DefaultFile();

                    metadata.setId(this.m_fileId);
                    metadata.setFolderId(this.m_folderId);
                    metadata.setFileName(newFileName);
                    metadata.setVersion(currentVersion);

                    modifiedColumns.add(com.openexchange.file.storage.File.Field.FILENAME);
                    fileAccess.saveFileMetadata(metadata, FileStorageFileAccess.DISTANT_FUTURE, modifiedColumns);
                    fileAccess.commit();

                    // retrieve the real filename from the metadata, it may have
                    // been changed from the requested one by the file service;
                    // add a short sleep in order to avoid asynchronous errors while
                    // getting the updated metadata, which may currently happen with
                    // cascaded DB backends
                    // (see also https://bugs.open-xchange.com/show_bug.cgi?id=26080)
                    final com.openexchange.file.storage.File newMetaData = fileAccess.getFileMetadata(this.m_fileId, currentVersion);
                    resultFileName = (null != newMetaData) ? newMetaData.getFileName() : newFileName;

                    if ((null != newFileVersion) && (newFileVersion.length > 0)) {
                        newFileVersion[0] = (null != newMetaData) ? newMetaData.getVersion() : metadata.getVersion();
                    }
                }
            } catch (Exception e) {
                //
            }
        }

        return resultFileName;
    }

    /**
     * Provides the latest version string of the file
     *
     * @param {Session} session
     *  The session to be used to access the file store.
     *
     * @return {String}
     *  The version string or null.
     */
    public String getVersionFromFile(Session session) {
        final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
        final IDBasedFileAccess fileAccess = ((null != fileFactory) && (null != session)) ? fileFactory.createAccess(session) : null;
        final String currentVersion = FileStorageFileAccess.CURRENT_VERSION;
        com.openexchange.file.storage.File metaData = null;
        String result = null;

        try {
            if (null != fileAccess) {
                metaData = fileAccess.getFileMetadata(this.m_fileId, currentVersion);
                result = metaData.getVersion();
            }
        } catch (OXException e) {
            ;
        }
        return result;
    }

    // - Statics ---------------------------------------------------------------

    /**
     * @param inputStm
     * @return The byte buffer of the given input stream. The given input stream is closed depending on the value of the parameter
     *         closeInputStream.
     */
    static public byte[] readBufferFromStream(InputStream inputStm, boolean closeInputStream) {
        byte[] buffer = null;

        if (null != inputStm) {
            try {
                final ByteArrayOutputStream byteArrayOutputStm = new ByteArrayOutputStream(DEFAULT_BUFFER_SIZE);
                final BufferedOutputStream outputStm = new BufferedOutputStream(byteArrayOutputStm, DEFAULT_BUFFER_SIZE);

                buffer = new byte[DEFAULT_BUFFER_SIZE];

                for (int read = 0; (read = inputStm.read(buffer)) != -1;) {
                    outputStm.write(buffer, 0, read);
                }

                outputStm.flush();
                buffer = byteArrayOutputStm.toByteArray();
                outputStm.close();
            } catch (IOException e) {
                buffer = null;
            }

            if (closeInputStream) {
                try {
                    inputStm.close();
                } catch (IOException e) {
                    ;
                }
            }
        }

        return buffer;
    }

    /**
     * @param outputStm
     * @param content
     */
    static public void writeBufferToStream(OutputStream outputStm, byte[] content) {
        copyStream(new BufferedInputStream(new ByteArrayInputStream(content), DEFAULT_BUFFER_SIZE), outputStm, true);
    }

    /**
     * Copies the content of the input stream to the outputr stream. The input stream gets closed after reading, the output stream gets
     * flushed after writing.
     *
     * @param srcStm The stream to read the content from.
     * @param dstStm The stream to write the content from <code>srcStm</code> to.
     */
    static public void copyStream(InputStream srcStm, OutputStream dstStm, boolean closeInputStream) {
        if ((null != srcStm) && (null != dstStm)) {
            final InputStream inputStm = new BufferedInputStream(srcStm, DEFAULT_BUFFER_SIZE);
            final OutputStream outputStm = new BufferedOutputStream(dstStm, DEFAULT_BUFFER_SIZE);
            final byte[] buf = new byte[DEFAULT_BUFFER_SIZE];
            int read = 0;

            try {
                while ((read = inputStm.read(buf)) != -1) {
                    outputStm.write(buf, 0, read);
                }

                outputStm.flush();
            } catch (IOException e) {
                ;
            } finally {
                if (closeInputStream) {
                    try {
                        inputStm.close();
                    } catch (IOException e) {
                        ;
                    }
                }
            }
        }
    }

    /**
     * @param zipFile the zip file to which the new entry is to be added
     * @param newEntryName the name of the new entry
     * @param newEntryContent the content to be written to the new entry
     * @return the inserted name of the entry. This might be different from the requested newEntryName to avoid duplicate entries. Returns
     *         null in case the new entry could not be added.
     */
    static public String addEntryToZipFile(File zipFile, String newEntryName, InputStream contentStm, boolean closeContentStm) {
        String entryName = newEntryName;
        String retName = null;

        if ((null != zipFile) && (null != entryName) && (null != contentStm) && zipFile.canWrite() && (entryName.length() > 0)) {
            File zipTmpFile = null;
            ZipFile zipReadFile = null;
            @SuppressWarnings("resource") ZipOutputStream zipOutputStm = null;

            try {
                (zipTmpFile = File.createTempFile("oxo", ".tmp")).deleteOnExit();
                zipOutputStm = new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(zipTmpFile)));

                final Enumeration<? extends ZipEntry> zipEntries = (zipReadFile = new ZipFile(zipFile)).entries();

                // copy all existing entries to new temporary ZipFile
                while (zipEntries.hasMoreElements()) {
                    final ZipEntry curZipEntry = zipEntries.nextElement();

                    zipOutputStm.putNextEntry(new ZipEntry(curZipEntry.getName()));
                    copyStream(new BufferedInputStream(zipReadFile.getInputStream(curZipEntry)), zipOutputStm, false);
                    zipOutputStm.closeEntry();
                }

                // add an underscore to the filenanme of the entryName,
                // if an entry with the same name already exists
                while (zipReadFile.getEntry(entryName) != null) {
                    final int pos = entryName.lastIndexOf(".");

                    if ((pos <= 0) || (pos == (entryName.length() - 1))) {
                        entryName = entryName + '_';
                    } else {
                        entryName = entryName.substring(0, pos) + '_' + entryName.substring(pos);
                    }
                }

                // add content of entryStream parameter as new ZipEntry with name entryName
                zipOutputStm.putNextEntry(new ZipEntry(entryName));
                copyStream(contentStm, zipOutputStm, false);
                zipOutputStm.closeEntry();

                // set new entry name as return value
                retName = entryName;
            } catch (Exception e) {
                ; // TODO (KA) consider error message
            } finally {
                // close contentStm
                if (closeContentStm) {
                    try {
                        contentStm.close();
                    } catch (IOException e) {
                        ;
                    }
                }

                // close zipReadFile
                if (null != zipReadFile) {
                    try {
                        zipReadFile.close();
                    } catch (IOException e) {
                        ;
                    } finally {
                        zipReadFile = null;
                    }
                }

                // close zipOutputStm
                if (null != zipOutputStm) {
                    try {
                        zipOutputStm.flush();
                        zipOutputStm.close();
                    } catch (IOException e) {
                        ;
                    } finally {
                        zipOutputStm = null;
                    }
                }

                // in case of success, move zipTmpFile to source
                // file, otherwise delete zipTmpFile and return null
                if (null != zipTmpFile) {
                    if (null != retName) {
                        try {
                            zipFile.delete();

                            if (!zipTmpFile.renameTo(zipFile)) {
                                retName = null;
                            }
                        } catch (Exception e) {
                            retName = null;
                        }
                    }

                    // if something went wrong (null == retName), delete tmpFile,
                    // oherwise, the tmpFile has become the original zipFile by filename
                    if (null == retName) {
                        zipTmpFile.delete();
                        zipTmpFile = null;
                    }
                }
            }
        }

        return retName;
    }

    /**
     * @param fileAccess
     * @param folderId
     * @return
     */
    static public boolean folderHasWriteAccess(IDBasedFileAccess fileAccess, String folderId) {
        boolean ret = true;

        // check write permission
        if (fileAccess instanceof FolderAware) {
            try {
                // create the info store file URL to check for the folder id
                final String infoStoreFileURL = "com.openexchange.infostore://infostore/" + folderId + "/0";
                final FileStorageFolder folder = ((FolderAware) fileAccess).optFolder(infoStoreFileURL);

                if (folder != null) {
                    final FileStoragePermission permission = folder.getOwnPermission();
                    ret = (permission != null) ? (permission.getWritePermission() > FileStoragePermission.NO_PERMISSIONS) : false;
                }
            } catch (final Exception e) {
                ;
            }
        }

        return ret;
    }

    /**
     * @param userLanguage
     * @return
     */
    static public String mapUserLanguageToLangCode(String userLanguage) {
        return (null != userLanguage) ? (userLanguage.indexOf('_') >= 0 ? userLanguage.replace('_', '-') : userLanguage) : null;
    }

    /**
     * @param fileName
     * @return The base name of the given filename
     */
    static public String getBaseName(String fileName) {
        return FilenameUtils.getBaseName(fileName);
    }

    /**
     * @param fileName
     * @return The extension of the given filename
     */
    static public String getExtension(String fileName) {
        String extension = "";
        int beginIndex = fileName.lastIndexOf(".");
        int endIndex = beginIndex;

        if (beginIndex > -1) {
            while (++endIndex < fileName.length()) {
                int c = fileName.codePointAt(endIndex);

                if (!(((c >= 'a') && (c <= 'z')) || ((c >= 'A') && (c <= 'Z'))))
                    break;
            }
        }

        if ((beginIndex > -1) && (endIndex <= fileName.length()) && (beginIndex < (endIndex - 1))) {
            extension = fileName.substring(beginIndex + 1, endIndex);
        }

        return extension.toLowerCase();
    }

    /**
     * @param fileName
     * @return the mime type of the fileName
     */
    static public String getMimeType(String fileName) {
        final String extension = getExtension(fileName);
        String mimeType = "";

        if (extension.length() > 0) {
            switch (getDocumentType(fileName)) {
            case DOCX: {
                mimeType = DOC_DOCX_TEXT;
                break;
            }

            case PPTX: {
                mimeType = DOC_DOCX_PRESENTATION;
                break;
            }

            case XLSX: {
                mimeType = DOC_DOCX_SPREADSHEET;
                break;
            }

            case ODF: {
                mimeType = DOC_ODF_TEXT;
                break;
            }

            default: {
                if (extension.equals(EXTENSION_DOCX)) {
                    mimeType = DOC_DOCX_TEXT;
                } else if (extension.equals(EXTENSION_XLSX)) {
                    mimeType = DOC_DOCX_SPREADSHEET;
                } else if (extension.equals(EXTENSION_PPTX)) {
                    mimeType = DOC_DOCX_PRESENTATION;
                } else if (extension.equals(EXTENSION_ODT)) {
                    mimeType = DOC_ODF_TEXT;
                } else if (extension.equals(EXTENSION_ODS)) {
                    mimeType = DOC_ODF_SPREADSHEET;
                } else if (extension.equals(EXTENSION_ODP)) {
                    mimeType = DOC_ODF_PRESENTATION;
                } else if (extension.equals(EXTENSION_ODG)) {
                    mimeType = DOC_ODF_GRAPHIC;
                } else if (extension.equals(EXTENSION_BMP)) {
                    mimeType = IMAGE_BMP;
                } else if (extension.equals(EXTENSION_EMF)) {
                    mimeType = IMAGE_EMF;
                } else if (extension.equals(EXTENSION_GIF)) {
                    mimeType = IMAGE_GIF;
                } else if (extension.equals(EXTENSION_JPG_1)) {
                    mimeType = IMAGE_JPEG;
                } else if (extension.equals(EXTENSION_JPG_2)) {
                    mimeType = IMAGE_JPEG;
                } else if (extension.equals(EXTENSION_PICT)) {
                    mimeType = IMAGE_PICT;
                } else if (extension.equals(EXTENSION_PNG)) {
                    mimeType = IMAGE_PNG;
                } else if (extension.equals(EXTENSION_TIFF)) {
                    mimeType = IMAGE_TIFF;
                } else if (extension.equals(EXTENSION_TIFF2)) {
                    mimeType = IMAGE_TIFF;
                } else if (extension.equals(EXTENSION_WMF)) {
                    mimeType = IMAGE_WMF;
                } else if (extension.equals(EXTENSION_XML)) {
                    mimeType = TEXT_XML;
                }

                break;
            }
            }
        }

        return ((mimeType.length() < 1) ? OCTET_STREAM : mimeType);
    }

    /**
     * @return
     */
    static public HashMap<String, DocumentType> getExtensionMap() {
        // Java DCL pattern with volatile member
        HashMap<String, DocumentType> serviceMap = m_extensionMap;

        if (null == serviceMap) {
            synchronized (m_Synchronizer) {
                if (null == (serviceMap = m_extensionMap)) {
                    serviceMap = m_extensionMap = new HashMap<String, DocumentType>();

                    m_extensionMap.put("docx", FileHelper.DocumentType.DOCX);
                    m_extensionMap.put("docx_ox", FileHelper.DocumentType.DOCX);
                    m_extensionMap.put("docm", FileHelper.DocumentType.DOCX);
                    m_extensionMap.put("docm_ox", FileHelper.DocumentType.DOCX);
                    m_extensionMap.put("dotx", FileHelper.DocumentType.DOCX);
                    m_extensionMap.put("dotx_ox", FileHelper.DocumentType.DOCX);
                    m_extensionMap.put("dotm", FileHelper.DocumentType.DOCX);
                    m_extensionMap.put("dotm_ox", FileHelper.DocumentType.DOCX);
                    m_extensionMap.put("pptx", FileHelper.DocumentType.PPTX);
                    m_extensionMap.put("pptx_ox", FileHelper.DocumentType.PPTX);
                    m_extensionMap.put("xlsx", FileHelper.DocumentType.XLSX);
                    m_extensionMap.put("xlsx_ox", FileHelper.DocumentType.XLSX);
                    m_extensionMap.put("odt", FileHelper.DocumentType.ODF);
                    m_extensionMap.put("odt_ox", FileHelper.DocumentType.ODF);
                    m_extensionMap.put("ott", FileHelper.DocumentType.ODF);
                    m_extensionMap.put("ott_ox", FileHelper.DocumentType.ODF);
                }
            }
        }

        return serviceMap;
    }

    /**
     * @param fileName
     * @return
     */
    static public DocumentType getDocumentType(String fileName) {
        final String fileExtension = getExtension(fileName);
        final HashMap<String, DocumentType> extensionMap = getExtensionMap();
        DocumentType documentType = DocumentType.UNDEFINED;

        if ((null != fileExtension) && extensionMap.containsKey(fileExtension)) {
            documentType = extensionMap.get(fileExtension);
        }

        return documentType;
    }

    /**
     * @param folderService
     * @param session
     * @return
     */
    static public String getUserFolderId(FolderService folderService, Session session) {
        String folderId = null;

        if ((null != folderService) && (null != session) && (session instanceof ServerSession)) {
            final String treeId = com.openexchange.folderstorage.FolderStorage.REAL_TREE_ID;
            final ContentType contentType = InfostoreContentType.getInstance();

            try {
                UserizedFolder defaultFolder = folderService.getDefaultFolder(
                    ((ServerSession) session).getUser(),
                    treeId,
                    contentType,
                    session,
                    null);

                if (null != defaultFolder) {
                    folderId = defaultFolder.getID();
                }
            } catch (OXException e) {
                ;
            }
        }

        return folderId;
    }

    // - Static members --------------------------------------------------------

    static private HashMap<String, FileHelper.DocumentType> m_extensionMap = null;

    static private volatile Object m_Synchronizer = new Object();

    // - Members --------------------------------------------------------

    private ServiceLookup m_services = null;

    private String m_folderId = null;

    private String m_fileId = null;

    private DocumentType m_documentType = DocumentType.UNDEFINED;

    private volatile String m_version = FileStorageFileAccess.CURRENT_VERSION;

    private boolean m_writeProtected = false;

    private Date m_lockedUntil = null;

    private String m_lockedByUser = "";
}
