/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.tools;

import java.math.BigInteger;
import java.util.Iterator;
import java.util.List;
import org.docx4j.wml.BooleanDefaultTrue;
import org.docx4j.wml.CTBorder;
import org.docx4j.wml.CTShd;
import org.docx4j.wml.CTTabStop;
import org.docx4j.wml.Color;
import org.docx4j.wml.Jc;
import org.docx4j.wml.JcEnumeration;
import org.docx4j.wml.ObjectFactory;
import org.docx4j.wml.PPr;
import org.docx4j.wml.PPrBase;
import org.docx4j.wml.PPrBase.Ind;
import org.docx4j.wml.PPrBase.OutlineLvl;
import org.docx4j.wml.PPrBase.PBdr;
import org.docx4j.wml.STBorder;
import org.docx4j.wml.STLineSpacingRule;
import org.docx4j.wml.Tabs;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.office.ooxml.docx.OperationDocument;

public class Paragraph {

    public static void applyParagraphProperties(OperationDocument operationDocument, ObjectFactory objectFactory, JSONObject paragraphProperties, PPr pPr)
        throws JSONException {

        if(paragraphProperties==null)
            return;

        Iterator<String> keys = paragraphProperties.keys();
        while(keys.hasNext()) {
            String attr = keys.next();
            Object value = paragraphProperties.get(attr);
            if(attr.equals("lineHeight")) {
                if(value instanceof JSONObject) {
                    Iterator<String> lineheightkeys = ((JSONObject)value).keys();
                    String lineheighttype = null;
                    int lineheightvalue = -1;
                    while(lineheightkeys.hasNext()) {
                        String lineheightattr = lineheightkeys.next();
                        if(lineheightattr.equals("type"))
                            lineheighttype = ((JSONObject)value).get(lineheightattr).toString();
                        else if (lineheightattr.equals("value"))
                            lineheightvalue = ((Integer)((JSONObject)value).get(lineheightattr)).intValue();
                    }
                    if(lineheighttype!=null&&lineheightvalue>-1) {
                        STLineSpacingRule lineSpacingRule = null;
                        BigInteger lineSpacingValue = null;
                        if(lineheighttype.equals("fixed")) {
                            lineSpacingRule = STLineSpacingRule.fromValue("exact");
                            lineSpacingValue = BigInteger.valueOf(((long)(lineheightvalue) * 144 ) / 254);
                        } else if (lineheighttype.equals("leading")) {
                            lineSpacingRule = STLineSpacingRule.fromValue("percent");
                            lineSpacingValue = BigInteger.valueOf(((long)(100) * 24 ) / 10);
                        } else if (lineheighttype.equals("atLeast")) {
                            lineSpacingRule = STLineSpacingRule.fromValue("atLeast");
                            lineSpacingValue = BigInteger.valueOf(((long)(lineheightvalue) * 144 ) / 254);
                        } else if (lineheighttype.equals("percent")) {
                            lineSpacingRule = STLineSpacingRule.fromValue("auto");
                            lineSpacingValue = BigInteger.valueOf(((long)(lineheightvalue) * 24 ) / 10);
                        }
                        if(lineSpacingRule!=null&&lineSpacingValue!=null) {
                            PPrBase.Spacing spacing = pPr.getSpacing();
                            if(spacing==null) {
                                spacing = objectFactory.createPPrBaseSpacing();
                                spacing.setParent(pPr);
                                pPr.setSpacing(spacing);
                            }
                            spacing.setLineRule(lineSpacingRule);
                            spacing.setLine(lineSpacingValue);
                        }
                    }
                }
                else
                    pPr.setSpacing(null);
            }
            else if(attr.equals("marginTop")) {
                PPrBase.Spacing spacing = pPr.getSpacing();
                if(value instanceof Integer) {
                    if(spacing==null) {
                        spacing = objectFactory.createPPrBaseSpacing();
                        spacing.setParent(pPr);
                        pPr.setSpacing(spacing);
                    }
                    spacing.setBefore(Utils.map100THMMToTwip((Integer)value));
                }
                else if(spacing!=null)
                    spacing.setBefore(null);
            }
            else if(attr.equals("marginBottom")) {
                PPrBase.Spacing spacing = pPr.getSpacing();
                if(value instanceof Integer) {
                    if(spacing==null) {
                        spacing = objectFactory.createPPrBaseSpacing();
                        spacing.setParent(pPr);
                        pPr.setSpacing(spacing);
                    }
                    spacing.setAfter(Utils.map100THMMToTwip((Integer)value));
                }
                else if(spacing!=null)
                    spacing.setAfter(null);
            }
            else if(attr.equals("alignment")) {
                if (value instanceof String ) {
                    Jc jc = pPr.getJc();
                    if(jc==null) {
                        jc = objectFactory.createJc();
                        jc.setParent(pPr);
                        pPr.setJc(jc);
                    }
                    if(((String)value).equals("left"))
                        jc.setVal(JcEnumeration.LEFT);
                    else if(((String)value).equals("center"))
                        jc.setVal(JcEnumeration.CENTER);
                    else if(((String)value).equals("right"))
                        jc.setVal(JcEnumeration.RIGHT);
                    if(((String)value).equals("justify"))
                        jc.setVal(JcEnumeration.DISTRIBUTE);
                }
                else
                    pPr.setJc(null);
            }
            else if(attr.equals("fillColor")) {
                if(value instanceof JSONObject) {
                    CTShd shd = pPr.getShd();
                    if(shd==null) {
                        shd = objectFactory.createCTShd();
                        shd.setParent(pPr);
                        pPr.setShd(shd);
                    }
                    JSONObject shdObject = (JSONObject)value;
                    Utils.initShdFromJSONColor(operationDocument, shdObject, shd);
                }
                else
                    pPr.setShd(null);
            }
            else if (attr.equals("outlineLevel")) {
                if (value instanceof Integer) {
                    Integer outlineLvl = (Integer)value;
                    OutlineLvl level = objectFactory.createPPrBaseOutlineLvl();
                    level.setParent(pPr);
                    level.setVal(BigInteger.valueOf(outlineLvl));
                    pPr.setOutlineLvl(level);
                }
                else
                    pPr.setOutlineLvl(null);
            }
            else if(attr.equals("listStyleId")) {
                if(value instanceof String) {
                    String sValue = (String)value;
                    if(sValue.length()==0)
                        throw new JSONException("docx export: apply paragraph attributes, listStyleId empty");

                    PPrBase.NumPr numPr = pPr.getNumPr();
                    if(numPr==null) {
                        numPr = objectFactory.createPPrBaseNumPr();
                        numPr.setParent(pPr);
                        pPr.setNumPr(numPr);
                    }
                    Integer numIdValue;
                    if(!java.lang.Character.isDigit(sValue.charAt(0))){
                        numIdValue = Integer.parseInt(sValue.substring(1));
                    } else {
                        numIdValue = Integer.parseInt(sValue);
                    }
                    if(numIdValue >= 0) {
                        PPrBase.NumPr.NumId numId = new PPrBase.NumPr.NumId();
                        numId.setVal(BigInteger.valueOf(numIdValue));
                        numPr.setNumId(numId);
                    } else {
                        pPr.setNumPr( null );
                    }
                }
                else
                    pPr.setNumPr(null);
            }
            else if(attr.equals("listLevel")) {
                if(value instanceof Integer) {
                    Integer intValue = ((Integer)value).intValue();
                    if(intValue<0)
                        pPr.setNumPr(null);
                    else {
                        PPrBase.NumPr numPr = pPr.getNumPr();
                        if(numPr==null) {
                            numPr = objectFactory.createPPrBaseNumPr();
                            numPr.setParent(pPr);
                            pPr.setNumPr(numPr);
                        }
                        PPrBase.NumPr.Ilvl ilvl = new PPrBase.NumPr.Ilvl();
                        ilvl.setVal(BigInteger.valueOf(intValue));
                        numPr.setIlvl(ilvl);
                    }
                }
                else
                    pPr.setNumPr(null);
            }
            else if(attr.equals("borderLeft") || attr.equals("borderRight") ||
                    attr.equals("borderTop") || attr.equals("borderBottom") ||
                    attr.equals("borderInside")) {

                boolean left = attr.equals("borderLeft");
                boolean right = attr.equals("borderRight");
                boolean top = attr.equals("borderTop");
                boolean bottom = attr.equals("borderBottom");

                if(value instanceof JSONObject) {
                    PBdr pBorder = pPr.getPBdr();
                    if(pBorder==null) {
                        pBorder = objectFactory.createPPrBasePBdr();
                        pBorder.setParent(pPr);
                        pPr.setPBdr(pBorder);
                    }
                    CTBorder border = left ? pBorder.getLeft() : right ? pBorder.getRight() : top ? pBorder.getTop() : bottom ? pBorder.getBottom() : pBorder.getBetween();

                    JSONObject borderObject = (JSONObject)value;
                    if( !borderObject.has("style") ) {
                        border = null;
                    } else {
                        if( border == null ) {
                            border = objectFactory.createCTBorder();
                        }
                        String style = borderObject.getString("style");
                        border.setVal(STBorder.fromValue( style ));
                        if(borderObject.has("color"))
                            Utils.initBorderColorFromJSONColor(operationDocument, borderObject.optJSONObject("color"), border);
                        border.setFrame(false);
                        border.setShadow(false);
                        if(borderObject.has("space"))
                            border.setSpace(Utils.map100THMMToTwip( (double)(borderObject.getLong("space") +10 ) / 20 ));
                        if(borderObject.has("width"))
                            border.setSz(Utils.map100THMMToTwip( (double)(borderObject.getLong("width") + 2 )* 2 / 5 ));
                    }
                    if(left)
                        pBorder.setLeft(border);
                    else if(right)
                        pBorder.setRight(border);
                    else if(top)
                        pBorder.setTop(border);
                    else if(bottom)
                        pBorder.setBottom(border);
                    else
                        pBorder.setBetween(border);
                }
                else {
                    PBdr pBorder = pPr.getPBdr();
                    if(pBorder != null){
                        if(left)
                            pBorder.setLeft(null);
                        else if(right)
                            pBorder.setRight(null);
                        else if(top)
                            pBorder.setTop(null);
                        else if(bottom)
                            pBorder.setBottom(null);
                        else
                            pBorder.setBetween(null);
                        if(pBorder.getLeft() == null && pBorder.getRight() == null && pBorder.getTop() == null &&
                            pBorder.getBottom() == null && pBorder.getBetween() == null)
                                pPr.setPBdr(null);
                    }

                }
            } else if ((attr.equals("indentLeft")) || (attr.equals("indentFirstLine")) ||
                        (attr.equals("indentRight"))) {

                boolean left = attr.equals("indentLeft");
                boolean right = attr.equals("indentRight");
                boolean firstLine = attr.equals("indentFirstLine");

                if(value instanceof Integer) {
                    Ind indent = pPr.getInd();
                    if (indent == null) {
                        indent = objectFactory.createPPrBaseInd();
                        indent.setParent(pPr);
                        pPr.setInd(indent);
                    }
                    Long longValue = ((Integer)value).longValue();
                    if (left) {
                        indent.setLeft(Utils.map100THMMToTwip(longValue));
                    }
                    else if (right) {
                        indent.setRight(Utils.map100THMMToTwip(longValue));
                    }
                    else if (firstLine) {
                        indent.setFirstLine(Utils.map100THMMToTwip(longValue));
                    }
                }
            }
            else if(attr.equals("contextualSpacing")){
                if(value instanceof Boolean) {
                    BooleanDefaultTrue contextSpacing = pPr.getContextualSpacing();
                    if(contextSpacing == null){
                        contextSpacing = objectFactory.createBooleanDefaultTrue();
                        pPr.setContextualSpacing(contextSpacing);
                    }
                    contextSpacing.setVal(((Boolean)value).booleanValue());
                }else{
                	pPr.setContextualSpacing(null);
                }
            }
        }
    }

    public static JSONObject createParagraphProperties(PPr paragraphProperties)
        throws JSONException {

        JSONObject jsonParagraphProperties = null;
        if(paragraphProperties!=null) {
            jsonParagraphProperties = new JSONObject();
            PPrBase.Spacing spacing = paragraphProperties.getSpacing();
            if(spacing!=null) {
                STLineSpacingRule lineSpacingRule = spacing.getLineRule();
                BigInteger lineSpacingValue = spacing.getLine();                    // value is always in twip
                if (lineSpacingRule!=null&&lineSpacingValue!=null) {
                    String type = null;
                    int value = -1;
                    if(lineSpacingRule==STLineSpacingRule.AT_LEAST) {
                        type = "atLeast";
                        value = (int)(lineSpacingValue.doubleValue() * 254 / 144);  // twip to 1/100mm
                    } else if (lineSpacingRule==STLineSpacingRule.AUTO) {
                        type = "percent";
                        value = (int)(lineSpacingValue.doubleValue() / 2.4);
                    } else if (lineSpacingRule==STLineSpacingRule.EXACT) {
                        type = "fixed";
                        value = (int)(lineSpacingValue.doubleValue() * 254 /144);   // twip to 1/100mm
                    }
                    if(type!=null&&value>-1) {
                        JSONObject jsonObject = new JSONObject();
                        jsonObject.put("type", type);
                        jsonObject.put("value", value);
                        jsonParagraphProperties.put("lineHeight", jsonObject);
                    }
                }
                BigInteger before = spacing.getBefore();
                if( before != null )
                    jsonParagraphProperties.put("marginTop", Utils.mapTwipTo100THMM(before));
                BigInteger after = spacing.getAfter();
                if( after != null )
                    jsonParagraphProperties.put("marginBottom", Utils.mapTwipTo100THMM(after));
            }

            BooleanDefaultTrue contextSpacing = paragraphProperties.getContextualSpacing();
            if(contextSpacing != null)
                jsonParagraphProperties.put("contextualSpacing", contextSpacing.isVal());

            Jc jc = paragraphProperties.getJc();
            if(jc!=null) {
                if (jc.getVal()==JcEnumeration.LEFT)
                    jsonParagraphProperties.put("alignment", "left");
                else if (jc.getVal()==JcEnumeration.CENTER)
                    jsonParagraphProperties.put("alignment", "center");
                else if (jc.getVal()==JcEnumeration.RIGHT)
                    jsonParagraphProperties.put("alignment", "right");
                else if (jc.getVal()==JcEnumeration.DISTRIBUTE)
                    jsonParagraphProperties.put("alignment", "justify");
                else if (jc.getVal()==JcEnumeration.BOTH)
                    jsonParagraphProperties.put("alignment", "justify");
                else    // we do not support quasimodo alignments->defaulting to left
                    jsonParagraphProperties.put("alignment", "left");
            }
            CTShd shd = paragraphProperties.getShd();
            if(shd!=null){
                Utils.jsonPut(
                    jsonParagraphProperties,
                    "fillColor",
                    Utils.createFillColor(shd));
            }
            PPrBase.NumPr numPr = paragraphProperties.getNumPr();
            if(numPr != null){
                PPrBase.NumPr.Ilvl ilvl = numPr.getIlvl();
                PPrBase.NumPr.NumId numId = numPr.getNumId();
                if(ilvl != null)
                    jsonParagraphProperties.put("listLevel", ilvl.getVal().intValue() );
                if(numId != null)
                    jsonParagraphProperties.put("listStyleId", 'L' + Integer.toString(numId.getVal().intValue()) );
            }
            OutlineLvl outlineLvl = paragraphProperties.getOutlineLvl();
            if (outlineLvl != null){
                jsonParagraphProperties.put("outlineLevel", outlineLvl.getVal().intValue() );
            }
            Tabs tabs = paragraphProperties.getTabs();
            if (tabs != null){
                List<CTTabStop> tabList = tabs.getTab();
                if ((tabList != null) && !tabList.isEmpty()){
                    JSONArray jsonTabstopArray = new JSONArray();
                    Iterator<CTTabStop> iter = tabList.iterator();
                    while (iter.hasNext()) {
                        CTTabStop tabstop = iter.next();
                        JSONObject jsonTabstop = new JSONObject();

                        BigInteger posValue = tabstop.getPos();
                        jsonTabstop.put("value", tabstop.getVal().value());
                        jsonTabstop.put("pos", Utils.mapTwipTo100THMM(posValue));
                        if (tabstop.getLeader() != null) {
                            jsonTabstop.put("fillChar", tabstop.getLeader().value());
                        }
                        jsonTabstopArray.put(jsonTabstop);
                    }
                    jsonParagraphProperties.put("tabStops", jsonTabstopArray);
                }
            }
            PBdr pBorder = paragraphProperties.getPBdr();
            if (pBorder != null){
                CTBorder lBorder = pBorder.getLeft();
                if( lBorder != null )
                    jsonParagraphProperties.put("borderLeft", createBorderProperties( lBorder ));
                CTBorder rBorder = pBorder.getRight();
                if( rBorder != null )
                    jsonParagraphProperties.put("borderRight", createBorderProperties( rBorder ));
                CTBorder tBorder = pBorder.getTop();
                if( tBorder != null )
                    jsonParagraphProperties.put("borderTop", createBorderProperties( tBorder ));
                CTBorder bBorder = pBorder.getBottom();
                if( bBorder != null )
                    jsonParagraphProperties.put("borderBottom", createBorderProperties( bBorder ));
                CTBorder iBorder = pBorder.getBetween();
                if( iBorder != null )
                    jsonParagraphProperties.put("borderInside", createBorderProperties( iBorder ));
// not used
//                    CTBorder barBorder = pBorder.getBar();
//                    if( barBorder != null )
//                        jsonParagraphProperties.put("borderbar", createBorderProperties( barBorder ));
            }
            Ind indent = paragraphProperties.getInd();
            if (indent != null){
                BigInteger hanging = indent.getHanging();
                if( hanging != null )
                    jsonParagraphProperties.put("indentFirstLine", - Utils.mapTwipTo100THMM(hanging));
                else{
                    BigInteger firstLine = indent.getFirstLine();
                    if( firstLine != null )
                        jsonParagraphProperties.put("indentFirstLine", Utils.mapTwipTo100THMM(firstLine));
                }
                BigInteger left = indent.getLeft();
                if( left != null )
                    jsonParagraphProperties.put("indentLeft", Utils.mapTwipTo100THMM(left));
                BigInteger right = indent.getRight();
                if( right != null )
                    jsonParagraphProperties.put("indentRight", Utils.mapTwipTo100THMM(right));
            }
        }
        return jsonParagraphProperties!=null&&jsonParagraphProperties.length()>0 ? jsonParagraphProperties : null;
    }
    private static JSONObject createBorderProperties( CTBorder borderLine )
        throws JSONException {

        JSONObject line = new JSONObject();
        Color color = new Color();
        color.setVal(borderLine.getColor());
        color.setThemeColor(borderLine.getThemeColor());
        color.setThemeTint(borderLine.getThemeTint());
        color.setThemeShade(borderLine.getThemeShade());
        Utils.jsonPut(line, "color", Utils.createColor(color));
        BigInteger sz = borderLine.getSz();
        if(sz != null)
            line.put("width", Utils.mapTwipTo100THMM(sz.longValue() * 5 / 2)); // Unit: 1/8 pt
        BigInteger space = borderLine.getSpace();
        if(space != null)
            line.put("space", Utils.mapTwipTo100THMM(space.longValue() * 20)); // Unit: pt
// not used
//            line.put("isframe",borderLine.isFrame());
//            line.put("isshadow", borderLine.isShadow());
        STBorder borderType = borderLine.getVal();
        if(borderType != null){
            line.put("style", borderType.value());
        }
        return line;
    }
}
