/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.realtime.impl;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import javax.annotation.concurrent.NotThreadSafe;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.documentconverter.Feature;
import com.openexchange.documentconverter.IManager;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.attach.AttachmentBase;
import com.openexchange.groupware.attach.Attachments;
import com.openexchange.groupware.ldap.User;
import com.openexchange.java.Streams;
import com.openexchange.mail.MailServletInterface;
import com.openexchange.mail.dataobjects.MailPart;
import com.openexchange.office.FilterException;
import com.openexchange.office.IExporter;
import com.openexchange.office.IImporter;
import com.openexchange.office.calcengine.client.CalcEngineClientFactory;
import com.openexchange.office.calcengine.client.ICalcEngineClient;
import com.openexchange.office.tools.ErrorCode;
import com.openexchange.office.tools.FileHelper;
import com.openexchange.office.tools.Resource;
import com.openexchange.office.tools.ResourceManager;
import com.openexchange.realtime.group.GroupDispatcher;
import com.openexchange.realtime.packet.ID;
import com.openexchange.realtime.packet.Message;
import com.openexchange.realtime.packet.Stanza;
import com.openexchange.realtime.payload.PayloadElement;
import com.openexchange.realtime.payload.PayloadTree;
import com.openexchange.realtime.payload.PayloadTreeNode;
import com.openexchange.realtime.util.ActionHandler;
import com.openexchange.realtime.util.ElementPath;
import com.openexchange.server.ServiceLookup;
import com.openexchange.server.services.ServerServiceRegistry;
import com.openexchange.session.Session;
import com.openexchange.sessiond.SessiondService;
import com.openexchange.timer.ScheduledTimerTask;
import com.openexchange.timer.TimerService;
import com.openexchange.tools.encoding.Base64;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.tools.session.ServerSessionAdapter;

/**
 * {@link Connection}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
final public @NotThreadSafe
class Connection extends GroupDispatcher {

    private static class PrepareLosingEditRightsTimeoutRunnable implements Runnable {

        private final Connection m_connection;
        private final Session m_session;
        private final ID m_fromId;

        public PrepareLosingEditRightsTimeoutRunnable(final Connection connection, Session session, ID fromId) {
            super();
            m_connection = connection;
            m_session = session;
            m_fromId = fromId;
        }

        @Override
        public void run() {
            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: RT handling waiting for answer [canloseeditrights]. Timeout reached, no answer from client: " + m_fromId.toString());
            }

            m_connection.canLoseEditRights(m_session, m_fromId, 0, true);
        }
    }

    private static class JSONDebugWriter extends java.io.Writer  {

        private final StringBuffer m_output;
        private static final int MAX_LENGTH = 1024;
        private int lastWrittenLength = -1;

        public JSONDebugWriter() {
            super();

            m_output = new StringBuffer(512);
        }

        public String getData() {
            return m_output.toString();
        }

        @Override
        public void close() throws IOException {
            // nothing
        }

        @Override
        public void flush() throws IOException {
            // nothing
        }

        @Override
        public void write(char[] cbuf, int off, int len) throws IOException {
            // Limit the amount of data written for an operation.
            // We assume that image data is written in chunks which exceeds
            // our own MAX_LENGTH
            if (lastWrittenLength < MAX_LENGTH) {
                len = Math.min(len, MAX_LENGTH);
                m_output.append(cbuf, off, len);
            } else if (len < MAX_LENGTH) {
                m_output.append(cbuf, off, len);
            }
            lastWrittenLength = len;
        }
    }

    /**
     * Initializes a new {@link Connection}.
     *
     * @param services
     * @param id
     */
    public Connection(ServiceLookup serviceLookup, ID id) {
        super(id, m_handler);

        // initialize ServiceLookup and ResourceManager
        m_services = serviceLookup;
        m_resourceManager = new ResourceManager(m_services);
        m_lastSaveTime = new Date();

        // resource layout is: folderId.fileId
        final String resource = id.getResource();

        if (null != resource) {
            final int fileIdDelimiter = resource.indexOf('.');
            String folderId = null;
            String fileId = null;

            if ((-1 < fileIdDelimiter) && (fileIdDelimiter < (resource.length() - 1))) {
                folderId = resource.substring(0, fileIdDelimiter);
                fileId = resource.substring(fileIdDelimiter + 1);
            }

            if ((null != folderId) && (folderId.length() > 0) && (null != fileId) && (fileId.length() > 0)) {
                m_fileHelper = new FileHelper(m_services, folderId, fileId);
            }
        }

        if (LOG.isDebugEnabled()) {
            if (null != m_fileHelper) {
                LOG.debug("RT connection: Established for resource: " + id.toString());
            } else {
                LOG.debug("RT connection: NOT established for resource: " + id.toString());
            }
        }

    }
    /*
     * (non-Javadoc)
     * @see com.openexchange.realtime.group.GroupDispatcher#firstJoined(com.openexchange.realtime.packet.ID)
     */
    @Override
    protected void firstJoined(ID id) {
        m_threadId = Thread.currentThread().getId();

        if (null != m_fileHelper) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: First client joined RT connection: " + ((null != id) ? id.toString() : "n/a"));
                LOG.debug("RT connection: Thread Id = " + String.valueOf(m_threadId));
            }
        } else if (LOG.isErrorEnabled()) {
            LOG.error("RT connection: Import of operations not possible due to invalid file parameters: " + ((null != id) ? id.toString() : ""));
        }

        firstEditClient = id;
        m_resourceManager.lockResources(true);
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.realtime.group.GroupDispatcher#onJoin(com.openexchange.realtime.packet.ID)
     */
    @Override
    protected void onJoin(ID id) {
        super.onJoin(id);

        if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Client joined: " + ((null != id) ? id.toString() : ""));
        }

        int curClientCount = 0;
        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: onJoin called from different thread " + String.valueOf(threadId));
        }

        synchronized(m_connectionStatus) {
            curClientCount = m_connectionStatus.getActiveClients();
            m_connectionStatus.setActiveClients(++curClientCount);
            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Number of current clients = " + String.valueOf(curClientCount));
                // add new member to member list
                m_memberNames.add(id.toString());
                for (int i = 0; i < m_memberNames.size(); i++) {
                    LOG.debug("RT connection: Member = " + m_memberNames.get(i));
                }
            }
        }

        // Automatically acquire the edit rights for the first client
        // The current state is transferred together with the actions
        // via getWelcomeMessage.
        if ((1 == curClientCount) && (null != id)) {
        	try {
                // No need that acquireEdit sends any update message
                // therefore set silent to true. As stated above
                // the current state is transferred via getWelcomeMessage.
                acquireEdit(null, id, true);
        	} catch (OXException e) {
        		;
        	}
        }
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.realtime.group.GroupDispatcher#onLeave(com.openexchange.realtime.packet.ID)
     */
    @Override
    protected void onLeave(ID id) {
        super.onLeave(id);

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: onLeave called from different thread " + String.valueOf(threadId));
        }

        if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Client " + id.toString() + " leaving => Flushing operations");
        }

        Session session = getServerSession(null, id);


        int curClientCount = 0;
        synchronized(m_connectionStatus) {

            curClientCount = m_connectionStatus.getActiveClients();
            if (curClientCount > 0) {
                m_connectionStatus.setActiveClients(--curClientCount);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: Number of current clients = " + String.valueOf(curClientCount));
                    // remove leaving member to member list
                    m_memberNames.remove(id.toString());
                    for (int i = 0; i < m_memberNames.size(); i++) {
                        LOG.debug("RT connection: Member = " + m_memberNames.get(i));
                    }
                }
            }

	        // Check if the leaving user wants the edit rights. If yes we have
	        // to remove the user to enable others to acquire them.
	        String wantsEditRightsUser = m_connectionStatus.getWantsEditRightsUserId();
	        String currentEditUser = m_connectionStatus.getCurrentEditingUserId();
	        if ((null != wantsEditRightsUser) && (wantsEditRightsUser.length() > 0)) {
                ID wantsEditRightsUserId = new ID(wantsEditRightsUser);
                if (wantsEditRightsUserId.equals(id)) {
                    // Reset wantsEditRightsUser from connectionStatus
                    m_connectionStatus.setWantsEditRightsUser("", "");
                }
	        }
	        if ((null != currentEditUser) && (currentEditUser.length() > 0)) {
                ID currentEditUserId = new ID(currentEditUser);
                if (currentEditUserId.equals(id)) {
                    // Reset current editing user from connectionStatus
                    m_connectionStatus.setCurrentEditingUser("", "");
                }
            }
        }

	    if (curClientCount > 0) {
	        updateClients(new MessageData(null, syncGetConnectionStatus()), null);
	    }

        flushDocument(session, false);
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.realtime.group.GroupDispatcher#onDispose()
     */
    @Override
    protected void onDispose(ID id) throws OXException {
        super.onDispose(id);

        if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Last client leaving => Flushing operations");
        }

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: onDispose called from different thread " + String.valueOf(threadId));
        }

        if( m_calcClient != null ){
            try {
                m_calcClient.destroyDocument(m_calcDocumentHandle);
            } catch (Exception e) {
                LOG.debug("RT connection: Error deleting spreadsheet document");
                LOG.debug(e.getStackTrace());
            }
        }
        flushDocument(getServerSession(null, id), true);

        m_resourceManager.lockResources(false);
        synchronized(m_connectionStatus) {
            cancelPrepareLosingEditRightsTimer();
            m_connectionStatus.setWantsEditRightsUser("", "");
            m_connectionStatus.setActiveClients(0);
        }
    }

    /* (non-Javadoc)
     * @see com.openexchange.realtime.group.GroupDispatcher#getWelcomeMessage(com.openexchange.realtime.packet.ID)
     */
    @Override
    public Stanza getWelcomeMessage(ID onBehalfOf) {
        final ID fromId = getId();
        Message welcomeMessage = null;
        int operationCount = 0;
        JSONObject lastOperation = null;
        JSONObject firstOperation = null;
        boolean opsFromDocument = true;

        if ((null != onBehalfOf) && (isMember(onBehalfOf))) {
            final ServerSession serverSession = getServerSession(null, onBehalfOf);
            final OXDocument oxDocument = new OXDocument(serverSession, m_fileHelper, m_resourceManager);
            final JSONObject jsonResult = new JSONObject();
            final JSONObject operations = new JSONObject();
            int messageChunkListSize = 0;

            // Initialize result object with operations from persistent document first.
            // Check for FilterExceptions that can occur due to problems with the file.
            try {
                ConnectionHelper.appendJSON(operations, oxDocument.getOperations(serverSession, getImporter(), null));
            } catch (final FilterException e) {
                ErrorCode errorCode = ErrorCode.LOADDOCUMENT_CANNOT_RETRIEVE_OPERATIONS_ERROR;
                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: getWelcomeMessage, Exception catched", e);
                }

                if (e.getErrorcode() == FilterException.ErrorCode.WRONG_PASSWORD
                 || e.getErrorcode() == FilterException.ErrorCode.UNSUPPORTED_ENCRYPTION_USED) {
                    errorCode = ErrorCode.LOADDOCUMENT_CANNOT_READ_PASSWORD_PROTECTED_ERROR;
                }

                // add client id
                JSONObject jsonExtraData = new JSONObject();
                try {
                    jsonExtraData.put(MessageData.KEY_CLIENT_ID, onBehalfOf.toString());
                } catch (JSONException jsonEx) {
                    ;
                }
                final MessageData messageData = new MessageData(null, syncGetConnectionStatus(), finalizeJSONResult(
                    errorCode.getAsJSON(),
                    null,
                    jsonExtraData));

                // create error welcome message
                welcomeMessage = new Message();
                welcomeMessage.setFrom(fromId);
                welcomeMessage.setTo(onBehalfOf);
                welcomeMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                    new PayloadElement(messageData, MessageData.class.getName(), "office", "getactions")).build()));

                return welcomeMessage;
            }

            if(m_fileHelper != null && (m_fileHelper.getDocumentType() == com.openexchange.office.tools.FileHelper.DocumentType.XLSX && m_calcDocumentHandle == null && operations.has("operations"))){
                final CalcEngineClientFactory calcFactory = m_services.getService(CalcEngineClientFactory.class);
                try {
                    m_calcClient = calcFactory.get();
                    m_calcDocumentHandle = m_calcClient.createDocument();
                    final int MAX_OPERATIONS = 1000;

                    JSONArray opArray = operations.getJSONArray("operations");
                    String result = null;
                    m_uiOperations = new JSONObject();
                    if(opArray.length() > MAX_OPERATIONS)
                    {
                        int opCount = opArray.length();
                        int copyPosition = 0;
                        JSONArray uiOperations = new JSONArray();

                        while(copyPosition < opCount){
                            JSONObject opPart = new JSONObject();
                            JSONArray partArray = new JSONArray();
                            for(int op = 0; op < MAX_OPERATIONS && copyPosition < opCount;++op){
                                partArray.put(op, opArray.get(copyPosition));
                                ++copyPosition;
                            }
                            opPart.put("operations", partArray);
                            result = m_calcClient.executeOperation(m_calcDocumentHandle, opPart.toString(false));
                            JSONObject resultObject = new JSONObject(result);
                            if(resultObject.has("uiOperations")){
                                try {
                                    JSONArray newUIOps = resultObject.getJSONArray("uiOperations");
                                    for (int i = 0; i < newUIOps.length(); i++) {
                                        uiOperations.put(newUIOps.get(i));
                                    }
                                } catch (JSONException e) {
                                    ;
                                }
                            }
                        }
                        try {
                            m_uiOperations.put("operations", uiOperations);
                        } catch (JSONException e) {
                            ;
                        }

                    }
                    else{
                        result = m_calcClient.executeOperation(m_calcDocumentHandle, operations.toString(false));
                        JSONObject resultObject = new JSONObject(result);
                        if(resultObject.has("uiOperations")){
                            try {
                                m_uiOperations.put("operations", resultObject.getJSONArray("uiOperations"));
                            } catch (JSONException e) {
                                ;
                            }
                        }
                        LOG.info(result);
                    }
                    m_calcClient.executeOperation(m_calcDocumentHandle, "{\"operations\":[{\"name\":\"loadFinished\"}]}");
                } catch (Exception e) {
                    LOG.debug("RT connection: Error executing spreadsheet operations");
                    LOG.debug(e.getStackTrace());
                }
            }

            // Determine the operation state number and append additional operations
            // applied to the document since the last save action.
            synchronized(m_messageChunkList) {
                messageChunkListSize = m_messageChunkList.size();

                if (m_messageChunkList.size() == 0) {
                    // If we don't have additional operations we have to send
                    // the current operation state number for the document. Therefore
                    // calculate or provide the current operation state number.
                    // Otherwise we don't provide the number from here but let
                    // the additional operations send it.
                    try {
                        JSONObject opsObject = (m_uiOperations != null) ? m_uiOperations : operations;
	                    operationCount = opsObject.getJSONArray("operations").length();
	                    lastOperation = opsObject.getJSONArray("operations").getJSONObject(operationCount - 1);
	                    if (! lastOperation.has("osn")) {

	                        if (m_connectionStatus.getOperationStateNumber() > 0) {
	                            lastOperation.put("osn", m_connectionStatus.getOperationStateNumber() - 1);
	                        } else {
	                            lastOperation.put("osn", operationCount - 1);
	                            // Set initial operation state number
	                            m_connectionStatus.setOperationStateNumber(operationCount);
	                        }
	                        lastOperation.put("opl", 1);
	                    }

	                } catch (JSONException e) {
	                    LOG.debug("RT connection: Error setting operation state number to last operation");
	                    LOG.debug(e.getStackTrace());
                    }
                }

                if (m_messageChunkList.size() > 0) {
                    opsFromDocument = false;
                    // Adding operations applied to the document since the last save action.
                    // These operations will provide the current operation state number, too.
                    ConnectionHelper.appendOperationChunks(operations, m_messageChunkList, false);

                    if (LOG.isDebugEnabled()) {
                        // Additional analyzing code to find root cause of missing operations
                        firstOperation = debugGetFirstOperationFromMessageChunkList(m_messageChunkList);
                        lastOperation = debugGetLastOperationFromMessageChunkList(m_messageChunkList);
                    }
                }
            }

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Connection.getWelcomeMessage, message chunk list size = " + Integer.toString(messageChunkListSize));
                if (null != firstOperation) {
                    LOG.debug("RT connection: Connection.getWelcomeMessage, first Operation in messsage chunk list: " + operationToString(firstOperation));
                }
                if (null != lastOperation) {
                    if (opsFromDocument) {
                        LOG.debug("RT connection: Connection.getWelcomeMessage, last Operation in document: " + operationToString(lastOperation));
                    } else {
                        LOG.debug("RT connection: Connection.getWelcomeMessage, last Operation in messsage chunk list: " + operationToString(lastOperation));
                    }
                }
            }

            // set writeProtected status
            m_connectionStatus.setWriteProtected(m_fileHelper.isWriteProtected());
            // set lock state
            m_connectionStatus.setLockState(m_fileHelper.lockedUntil() != null);
            // set locked by user
            m_connectionStatus.setLockedByUser(m_fileHelper.getLockedByUser());

            // add additional client startup data
            final JSONObject jsonExtraData = new JSONObject();

            // add client id
            try {
                jsonExtraData.put(MessageData.KEY_CLIENT_ID, onBehalfOf.toString());
            } catch (JSONException e) {
                ;
            }

            // putting all operations into an 'actions' object
            final JSONObject actionData = new JSONObject();
            try {
                if (m_uiOperations != null) {
                    actionData.put("actions", (new JSONArray()).put(m_uiOperations));
                }
                else
                if (!operations.isEmpty()) {
                    actionData.put("actions", (new JSONArray()).put(operations));
                } else {
                    jsonExtraData.put("hasErrors", true);
                }
            } catch (JSONException e) {
                ;
            }

            final MessageData messageData = new MessageData(new MessageChunk(actionData, getId()), syncGetConnectionStatus(), finalizeJSONResult(
                jsonResult,
                null,
                jsonExtraData));

            // set operations chunk of collected JSON objects as payload
            welcomeMessage = new Message();
            welcomeMessage.setFrom(fromId);
            welcomeMessage.setTo(onBehalfOf);
            welcomeMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(messageData, MessageData.class.getName(), "office", "getactions")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Sending [welcome message] to: " + onBehalfOf.toString());
            }
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't send [welcome message] since the requester was no member: " + ((null != onBehalfOf) ? onBehalfOf.toString() : "null"));
        }

        return welcomeMessage;
    }

    /* (non-Javadoc)
     * @see com.openexchange.realtime.group.GroupDispatcher#getSignOffMessage(com.openexchange.realtime.packet.ID)
     */
    @Override
    public Stanza getSignOffMessage(ID onBehalfOf) {
        final ID fromId = getId();
        Message signoffMessage = null;

        if ((null != onBehalfOf) && (isMember(onBehalfOf))) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: [signoff message] for: " + onBehalfOf.toString());
            }

            final ServerSession serverSession = getServerSession(null, onBehalfOf);
            final JSONObject jsonResult = flushDocument(serverSession, false);

            // send back the JSON result object returned by getDocument
            signoffMessage = new Message();
            signoffMessage.setFrom(fromId);
            signoffMessage.setTo(onBehalfOf);
            signoffMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(jsonResult, null, null), "json", "office", "closedocument")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Sending [signoff message] to: " + onBehalfOf.toString());
            }
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't send [signoff message] since the requester was no member: " + ((null != onBehalfOf) ? onBehalfOf.toString() : "null"));
        }

        return signoffMessage;
    }

    /**
     * Stanzas layout: ApplyActions { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666"
     * element: "message", payloads: [{ element: "action", data: "updateactions" }, { namespace: "office", element: "actions", data:
     * "[{operations: [...]}, ...]" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public Message handleApplyActions(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();
        final String editor = m_connectionStatus.getCurrentEditingUserId();
        final ID editorId = (editor != null) && (editor.length() > 0) ? new ID(editor) : null;
        boolean stanzaFromEditor = (null != editor) && (null != fromId) && (fromId.equals(editorId));
        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleApplyActions called from different thread " + String.valueOf(threadId));
        }

        // Check that we have a ID for the sender stanza and that the sender
        // has the edit rights.
        final Message returnMessage = new Message();
        if ((null != fromId) && isMember(fromId) && stanzaFromEditor) {
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "actions"), true);
            MessageChunk actionChunk = new MessageChunk(jsonRequest, fromId);
            MessageChunk uiActionChunk = null;
            JSONObject jsonResult = new JSONObject();

            // increase number of received actions
            m_nActionsReceived++;

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: [applyActions] received, applied from: " + fromId.toString());
            }

            boolean sendToAll = false;
            // update spreadsheet document
            if( m_calcClient != null && actionChunk.getOperations().has("actions")){
                JSONObject sheetOperations = new JSONObject();
                ConnectionHelper.appendJSON(sheetOperations, actionChunk.getOperations(), false);
                try {
                    String result = m_calcClient.executeOperation(m_calcDocumentHandle, sheetOperations.toString() );
                    // -> the jsonResult from the calc engine has to be added to the message
                    // that will be sent to all clients, including the sender of the operation.
                    sendToAll = true;
                    LOG.info(result);
                    jsonResult = new JSONObject(result);
                        try {
                        if(jsonResult.has("changedOperations")){
                            JSONObject actions = new JSONObject();
                            JSONArray changedOperations = new JSONArray();
                            changedOperations.put(0, jsonResult.getJSONObject("changedOperations"));
                            actions.put("actions", changedOperations);
                            actionChunk = new MessageChunk(actions, actionChunk.getSender());
                            jsonResult.remove("changedOperations");
                        }
                        JSONObject actions = new JSONObject();
                        JSONArray uiActionsArray = new JSONArray();
                        JSONObject uiOperations = new JSONObject();
                        JSONArray sourceOperations = sheetOperations.getJSONArray("operations");
                        JSONObject firstSourceOp = sourceOperations.getJSONObject(0);
                        if(jsonResult.has("uiOperations"))
                        {
                            JSONArray uiOperationsArray = jsonResult.getJSONArray("uiOperations");
                            int osnDiff = sourceOperations.length() - uiOperationsArray.length();
                            int opState = m_connectionStatus.getOperationStateNumber();
                            for(int op = 0; op < uiOperationsArray.length(); ++op)
                            {
                                uiOperationsArray.getJSONObject(op).put("opl", 1).put("osn", opState);
                            }
                            if( osnDiff > 0 ) {
                                uiOperationsArray.getJSONObject(uiOperationsArray.length() - 1).put("opl", osnDiff + 1);
                            }
                            uiOperations.put("operations", uiOperationsArray);
                        }
                        else
                        {
                            //create a dummy operation to align osl/opn
                            int osn = firstSourceOp.getInt("osn");
                            JSONObject dummyOp = new JSONObject();
                            dummyOp.put("name", "noOp");
                            dummyOp.put("osn", osn);
                            dummyOp.put("opl", sourceOperations.length());
                            JSONArray operationsArray = new JSONArray();
                            operationsArray.put(0, dummyOp);
                            uiOperations.put("operations", operationsArray);
                        }
                        uiActionsArray.put(0, uiOperations);

                        actions.put("actions", uiActionsArray);
                        uiActionChunk = new MessageChunk(actions, fromId);
                    } catch (JSONException e) {
                        LOG.debug(e.getStackTrace());
                    }

                } catch (Exception e) {
                    LOG.debug("RT connection: Error executing spreadsheet operation");
                    LOG.debug( e.getStackTrace() );
                }
            }

            // handling operation state number for operations from clients
            try {
                MessageChunk osnActionChunk = uiActionChunk != null ? uiActionChunk : actionChunk;
                if (osnActionChunk.getOperations().hasAndNotNull("actions")) {
                    JSONArray allActions = osnActionChunk.getOperations().getJSONArray("actions");
                    JSONObject lastAction = allActions.getJSONObject(allActions.length() - 1);

                    if (lastAction.has("operations")) {
                        JSONArray allOps = lastAction.getJSONArray("operations");
                        JSONObject lastOp = allOps.getJSONObject(allOps.length() - 1);

                        if (lastOp.has("osn") && lastOp.has("opl")) {
                            m_connectionStatus.setOperationStateNumber(lastOp.getInt("osn") + lastOp.getInt("opl"));
                            if (LOG.isDebugEnabled()) {
                                LOG.debug("RT connection: handleApplyActions, lastOp received = " + operationToString(lastOp));
                                LOG.debug("RT connection: handleApplyActions, current server osn = " + m_connectionStatus.getOperationStateNumber());
                            }
                        }
                    }
                }
            } catch (JSONException e) {
                LOG.debug("RT connection: handleApplyActions: Error setting operation state number to operation");
                LOG.debug(e.getStackTrace());
            }

            // update the shared state
            int msgChunkListSize = 0;
            synchronized(m_messageChunkList) {
                m_messageChunkList.add(actionChunk);
                if(uiActionChunk != null) {
                    ConnectionHelper.appendJSON(m_uiOperations, uiActionChunk.toJSON());
                }
                msgChunkListSize = m_messageChunkList.size();
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: handleApplyActions, message chunk list size = " + String.valueOf(msgChunkListSize));
            }

            final MessageData messageData = new MessageData(uiActionChunk != null ? uiActionChunk : actionChunk, syncGetConnectionStatus(), finalizeJSONResult(
                jsonResult,
                jsonRequest,
                null));

            // Send the message to all participants (excluding the one who sent it originally)
            returnMessage.setFrom(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(messageData, MessageData.class.getName(), "office", "update")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: relaying [applyActions], applied from: " + fromId.toString());
            }

            relayToAllExceptSender(returnMessage, stanza);

            if (sendToAll) {
                // update all Clients with the result, too
                updateClients(new MessageData(null, null, jsonResult), fromId);
                m_nActionsReceived = 0;
            } else if (m_nActionsReceived >= ACTIONS_UNTIL_FORCE_UPDATE) {
                // send update to sender, too
                m_nActionsReceived = 0;

                final Message senderMessage = new Message();
                final MessageData senderMessageData = new MessageData(null, syncGetConnectionStatus(), null);
                senderMessage.setFrom(fromId);
                senderMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                    new PayloadElement(senderMessageData, MessageData.class.getName(), "office", "update")).build()));
                if (stanza.getTracer() != null) {
                    senderMessage.setTracer(stanza.getTracer());
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: Relaying [update] to editor for keep alive, applied from: " + fromId.toString());
                }
                relayToID(senderMessage, fromId);
            }

        } else {
            if (null != fromId && !stanzaFromEditor) {
                // The sender has no edit rights, therefore we send a hang up message
                // so the client will be notified and can reload the document.
                final MessageData messageData = new MessageData(null, syncGetConnectionStatus(), null);

                // Send the message to the sender to hang up
                returnMessage.setFrom(fromId);
                returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                    new PayloadElement(messageData, MessageData.class.getName(), "office", "hangup")).build()));

                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: Relaying [hangup] due to missing edit rights, applied from: " + fromId.toString());
                }
                if (stanza.getTracer() != null) {
                    returnMessage.setTracer(stanza.getTracer());
                }
                relayToID(returnMessage, fromId);
            }

            if (LOG.isDebugEnabled()) {
                if (!isMember(fromId)) {
                    LOG.debug("RT connection: Didn't relay [applyActions] since the requester was no member: " + ((null != fromId) ? fromId.toString() : "null"));
                }
                if (!stanzaFromEditor) {
                    LOG.debug("RT connection: Didn't relay [applyActions] because the requester doesn't have the edit rights");
                }
            }
        }
        return returnMessage;
    }

    /**
     * Stanzas layout: AcquireEdit { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "acquireedit" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleAcquireEdit(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleAcquireEdit called from different thread " + String.valueOf(threadId));
        }

        if ((null != fromId) && isMember(fromId)) {
            final Session serverSession = getServerSession(null, fromId);
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "state"), false);
            int requesterOSN = 0;

            try {
                requesterOSN = jsonRequest.getInt("osn");
            } catch (JSONException e) {
                // handled by following code
            }

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Handling [acquireEdit], called from: " + fromId.toString() + " with client-osn: " + String.valueOf(requesterOSN) + " server-osn: " + String.valueOf(this.m_connectionStatus.getOperationStateNumber()));
            }

            acquireEdit(serverSession, fromId, false);
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't handle [acquireEdit] since the requester was no member: " + ((null != fromId) ? fromId.toString() : "null"));
        }
    }

    /**
     * Stanzas layout: CanLoseEditRights { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "canloseeditrights" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleCanLoseEditRights(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleCanLoseEditRights called from different thread " + String.valueOf(threadId));
        }

        if ((null != fromId) && isMember(fromId)) {
            final Session serverSession = getServerSession(null, fromId);
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "state"), false);
            int requesterOSN = 0;

            try {
                requesterOSN = jsonRequest.getInt("osn");
            } catch (JSONException e) {
                // handled by following code
            }

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Handling [canLoseEditRights], called from: " + fromId.toString() + " with client-osn: " + String.valueOf(requesterOSN) + " server-osn: " + String.valueOf(this.m_connectionStatus.getOperationStateNumber()));
            }

            canLoseEditRights(serverSession, fromId, requesterOSN, false);

        } else if (LOG.isDebugEnabled()) {
            LOG.debug("Didn't handle [canLoseEditRights] since the requester was no member: " + ((null != fromId) ? fromId.toString() : "null"));
        }
    }

    /**
     * Stanzas layout: AddResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "addresource" }, { namespace: "office", element: "resource", data: "{... : ...,}" }]
     * }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleAddResource(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleAddResource called from different thread " + String.valueOf(threadId));
        }

        // don't check for isMember here, since this message was
        // sent as an internal backend call
        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "resource"), false);
            final ServerSession serverSession = getServerSession(jsonRequest, fromId);
            final JSONObject jsonResult = addResource(serverSession, jsonRequest);

            // send back the JSON result object returned by addResource
            returnMessage.setFrom(getId());
            returnMessage.setTo(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(jsonResult, jsonRequest, null), "json", "office", "resource")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Handling [addResource], called from: " + fromId.toString());
            }

            // send the message to the one who requested to add the resource
            send(returnMessage);
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't handle [addResource] since the requester was no member: " + ((null != fromId) ? fromId.toString() : "null"));
        }
    }
    /**
     * Stanzas layout: FlushDocument { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "flushdocument" }, { namespace: "office", element: "document", data: "{... : ...,}" }]
     * }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleFlushDocument(Stanza stanza) throws OXException {
        ID fromId = getIDFromStanza(stanza);

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleFlushDocument called from different thread " + String.valueOf(threadId));
        }

        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "document"), false);
            final Session serverSession = getServerSession(jsonRequest, fromId);
            final JSONObject jsonResult = flushDocument(serverSession, false);

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Handling [flushDocument], called from: " + fromId.toString());
            }

            // send back the JSON result object returned by flushDocument
            returnMessage.setFrom(getId());
            returnMessage.setTo(stanza.getFrom());
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(jsonResult, jsonRequest, null), "json", "office", "flushdocument")).build()));
            send(returnMessage);
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't handle [flushDocument] since the requester was no member: " + ((null != fromId) ? fromId.toString() : "null"));
        }
    }

    private void processUndoRedo(Stanza stanza, boolean isUndo) throws OXException {
        ID fromId = getIDFromStanza(stanza);

        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "document"), false);
            //final Session serverSession = getServerSession(jsonRequest, fromId);
            Stanza applyStanza = new Message();
            applyStanza.setFrom(fromId);
            JSONObject undoAction = new JSONObject();
            JSONArray actions = new JSONArray();
            JSONObject operationObj = new JSONObject();
            JSONArray operations = new JSONArray();
            JSONObject undo = new JSONObject();

            if (LOG.isDebugEnabled()) {
                if (isUndo) {
                    LOG.debug("RT connection: Handling [undo], called from: " + fromId.toString());
                } else {
                    LOG.debug("RT connection: Handling [redo], called from: " + fromId.toString());
                }
            }

            try {
                undo.put("name", isUndo ? "undo" : "redo");
                undo.put("osn", m_connectionStatus.getOperationStateNumber());
                undo.put("opl", 1);
                operations.put(0, undo);
                operationObj.put("operations", operations );

                actions.put(0, operationObj);
                undoAction.put("actions", actions);
            } catch (JSONException e) {
                ;
            }
            applyStanza.addPayload(
                new PayloadTree(PayloadTreeNode.builder().withPayload(
                    new PayloadElement(finalizeJSONResult(undoAction, jsonRequest, null), "json", "office", "actions")).build()
                ));
            Message applyMessage = handleApplyActions(applyStanza);
            JSONObject jsonResult = new JSONObject();

            ElementPath path =  new ElementPath("office", "update");
            for( PayloadTree loadIter : applyMessage.getPayloads(path) ){
                loadIter.getNumberOfNodes();
                Object data = loadIter.getRoot().getData();
                if( data instanceof MessageData ) {
                    jsonResult = ((MessageData)data).toJSON();
                }
            }

            // send back the JSON result object returned by handleApplyActions
            returnMessage.setFrom(getId());
            returnMessage.setTo(stanza.getFrom());
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(jsonResult, jsonRequest, null), "json", "office", "undo")).build()));
            send(returnMessage);
        }
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "undo" }, { namespace: "office", element: "document", data: "{... : ...,}" }]
     * }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleUndo(Stanza stanza) throws OXException {
        processUndoRedo(stanza, true);
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "redo" }, { namespace: "office", element: "document", data: "{... : ...,}" }]
     * }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleRedo(Stanza stanza) throws OXException {
        processUndoRedo(stanza, false);
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "getdocument" }, { namespace: "office", element: "document", data: "{... : ...,}" }]
     * }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleGetDocument(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleGetDocument called from different thread " + String.valueOf(threadId));
        }

        // don't check for isMember here, since this message was
        // sent as an internal backend call
        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "document"), false);
            final ServerSession serverSession = getServerSession(jsonRequest, fromId);
            final JSONObject jsonResult = getDocument(serverSession, jsonRequest);

            // send back the JSON result object returned by getDocument
            returnMessage.setFrom(getId());
            returnMessage.setTo(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(jsonResult, jsonRequest, null), "json", "office", "document")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Handling [getDocument], sending to: " + fromId.toString());
            }

            // send the message to the one who requested to get the document
            send(returnMessage);
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't handle [getDocument] since the requester was no member: " + ((null != fromId) ? fromId.toString() : "null"));
        }
    }

    /**
     * Stanzas layout: RenameDocument { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666"
     * element: "message", payloads: [{ element: "action", data: "renamedocument" }, { namespace: "office", element: "document", data:
     * "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleRenameDocument(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleRenameDocument called from different thread " + String.valueOf(threadId));
        }

        // don't check for isMember here, since this message was
        // sent as an internal backend call
        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "document"), false);
            final Session serverSession = getServerSession(jsonRequest, fromId);
            final JSONObject jsonResult = renameDocument(serverSession, jsonRequest);

            returnMessage.setFrom(getId());
            returnMessage.setTo(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(jsonResult, jsonRequest, null), "json", "office", "document")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Sending [renameDocument] result to: " + stanza.getFrom().toString());
            }

            // always send the message to the one who requested to rename the document
            send(returnMessage);

            // if rename was successful update all clients with the result, too
            try {
                final JSONObject errorCode = jsonResult.getJSONObject(MessageData.KEY_FILE_CHANGERESULT);
                if ((null != errorCode) && (ErrorCode.getErrorCodeFromJSON(errorCode, -1) == 0)) {
                    jsonResult.remove(MessageData.KEY_FILE_CHANGERESULT);

                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Relaying [renameDocument] result, called from: " + stanza.getFrom().toString());
                    }

                    // update all Clients with the result, too
                    updateClients(new MessageData(null, null, jsonResult), null);
                }
            } catch (final JSONException e) {
                ;
            }
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't handle [renameDocument] since the requester was no member: " + ((null != fromId) ? fromId.toString() : "null"));
        }
    }
    /**
     * Stanzas layout: UpdateView { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666"
     * element: "message", payloads: [{ element: "action", data: "updateview" }, { namespace: "office", element: "spreadsheet", data:
     * "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleUpdateView(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleUpdateView called from different thread " + String.valueOf(threadId));
        }

        // don't check for isMember here, since this message was
        // sent as an internal backend call
        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "spreadsheet"), false);
            final Session serverSession = getServerSession(jsonRequest, fromId);
            final JSONObject jsonResult = getSpreadsheetViewUpdate(serverSession, jsonRequest);

            returnMessage.setFrom(getId());
            returnMessage.setTo(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(jsonResult, jsonRequest, null), "json", "office", "spreadsheet")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Sending [updateView] result to: " + stanza.getFrom().toString());
            }

            // send the message to the one who requested to rename the document
            send(returnMessage);

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Relaying [updateView] result, called from: " + stanza.getFrom().toString());
            }

            // update all Clients with the result, too
            updateClients(new MessageData(null, null, jsonResult), null);
        }
    }

    /**
     * Checks if a saving operation for this connection is
     * currently in progress.
     *
     * @return
     *  TRUE if there is a saving in progress otherwise
     *  FALSE.
     */
    public boolean savingDocumentInProgress() {
        return (this.m_saveDocumentInProgress.get() > 0);
    }

    /**
     * Provides the time stamp of the latest save
     * operation.
     *
     * @return
     *  The date of the latest save operation.
     */
    public Date getSaveTimeStamp() {
        return this.m_lastSaveTime;
    }

    /**
     * Provides the number of operations not applied
     * to a saved revision of the document.
     *
     * @return
     *  The number of pending operations (not saved
     *  in the document file yet).
     */
    public long getNumberOfPendingOperations() {
        long num = 0;
        synchronized(m_messageChunkList) {
            num = this.m_messageChunkList.size();
        }
        return num;
    }

    /**
     * Provides the fact if the document has been
     * saved with revision at least once.
     *
     * @return
     *  TRUE if the document was saved with this
     *  connection or FALSE if not.
     */
    public boolean savedWithRevision() {
        return this.m_savedWithRevision;
    }

    /**
     * Tries to flush a modified document
     * @return
     */
    public boolean saveDocumentOnDispose() {
        final String editingUserId = m_connectionStatus.getCurrentEditingUserId();
        boolean modified = false;
        boolean result = false;

        synchronized(m_messageChunkList) {
            modified = !m_messageChunkList.isEmpty();
        }

        if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Save document on dispose initiated");
        }
        // Try to use the session information from the edit user.
        // If a client leaves the document will be saved, therefore
        // there must be an editing user if we have operations in the
        // message chunk list.
        if (modified && (null != editingUserId)) {
            ID userId = null;
            try {
                userId = new ID(editingUserId);
            } catch (Exception e) {
                LOG.error("RT connection: Save document on dispose => no user for saving found!");
                LOG.error(e.getMessage(), e);
            }

            if (null != userId) {
                try {
                    Session session = userId.toSession();
                    flushDocument(session, true);
                } catch (OXException e) {
                    LOG.error("RT connection: Flushing document on dispose resulted in exception!");
                    LOG.error(e.getMessage(), e);
                }
            }
        } else {
            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Save document on dispose - no need to flush document");
            }
        }

        return result;
    }

    /**
     * Make a fail safe revision of the current
     * document. This method is called by a timer thread
     * and therefore must be implemented thread-safe.
     *
     * @param withoutRevision
     *  Specifies if the document should be saved with/without
     *  a new revision. TRUE means WITHOUT a new
     *  revision, FALSE creates a new revision.
     *
     * @return
     *  TRUE if the document has been saved
     *  successfully, otherwise FALSE.
     */
    public boolean failSafeSaveDocument(boolean withoutRevision) {
        final String editingUserId = m_connectionStatus.getCurrentEditingUserId();
        final String revisionless = withoutRevision ? "without new revision" : "";
        boolean result = false;

        if (editingUserId != null) {
            ID userId = new ID(editingUserId);
            Session session = null;

            // retrieve session information from the edit user
            // if there is no editing user we don't need to make a
            // fail safe save
            try {
                session = userId.toSession();
            } catch (OXException e) {
                // nothing TODO
            }

            if (null != session) {

                // make sure we are the only method to trigger the saving process
                if (this.m_saveDocumentInProgress.getAndIncrement() > 0) {
                    // there is a saving in progress so just leave
                    this.m_saveDocumentInProgress.decrementAndGet();
                    return false;
                }

                // close document in every case to write left over operations,
                // if nothing needs to be written, newFileVersion will be the empty string
                try {
                    int messageChunkListSize = 0;
                    JSONObject outputOperation = null;

                    ArrayList<MessageChunk> currentMessages = null;
                    synchronized(m_messageChunkList) {
                        currentMessages = (ArrayList<MessageChunk>)m_messageChunkList.clone();
                    }

                    final int[] errorCode = { MessageData.SAVE_ERROR_NONE };
                    String newFileVersion = null;

                    if (!currentMessages.isEmpty()) {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("RT connection: Fail safe save triggered: Trying to save document " + revisionless);
                        }

                        if (LOG.isDebugEnabled()) {
                            // Additional analyzing code to find root cause of missing operations
                            LOG.debug("RT connection: Fail safe save (before save): Current message chunk list size = " + currentMessages.size());
                            outputOperation = debugGetLastOperationFromMessageChunkList(currentMessages);
                            if (null != outputOperation) {
                                LOG.debug("RT connection: Fail safe save (before save): Last Operation in messsage chunk list = " + operationToString(outputOperation));
                                outputOperation = null;
                            }
                        }

                        final OXDocument oxDocument = new OXDocument(session, m_fileHelper, m_resourceManager);

                        newFileVersion = oxDocument.save(session, getExporter(), m_resourceManager, currentMessages, errorCode, withoutRevision);

                        // clean messageChunkList in case the messageChunkList has been flushed
                        if ((null != newFileVersion) && (newFileVersion.length() > 0)) {

                            int currSize = 0;
                            synchronized(m_messageChunkList) {
                                m_savedWithRevision = !withoutRevision;
                                currSize = m_messageChunkList.size();

                                // Remove all operations which have been saved recently
                                // leaving just the new ones added during the save process.
                                int index = currentMessages.size() - 1;

                                if (index == m_messageChunkList.size() - 1) {
                                    m_messageChunkList.clear(); // optimization
                                } else if ((index > 0) && ((index + 1) < m_messageChunkList.size())) {
                                    // New operation list must only hold the new operations
                                    // therefore we need to remove the old ones.
                                    ArrayList<MessageChunk> copy = (ArrayList<MessageChunk>)m_messageChunkList.clone();
                                    List<MessageChunk> newPart = copy.subList(index + 1, m_messageChunkList.size());
                                    m_messageChunkList.clear();
                                    m_messageChunkList.addAll(newPart);
                                }
                                if (LOG.isDebugEnabled()) {
                                    messageChunkListSize = m_messageChunkList.size();
                                    outputOperation = debugGetFirstOperationFromMessageChunkList(m_messageChunkList);
                                }
                            }
                            m_lastSaveTime = new Date();
                            if (LOG.isDebugEnabled()) {
                                LOG.debug("RT connection: Fail safe save (after save): Before adjustment, message chunk list size = " + currSize);
                                LOG.debug("RT connection: Fail safe save (after save): After adjustment, message chunk list size = " + messageChunkListSize);
                                if (null != outputOperation) {
                                    LOG.debug("RT connection: Fail safe save (after save): First operation in messsage chunk list = " + operationToString(outputOperation));
                                }
                            }
                        } else {
                            if (LOG.isDebugEnabled()) {
                                LOG.debug("RT connection: Fail safe save, documentsave reports no new file version => Possible loss of document data!");
                            }
                        }
                    }

                    if (LOG.isWarnEnabled()) {
                        if ((MessageData.SAVE_ERROR_NOBACKUP == errorCode[0])) {
                            LOG.warn("RT connection: Fail safe " + revisionless + ": Save document operations failed completely!");
                        } else if ((MessageData.SAVE_ERROR_OPERATIONS == errorCode[0])) {
                            LOG.warn("RT connection: Fail safe " + revisionless + ": Save to native document format failed => OX document replacement archive has been written!");
                        }
                    }

                    if (LOG.isDebugEnabled()) {
                        if ((null != newFileVersion) && (newFileVersion.length() > 0)) {
                            LOG.debug("RT connection: Fail safe " + revisionless + ": Flushing operations to native document succeeded");
                        } else {
                            LOG.debug("RT connection: Fail safe " + revisionless + ": No need to save document");
                        }
                    }

                    result = true;
                } catch (Exception e) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Exception detected while doing a fail safe save", e);
                    }
                } finally {
                    m_saveDocumentInProgress.decrementAndGet();
                }
            }
        }

        if (result) {
            // Trigger update message so clients can update files view
            m_connectionStatus.setFailSafeSaveDone(true);
            updateClients(new MessageData(null, syncGetConnectionStatus()), null);
            m_connectionStatus.setFailSafeSaveDone(false);
        }

        return result;
    }

    /**
     * @param messageData
     * @param fromId
     */
    private void updateClients(MessageData messageData, ID fromId) {
        final Message updateMessage = new Message();
        final ID senderId = (null != fromId) ? fromId : getId();

        if (null != senderId) {
            updateMessage.setFrom(senderId);
            updateMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(messageData, MessageData.class.getName(), "office", "update")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Handling [updateClients], originated by: " + senderId.toString());
            }

            try {
                relayToAll(updateMessage);
            } catch (OXException e) {
                ;
            }
        }
    }

    /**
     * Internal method to process "acquireeditrights" from a client.
     *
     * @param {ID} fromId
     */
    private void acquireEdit(Session session, ID fromId, boolean silent) throws OXException {
        Session curSession = session;
        String displayUserName = "";

        if ((null == curSession) && (null != fromId)) {
            try {
                curSession = fromId.toSession();
            } catch (OXException e) {
                curSession = null;
            }
        }

        if ((null != curSession) && (curSession instanceof ServerSession)) {
            final User curUser = ((ServerSession) curSession).getUser();

            if (null != curUser) {
                displayUserName = curUser.getDisplayName();
            }
        }

        if (null != fromId) {

            String wantsEditRightsUserId = "";
            String hasEditRightsUserId = "";

            synchronized(m_connectionStatus) {
                wantsEditRightsUserId = m_connectionStatus.getWantsEditRightsUserId();
                hasEditRightsUserId = m_connectionStatus.getCurrentEditingUserId();
            }

            // Check if user tries to acquire edit rights although she has it already.
            // Can be related to a slow connection and the user tries again and again
            // to acquire the rights.
            if ((null != hasEditRightsUserId) && (hasEditRightsUserId.length() > 0)) {
                ID currentEditorId = new ID(hasEditRightsUserId);
                if (fromId.equals(currentEditorId)) {
                    return;
                }
            }

            if ((null == hasEditRightsUserId) || (hasEditRightsUserId.length() == 0)) {
                ConnectionStatus connectionStatus = null;
                synchronized(m_connectionStatus) {
                    // first acquire edit rights on a non-acquired document can be directly approved
                    m_connectionStatus.setCurrentEditingUser(fromId.toString(), displayUserName);
                    connectionStatus = (ConnectionStatus)m_connectionStatus.clone();
                }
                if (!silent) {
                    updateClients(new MessageData(null,connectionStatus), null);
                }
            } else if ((null == wantsEditRightsUserId) || (wantsEditRightsUserId.length() == 0)) {
                ID currentEditorId = new ID(hasEditRightsUserId);
                Message returnMessageLose = null;
                Message returnMessageGet = null;
                MessageData messageData = null;

                synchronized(m_connectionStatus) {
                    // Start "preparelosingeditrights" timeout timer to prevent that we have a document
                    // where the edit rights are pending forever (e.g. the current editor loses connection
                    // while we are switching the edit rights - waiting for canLoseEditRights)
                    cancelPrepareLosingEditRightsTimer();
                    startPrepareLosingEditRightsTimer(session, currentEditorId);

                    // store user who wants to receive the edit rights
                    m_connectionStatus.setWantsEditRightsUser(fromId.toString(), displayUserName);

                    if (!silent) {
                        // Send the message to the client who will lose the edit rights
                        returnMessageLose = new Message();
                        returnMessageGet = new Message();
                        messageData = new MessageData(null, (ConnectionStatus)m_connectionStatus.clone(), null);
                    }
                }

                if (!silent) {
                    // send info notification to the client which will receive the edit rights
                    returnMessageGet.setFrom(fromId);
                    returnMessageGet.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                        new PayloadElement(messageData, MessageData.class.getName(), "office", "acceptedacquireeditrights")).build()));

                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Relaying [acceptedacquireeditrights] to client requested edit rights, applied from: " + fromId.toString());
                    }

                    relayToID(returnMessageGet, fromId);

                    // send message to the client which will lose the edit rights for preparation
                    returnMessageLose.setFrom(currentEditorId);
                    returnMessageLose.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                        new PayloadElement(messageData, MessageData.class.getName(), "office", "preparelosingeditrights")).build()));

                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Relaying [preparelosingeditrights] to client with edit rights, to: " + currentEditorId.toString());
                    }

                    relayToID(returnMessageLose, currentEditorId);
                }
            } else {
                ID wantsEditorRightsId = new ID(wantsEditRightsUserId);

                if (!fromId.equals(wantsEditorRightsId) && !silent) {
                    // Someone else already wants to acquire the edit rights. Therefore we
                    // have to decline this request until the edit rights changed.

                    final Message returnMessage = new Message();
                    final MessageData messageData = new MessageData(null, syncGetConnectionStatus(), null);

                    returnMessage.setFrom(fromId);
                    returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                        new PayloadElement(messageData, MessageData.class.getName(), "office", "declinedacquireeditrights")).build()));

                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Relaying [declinedacquireeditrights] to client requested edit rights while there is a switch in progress, applied from: " + fromId.toString());
                    }

                    relayToID(returnMessage, fromId);
                }
        	}
        }
    }

    /**
     * Implementation to handle the response from the current editor
     * client to give up the edit rights.
     * This method is called by the "preparelosingeditrights" timer thread
     * and the RT core. Therefore it must be thread-safe implemented.
     *
     * @param {Session} session
     *  The session of the user who was notified via "preparelosingeditrights"
     *
     * @param {ID} fromId
     *  The id who was notified via "preparelosingeditrights".
     *
     * @param {int} clientOSN
     *  The osn of the client responding to the "preparelosingeditrights" notification.
     *
     * @param {boolean} timeout
     *  Specifies that the method is called from the timeout callback.
     */
    private void canLoseEditRights(Session session, ID fromId, int clientOSN, boolean timeout) {

        ConnectionStatus statusToSend = null;
        int serverOSN = clientOSN;
        ID currentEditorId = null;

        synchronized(m_connectionStatus) {

            currentEditorId = new ID(this.m_connectionStatus.getCurrentEditingUserId());
            if (currentEditorId.equals(fromId)) {
                String newEditorId = m_connectionStatus.getWantsEditRightsUserId();

                this.cancelPrepareLosingEditRightsTimer();

                // Current editor completed to send her latest changes. It's now safe
                // to switch the edit rights.
                m_connectionStatus.setCurrentEditingUser(newEditorId, m_connectionStatus.getWantsEditRightsUserName());
                // Remove the requester so others can acquire the edit rights
                m_connectionStatus.setWantsEditRightsUser("", "");
                // Create copy of the connection status we can use updateClients
                // outside our synchronized and have a consistent state
                statusToSend = (ConnectionStatus)m_connectionStatus.clone();
                serverOSN = m_connectionStatus.getOperationStateNumber();

	            if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: Switching edit rights to client: " + newEditorId);
                }
            }
        }

        if (statusToSend != null) {
            updateClients(new MessageData(null, statusToSend), null);
        }

        // Check the current server osn and determine if the client losing
        // edit rights is out of sync. Do this only if we don't have a timeout
        // scenario.
        if (!timeout && (serverOSN != clientOSN) && (null != currentEditorId)) {

            try {
                final Message returnMessage = new Message();
                final MessageData messageData = new MessageData(null, statusToSend);

                // Send the message to the sender to hang up
                returnMessage.setFrom(fromId);
                returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                    new PayloadElement(messageData, MessageData.class.getName(), "office", "hangup")).build()));

                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: Relaying [hangup] to client that has recently lost edit rights due to inconsitent osn, called from: " + fromId.toString() + " with client-osn: " + String.valueOf(clientOSN) + " server-osn: " + String.valueOf(this.m_connectionStatus.getOperationStateNumber()));
                }

                relayToID(returnMessage, currentEditorId);
            } catch (OXException e) {
                // Nothing to do. Sending the message here is just a faster way
                // to provide this information to the related client.
            }
        }
    }

    /**
     * @param session
     * @param jsonRequest
     * @return
     */
    private JSONObject addResource(Session session, JSONObject jsonRequest) {
        JSONObject jsonResult = new JSONObject();

        if ((null != session) && (null != jsonRequest)) {
            final String managedResourceId = jsonRequest.optString("resourceid", "");

            // we got a file id to create a distributed managed file resource from
            if (managedResourceId.length() > 0) {
                final Resource managedResource = m_resourceManager.createManagedResource(managedResourceId);

                if (null != managedResource) {
                    m_resourceManager.addResource(managedResource);

                    try {
                        jsonResult.put("resourceid", managedResource.getManagedId());
                    } catch (JSONException e) {
                        if (LOG.isErrorEnabled()) {
                            LOG.error("RT connection: addResource - could not retrieve managed resource id");
                        }
                    }
                } else {
                    if (LOG.isErrorEnabled()) {
                        LOG.error("RT connection: addResource - managed resource could not be retrieved from id: " + managedResourceId);
                    }
                }
            } else {
                // try to create the resource from the given data
                final String data64 = jsonRequest.optString("dataurl", "");
                byte[] data = null;

                if (null != data64) {
                    final String searchPattern = "base64,";
                    int pos = data64.indexOf(searchPattern);

                    if ((-1 != pos) && ((pos += searchPattern.length()) < (data64.length() - 1))) {
                        data = Base64.decode(data64.substring(pos));
                    }
                }

                if ((data != null) && (data.length > 0)) {
                    final String fileName = jsonRequest.optString("filename", "");
                    final String fileExtension = FileHelper.getExtension(fileName);

                    String newExtension = jsonRequest.optString("add_ext", "");

                    // extension to add to resource fileName
                    if (newExtension.length() < 1) {
                        newExtension = fileExtension;
                    }

                    // crc32 of resource data
                    int crc32 = jsonRequest.optInt("add_crc32", 0);

                    if (0 == crc32) {
                        crc32 = Resource.getCRC32(data);
                    }

                    // resource id
                    final long uid = m_resourceManager.addResource(crc32, data);

                    try {
                        String resourceName = null;

                        switch (m_fileHelper.getDocumentType()) {
                        case DOCX: {
                            resourceName = "word/media/uid" + Long.toHexString(uid) + "." + newExtension;
                            break;
                        }
                        case PPTX: {
                            resourceName = "ppt/media/uid" + Long.toHexString(uid) + "." + newExtension;
                            break;
                        }
                        case XLSX: {
                            resourceName = "xl/media/uid" + Long.toHexString(uid) + "." + newExtension;
                            break;
                        }
                        case ODT: {
                            resourceName = "Pictures/uid" + Long.toHexString(uid) + "." + newExtension;
                            break;
                        }
                        case ODS: {
                            resourceName = "Pictures/uid" + Long.toHexString(uid) + "." + newExtension;
                            break;
                        }

                        default: {
                            break;
                        }
                        }

                        if (null != resourceName) {
                            jsonResult.put("added_filename", resourceName);
                        }
                    } catch (Exception e) {
                        ;
                    }
                }
            }
        }

        return checkJSONResult(jsonResult);
    }

    /**
     * @param session
     * @param jsonRequest
     * @return
     */
    private JSONObject getDocument(ServerSession session, JSONObject jsonRequest) {
        JSONObject jsonResult = new JSONObject();

        if ((null != session) && (null != jsonRequest)) {
            final String documentFormat = jsonRequest.optString("documentformat", "native"); // ["native", "pdf"]
            final String returnType = jsonRequest.optString("returntype", "dataurl"); // ["dataurl", "resourceid"]
            final String source = jsonRequest.optString("source", "file"); //
            final String folder = jsonRequest.optString("folder");
            final String id  = jsonRequest.optString("id");
            final String module = jsonRequest.optString("module");
            final String attachment = jsonRequest.optString("attachment");

            final int[] errorCode = { MessageData.SAVE_ERROR_NONE };
            ArrayList<MessageChunk> currentMessages = null;

            synchronized(m_messageChunkList) {
                currentMessages = (ArrayList<MessageChunk>)m_messageChunkList.clone();
            }
            InputStream documentStm = null;

            MailServletInterface mailInterface = null;
            try {
                if (source.equals("mail")) {
                    try {
                        mailInterface = MailServletInterface.getInstance(session);
                        MailPart part = mailInterface.getMessageAttachment(folder, id, attachment, false);
                        if (part != null) {
                            documentStm = part.getInputStream();
                        }
                    } catch (final Exception e) {
                        documentStm = null;
                    }
                } else if (source.equals("task")) {
                    try {
                        final AttachmentBase ATTACHMENT_BASE = Attachments.getInstance();
                        documentStm = ATTACHMENT_BASE.getAttachedFile(session, Integer.parseInt(folder), Integer.parseInt(attachment), Integer.parseInt(module), Integer.parseInt(id), session.getContext(), session.getUser(), session.getUserConfiguration());
                    } catch (final Exception e) {
                        documentStm = null;
                    }
                } else {
                    final OXDocument oxDocument = new OXDocument(session, m_fileHelper, m_resourceManager);
                    documentStm = oxDocument.getResolvedDocumentStream(session, getExporter(), m_resourceManager, currentMessages, null, errorCode);
                }
                // convert document stream content to PDF, if requested
                if (null != documentStm) {
                    String fileName = jsonRequest.optString("filename", "document");
                    String mimeType = jsonRequest.optString("mimetype", "");

                    if (documentFormat.equals("pdf") && (MessageData.SAVE_ERROR_NONE == errorCode[0])) {
                        final IManager dcManager = m_services.getService(IManager.class);

                        // convert content of document stream to PDF
                        if ((null != dcManager) && dcManager.hasFeature(Feature.DOCUMENTCONVERTER)) {
                            final HashMap<String, Object> jobProperties = new HashMap<String, Object>(8);
                            final HashMap<String, Object> resultProperties = new HashMap<String, Object>(8);

                            jobProperties.put("InputStream", documentStm);
                            jobProperties.put("InfoFilename", fileName);
                            final InputStream pdfDocumentStm = dcManager.convert("pdf", jobProperties, resultProperties);

                            try {
                                documentStm.close();
                            } catch (IOException e) {
                                ;
                            } finally {
                                // set filename and mimetype of conversion result
                                documentStm = pdfDocumentStm;
                                fileName = fileName + ".pdf";
                                mimeType = "application/pdf";
                            }
                        }
                    }

                    if (mimeType.length() < 1) {
                        mimeType = FileHelper.getMimeType(fileName);
                    }

                    // set resulting data at JSONObject; document
                    // stream might be null after a conversion error
                    if (null != documentStm) {
                        // add resourceid or dataurl to the result object, depending on the value of returnType
                        ConnectionHelper.appendData(m_resourceManager, jsonResult, documentStm, returnType, mimeType);

                        // add filename and mimetype properties to the valid result object
                        if (!jsonResult.isEmpty()) {
                            try {
                                jsonResult.put("resourcename", fileName);
                                jsonResult.put("mimetype", mimeType);
                            } catch (JSONException e) {
                                ;
                            }
                        }

                        // close document stream
                        try {
                            documentStm.close();
                        } catch (IOException e) {
                            ;
                        }
                    }
                }
            } finally {
                Streams.close(mailInterface);
            }
        }

        return checkJSONResult(jsonResult);
    }

    /**
     * @return
     */
    private JSONObject renameDocument(Session session, JSONObject jsonRequest) {
        JSONObject jsonResult = new JSONObject();
        String requestedNewFileName = "";

        try {
            String newFileName = null;
            final String[] newFileVersion = { "" };
            final ErrorCode errorCode[] = { ErrorCode.NO_ERROR };

            if ((null != session) && (null != jsonRequest) && (null != m_fileHelper) && jsonRequest.has(MessageData.KEY_FILE_NAME)) {
                requestedNewFileName = jsonRequest.getString(MessageData.KEY_FILE_NAME);
                newFileName = m_fileHelper.renameDocument(session, requestedNewFileName, newFileVersion, errorCode);
            }

            if (null != newFileName) {
                jsonResult.put(MessageData.KEY_FILE_NAME, newFileName);
                jsonResult.put(MessageData.KEY_FILE_VERSION, newFileVersion[0]);
                jsonResult.put(MessageData.KEY_FILE_CHANGERESULT, errorCode[0].getAsJSON());
            } else {
                // In case of an error set at least the change result entry
                jsonResult.put(MessageData.KEY_FILE_CHANGERESULT, errorCode[0].getAsJSON());
            }
        } catch (JSONException e) {
            if (LOG.isWarnEnabled()) {
                LOG.warn("RT connection: renameDocument - failed to rename document to " + requestedNewFileName, e);
            }
            // Fix for bug 29261: Don't return an empty jsonResult which results in a response
            // where the property "hasErrors" is set to "true". This forces the client to immediately
            // show an error bubble and stop accepting server messages.
            try {
                // provide change result to the client
                jsonResult.put(MessageData.KEY_FILE_CHANGERESULT, ErrorCode.RENAMEDOCUMENT_FAILED_ERROR.getAsJSON());
            } catch (JSONException je) {
                ;
            }
        }

        return checkJSONResult(jsonResult);
    }
    /**
     * @return
     */
    private JSONObject getSpreadsheetViewUpdate(Session session, JSONObject jsonRequest) {
        JSONObject updateViewOperation = createUpdateViewOperation(jsonRequest);
        JSONObject resultObject = null;
        try {
            String result = m_calcClient.executeOperation(m_calcDocumentHandle, updateViewOperation.toString());
            resultObject = new JSONObject(result);
        } catch (Exception e) {
            LOG.debug(e.getStackTrace());
        }

        return checkJSONResult(resultObject);
    }
    private JSONObject createUpdateViewOperation(JSONObject requestData) {
        final JSONArray operationsArray = new JSONArray();
        JSONObject operations = new JSONObject();
        try {
            requestData.put("name", "updateView");
            LOG.info(requestData.toString());
            operationsArray.put(requestData);
            operations.put("operations", operationsArray);
        } catch (JSONException e) {
          e.printStackTrace();
        }
        return operations;
    }
    /**
     * @return The Importer for the current file helper object
     */
    private IImporter getImporter() {
        IImporter importer = null;

        if (null != m_fileHelper) {
            switch (m_fileHelper.getDocumentType()) {
            case DOCX: {
                importer = m_services.getService(com.openexchange.office.ooxml.docx.Importer.class);
                break;
            }

            case PPTX: {
                importer = m_services.getService(com.openexchange.office.ooxml.pptx.Importer.class);
                break;
            }

            case XLSX: {
                importer = m_services.getService(com.openexchange.office.ooxml.xlsx.Importer.class);
                break;
            }

            case ODT: case ODS: {
                importer = m_services.getService(com.openexchange.office.odf.Importer.class);
                break;
            }

            default: {
                break;
            }
            }
        }

        return importer;
    }

    /**
     * @return The Exporter for the current file helper object
     */
    private IExporter getExporter() {
        IExporter exporter = null;

        if (null != m_fileHelper) {
            switch (m_fileHelper.getDocumentType()) {

            case DOCX: {
                exporter = m_services.getService(com.openexchange.office.ooxml.docx.Exporter.class);
                break;
            }

            case PPTX: {
                exporter = m_services.getService(com.openexchange.office.ooxml.pptx.Exporter.class);
                break;
            }

            case XLSX: {
                exporter = m_services.getService(com.openexchange.office.ooxml.xlsx.Exporter.class);
                break;
            }

            case ODT: case ODS: {
                exporter = m_services.getService(com.openexchange.office.odf.Exporter.class);
                break;
            }

            default: {
                break;
            }
            }
        }

        return exporter;
    }

    /**
     * Provides thread-safe the current connection status
     * as a shallow clone.
     *
     * @return
     *  The current connection status as a shallow clone.
     */
    private ConnectionStatus syncGetConnectionStatus() {
        synchronized(m_connectionStatus) {
            return (ConnectionStatus)m_connectionStatus.clone();
        }
    }

    /**
     * @param stanza
     * @param elementPath
     * @param keepActions
     * @return
     */
    static private JSONObject getJSONRequest(Stanza stanza, ElementPath elementPath, boolean keepActions) {
        final JSONObject jsonRequest = new JSONObject();

        // We're iterating over all messages that are constructed with the office.operations element path
        for (final PayloadTree payload : stanza.getPayloads(elementPath)) {
            final Object data = payload.getRoot().getData();

            if (data instanceof JSONArray) {
                final JSONArray jsonArray = (JSONArray) data;
                final JSONObject jsonObject = new JSONObject();

                try {
                    jsonObject.put("actions", jsonArray);
                } catch (JSONException e) {
                    ;
                }

                ConnectionHelper.appendJSON(jsonRequest, jsonObject, keepActions);
            } else if (data instanceof JSONObject) {
                ConnectionHelper.appendJSON(jsonRequest, (JSONObject) data, keepActions);
            }
        }

        return jsonRequest;
    }

    /**
     * @param jsonResult
     * @param jsonRequest
     * @param jsonExtra
     * @return
     */
    static private JSONObject finalizeJSONResult(JSONObject jsonResult, JSONObject jsonRequest, JSONObject jsonExtra) {
        if (null != jsonResult) {
            // add unique id element from JSON requests to JSON result
            if (null != jsonRequest) {
                try {
                    // add unique message id, if given in request
                    if (jsonRequest.has(MessageData.KEY_UNIQUE_ID)) {
                        jsonResult.put(MessageData.KEY_UNIQUE_ID, jsonRequest.get(MessageData.KEY_UNIQUE_ID));
                    }
                } catch (JSONException e) {
                    ;
                }
            }

            // add extra data, if set
            if (null != jsonExtra) {
                ConnectionHelper.appendJSON(jsonResult, jsonExtra);
            }
        }

        return jsonResult;
    }

    /**
     * @param jsonResult
     */
    static private JSONObject checkJSONResult(JSONObject jsonResult) {
        if ((null != jsonResult) && jsonResult.isEmpty()) {
            try {
                jsonResult.put("hasErrors", true);
            } catch (JSONException e) {
                ;
            }
        }

        return jsonResult;
    }

    /**
     * Retrieve ID from stanza.
     *
     * @param {Stanza} stanza
     * @return {ID}
     *  The id of the user who send the request. If the
     *  request was done synchronously this is the "real"
     *  ID and not the synthetic.
     *  ATTENTION: In that case the fromID must be used
     *  for the answer message.setTo() method!
     */
    static private ID getIDFromStanza(Stanza stanza) {
        ID id = null;
        if (null != stanza) {
            id = stanza.getOnBehalfOf();
            if (null == id) {
                id = stanza.getFrom();
            }
        }

        return id;
    }

    /**
     * @param jsonRequest
     * @return
     */
    static private ServerSession getServerSession(JSONObject jsonRequest, ID sessionId) {
        ServerSession serverSession = null;

        // try to get session from json request
        if (null != jsonRequest) {
            try {
                if (jsonRequest.has("session")) {
                    final JSONObject jsonSession = jsonRequest.getJSONObject("session");
                    if (null != jsonSession) {
                        final Session session = ServerServiceRegistry.getInstance().getService(SessiondService.class, true).getSession(jsonSession.getString("resource"));
                        serverSession = ServerSessionAdapter.valueOf(session);
                    }
                }
            } catch (Exception e) {
                ;
            }
        }

        // try to get session from ID
        if ((null == serverSession)) {
            try {
                serverSession = (null != sessionId) ? sessionId.toSession() : null;
            } catch (Exception e) {
                if (LOG.isErrorEnabled()) {
                    LOG.error("RT connection: Message handler not able to get server session!");
                }
            }
        }

        return serverSession;
    }

    /**
     * @param id
     */
    private JSONObject flushDocument(Session session, boolean forceSave) {
        final JSONObject jsonResult = new JSONObject();

        if (null != session) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: flushDocument called");
            }
            // We only want to have just ONE save document process with
            // only ONE exception: if forceSave is set (only if the connection
            // is going to be disposed) we have to write the latest state.
            int saveCount = this.m_saveDocumentInProgress.getAndIncrement();
            if ((saveCount > 0) && !forceSave) {
                this.m_saveDocumentInProgress.getAndDecrement();
                return jsonResult;
            }

            // close document in every case to write left over operations,
            // if nothing needs to be written, newFileVersion will be the empty string
            try {
                final int[] errorCode = { MessageData.SAVE_ERROR_NONE };
                String newFileVersion = null;
                String oldFileVersion = null;
                ArrayList<MessageChunk> currentMessages = null;
                int osn = -1;
                int currSize = 0;

                synchronized(m_messageChunkList) {
                    if (!m_messageChunkList.isEmpty()) {
                        currentMessages = (ArrayList<MessageChunk>)m_messageChunkList.clone();
                    }
                    osn = m_connectionStatus.getOperationStateNumber();
                }

                if (currentMessages != null && !currentMessages.isEmpty()) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Flushing operations, current message chunk list size = " + currentMessages.size());
                        LOG.debug("RT connection: Flushing operations, current server osn = " + String.valueOf(osn));
                        JSONObject outputOperation = this.debugGetLastOperationFromMessageChunkList(currentMessages);
                        if (null != outputOperation) {
                            LOG.debug("RT connection: Flushing operations (before save): last operation in current message chunk list = " + operationToString(outputOperation));
                        }
                    }

                    oldFileVersion = m_fileHelper.getVersionFromFile(session);
                    final OXDocument oxDocument = new OXDocument(session, m_fileHelper, m_resourceManager);

                    newFileVersion = oxDocument.save(session, getExporter(), m_resourceManager, currentMessages, errorCode, false);

                    // clean messageChunkList in case the messageChunkList has been flushed
                    if ((null != newFileVersion) && (newFileVersion.length() > 0)) {
                        int messageChunkListSize = 0;
                        JSONObject outputOperation = null;

                        synchronized(m_messageChunkList) {
                            m_savedWithRevision = true;
                            currSize = m_messageChunkList.size();

                            // Remove all operations which have been saved recently
                            // leaving just the new ones added during the save process.
                            int index = currentMessages.size() - 1;

                            if (index == m_messageChunkList.size() - 1) {
                                m_messageChunkList.clear(); // optimization
                            } else if ((index > 0) && ((index + 1) < m_messageChunkList.size())) {
                                // New operation list must only hold the new operations
                                // therefore we need to remove the old ones.
                                ArrayList<MessageChunk> copy = (ArrayList<MessageChunk>)m_messageChunkList.clone();
                                List<MessageChunk> newPart = copy.subList(index + 1, m_messageChunkList.size());
                                m_messageChunkList.clear();
                                m_messageChunkList.addAll(newPart);
                            }
                            if (LOG.isDebugEnabled()) {
                                messageChunkListSize = m_messageChunkList.size();
                                outputOperation = debugGetFirstOperationFromMessageChunkList(m_messageChunkList);
                            }
                            m_lastSaveTime = new Date();
                            osn = m_connectionStatus.getOperationStateNumber();
                        }

                        if (LOG.isDebugEnabled()) {
                            LOG.debug("RT connection: Flushing operations completed.");
                            LOG.debug("RT connection: Flushing operations, current server osn = " + String.valueOf(osn));
                            LOG.debug("RT connection: Flushing operations (after save): Before adjustment, message chunk list size = " + currSize);
                            LOG.debug("RT connection: Flushing operations (after save): After adjustment, message chunk list size = " + messageChunkListSize);
                            if (null != outputOperation) {
                                LOG.debug("RT connection: Flushing operation (after save): First operation in messsage chunk list = " + operationToString(outputOperation));
                            }
                        }

                        if ((null != oldFileVersion) && (oldFileVersion.equals(newFileVersion))) {
                            // logical check: a successful revision save must create a new file version
                            if (LOG.isWarnEnabled()) {
                                LOG.warn("RT connection: Flushing operations successful but file version didn't change => Possible loss of document data!");
                            }
                        }
                    } else {
                        if (LOG.isWarnEnabled()) {
                            LOG.warn("RT connection: Flushing operations reports no new file version => Possible loss of document data!");
                        }
                    }
                }

                try {
                    // set JSON result error code and new file version
                    jsonResult.put(MessageData.KEY_ERROR_CODE, errorCode[0]);

                    if (null != newFileVersion) {
                        jsonResult.put(MessageData.KEY_FILE_VERSION, newFileVersion);
                    }
                } catch (JSONException e) {
                    ;
                }

                if (LOG.isWarnEnabled()) {
                    if ((MessageData.SAVE_ERROR_NOBACKUP == errorCode[0])) {
                        LOG.warn("RT connection: Flushing operations failed completely => Possible loss of document data!");
                    } else if ((MessageData.SAVE_ERROR_OPERATIONS == errorCode[0])) {
                        LOG.warn("RT connection: Flushing operations to native document format failed => OX document replacement archive has been written!");
                    }
                }

                if (LOG.isDebugEnabled()) {
                    if ((null != newFileVersion) && (newFileVersion.length() > 0)) {
                        LOG.info("RT connection: Flushing operations to native document succeeded");
                    } else {
                        LOG.info("RT connection: Closing document succeeded without needing to flush operations");
                    }
                }
            } catch (Exception e) {
                LOG.error("RT connection: Exception while flushing operations to document!");
                LOG.error(e.getMessage(), e);
            } finally {
                m_saveDocumentInProgress.decrementAndGet();
            }
        }

        return jsonResult;
    }

    /**
     * Start a new "preparelosingeditrights" timer.
     *
     * @param {Session} session
     * @param {ID} id
     */
    private void startPrepareLosingEditRightsTimer(Session session, ID id) {
        TimerService timerService = m_services.getService(TimerService.class);

        if (timerService != null) {
            m_prepareLosingEditRightsTimer = timerService.schedule(
                new PrepareLosingEditRightsTimeoutRunnable(this, session, id), 20000);
        }
    }

    /**
     * Cancel a running "preparelosingeditrights" timer
     */
    private void cancelPrepareLosingEditRightsTimer() {
		// Cancel possible timeout timer
		if (m_prepareLosingEditRightsTimer != null) {
			m_prepareLosingEditRightsTimer.cancel();
			m_prepareLosingEditRightsTimer = null;
		}
    }

    private JSONObject debugGetFirstOperationFromMessageChunkList(ArrayList<MessageChunk> chunkList) {
        JSONObject result = null;

        if (chunkList.size() > 0) {
            MessageChunk lastChunk = chunkList.get(0);
            JSONObject opsObject = lastChunk.getOperations();
            try {
                JSONArray actions = opsObject.getJSONArray("actions");
                if (actions.length() > 0) {
                    opsObject = actions.getJSONObject(0);
                    int operationCount = opsObject.getJSONArray("operations").length();
                    if (operationCount > 0) {
                        result = opsObject.getJSONArray("operations").getJSONObject(0);
                    }
                }
            } catch (final JSONException e) {
                ; // do nothing
            } catch (final Throwable e) {
                ; // do nothing
            }
        }

        return result;
    }

    private JSONObject debugGetLastOperationFromMessageChunkList(ArrayList<MessageChunk> chunkList) {
        JSONObject result = null;

        if (chunkList.size() > 0) {
            MessageChunk lastChunk = chunkList.get(chunkList.size() - 1);
            JSONObject opsObject = lastChunk.getOperations();
            try {
                JSONArray actions = opsObject.getJSONArray("actions");
                if (actions.length() > 0) {
                    opsObject = actions.getJSONObject(actions.length() - 1);
                    int operationCount = opsObject.getJSONArray("operations").length();
                    if (operationCount > 0) {
                        result = opsObject.getJSONArray("operations").getJSONObject(operationCount - 1);
                    }
                }
            } catch (final JSONException e) {
                ; // do nothing
            } catch (final Throwable e) {
                ; // do nothing
            }
        }

        return result;
    }

    private String operationToString(JSONObject operation) {
        String result = "";

        if (null != operation) {
            try {
                JSONDebugWriter myWriter = new JSONDebugWriter();
                operation.write(myWriter);
                result = myWriter.getData();
            } catch (final Exception e) {
                // nothing
            }
        }

        return result;
    }

    // - Members ----------------------------------------------------------------

    static private final org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(Connection.class);

    static private final ActionHandler m_handler = new ActionHandler(Connection.class);

    static private final long ACTIONS_UNTIL_FORCE_UPDATE = 20;

    private ServiceLookup m_services = null;

    private FileHelper m_fileHelper = null;

    private ResourceManager m_resourceManager = null;

    private final ArrayList<MessageChunk> m_messageChunkList = new ArrayList<MessageChunk>();

    private JSONObject m_uiOperations = null;

    private final ConnectionStatus m_connectionStatus = new ConnectionStatus();

    private ICalcEngineClient m_calcClient = null;

    private String m_calcDocumentHandle;

    private ID firstEditClient = null;

    private boolean m_savedWithRevision = false;

    private final AtomicInteger m_saveDocumentInProgress = new AtomicInteger();

    private Date m_lastSaveTime = null;

    private ScheduledTimerTask m_prepareLosingEditRightsTimer = null;

    private final ArrayList<String> m_memberNames = new ArrayList<String>();

    private long m_threadId = 0;

    private long m_nActionsReceived = 0;
}
