/*
 *  Copyright 2007-2008, Plutext Pty Ltd.
 *
 *  This file is part of docx4j.

    docx4j is licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.

    You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

    Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.

 */
package org.docx4j.jaxb;


import java.util.HashMap;
import java.util.Map;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import org.apache.commons.logging.Log;
import com.openexchange.log.LogFactory;

public class Context {

    private final static Map<String, JAXBContext> contexts = new HashMap<String, JAXBContext>();

    private static final JAXBContext getContext(final String className)
    {
        synchronized(contexts)
        {
            JAXBContext jc = null;
            if(contexts.containsKey(className)) {
                jc = contexts.get(className);
            }
            if(jc==null) {
                try {
                    Context tmp = new Context();
                    java.lang.ClassLoader classLoader = tmp.getClass().getClassLoader();
                    jc = JAXBContext.newInstance(className, classLoader);
                } catch (JAXBException e) {
                    throw new RuntimeException(e);
                }
                contexts.put(className, jc);
            }
            return jc;
        }
    }

    public static final JAXBContext getJc() {
        return getContext("org.docx4j.wml:" +
            "org.docx4j.dml:org.docx4j.dml.chart:org.docx4j.dml.chartDrawing:org.docx4j.dml.compatibility:org.docx4j.dml.diagram:org.docx4j.dml.lockedCanvas:org.docx4j.dml.picture:org.docx4j.dml.wordprocessingDrawing:org.docx4j.dml.spreadsheetdrawing:org.docx4j.dml.diagram2008:" +
            // All VML stuff is here, since compiling it requires WML and DML (and MathML), but not PML or SML
            "org.docx4j.vml:org.docx4j.vml.officedrawing:org.docx4j.vml.wordprocessingDrawing:org.docx4j.vml.presentationDrawing:org.docx4j.vml.spreadsheetDrawing:org.docx4j.vml.root:" +
            "org.opendope.xpaths:org.opendope.conditions:org.opendope.questions:org.opendope.components:org.opendope.SmartArt.dataHierarchy:" +
            "org.docx4j.math:" +
            "org.docx4j.sharedtypes:org.docx4j.bibliography");
    }

    public static final JAXBContext getJcThemePart() {
        return getJc();
    }

    public static final JAXBContext getJcDocPropsCore() {
        return getContext("org.docx4j.docProps.core:org.docx4j.docProps.core.dc.elements:org.docx4j.docProps.core.dc.terms");
    }

    public static final JAXBContext getJcDocPropsCustom() {
        return getContext("org.docx4j.docProps.custom");
    }

    public static final JAXBContext getJcDocPropsExtended() {
        return getContext("org.docx4j.docProps.extended");
    }

    public static final JAXBContext getJcRelationships() {
        return getContext("org.docx4j.relationships");
    }

    public static final JAXBContext getJcCustomXmlProperties() {
        return getContext("org.docx4j.customXmlProperties");
    }

    public static final JAXBContext getJcContentTypes() {
        return getContext("org.docx4j.openpackaging.contenttype");
    }

    public static final JAXBContext getJcXmlPackage() {
        return getContext("org.docx4j.xmlPackage");
    }

    public static final JAXBContext getXslFoContext() {
        return getContext("org.plutext.jaxb.xslfo");
    }

    public static final JAXBContext getJcSectionModel() {
        return getContext("org.docx4j.model.structure.jaxb");
    }

    public static final JAXBContext getJcXmlDSig() {
        return getContext("org.plutext.jaxb.xmldsig");
    }

    public static final JAXBContext getJcSML() {
        return getContext("org.xlsx4j.sml");
    }

    public static final JAXBContext getJcPML() {
        return getContext("org.pptx4j.pml:" +
        "org.docx4j.dml:org.docx4j.dml.chart:org.docx4j.dml.chartDrawing:org.docx4j.dml.compatibility:org.docx4j.dml.diagram:org.docx4j.dml.lockedCanvas:org.docx4j.dml.picture:org.docx4j.dml.wordprocessingDrawing:org.docx4j.dml.spreadsheetdrawing");
    }

	private static Log log = LogFactory.getLog(Context.class);

	static {

		// Display diagnostic info about version of JAXB being used.
		Object namespacePrefixMapper;
		try {
			namespacePrefixMapper = NamespacePrefixMapperUtils.getPrefixMapper();
			if ( namespacePrefixMapper.getClass().getName().equals("org.docx4j.jaxb.NamespacePrefixMapperSunInternal") ) {
				// Java 6
				log.info("Using Java 6/7 JAXB implementation");
			} else {
				log.info("Using JAXB Reference Implementation");
			}
		} catch (JAXBException e) {
			log.error("PANIC! No suitable JAXB implementation available");
		}
	}

	private static org.docx4j.wml.ObjectFactory wmlObjectFactory;
	public static org.docx4j.wml.ObjectFactory getWmlObjectFactory() {

		if (wmlObjectFactory==null) {
			wmlObjectFactory = new org.docx4j.wml.ObjectFactory();
		}
		return wmlObjectFactory;
	}

    private static org.xlsx4j.sml.ObjectFactory smlObjectFactory;
    public static org.xlsx4j.sml.ObjectFactory getsmlObjectFactory() {

        if (smlObjectFactory==null) {
            smlObjectFactory = new org.xlsx4j.sml.ObjectFactory();
        }
        return smlObjectFactory;
    }

    private static org.pptx4j.pml.ObjectFactory pmlObjectFactory;
    public static org.pptx4j.pml.ObjectFactory getpmlObjectFactory() {

        if (pmlObjectFactory==null) {
            pmlObjectFactory = new org.pptx4j.pml.ObjectFactory();
        }
        return pmlObjectFactory;
    }

    private static org.docx4j.vml.wordprocessingDrawing.ObjectFactory vmlWordprocessingDrawingObjectFactory;
    public static org.docx4j.vml.wordprocessingDrawing.ObjectFactory getVmlWordprocessingDrawingObjectFactory() {
        if(vmlWordprocessingDrawingObjectFactory==null) {
            vmlWordprocessingDrawingObjectFactory = new org.docx4j.vml.wordprocessingDrawing.ObjectFactory();
        }
        return vmlWordprocessingDrawingObjectFactory;
    }

}
