/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx;

import java.io.InputStream;
import java.util.List;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import org.docx4j.dml.Theme;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.Docx4JException;
import org.docx4j.openpackaging.packages.SpreadsheetMLPackage;
import org.docx4j.openpackaging.parts.PartName;
import org.docx4j.openpackaging.parts.ThemePart;
import org.docx4j.openpackaging.parts.SpreadsheetML.SharedStrings;
import org.docx4j.openpackaging.parts.SpreadsheetML.Styles;
import org.json.JSONArray;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.xlsx4j.sml.CTRst;
import org.xlsx4j.sml.CTSst;
import org.xlsx4j.sml.CTStylesheet;
import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.FilterException.ErrorCode;
import com.openexchange.office.ooxml.xlsx.operations.ApplyOperationHelper;
import com.openexchange.office.ooxml.xlsx.operations.CreateOperationHelper;
import com.openexchange.office.tools.ResourceManager;
import com.openexchange.server.ServiceLookup;

public class OperationDocument extends com.openexchange.office.ooxml.OperationDocument {

    final private SpreadsheetMLPackage opcPackage;
    protected CTStylesheet stylesheet = null;
    protected CTSst sharedStrings = null;

    public OperationDocument(ServiceLookup _services, InputStream _inputDocumentStream, ResourceManager _resourceManager, DocumentProperties _documentProperties)
        throws FilterException {

        super(_services, _inputDocumentStream, _resourceManager, _documentProperties);
        try {
            opcPackage = (SpreadsheetMLPackage)SpreadsheetMLPackage.load(_inputDocumentStream);

            // styles
            Styles styles = getPackage().getWorkbookPart().getStyles();
            if(styles!=null)
                stylesheet = styles.getJaxbElement();

            // shared strings
            SharedStrings _sharedStrings = getPackage().getWorkbookPart().getSharedStrings();
            if(_sharedStrings!=null)
                sharedStrings = _sharedStrings.getJaxbElement();
        }
        catch(Docx4JException e) {
            ErrorCode errorCode = ErrorCode.CRITICAL_ERROR;
            if(e.getCause() instanceof java.security.InvalidKeyException) {
                errorCode = ErrorCode.WRONG_PASSWORD;
            }
            else if(e.getCause() instanceof org.apache.poi.EncryptedDocumentException) {
                errorCode = ErrorCode.UNSUPPORTED_ENCRYPTION_USED;
            }
            throw new FilterException(e, errorCode);
        }
    }

    // the constructor without inputStream creates an empty document
    public OperationDocument(ServiceLookup _services, DocumentProperties _documentProperties)
        throws FilterException {

        super(_services, null, null, _documentProperties);
        try {
            opcPackage = SpreadsheetMLPackage.createPackage();

            ThemePart themePart = getDefaultThemePart(new PartName("/xl/theme/theme1.xml"));
            getPackage().addTargetPart(themePart);
            getPackage().createWorksheetPart(new PartName("/xl/worksheets/sheet1.xml"), "Sheet1", 1);
            getStylesheet(true);
        }
        catch (Exception e) {
            throw new FilterException(e, ErrorCode.CRITICAL_ERROR);
        }
    }

    public CTStylesheet getStylesheet(boolean createIfMissing)
        throws Exception {

        if(stylesheet==null&&createIfMissing) {
            Styles styles = new Styles(new PartName("/xl/styles.xml"));
            stylesheet = createStylesheet();
            styles.setJaxbElement(stylesheet);
            getPackage().getWorkbookPart().addTargetPart(styles);
        }
        return stylesheet;
    }

    public CTSst getSharedStrings(boolean createIfMissing)
        throws Exception {

        if(sharedStrings==null&&createIfMissing) {
            SharedStrings _sharedStrings = new SharedStrings(new PartName("/xl/sharedStrings.xml"));
            sharedStrings = Context.getsmlObjectFactory().createCTSst();
            _sharedStrings.setJaxbElement(sharedStrings);
            getPackage().getWorkbookPart().addTargetPart(_sharedStrings);
        }
        return sharedStrings;
    }

    // returns null if something goes wrong ...
    public String getSharedString(String index) {

        String sharedString = null;
        if(sharedStrings!=null) {
            List<CTRst> si = sharedStrings.getSi();
            try  {
                int i = Integer.parseInt(index);
                if(i>=0&&i<si.size()) {
                    CTRst ctRst = si.get(i);
                    sharedString = ctRst.getT();
                }
            }
            catch(NumberFormatException e) {
                // TODO: what to do now ?
            }
        }
        return sharedString;
    }

    //
    public String escapeString(String s) {
        if(s==null||s.length()==0)
            return s;
        char char0 = s.charAt(0);
        return ((char0=='\'')||(char0=='=')||(char0=='#')) ? "'" + s : s;
    }

    @Override
    public void applyOperations(String applyOperations) throws FilterException {

        if (applyOperations != null){
            int i = 0;
            JSONObject op = null;
            try {
                ApplyOperationHelper applyOperationHelper = new ApplyOperationHelper(this);
                final JSONArray aOperations = new JSONArray(new JSONTokener(applyOperations));
                for (i = 0; i < aOperations.length(); i++) {
                    op = (JSONObject) aOperations.get(i);
                    String opName = op.getString("name");
                    if (opName.equals("setCellContents"))
                        applyOperationHelper.setCellContents(op.getInt("sheet"), op.getJSONArray("start"), op.getJSONArray("contents"));
                    else if(opName.equals("fillCellRange"))
                        applyOperationHelper.fillCellRange(op.getInt("sheet"), op.getJSONArray("start"), op.optJSONArray("end"), op.has("value"), op.opt("value"), op.optJSONObject("attrs"), op.optInt("shared", -1), op.optJSONArray("ref"));
                    else if (opName.equals("setRowAttributes"))
                        applyOperationHelper.setRowAttributes(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1), op.getJSONObject("attrs"));
                    else if (opName.equals("setColumnAttributes"))
                        applyOperationHelper.setColumnAttributes(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1), op.getJSONObject("attrs"));
                    else if (opName.equals("insertRows"))
                        applyOperationHelper.insertRows(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1));
                    else if (opName.equals("deleteRows"))
                        applyOperationHelper.deleteRows(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1));
                    else if (opName.equals("insertSheet"))
                        applyOperationHelper.insertSheet(op.getInt("sheet"), op.getString("sheetName"), op.optJSONObject("attrs"));
                    else if (opName.equals("moveSheet"))
                        applyOperationHelper.moveSheet(op.getInt("sheet"), op.getInt("to"));
                    else if (opName.equals("deleteSheet"))
                        applyOperationHelper.deleteSheet(op.getInt("sheet"));
                    else if (opName.equals("setSheetName"))
                        applyOperationHelper.setSheetName(op.getInt("sheet"), op.getString("sheetName"));
                    else if (opName.equals("setSheetAttributes"))
                        applyOperationHelper.setSheetAttributes(op.getInt("sheet"), op.getJSONObject("attrs"));
                    else if (opName.equals("mergeCells"))
                        applyOperationHelper.mergeCells(op.getInt("sheet"), op.getJSONArray("start"), op.optJSONArray("end"), op.optBoolean("keepContent", false), op.optString("type", "merge"));
                }
            }
            catch(Exception e) {
                String message = e.getMessage();
                if(op!=null) {
                    message += ", operation:" + Integer.toString(i) + " " + op.toString();
                }
                throw new FilterException(message, ErrorCode.CRITICAL_ERROR);
            }
        }
    }

    @Override
    public JSONObject getOperations()
        throws Exception {

        final JSONObject aOperations = new JSONObject();
        final JSONArray operationsArray = new JSONArray();
        CreateOperationHelper createOperationHelper = new CreateOperationHelper(this, operationsArray);
        createOperationHelper.createDocumentDefaults(getUserLanguage());
        createOperationHelper.CreateStyleOperations();
        createOperationHelper.CreateThemeOperations();
        createOperationHelper.createOperations();
        aOperations.put("operations", operationsArray);

        return aOperations;
    }

    @Override
    public SpreadsheetMLPackage getPackage() {
        return opcPackage;
    }

    @Override
    public Theme getTheme() {
        try {
            return ((ThemePart)getPackage().getParts().get(new PartName("/xl/theme/theme1.xml"))).getJaxbElement();
        }
        catch (Exception e) {
            return null;
        }
    }

    private CTStylesheet createStylesheet() throws JAXBException {

        final String defaultStylesheet = "<styleSheet xmlns=\"http://schemas.openxmlformats.org/spreadsheetml/2006/main\"" +
        		"    xmlns:mc=\"http://schemas.openxmlformats.org/markup-compatibility/2006\"" +
        		"    mc:Ignorable=\"x14ac\"" +
        		"    xmlns:x14ac=\"http://schemas.microsoft.com/office/spreadsheetml/2009/9/ac\">" +
        		"    <fonts count=\"1\" x14ac:knownFonts=\"1\">" +
        		"        <font>" +
        		"            <sz val=\"11\" />" +
        		"            <color theme=\"1\" />" +
        		"            <name val=\"Calibri\" />" +
        		"            <family val=\"2\" />" +
        		"            <scheme val=\"minor\" />" +
        		"        </font>" +
        		"    </fonts>" +
        		"    <fills count=\"2\">" +
        		"        <fill>" +
        		"            <patternFill patternType=\"none\" />" +
        		"        </fill>" +
        		"        <fill>" +
        		"            <patternFill patternType=\"gray125\" />" +
        		"        </fill>" +
        		"    </fills>" +
        		"    <borders count=\"1\">" +
        		"        <border>" +
        		"            <left />" +
        		"            <right />" +
        		"            <top />" +
        		"            <bottom />" +
        		"            <diagonal />" +
        		"        </border>" +
        		"    </borders>" +
        		"    <cellStyleXfs count=\"1\">" +
        		"        <xf numFmtId=\"0\" fontId=\"0\" fillId=\"0\" borderId=\"0\" />" +
        		"    </cellStyleXfs>" +
        		"    <cellXfs count=\"2\">" +
        		"        <xf numFmtId=\"0\" fontId=\"0\" fillId=\"0\" borderId=\"0\" xfId=\"0\" />" +
        		"    </cellXfs>" +
        		"    <cellStyles count=\"1\">" +
        		"        <cellStyle name=\"Standard\" xfId=\"0\" builtinId=\"0\" />" +
        		"    </cellStyles>" +
        		"    <dxfs count=\"0\" />" +
        		"    <tableStyles count=\"0\" defaultTableStyle=\"TableStyleMedium2\"" +
        		"        defaultPivotStyle=\"PivotStyleLight16\" />" +
        		"</styleSheet>";

        java.util.HashMap<String, String> mappings = new java.util.HashMap<String, String>();
        Object o = org.docx4j.XmlUtils.unmarshallFromTemplate(defaultStylesheet, mappings, getPackage().getWorkbookPart().getJAXBContext());
        return o instanceof JAXBElement ? (CTStylesheet)((JAXBElement<?>)o).getValue() : (CTStylesheet)o;
    }
}
