/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.tools.jobpool.impl;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.lang.Validate;

import com.openexchange.office.tools.jobpool.IJob;
import com.openexchange.office.tools.jobpool.IJobPool;
import com.openexchange.office.tools.jobpool.JobBase;
import com.openexchange.office.tools.jobpool.JobEnvironment;
import com.openexchange.office.tools.jobpool.JobPoolConfig;

//=============================================================================
public class JobPool implements IJobPool
{
	//-------------------------------------------------------------------------
	public static final int DEFAULT_POOL_SIZE = 5;
	
	//-------------------------------------------------------------------------
	public JobPool ()
	    throws Exception
	{}
	
	//-------------------------------------------------------------------------
	/** set the job type for all jobs managed by this pool.
	 * 
	 *  @note	We request all jobs managed by this pool needs to support
	 *  		some base functionality ... so we enforce JobBase as base
	 *  		class of all job implementations.
	 * 
	 *  @param	aType [IN]
	 *  		the job type.
	 */
	public synchronized void setJobType (final Class< ? extends JobBase > aType)
	    throws Exception
	{
		m_aType = aType;
	}

	//-------------------------------------------------------------------------
	@Override
	public synchronized void configureJobPool (final JobPoolConfig aConfig)
	    throws Exception
	{
		final Integer nPoolSize = aConfig.get(JobPoolConfig.PROP_POOL_SIZE, DEFAULT_POOL_SIZE);
		Validate.isTrue(nPoolSize > 0, "Invalid config '"+JobPoolConfig.PROP_POOL_SIZE+"'. Needs to be > 0.");
		
		//final Boolean bBlockOnLease = aConfig.get(JobPoolConfig.PROP_BLOCK_ON_LEASE, Boolean.TRUE);
		
		// TODO implement me :-)
		
		m_aConfig = aConfig;
	}

	//-------------------------------------------------------------------------
	@Override
	public synchronized void configureDefaultJobEnvironment (final JobEnvironment aEnv)
		throws Exception
	{
		m_aDefaultEnv = aEnv;
	}

	//-------------------------------------------------------------------------
	@Override
	public synchronized void configureJobEnvironment (final int            nJobId,
										              final JobEnvironment aEnv  )
	    throws Exception
	{
		final int nPoolSize   = m_aConfig.get(JobPoolConfig.PROP_POOL_SIZE, DEFAULT_POOL_SIZE);
		final int nActualSize = mem_Pool ().size();
		
		Validate.isTrue(nActualSize<=nPoolSize, "Pool size exceeded ! Pool configuration and count of provided job environment needs to be in sync.");
		
		JobDescriptor rDesc = impl_getOrCreateJobDescriptor (nJobId);
		aEnv.setId(nJobId);
		rDesc.aEnv = aEnv;
	}
	
	//-------------------------------------------------------------------------
	@Override
	public synchronized IJob leaseJob()
		throws Exception
	{
		final Job aJob = impl_leaseJob (null);
		return aJob;
	}

	//-------------------------------------------------------------------------
	public synchronized IJob leaseJob(final int nJobId)
		throws Exception
	{
		final Job aJob = impl_leaseJob (nJobId);
		return aJob;
	}

	//-------------------------------------------------------------------------
	@Override
	public synchronized void freeJob(final IJob iJob)
		throws Exception
	{
		final Job           aJobWrapper = (Job) iJob;
		final JobBase       aJobImpl    = aJobWrapper.getJob ();
		final int           nJobId      = aJobImpl   .getId  ();
		final JobDescriptor aDesc       = impl_getOrCreateJobDescriptor(nJobId);

		aJobWrapper.bindToJob(null );
		aDesc.aEnv.setInUse  (false);
		
		// TODO implement me : notify leaseJob .. which is might be in blocked mode and waits for free jobs ...
	}

	//-------------------------------------------------------------------------
	/** has to be called from our Job implementation in error case only.
	 *  Not allowed to be called from outside ...
	 *  
	 *  @param	aJob [IN]
	 *  		the job to be disabled - because its 'non functional' any longer.
	 */
	protected synchronized void deactivateJob (final Job aJobWrapper)
	    throws Exception
	{
		// THIS will be the real "delete" of a job (in combination with some tricky use of weak references anywhere else .-)
		// Use a local copy for the rest of this method to remove it's references in some further container structures ...

		final JobBase       aJobImpl = aJobWrapper.getJob();
		final int           nJobId   = aJobImpl   .getId  ();
		final JobDescriptor aDesc    = impl_getOrCreateJobDescriptor(nJobId);

		aJobWrapper.bindToJob(null );
		aDesc.aEnv.setInUse  (false);
		aDesc.aJob = null; // !!! thats the "killer" .-)

		try
		{
			// TODO fix me : might be its a good idea to do that within a separate thread ...
			//               to prevent THIS thread from blocking in case deactivating will be blocked
			//               synchronous by any reason ... ?!
		    aJobImpl.deactivateJob();
		}
		catch (Throwable ex)
		{
			// IGNORE ERRORS HERE !
			// Might be those job is already dead ...
		}
	}

	//-------------------------------------------------------------------------
	public synchronized JobPoolConfig getConfig ()
	    throws Exception
	{
		if (m_aConfig != null)
			return m_aConfig;
		
		return JobPoolConfig.EMPTY_CONFIG();
	}

	//-------------------------------------------------------------------------
	private Job impl_leaseJob(final Integer nJobId)
		throws Exception
	{
		final boolean       bBlock    = m_aConfig.get(JobPoolConfig.PROP_BLOCK_ON_LEASE, Boolean.TRUE);
  	          long          nTimeout  = m_aConfig.get(JobPoolConfig.PROP_LEASE_BLOCK_TIMEOUT, JobPoolConfig.DEFAULT_LEASE_BLOCK_TIMEOUT_IN_MS);
		      long          nGameOver = System.currentTimeMillis() + nTimeout;
  	          JobDescriptor aDesc     = null;
		      Job           aJob      = null;
		      
		while (true)
		{
			if (nJobId == null)
				aDesc = impl_getFreeJob ();
			else
			{
				aDesc = impl_getOrCreateJobDescriptor(nJobId);
				if (aDesc.aEnv.isInUse())
					aDesc = null;
			}

			if (aDesc != null)
				aJob = impl_prepareJobForLease (aDesc);
			
			if (aJob != null)
				break;
			
			if (! bBlock)
				break;
			
			Thread.sleep (250); // TODO fix me : find better solution (e.g. notify on freeJob ...)
			
			final long nNow = System.currentTimeMillis();
			System.out.println("now       = "+nNow     );
			System.out.println("game over = "+nGameOver);
			if (nNow > nGameOver)
			{
				System.out.println ("... GIVE UP.");
				break;
			}
		}
		
		if (aJob == null)
			throw new Exception ("No free job available.");

		return aJob;
	}

	//-------------------------------------------------------------------------
	private Job impl_prepareJobForLease (final JobDescriptor aDesc)
	    throws Exception
	{
		JobBase aJobImpl   = aDesc.aJob;
		        aDesc.aJob = null;
		int     nRetry     = 3;

		while (true)
		{
			--nRetry;

			if (nRetry < 1)
				break;
			
			if (aJobImpl == null)
			{
				aJobImpl = m_aType.newInstance();
				aJobImpl.setEnvironment(aDesc.aEnv);
				aJobImpl.activateJob   ();
			}
	
			try
			{
				aJobImpl.validateJob();
			}
			catch (Throwable ex)
			{
				aJobImpl = null;
				continue;
			}
		}

		if (aJobImpl == null)
			return null;
		
		final Job aJobWrapper = new Job ();

		aJobWrapper.bindToPool(this    );
		aJobWrapper.bindToJob (aJobImpl);

		aDesc.aJob = aJobImpl;
		aDesc.aEnv.setInUse(true);
		
		return aJobWrapper;
	}

	//-------------------------------------------------------------------------
	private JobDescriptor impl_getFreeJob ()
	    throws Exception
	{
		final Map< Integer, JobDescriptor >               aPool       = mem_Pool ();
		final Iterator< Entry< Integer, JobDescriptor > > pIt         = aPool.entrySet().iterator();
		final int                                         nPoolSize   = m_aConfig.get(JobPoolConfig.PROP_POOL_SIZE, DEFAULT_POOL_SIZE);
		final int                                         nActualSize = aPool.size();

		while (pIt.hasNext())
		{
			final Entry< Integer, JobDescriptor > aEntry = pIt.next ();
			final int                             nJobId = aEntry.getKey  ();
			final JobDescriptor                   aDesc  = aEntry.getValue();

			if (aDesc.aEnv == null)
			{
				aDesc.aEnv = JobEnvironment.clone(m_aDefaultEnv);
				aDesc.aEnv.setId(nJobId);
			}
			
			if (aDesc.aEnv.isInUse())
				continue;
			
			return aDesc;
		}
		
		if (nActualSize < nPoolSize)
		{
			final JobDescriptor aDesc  = new JobDescriptor ();
			final int           nJobId = impl_getFreeJobId ();

			aDesc.aEnv = JobEnvironment.clone(m_aDefaultEnv);
			aDesc.aEnv.setId(nJobId);
			aPool.put(nJobId, aDesc);

			return aDesc;
		}
		
		return null;
	}

	//-------------------------------------------------------------------------
	private JobDescriptor impl_getOrCreateJobDescriptor (final int nJobId)
	    throws Exception
	{
		final Map< Integer, JobDescriptor > aPool = mem_Pool ();
			  JobDescriptor                 aDesc = aPool.get(nJobId);
			  
		if (aDesc == null)
		{
			aDesc = new JobDescriptor ();
			aPool.put(nJobId, aDesc);
		}
		
		if (aDesc.aEnv == null)
		{
			aDesc.aEnv = JobEnvironment.clone(m_aDefaultEnv);
			aDesc.aEnv.setId(nJobId);
		}
		
		return aDesc;
	}

	//-------------------------------------------------------------------------
	private int impl_getFreeJobId ()
	    throws Exception
	{
		final Map< Integer, JobDescriptor > aPool = mem_Pool ();
		for (int nId=0; nId<Integer.MAX_VALUE; ++nId)
		{
			if ( ! aPool.containsKey(nId))
				return nId;
		}
		
		throw new Exception ("Unexpected situation or unexpected use of pool detected ! No free IDs in range of MAX.INT !");
	}
	
	//-------------------------------------------------------------------------
	private Map< Integer, JobDescriptor > mem_Pool ()
	    throws Exception
	{
		if (m_aPool == null)
		{
			Map< Integer, JobDescriptor > aPool = new HashMap< Integer, JobDescriptor > ();
			m_aPool = aPool;
		}
		return m_aPool;
	}

	//-------------------------------------------------------------------------
	private class JobDescriptor 
	{
		public JobEnvironment aEnv = null;
		public JobBase        aJob = null;
	}
	
	//-------------------------------------------------------------------------
	private JobPoolConfig m_aConfig = JobPoolConfig.DEFAULT_CONFIG();

	//-------------------------------------------------------------------------
	private Class< ? extends JobBase > m_aType = null;

	//-------------------------------------------------------------------------
	private JobEnvironment m_aDefaultEnv = JobEnvironment.EMPTY_ENVIRONMENT();
	
	//-------------------------------------------------------------------------
	private Map< Integer, JobDescriptor > m_aPool = null;
}
