/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.calcengine.client.impl;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang.Validate;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.openexchange.office.calcengine.JCalcDocumentWrapper;
import com.openexchange.office.calcengine.client.CalcEngineConfig;
import com.openexchange.office.calcengine.client.CalcEngineDescriptor;
import com.openexchange.office.calcengine.client.ICalcEngineClient;
import com.openexchange.office.tools.jobpool.IJob;
import com.openexchange.office.tools.jobpool.IJobPool;
import com.openexchange.office.tools.jobpool.JobArguments;
import com.openexchange.office.tools.jobpool.JobEnvironment;
import com.openexchange.office.tools.jobpool.JobPoolConfig;
import com.openexchange.office.tools.jobpool.JobPoolManager;
import com.openexchange.office.tools.jobpool.JobResults;

//=============================================================================
/** Run the calc engine in 'local server' mode.
 *  Call chain will be the following :
 *  
 *  <ul>
 *  <li>calli</li>
 *  <li>-> ICalcEngineClient</li>
 *  <li>-> CalcEngineClientLocal</li>
 *  <li>-> JobPool</li>
 *  <li>-> Job-01</li>
 *	<li>...</li>
 *  <li>-> Job-XX</li>
 *  </ul>
 *  
 *  where every job will do this :
 *
 *  <ul>
 *  <li>Job-XX</li>
 *  <li>-> CalcEngineJob (HTTP)</li>
 *  <li>...</li>
 *  <li>-> (HTTP) CalcEngineWorkerServlet</li>
 *  <li>-> ICalcEngineClient</li>
 *  <li>-> CalcEngineClientInline</li>
 *  <li>-> see how it work inside {@link CalcEngineClientInline} !</li>
 *  </ul>
 */
public class CalcEngineClientLocal implements ICalcEngineClient
{
    //-------------------------------------------------------------------------
    private static final Log LOG = LogFactory.getLog(CalcEngineClientLocal.class);

    //-------------------------------------------------------------------------
    private CalcEngineClientLocal ()
        throws Exception
    {}
    
    //-------------------------------------------------------------------------
    public synchronized static CalcEngineClientLocal create()
        throws Exception
    {
    	if (m_gSingleton == null)
    		m_gSingleton = new CalcEngineClientLocal ();
    	return m_gSingleton;
    }

    //-------------------------------------------------------------------------
    public synchronized void describeEnvironment (final CalcEngineDescriptor aDesc)
        throws Exception
    {
    	m_aDescriptor = aDesc;
    }
    
    //-------------------------------------------------------------------------
    @Override
    public synchronized String createDocument()
        throws Exception
    {
    	// TODO CHANGE THE API !
    	// Doc handle must be calculated by ME ... not by C++ process.
    	// Otherwise different calc engine process return same handle ...
    	// or use UUID.toString as ID !

    	final IJobPool     iPool    = mem_Pool ();
    	final IJob         iJob     = iPool.leaseJob();
    	final JobArguments lArgs    = JobArguments.EMPTY_ARGUMENTS();
    	
    	lArgs.set(CalcEngineJob.ARG_OPERATION, CalcEngineJob.OP_CREATE_DOCUMENT);

    	final JobResults lResults = iJob.executeSynchronous(lArgs);
    	if ( ! lResults.wasSuccessfully())
    		throw new Exception ("Creating new document was not successfully.");
    	
    	final String sHandle = lResults.get(CalcEngineJob.RESULT_DOCUMENT_HANDLE);
    	Validate.notEmpty(sHandle, "Unexpected and missing doc handle.");

    	final int                    nJobId     = iJob.getId    ();
    	final Map< String, Integer > aStickiMap = mem_StickiMap ();
    	aStickiMap.put (sHandle, nJobId);
    	
    	return sHandle;
    };
    
    //-------------------------------------------------------------------------
    @Override
    public synchronized void destroyDocument(final String sHandle)
        throws Exception
    {
    	final IJob         iJob  = impl_getStickiJob (sHandle);
    	final JobArguments lArgs = JobArguments.EMPTY_ARGUMENTS();
    	
    	lArgs.set(CalcEngineJob.ARG_OPERATION      , CalcEngineJob.OP_DESTROY_DOCUMENT);
    	lArgs.set(CalcEngineJob.ARG_DOCUMENT_HANDLE, sHandle                          );

    	final JobResults lResults = iJob.executeSynchronous(lArgs);
    	
    	if ( ! lResults.wasSuccessfully())
    		throw new Exception ("Destroy of document '"+sHandle+"' was not successfully.");   	
    };

    //-------------------------------------------------------------------------
    @Override
    public synchronized String executeOperation(final String sHandle       ,
                                   				final String sOperationJSON)
        throws Exception
    {
    	final IJob         iJob  = impl_getStickiJob (sHandle);
    	final JobArguments lArgs = JobArguments.EMPTY_ARGUMENTS();
    	
    	lArgs.set(CalcEngineJob.ARG_OPERATION      , CalcEngineJob.OP_EXECUTE_OPERATION);
    	lArgs.set(CalcEngineJob.ARG_DOCUMENT_HANDLE, sHandle                           );
    	lArgs.set(CalcEngineJob.ARG_OPERATIONLIST  , sOperationJSON                    );

    	final JobResults lResults = iJob.executeSynchronous(lArgs);
    	
    	if ( ! lResults.wasSuccessfully())
    		throw new Exception ("Execution of operations on document '"+sHandle+"' was not successfully.");
    
    	final String sResult = lResults.get(CalcEngineJob.RESULT_4_EXECUTION);
    	return sResult;
    };
    //-------------------------------------------------------------------------
    @Override
    public String getVersion()
        throws Exception
    {
        return "";
    }
    //-------------------------------------------------------------------------
    private IJob impl_getStickiJob (final String sHandle)
        throws Exception
    {
    	final Map< String, Integer > aStickiMap = mem_StickiMap ();
    	final Integer                nJobId     = aStickiMap.get (sHandle);
    	
    	Validate.isTrue (nJobId != null, "No calc engine for this handle '"+sHandle+"'.");
    	
    	final IJobPool iPool = mem_Pool ();
    	final IJob     iJob  = iPool.leaseJob(nJobId);
    	
    	return iJob;
    }

    //-------------------------------------------------------------------------
    private IJobPool mem_Pool ()
        throws Exception
    {
    	if (m_iPool == null)
    	{
    		final IJobPool         iPool       = JobPoolManager.accessPool(CalcEngineJob.class);
    		final JobPoolConfig    aPoolConfig = JobPoolConfig.DEFAULT_CONFIG();
    		final CalcEngineConfig aCEConfig   = CalcEngineConfig.get();
    		final int              nPortMin    = aCEConfig.getWorkerPortRangeMin();
    		final int              nPortMax    = aCEConfig.getWorkerPortRangeMax();
    		
    		// check for port>0 is already done within configuration !
    		Validate.isTrue(nPortMax>=nPortMin, "Wrong configuration for worker port range. Range must be (at least) 1 and max needs to be greater then min.");
    		
    		final int nRange      = (nPortMax - nPortMin) + 1;
    		final int nJobTimeout = aCEConfig.getJobTimeout();
    		
    		aPoolConfig.set(JobPoolConfig.PROP_BLOCK_ON_LEASE       , Boolean.TRUE);
    		aPoolConfig.set(JobPoolConfig.PROP_POOL_SIZE            , nRange      );
    		aPoolConfig.set(JobPoolConfig.PROP_JOB_EXECUTION_TIMEOUT, nJobTimeout );
    		
    		iPool.configureJobPool(aPoolConfig);

    		int nJob = 0;
    		for (int nPort=nPortMin; nPort<=nPortMax; ++nPort, ++nJob)
    		{
    			final JobEnvironment aEnv = JobEnvironment.EMPTY_ENVIRONMENT();
    			aEnv.set(CalcEngineJob.ENV_JAR , nPort);
    			aEnv.set(CalcEngineJob.ENV_PORT, nPort);
    			iPool.configureJobEnvironment(nJob, aEnv);
    		}

    		m_iPool = iPool;
    	}
    	return m_iPool;
    }

    //-------------------------------------------------------------------------
    private Map< String, Integer > mem_StickiMap ()
        throws Exception
    {
    	if (m_aStickiMap == null)
    		m_aStickiMap = new HashMap< String, Integer > ();
    	return m_aStickiMap;
    }

    //-------------------------------------------------------------------------
    private static CalcEngineClientLocal m_gSingleton = null;

    //-------------------------------------------------------------------------
    private CalcEngineDescriptor m_aDescriptor = null;

    //-------------------------------------------------------------------------
    private IJobPool m_iPool = null;

    //-------------------------------------------------------------------------
    private Map< String, Integer > m_aStickiMap = null;
}
