/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/utils/scheduler',
    'io.ox/office/tk/object/baseobject'
], function (Utils, IteratorUtils, Scheduler, BaseObject) {

    'use strict';

    // jQuery.Deferred base test ==============================================

    describe('class jQuery.Deferred', function () {
        it('should exist', function () {
            expect($.Deferred).to.be.a('function');
        });
        it('should create a pending deferred object', function () {
            var def = $.Deferred();
            expect(def).to.be.an('object');
            expect(def).to.respondTo('state');
            expect(def).to.respondTo('promise');
            expect(def).to.respondTo('resolve');
            expect(def).to.respondTo('reject');
            expect(def).to.respondTo('done');
            expect(def).to.respondTo('fail');
            expect(def).to.respondTo('then');
        });
        it('should have pending state', function () {
            expect($.Deferred().state()).to.equal('pending');
        });
        it('should resolve immediately', function () {
            var def = $.Deferred();
            expect(def.resolve()).to.equal(def);
            expect(def.state()).to.equal('resolved');
        });
        it('should reject immediately', function () {
            var def = $.Deferred();
            expect(def.reject()).to.equal(def);
            expect(def.state()).to.equal('rejected');
        });
    });

    // class BaseObject =======================================================

    describe('Toolkit class BaseObject', function () {

        it('should exist', function () {
            expect(BaseObject).to.be.a('function');
        });

        // Scheduler must be empty, otherwise some tests may fail unexpectedly because of other
        // running background timer loops. If this test fails, there may exist another unit test
        // module whare an application has been created globally. Check that the function call
        // AppHelper.createXYZApplication() is located locally inside the callback of the
        // outermost "describe()" of the unit test.
        it('should run with clean timer scheduler', function () {
            expect(Scheduler.getPendingTimers().length).to.equal(0);
            expect(Scheduler.getPendingDeferreds().length).to.equal(0);
        });

        // private helpers ----------------------------------------------------

        var clock = null;
        before(function () { clock = sinon.useFakeTimers(); });
        after(function () { clock.restore(); });

        // ensure empty timer queue after each test
        afterEach(function () { clock.tick(10000); });

        // the minimum delay of the timer scheduler
        var INT = Scheduler.INTERVAL;

        // constants ----------------------------------------------------------

        describe('constant "SINGLETON"', function () {
            expect(BaseObject).to.have.a.property('SINGLETON').that.is.a('string');
        });

        // protected methods --------------------------------------------------

        describe('method "registerDestructor"', function () {
            it('should exist', function () {
                expect(BaseObject).to.respondTo('registerDestructor');
            });
            it('should register a destructor callback', function () {
                var obj = new BaseObject('window-0'), spy = sinon.spy();
                obj.registerDestructor(spy);
                expect(spy.called).to.equal(false);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "getWindowId"', function () {
            it('should exist', function () {
                expect(BaseObject).to.respondTo('getWindowId');
            });
            it('should return the window identifier', function () {
                var obj1 = new BaseObject('window-0');
                expect(obj1.getWindowId()).to.equal('window-0');
                var obj2 = new BaseObject(obj1);
                expect(obj2.getWindowId()).to.equal('window-0');
            });
        });

        describe('method "getUid"', function () {
            var obj1 = new BaseObject('window-0'), obj2 = new BaseObject('window-0');
            it('should exist', function () {
                expect(BaseObject).to.respondTo('getUid');
            });
            it('should return a UID', function () {
                expect(obj1.getUid()).to.be.a('string');
            });
            it('should return the same UID for the same object', function () {
                expect(obj1.getUid()).to.equal(obj1.getUid());
            });
            it('should return different UIDs for different objects', function () {
                expect(obj1.getUid()).to.not.equal(obj2.getUid());
            });
        });

        // promises -----------------------------------------------------------

        describe('method "createAbortablePromise"', function () {
            var obj = new BaseObject('window-0');
            it('should exist', function () {
                expect(BaseObject).to.respondTo('createAbortablePromise');
            });
            it('should create an abortable promise', function () {
                var promise = obj.createAbortablePromise($.Deferred());
                expect(promise).to.be.an('object');
                expect(promise).to.respondTo('state');
                expect(promise).to.respondTo('promise');
                expect(promise).to.respondTo('abort');
                expect(promise.state()).to.equal('pending');
                promise.abort();
                expect(promise.state()).to.equal('rejected');
            });
            it('should create an abortable promise of a resolved deferred object', function () {
                var promise = obj.createAbortablePromise($.Deferred().resolve());
                expect(promise).to.be.an('object');
                expect(promise).to.respondTo('abort');
                expect(promise.state()).to.equal('resolved');
                promise.abort();
                expect(promise.state()).to.equal('resolved');
            });
            it('should invoke the callback on abort', function () {
                var spy = sinon.spy();
                var promise = obj.createAbortablePromise($.Deferred(), spy);
                expect(spy.called).to.equal(false);
                promise.abort();
                expect(spy.calledOnce).to.equal(true);
                expect(spy.calledOn(obj)).to.equal(true);
                expect(spy.calledWith('abort')).to.equal(true);
            });
            it('should abort the promise after a timeout', function () {
                var promise = obj.createAbortablePromise($.Deferred(), null, 10);
                expect(promise.state()).to.equal('pending');
                clock.tick(11);
                expect(promise.state()).to.equal('rejected');
            });
            it('should override promise.then() to return an abortable promise', function () {
                var spy = sinon.spy();
                var promise1 = obj.createAbortablePromise($.Deferred(), spy).fail(spy);
                var promise2 = promise1.then(null, spy, null, spy);
                expect(promise1).to.not.equal(promise2);
                expect(promise2).to.respondTo('abort');
                promise2.abort(42);
                expect(promise1.state()).to.equal('rejected');
                expect(promise2.state()).to.equal('rejected');
                expect(spy.callCount).to.equal(4);
                expect(spy.alwaysCalledWith(42)).to.equal(true);

            });
            it('should abort chained abortable promises', function () {
                var promise1 = obj.createAbortablePromise($.Deferred().resolve());
                var promise2 = promise1.then(_.constant(obj.createAbortablePromise($.Deferred())));
                var promise3 = promise2.then(_.constant(obj.createAbortablePromise($.Deferred())));
                expect(promise1.state()).to.equal('resolved');
                expect(promise2.state()).to.equal('pending');
                expect(promise3.state()).to.equal('pending');
                promise3.abort();
                expect(promise1.state()).to.equal('resolved');
                expect(promise2.state()).to.equal('rejected');
                expect(promise3.state()).to.equal('rejected');
            });
            it('should abort the promise on destruction', function () {
                var spy = sinon.spy(),
                    promise = obj.createAbortablePromise($.Deferred(), spy);
                expect(promise.state()).to.equal('pending');
                obj.destroy();
                expect(promise.state()).to.equal('rejected');
                expect(spy.calledWith('destroy')).to.equal(true);
            });
        });

        describe('method "createResolvedPromise"', function () {
            var obj = new BaseObject('window-0');
            it('should exist', function () {
                expect(BaseObject).to.respondTo('createResolvedPromise');
            });
            it('should create a resolved abortable promise', function () {
                var promise = obj.createResolvedPromise(42);
                expect(promise).to.be.an('object');
                expect(promise).to.respondTo('abort');
                expect(promise.state()).to.equal('resolved');
                promise.abort();
                expect(promise.state()).to.equal('resolved');
            });
            it('should override promise.then() to return an abortable promise', function () {
                var promise1 = obj.createResolvedPromise(42);
                var promise2 = promise1.then();
                expect(promise1).to.not.equal(promise2);
                expect(promise2).to.respondTo('abort');
                expect(promise2.state()).to.equal('resolved');
            });
        });

        describe('method "createRejectedPromise"', function () {
            var obj = new BaseObject('window-0');
            it('should exist', function () {
                expect(BaseObject).to.respondTo('createRejectedPromise');
            });
            it('should create a rejected abortable promise', function () {
                var promise = obj.createRejectedPromise(42);
                expect(promise).to.be.an('object');
                expect(promise).to.respondTo('abort');
                expect(promise.state()).to.equal('rejected');
                promise.abort();
                expect(promise.state()).to.equal('rejected');
            });
            it('should override promise.then() to return an abortable promise', function () {
                var promise1 = obj.createRejectedPromise(42);
                var promise2 = promise1.then();
                expect(promise1).to.not.equal(promise2);
                expect(promise2).to.respondTo('abort');
                expect(promise2.state()).to.equal('rejected');
            });
        });

        describe('method "convertToPromise"', function () {
            var obj = new BaseObject('window-0');
            it('should exist', function () {
                expect(BaseObject).to.respondTo('convertToPromise');
            });
            it('should create a resolved promise', function () {
                var promise = obj.convertToPromise(42);
                expect(promise).to.be.an('object');
                expect(promise).to.respondTo('abort');
                expect(promise.state()).to.equal('resolved');
            });
            it('should create a rejected promise from constant value', function () {
                var promise1 = obj.convertToPromise(41, 42);
                var promise2 = obj.convertToPromise(42, 42);
                expect(promise1.state()).to.equal('resolved');
                expect(promise2.state()).to.equal('rejected');
            });
            it('should create a rejected promise from predicate', function () {
                var spy = sinon.spy(function (value) { return value === 42; });
                var promise1 = obj.convertToPromise(41, spy);
                var promise2 = obj.convertToPromise(42, spy);
                expect(promise1.state()).to.equal('resolved');
                expect(promise2.state()).to.equal('rejected');
                expect(spy.alwaysCalledOn(obj)).to.equal(true);
            });
            it('should return an existing promise', function () {
                var promise1 = obj.createResolvedPromise(42);
                var promise2 = obj.createRejectedPromise(42);
                var promise3 = new $.Deferred().promise();
                expect(obj.convertToPromise(promise1)).to.equal(promise1);
                expect(obj.convertToPromise(promise2)).to.equal(promise2);
                expect(obj.convertToPromise(promise3)).to.equal(promise3);
            });
        });

        describe('method "createDeferred"', function () {
            var obj = new BaseObject('window-0');
            it('should exist', function () {
                expect(BaseObject).to.respondTo('createDeferred');
            });
            it('should create a pending deferred object', function () {
                var def = obj.createDeferred('msg');
                expect(def).to.be.an('object');
                expect(def).to.respondTo('state');
                expect(def).to.respondTo('promise');
                expect(def).to.respondTo('resolve');
                expect(def).to.respondTo('reject');
                expect(def.state()).to.equal('pending');
            });
            it('should reject the deferred object on destruction', function () {
                var spy = sinon.spy();
                var def = obj.createDeferred('msg').fail(spy);
                expect(def.state()).to.equal('pending');
                obj.destroy();
                expect(def.state()).to.equal('rejected');
                expect(spy.calledWith('destroy')).to.equal(true);
            });
        });

        describe('method "waitForSuccess"', function () {
            it('should exist', function () {
                expect(BaseObject).to.respondTo('waitForSuccess');
            });
            it('should wait for promise\'s success', function () {
                var obj = new BaseObject('window-0');
                var def1 = $.Deferred(), spy1 = sinon.spy();
                var def2 = $.Deferred(), spy2 = sinon.spy();
                obj.waitForSuccess(def1, spy1);
                obj.waitForSuccess(def2, spy2);
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(false);
                def1.resolve();
                def2.reject();
                expect(spy1.called).to.equal(true);
                expect(spy2.called).to.equal(false);
            });
        });

        describe('method "waitForFailure"', function () {
            it('should exist', function () {
                expect(BaseObject).to.respondTo('waitForFailure');
            });
            it('should wait for promise\'s failure', function () {
                var obj = new BaseObject('window-0');
                var def1 = $.Deferred(), spy1 = sinon.spy();
                var def2 = $.Deferred(), spy2 = sinon.spy();
                obj.waitForFailure(def1, spy1);
                obj.waitForFailure(def2, spy2);
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(false);
                def1.resolve();
                def2.reject();
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(true);
            });
        });

        describe('method "waitForAny"', function () {
            it('should exist', function () {
                expect(BaseObject).to.respondTo('waitForAny');
            });
            it('should wait for promise\'s success or failure', function () {
                var obj = new BaseObject('window-0');
                var def1 = $.Deferred(), spy1 = sinon.spy();
                var def2 = $.Deferred(), spy2 = sinon.spy();
                obj.waitForAny(def1, spy1);
                obj.waitForAny(def2, spy2);
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(false);
                def1.resolve();
                def2.reject();
                expect(spy1.called).to.equal(true);
                expect(spy2.called).to.equal(true);
            });
        });

        // events -------------------------------------------------------------

        describe('method "listenTo"', function () {
            var obj = new BaseObject('window-0');
            it('should exist', function () {
                expect(BaseObject).to.respondTo('listenTo');
            });
            it('should call the event handler', function () {
                var src = $({}), spy = sinon.spy();
                obj.listenTo(src, 'test', spy);
                expect(spy.called).to.equal(false);
                src.trigger('wrong');
                expect(spy.called).to.equal(false);
                src.trigger('test');
                expect(spy.called).to.equal(true);
            });
            it('should call the event handler for multiple event types', function () {
                var src = $({}), spy = sinon.spy();
                obj.listenTo(src, 'test1 test2', spy);
                expect(spy.called).to.equal(false);
                src.trigger('wrong');
                expect(spy.called).to.equal(false);
                src.trigger('test1');
                expect(spy.calledOnce).to.equal(true);
                src.trigger('test2');
                expect(spy.calledTwice).to.equal(true);
            });
            it('should call the event handlers of an event map', function () {
                var src = $({}), spy1 = sinon.spy(), spy2 = sinon.spy();
                obj.listenTo(src, { test1: spy1, 'test1 test2': spy2 });
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(false);
                src.trigger('wrong');
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(false);
                src.trigger('test1');
                expect(spy1.calledOnce).to.equal(true);
                expect(spy2.calledOnce).to.equal(true);
                src.trigger('test2');
                expect(spy1.calledOnce).to.equal(true);
                expect(spy2.calledTwice).to.equal(true);
            });
        });

        describe('method "stopListeningTo"', function () {
            it('should exist', function () {
                expect(BaseObject).to.respondTo('stopListeningTo');
            });
            it('should unregister all event listeners', function () {
                var obj = new BaseObject('window-0'), src = $({}), spy1 = sinon.spy(), spy2 = sinon.spy();
                obj.listenTo(src, 'test1 test2', spy1).listenTo(src, 'test1 test2', spy2);
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(false);
                src.trigger('test1');
                expect(spy1.calledOnce).to.equal(true);
                expect(spy2.calledOnce).to.equal(true);
                obj.stopListeningTo(src);
                src.trigger('test1').trigger('test2');
                expect(spy1.calledOnce).to.equal(true);
                expect(spy2.calledOnce).to.equal(true);
            });
            it('should unregister a specific event type', function () {
                var obj = new BaseObject('window-0'), src = $({}), spy1 = sinon.spy(), spy2 = sinon.spy();
                obj.listenTo(src, 'test1 test2 test3', spy1).listenTo(src, 'test1 test2 test3', spy2);
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(false);
                src.trigger('test1');
                expect(spy1.calledOnce).to.equal(true);
                expect(spy2.calledOnce).to.equal(true);
                obj.stopListeningTo(src, 'test2 test3');
                src.trigger('test1').trigger('test2').trigger('test3');
                expect(spy1.calledTwice).to.equal(true);
                expect(spy2.calledTwice).to.equal(true);
            });
            it('should unregister a specific event listener', function () {
                var obj = new BaseObject('window-0'), src = $({}), spy1 = sinon.spy(), spy2 = sinon.spy();
                obj.listenTo(src, 'test1 test2', spy1).listenTo(src, 'test1 test2', spy2);
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(false);
                src.trigger('test1');
                expect(spy1.calledOnce).to.equal(true);
                expect(spy2.calledOnce).to.equal(true);
                obj.stopListeningTo(src, spy2);
                src.trigger('test1').trigger('test2');
                expect(spy1.calledThrice).to.equal(true);
                expect(spy2.calledOnce).to.equal(true);
            });
            it('should unregister a specific event type and listener', function () {
                var obj = new BaseObject('window-0'), src = $({}), spy1 = sinon.spy(), spy2 = sinon.spy();
                obj.listenTo(src, 'test1 test2 test3', spy1).listenTo(src, 'test1 test2 test3', spy2);
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(false);
                src.trigger('test1');
                expect(spy1.calledOnce).to.equal(true);
                expect(spy2.calledOnce).to.equal(true);
                obj.stopListeningTo(src, 'test2 test3', spy2);
                src.trigger('test1').trigger('test2').trigger('test3');
                expect(spy1.callCount).to.equal(4);
                expect(spy2.calledTwice).to.equal(true);
            });
            it('should unregister listeners from an event map', function () {
                var obj = new BaseObject('window-0'), src = $({}), spy1 = sinon.spy(), spy2 = sinon.spy();
                obj.listenTo(src, 'test1 test2 test3', spy1).listenTo(src, 'test1 test2 test3', spy2);
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(false);
                src.trigger('test1');
                expect(spy1.calledOnce).to.equal(true);
                expect(spy2.calledOnce).to.equal(true);
                obj.stopListeningTo(src, { test1: spy1, 'test2 test3': spy2 });
                src.trigger('test1').trigger('test2').trigger('test3');
                expect(spy1.calledThrice).to.equal(true);
                expect(spy2.calledTwice).to.equal(true);
            });
        });

        describe('method "waitForEvent"', function () {
            it('should exist', function () {
                expect(BaseObject).to.respondTo('waitForEvent');
            });
            it('should wait for an event', function () {
                var obj = new BaseObject('window-0'), src = $({}), spy = sinon.spy();
                var promise = obj.waitForEvent(src, 'test').done(spy);
                expect(promise.state()).to.equal('pending');
                clock.tick(10);
                expect(promise.state()).to.equal('pending');
                src.trigger('other');
                expect(promise.state()).to.equal('pending');
                src.trigger('test', [42, 'abc']);
                expect(promise.state()).to.equal('resolved');
                sinon.assert.calledOnce(spy);
                expect(spy.getCall(0).args[1]).to.equal(42);
                expect(spy.getCall(0).args[2]).to.equal('abc');
            });
            it('should reject on a timeout', function () {
                var obj = new BaseObject('window-0'), src = $({}), spy = sinon.spy();
                var promise = obj.waitForEvent(src, 'test', 15).fail(spy);
                expect(promise.state()).to.equal('pending');
                clock.tick(10);
                expect(promise.state()).to.equal('pending');
                src.trigger('other');
                expect(promise.state()).to.equal('pending');
                clock.tick(10);
                expect(promise.state()).to.equal('rejected');
                sinon.assert.calledOnce(spy);
                sinon.assert.calledWithExactly(spy, 'timeout');
            });
        });

        // timers -------------------------------------------------------------

        describe('method "executeDelayed"', function () {
            var obj = new BaseObject('window-0');
            it('should exist', function () {
                expect(BaseObject).to.respondTo('executeDelayed');
            });
            it('should invoke callback delayed', function () {
                var spy = sinon.spy();
                var timer = obj.executeDelayed(spy, 'msg');
                expect(spy.called).to.equal(false);
                expect(timer.state()).to.equal('pending');
                clock.tick(2);
                expect(timer.state()).to.equal('resolved');
                expect(spy.calledOnce).to.equal(true);
                expect(spy.calledOn(obj)).to.equal(true);
            });
            it('should invoke callbacks after specified delay time', function () {
                var spy = sinon.spy();
                obj.executeDelayed(spy, 'msg', { delay: 10 });
                expect(spy.called).to.equal(false);
                clock.tick(9);
                expect(spy.called).to.equal(false);
                clock.tick(2);
                expect(spy.called).to.equal(true);
            });
            it('should accept delay time directly as numeric parameter', function () {
                var spy = sinon.spy();
                obj.executeDelayed(spy, 'msg', 10);
                expect(spy.called).to.equal(false);
                clock.tick(9);
                expect(spy.called).to.equal(false);
                clock.tick(2);
                expect(spy.called).to.equal(true);
            });
            it('should invoke callbacks in correct order', function () {
                var spy1 = sinon.spy(), spy2 = sinon.spy();
                obj.executeDelayed(spy1, 'msg', 12); // fires at 12
                obj.executeDelayed(spy2, 'msg', 10); // fires at 10
                clock.tick(11); // timer fires at 10
                expect(spy1.called).to.equal(false);
                expect(spy2.called).to.equal(true);
                // move time to 13
                clock.tick(2); // timer fires at 12
                expect(spy1.called).to.equal(true);
            });
            it('should allow to abort a timer', function () {
                var spy = sinon.spy();
                var timer = obj.executeDelayed(spy, 'msg', 10);
                clock.tick(9);
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.called).to.equal(false);
            });
            it('should forward the callback result', function () {
                var stub = sinon.stub().returns(42);
                var spy = sinon.spy();
                var timer = obj.executeDelayed(stub, 'msg', 10).done(spy);
                clock.tick(20);
                expect(stub.called).to.equal(true);
                expect(timer.state()).to.equal('resolved');
                expect(spy.calledWith(42)).to.equal(true);
            });
            it('should defer timer promise to callback promise', function () {
                var def = $.Deferred();
                var stub = sinon.stub().returns(def.promise());
                var spy = sinon.spy();
                var timer = obj.executeDelayed(stub, 'msg', 10).done(spy);
                clock.tick(20);
                expect(stub.called).to.equal(true);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(false);
                def.resolve(42);
                expect(timer.state()).to.equal('resolved');
                expect(spy.calledWith(42)).to.equal(true);
            });
            it('should abort the nested promise', function () {
                var promise = obj.createAbortablePromise($.Deferred());
                var spy = sinon.spy(_.constant(promise));
                var timer = obj.executeDelayed(spy, 'msg', 10);
                clock.tick(9);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(false);
                clock.tick(2);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(true);
                expect(promise.state()).to.equal('pending');
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                expect(promise.state()).to.equal('rejected');
            });
            it('should not invoke callbacks after destruction', function () {
                var spy = sinon.spy();
                var timer = obj.executeDelayed(spy, 'msg', 10);
                clock.tick(9);
                obj.destroy();
                clock.tick(1000);
                expect(spy.called).to.equal(false);
                expect(timer.state()).to.equal('rejected');
            });
        });

        describe('method "repeatDelayed"', function () {
            var obj = new BaseObject('window-0');
            it('should exist', function () {
                expect(BaseObject).to.respondTo('repeatDelayed');
            });
            it('should invoke callback delayed and stop on Utils.BREAK', function () {
                var stub = sinon.stub();
                var timer = obj.repeatDelayed(stub, 'msg', { delay: 0 });
                stub.onThirdCall().returns(Utils.BREAK);
                expect(stub.called).to.equal(false);
                clock.tick(1000);
                expect(stub.calledThrice).to.equal(true);
                expect(stub.alwaysCalledOn(obj)).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should pass iteration index to callback', function () {
                var stub = sinon.stub();
                var timer = obj.repeatDelayed(stub, 'msg', { delay: 0 });
                stub.onThirdCall().returns(Utils.BREAK);
                clock.tick(1000);
                expect(stub.getCall(0).calledWith(0)).to.equal(true);
                expect(stub.getCall(1).calledWith(1)).to.equal(true);
                expect(stub.getCall(2).calledWith(2)).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should invoke callbacks after specified delay time', function () {
                var stub = sinon.stub();
                var timer = obj.repeatDelayed(stub, 'msg', { delay: 100 }); // fires at 100, 200, 300, ...
                stub.onThirdCall().returns(Utils.BREAK);
                clock.tick(99);
                expect(stub.called).to.equal(false);
                clock.tick(2);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(98);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(2);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(98);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(2);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should accept delay time directly as numeric parameter', function () {
                var stub = sinon.stub();
                var timer = obj.repeatDelayed(stub, 'msg', 100);
                stub.onThirdCall().returns(Utils.BREAK);
                clock.tick(99);
                expect(stub.called).to.equal(false);
                clock.tick(2);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(100);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(100);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should run first iteration synchronously', function () {
                var stub = sinon.stub();
                stub.onThirdCall().returns(Utils.BREAK);
                var timer = obj.repeatDelayed(stub, 'msg');
                expect(stub.calledOnce).to.equal(true);
                clock.tick(10000);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should use different repetition delay time', function () {
                var stub = sinon.stub();
                var timer = obj.repeatDelayed(stub, 'msg', { delay: 10, repeatDelay: 100 });
                stub.onThirdCall().returns(Utils.BREAK);
                clock.tick(9);
                expect(stub.called).to.equal(false);
                clock.tick(2);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(98);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(2);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(98);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(2);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should use specified number of cycles', function () {
                var stub = sinon.stub();
                var timer = obj.repeatDelayed(stub, 'msg', { delay: 100, cycles: 3 });
                clock.tick(99);
                expect(stub.called).to.equal(false);
                clock.tick(2);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(100);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(100);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should allow to abort a timer', function () {
                var spy = sinon.spy();
                var timer = obj.repeatDelayed(spy, 'msg', 100);
                clock.tick(250);
                expect(spy.calledTwice).to.equal(true);
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.calledTwice).to.equal(true);
            });
            it('should defer next iteration to callback promise', function () {
                var stub = sinon.stub();
                var def = $.Deferred();
                var timer = obj.repeatDelayed(stub, 'msg', 100);
                stub.onSecondCall().returns(def);
                stub.onThirdCall().returns(Utils.BREAK);
                clock.tick(150);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(100);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(200);
                expect(stub.calledTwice).to.equal(true);
                def.resolve();
                clock.tick(99);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(2);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should abort on rejected callback promise', function () {
                var stub = sinon.stub();
                var timer = obj.repeatDelayed(stub, 'msg', 100);
                stub.onThirdCall().returns($.Deferred().reject());
                clock.tick(250);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(100);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('rejected');
            });
            it('should abort the nested promise', function () {
                var promise = obj.createAbortablePromise($.Deferred());
                var spy = sinon.spy(_.constant(promise));
                var timer = obj.repeatDelayed(spy, 'msg', 100);
                clock.tick(99);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(false);
                clock.tick(2);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(true);
                expect(promise.state()).to.equal('pending');
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                expect(promise.state()).to.equal('rejected');
            });
            it('should reduce delay time after a pending promise in fastAsync mode', function () {
                var stub = sinon.stub();
                var def1 = $.Deferred();
                var def2 = $.Deferred();
                var timer = obj.repeatDelayed(stub, 'msg', { delay: 100, fastAsync: true });
                stub.onFirstCall().returns(def1);
                stub.onSecondCall().returns(def2);
                stub.onThirdCall().returns(Utils.BREAK);
                clock.tick(110);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(40);
                def1.resolve();
                clock.tick(40);
                expect(stub.calledOnce).to.equal(true);
                clock.tick(20);
                expect(stub.calledTwice).to.equal(true);
                clock.tick(200);
                expect(stub.calledTwice).to.equal(true);
                def2.resolve();
                clock.tick(1);
                expect(stub.calledThrice).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should not invoke callbacks after destruction', function () {
                var spy = sinon.spy();
                var timer = obj.repeatDelayed(spy, 'msg', 100);
                clock.tick(250);
                expect(spy.calledTwice).to.equal(true);
                obj.destroy();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.calledTwice).to.equal(true);
            });
        });

        describe('method "repeatSliced"', function () {

            // Creates a callback spy that simulates to execute for 150ms, returns 'ret'
            // for the first four invocations, and Utils.BREAK for the fifth invocation.
            function makeSpy(ret) {
                return sinon.spy(function (i) {
                    clock.tick(150);
                    return (i === 4) ? Utils.BREAK : ret;
                });
            }

            var obj = new BaseObject('window-0');

            it('should exist', function () {
                expect(BaseObject).to.respondTo('repeatSliced');
            });
            it('should invoke callback in time slices and stop on Utils.BREAK', function () {
                var spy = makeSpy(null);
                var timer = obj.repeatSliced(spy, 'msg', { delay: 0 });
                expect(spy.called).to.equal(false);
                // tick once to trigger first slice; spy will run twice (150 each)
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                // clock is now 1 after slice#1, next cycle starts in INT-1
                clock.tick(INT - 2);
                expect(spy.callCount).to.equal(2);
                // clock is now 1 before slice#2
                clock.tick(1);
                expect(spy.callCount).to.equal(4);
                // clock is now after slice#2, next cycle starts in INT
                clock.tick(INT - 1);
                expect(spy.callCount).to.equal(4);
                // clock is now 1 before slice#3
                clock.tick(1);
                expect(spy.callCount).to.equal(5);
                expect(spy.alwaysCalledOn(obj)).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should pass iteration index to callback', function () {
                var spy = makeSpy();
                var timer = obj.repeatSliced(spy, 'msg', { delay: 0 });
                // run all time slices at once
                clock.tick(10000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
                expect(spy.getCall(0).calledWith(0)).to.equal(true);
                expect(spy.getCall(1).calledWith(1)).to.equal(true);
                expect(spy.getCall(2).calledWith(2)).to.equal(true);
                expect(spy.getCall(3).calledWith(3)).to.equal(true);
                expect(spy.getCall(4).calledWith(4)).to.equal(true);
            });
            it('should invoke callbacks after specified delay/slice/interval times', function () {
                var spy = makeSpy();
                var timer = obj.repeatSliced(spy, 'msg', { delay: 10, slice: 200, interval: 100 });
                expect(spy.called).to.equal(false);
                clock.tick(9);
                expect(spy.called).to.equal(false);
                // clock is now at 9, tick once to trigger the first cycle
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                // clock is now after first slice, the next cycle starts in 100
                clock.tick(99);
                expect(spy.callCount).to.equal(2);
                clock.tick(1);
                expect(spy.callCount).to.equal(4);
                // clock is now after slice#2, the next cycle starts in 100
                clock.tick(100);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should use specified number of cycles', function () {
                var spy = makeSpy();
                var timer = obj.repeatSliced(spy, 'msg', { cycles: 3, interval: 100 });
                // tick once to trigger the first slice; spy will run twice (150 each)
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                // clock is now 1 after slice#1, next time slice starts in 99
                clock.tick(98);
                expect(spy.callCount).to.equal(2);
                // clock is now 1 before slice#2
                clock.tick(1);
                expect(spy.callCount).to.equal(3);
                expect(timer.state()).to.equal('resolved');
            });
            it('should run first iteration synchronously', function () {
                var spy = makeSpy();
                var timer = obj.repeatSliced(spy, 'msg');
                expect(spy.callCount).to.equal(2);
                clock.tick(10000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should allow to abort a timer', function () {
                var spy = makeSpy();
                var timer = obj.repeatSliced(spy, 'msg', { delay: 0 });
                // tick once to trigger the first cycle; spy will run twice (150 each)
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
            });
            it('should defer next iteration to callback promise', function () {
                var def = $.Deferred();
                var spy = makeSpy(def.promise());
                var timer = obj.repeatSliced(spy, 'msg', { delay: 0 });
                // tick once to trigger the first cycle; spy will run once (150 each) and return a pending promise
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                // loop will wait for the promise
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                // resolve the deferred object to let the loop continue
                def.resolve();
                expect(timer.state()).to.equal('pending');
                expect(spy.callCount).to.equal(1);
                // trigger the next cycles, spy will run twice because it returns a resolved promise
                clock.tick(1);
                expect(spy.callCount).to.equal(3);
                clock.tick(1000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should abort on rejected callback promise', function () {
                var def = $.Deferred();
                var spy = makeSpy(def.promise());
                var timer = obj.repeatSliced(spy, 'msg', { delay: 0 });
                // tick once to trigger the first cycle; spy will run once (150 each) and return a pending promise
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                // clock is now 1 after first cycle, loop will wait for the promise
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                // reject the deferred object
                def.reject();
                expect(spy.callCount).to.equal(1);
                expect(timer.state()).to.equal('rejected');
                // loop must not continue after a rejected promise
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
            });
            it('should abort the nested promise', function () {
                var promise = obj.createAbortablePromise($.Deferred());
                var spy = sinon.spy(_.constant(promise));
                var timer = obj.repeatSliced(spy, 'msg', { delay: 10 });
                clock.tick(9);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(false);
                clock.tick(2);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(true);
                expect(promise.state()).to.equal('pending');
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                expect(promise.state()).to.equal('rejected');
            });
            it('should not invoke callbacks after destruction', function () {
                var spy = makeSpy();
                var timer = obj.repeatSliced(spy, 'msg', { delay: 0 });
                // tick once to trigger the first cycle; spy will run twice (150 each)
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                // destroy the object
                obj.destroy();
                expect(timer.state()).to.equal('rejected');
                // loop must not continue after destruction
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
            });
        });

        describe('method "iterateSliced"', function () {

            // See test cases for BaseObject.repeatSliced() above for details about how
            // these tests work.

            // Creates a callback spy that simulates to execute for 150ms, and returns 'ret'.
            function makeSpy(ret) {
                return sinon.spy(function () {
                    clock.tick(150);
                    return ret;
                });
            }

            var obj = new BaseObject('window-0');

            it('should exist', function () {
                expect(BaseObject).to.respondTo('iterateSliced');
            });
            it('should invoke callback in time slices', function () {
                var spy = makeSpy();
                var timer = obj.iterateSliced(IteratorUtils.createIndexIterator(5), spy, 'msg', { delay: 0 });
                expect(spy.called).to.equal(false);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                clock.tick(INT - 2);
                expect(spy.callCount).to.equal(2);
                clock.tick(1);
                expect(spy.callCount).to.equal(4);
                clock.tick(INT - 1);
                expect(spy.callCount).to.equal(4);
                clock.tick(1);
                expect(spy.callCount).to.equal(5);
                expect(spy.alwaysCalledOn(obj)).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should stop on Utils.BREAK', function () {
                var stub = sinon.stub();
                stub.onSecondCall().returns(Utils.BREAK);
                var timer = obj.iterateSliced(IteratorUtils.createIndexIterator(5), stub, 'msg', { delay: 0 });
                expect(stub.called).to.equal(false);
                clock.tick(1);
                expect(stub.callCount).to.equal(2);
                expect(timer.state()).to.equal('resolved');
            });
            it('should pass iterator results to callback', function () {
                var spy = makeSpy();
                var timer = obj.iterateSliced(IteratorUtils.createIndexIterator(5), spy, 'msg', { delay: 0 });
                clock.tick(1000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
                expect(spy.getCall(0).calledWith(0, { done: false, value: 0 })).to.equal(true);
                expect(spy.getCall(1).calledWith(1, { done: false, value: 1 })).to.equal(true);
                expect(spy.getCall(2).calledWith(2, { done: false, value: 2 })).to.equal(true);
                expect(spy.getCall(3).calledWith(3, { done: false, value: 3 })).to.equal(true);
                expect(spy.getCall(4).calledWith(4, { done: false, value: 4 })).to.equal(true);
            });
            it('should invoke callbacks after specified delay/slice/interval time', function () {
                var spy = makeSpy();
                var timer = obj.iterateSliced(IteratorUtils.createIndexIterator(5), spy, 'msg', { delay: 10, slice: 200, interval: 100 });
                expect(spy.called).to.equal(false);
                clock.tick(9);
                expect(spy.called).to.equal(false);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                clock.tick(100);
                expect(spy.callCount).to.equal(4);
                clock.tick(100);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should run first iteration synchronously', function () {
                var spy = makeSpy();
                var timer = obj.iterateSliced(IteratorUtils.createIndexIterator(5), spy, 'msg');
                expect(spy.callCount).to.equal(2);
                clock.tick(10000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should allow to abort a timer', function () {
                var spy = makeSpy();
                var timer = obj.iterateSliced(IteratorUtils.createIndexIterator(5), spy, 'msg', { delay: 0 });
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
            });
            it('should defer next iteration to callback promise', function () {
                var def = $.Deferred();
                var spy = makeSpy(def.promise());
                var timer = obj.iterateSliced(IteratorUtils.createIndexIterator(5), spy, 'msg', { delay: 0 });
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.resolve();
                expect(timer.state()).to.equal('pending');
                clock.tick(1);
                expect(spy.callCount).to.equal(3);
                clock.tick(INT - 1);
                expect(spy.callCount).to.equal(5);
                // last array element visited, but iterator did not return the final {done:true} yet
                expect(timer.state()).to.equal('pending');
                clock.tick(INT);
                expect(timer.state()).to.equal('resolved');
            });
            it('should abort on rejected callback promise', function () {
                var def = $.Deferred();
                var spy = makeSpy(def.promise());
                var timer = obj.iterateSliced(IteratorUtils.createIndexIterator(5), spy, 'msg', { delay: 0 });
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.reject();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
            });
            it('should abort the nested promise', function () {
                var promise = obj.createAbortablePromise($.Deferred());
                var spy = sinon.spy(_.constant(promise));
                var timer = obj.iterateSliced(IteratorUtils.createIndexIterator(5), spy, 'msg', { delay: 10 });
                clock.tick(9);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(false);
                clock.tick(2);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(true);
                expect(promise.state()).to.equal('pending');
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                expect(promise.state()).to.equal('rejected');
            });
            it('should not invoke callbacks after destruction', function () {
                var spy = makeSpy();
                var timer = obj.iterateSliced(IteratorUtils.createIndexIterator(5), spy, 'msg', { delay: 0 });
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                obj.destroy();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
            });
        });

        describe('method "reduceSliced"', function () {

            // See test cases for BaseObject.repeatSliced() above for details about how
            // these tests work.

            // Creates a callback spy that simulates to execute for 150ms, and returns 'ret'.
            function makeSpy(ret) {
                return sinon.spy(function () {
                    clock.tick(150);
                    return ret;
                });
            }

            // Creates a callback spy that simulates to execute for 150ms, and returns the sum of its arguments
            function makeSumSpy() {
                return sinon.spy(function (a, b) {
                    clock.tick(150);
                    return a + b;
                });
            }

            var obj = new BaseObject('window-0');

            it('should exist', function () {
                expect(BaseObject).to.respondTo('reduceSliced');
            });
            it('should invoke callback in time slices', function () {
                var spy = makeSpy();
                var timer = obj.reduceSliced(1, IteratorUtils.createIntervalIterator(2, 6), spy, 'msg', { delay: 0 });
                expect(spy.called).to.equal(false);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                clock.tick(INT - 2);
                expect(spy.callCount).to.equal(2);
                clock.tick(1);
                expect(spy.callCount).to.equal(4);
                clock.tick(INT - 1);
                expect(spy.callCount).to.equal(4);
                clock.tick(1);
                expect(spy.callCount).to.equal(5);
                expect(spy.alwaysCalledOn(obj)).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should pass correct values to callback function', function (done) {
                var spy = makeSumSpy();
                var timer = obj.reduceSliced(1, IteratorUtils.createIntervalIterator(2, 6), spy, 'msg', { delay: 0 });
                clock.tick(10000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
                expect(spy.getCall(0).calledWith(1, 2, { done: false, value: 2 })).to.equal(true);
                expect(spy.getCall(1).calledWith(3, 3, { done: false, value: 3 })).to.equal(true);
                expect(spy.getCall(2).calledWith(6, 4, { done: false, value: 4 })).to.equal(true);
                expect(spy.getCall(3).calledWith(10, 5, { done: false, value: 5 })).to.equal(true);
                expect(spy.getCall(4).calledWith(15, 6, { done: false, value: 6 })).to.equal(true);
                timer.done(function (result) { expect(result).to.equal(21); done(); });
            });
            it('should allow to abort a timer', function () {
                var spy = makeSpy();
                var timer = obj.reduceSliced(1, IteratorUtils.createIntervalIterator(2, 6), spy, 'msg', { delay: 0 });
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
            });
            it('should defer next iteration to callback promise', function () {
                var def = $.Deferred();
                var spy = makeSpy(def.promise());
                var timer = obj.reduceSliced(1, IteratorUtils.createIntervalIterator(2, 6), spy, 'msg', { delay: 0 });
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.resolve();
                expect(timer.state()).to.equal('pending');
                clock.tick(1);
                expect(spy.callCount).to.equal(3);
                clock.tick(INT - 1);
                expect(spy.callCount).to.equal(5);
                // last array element visited, but iterator did not return the final {done:true} yet
                expect(timer.state()).to.equal('pending');
                clock.tick(INT);
                expect(timer.state()).to.equal('resolved');
            });
            it('should abort on rejected callback promise', function () {
                var def = $.Deferred();
                var spy = makeSpy(def.promise());
                var timer = obj.reduceSliced(1, IteratorUtils.createIntervalIterator(2, 6), spy, 'msg', { delay: 0 });
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.reject();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
            });
            it('should not invoke callbacks after destruction', function () {
                var spy = makeSpy();
                var timer = obj.reduceSliced(1, IteratorUtils.createIntervalIterator(2, 6), spy, 'msg', { delay: 0 });
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                obj.destroy();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
            });
        });

        describe('method "iterateArraySliced"', function () {

            // See test cases for BaseObject.repeatSliced() above for details about how
            // these tests work.

            // Creates a callback spy that simulates to execute for 150ms, and returns 'ret'.
            function makeSpy(ret) {
                return sinon.spy(function () {
                    clock.tick(150);
                    return ret;
                });
            }

            var ARRAY = ['a', 'b', 'c', 'd', 'e'];
            var obj = new BaseObject('window-0');

            it('should exist', function () {
                expect(BaseObject).to.respondTo('iterateArraySliced');
            });
            it('should return a resolved promise, if an empty array is given', function () {
                var spy = sinon.spy();
                var timer = obj.iterateArraySliced([], _.noop, 'msg', { delay: 0 });
                expect(timer).to.be.an('object');
                expect(timer.state()).to.equal('resolved');
                expect(spy.callCount).to.equal(0);
            });
            it('should invoke callback in time slices', function () {
                var spy = makeSpy();
                var timer = obj.iterateArraySliced(ARRAY, spy, 'msg', { delay: 0 });
                expect(spy.called).to.equal(false);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                clock.tick(INT - 2);
                expect(spy.callCount).to.equal(2);
                clock.tick(1);
                expect(spy.callCount).to.equal(4);
                clock.tick(INT - 1);
                expect(spy.callCount).to.equal(4);
                clock.tick(1);
                expect(spy.callCount).to.equal(5);
                expect(spy.alwaysCalledOn(obj)).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should stop on Utils.BREAK', function () {
                var stub = sinon.stub();
                stub.onSecondCall().returns(Utils.BREAK);
                var timer = obj.iterateArraySliced(ARRAY, stub, 'msg', { delay: 0 });
                expect(stub.called).to.equal(false);
                clock.tick(1);
                expect(stub.callCount).to.equal(2);
                expect(timer.state()).to.equal('resolved');
            });
            it('should pass array elements to callback', function () {
                var spy = makeSpy();
                var timer = obj.iterateArraySliced(ARRAY, spy, 'msg', { delay: 0 });
                clock.tick(10000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
                expect(spy.getCall(0).calledWith('a', 0, ARRAY)).to.equal(true);
                expect(spy.getCall(1).calledWith('b', 1, ARRAY)).to.equal(true);
                expect(spy.getCall(2).calledWith('c', 2, ARRAY)).to.equal(true);
                expect(spy.getCall(3).calledWith('d', 3, ARRAY)).to.equal(true);
                expect(spy.getCall(4).calledWith('e', 4, ARRAY)).to.equal(true);
            });
            it('should process array elements in reversed order', function () {
                var spy = makeSpy();
                var timer = obj.iterateArraySliced(ARRAY, spy, 'msg', { reverse: true });
                clock.tick(10000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
                expect(spy.getCall(0).calledWith('e', 4, ARRAY)).to.equal(true);
                expect(spy.getCall(1).calledWith('d', 3, ARRAY)).to.equal(true);
                expect(spy.getCall(2).calledWith('c', 2, ARRAY)).to.equal(true);
                expect(spy.getCall(3).calledWith('b', 1, ARRAY)).to.equal(true);
                expect(spy.getCall(4).calledWith('a', 0, ARRAY)).to.equal(true);
            });
            it('should work with array-like objects', function () {
                var spy = makeSpy();
                var timer = obj.iterateArraySliced('abcde', spy, 'msg', { delay: 0 });
                clock.tick(10000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
                expect(spy.getCall(0).calledWith('a', 0, 'abcde')).to.equal(true);
                expect(spy.getCall(1).calledWith('b', 1, 'abcde')).to.equal(true);
                expect(spy.getCall(2).calledWith('c', 2, 'abcde')).to.equal(true);
                expect(spy.getCall(3).calledWith('d', 3, 'abcde')).to.equal(true);
                expect(spy.getCall(4).calledWith('e', 4, 'abcde')).to.equal(true);
            });
            it('should invoke callbacks after specified delay/slice/interval time', function () {
                var spy = makeSpy();
                var timer = obj.iterateArraySliced(ARRAY, spy, 'msg', { delay: 10, slice: 200, interval: 100 });
                expect(spy.called).to.equal(false);
                clock.tick(9);
                expect(spy.called).to.equal(false);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                clock.tick(100);
                expect(spy.callCount).to.equal(4);
                clock.tick(100);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should run first iteration synchronously', function () {
                var spy = makeSpy();
                var timer = obj.iterateArraySliced(ARRAY, spy, 'msg');
                expect(spy.callCount).to.equal(2);
                clock.tick(10000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should allow to abort a timer', function () {
                var spy = makeSpy();
                var timer = obj.iterateArraySliced(ARRAY, spy, 'msg', { delay: 0 });
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
            });
            it('should defer next iteration to callback promise', function () {
                var def = $.Deferred();
                var spy = makeSpy(def.promise());
                var timer = obj.iterateArraySliced(ARRAY, spy, 'msg', { delay: 0 });
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.resolve();
                expect(timer.state()).to.equal('pending');
                clock.tick(1);
                expect(spy.callCount).to.equal(3);
                clock.tick(INT - 1);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should abort on rejected callback promise', function () {
                var def = $.Deferred();
                var spy = makeSpy(def.promise());
                var timer = obj.iterateArraySliced(ARRAY, spy, 'msg', { delay: 0 });
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.reject();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
            });
            it('should abort the nested promise', function () {
                var promise = obj.createAbortablePromise($.Deferred());
                var spy = sinon.spy(_.constant(promise));
                var timer = obj.iterateArraySliced(ARRAY, spy, 'msg', { delay: 10 });
                clock.tick(9);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(false);
                clock.tick(2);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(true);
                expect(promise.state()).to.equal('pending');
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                expect(promise.state()).to.equal('rejected');
            });
            it('should not invoke callbacks after destruction', function () {
                var spy = makeSpy();
                var timer = obj.iterateArraySliced(ARRAY, spy, 'msg', { delay: 0 });
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                obj.destroy();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
            });
        });

        describe('method "iterateObjectSliced"', function () {

            // See test cases for BaseObject.repeatSliced() above for details about how
            // these tests work.

            // Creates a callback spy that simulates to execute for 150ms, and returns 'ret'.
            function makeSpy(ret) {
                return sinon.spy(function () {
                    clock.tick(150);
                    return ret;
                });
            }

            var OBJECT = { a: 1, b: 2, c: 3, d: 4, e: 5 };
            var obj = new BaseObject('window-0');

            it('should exist', function () {
                expect(BaseObject).to.respondTo('iterateObjectSliced');
            });
            it('should invoke callback in time slices', function () {
                var spy = makeSpy();
                var timer = obj.iterateObjectSliced(OBJECT, spy, 'msg', { delay: 0 });
                expect(spy.called).to.equal(false);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                clock.tick(INT - 2);
                expect(spy.callCount).to.equal(2);
                clock.tick(1);
                expect(spy.callCount).to.equal(4);
                clock.tick(INT - 1);
                expect(spy.callCount).to.equal(4);
                clock.tick(1);
                expect(spy.callCount).to.equal(5);
                expect(spy.alwaysCalledOn(obj)).to.equal(true);
                expect(timer.state()).to.equal('resolved');
            });
            it('should stop on Utils.BREAK', function () {
                var stub = sinon.stub();
                stub.onSecondCall().returns(Utils.BREAK);
                var timer = obj.iterateObjectSliced(OBJECT, stub, 'msg', { delay: 0 });
                expect(stub.called).to.equal(false);
                clock.tick(1);
                expect(stub.callCount).to.equal(2);
                expect(timer.state()).to.equal('resolved');
            });
            it('should pass object properties to callback', function () {
                var spy = makeSpy();
                var timer = obj.iterateObjectSliced(OBJECT, spy, 'msg', { delay: 0 });
                clock.tick(10000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
                expect(spy.getCall(0).calledWith(1, 'a', OBJECT)).to.equal(true);
                expect(spy.getCall(1).calledWith(2, 'b', OBJECT)).to.equal(true);
                expect(spy.getCall(2).calledWith(3, 'c', OBJECT)).to.equal(true);
                expect(spy.getCall(3).calledWith(4, 'd', OBJECT)).to.equal(true);
                expect(spy.getCall(4).calledWith(5, 'e', OBJECT)).to.equal(true);
            });
            it('should invoke callbacks after specified delay/slice/interval time', function () {
                var spy = makeSpy();
                var timer = obj.iterateObjectSliced(OBJECT, spy, 'msg', { delay: 10, slice: 200, interval: 100 });
                expect(spy.called).to.equal(false);
                clock.tick(9);
                expect(spy.called).to.equal(false);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                clock.tick(100);
                expect(spy.callCount).to.equal(4);
                clock.tick(100);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should run first iteration synchronously', function () {
                var spy = makeSpy();
                var timer = obj.iterateObjectSliced(OBJECT, spy, 'msg');
                expect(spy.callCount).to.equal(2);
                clock.tick(10000);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should allow to abort a timer', function () {
                var spy = makeSpy();
                var timer = obj.iterateObjectSliced(OBJECT, spy, 'msg', { delay: 0 });
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
            });
            it('should defer next iteration to callback promise', function () {
                var def = $.Deferred();
                var spy = makeSpy(def.promise());
                var timer = obj.iterateObjectSliced(OBJECT, spy, 'msg', { delay: 0 });
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.resolve();
                expect(timer.state()).to.equal('pending');
                clock.tick(1);
                expect(spy.callCount).to.equal(3);
                clock.tick(INT - 1);
                expect(spy.callCount).to.equal(5);
                expect(timer.state()).to.equal('resolved');
            });
            it('should abort on rejected callback promise', function () {
                var def = $.Deferred();
                var spy = makeSpy(def.promise());
                var timer = obj.iterateObjectSliced(OBJECT, spy, 'msg', { delay: 0 });
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
                def.reject();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(1);
            });
            it('should abort the nested promise', function () {
                var promise = obj.createAbortablePromise($.Deferred());
                var spy = sinon.spy(_.constant(promise));
                var timer = obj.iterateObjectSliced(OBJECT, spy, 'msg', { delay: 10 });
                clock.tick(9);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(false);
                clock.tick(2);
                expect(timer.state()).to.equal('pending');
                expect(spy.called).to.equal(true);
                expect(promise.state()).to.equal('pending');
                timer.abort();
                expect(timer.state()).to.equal('rejected');
                expect(promise.state()).to.equal('rejected');
            });
            it('should not invoke callbacks after destruction', function () {
                var spy = makeSpy();
                var timer = obj.iterateObjectSliced(OBJECT, spy, 'msg', { delay: 0 });
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
                obj.destroy();
                expect(timer.state()).to.equal('rejected');
                clock.tick(1000);
                expect(spy.callCount).to.equal(2);
            });
        });

        // method generators --------------------------------------------------

        describe('method "createSynchronizedMethod"', function () {
            var obj = new BaseObject('window-0');
            it('should exist', function () {
                expect(BaseObject).to.respondTo('createSynchronizedMethod');
            });
            it('should invoke callback serialized', function () {
                var defs = _.times(3, function () { return new $.Deferred(); });
                var stub = sinon.stub();
                var method = obj.createSynchronizedMethod('name', stub);
                defs.forEach(function (def, index) { stub.onCall(index).returns(def); });
                expect(stub.callCount).to.equal(0);
                var promise0 = method();
                expect(stub.callCount).to.equal(1);
                expect(promise0.state()).to.equal('pending');
                var promise1 = method();
                expect(stub.callCount).to.equal(1);
                expect(promise1.state()).to.equal('pending');
                var promise2 = method();
                expect(stub.callCount).to.equal(1);
                expect(promise2.state()).to.equal('pending');
                clock.tick(100);
                defs[0].resolve();
                expect(stub.callCount).to.equal(1);
                clock.tick(100);
                expect(stub.callCount).to.equal(2);
                expect(promise0.state()).to.equal('resolved');
                expect(promise1.state()).to.equal('pending');
                clock.tick(100);
                defs[2].reject();
                expect(stub.callCount).to.equal(2);
                expect(promise1.state()).to.equal('pending');
                expect(promise2.state()).to.equal('pending');
                clock.tick(100);
                defs[1].resolve();
                expect(stub.callCount).to.equal(2);
                clock.tick(100);
                expect(stub.callCount).to.equal(3);
                expect(promise1.state()).to.equal('resolved');
                expect(promise2.state()).to.equal('rejected');
            });
            it('should pass parameters to callback', function () {
                var defs = _.times(3, function () { return new $.Deferred(); });
                var stub = sinon.stub();
                var method = obj.createSynchronizedMethod('name', stub);
                defs.forEach(function (def, index) { stub.onCall(index).returns(def); });
                expect(stub.callCount).to.equal(0);
                method(1, 'a');
                method(2, 'b');
                method(3, 'c');
                clock.tick(20);
                defs[0].resolve();
                clock.tick(100);
                defs[1].resolve();
                clock.tick(100);
                defs[2].resolve();
                clock.tick(100);
                expect(stub.callCount).to.equal(3);
                expect(stub.getCall(0).calledWith(1, 'a')).to.equal(true);
                expect(stub.getCall(1).calledWith(2, 'b')).to.equal(true);
                expect(stub.getCall(2).calledWith(3, 'c')).to.equal(true);
            });
            it('should invoke callback with current context', function () {
                var spy = sinon.spy();
                var method = obj.createSynchronizedMethod('name', spy);
                method.call(obj);
                expect(spy.called).to.equal(true);
                expect(spy.calledOn(obj)).to.equal(true);
            });
            it('should synchronize recursive invocations', function () {
                var running = false;
                var spy = sinon.spy(function (i) {
                    expect(running).to.equal(false);
                    running = true;
                    if (i < 3) { method(i + 1); }
                    running = false;
                });
                var method = obj.createSynchronizedMethod('name', spy);
                method(1);
            });
            it('should not invoke callbacks after destruction', function () {
                var defs = _.times(3, function () { return new $.Deferred(); });
                var stub = sinon.stub();
                var method = obj.createSynchronizedMethod('name', stub);
                _.each(defs, function (def, index) { stub.onCall(index).returns(def); });
                var promises = _.times(defs.length, method);
                clock.tick(100);
                defs[0].resolve();
                clock.tick(100);
                expect(stub.callCount).to.equal(2);
                expect(promises[0].state()).to.equal('resolved');
                clock.tick(100);
                expect(promises[1].state()).to.equal('pending');
                expect(promises[2].state()).to.equal('pending');
                obj.destroy();
                defs[1].resolve();
                clock.tick(1000);
                defs[2].resolve();
                expect(stub.callCount).to.equal(2);
                expect(promises[1].state()).to.equal('rejected');
                expect(promises[2].state()).to.equal('rejected');
            });
        });

        describe('method "createDebouncedMethod"', function () {
            var obj = new BaseObject('window-0');
            it('should exist', function () {
                expect(BaseObject).to.respondTo('createDebouncedMethod');
            });
            it('should invoke callback debounced', function () {
                var spy1 = sinon.spy(), spy2 = sinon.spy();
                var method = obj.createDebouncedMethod('name', spy1, spy2);
                expect(spy1.callCount).to.equal(0);
                expect(spy2.callCount).to.equal(0);
                method();
                expect(spy1.callCount).to.equal(1);
                expect(spy2.callCount).to.equal(0);
                method();
                expect(spy1.callCount).to.equal(2);
                expect(spy2.callCount).to.equal(0);
                method();
                expect(spy1.callCount).to.equal(3);
                expect(spy2.callCount).to.equal(0);
                clock.tick(10);
                expect(spy1.callCount).to.equal(3);
                expect(spy2.callCount).to.equal(1);
            });
            it('should pass parameters to callbacks', function () {
                var spy1 = sinon.spy(_.identity), spy2 = sinon.spy();
                var method = obj.createDebouncedMethod('name', spy1, spy2);
                method('a', 1);
                method('b', 2);
                method('c', 3);
                clock.tick(10);
                expect(spy1.callCount).to.equal(3);
                expect(spy1.getCall(0).calledWith('a', 1)).to.equal(true);
                expect(spy1.getCall(1).calledWith('b', 2)).to.equal(true);
                expect(spy1.getCall(2).calledWith('c', 3)).to.equal(true);
                expect(spy2.callCount).to.equal(1);
                expect(spy2.getCall(0).calledWith('c')).to.equal(true);
            });
            it('should invoke callbacks with current context', function () {
                var spy = sinon.spy();
                var method = obj.createDebouncedMethod('name', spy, spy);
                method.call(obj);
                clock.tick(10);
                expect(spy.calledTwice).to.equal(true);
                expect(spy.alwaysCalledOn(obj)).to.equal(true);
            });
            it('should use specified delay time', function () {
                var spy = sinon.spy();
                var method = obj.createDebouncedMethod('name', _.noop, spy, { delay: 10 });
                method();
                clock.tick(2);
                expect(spy.callCount).to.equal(0);
                method();
                clock.tick(9);
                expect(spy.callCount).to.equal(0);
                method();
                clock.tick(9);
                expect(spy.callCount).to.equal(0);
                method();
                clock.tick(9);
                expect(spy.callCount).to.equal(0);
                method();
                clock.tick(11);
                expect(spy.callCount).to.equal(1);
            });
            it('should use specified maximum delay time', function () {
                var spy = sinon.spy();
                var method = obj.createDebouncedMethod('name', _.noop, spy, { delay: 100, maxDelay: 300 });
                method();
                clock.tick(99);
                method();
                clock.tick(99);
                method();
                clock.tick(99);
                expect(spy.callCount).to.equal(0);
                method();
                clock.tick(2);
                expect(spy.callCount).to.equal(0);
                clock.tick(1);
                expect(spy.callCount).to.equal(1);
                method();
                clock.tick(99);
                expect(spy.callCount).to.equal(1);
                clock.tick(1);
                expect(spy.callCount).to.equal(2);
            });
            it('should accept null as direct callback', function () {
                obj.method = obj.createDebouncedMethod('name', null, _.noop);
                expect(obj.method()).to.equal(obj);
            });
            it('should not invoke callback after destruction', function () {
                var spy = sinon.spy();
                var method = obj.createDebouncedMethod('name', _.noop, spy);
                method();
                obj.destroy();
                clock.tick(1000);
                expect(spy.callCount).to.equal(0);
            });
        });

        // destruction --------------------------------------------------------

        describe('method "destroy"', function () {

            var obj = new BaseObject('window-0');
            var src1 = $({}), src2 = $({});
            var def = $.Deferred();
            var spy1 = sinon.spy();
            var spy2 = sinon.spy();
            var spy3 = sinon.spy();

            obj.registerDestructor(spy1);
            obj.registerDestructor(spy2);
            obj.listenTo(src1, 'test1', spy3);
            obj.listenTo(src2, 'test2', spy3);
            obj.waitForSuccess(def, spy3);

            it('should exist', function () {
                expect(BaseObject).to.respondTo('destroy');
            });
            it('should call the registered destructors in reversed order', function () {
                obj.myProperty = true;
                obj.registerDestructor(function () {
                    expect(obj.destroying).to.equal(true);
                });
                expect(obj.destroyed).to.be.undefined;
                obj.destroy();
                expect(obj.destroyed).to.equal(true);
                expect(spy1.called).to.equal(true);
                expect(spy2.called).to.equal(true);
                expect(spy2.calledBefore(spy1)).to.equal(true);
            });
            it('should delete the prototype and public methods', function () {
                expect(obj).not.to.have.a.property('myProperty');
            });
            it('should release event handlers', function () {
                src1.trigger('test1');
                expect(spy3.called).to.equal(false);
            });
            it('should release promise handlers', function () {
                def.resolve();
                expect(spy3.called).to.equal(false);
            });
        });
    });

    // ========================================================================
});
