/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/container/valueset'
], function (ValueSet) {

    'use strict';

    // class ValueSet ========================================================

    describe('Toolkit class ValueSet', function () {

        it('should exist', function () {
            expect(ValueSet).to.be.a('function');
        });

        // static methods -----------------------------------------------------

        describe('method "pluck"', function () {
            it('should exist', function () {
                expect(ValueSet).itself.to.respondTo('pluck');
            });
            it('should create a set from an array', function () {
                var set = ValueSet.pluck(_.identity, [{ p: 'a' }, { p: 'b' }, { p: 'c' }, { p: 'a', q: 42 }], 'p');
                expect(set).to.be.an.instanceof(ValueSet);
                expect(set._cont).to.deep.equal({ a: 'a', b: 'b', c: 'c' });
            });
            var OBJ = { a: { p: 1 }, b: { p: 2 }, c: { p: 3 }, d: { p: 4, q: 42 } };
            var EXP = { 1: 1, 2: 2, 3: 3, 4: 4 };
            it('should create a set from an object', function () {
                var set = ValueSet.pluck(_.identity, OBJ, 'p');
                expect(set._cont).to.deep.equal(EXP);
            });
        });

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a set from an array', function () {
                var set = new ValueSet(_.identity, ['a', 'b', 'c', 'a']);
                expect(set._cont).to.deep.equal({ a: 'a', b: 'b', c: 'c' });
            });
            var OBJ = { a: 1, b: 2, c: 3, d: 1 };
            var EXP = { 1: 1, 2: 2, 3: 3 };
            it('should create a set from an object', function () {
                var set = new ValueSet(_.identity, OBJ);
                expect(set._cont).to.deep.equal(EXP);
            });
            it('should create a set from another set', function () {
                var set = new ValueSet(_.identity, new ValueSet(_.identity, OBJ));
                expect(set._cont).to.deep.equal(EXP);
            });
            it('should create a set with a specific keyer function', function () {
                var set = new ValueSet(function (value) { return 'k' + value; }, OBJ);
                expect(set._cont).to.deep.equal({ k1: 1, k2: 2, k3: 3 });
            });
            it('should create a set with a specific keyer property', function () {
                var set = new ValueSet('p', [{ p: 'a' }, { p: 'b' }]);
                expect(set._cont).to.deep.equal({ a: { p: 'a' }, b: { p: 'b' } });
            });
            it('should create a set with a specific keyer method', function () {
                var obj1 = { p: _.constant('a') };
                var obj2 = { p: _.constant('b') };
                var set = new ValueSet('p()', [obj1, obj2]);
                expect(set._cont).to.have.a.property('a', obj1);
                expect(set._cont).to.have.a.property('b', obj2);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "empty"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('empty');
            });
            it('should return whether the set is empty', function () {
                var set = new ValueSet(_.identity);
                expect(set.empty()).to.equal(true);
                set.insert('a');
                expect(set.empty()).to.equal(false);
            });
        });

        describe('method "single"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('single');
            });
            it('should return whether the set contains a single element', function () {
                var set = new ValueSet(_.identity);
                expect(set.single()).to.equal(false);
                set.insert('a');
                expect(set.single()).to.equal(true);
                set.insert('b');
                expect(set.single()).to.equal(false);
            });
        });

        describe('method "clear"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('clear');
            });
            it('should clear the set', function () {
                var set = new ValueSet(_.identity, ['a']);
                expect(set.empty()).to.equal(false);
                expect(set.clear()).to.equal(set);
                expect(set.empty()).to.equal(true);
            });
        });

        describe('method "has"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('has');
            });
            it('should return whether the element exists', function () {
                var set = new ValueSet('p', [{ p: 'a' }]);
                expect(set.has({ p: 'a' })).to.equal(true);
                expect(set.has({ p: 'b' })).to.equal(false);
            });
        });

        describe('method "hasKey"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('hasKey');
            });
            it('should return whether the element exists', function () {
                var set = new ValueSet('p', [{ p: 'a' }]);
                expect(set.hasKey('a')).to.equal(true);
                expect(set.hasKey('b')).to.equal(false);
            });
        });

        describe('method "get"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('get');
            });
            it('should return an element', function () {
                var obj = { p: 'a' };
                var set = new ValueSet('p', [obj]);
                expect(set.get('a')).to.equal(obj);
                expect(set.get('b')).to.equal(undefined);
                expect(set.get('a', null)).to.equal(obj);
                expect(set.get('b', null)).to.equal(null);
            });
        });

        describe('method "getAny"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('getAny');
            });
            it('should return an element', function () {
                var set = new ValueSet(_.identity);
                expect(set.getAny()).to.equal(undefined);
                set.insert('a');
                set.insert('b');
                set.insert('c');
                var v1 = set.getAny();
                expect(v1).to.be.a('string');
                var v2 = set.getAny();
                expect(v2).to.equal(v1);
            });
        });

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('clone');
            });
            var set = new ValueSet(_.identity);
            set.insert('a');
            it('should create a shallow clone', function () {
                var clone = set.clone();
                expect(clone.get('a')).to.equal('a');
                expect(clone.get('b')).to.equal(undefined);
            });
            it('should create a deep clone', function () {
                var spy = sinon.spy(_.constant(42));
                var context = {};
                var clone = set.clone(spy, context);
                expect(clone.get('a')).to.equal(undefined);
                expect(clone.get('42')).to.equal(42);
                sinon.assert.alwaysCalledOn(spy, context);
            });
        });

        describe('method "insert"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('insert');
            });
            it('should insert set elements', function () {
                var obj1 = { p: 'a' }, obj2 = { p: 'b' };
                var set = new ValueSet('p');
                expect(set.has(obj1)).to.equal(false);
                expect(set.insert(obj1)).to.equal(obj1);
                expect(set.has(obj1)).to.equal(true);
                expect(set.get('a')).to.equal(obj1);
                expect(set.insert(obj2)).to.equal(obj2);
                expect(set.get('a')).to.equal(obj1);
                expect(set.get('b')).to.equal(obj2);
            });
        });

        describe('method "remove"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('remove');
            });
            it('should remove set elements', function () {
                var obj1 = { p: 'a' }, obj2 = { p: 'b' };
                var set = new ValueSet('p', [obj1, obj2]);
                expect(set.has(obj1)).to.equal(true);
                expect(set.remove(obj1)).to.equal(true);
                expect(set.has(obj1)).to.equal(false);
                expect(set.remove(obj1)).to.equal(false);
                expect(set.remove({ p: 'b' })).to.equal(true);
                expect(set.remove({ p: 'c' })).to.equal(false);
            });
        });

        describe('method "merge"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('merge');
            });
            it('should merge the sets', function () {
                var set = new ValueSet(_.identity, ['a', 'b']);
                expect(set.has('c')).to.equal(false);
                expect(set.merge(['b', 'c'])).to.equal(set);
                expect(set.has('a')).to.equal(true);
                expect(set.has('b')).to.equal(true);
                expect(set.has('c')).to.equal(true);
            });
        });

        describe('method "with"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('with');
            });
            it('should invoke callback for existing elements', function () {
                var set = new ValueSet('p', [{ p: 'a' }]);
                var spy = sinon.spy(_.constant('abc'));
                var context = {};
                expect(set.with('b', spy, context)).to.equal(undefined);
                sinon.assert.callCount(spy, 0);
                expect(set.with('a', spy, context)).to.equal('abc');
                sinon.assert.callCount(spy, 1);
                sinon.assert.alwaysCalledOn(spy, context);
            });
        });

        describe('method "keys"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('keys');
            });
            it('should return all keys', function () {
                var set = new ValueSet('p', [{ p: 'a' }, { p: 'b' }]);
                var keys = set.keys();
                expect(keys).to.be.an('array');
                expect(keys.sort()).to.deep.equal(['a', 'b']);
            });
            it('should return an empty array for an empty set', function () {
                expect(new ValueSet(_.identity).keys()).to.deep.equal([]);
            });
        });

        describe('method "values"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('values');
            });
            it('should return all values', function () {
                var set = new ValueSet('p', [{ p: 'a' }, { p: 'b' }]);
                var values = set.values();
                expect(values).to.be.an('array').with.length(2);
                values = _.sortBy(values, 'p');
                expect(values).to.deep.equal([{ p: 'a' }, { p: 'b' }]);
            });
            it('should return an empty array for an empty set', function () {
                expect(new ValueSet(_.identity).values()).to.deep.equal([]);
            });
        });

        describe('method "toObject"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('toObject');
            });
            it('should return the internal map', function () {
                var set = new ValueSet('p', [{ p: 'a' }, { p: 'b' }]);
                expect(set.toObject()).to.equal(set._cont);
            });
        });

        describe('method "iterator"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('iterator');
            });
            it('should return a property iterator', function () {
                var set = new ValueSet(function (v) { return 'k' + v; }, [1, 2]);
                function matcher(result) { return !result.done && /^k(11|22)$/.test(result.key + result.value); }
                var it = set.iterator();
                expect(it).to.respondTo('next');
                expect(it.next()).to.satisfy(matcher);
                expect(it.next()).to.satisfy(matcher);
                expect(it.next()).to.deep.equal({ done: true });
            });
        });

        describe('method "keyIterator"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('keyIterator');
            });
            it('should return a key iterator', function () {
                var set = new ValueSet(function (v) { return 'k' + v; }, [1, 2]);
                function matcher(result) { return !result.done && /^k(11|22)$/.test(result.value + result.element); }
                var it = set.keyIterator();
                expect(it).to.respondTo('next');
                expect(it.next()).to.satisfy(matcher);
                expect(it.next()).to.satisfy(matcher);
                expect(it.next()).to.deep.equal({ done: true });
            });
        });

        describe('method "forEach"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('forEach');
            });
            it('should visit all elements', function () {
                var set = new ValueSet(function (v) { return 'k' + v; }, [1, 2]);
                var spy = sinon.spy(), context = {};
                expect(set.forEach(spy, context)).to.equal(set);
                sinon.assert.callCount(spy, 2);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy, 1, 'k1', set);
                sinon.assert.calledWithExactly(spy, 2, 'k2', set);
            });
        });

        describe('method "forEachKey"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('forEachKey');
            });
            it('should visit all keys and elements', function () {
                var set = new ValueSet(function (v) { return 'k' + v; }, [1, 2]);
                var spy = sinon.spy(), context = {};
                expect(set.forEachKey(spy, context)).to.equal(set);
                sinon.assert.callCount(spy, 2);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy, 'k1', 1, set);
                sinon.assert.calledWithExactly(spy, 'k2', 2, set);
            });
        });

        describe('method "some"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('some');
            });
        });

        describe('method "every"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('every');
            });
        });

        describe('method "map"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('map');
            });
        });

        describe('method "filter"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('filter');
            });
        });

        describe('method "reject"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('reject');
            });
        });

        describe('method "pluck"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('pluck');
            });
        });

        describe('method "sortBy"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('sortBy');
            });
            it('should sort by callback value', function () {
                var set = new ValueSet(_.identity, [-2, -1]);
                var spy = sinon.spy(Math.abs);
                expect(set.sortBy(spy, 42)).to.deep.equal([-1, -2]);
                expect(spy.alwaysCalledOn(42)).to.equal(true);
            });
            it('should sort by property', function () {
                var set = new ValueSet('p', [{ p: 2 }, { p: 1 }]);
                expect(set.sortBy('p')).to.deep.equal([{ p: 1 }, { p: 2 }]);
            });
        });

        describe('method "mapObject"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('mapObject');
            });
        });

        describe('method "pick"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('pick');
            });
        });

        describe('method "omit"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('omit');
            });
        });

        describe('method "find"', function () {
            it('should exist', function () {
                expect(ValueSet).to.respondTo('find');
            });
        });
    });

    // ========================================================================
});
