/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/textframework/model/listhandlermixin',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/listutils'
], function (AppHelper, ListHandlerMixin, AttributeUtils, DOM, ListUtils) {

    'use strict';

    // mix-in class UpdateListsMixin ==========================================

    describe('Textframework mix-in class ListHandlerMixin', function () {

        // private helpers ----------------------------------------------------

        var layoutId_1 = 'layout1',
            masterId_1 = 'master1',
            slideDefaultName = 'slide_1',
            firstStandardDrawing = null, // the first drawing in the standard slide
            firstStandardParagraphs = null, // the paragraphs in the first drawing in the standard slide
            secondStandardDrawing = null, // the second drawing in the standard slide
            secondStandardParagraphs = null, // the paragraphs in the second drawing in the standard slide
            explicitAttributes = null,  // the explicit attributes of a paragraph
            textFrameDrawing = null, // the text frame drawing in the standard slide
            textFrameParagraphs = null, // the paragraphs in the text frame drawing in the standard slide
            para1 = null, // the first paragraph in second drawing on standard slide
            para2 = null, // the second paragraph in second drawing on standard slide
            para3 = null, // the third paragraph in second drawing on standard slide
            para4 = null, // the fourth paragraph in second drawing on standard slide
            para5 = null, // the fifth paragraph in second drawing on standard slide
            para6 = null, // the sixth paragraph in second drawing on standard slide

            // the operations to be applied by the document model
            OPERATIONS = [
                {
                    name: 'setDocumentAttributes',
                    attrs: {
                        page: { width: 33866, height: 19050, orientation: 'landscape' },
                        defaultTextListStyles: {
                            l1: { character: { fontSize: 18, fontName: '+mn-lt', color: { type: 'scheme', value: 'text1' } }, paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 0 } },
                            l2: { character: { fontSize: 18, fontName: '+mn-lt', color: { type: 'scheme', value: 'text1' } }, paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 1270 } },
                            l3: { character: { fontSize: 18, fontName: '+mn-lt', color: { type: 'scheme', value: 'text1' } }, paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 2540 } }
                        }
                    }
                },
                {
                    name: 'insertMasterSlide', id: masterId_1, attrs: {
                        listStyles: {
                            title: {
                                l1: {
                                    character: { fontSize: 44, fontName: '+mj-lt', type: 'solid', color: { type: 'scheme', value: 'text1' } },
                                    paragraph: { lineHeight: { type: 'percent', value: 90 }, spacingBefore: { type: 'percent', value: 0 }, defaultTabSize: 2540, bullet: { type: 'none' }, alignment: 'left' }
                                }
                            },
                            body: {
                                l1: {
                                    character: { fontSize: 28, fontName: '+mn-lt', type: 'solid', color: { type: 'scheme', value: 'text1' } },
                                    paragraph: { lineHeight: { type: 'percent', value: 90 }, spacingBefore: { type: 'fixed', value: 35 }, defaultTabSize: 2540, bullet: { type: 'character', character: '-' }, bulletFont: { followText: false, name: 'Arial' }, alignment: 'left', indentLeft: 635, indentFirstLine: -634 }
                                },
                                l2: {
                                    character: { fontSize: 24, fontName: '+mn-lt', type: 'solid', color: { type: 'scheme', value: 'text1' } },
                                    paragraph: { lineHeight: { type: 'percent', value: 90 }, spacingBefore: { type: 'fixed', value: 17 }, defaultTabSize: 2540, bullet: { type: 'character', character: '\u2022' }, bulletFont: { followText: false, name: 'Arial' }, alignment: 'left', indentLeft: 1905, indentFirstLine: -634 }
                                },
                                l3: {
                                    character: { fontSize: 20, fontName: '+mn-lt', type: 'solid', color: { type: 'scheme', value: 'text1' } },
                                    paragraph: { lineHeight: { type: 'percent', value: 90 }, spacingBefore: { type: 'fixed', value: 17 }, defaultTabSize: 2540, bullet: { type: 'character', character: '\u2022' }, bulletFont: { followText: false, name: 'Arial' }, alignment: 'left', indentLeft: 3175, indentFirstLine: -634 }
                                },
                                l4: {
                                    character: { fontSize: 18, fontName: '+mn-lt', type: 'solid', color: { type: 'scheme', value: 'text1' } },
                                    paragraph: { lineHeight: { type: 'percent', value: 90 }, spacingBefore: { type: 'fixed', value: 17 }, defaultTabSize: 2540, bullet: { type: 'character', character: '\u2022' }, bulletFont: { followText: false, name: 'Arial' }, alignment: 'left', indentLeft: 4445, indentFirstLine: -634 }
                                },
                                l5: {
                                    character: { fontSize: 18, fontName: '+mn-lt', type: 'solid', color: { type: 'scheme', value: 'text1' } },
                                    paragraph: { lineHeight: { type: 'percent', value: 90 }, spacingBefore: { type: 'fixed', value: 17 }, defaultTabSize: 2540, bullet: { type: 'character', character: '\u2022' }, bulletFont: { followText: false, name: 'Arial' }, alignment: 'left', indentLeft: 5715, indentFirstLine: -634 }
                                },
                                l6: {
                                    character: { fontSize: 18, fontName: '+mn-lt', type: 'solid', color: { type: 'scheme', value: 'text1' } },
                                    paragraph: { lineHeight: { type: 'percent', value: 90 }, spacingBefore: { type: 'fixed', value: 17 }, defaultTabSize: 2540, bullet: { type: 'character', character: '\u2022' }, bulletFont: { followText: false, name: 'Arial' }, alignment: 'left', indentLeft: 6985, indentFirstLine: -634 }
                                },
                                l7: {
                                    character: { fontSize: 18, fontName: '+mn-lt', type: 'solid', color: { type: 'scheme', value: 'text1' } },
                                    paragraph: { lineHeight: { type: 'percent', value: 90 }, spacingBefore: { type: 'fixed', value: 17 }, defaultTabSize: 2540, bullet: { type: 'character', character: '\u2022' }, bulletFont: { followText: false, name: 'Arial' }, alignment: 'left', indentLeft: 8255, indentFirstLine: -634 }
                                },
                                l8: {
                                    character: { fontSize: 18, fontName: '+mn-lt', type: 'solid', color: { type: 'scheme', value: 'text1' } },
                                    paragraph: { lineHeight: { type: 'percent', value: 90 }, spacingBefore: { type: 'fixed', value: 17 }, defaultTabSize: 2540, bullet: { type: 'character', character: '\u2022' }, bulletFont: { followText: false, name: 'Arial' }, alignment: 'left', indentLeft: 9525, indentFirstLine: -634 }
                                },
                                l9: {
                                    character: { fontSize: 18, fontName: '+mn-lt', type: 'solid', color: { type: 'scheme', value: 'text1' } },
                                    paragraph: { lineHeight: { type: 'percent', value: 90 }, spacingBefore: { type: 'fixed', value: 17 }, defaultTabSize: 2540, bullet: { type: 'character', character: '\u2022' }, bulletFont: { followText: false, name: 'Arial' }, alignment: 'left', indentLeft: 10795, indentFirstLine: -634 }
                                }
                            },
                            other: {
                                l1: {
                                    character: { fontSize: 18, fontName: '+mn-lt', type: 'solid', color: { type: 'scheme', value: 'text1' } },
                                    paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 0 }
                                },
                                l2: {
                                    character: { fontSize: 18, fontName: '+mn-lt', type: 'solid', color: { type: 'scheme', value: 'text1' } },
                                    paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 1270 }
                                },
                                l3: {
                                    character: { fontSize: 18, fontName: '+mn-lt', type: 'solid', color: { type: 'scheme', value: 'text1' } },
                                    paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 2540 }
                                },
                                l4: {
                                    character: { fontSize: 18, fontName: '+mn-lt', type: 'solid', color: { type: 'scheme', value: 'text1' } },
                                    paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 3810 }
                                },
                                l5: {
                                    character: { fontSize: 18, fontName: '+mn-lt', type: 'solid', color: { type: 'scheme', value: 'text1' } },
                                    paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 5080 }
                                },
                                l6: {
                                    character: { fontSize: 18, fontName: '+mn-lt', type: 'solid', color: { type: 'scheme', value: 'text1' } },
                                    paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 6350 }
                                },
                                l7: {
                                    character: { fontSize: 18, fontName: '+mn-lt', type: 'solid', color: { type: 'scheme', value: 'text1' } },
                                    paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 7620 }
                                },
                                l8: {
                                    character: { fontSize: 18, fontName: '+mn-lt', type: 'solid', color: { type: 'scheme', value: 'text1' } },
                                    paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 8890 }
                                },
                                l9: {
                                    character: { fontSize: 18, fontName: '+mn-lt', type: 'solid', color: { type: 'scheme', value: 'text1' } },
                                    paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 10160 }
                                }
                            }
                        }
                    }
                },
//                { name: 'insertDrawing', start: [0, 0], target: masterId_1, type: 'shape', attrs: { presentation: { phType: 'title' }, shape: { anchor: 'centered', paddingLeft: 254, paddingRight: 254, paddingTop: 127, paddingBottom: 127 }, drawing: { name: 'T1', left: 2328, top: 1014, width: 29210, height: 3682 }, geometry: { presetShape: 'rect', avList: {} }}},
//                { name: 'insertParagraph', start: [0, 0, 0], target: masterId_1 },
//                { name: 'insertText', start: [0, 0, 0, 0], target: masterId_1, text: 'Master title format' },
//                { name: 'insertDrawing', start: [0, 1], target: masterId_1, type: 'shape', attrs: { presentation: { phType: 'body', phIndex: 1 }, drawing: { name: 'T2', left: 2328, top: 5071, width: 29210, height: 12087 }, geometry: { presetShape: 'rect', avList: {} }}},
//                { name: 'insertParagraph', start: [0, 1, 0], target: masterId_1, attrs: { paragraph: { level: 0 }}},
//                { name: 'insertText', start: [0, 1, 0, 0], target: masterId_1,text: 'Format templates' },
//                { name: 'insertParagraph', start: [0, 1, 1], target: masterId_1, attrs: { paragraph: { level: 1 }}},
//                { name: 'insertText', start: [0, 1, 1, 0], target: masterId_1,text: 'Second level' },
//                { name: 'insertParagraph', start: [0, 1, 2], target: masterId_1, attrs: { paragraph: { level: 2 }}},
//                { name: 'insertText', start: [0, 1, 2, 0], target: masterId_1,text: 'Third level' },
//                { name: 'insertParagraph', start: [0, 1, 3], target: masterId_1, attrs: { paragraph: { level: 3 }}},
//                { name: 'insertText', start: [0, 1, 3, 0], target: masterId_1, text: 'Fourth level' },
//                { name: 'insertParagraph', start: [0, 1, 4], target: masterId_1, attrs: { paragraph: { level: 4 }}},
//                { name: 'insertText', start: [0, 1, 4, 0], target: masterId_1, text: 'Fifth level' },
//                { name: 'insertDrawing', start: [0, 2],target: masterId_1, type: 'shape', attrs: { presentation: { phType: 'dt', phSize: 'half', phIndex: 2 }, listStyle: { l1: { character: { fontSize: 12, type: 'solid', color: { type: 'scheme', value: 'text1', transformations: [{ type: 'tint', value: 75000 }] }}, paragraph: { alignment: 'left' }}}, shape: { anchor: 'centered', paddingLeft: 254, paddingRight: 254, paddingTop: 127, paddingBottom: 127 }, drawing: { name: 'D3', left: 2328, top: 17657, width: 7620, height: 1014 }, geometry: { presetShape: 'rect', avList: {} }}},
//                { name: 'insertParagraph', start: [0, 2, 0], target: masterId_1 },
//                { name: 'insertDrawing', start: [0, 3], target: masterId_1, type: 'shape', attrs: { presentation: { phType: 'ftr', phSize: 'quarter', phIndex: 3 }, listStyle: { l1: { character: { fontSize: 12, type: 'solid', color: { type: 'scheme', value: 't1', transformations: [{ type: 'tint', value: 75000 }] }}, paragraph: { alignment: 'center' }}}, shape: { anchor: 'centered', paddingLeft: 254, paddingRight: 254, paddingTop: 127, paddingBottom: 127 }, drawing: { name: 'F4', left: 11218, top: 17657, width: 11430, height: 1014 }, geometry: { presetShape: 'rect', avList: {} }}},
//                { name: 'insertParagraph', start: [0, 3, 0], target: masterId_1 },
//                { name: 'insertDrawing', start: [0, 4], target: masterId_1, type: 'shape', attrs: { presentation: { phType: 'sldNum', phSize: 'quarter', phIndex: 4 }, listStyle: { l1: { character: { fontSize: 12, type: 'solid', color: { type: 'scheme', value: 't1', transformations: [{ type: 'tint', value: 75000 }] }}, paragraph: { alignment: 'right' }}}, shape: { anchor: 'centered', paddingLeft: 254, paddingRight: 254, paddingTop: 127, paddingBottom: 127 }, drawing: { name: 'F5', left: 23918, top: 17657, width: 7620, height: 1014 }, geometry: { presetShape: 'rect', avList: {} }}},
//                { name: 'insertParagraph', start: [0, 4, 0], target: masterId_1 },
                { name: 'insertLayoutSlide', id: layoutId_1, target: masterId_1 },
//                { name: 'insertDrawing', start: [0, 0], target: layoutId_1, type: 'shape', attrs: { presentation: { phType: 'title' }, drawing: { name: 'Title 1' }}},
//                { name: 'insertParagraph', start: [0, 0, 0], target: layoutId_1 },
//                { name: 'insertText', start: [0, 0, 0, 0], target: layoutId_1, text: 'Title master format' },
//                { name: 'insertDrawing', start: [0, 1], target: layoutId_1, type: 'shape', attrs: { presentation: { phIndex: 1 }, drawing: { name: 'I2' }}},
//                { name: 'insertParagraph', start: [0, 1, 0], target: layoutId_1, attrs: { paragraph: { level: 0 }}},
//                { name: 'insertText', start: [0, 1, 0, 0], target: layoutId_1, text: 'Format templates' },
//                { name: 'insertParagraph', start: [0, 1, 1], target: layoutId_1, attrs: { paragraph: { level: 1 }}},
//                { name: 'insertText', start: [0, 1, 1, 0], target: layoutId_1, text: 'Second level' },
//                { name: 'insertParagraph', start: [0, 1, 2], target: layoutId_1, attrs: { paragraph: { level: 2 }}},
//                { name: 'insertText', start: [0, 1, 2, 0], target: layoutId_1, text: 'Third level' },
//                { name: 'insertParagraph', start: [0, 1, 3], target: layoutId_1, attrs: { paragraph: { level: 3 }}},
//                { name: 'insertText', start: [0, 1, 3, 0], target: layoutId_1, text: 'Fourth level' },
//                { name: 'insertParagraph', start: [0, 1, 4], target: layoutId_1, attrs: { paragraph: { level: 4 }}},
//                { name: 'insertText', start: [0, 1, 4, 0], target: layoutId_1, text: 'Fifth level' },
                { name: 'insertDrawing', start: [0, 0], target: layoutId_1, type: 'shape', attrs: { presentation: { phType: 'dt', phSize: 'half', phIndex: 10 }, drawing: { name: 'D3' } } },
                { name: 'insertParagraph', start: [0, 0, 0], target: layoutId_1 },
                { name: 'insertDrawing', start: [0, 1], target: layoutId_1, type: 'shape', attrs: { presentation: { phType: 'ftr', phSize: 'quarter', phIndex: 11 }, drawing: { name: 'F4' } } },
                { name: 'insertParagraph', start: [0, 1, 0], target: layoutId_1 },
                { name: 'insertDrawing', start: [0, 2], target: layoutId_1, type: 'shape', attrs: { presentation: { phType: 'sldNum', phSize: 'quarter', phIndex: 12 }, drawing: { name: 'F5' } } },
                { name: 'insertParagraph', start: [0, 2, 0], target: layoutId_1 },
                { name: 'insertSlide', start: 0, target: layoutId_1 },
                { name: 'insertDrawing', start: [0, 0], type: 'shape', attrs: { presentation: { phType: 'title' }, drawing: { name: 'Title 1' } } },
                { name: 'insertParagraph', start: [0, 0, 0] },
                { name: 'insertText', start: [0, 0, 0, 0], text: 'Title' },
                { name: 'insertDrawing', start: [0, 1], type: 'shape', attrs: { presentation: { phIndex: 1 }, drawing: { name: 'I2' } } },
                { name: 'insertParagraph', start: [0, 1, 0] },
                { name: 'insertText', start: [0, 1, 0, 0], text: 'Item 1' },
                { name: 'insertParagraph', start: [0, 1, 1] },
                { name: 'insertText', start: [0, 1, 1, 0], text: 'Item 2' },
                { name: 'insertParagraph', start: [0, 1, 2] },
                { name: 'insertText', start: [0, 1, 2, 0], text: 'Item 3' },
                { name: 'insertParagraph', start: [0, 1, 3] },
                { name: 'insertText', start: [0, 1, 3, 0], text: 'Item 4' },
                { name: 'insertParagraph', start: [0, 1, 4] },
                { name: 'insertText', start: [0, 1, 4, 0], text: 'Item 5' },
                { name: 'insertParagraph', start: [0, 1, 5] },
                { name: 'insertText', start: [0, 1, 5, 0], text: 'Item 6' },
                { name: 'insertDrawing', attrs: { drawing: { width: 8000, left: 5000, top: 3000 }, shape: { autoResizeHeight: true }, line: { color: { type: 'rgb', value: '000000' }, style: 'single', type: 'none', width: 26 }, fill: { color: { type: 'auto' }, type: 'solid' } }, start: [0, 2], type: 'shape' },
                { name: 'insertParagraph', start: [0, 2, 0] },
                { name: 'insertText', start: [0, 2, 0, 0], text: 'Text frame text in paragraph 1' },
                { name: 'insertParagraph', start: [0, 2, 1] },
                { name: 'insertText', start: [0, 2, 1, 0], text: 'Text frame text in paragraph 2' },
                { name: 'insertParagraph', start: [0, 2, 2] },
                { name: 'insertText', start: [0, 2, 2, 0], text: 'Text frame text in paragraph 3' }
            ];

        var model;
        AppHelper.createPresentationApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(ListHandlerMixin).to.be.a('function');
        });

        // public methods -----------------------------------------------------

        describe('method "isStandardSlideId"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('isStandardSlideId');
            });
            it('should return whether the specified ID is the ID of a normal (standard) slide', function () {
                expect(model.isStandardSlideId(layoutId_1)).to.equal(false);
                expect(model.isStandardSlideId(masterId_1)).to.equal(false);
                expect(model.isStandardSlideId(slideDefaultName)).to.equal(true);
            });
        });

        describe('method "getActiveSlideId"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getActiveSlideId');
            });
            it('should return the ID of the active slide', function () {
                expect(model.getActiveSlideId()).to.equal(slideDefaultName);
            });
        });

        describe('method "DOM.isListParagraphNode"', function () {
            it('should exist', function () {
                expect(DOM).to.respondTo('isListParagraphNode');
            });
            it('should check whether the specified node is a list paragraph', function () {
                model.getSelection().setTextSelection([0, 0, 0, 0]);
                firstStandardDrawing = model.getSelection().getSelectedTextFrameDrawing();
                expect(firstStandardDrawing.length).to.equal(1);
                firstStandardParagraphs = firstStandardDrawing.find(DOM.PARAGRAPH_NODE_SELECTOR);
                expect(firstStandardParagraphs.length).to.equal(1);
                model.getSelection().setTextSelection([0, 1, 0, 0]);
                secondStandardDrawing = model.getSelection().getSelectedTextFrameDrawing();
                expect(secondStandardDrawing.length).to.equal(1);
                secondStandardParagraphs = secondStandardDrawing.find(DOM.PARAGRAPH_NODE_SELECTOR);
                expect(secondStandardParagraphs.length).to.equal(6);
                expect(DOM.isListParagraphNode(firstStandardParagraphs[0])).to.equal(false);
                expect(DOM.isListParagraphNode(secondStandardParagraphs[0])).to.equal(true);
            });
        });

        describe('method "DOM.isListLabelNode"', function () {
            it('should exist', function () {
                expect(DOM).to.respondTo('isListLabelNode');
            });
            it('should check whether the specified node is a list label in a list paragraph', function () {
                expect(DOM.isListLabelNode(firstStandardParagraphs[0].childNodes[0])).to.equal(false);
                expect(DOM.isListLabelNode(secondStandardParagraphs[0].childNodes[0])).to.equal(true);
            });
        });

        describe('method "isListIndentChangeable"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('isListIndentChangeable');
            });
            it('should check whether the list level at the currently selected paragraph is changeable', function () {
                model.getSelection().setTextSelection([0, 0, 0, 0]);
                expect(model.isListIndentChangeable()).to.equal(false);
                model.getSelection().setTextSelection([0, 1, 0, 0]);
                expect(model.isListIndentChangeable()).to.equal(true);
            });
        });

        describe('method "getMaxListLevel"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getMaxListLevel');
            });
            it('should receive the maximum list level for the drawing that contains the selection', function () {
                model.getSelection().setTextSelection([0, 0, 0, 0]);
                expect(model.getMaxListLevel()).to.equal(0);
                model.getSelection().setTextSelection([0, 1, 0, 0]);
                expect(model.getMaxListLevel()).to.equal(8);
            });
        });

        describe('method "changeListIndent"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('changeListIndent');
            });
            it('should change the level of list indent for the specified paragraph', function () {

                expect($(firstStandardParagraphs[0].childNodes[0]).text()).to.equal('Title');

                expect(secondStandardParagraphs[0].childNodes.length).to.equal(2);
                expect($(secondStandardParagraphs[0].childNodes[0]).text()).to.equal('-');
                expect($(secondStandardParagraphs[0].childNodes[1]).text()).to.equal('Item 1');

                model.getSelection().setTextSelection([0, 0, 0, 0]);

                explicitAttributes = AttributeUtils.getExplicitAttributes(firstStandardParagraphs[0]);
                expect(explicitAttributes.paragraph).to.be.undefined;

                model.changeListIndent(); // increasing indent of first paragraph

                explicitAttributes = AttributeUtils.getExplicitAttributes(firstStandardParagraphs[0]);
                expect(explicitAttributes.paragraph).to.be.undefined;  // no effect in title

                model.changeListIndent({ increase: false }); // decreasing indent of first paragraph

                explicitAttributes = AttributeUtils.getExplicitAttributes(firstStandardParagraphs[0]);
                expect(explicitAttributes.paragraph).to.be.undefined;  // no effect in title

                model.getSelection().setTextSelection([0, 1, 0, 0]);

                model.changeListIndent(); // increasing indent of first paragraph

                explicitAttributes = AttributeUtils.getExplicitAttributes(secondStandardParagraphs[0]);
                expect(explicitAttributes.paragraph.level).to.equal(1);

                model.changeListIndent(); // increasing indent of first paragraph

                explicitAttributes = AttributeUtils.getExplicitAttributes(secondStandardParagraphs[0]);
                expect(explicitAttributes.paragraph.level).to.equal(2);

                model.changeListIndent({ increase: false }); // decreasing indent of first paragraph

                explicitAttributes = AttributeUtils.getExplicitAttributes(secondStandardParagraphs[0]);
                expect(explicitAttributes.paragraph.level).to.equal(1);

                model.changeListIndent({ increase: false }); // decreasing indent of first paragraph

                explicitAttributes = AttributeUtils.getExplicitAttributes(secondStandardParagraphs[0]);
                expect(explicitAttributes.paragraph.level).to.equal(0);

            });
        });

        describe('method "setListStyleId"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('setListStyleId');
            });
            it('should set the specified list style to the selected paragraph', function () {

                model.getSelection().setTextSelection([0, 0, 0, 0]);

                explicitAttributes = AttributeUtils.getExplicitAttributes(firstStandardParagraphs[0]);
                expect(explicitAttributes.paragraph).to.be.undefined;

                model.setListStyleId('bullet', ListUtils.DEFAULT_LIST_STYLE_ID, '');

                explicitAttributes = AttributeUtils.getExplicitAttributes(firstStandardParagraphs[0]);
                expect(explicitAttributes.paragraph.bullet.type).to.equal('character');

                model.setListStyleId('bullet', ListUtils.DEFAULT_LIST_STYLE_ID, ListHandlerMixin.DEFAULT_BULLET_LISTSTYLE);

                explicitAttributes = AttributeUtils.getExplicitAttributes(firstStandardParagraphs[0]);
                expect(explicitAttributes.paragraph.bullet.type).to.equal('none');
            });
        });

        describe('method "getListStyleId"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getListStyleId');
            });
            it('should get the list style to a selected paragraph', function () {

                var newListStyleId = 'L20014';

                model.getSelection().setTextSelection([0, 0, 0, 0]);

                explicitAttributes = AttributeUtils.getExplicitAttributes(firstStandardParagraphs[0]);
                expect(explicitAttributes.paragraph.bullet.type).to.equal('none');

                model.setListStyleId('bullet', newListStyleId, '');

                explicitAttributes = AttributeUtils.getExplicitAttributes(firstStandardParagraphs[0]);
                expect(explicitAttributes.paragraph.bullet.type).to.equal('character');

                expect(model.getListStyleId('bullet', explicitAttributes.paragraph)).to.equal(newListStyleId);
            });
        });

        describe('method "getMaxListLevel"', function () {
            it('should receive the maximum list level of a paragraph inside a text frame', function () {
                model.getSelection().setTextSelection([0, 2, 0, 0]);
                expect(model.getMaxListLevel()).to.equal(2);
            });
        });

        describe('method "setListStyleId"', function () {
            it('should set the default numbering list style over all selected paragraphs', function () {

                model.getSelection().setTextSelection([0, 2, 0, 0], [0, 2, 2, 2]); // selecting three paragraphs

                textFrameDrawing = model.getSelection().getSelectedTextFrameDrawing();
                expect(textFrameDrawing.length).to.equal(1);
                textFrameParagraphs = textFrameDrawing.find(DOM.PARAGRAPH_NODE_SELECTOR);
                expect(textFrameParagraphs.length).to.equal(3);

                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[0]);
                expect(explicitAttributes.paragraph).to.be.undefined;
                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[1]);
                expect(explicitAttributes.paragraph).to.be.undefined;
                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[2]);
                expect(explicitAttributes.paragraph).to.be.undefined;

                model.setListStyleId('numbering', ListHandlerMixin.DEFAULT_NUMBERING_TYPE, '');

                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[0]);
                expect(explicitAttributes.paragraph.bullet.type).to.equal('numbering');
                expect(explicitAttributes.paragraph.bullet.numType).to.equal(ListHandlerMixin.DEFAULT_NUMBERING_TYPE);
                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[1]);
                expect(explicitAttributes.paragraph.bullet.type).to.equal('numbering');
                expect(explicitAttributes.paragraph.bullet.numType).to.equal(ListHandlerMixin.DEFAULT_NUMBERING_TYPE);
                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[2]);
                expect(explicitAttributes.paragraph.bullet.type).to.equal('numbering');
                expect(explicitAttributes.paragraph.bullet.numType).to.equal(ListHandlerMixin.DEFAULT_NUMBERING_TYPE);
            });
        });

        describe('method "setListStyleId"', function () {
            it('should expand the selection of a new numbering list style automatically', function () {

                var newNumberingListStyle = 'arabicParenBoth';

                model.getSelection().setTextSelection([0, 2, 1, 1], [0, 2, 1, 2]); // selecting only one paragraph

                model.setListStyleId('numbering', newNumberingListStyle, ListHandlerMixin.DEFAULT_NUMBERING_TYPE);

                // all three paragraphs have the new numbering style assigned
                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[0]);
                expect(explicitAttributes.paragraph.bullet.type).to.equal('numbering');
                expect(explicitAttributes.paragraph.bullet.numType).to.equal(newNumberingListStyle);
                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[1]);
                expect(explicitAttributes.paragraph.bullet.type).to.equal('numbering');
                expect(explicitAttributes.paragraph.bullet.numType).to.equal(newNumberingListStyle);
                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[2]);
                expect(explicitAttributes.paragraph.bullet.type).to.equal('numbering');
                expect(explicitAttributes.paragraph.bullet.numType).to.equal(newNumberingListStyle);
            });
        });

        describe('method "setListStyleId"', function () {
            it('should not expand the selection automatically, if more than one paragraph is selected', function () {

                var oldNumberingListStyle = 'arabicParenBoth',
                    newNumberingListStyle = 'romanLcPeriod';

                model.getSelection().setTextSelection([0, 2, 1, 1], [0, 2, 2, 2]); // selecting two paragraphs

                model.setListStyleId('numbering', newNumberingListStyle, oldNumberingListStyle);

                // all three paragraphs have the new numbering style assigned
                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[0]);
                expect(explicitAttributes.paragraph.bullet.type).to.equal('numbering');
                expect(explicitAttributes.paragraph.bullet.numType).to.equal(oldNumberingListStyle);  // old style remains
                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[1]);
                expect(explicitAttributes.paragraph.bullet.type).to.equal('numbering');
                expect(explicitAttributes.paragraph.bullet.numType).to.equal(newNumberingListStyle);
                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[2]);
                expect(explicitAttributes.paragraph.bullet.type).to.equal('numbering');
                expect(explicitAttributes.paragraph.bullet.numType).to.equal(newNumberingListStyle);
            });
        });

        describe('method "changeListIndent"', function () {
            it('should find the correct numbering type after increase and decrease of list level', function () {

                var oldNumberingListStyle = 'arabicParenBoth',
                    newNumberingListStyle = 'romanLcPeriod';

                model.getSelection().setTextSelection([0, 2, 0, 0]);  // selecting the first paragraph

                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[0]);
                expect(explicitAttributes.paragraph.bullet.numType).to.equal(oldNumberingListStyle); // old value

                model.changeListIndent(); // increasing indent of first paragraph

                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[0]);
                expect(explicitAttributes.paragraph.bullet.numType).to.equal(oldNumberingListStyle); // still old value

                model.changeListIndent({ increase: false }); // decreasing indent of first paragraph

                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[0]);
                expect(explicitAttributes.paragraph.bullet.numType).to.equal(newNumberingListStyle); // new value
            });
        });

        describe('method "changeListIndent"', function () {
            it('should not modify the numbering type after increase and decrease of list level, if character bullet list follows', function () {

                var oldNumberingListStyle = 'arabicParenBoth',
                    newNumberingListStyle = 'romanLcPeriod',
                    bulletListStyleId = 'L20014';

                model.getSelection().setTextSelection([0, 2, 1, 0]);  // selecting the second paragraph
                model.setListStyleId('bullet', bulletListStyleId, ''); // second paragraph gets character bullet list
                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[1]);
                expect(explicitAttributes.paragraph.bullet.type).to.equal('character');

                model.getSelection().setTextSelection([0, 2, 0, 0]);  // selecting the first paragraph
                model.setListStyleId('numbering', oldNumberingListStyle, newNumberingListStyle);  // setting list style to 'oldNumberingListStyle'
                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[0]);
                expect(explicitAttributes.paragraph.bullet.type).to.equal('numbering');
                expect(explicitAttributes.paragraph.bullet.numType).to.equal(oldNumberingListStyle);

                // now increasing and decreasing level of third paragraph
                model.getSelection().setTextSelection([0, 2, 2, 0]);  // selecting the third paragraph
                model.changeListIndent(); // increasing indent of third paragraph
                model.changeListIndent({ increase: false }); // decreasing indent of third paragraph

                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[2]);
                expect(explicitAttributes.paragraph.bullet.type).to.equal('numbering');
                expect(explicitAttributes.paragraph.bullet.numType).to.equal(newNumberingListStyle);  // -> not switched to oldNumberingListStyle
            });
        });

        describe('method "changeListIndent"', function () {
            it('should modify the numbering type after increase and decrease of list level, if no bullet list follows', function () {

                var oldNumberingListStyle = 'arabicParenBoth',
                    newNumberingListStyle = 'romanLcPeriod';

                model.getSelection().setTextSelection([0, 2, 1, 0]);  // selecting the second paragraph
                model.setListStyleId('numbering', oldNumberingListStyle, ''); // second paragraph gets numbering list again
                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[1]);
                expect(explicitAttributes.paragraph.bullet.type).to.equal('numbering');

                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[2]);  // third paragraph
                expect(explicitAttributes.paragraph.bullet.type).to.equal('numbering');
                expect(explicitAttributes.paragraph.bullet.numType).to.equal(newNumberingListStyle);  // -> check of existing numbering list style

                // now increasing and decreasing level of third paragraph
                model.getSelection().setTextSelection([0, 2, 2, 0]);  // selecting the third paragraph
                model.changeListIndent(); // increasing indent of third paragraph
                model.changeListIndent({ increase: false }); // decreasing indent of third paragraph

                explicitAttributes = AttributeUtils.getExplicitAttributes(textFrameParagraphs[2]);
                expect(explicitAttributes.paragraph.bullet.type).to.equal('numbering');
                expect(explicitAttributes.paragraph.bullet.numType).to.equal(oldNumberingListStyle);  // -> switched to oldNumberingListStyle
            });
        });

        // checks with empty paragraphs (using asynchronous list update via 'updateListsDebounced')

        describe('method "updateLists"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('updateLists');
            });

            it('should have a correct prepared drawing on standard slide with 6 paragraphs', function () {

                expect(secondStandardParagraphs.length).to.equal(6);

                para1 = $(secondStandardParagraphs[0]);
                para2 = $(secondStandardParagraphs[1]);
                para3 = $(secondStandardParagraphs[2]);
                para4 = $(secondStandardParagraphs[3]);
                para5 = $(secondStandardParagraphs[4]);
                para6 = $(secondStandardParagraphs[5]);

                expect(DOM.isListParagraphNode(para1)).to.equal(true);
                expect(DOM.isListParagraphNode(para2)).to.equal(true);
                expect(DOM.isListParagraphNode(para3)).to.equal(true);
                expect(DOM.isListParagraphNode(para4)).to.equal(true);
                expect(DOM.isListParagraphNode(para5)).to.equal(true);
                expect(DOM.isListParagraphNode(para6)).to.equal(true);
            });
        });

        describe('method "isNumberedListParagraph"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('isNumberedListParagraph');
            });

            it('should check that paragraphs are correctly switched from bullet list to numbered list', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para1).paragraph)).to.equal(false);
                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para2).paragraph)).to.equal(false);
                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para3).paragraph)).to.equal(false);
                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para4).paragraph)).to.equal(false);
                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para5).paragraph)).to.equal(false);
                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para6).paragraph)).to.equal(false);

                model.getSelection().setTextSelection([0, 1, 0, 0], [0, 1, 5, 1]); // selecting all six paragraphs

                model.setListStyleId('numbering', ListHandlerMixin.DEFAULT_NUMBERING_TYPE, ''); // setting numbering list style to all paragraphs

                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para1).paragraph)).to.equal(true);
                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para2).paragraph)).to.equal(true);
                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para3).paragraph)).to.equal(true);
                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para4).paragraph)).to.equal(true);
                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para5).paragraph)).to.equal(true);
                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para6).paragraph)).to.equal(true);

                // debounced list update -> synchronous check: still the old list bullets
                expect(para1.children().first().text()).to.equal('-');
                expect(para2.children().first().text()).to.equal('-');
                expect(para3.children().first().text()).to.equal('-');
                expect(para4.children().first().text()).to.equal('-');
                expect(para5.children().first().text()).to.equal('-');
                expect(para6.children().first().text()).to.equal('-');

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    // checking state after formatting of list
                    expect(para1.children().first().text()).to.equal('1.');
                    expect(para2.children().first().text()).to.equal('2.');
                    expect(para3.children().first().text()).to.equal('3.');
                    expect(para4.children().first().text()).to.equal('4.');
                    expect(para5.children().first().text()).to.equal('5.');
                    expect(para6.children().first().text()).to.equal('6.');
                    done();
                });
            });

        });

        describe('empty list paragraphs', function () {

            it('should not be counted in numbered lists', function (done) {

                var deletePromise = null;
                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', $.noop); // avoid resolving 'listFormattingDef' too early

                model.getSelection().setTextSelection([0, 1, 1, 0], [0, 1, 1, 6]); // selecting all content of second paragraph

                deletePromise = model.deleteSelected();
                deletePromise.always(function () {

                    expect(deletePromise.state()).to.equal('resolved');

                    model.on('listformatting:done', function () {
                        listFormattingDef.resolve();
                    });

                    listFormattingDef.always(function () {

                        expect(listFormattingDef.state()).to.equal('resolved');

                        expect(para1.children().first().text()).to.equal('1.');
                        expect(para2.children().first().text()).to.equal('2.'); // empty paragraph gets number increased by 1, but stops further increase
                        expect(para3.children().first().text()).to.equal('2.');
                        expect(para4.children().first().text()).to.equal('3.');
                        expect(para5.children().first().text()).to.equal('4.');
                        expect(para6.children().first().text()).to.equal('5.');
                        done();
                    });
                });

            });

            it('should have the class attributes that handle the opacity, if the selection is inside the paragraph', function () {
                expect(para2.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS)).to.equal(true);
                expect(para2.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_SELECTED_CLASS)).to.equal(true);
            });

            it('should not have the class attributes that handle the opacity, if the selection is outside the paragraph', function () {
                model.getSelection().setTextSelection([0, 1, 0, 0]); // setting cursor into another paragraph
                expect(para2.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS)).to.equal(true);
                expect(para2.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_SELECTED_CLASS)).to.equal(false);
            });

            it('should not be counted in numbered lists if text is deleted (preparation for undo)', function (done) {

                var deletePromise = null;
                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', $.noop); // avoid resolving 'listFormattingDef' too early

                model.getSelection().setTextSelection([0, 1, 3, 0], [0, 1, 3, 6]); // selecting all content of fourth paragraph

                deletePromise = model.deleteSelected();
                deletePromise.always(function () {

                    expect(deletePromise.state()).to.equal('resolved');

                    model.on('listformatting:done', function () {
                        listFormattingDef.resolve();
                    });

                    listFormattingDef.always(function () {

                        expect(listFormattingDef.state()).to.equal('resolved');

                        expect(para1.children().first().text()).to.equal('1.');
                        expect(para2.children().first().text()).to.equal('2.'); // empty paragraph
                        expect(para3.children().first().text()).to.equal('2.');
                        expect(para4.children().first().text()).to.equal('3.'); // empty paragraph
                        expect(para5.children().first().text()).to.equal('3.');
                        expect(para6.children().first().text()).to.equal('4.');

                        expect(para4.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS)).to.equal(true);
                        expect(para4.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_SELECTED_CLASS)).to.equal(true);

                        done();
                    });
                });

            });

            it('should be counted in numbered lists if text is inserted via undo', function (done) {

                var undoPromise = model.getUndoManager().undo();
                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', $.noop); // avoid resolving 'listFormattingDef' too early

                undoPromise.always(function () {

                    expect(undoPromise.state()).to.equal('resolved');

                    model.on('listformatting:done', function () {
                        listFormattingDef.resolve();
                    });

                    listFormattingDef.always(function () {

                        expect(para1.children().first().text()).to.equal('1.');
                        expect(para2.children().first().text()).to.equal('2.'); // empty paragraph
                        expect(para3.children().first().text()).to.equal('2.');
                        expect(para4.children().first().text()).to.equal('3.'); // no longer empty paragraph
                        expect(para5.children().first().text()).to.equal('4.');
                        expect(para6.children().first().text()).to.equal('5.');

                        expect(para4.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS)).to.equal(false);
                        expect(para4.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_SELECTED_CLASS)).to.equal(false);

                        done();
                    });
                });
            });

            it('should be counted in numbered lists if text is inserted into paragraph', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([0, 1, 1, 0]); // setting cursor into empty paragraph
                model.insertText('c', [0, 1, 1, 0]);  // inserting one character into empty paragraph
                model.getSelection().setTextSelection([0, 1, 1, 1]); // setting cursor behind the new text

                listFormattingDef.always(function () {

                    expect(para1.children().first().text()).to.equal('1.');
                    expect(para2.children().first().text()).to.equal('2.'); // no longer empty paragraph
                    expect(para3.children().first().text()).to.equal('3.');
                    expect(para4.children().first().text()).to.equal('4.');
                    expect(para5.children().first().text()).to.equal('5.');
                    expect(para6.children().first().text()).to.equal('6.');

                    expect(para2.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS)).to.equal(false);
                    expect(para2.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_SELECTED_CLASS)).to.equal(false);

                    done();
                });
            });

            // additional checks for synchronous list update (calling model.updateLists() directly)

            it('should not be counted in numbered lists if text is deleted (synchronous list update)', function (done) {

                var deletePromise = null;

                model.getSelection().setTextSelection([0, 1, 1, 0], [0, 1, 1, 1]); // selecting all content of second paragraph

                deletePromise = model.deleteSelected();
                deletePromise.always(function () {

                    expect(deletePromise.state()).to.equal('resolved');

                    // updating lists synchronously
                    model.updateLists(null, secondStandardParagraphs);

                    expect(para1.children().first().text()).to.equal('1.');
                    expect(para2.children().first().text()).to.equal('2.'); // empty paragraph
                    expect(para3.children().first().text()).to.equal('2.');
                    expect(para4.children().first().text()).to.equal('3.');
                    expect(para5.children().first().text()).to.equal('4.');
                    expect(para6.children().first().text()).to.equal('5.');

                    expect(para2.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS)).to.equal(true);
                    expect(para2.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_SELECTED_CLASS)).to.equal(true);

                    done();
                });

            });

            it('should be counted in numbered lists if text is inserted into paragraph (synchronous list update)', function () {

                model.getSelection().setTextSelection([0, 1, 1, 0]); // setting cursor into empty paragraph
                model.insertText('c', [0, 1, 1, 0]);  // inserting one character into empty paragraph
                model.getSelection().setTextSelection([0, 1, 1, 1]); // setting cursor behind the new text

                // updating lists synchronously (this is done debounced during working with document)
                model.updateLists(null, secondStandardParagraphs);

                expect(para1.children().first().text()).to.equal('1.');
                expect(para2.children().first().text()).to.equal('2.'); // no longer empty paragraph
                expect(para3.children().first().text()).to.equal('3.');
                expect(para4.children().first().text()).to.equal('4.');
                expect(para5.children().first().text()).to.equal('5.');
                expect(para6.children().first().text()).to.equal('6.');

                expect(para2.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS)).to.equal(false);
                expect(para2.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_SELECTED_CLASS)).to.equal(false);
            });

        });

    });

    // ========================================================================

    describe('Presentation mix-in class ListHandlerMixin, ODF', function () {

        // private helpers ----------------------------------------------------

        var masterId_1 = 'master1',
            slideDefaultName = 'slide_1',
            firstStandardDrawing = null, // the first drawing in the standard slide
            firstStandardParagraphs = null, // the paragraphs in the first drawing in the standard slide
            secondStandardDrawing = null, // the second drawing in the standard slide
            secondStandardParagraphs = null, // the paragraphs in the second drawing in the standard slide
            explicitAttributes = null,  // the explicit attributes of a paragraph
            para1 = null, // the first paragraph in second drawing on standard slide
            para2 = null, // the second paragraph in second drawing on standard slide
            para3 = null, // the third paragraph in second drawing on standard slide
            para4 = null, // the fourth paragraph in second drawing on standard slide
            para5 = null, // the fifth paragraph in second drawing on standard slide
            para6 = null, // the sixth paragraph in second drawing on standard slide
            // the operations to be applied by the document model
            OPERATIONS = [
                {
                    name: 'setDocumentAttributes',
                    attrs: {
                        page: { marginRight: 0, width: 28000, marginBottom: 0, marginTop: 0, marginLeft: 0, height: 21000 }
                    }
                },
                {
                    name: 'insertMasterSlide', id: masterId_1, attrs: {
                        listStyles: {
                            title: {
                                l1: { paragraph: { bulletFont: { followText: false, name: 'StarSymbol' }, alignment: 'center', indentLeft: 600 }, character: { color: { type: 'auto' }, italicComplex: false, underline: false, strike: 'none', fontNameComplex: 'Lucida Sans', bold: false, italic: false, caps: 'none', fillColor: { type: 'auto' }, fontName: 'Liberation Sans', boldComplex: false, fontNameAsian: 'Microsoft YaHei', fontSize: 44, fontSizeAsian: 44, boldAsian: false, fontSizeComplex: 44, italicAsian: false } }
                            },
                            body: {
                                l1: { paragraph: { indentFirstLine: -600, bulletFont: { followText: false, name: 'Arial' }, bulletSize: { type: 'followText' }, spacingBefore: { type: 'fixed', value: 500 }, bullet: { character: '-', type: 'character' }, indentLeft: 900, spacingAfter: { type: 'fixed', value: 500 } }, character: { color: { type: 'auto' }, italicComplex: false, underline: false, strike: 'none', fontNameComplex: 'Lucida Sans', bold: false, italic: false, caps: 'none', fillColor: { type: 'auto' }, fontName: 'Liberation Sans', boldComplex: false, fontNameAsian: 'Microsoft YaHei', fontSize: 32, fontSizeAsian: 32, boldAsian: false, fontSizeComplex: 32, italicAsian: false } },
                                l2: { paragraph: { indentFirstLine: 600, bulletFont: { followText: false, name: 'Arial' }, bulletSize: { type: 'followText' }, spacingBefore: { type: 'fixed', value: 400 }, bullet: { character: '\u2022', type: 'character' }, indentLeft: 900, spacingAfter: { type: 'fixed', value: 400 } }, character: { color: { type: 'auto' }, italicComplex: false, underline: false, strike: 'none', fontNameComplex: 'Lucida Sans', bold: false, italic: false, caps: 'none', fillColor: { type: 'auto' }, fontName: 'Liberation Sans', boldComplex: false, fontNameAsian: 'Microsoft YaHei', fontSize: 28, fontSizeAsian: 28, boldAsian: false, fontSizeComplex: 28, italicAsian: false } },
                                l3: { paragraph: { indentFirstLine: 1200, bulletFont: { followText: false, name: 'Arial' }, bulletSize: { type: 'followText' }, spacingBefore: { type: 'fixed', value: 400 }, bullet: { character: '\u2022', type: 'character' }, indentLeft: 900, spacingAfter: { type: 'fixed', value: 400 } }, character: { color: { type: 'auto' }, italicComplex: false, underline: false, strike: 'none', fontNameComplex: 'Lucida Sans', bold: false, italic: false, caps: 'none', fillColor: { type: 'auto' }, fontName: 'Liberation Sans', boldComplex: false, fontNameAsian: 'Microsoft YaHei', fontSize: 28, fontSizeAsian: 28, boldAsian: false, fontSizeComplex: 28, italicAsian: false } },
                                l4: { paragraph: { indentFirstLine: 1800, bulletFont: { followText: false, name: 'Arial' }, bulletSize: { type: 'followText' }, spacingBefore: { type: 'fixed', value: 400 }, bullet: { character: '\u2022', type: 'character' }, indentLeft: 900, spacingAfter: { type: 'fixed', value: 400 } }, character: { color: { type: 'auto' }, italicComplex: false, underline: false, strike: 'none', fontNameComplex: 'Lucida Sans', bold: false, italic: false, caps: 'none', fillColor: { type: 'auto' }, fontName: 'Liberation Sans', boldComplex: false, fontNameAsian: 'Microsoft YaHei', fontSize: 28, fontSizeAsian: 28, boldAsian: false, fontSizeComplex: 28, italicAsian: false } },
                                l5: { paragraph: { indentFirstLine: 2400, bulletFont: { followText: false, name: 'Arial' }, bulletSize: { type: 'followText' }, spacingBefore: { type: 'fixed', value: 400 }, bullet: { character: '\u2022', type: 'character' }, indentLeft: 900, spacingAfter: { type: 'fixed', value: 400 } }, character: { color: { type: 'auto' }, italicComplex: false, underline: false, strike: 'none', fontNameComplex: 'Lucida Sans', bold: false, italic: false, caps: 'none', fillColor: { type: 'auto' }, fontName: 'Liberation Sans', boldComplex: false, fontNameAsian: 'Microsoft YaHei', fontSize: 28, fontSizeAsian: 28, boldAsian: false, fontSizeComplex: 28, italicAsian: false } },
                                l6: { paragraph: { indentFirstLine: 3000, bulletFont: { followText: false, name: 'Arial' }, bulletSize: { type: 'followText' }, spacingBefore: { type: 'fixed', value: 400 }, bullet: { character: '\u2022', type: 'character' }, indentLeft: 900, spacingAfter: { type: 'fixed', value: 400 } }, character: { color: { type: 'auto' }, italicComplex: false, underline: false, strike: 'none', fontNameComplex: 'Lucida Sans', bold: false, italic: false, caps: 'none', fillColor: { type: 'auto' }, fontName: 'Liberation Sans', boldComplex: false, fontNameAsian: 'Microsoft YaHei', fontSize: 28, fontSizeAsian: 28, boldAsian: false, fontSizeComplex: 28, italicAsian: false } },
                                l7: { paragraph: { indentFirstLine: 3600, bulletFont: { followText: false, name: 'Arial' }, bulletSize: { type: 'followText' }, spacingBefore: { type: 'fixed', value: 400 }, bullet: { character: '\u2022', type: 'character' }, indentLeft: 900, spacingAfter: { type: 'fixed', value: 400 } }, character: { color: { type: 'auto' }, italicComplex: false, underline: false, strike: 'none', fontNameComplex: 'Lucida Sans', bold: false, italic: false, caps: 'none', fillColor: { type: 'auto' }, fontName: 'Liberation Sans', boldComplex: false, fontNameAsian: 'Microsoft YaHei', fontSize: 28, fontSizeAsian: 28, boldAsian: false, fontSizeComplex: 28, italicAsian: false } },
                                l8: { paragraph: { indentFirstLine: 4200, bulletFont: { followText: false, name: 'Arial' }, bulletSize: { type: 'followText' }, spacingBefore: { type: 'fixed', value: 400 }, bullet: { character: '\u2022', type: 'character' }, indentLeft: 900, spacingAfter: { type: 'fixed', value: 400 } }, character: { color: { type: 'auto' }, italicComplex: false, underline: false, strike: 'none', fontNameComplex: 'Lucida Sans', bold: false, italic: false, caps: 'none', fillColor: { type: 'auto' }, fontName: 'Liberation Sans', boldComplex: false, fontNameAsian: 'Microsoft YaHei', fontSize: 28, fontSizeAsian: 28, boldAsian: false, fontSizeComplex: 28, italicAsian: false } },
                                l9: { paragraph: { indentFirstLine: 4800, bulletFont: { followText: false, name: 'Arial' }, bulletSize: { type: 'followText' }, spacingBefore: { type: 'fixed', value: 400 }, bullet: { character: '\u2022', type: 'character' }, indentLeft: 900, spacingAfter: { type: 'fixed', value: 400 } }, character: { color: { type: 'auto' }, italicComplex: false, underline: false, strike: 'none', fontNameComplex: 'Lucida Sans', bold: false, italic: false, caps: 'none', fillColor: { type: 'auto' }, fontName: 'Liberation Sans', boldComplex: false, fontNameAsian: 'Microsoft YaHei', fontSize: 28, fontSizeAsian: 28, boldAsian: false, fontSizeComplex: 28, italicAsian: false } }
                            }
                        },
                        fill: { type: 'solid' }
                    }
                },
                { name: 'insertDrawing', start: [0, 0], type: 'shape', attrs: { presentation: { phType: 'title' }, drawing: { top: 837, left: 1400, width: 25199, height: 3506 } }, target: masterId_1 },
                { name: 'insertParagraph', start: [0, 0, 0], target: masterId_1, attrs: { paragraph: { bullet: { type: 'none' } } } },
                { name: 'insertText', start: [0, 0, 0, 0], target: masterId_1, text: 'Click to edit Master title style' },
                { name: 'insertDrawing', start: [0, 1], type: 'shape', attrs: { presentation: { phType: 'body' }, drawing: { top: 4914, left: 1400, width: 25199, height: 12179 } }, target: masterId_1 },
                { name: 'insertParagraph', start: [0, 1, 0], target: masterId_1, attrs: { paragraph: { level: 0 } } },
                { name: 'insertText', start: [0, 1, 0, 0], target: masterId_1, text: 'Edit Master text styles' },
                { name: 'insertParagraph', start: [0, 1, 1], target: masterId_1, attrs: { paragraph: { level: 1 } } },
                { name: 'insertText', start: [0, 1, 1, 0], target: masterId_1, text: 'Second level' },
                { name: 'insertSlide', start: [0], target: masterId_1, attrs: { slide: { isSlideNum: false, isDate: false, isFooter: false } } },
                { name: 'insertDrawing', start: [0, 0], type: 'shape', attrs: { presentation: { userTransformed: true, phType: 'title' }, drawing: { top: 837, left: 1400, width: 25199, height: 3506 } } },
                { name: 'insertParagraph', start: [0, 0, 0] },
                { name: 'insertText', start: [0, 0, 0, 0], text: 'Title' },
                { name: 'insertDrawing', start: [0, 1], type: 'shape', attrs: { presentation: { phType: 'body' }, drawing: { top: 4914, left: 1400, width: 25199, height: 12179 } } },
                { name: 'insertParagraph', start: [0, 1, 0], attrs: { paragraph: { level: 0, listStyleId: 'L1' } } },
                { name: 'insertText', start: [0, 1, 0, 0], text: 'Item 1' },
                { name: 'insertParagraph', start: [0, 1, 1], attrs: { paragraph: { level: 0, listStyleId: 'L1' } } },
                { name: 'insertText', start: [0, 1, 1, 0], text: 'Item 2' },
                { name: 'insertParagraph', start: [0, 1, 2], attrs: { paragraph: { level: 0, listStyleId: 'L1' } } },
                { name: 'insertText', start: [0, 1, 2, 0], text: 'Item 3' },
                { name: 'insertParagraph', start: [0, 1, 3], attrs: { paragraph: { level: 0, listStyleId: 'L1' } } },
                { name: 'insertText', start: [0, 1, 3, 0], text: 'Item 4' },
                { name: 'insertParagraph', start: [0, 1, 4], attrs: { paragraph: { level: 0, listStyleId: 'L1' } } },
                { name: 'insertText', start: [0, 1, 4, 0], text: 'Item 5' },
                { name: 'insertParagraph', start: [0, 1, 5], attrs: { paragraph: { level: 0, listStyleId: 'L1' } } },
                { name: 'insertText', start: [0, 1, 5, 0], text: 'Item 6' }
            ];

        var model;
        AppHelper.createPresentationApp('odf', OPERATIONS).done(function (app) {
            model = app.getModel();
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(ListHandlerMixin).to.be.a('function');
        });

        // public methods -----------------------------------------------------

        describe('method "isStandardSlideId"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('isStandardSlideId');
            });
            it('should return whether the specified ID is the ID of a normal (standard) slide', function () {
                expect(model.isStandardSlideId(masterId_1)).to.equal(false);
                expect(model.isStandardSlideId(slideDefaultName)).to.equal(true);
            });
        });

        describe('method "getActiveSlideId"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getActiveSlideId');
            });
            it('should return the ID of the active slide', function () {
                expect(model.getActiveSlideId()).to.equal(slideDefaultName);
            });
        });

        describe('method "DOM.isListParagraphNode"', function () {
            it('should exist', function () {
                expect(DOM).to.respondTo('isListParagraphNode');
            });
            it('should check whether the specified node is a list paragraph', function () {
                model.getSelection().setTextSelection([0, 0, 0, 0]);
                firstStandardDrawing = model.getSelection().getSelectedTextFrameDrawing();
                expect(firstStandardDrawing.length).to.equal(1);
                firstStandardParagraphs = firstStandardDrawing.find(DOM.PARAGRAPH_NODE_SELECTOR);
                expect(firstStandardParagraphs.length).to.equal(1);
                model.getSelection().setTextSelection([0, 1, 0, 0]);
                secondStandardDrawing = model.getSelection().getSelectedTextFrameDrawing();
                expect(secondStandardDrawing.length).to.equal(1);
                secondStandardParagraphs = secondStandardDrawing.find(DOM.PARAGRAPH_NODE_SELECTOR);
                expect(secondStandardParagraphs.length).to.equal(6);
                expect(DOM.isListParagraphNode(firstStandardParagraphs[0])).to.equal(false);
                expect(DOM.isListParagraphNode(secondStandardParagraphs[0])).to.equal(true);
            });
        });

        describe('method "DOM.isListLabelNode"', function () {
            it('should exist', function () {
                expect(DOM).to.respondTo('isListLabelNode');
            });
            it('should check whether the specified node is a list label in a list paragraph', function () {
                expect(DOM.isListLabelNode(firstStandardParagraphs[0].childNodes[0])).to.equal(false);
                expect(DOM.isListLabelNode(secondStandardParagraphs[0].childNodes[0])).to.equal(true);
            });
        });

        describe('method "isListIndentChangeable"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('isListIndentChangeable');
            });
            it('should check whether the list level at the currently selected paragraph is changeable', function () {
                model.getSelection().setTextSelection([0, 0, 0, 0]);
                expect(model.isListIndentChangeable()).to.equal(false);
                model.getSelection().setTextSelection([0, 1, 0, 0]);
                expect(model.isListIndentChangeable()).to.equal(true);
            });
        });

        describe('method "getMaxListLevel"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getMaxListLevel');
            });
            it('should receive the maximum list level for the drawing that contains the selection', function () {
                model.getSelection().setTextSelection([0, 0, 0, 0]);
                expect(model.getMaxListLevel()).to.equal(0);
                model.getSelection().setTextSelection([0, 1, 0, 0]);
                expect(model.getMaxListLevel()).to.equal(8);
            });
        });

        describe('method "changeListIndent"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('changeListIndent');
            });
            it('should change the level of list indent for the specified paragraph', function () {

                expect($(firstStandardParagraphs[0].childNodes[0]).text()).to.equal('Title');

                expect(secondStandardParagraphs[0].childNodes.length).to.equal(2);
                expect($(secondStandardParagraphs[0].childNodes[0]).text()).to.equal('-');
                expect($(secondStandardParagraphs[0].childNodes[1]).text()).to.equal('Item 1');

                model.getSelection().setTextSelection([0, 0, 0, 0]);

                explicitAttributes = AttributeUtils.getExplicitAttributes(firstStandardParagraphs[0]);
                expect(explicitAttributes.paragraph).to.be.undefined;

                model.changeListIndent(); // increasing indent of first paragraph

                explicitAttributes = AttributeUtils.getExplicitAttributes(firstStandardParagraphs[0]);
                expect(explicitAttributes.paragraph).to.be.undefined;  // no effect in title

                model.changeListIndent({ increase: false }); // decreasing indent of first paragraph

                explicitAttributes = AttributeUtils.getExplicitAttributes(firstStandardParagraphs[0]);
                expect(explicitAttributes.paragraph).to.be.undefined;  // no effect in title

                model.getSelection().setTextSelection([0, 1, 0, 0]);

                model.changeListIndent(); // increasing indent of first paragraph

                explicitAttributes = AttributeUtils.getExplicitAttributes(secondStandardParagraphs[0]);
                expect(explicitAttributes.paragraph.level).to.equal(1);

                model.changeListIndent(); // increasing indent of first paragraph

                explicitAttributes = AttributeUtils.getExplicitAttributes(secondStandardParagraphs[0]);
                expect(explicitAttributes.paragraph.level).to.equal(2);

                model.changeListIndent({ increase: false }); // decreasing indent of first paragraph

                explicitAttributes = AttributeUtils.getExplicitAttributes(secondStandardParagraphs[0]);
                expect(explicitAttributes.paragraph.level).to.equal(1);

                model.changeListIndent({ increase: false }); // decreasing indent of first paragraph

                explicitAttributes = AttributeUtils.getExplicitAttributes(secondStandardParagraphs[0]);
                expect(explicitAttributes.paragraph.level).to.equal(0);

            });
        });

        // checks with empty paragraphs (using asynchronous list update via 'updateListsDebounced')

        describe('method "updateLists"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('updateLists');
            });

            it('should have a correct prepared drawing on standard slide with 6 paragraphs', function () {

                expect(secondStandardParagraphs.length).to.equal(6);

                para1 = $(secondStandardParagraphs[0]);
                para2 = $(secondStandardParagraphs[1]);
                para3 = $(secondStandardParagraphs[2]);
                para4 = $(secondStandardParagraphs[3]);
                para5 = $(secondStandardParagraphs[4]);
                para6 = $(secondStandardParagraphs[5]);

                expect(DOM.isListParagraphNode(para1)).to.equal(true);
                expect(DOM.isListParagraphNode(para2)).to.equal(true);
                expect(DOM.isListParagraphNode(para3)).to.equal(true);
                expect(DOM.isListParagraphNode(para4)).to.equal(true);
                expect(DOM.isListParagraphNode(para5)).to.equal(true);
                expect(DOM.isListParagraphNode(para6)).to.equal(true);
            });
        });

        describe('method "isNumberedListParagraph"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('isNumberedListParagraph');
            });

            it('should check that paragraphs are correctly switched from bullet list to numbered list', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para1).paragraph)).to.equal(false);
                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para2).paragraph)).to.equal(false);
                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para3).paragraph)).to.equal(false);
                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para4).paragraph)).to.equal(false);
                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para5).paragraph)).to.equal(false);
                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para6).paragraph)).to.equal(false);

                model.getSelection().setTextSelection([0, 1, 0, 0], [0, 1, 5, 1]); // selecting all six paragraphs

                model.setListStyleId('numbering', ListHandlerMixin.DEFAULT_NUMBERING_TYPE, ''); // setting numbering list style to all paragraphs

                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para1).paragraph)).to.equal(true);
                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para2).paragraph)).to.equal(true);
                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para3).paragraph)).to.equal(true);
                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para4).paragraph)).to.equal(true);
                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para5).paragraph)).to.equal(true);
                expect(model.isNumberedListParagraph(model.getParagraphStyles().getElementAttributes(para6).paragraph)).to.equal(true);

                // debounced list update -> synchronous check: still the old list bullets
                expect(para1.children().first().text()).to.equal('-');
                expect(para2.children().first().text()).to.equal('-');
                expect(para3.children().first().text()).to.equal('-');
                expect(para4.children().first().text()).to.equal('-');
                expect(para5.children().first().text()).to.equal('-');
                expect(para6.children().first().text()).to.equal('-');

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    // checking state after formatting of list
                    expect(para1.children().first().text()).to.equal('1.');
                    expect(para2.children().first().text()).to.equal('2.');
                    expect(para3.children().first().text()).to.equal('3.');
                    expect(para4.children().first().text()).to.equal('4.');
                    expect(para5.children().first().text()).to.equal('5.');
                    expect(para6.children().first().text()).to.equal('6.');
                    done();
                });
            });

        });

        describe('empty list paragraphs', function () {

            it('should be counted in numbered lists, not like in OOXML', function (done) {

                var deletePromise = null;
                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', $.noop); // avoid resolving 'listFormattingDef' too early

                model.getSelection().setTextSelection([0, 1, 1, 0], [0, 1, 1, 6]); // selecting all content of second paragraph

                deletePromise = model.deleteSelected();
                deletePromise.always(function () {

                    expect(deletePromise.state()).to.equal('resolved');

                    model.on('listformatting:done', function () {
                        listFormattingDef.resolve();
                    });

                    listFormattingDef.always(function () {

                        expect(listFormattingDef.state()).to.equal('resolved');

                        expect(para1.children().first().text()).to.equal('1.');
                        expect(para2.children().first().text()).to.equal('2.'); // empty paragraph gets number increased by 1 and does NOT stop further increase
                        expect(para3.children().first().text()).to.equal('3.');
                        expect(para4.children().first().text()).to.equal('4.');
                        expect(para5.children().first().text()).to.equal('5.');
                        expect(para6.children().first().text()).to.equal('6.');
                        done();
                    });
                });

            });

            it('should have the class attributes that mark the paragraph as empty list paragraph', function () {
                expect(para2.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS)).to.equal(true);
            });

            it('should be counted in numbered lists if text is deleted (preparation for undo), not like in OOXML', function (done) {

                var deletePromise = null;
                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', $.noop); // avoid resolving 'listFormattingDef' too early

                model.getSelection().setTextSelection([0, 1, 3, 0], [0, 1, 3, 6]); // selecting all content of fourth paragraph

                deletePromise = model.deleteSelected();
                deletePromise.always(function () {

                    expect(deletePromise.state()).to.equal('resolved');

                    model.on('listformatting:done', function () {
                        listFormattingDef.resolve();
                    });

                    listFormattingDef.always(function () {

                        expect(listFormattingDef.state()).to.equal('resolved');

                        expect(para1.children().first().text()).to.equal('1.');
                        expect(para2.children().first().text()).to.equal('2.'); // empty paragraph
                        expect(para3.children().first().text()).to.equal('3.');
                        expect(para4.children().first().text()).to.equal('4.'); // empty paragraph
                        expect(para5.children().first().text()).to.equal('5.');
                        expect(para6.children().first().text()).to.equal('6.');

                        expect(para4.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS)).to.equal(true);

                        done();
                    });
                });

            });

            it('should be counted in numbered lists if text is inserted via undo', function (done) {

                var undoPromise = model.getUndoManager().undo();
                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', $.noop); // avoid resolving 'listFormattingDef' too early

                undoPromise.always(function () {

                    expect(undoPromise.state()).to.equal('resolved');

                    model.on('listformatting:done', function () {
                        listFormattingDef.resolve();
                    });

                    listFormattingDef.always(function () {

                        expect(para1.children().first().text()).to.equal('1.');
                        expect(para2.children().first().text()).to.equal('2.'); // empty paragraph
                        expect(para3.children().first().text()).to.equal('3.');
                        expect(para4.children().first().text()).to.equal('4.'); // no longer empty paragraph
                        expect(para5.children().first().text()).to.equal('5.');
                        expect(para6.children().first().text()).to.equal('6.');

                        expect(para4.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS)).to.equal(false);

                        done();
                    });
                });
            });

            it('should be counted in numbered lists if text is inserted into paragraph', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([0, 1, 1, 0]); // setting cursor into empty paragraph
                model.insertText('c', [0, 1, 1, 0]);  // inserting one character into empty paragraph
                model.getSelection().setTextSelection([0, 1, 1, 1]); // setting cursor behind the new text

                listFormattingDef.always(function () {

                    expect(para1.children().first().text()).to.equal('1.');
                    expect(para2.children().first().text()).to.equal('2.'); // no longer empty paragraph
                    expect(para3.children().first().text()).to.equal('3.');
                    expect(para4.children().first().text()).to.equal('4.');
                    expect(para5.children().first().text()).to.equal('5.');
                    expect(para6.children().first().text()).to.equal('6.');

                    expect(para2.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS)).to.equal(false);

                    done();
                });
            });

            // additional checks for synchronous list update (calling model.updateLists() directly)

            it('should be counted in numbered lists if text is deleted (synchronous list update), not like in OOXML', function (done) {

                var deletePromise = null;

                model.getSelection().setTextSelection([0, 1, 1, 0], [0, 1, 1, 1]); // selecting all content of second paragraph

                deletePromise = model.deleteSelected();
                deletePromise.always(function () {

                    expect(deletePromise.state()).to.equal('resolved');

                    // updating lists synchronously
                    model.updateLists(null, secondStandardParagraphs);

                    expect(para1.children().first().text()).to.equal('1.');
                    expect(para2.children().first().text()).to.equal('2.'); // empty paragraph
                    expect(para3.children().first().text()).to.equal('3.');
                    expect(para4.children().first().text()).to.equal('4.');
                    expect(para5.children().first().text()).to.equal('5.');
                    expect(para6.children().first().text()).to.equal('6.');

                    expect(para2.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS)).to.equal(true);

                    done();
                });

            });

            it('should be counted in numbered lists if text is inserted into paragraph (synchronous list update)', function () {

                model.getSelection().setTextSelection([0, 1, 1, 0]); // setting cursor into empty paragraph
                model.insertText('c', [0, 1, 1, 0]);  // inserting one character into empty paragraph
                model.getSelection().setTextSelection([0, 1, 1, 1]); // setting cursor behind the new text

                // updating lists synchronously (this is done debounced during working with document)
                model.updateLists(null, secondStandardParagraphs);

                expect(para1.children().first().text()).to.equal('1.');
                expect(para2.children().first().text()).to.equal('2.'); // no longer empty paragraph
                expect(para3.children().first().text()).to.equal('3.');
                expect(para4.children().first().text()).to.equal('4.');
                expect(para5.children().first().text()).to.equal('5.');
                expect(para6.children().first().text()).to.equal('6.');

                expect(para2.hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS)).to.equal(false);
            });

        });

    });

});
