/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */
define([
    'globals/apphelper',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/model/groupoperationmixin',
    'io.ox/office/textframework/utils/dom'
], function (AppHelper, DrawingFrame, GroupOperationMixin, DOM) {

    'use strict';

    // class PresentationModel ================================================

    describe('Presentation class GroupOperationMixin', function () {

        // private helpers ----------------------------------------------------

        var model = null,
            layoutId_1 = 'layout1',
            layoutId_2 = 'layout2',
            masterId_1 = 'master1',
            slide1DefaultName = 'slide_1',
            // slide2DefaultName = 'slide_2',
            activeSlide = null,

            // the operations to be applied by the document model
            OPERATIONS = [
                { name: 'setDocumentAttributes', attrs: { page: { width: 33866, height: 19050, orientation: 'landscape' } } },
                { name: 'insertMasterSlide', id: masterId_1 },
                { name: 'insertLayoutSlide', id: layoutId_1, target: masterId_1 },
                { name: 'insertLayoutSlide', id: layoutId_2, target: masterId_1 },
                { name: 'insertSlide', start: [0], target: layoutId_1, attrs: { fill: { type: 'solid', color: { type: 'rgb', value: 'FF0000' } } } },
                { name: 'insertDrawing', start: [0, 0], type: 'shape', attrs: { presentation: { phType: 'ctrTitle' }, drawing: { name: 'Title', noGroup: true } } },
                { name: 'insertParagraph', start: [0, 0, 0] },
                { name: 'insertText', text: 'Title', start: [0, 0, 0, 0] },
                { name: 'insertDrawing', start: [0, 1], type: 'shape', attrs: { presentation: { phType: 'subTitle', phIndex: 1 }, drawing: { name: 'Subtitle', noGroup: true } } },
                { name: 'insertParagraph', start: [0, 1, 0] },
                { name: 'insertText', text: 'SubTitle', start: [0, 1, 0, 0] },
                { name: 'insertDrawing', start: [0, 2], type: 'shape', attrs: { drawing: { name: 'rect', width: 2000, height: 3000, left: 1000, top: 1000 }, geometry: { presetShape: 'rect', avList: {} } } },
                { name: 'insertParagraph', start: [0, 2, 0] },
                { name: 'insertText', text: 'Shape 1', start: [0, 2, 0, 0] },
                { name: 'insertDrawing', start: [0, 3], type: 'shape', attrs: { drawing: { name: 'rect', width: 2000, height: 2000, left: 3000, top: 3000 }, geometry: { presetShape: 'rect', avList: {} } } },
                { name: 'insertParagraph', start: [0, 3, 0] },
                { name: 'insertText', text: 'Shape 2', start: [0, 3, 0, 0] },
                { name: 'insertDrawing', start: [0, 4], type: 'shape', attrs: { drawing: { name: 'rect', width: 2000, height: 3000, left: 6000, top: 6000 }, geometry: { presetShape: 'rect', avList: {} } } },
                { name: 'insertParagraph', start: [0, 4, 0] },
                { name: 'insertText', text: 'Shape 4', start: [0, 4, 0, 0] },
                { name: 'insertSlide', start: [1], target: layoutId_1, attrs: { fill: { type: 'solid', color: { type: 'rgb', value: 'FF0000' } } } },
                { name: 'insertDrawing', start: [1, 0], type: 'shape', attrs: { presentation: { phType: 'ctrTitle' }, drawing: { name: 'Titel 1', noGroup: true } } },
                { name: 'insertParagraph', start: [1, 0, 0] },
                { name: 'insertText', text: 'Slide 2', start: [1, 0, 0, 0] }
            ];

        AppHelper.createPresentationApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(GroupOperationMixin).to.be.a('function');
        });

        // public methods -----------------------------------------------------

        describe('method "getSlideById"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getSlideById');
            });
            it('should return the standard IDs for the active slides', function () {
                // checking if the document is correctly prepared
//                model.getSelection().setTextSelection([1, 0, 0, 0]);  // selecting the second slide
//                expect(model.getActiveSlideId()).to.equal(slide2DefaultName);
//                activeSlide = model.getSlideById(model.getActiveSlideId());
//                expect(activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(1);

                model.getSelection().setTextSelection([0, 0, 0, 0]);  // selecting the first slide
                activeSlide = model.getSlideById(model.getActiveSlideId());
                expect(model.getActiveSlideId()).to.equal(slide1DefaultName);
                expect(activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(5);
            });
        });

        describe('method "setMultiDrawingSelectionByPosition"', function () {
            it('should exist', function () {
                expect(model.getSelection()).to.respondTo('setMultiDrawingSelectionByPosition');
            });
            it('should allow to set a multi drawing selection with the specified logical positions', function () {
                // checking if the document is correctly prepared
                model.getSelection().setMultiDrawingSelectionByPosition([[0, 1], [0, 2], [0, 3]]);  // selecting three drawings by position
                expect(model.getSelection().getMultiSelection().length).to.equal(3);

                model.getSelection().setMultiDrawingSelectionByPosition([[0, 1], [0, 2]]);  // selecting two drawings by position
                expect(model.getSelection().getMultiSelection().length).to.equal(2);

                model.getSelection().clearMultiSelection();  // selecting three drawings by position
                expect(model.getSelection().getMultiSelection().length).to.equal(0);

                model.getSelection().setMultiDrawingSelectionByPosition([[0, 1], [0, 2], [0, 4]]);  // selecting three drawings by position
                expect(model.getSelection().getMultiSelection().length).to.equal(3);
            });
        });

        describe('method "setMultiDrawingSelection"', function () {
            it('should exist', function () {
                expect(model.getSelection()).to.respondTo('setMultiDrawingSelection');
            });
            it('should allow to set a multi drawing selection with the specified drawing nodes', function () {
                var allDrawings = null;
                activeSlide = model.getSlideById(model.getActiveSlideId());
                allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);

                expect(allDrawings.length).to.equal(5);

                model.getSelection().setMultiDrawingSelection([allDrawings[0], allDrawings[1], allDrawings[3]]);  // selecting three drawings by node
                expect(model.getSelection().getMultiSelection().length).to.equal(3);
            });
        });

        describe('method "isGroupableSelection"', function () {
            it('should exist', function () {
                expect(model.getSelection()).to.respondTo('isGroupableSelection');
            });
            it('should return that a multi drawing selection with place holder drawing is not groupable', function () {
                model.getSelection().setMultiDrawingSelectionByPosition([[0, 1], [0, 2], [0, 4]]);
                expect(model.getSelection().isGroupableSelection()).to.equal(false);
            });
            it('should return that a multi drawing selection without place holder drawing is groupable', function () {
                model.getSelection().setMultiDrawingSelectionByPosition([[0, 2], [0, 3], [0, 4]]);
                expect(model.getSelection().isGroupableSelection()).to.equal(true);
            });
        });

        describe('method "groupDrawings"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('groupDrawings');
            });
            it('should group the selected drawings', function () {
                var allDrawings = null;
                model.groupDrawings();
                allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                expect(allDrawings.length).to.equal(3); // only the remaining children
                expect(DrawingFrame.isGroupDrawingFrame(allDrawings[2])).to.equal(true); // at position [0, 2] must be a group
            });
        });

        describe('method "isUngroupableSelection"', function () {
            it('should exist', function () {
                expect(model.getSelection()).to.respondTo('isUngroupableSelection');
            });
            it('should return that a multi drawing selection with place holder drawing is not groupable', function () {
                // model.getSelection().setMultiDrawingSelectionByPosition([[0, 1], [0, 2], [0, 3]]);
                model.getSelection().selectAll();
                expect(model.getSelection().isUngroupableSelection()).to.equal(true);
            });
        });

        describe('method "ungroupDrawings"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('ungroupDrawings');
            });
            it('should ungroup the selected drawings', function () {
                var allDrawings = null;
                model.getSelection().selectAll();
                expect(model.getSelection().getMultiSelection().length).to.equal(3);
                expect(model.getSelection().isUngroupableSelection()).to.equal(true);
                model.ungroupDrawings();
                allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                expect(allDrawings.length).to.equal(5); // only the remaining children
                expect(DrawingFrame.isGroupDrawingFrame(allDrawings[2])).to.equal(false); // at position [0, 2] must no longer be a group
                model.getSelection().selectAll();
                expect(model.getSelection().getMultiSelection().length).to.equal(5);
                expect(model.getSelection().isUngroupableSelection()).to.equal(false);
            });
        });

    });

    // ========================================================================

});
