/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/text/model/model',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/editframework/utils/attributeutils'
], function (AppHelper, DrawingFrame, TextModel, DOM, Position, AttributeUtils) {

    'use strict';

    // class TextModel ================================================

    describe('Text class TextModel', function () {

        // private helpers ----------------------------------------------------

        var model = null,
            firstParagraph = null,
            secondParagraph = null,
            secondParagraphLength = 241;

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes',
                attrs: {
                    document:  { defaultTabStop: 1270, zoom: { value: 100 } },
                    page:      { width: 21590, height: 27940, marginLeft: 2540, marginTop: 2540, marginRight: 2540, marginBottom: 2540, marginHeader: 1248, marginFooter: 1248 },
                    character: { fontName: 'Arial', fontSize: 11, language: 'en-US', languageEa: 'en-US', languageBidi: 'ar-SA' },
                    paragraph: { lineHeight: { type: 'percent', value: 119 }, marginBottom: 352 }
                }
            },
            { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading1', styleName: 'heading 1',
                attrs: {
                    character: { bold: true, fontName: 'Times New Roman', fontSize: 14, color: { transformations: [{ type: 'shade', value: 74902 }], type: 'scheme', value: 'accent1' } },
                    paragraph: { marginTop: 846, outlineLevel: 0, nextStyleId: 'Normal' }
                },
                parent: 'Normal',
                uiPriority: 9
            },
            { name: 'insertParagraph', start: [0] },
            { name: 'insertText', text: 'Hello World.', start: [0, 0] },

            { name: 'splitParagraph', start: [0, 12] },
            { name: 'insertText', start: [1, 0], text: [
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet',
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet',
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet.'
            ].join('') }
        ];

        AppHelper.createTextApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
            var node = model.getNode();
            firstParagraph = $(Position.getParagraphElement(node, [0]));
            secondParagraph = $(Position.getParagraphElement(node, [1]));
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(TextModel).to.be.a('function');
        });

        describe('method "insertShapeWithDefaultBox"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('insertShapeWithDefaultBox');
            });

            it('should insert a shape with default size at the specified position', function () {

                var paraChildren = firstParagraph.children();
                var drawingNode = null;

                expect(paraChildren.length).to.equal(1);
                expect(firstParagraph.text()).to.equal('Hello World.');

                model.getSelection().setTextSelection([0, 6], [0, 7]); // Selecting the 'W' of world

                model.insertShapeWithDefaultBox('rect');

                paraChildren = firstParagraph.children();
                expect(paraChildren.length).to.equal(3);

                // checking the new two text spans
                expect($(paraChildren[0]).text()).to.equal('Hello ');
                expect($(paraChildren[2]).text()).to.equal('World.');

                // checking the new inserted drawing
                // -> inserted at the start point of the selection
                drawingNode = $(paraChildren[1]);

                // checking the assigned classes
                expect(drawingNode.hasClass('drawing')).to.equal(true);
                expect(drawingNode.hasClass('inline')).to.equal(true);
                expect(drawingNode.hasClass('selected')).to.equal(true);
                expect(drawingNode.hasClass('movable')).to.equal(true);
                expect(drawingNode.hasClass('resizable')).to.equal(true);

                // checking the drawing type
                expect(DrawingFrame.isShapeDrawingFrame(paraChildren[1])).to.equal(true);

                // checking the default size
                expect(drawingNode.data().attributes.drawing.width).to.equal(model.getDefaultShapeSize().width);
                expect(drawingNode.data().attributes.drawing.height).to.equal(model.getDefaultShapeSize().height);
            });

        });

        describe('method "insertShape"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('insertShape');
            });

            // test if shapes are inserted with the correct fill style
            it('Text: should insert a shape with fill', function () {
                var paraChildren = firstParagraph.children();
                // check inital state
                expect(paraChildren.length).to.equal(3);

                // insert a shape with fill
                model.insertShape('ellipse', { width: 200, height: 200 });
                paraChildren = firstParagraph.children();
                expect(paraChildren.length).to.equal(5);

                // checking the new inserted drawing
                // -> inserted at the start point of the selection
                var drawingNode = $(paraChildren[1]);

                expect(AttributeUtils.getExplicitAttributes(drawingNode).fill.type).to.equal('solid');
                expect(AttributeUtils.getExplicitAttributes(drawingNode).drawing.name).to.equal('ellipse');
            });

            it('Text: should insert a shape with no fill', function () {
                var paraChildren = firstParagraph.children();
                // check inital state
                expect(paraChildren.length).to.equal(5);

                // insert a shape with no fill
                model.insertShape('leftBrace', { width: 200, height: 200 });
                paraChildren = firstParagraph.children();
                expect(paraChildren.length).to.equal(7);

                // checking the new inserted drawing
                // -> inserted at the start point of the selection
                var drawingNode = $(paraChildren[1]);

                expect(AttributeUtils.getExplicitAttributes(drawingNode).fill).to.deep.equal({ type: 'none' });
                expect(AttributeUtils.getExplicitAttributes(drawingNode).drawing.name).to.equal('leftBrace');
            });
        });

        describe('Checks if hint for non auto resize shape is set', function () {
            var drawingNode = $();
            before(function (done) {
                model.getSelection().setTextSelection([0, 0]);
                $('body').append(model.getNode());
                model.insertShape('rect', { width: 3000, height: 800 }).then(function () {
                    model.getSelection().setTextSelection([0, 0, 0, 0]);
                    drawingNode = model.getSelection().getAnyTextFrameDrawing({ forceTextFrame: true });
                    model.handleTextFrameAutoFit(false);
                    model.insertText('lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet, lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet', [0, 0, 0, 0]);
                    done();
                });
            });

            it('Should check if hint arrow is set', function () {
                expect(drawingNode.children('.textframecontent').hasClass('b-arrow')).to.equal(true);
            });
            after(function (done) {
                model.getNode().remove();
                done();
            });
        });

        // checking the selection after undo
        describe('Check, if selection is set correctly after undo', function () {

            it('Checks selection after undo and redo of setAttributes operation', function (done) {

                var boldSpan = null;
                var selection = model.getSelection();
                var promise = null; // the undo promise

                expect(secondParagraph.children().length).to.equal(1);
                expect(Position.getParagraphNodeLength(secondParagraph)).to.equal(secondParagraphLength);

                selection.setTextSelection([1, 6], [1, 11]); // Selecting 'ipsum'

                // marking word as 'bold'
                model.setAttribute('character', 'bold', true);

                // checking number of children in paragraph
                expect(secondParagraph.children().length).to.equal(3);
                expect(Position.getParagraphNodeLength(secondParagraph)).to.equal(secondParagraphLength);

                boldSpan = $(secondParagraph.children()[1]);
                expect(boldSpan.css('font-weight')).to.equal('bold');

                // setting the cursor to another position
                selection.setTextSelection([1, 2]);

                expect(selection.isTextCursor()).to.equal(true);
                expect(selection.getStartPosition()[0]).to.equal(1);
                expect(selection.getStartPosition()[1]).to.equal(2);
                expect(selection.getEndPosition()[0]).to.equal(1);
                expect(selection.getEndPosition()[1]).to.equal(2);

                promise = model.getUndoManager().undo();

                promise.always(function () {

                    expect(promise.state()).to.equal('resolved');

                    expect(secondParagraph.children().length).to.equal(1); // only one span left
                    expect(Position.getParagraphNodeLength(secondParagraph)).to.equal(secondParagraphLength);

                    // the old selection must be restored after undo
                    expect(selection.isTextCursor()).to.equal(false);
                    expect(selection.getStartPosition()[0]).to.equal(1);
                    expect(selection.getStartPosition()[1]).to.equal(6);
                    expect(selection.getEndPosition()[0]).to.equal(1);
                    expect(selection.getEndPosition()[1]).to.equal(11);

                    // setting the cursor to another position
                    selection.setTextSelection([1, 2]);

                    expect(selection.isTextCursor()).to.equal(true);
                    expect(selection.getStartPosition()[0]).to.equal(1);
                    expect(selection.getStartPosition()[1]).to.equal(2);
                    expect(selection.getEndPosition()[0]).to.equal(1);
                    expect(selection.getEndPosition()[1]).to.equal(2);

                    // also after redo the selection must be restored correctly
                    promise = model.getUndoManager().redo();

                    promise.always(function () {

                        expect(promise.state()).to.equal('resolved');

                        expect(secondParagraph.children().length).to.equal(3); // three spans
                        expect(Position.getParagraphNodeLength(secondParagraph)).to.equal(secondParagraphLength);

                        boldSpan = $(secondParagraph.children()[1]);
                        expect(boldSpan.css('font-weight')).to.equal('bold');

                        // the old selection must be restored after undo
                        expect(selection.isTextCursor()).to.equal(false);
                        expect(selection.getStartPosition()[0]).to.equal(1);
                        expect(selection.getStartPosition()[1]).to.equal(6);
                        expect(selection.getEndPosition()[0]).to.equal(1);
                        expect(selection.getEndPosition()[1]).to.equal(11);

                        // and set to old state again
                        promise = model.getUndoManager().undo();
                        promise.always(function () {
                            expect(promise.state()).to.equal('resolved');
                            expect(secondParagraph.children().length).to.equal(1); // one span
                            expect(Position.getParagraphNodeLength(secondParagraph)).to.equal(secondParagraphLength);
                            done();
                        });
                    });

                });

            });

            it('Checks selection after undo of delete operation', function (done) {

                var selection = model.getSelection();
                var promise = null; // the undo promise

                expect(secondParagraph.children().length).to.equal(1);
                expect(Position.getParagraphNodeLength(secondParagraph)).to.equal(secondParagraphLength);

                // selecting one word
                expect(selection.isTextCursor()).to.equal(false);
                expect(selection.getStartPosition()[0]).to.equal(1);
                expect(selection.getStartPosition()[1]).to.equal(6);
                expect(selection.getEndPosition()[0]).to.equal(1);
                expect(selection.getEndPosition()[1]).to.equal(11);

                model.deleteSelected({ deleteKey: true });

                expect(secondParagraph.children().length).to.equal(1); // still one span
                expect(Position.getParagraphNodeLength(secondParagraph)).to.equal(secondParagraphLength - 5); // word removed

                expect(selection.isTextCursor()).to.equal(true); // text cursor at start of delete range
                expect(selection.getStartPosition()[0]).to.equal(1);
                expect(selection.getStartPosition()[1]).to.equal(6);
                expect(selection.getEndPosition()[0]).to.equal(1);
                expect(selection.getEndPosition()[1]).to.equal(6);

                // setting the cursor to another position
                selection.setTextSelection([1, 2]);

                expect(selection.isTextCursor()).to.equal(true); // text cursor at start of delete range
                expect(selection.getStartPosition()[0]).to.equal(1);
                expect(selection.getStartPosition()[1]).to.equal(2);
                expect(selection.getEndPosition()[0]).to.equal(1);
                expect(selection.getEndPosition()[1]).to.equal(2);

                // inserting bringing drawing1 to the front again
                promise = model.getUndoManager().undo();

                promise.always(function () {

                    expect(promise.state()).to.equal('resolved');

                    expect(secondParagraph.children().length).to.equal(1); // only one span after inserting the text again
                    expect(Position.getParagraphNodeLength(secondParagraph)).to.equal(secondParagraphLength);

                    // the old selection must be restored after undo
                    expect(selection.isTextCursor()).to.equal(false);
                    expect(selection.getStartPosition()[0]).to.equal(1);
                    expect(selection.getStartPosition()[1]).to.equal(6);
                    expect(selection.getEndPosition()[0]).to.equal(1);
                    expect(selection.getEndPosition()[1]).to.equal(11);

                    done();
                });

            });
        });

    });

    // test for ODF files

    describe('Text class TextModel in ODF', function () {

        // private helpers ----------------------------------------------------

        var model = null,
            firstParagraph = null,
            drawings = null,
            drawing = null,
            canvasNode = null,
            drawingAttrs = null,
            drawing1Name = 'Rect1',
            drawing2Name = 'Heart1',
            drawingWidth = 0,
            drawingHeight = 0,
            canvasWidth = 0,
            canvasHeight = 0,
            drawing1Text = 'Rect',
            drawing2Text = 'Heart',
            drawing1WidthHmm = 2000,
            drawing1HeightHmm = 3700,
            drawing2WidthHmm = 4000,
            drawing2HeightHmm = 5000,
            drawing1WidthPx = 76,
            drawing1HeightPx = 140,
            canvas1WidthPx = drawing1WidthPx + 2,
            canvas1HeightPx = drawing1HeightPx + 2,
            drawing2WidthPx = 151,
            drawing2HeightPx = 189,
            canvas2WidthPx = drawing2WidthPx + 2,
            canvas2HeightPx = drawing2HeightPx + 2,
            tolerance = 2,

            // Info: The width of the drawing is adapted to the page width. This is set in this case to 21 cm minus 4 cm for the margins.
            //       Therefore there is a call in pagestyles.js: page.css({ width: '170mm' })
            //       The jQuery call in a unit test 'page.width()' then returns simply '170' which is interpreted in the normal code as
            //       pixel value. Therefore in a unit test the page width is reduced to '170 px' and the drawing width is automatically
            //       reduced to this value, if it is larger.

            // the operations to be applied by the document model
            OPERATIONS = [
                { name: 'setDocumentAttributes',
                    attrs: {
                        document:  { defaultTabStop: 1270, fileFormat: 'odf', zoom: { value: 100 } },
                        page:      { width: 21001, height: 29700, marginLeft: 2000, marginTop: 2000, marginRight: 2000, marginBottom: 2000 }
                    }
                },
                { name: 'insertStyleSheet', type: 'drawing', styleId: '_default', default: true, attrs: { character: { fontName: 'Liberation·Serif', color: { type: 'auto' }, fontNameAsian: 'SimSun', fontNameComplex: 'Lucida·Sans', language: 'de-DE', fontSize: 12, fontSizeAsian: 10.5, fontSizeComplex: 12 }, line: { color: { type: 'rgb', value: '3465a4' }, style: 'solid', type: 'solid' }, fill: { color: { type: 'rgb', value: '729fcf' }, type: 'solid' } } },
                { name: 'insertStyleSheet', type: 'paragraph', styleId: '_default', default: true, attrs: { character: { fontName: 'Liberation·Serif', color: { type: 'auto' }, fontNameAsian: 'SimSun', fontNameComplex: 'Lucida·Sans', language: 'de-DE', fontSize: 12, fontSizeAsian: 10.5, fontSizeComplex: 12 } } },
                { name: 'insertParagraph', start: [0], attrs: { styleId: 'Standard' } },
                { name: 'insertDrawing', start: [0, 0], type: 'shape', attrs: { shape: { paddingBottom: 200, paddingRight: 600, paddingTop: 200, paddingLeft: 600 }, line: { width: 26, style: 'solid', type: 'solid' }, drawing: { name: drawing1Name, anchorHorOffset: 2302, anchorVertOffset: 1270, anchorHorBase: 'column', top: 1200, inline: false, left: 2300, anchorHorAlign: 'offset', anchorVertAlign: 'offset', width: drawing1WidthHmm, anchorVertBase: 'paragraph', height: drawing1HeightHmm }, geometry: { presetShape: 'rect', hostData: 'H4sIAAAAAAAAAJWRwQrCMAyGX6X0PqbzJlbBs/oA3sKWrYW20Sxu06d3bCooU5BSKPn5viR0VTC0\r\nS4wWYo5FUiEFFL6qoRwcM3Fiid2NooA3ugRfo36LG2Rx+Wco2EkCjFAbPVMztSkX/c20qptq2Ths\r\nt9SNSX+ezPWERhN1wSeMuTzKr+lOINbo/cDsBuPLO7wydVQHvV6N1PkC4iiOjgihV5fPTiVxuHgw\r\n2lPVukJsmvVc+gb+8swnPBZdZeVPUfZV9Jdm8W2vCUs6/eXrO8ipjHsLAgAA\r\n' }, fill: { type: 'solid' } } },
                { name: 'insertParagraph', start: [0, 0, 0], attrs: { paragraph: { alignment: 'center' } } },
                { name: 'insertText', start: [0, 0, 0, 0], text: drawing1Text },
                { name: 'insertDrawing', start: [0, 1], type: 'shape', attrs: { shape: { paddingBottom: 300, paddingRight: 500, paddingTop: 300, paddingLeft: 500 }, line: { width: 26, style: 'solid', type: 'solid' }, drawing: { name: drawing2Name, anchorHorOffset: 10001, anchorVertOffset: 4286, anchorHorBase: 'column', top: 4200, inline: false, left: 10000, anchorHorAlign: 'offset', anchorVertAlign: 'offset', width: drawing2WidthHmm, anchorVertBase: 'paragraph', height: drawing2HeightHmm }, geometry: { presetShape: 'heart', hostData: 'H4sIAAAAAAAAAJXU3WqDMBgG4FuRHLYEE/9aS62wHW8XsLNQowmo6WKqdle/+FeoM4MgiZD4PQkv\r\nH54zSboTrRmprzSDBRUVVfLhjMsVl1JIyITkP6JWpExATsqGgpftlkrFr+tNRXsFiaSkSUCax06a\r\nY2+Y0DBh4DRtcWo57d5EnwDkjM9S+rjRBAjRVyVklEg1rz9veSOKJeBDS97kvutXqMdRj2g5atzz\r\n9eQvK8H8yVz15XyCy3miv+9EcVFPB9Wk0hfIl/vkQlb3kiSgFEXHM8V2QewGR13rvhT/Z2GjhZGt\r\n5Zks17Ny/LUzJLNHMM2RYwUFRgjbQeEmNDAQWUGRAUK20GEja0Z5wZTrW0HHNYTGgA52AcXmNnIj\r\nu4Y0d3doS2019xTSzrOMCW819xx4YCf9ae+ntOG42//Byy/XmZsDIAUAAA==\r\n' }, fill: { type: 'solid' } } },
                { name: 'insertParagraph', start: [0, 1, 0], attrs: { paragraph: { alignment: 'center' } } },
                { name: 'insertText', start: [0, 1, 0, 0], text: drawing2Text }
            ];

        AppHelper.createTextApp('odf', OPERATIONS).done(function (app) {
            model = app.getModel();
            var node = model.getNode();
            firstParagraph  = $(Position.getParagraphElement(node, [0]));
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(TextModel).to.be.a('function');
        });

        describe('Checking the size of the Canvas nodes inside the drawing in ODF text files', function () {

            it('should contain two drawings in the first paragraph', function () {
                drawings = firstParagraph.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                expect(drawings.length).to.equal(2);
            });

            it('drawings in first paragraph should have the correct name', function () {
                drawingAttrs = AttributeUtils.getExplicitAttributes(drawings[0], { family: 'drawing' });
                expect(drawingAttrs.name).to.equal(drawing1Name);
                drawingAttrs = AttributeUtils.getExplicitAttributes(drawings[1], { family: 'drawing' });
                expect(drawingAttrs.name).to.equal(drawing2Name);
            });

            it('drawings in first paragraph should have the correct text content', function () {
                expect($(drawings[0]).text()).to.equal(drawing1Text);
                expect($(drawings[1]).text()).to.equal(drawing2Text);
            });

            it('drawings should have the specified width and height in hmm', function () {
                drawingAttrs = AttributeUtils.getExplicitAttributes(drawings[0], { family: 'drawing' });
                expect(drawingAttrs.width).to.equal(drawing1WidthHmm);
                expect(drawingAttrs.height).to.equal(drawing1HeightHmm);
                drawingAttrs = AttributeUtils.getExplicitAttributes(drawings[1], { family: 'drawing' });
                expect(drawingAttrs.width).to.equal(drawing2WidthHmm);
                expect(drawingAttrs.height).to.equal(drawing2HeightHmm);
            });

            it('first drawing without line ends should have only a slightly increased canvas', function () {

                // first drawing in first paragraph: the rectangle shape
                drawing = $(drawings[0]);

                canvasNode = DrawingFrame.getCanvasNode(drawing);
                expect(canvasNode.length).to.equal(1);

                drawingWidth = drawing.width();
                drawingHeight = drawing.height();
                canvasWidth = canvasNode.width();
                canvasHeight = canvasNode.height();

                expect(drawingWidth).to.be.within(drawing1WidthPx - tolerance, drawing1WidthPx + tolerance);
                expect(drawingHeight).to.be.within(drawing1HeightPx - tolerance, drawing1HeightPx + tolerance);
                expect(canvasWidth).to.be.within(canvas1WidthPx - tolerance, canvas1WidthPx + tolerance);
                expect(canvasHeight).to.be.within(canvas1HeightPx - tolerance, canvas1HeightPx + tolerance);
            });

            it('second drawing without line ends should have only a slightly increased canvas', function () {

                // second drawing in first paragraph: the heart shape
                drawing = $(drawings[1]);
                canvasNode = DrawingFrame.getCanvasNode(drawing);
                expect(canvasNode.length).to.equal(1);

                drawingWidth = drawing.width();
                drawingHeight = drawing.height();
                canvasWidth = canvasNode.width();
                canvasHeight = canvasNode.height();

                expect(drawingWidth).to.be.within(drawing2WidthPx - tolerance, drawing2WidthPx + tolerance);
                expect(drawingHeight).to.be.within(drawing2HeightPx - tolerance, drawing2HeightPx + tolerance);
                expect(canvasWidth).to.be.within(canvas2WidthPx - tolerance, canvas2WidthPx + tolerance);
                expect(canvasHeight).to.be.within(canvas2HeightPx - tolerance, canvas2HeightPx + tolerance);
            });

        });

    });

    // test for grouped shapes in ODF files

    describe('Grouped Shapes in class TextModel in ODF', function () {

        // private helpers ----------------------------------------------------

        var model = null,
            firstParagraph = null,
            groupDrawing = null,
            groupedDrawings = null,
            groupedDrawing1 = null,
            groupedDrawing2 = null,
            drawing1Text = 'Hello',
            drawing2Text = 'World',
            groupDrawingHeightPx = 268,
            groupDrawingWidthPx = 109,
            groupedDrawing1HeightPx = 191,
            groupedDrawing1WidthPx = 78,
            groupedDrawing2HeightPx = 170,
            groupedDrawing2WidthPx = 80,
            groupedDrawing1TopOffsetPx = 0,
            groupedDrawing1LeftOffsetPx = 0,
            groupedDrawing2TopOffsetPx = 98,
            groupedDrawing2LeftOffsetPx = 29,
            groupedDrawing1Color = '46b030',
            groupedDrawing2Color = '92d050',
            groupedDrawing1ColorArray = [70, 176, 48],
            groupedDrawing2ColorArray = [146, 208, 80],
            tolerance = 2,

            // Info: The width of the drawing is adapted to the page width. This is set in this case to 21 cm minus 4 cm for the margins.
            //       Therefore there is a call in pagestyles.js: page.css({ width: '170mm' })
            //       The jQuery call in a unit test 'page.width()' then returns simply '170' which is interpreted in the normal code as
            //       pixel value. Therefore in a unit test the page width is reduced to '170 px' and the drawing width is automatically
            //       reduced to this value, if it is larger.

            // the operations to be applied by the document model
            OPERATIONS = [
                { name: 'setDocumentAttributes',
                    attrs: {
                        document:  { defaultTabStop: 1270, fileFormat: 'odf', zoom: { value: 100 } },
                        page:      { width: 21001, height: 29700, marginLeft: 2000, marginTop: 2000, marginRight: 2000, marginBottom: 2000 }
                    }
                },
                { name: 'insertStyleSheet', type: 'drawing', styleId: '_default', default: true, attrs: { character: { fontName: 'Liberation·Serif', color: { type: 'auto' }, fontNameAsian: 'SimSun', fontNameComplex: 'Lucida·Sans', language: 'de-DE', fontSize: 12, fontSizeAsian: 10.5, fontSizeComplex: 12 }, line: { color: { type: 'rgb', value: '3465a4' }, style: 'solid', type: 'solid' }, fill: { color: { type: 'rgb', value: '729fcf' }, type: 'solid' } } },
                { name: 'insertStyleSheet', type: 'paragraph', styleId: '_default', default: true, attrs: { character: { fontName: 'Liberation·Serif', color: { type: 'auto' }, fontNameAsian: 'SimSun', fontNameComplex: 'Lucida·Sans', language: 'de-DE', fontSize: 12, fontSizeAsian: 10.5, fontSizeComplex: 12 } } },
                { name: 'insertParagraph', start: [0], attrs: { styleId: 'Standard' } },
                // inserting a drawing group
                { name: 'insertDrawing', start: [0, 0], type: 'group', attrs: { drawing: { width: 2888, height: 7092, anchorHorOffset: 231, textWrapSide: 'both', anchorHorBase: 'column', anchorVertAlign: 'offset', marginLeft: 318, textWrapMode: 'through', marginRight: 370, indentRight: 370, anchorVertOffset: 993, anchorLayerOrder: 0, inline: false, anchorHorAlign: 'offset', marginBottom: 53, indentLeft: 318, marginTop: 0, anchorVertBase: 'paragraph' } } },
                // inserting an ellipse into the group
                { name: 'insertDrawing', start: [0, 0, 0], type: 'shape', attrs: { drawing: { top: 993, left: 231, width: 1063, height: 5054 }, shape: { paddingBottom: 127, paddingRight: 254, anchor: 'centered', paddingTop: 127, paddingLeft: 254 }, line: { color: { type: 'rgb', value: '43729d' }, width: 35, style: 'solid', type: 'solid' }, geometry: { presetShape: 'ellipse', hostData: 'H4sIAAAAAAAAAK2VXW+DIBSG/wrhynVxivWjbWqb7GZX2w/YHWlRSdTjkH7t1w80bdrhljGrAcmB\r\n8/geeYnLraCHBasLWm/Y1s0ZVEyKE+rCFRcChFuA4J9QS1qmOKNly/DN9J4JyTffJyU7SpcKRtsU\r\nr7MIrbOZarFqc4zafb7Yc3Z4hmOKfdTd57xTw1IMcKxKl5Ulby7Ei8iGyiLFryppnSXoHb31CwDg\r\npzUvqtejQDXSddPrSKi76DqidZLkOqLVk3n3ttWy1/Oxo5JD3auraaV0Z+cyMhDVrqQpLiE/8K0s\r\nvEDleTeJv3HId46WMtlA6zR84gSJr68Hj/izfmQFDwZEFoznhbRUOTVVBmjS8voeKsM7fcrIFBmi\r\nR99V+4msQPEgSGNc3wqU3Ov7z0xFSV/a1K60+SBIYyxLI0MHoK/NjmMcgPPu22EMqzsXM3qxfjxZ\r\n1mdY3onCMTzD5Q6JgzFAw+4jBRqud/5LMmw/Upnh/pE84xD8hecN/0BXX7yVBJdZBwAA\r\n' }, fill: { color: { type: 'rgb', value: groupedDrawing1Color }, type: 'solid' } } },
                { name: 'insertParagraph', start: [0, 0, 0, 0], attrs: { paragraph: { marginBottom: 0, alignment: 'left', marginTop: 0 } } },
                { name: 'insertText', start: [0, 0, 0, 0, 0], text: drawing1Text },
                { name: 'setAttributes', attrs: { character: { color: { type: 'rgb', value: 'ffffff' }, underline: false, italicComplex: false, strike: 'none', letterSpacing: 'normal', bold: false, italic: false, caps: 'none', fontName: 'Calibri2', boldComplex: false, vertAlign: 'baseline', fontSize: 11, fontSizeAsian: 11, boldAsian: false, fontSizeComplex: 11, italicAsian: false } }, start: [0, 0, 0, 0, 0], end: [0, 0, 0, 0, 4] },
                // inserting a rectangle into the group
                { name: 'insertDrawing', start: [0, 0, 1], type: 'shape', attrs: { drawing: { top: 3579, left: 626, width: 1093, height: 4506 }, shape: { paddingBottom: 127, paddingRight: 254, anchor: 'centered', paddingTop: 127, paddingLeft: 254 }, line: { color: { type: 'rgb', value: '43729d' }, width: 35, style: 'solid', type: 'solid' }, geometry: { presetShape: 'rect', hostData: 'H4sIAAAAAAAAAJWRwQrCMAyGX6X0PqbzJlbBs/oA3sKWrYW20Sxu06d3bCooU5BSKPn5viR0VTC0\r\nS4wWYo5FUiEFFL6qoRwcM3Fiid2NooA3ugRfo36LG2Rx+Wco2EkCjFAbPVMztSkX/c20qptq2Ths\r\nt9SNSX+ezPWERhN1wSeMuTzKr+lOINbo/cDsBuPLO7wydVQHvV6N1PkC4iiOjgihV5fPTiVxuHgw\r\n2lPVukJsmvVc+gb+8swnPBZdZeVPUfZV9Jdm8W2vCUs6/eXrO8ipjHsLAgAA\r\n' }, fill: { color: { type: 'rgb', value: groupedDrawing2Color }, type: 'solid' } } },
                { name: 'insertParagraph', start: [0, 0, 1, 0], attrs: { paragraph: { lineHeight: { type: 'percent', value: 100 }, marginBottom: 0, alignment: 'center', marginTop: 0 } } },
                { name: 'insertText', start: [0, 0, 1, 0, 0], text: drawing2Text },
                { name: 'setAttributes', attrs: { character: { color: { type: 'rgb', value: 'ffffff' }, underline: false, italicComplex: false, strike: 'none', letterSpacing: 'normal', bold: false, italic: false, caps: 'none', fontName: 'Calibri2', boldComplex: false, vertAlign: 'baseline', fontSize: 11, fontSizeAsian: 11, boldAsian: false, fontSizeComplex: 11, italicAsian: false } }, start: [0, 0, 1, 0, 0], end: [0, 0, 1, 0, 4] }
            ];

        AppHelper.createTextApp('odf', OPERATIONS).done(function (app) {
            model = app.getModel();
            var node = model.getNode();
            firstParagraph  = $(Position.getParagraphElement(node, [0]));
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(TextModel).to.be.a('function');
        });

        describe('checking group drawing and grouped drawings', function () {

            it('should contain one group drawing in the first paragraph', function () {
                groupDrawing = firstParagraph.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                expect(groupDrawing.length).to.equal(1);
                expect(DrawingFrame.isGroupDrawingFrame(groupDrawing)).to.equal(true);
            });

            it('should contain two grouped drawings in the group drawing', function () {
                groupedDrawings = groupDrawing.find(DOM.ABSOLUTE_DRAWING_SELECTOR);
                expect(groupedDrawings.length).to.equal(2);
                groupedDrawing1 = $(groupedDrawings[0]);
                groupedDrawing2 = $(groupedDrawings[1]);
                expect(DrawingFrame.isGroupedDrawingFrame(groupedDrawings[0])).to.equal(true);
                expect(DrawingFrame.isGroupedDrawingFrame(groupedDrawings[1])).to.equal(true);
                expect(groupedDrawing1.text()).to.equal(drawing1Text);
                expect(groupedDrawing2.text()).to.equal(drawing2Text);
            });

            // checking the size of the group drawing
            it('should have the correct size for the group drawing', function () {
                expect(groupDrawing.width()).to.be.within(groupDrawingWidthPx - tolerance, groupDrawingWidthPx + tolerance);
                expect(groupDrawing.height()).to.be.within(groupDrawingHeightPx - tolerance, groupDrawingHeightPx + tolerance);
            });

            it('should have the correct size for the grouped drawings', function () {
                expect(groupedDrawing1.height()).to.be.within(groupedDrawing1HeightPx - tolerance, groupedDrawing1HeightPx + tolerance);
                expect(groupedDrawing1.width()).to.be.within(groupedDrawing1WidthPx - tolerance, groupedDrawing1WidthPx + tolerance);
                expect(groupedDrawing2.height()).to.be.within(groupedDrawing2HeightPx - tolerance, groupedDrawing2HeightPx + tolerance);
                expect(groupedDrawing2.width()).to.be.within(groupedDrawing2WidthPx - tolerance, groupedDrawing2WidthPx + tolerance);
            });

            it('should have the correct vertical offset for the grouped drawings', function () {
                expect(parseInt(groupedDrawing1.css('top'), 10)).to.be.within(groupedDrawing1TopOffsetPx - tolerance, groupedDrawing1TopOffsetPx + tolerance);
                expect(parseInt(groupedDrawing2.css('top'), 10)).to.be.within(groupedDrawing2TopOffsetPx - tolerance, groupedDrawing2TopOffsetPx + tolerance);
            });

            it('should have the correct horizontal offset for the grouped drawings', function () {
                expect(parseInt(groupedDrawing1.css('left'), 10)).to.be.within(groupedDrawing1LeftOffsetPx - tolerance, groupedDrawing1LeftOffsetPx + tolerance);
                expect(parseInt(groupedDrawing2.css('left'), 10)).to.be.within(groupedDrawing2LeftOffsetPx - tolerance, groupedDrawing2LeftOffsetPx + tolerance);
            });

            it('should have the correct background color set as explicit attributes', function () {
                var attrs1 = AttributeUtils.getExplicitAttributes(groupedDrawing1);
                var attrs2 = AttributeUtils.getExplicitAttributes(groupedDrawing2);
                expect(attrs1.fill.color.value).to.equal(groupedDrawing1Color);
                expect(attrs2.fill.color.value).to.equal(groupedDrawing2Color);
            });

            it('should have the correct background color set in the canvas node', function () {
                var canvasNode1 = DrawingFrame.getCanvasNode(groupedDrawing1);
                var canvasNode2 = DrawingFrame.getCanvasNode(groupedDrawing2);
                expect(canvasNode1.length).to.equal(1);
                expect(canvasNode2.length).to.equal(1);
                var imageData1 = canvasNode1[0].getContext('2d').getImageData(20, 20, 1, 1).data; // filled point in ellipse
                var imageData2 = canvasNode2[0].getContext('2d').getImageData(2, 2, 1, 1).data; // filled point in rectangle
                expect(imageData1[0]).to.equal(groupedDrawing1ColorArray[0]);
                expect(imageData1[1]).to.equal(groupedDrawing1ColorArray[1]);
                expect(imageData1[2]).to.equal(groupedDrawing1ColorArray[2]);
                expect(imageData2[0]).to.equal(groupedDrawing2ColorArray[0]);
                expect(imageData2[1]).to.equal(groupedDrawing2ColorArray[1]);
                expect(imageData2[2]).to.equal(groupedDrawing2ColorArray[2]);
            });

        });

    });

});
