/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/text/components/drawing/drawingresize',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/tk/utils'
], function (AppHelper, DrawingResize, DOM, Utils) {

    'use strict';

    // class DrawingResize ==============================================

    describe('Text class DrawingResize', function () {

        // private helpers ----------------------------------------------------
        var textApp = null;
        var model = null;
        var selection = null;

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes',
                attrs: {
                    document:  { defaultTabStop: 1270, zoom: { value: 100 } },
                    page:      { width: 21590, height: 27940, marginLeft: 2540, marginTop: 2540, marginRight: 2540, marginBottom: 2540, marginHeader: 1248, marginFooter: 1248 },
                    character: { fontName: 'Arial', fontSize: 11, language: 'en-US', languageEa: 'en-US', languageBidi: 'ar-SA' },
                    paragraph: { lineHeight: { type: 'percent', value: 119 }, marginBottom: 352 }
                }
            },
            { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading1', styleName: 'heading 1',
                attrs: {
                    character: { bold: true, fontName: 'Times New Roman', fontSize: 14, color: { transformations: [{ type: 'shade', value: 74902 }], type: 'scheme', value: 'accent1' } },
                    paragraph: { marginTop: 846, outlineLevel: 0, nextStyleId: 'Normal' }
                },
                parent: 'Normal',
                uiPriority: 9
            },
            { name: 'insertParagraph', start: [0] },
            { name: 'insertText', text: 'Hello World.', start: [0, 0] },
            { name: 'insertDrawing', start: [0, 0], type: 'shape', attrs: { drawing: { name: 'TextBox 1', width: 5000, height: 1000 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'none' } } },
            { name: 'setAttributes', start: [0, 0], attrs: { drawing: { inline: false, anchorHorBase: 'column', anchorVertBase: 'paragraph', anchorHorAlign: 'right', anchorHorOffset: 0, anchorVertOffset: 0 } } },
            { name: 'insertParagraph', start: [0, 0, 0], attrs: { character: { color: { type: 'scheme', value: 'light1' } }, paragraph: { alignment: 'center' } } }
        ];

        AppHelper.createTextApp('ooxml', OPERATIONS).done(function (app) {
            textApp = app;
            model = app.getModel();
            selection = model.getSelection();
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(DrawingResize).to.be.a('object');
        });

        function triggerTrackingEvent(node, type, x, y, options) {
            var positionNode = Utils.getObjectOption(options, 'drawingNode', node);
            var isShiftKey = Utils.getBooleanOption(options, 'shiftKey', false);
            var startX = parseFloat(positionNode.css('left'));
            var startY = parseFloat(positionNode.css('top'));
            var event = new $.Event(type, {
                which: 1,
                startX: startX,
                startY: startY,
                pageX: x + startX,
                pageY: y + startY,
                shiftKey: isShiftKey
            });
            node.trigger(event);
        }

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(DrawingResize).to.be.a('object');
        });

        // public methods -----------------------------------------------------

        describe('method "DrawingResize.drawDrawingSelection"', function () {

            it('should exist', function () {
                expect(DrawingResize).to.respondTo('drawDrawingSelection');
            });

            it('should rotate drawing correctly', function () {
                selection.setTextSelection([0, 0], [0, 1]);

                var firstDrawing = selection.getSelectedDrawing();
                var rotateHandle = firstDrawing.find('.rotate-handle');
                var oldAttributes = firstDrawing.data('attributes');

                firstDrawing.css({ position: 'absolute', left: 0 });

                expect(oldAttributes.drawing.anchorHorOffset).to.be.closeTo(0, 20);
                expect(oldAttributes.drawing.anchorVertOffset).to.be.closeTo(0, 20);
                expect(oldAttributes.drawing.rotation).not.exist;

                DrawingResize.drawDrawingSelection(textApp, firstDrawing);

                triggerTrackingEvent(rotateHandle, 'tracking:start', 0, 0, { drawingNode: firstDrawing });
                triggerTrackingEvent(rotateHandle, 'tracking:move', 500, 19, { drawingNode: firstDrawing }); // height 38px, so divide /2
                triggerTrackingEvent(rotateHandle, 'tracking:end', 500, 19, { drawingNode: firstDrawing });

                var newAttributes = firstDrawing.data('attributes');
                expect(newAttributes.drawing.rotation).not.to.be.equal(0);
                expect(newAttributes.drawing.rotation).to.be.equal(90);

                expect(newAttributes.drawing.anchorHorOffset).to.be.closeTo(0, 20);
                expect(newAttributes.drawing.anchorVertOffset).to.be.closeTo(0, 20);

                DrawingResize.drawDrawingSelection(textApp, firstDrawing);
                firstDrawing.css({ position: 'absolute', left: 0 });

                triggerTrackingEvent(rotateHandle, 'tracking:start', 79.5, 0, { drawingNode: firstDrawing }); // 159px is drawing width, /2 because rotate handle is in middle
                triggerTrackingEvent(rotateHandle, 'tracking:move', 79.5, 1000, { drawingNode: firstDrawing });
                triggerTrackingEvent(rotateHandle, 'tracking:end', 79.5, 1000, { drawingNode: firstDrawing });

                newAttributes = firstDrawing.data('attributes');

                expect(newAttributes.drawing.rotation).not.to.be.equal(90);
                expect(newAttributes.drawing.rotation).to.be.equal(180);

                DrawingResize.drawDrawingSelection(textApp, firstDrawing);
                firstDrawing.css({ position: 'absolute', left: 0 });

                triggerTrackingEvent(rotateHandle, 'tracking:start', 0, 0, { drawingNode: firstDrawing });
                triggerTrackingEvent(rotateHandle, 'tracking:move', -100, 19, { drawingNode: firstDrawing });
                triggerTrackingEvent(rotateHandle, 'tracking:end', -100, 19, { drawingNode: firstDrawing });

                newAttributes = firstDrawing.data('attributes');

                expect(newAttributes.drawing.rotation).not.to.be.equal(180);
                expect(newAttributes.drawing.rotation).to.be.equal(270);
            });

            it('should rotate drawing correctly with step 15 degree (shift key)', function () {
                selection.setTextSelection([0, 0], [0, 1]);

                var firstDrawing = selection.getSelectedDrawing();
                var rotateHandle = firstDrawing.find('.rotate-handle');

                firstDrawing.css({ position: 'absolute', left: 0 });
                DrawingResize.drawDrawingSelection(textApp, firstDrawing);

                // shift key works as rounding, as it will round to closest step 15 angle
                triggerTrackingEvent(rotateHandle, 'tracking:start', 0, 0, { drawingNode: firstDrawing, shiftKey: true });
                triggerTrackingEvent(rotateHandle, 'tracking:move', 500, 15, { drawingNode: firstDrawing, shiftKey: true }); // height 38px, so divide /2
                triggerTrackingEvent(rotateHandle, 'tracking:end', 500, 15, { drawingNode: firstDrawing, shiftKey: true });

                var newAttributes = firstDrawing.data('attributes');
                expect(newAttributes.drawing.rotation).not.to.be.equal(0);
                expect(newAttributes.drawing.rotation).to.be.equal(90);

                expect(newAttributes.drawing.anchorHorOffset).to.be.closeTo(0, 20);
                expect(newAttributes.drawing.anchorVertOffset).to.be.closeTo(0, 20);

                DrawingResize.drawDrawingSelection(textApp, firstDrawing);
                firstDrawing.css({ position: 'absolute', left: 0 });

                triggerTrackingEvent(rotateHandle, 'tracking:start', 75.5, 0, { drawingNode: firstDrawing, shiftKey: true }); // 159px is drawing width, /2 because rotate handle is in middle
                triggerTrackingEvent(rotateHandle, 'tracking:move', 75.5, 1000, { drawingNode: firstDrawing, shiftKey: true });
                triggerTrackingEvent(rotateHandle, 'tracking:end', 75.5, 1000, { drawingNode: firstDrawing, shiftKey: true });

                newAttributes = firstDrawing.data('attributes');

                expect(newAttributes.drawing.rotation).not.to.be.equal(90);
                expect(newAttributes.drawing.rotation).to.be.equal(180);

                DrawingResize.drawDrawingSelection(textApp, firstDrawing);
                firstDrawing.css({ position: 'absolute', left: 0 });

                triggerTrackingEvent(rotateHandle, 'tracking:start', 0, 0, { drawingNode: firstDrawing, shiftKey: true });
                triggerTrackingEvent(rotateHandle, 'tracking:move', -90, 25, { drawingNode: firstDrawing, shiftKey: true });
                triggerTrackingEvent(rotateHandle, 'tracking:end', -90, 25, { drawingNode: firstDrawing, shiftKey: true });

                newAttributes = firstDrawing.data('attributes');
                expect(newAttributes.drawing.rotation).not.to.be.equal(180);
                expect(newAttributes.drawing.rotation).to.be.equal(270);
            });
        });
    });

    // ========================================================================
});
